/*
 * lib/clipboard.c, part of W
 * (C) 94-03/96 by Torsten Scherer (TeSche)
 * itschere@techfak.uni-bielefeld.de
 *
 * This is based on the selection.c in the W Toolkit and moved into W
 * library in preparation of W server clipboard (when client side clipboard
 * functions have the be able to manage the server socket buffer), which
 * will allow data exchange between clients on different hosts.
 *
 * Changes:
 * - Changed to earlier `agreed' clipboard standard where clips of different
 *   type reside on a certain directory as _separate_ files.  Clip type and
 *   size are taken from the clip file name and size instead of reading them
 *   from the clip file itself.
 * - Changed clipboard permissions to 0600 (just to be sure...).
 * - Added clipboard open, append and close functions to ease on the fly
 *   clip translation.
 * - Used unix open/read/write/close functions instead of C streams so that
 *   I can refer to clips with an int (file descriptor).
 * - Added TRACE stuff to functions.
 *
 * (w) 1996 by Kay Roemer, 1997 by Eero Tamminen
 */
#include <fcntl.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/stat.h>
#include "Wlib.h"
#include "proto.h"

#define SEL_DIR		"/tmp/wclips"
#define BUF_SIZE	(16 + W_SELTYPE_MAX)	/* strlen(SEL_DIR) + 2 + ... */


static int
dolock (int fd, int mode)
{
#ifdef __MINT__
  /* this locking doesn't work on the MiNT ram-fs where most(?) of the
   * MiNT users keep their /tmp. :-(
   *
   * Normally locking shouldn't be necessary because clipboard is mostly
   * accessed only by one client at the time (keyboard focus / mouse (used
   * by user to initiate clipboard accesses) is in one window at the
   * time...).
   */
  return 0;
#else
  /*
   * different OSes might have differing lock schemes... */
  struct flock fl;

  fl.l_type = mode;
  fl.l_whence = 0;
  fl.l_len = 0;
  fl.l_pid = 0;

  return fcntl (fd, F_SETLKW, &fl);
#endif
}

void
w_selclose (w_clipboard_t file)
{
  TRACESTART();
  if(file)
  {
    dolock (file, F_UNLCK);
    close (file);
  }
  TRACEPRINT(("w_selclose(%d)\n", file));
  TRACEEND();
}

w_clipboard_t
w_selappend(w_clipboard_t file, char *data, long len)
{
  TRACESTART();
  if (file >= 0 && len > 0)
  {
    if (write (file, data, len) == len)
    {
      TRACEPRINT(("w_selappend(%d,%p,%ld) -> %d\n", file, data, len, file));
      TRACEEND();
      return file;
    }
  }
  TRACEPRINT(("w_selappend(%d,%p,%ld) -> -1\n", file, data, len));
  TRACEEND();
  return -1;
}

w_clipboard_t
w_selopen (char *type)
{
  char name[BUF_SIZE] = SEL_DIR;
  w_clipboard_t file;
  int len;

  TRACESTART();
  mkdir(name, 0700);			/* permissions won't matter */

  len = strlen(name);
  name[len++] = '/';
  strncpy(&name[len], type, W_SELTYPE_MAX);
  if((file = open (name, O_WRONLY|O_CREAT, 0600)) < 0)
  {
    TRACEPRINT(("w_selopen(`%s') -> unable to open\n", type));
    TRACEEND();
    return -1;
  }
  if (dolock (file, F_WRLCK) < 0)
  {
    TRACEPRINT(("w_selopen(`%s') -> unable to lock\n", type));
    TRACEEND();
    close (file);
    return -1;
  }
  ftruncate (file, 0);
  TRACEPRINT(("w_selopen(`%s') -> %d\n", type, file));
  TRACEEND();
  return file;
}

int
w_putselection (char *type, char *data, long len)
{
  w_clipboard_t file;

  TRACESTART();

  if((file = w_selopen(type)) < 0)
  {
    TRACEPRINT(("w_putselection(`%s',%p,%ld) -> unable to open\n", type, data, len));
    TRACEEND();
    return -1;
  }
  w_selappend(file, data, len);
  w_selclose(file);
  TRACEPRINT(("w_putselection(`%s',%p,%ld) -> 0\n", type, data, len));
  TRACEEND();
  return 0;
}

w_selection_t *
w_getselection (char *type)
{
  char name[BUF_SIZE] = SEL_DIR;
  w_selection_t *sel = NULL;
  struct stat st;
  int file;
  int len;

  TRACESTART();

  /* make sure there's a clipboard directory */
  mkdir(name, 0700);

  len = strlen(name);
  name[len++] = '/';
  strncpy(&name[len], type, W_SELTYPE_MAX);
  if((file = open (name, O_RDONLY)) < 0)
  {
    TRACEPRINT(("w_getselection(`%s') -> unable to open clip type\n", type));
    TRACEEND();
    return NULL;
  }
  if (dolock (file, F_RDLCK) < 0)
  {
    TRACEPRINT(("w_getselection(`%s') -> unable to lock clip type\n", type));
    TRACEEND();
    close (file);
    return NULL;
  }

  if((sel = calloc (1, sizeof (w_selection_t))))
  {
    strncpy(sel->type, type, W_SELTYPE_MAX);

    stat(name, &st);
    if((sel->data = malloc(st.st_size+1)))
    {
      sel->len = read(file, sel->data, st.st_size);
      sel->data[sel->len] = 0;
      dolock (file, F_UNLCK);
      close (file);
      TRACEPRINT(("w_getselection(`%s') -> %p\n", type, sel));
      TRACEEND();
      return sel;
    }
    free(sel);
  }
  dolock (file, F_UNLCK);
  close (file);
  TRACEPRINT(("w_getselection(`%s') -> alloc failed\n", type));
  TRACEEND();
  return NULL;
}

void
w_freeselection (w_selection_t *sel)
{
  TRACESTART();
  if(sel)
  {
    free (sel->data);
    free (sel);
  }
  TRACEPRINT(("w_freeselection(%p)\n", sel));
  TRACEEND();
}
