/*
 * lib/block.c, part of Wlib, part of W
 * (C) 94-07/96 by Torsten Scherer (TeSche)
 * itschere@techfak.uni-bielefeld.de
 *
 * functions to get/put an image from/to screen and to allocate/free
 * bitmap.
 *
 * CHANGES:
 * - Added color and trace stuff to w_alloc/freebm().
 * - moved bitmap convertion functions to convert.c file.
 * - Fixed get/putblock functions.
 *   ++eero 5/97
 *
 * - Changed putblock to do image convertion/sending line-by-line.
 * - Moved bitmap alloc/free functions to convert.c.
 *   ++eero 5/97
 *
 * TODO:
 * - w_getblock() should probably get also the window palette.
 */

#include <stdio.h>
#include <stdlib.h>
#include "Wlib.h"
#include "proto.h"


/*****************************************************************************/

/* get a block
 */
BITMAP *
w_getblock(WWIN *win, short x0, short y0, short width, short height)
{
	char *data;
	BITMAP *ret;
	GETBLKREQP paket;
	GETBLKDATAP paket2;
	PAKET *rawdata;
	long size, todo;
	int datasize;

	TRACESTART();

	if ((width < 1) || (height < 1)) {
		TRACEPRINT(("w_getblock(%p,%i,%i,%i,%i) -> illegal size\n",\
		            win, x0, y0, width, height));
		TRACEEND();
	}

	if ((data = _check_window(win))) {
		TRACEPRINT(("w_getblock(%p,%i,%i,%i,%i) -> %s\n",\
		            win, x0, y0, width, height, data));
		TRACEEND();
		return NULL;
	}

	ret = w_allocbm (width, height, _wserver.type, 1<<_wserver.planes);
	if (!ret) {
		TRACEPRINT(("w_getblock(%p,%i,%i,%i,%i) -> alloc failed\n",\
		            win, x0, y0, width, height));
		TRACEEND();
		return NULL;
	}
	size = ret->height * ret->unitsize * ret->upl;

	paket.len = sizeof(GETBLKREQP);
	paket.type = PAK_GETBLKREQ;
	paket.handle = htons(win->handle);
	paket.x0 = htons(x0);
	paket.y0 = htons(y0);
	paket.width = htons(width);
	paket.height = htons(height);
	_send_paket((PAKET *)&paket);

	todo = ntohl(((LRETP *)_wait4paket(PAK_LRET))->ret);
	if (todo != size) {
		w_freebm (ret);
		TRACEPRINT(("w_getblock(%p,%i,%i,%i,%i) -> NULL (%ld!=%ld)\n",\
		            win, x0, y0, width, height, size, todo));
		TRACEEND();
		return NULL;
	}

	data = ret->data;
	while (todo > 0) {
		paket2.len = sizeof(GETBLKDATAP);
		paket2.type = PAK_GETBLKDATA;
		_send_paket((PAKET *)&paket2);

		rawdata = _wait4paket(PAK_RAWDATA);
		datasize = rawdata->len - (sizeof(*rawdata)-sizeof(rawdata->data));
		if (todo - datasize < 0) {
			w_freebm (ret);
			TRACEPRINT(("w_getblock(%p,%i,%i,%i,%i) -> NULL\n",\
			            win, x0, y0, width, height));
			TRACEEND();
			return NULL;
		}
		memcpy(data, rawdata->data, datasize);
		data += datasize;
		todo -= datasize;
	}

	TRACEPRINT(("w_getblock(%p,%i,%i,%i,%i) -> %p\n",\
		    win, x0, y0, width, height, ret));
	TRACEEND();

	/* with a bogus palette... */
	return ret;
}


/* put a block
 */
short
w_putblock (BITMAP *bm, WWIN *win, short x1, short y1)
{
	char *cptr, *data;
#ifdef __MINT__
	static PAKET dpaket;	/* quite big, so let's save stack */
#else
	PAKET dpaket;
#endif
	PUTBLKREQP paket;
	long size, todo;
	BITMAP *use = NULL;
	uchar *line, *(*conv)(BITMAP *, BITMAP *, uchar *);
	int width, height, colors;
	int datasize;

	TRACESTART();

	if (!bm) {
		cptr = "no BITMAP";
		goto error;
	}
	if ((cptr = _check_window(win))) {
		goto error;
	}

	width = bm->width;
	height = bm->height;

	/* set explicitly */
	if (bm->type == BM_PACKEDMONO) {
		colors = 2;
	} else {
		colors = bm->colors;
		if (!colors) {
			colors = 1 << _wserver.planes;
		}
	}

	if (!(use = w_allocbm(width, 1, _wserver.type, colors))) {
		cptr = "work bitmap allocation failed";
		goto error;
	}

	if (!(conv = w_convertFunction(bm, use))) {
		cptr = "no bitmap conversion function";
		goto error;
	}

	size = height * use->unitsize * use->upl;

	paket.len = sizeof(PUTBLKREQP);
	paket.type = PAK_PUTBLKREQ;
	paket.width = htons(width);
	paket.height = htons(height);
	paket.handle = htons(win->handle);
	paket.x1 = htons(x1);
	paket.y1 = htons(y1);
	paket.shmKey = htonl(0);
	_send_paket((PAKET *)&paket);

	todo = ntohl(((LRETP *)_wait4paket(PAK_LRET))->ret);
	if (todo != size) {
		if( use != bm) {
			w_freebm(use);
		}
		TRACEPRINT(("w_putblock(%p,%p,%i,%i) -> -1 (%ld!=%ld)\n",\
		           bm, win, x1, y1, todo, size));
		TRACEEND();
		return -1;
	}

	/* doing it line by line might add one additional rawpacket / image
	 * line.  It means 4 more bytes / line (packet header).  However,
	 * we'll need less memory.  IMHO a good bargain. ++eero 10/97
	 */
	line = bm->data;
	while (--height >= 0) {

		line = (*conv)(bm, use, line);
		todo = use->unitsize * use->upl;
		data = use->data;

		while (todo > 0) {
			datasize = todo;
			if (datasize > sizeof(dpaket.data)) {
				datasize = sizeof(dpaket.data);
			}
			memcpy(dpaket.data, data, datasize);
			/* this has to be here as _send_paket() converts
			 * it every time...
			 */
			dpaket.type = PAK_RAWDATA;
			dpaket.len = datasize + (sizeof(dpaket) - sizeof(dpaket.data));
			_send_paket((PAKET *)&dpaket);
			data += datasize;
			todo -= datasize;
		}
	}
	w_freebm (use);

	TRACEPRINT(("w_putblock(%p,%p,%i,%i) -> 0\n", bm, win, x1, y1));
	TRACEEND();
	return 0;

error:
	if (use) {
		w_freebm (use);
	}
	TRACEPRINT(("w_putblock(%p,%p,%i,%i) -> %s\n",\
	           bm, win, x1, y1, cptr));
	TRACEEND();
	return -1;
}
