/* W window system interface for John Beale's HF-Lab v0.90
 *
 * HF-Lab is a really nice command line heightfield creator with a X11
 * heightfield previewer which can do light shading.  If you'll replace the
 * x_iface.c file with this, you can then do previews under W before
 * raytracing the heightfield...
 *
 * When using HF-Lab 'view' with monochrome W server, either set 'contrast
 * maximation' ON or use s2 option (slope shading). I myself prefer the
 * p2 drawing density (p1 = no skip).
 *
 * (w) 1997 by Eero Tamminen
 *
 * NOTES:
 *   X_PutImagemap data should be long aligned (evenly divisable by four),
 *   so don't use any funny preview window widths.
 *
 * TODO:
 *   After W color routines stabilize, use colors instead of pattern in
 *   X_Point/X_FillPoly drawing routines for DIRECT8 W server.
 */

#include <stdio.h>
#include <stdlib.h>
#include <Wlib.h>

static WSERVER *wserver;

static WWIN *win;
static rgb_t *gray;
static int colors;

#define MAX_COLORS	256
#define WIN_PROPERTIES	(W_TITLE | W_MOVE)

/* If this succeeds (returns 0), it's not supposed to be called again with
 * different values...
 */
int init_graph(int wx, int wy, int ncols)
{
	int intensity;

	if (!wserver) {
		if (!(wserver = w_init())) {
			fprintf(stderr, "unable to connect W server\n");
			return -1;
		}
	}
	if (!win) {
		if (!(win = w_create(wx, wy, WIN_PROPERTIES))) {
			fprintf(stderr, "unable to create W window\n");
			return -1;
		}
	}
	/* clear to black */
	w_setmode(win, M_DRAW);
	w_pbox(win, 0, 0, win->width, win->height);

	if(w_open(win, UNDEF, UNDEF) < 0) {
		fprintf(stderr, "unable to open W window\n");
		return -1;
	}
	if (!gray) {
		if (ncols > MAX_COLORS) {
			ncols = MAX_COLORS;
		}
		if (!(gray = malloc(sizeof(rgb_t) * ncols))) {
			fprintf(stderr, "palette alloc failed\n");
			return -1;
		}
		colors = ncols;
		while (--ncols >= 0) {
			intensity = 256L * ncols / colors;
			gray[ncols].red   = intensity;
			gray[ncols].green = intensity;
			gray[ncols].blue  = intensity;
		}
	}
	printf("using %dx%d window in %d colors.\n", wx, wy, colors);
	return 0;
}



void close_graph(void)
{
	if (win) {
		w_close(win);
	}
}

int GetMaxX(void)
{
	if (win) {
		return win->width;
	}
	if (wserver) {
		return wserver->width;
	}
	return 0;
}

int GetMaxY(void)
{
	if (win) {
		return win->height;
	}
	if (wserver) {
		return wserver->height;
	}
	return 0;
}

void Gflush(void)
{
	w_flush();
}

static void assert(WWIN *win)
{
	if (!win) {
		fprintf(stderr, "no window to draw to, exiting...\n");
		exit(-1);
	}
}

void Clear(void)
{
	assert(win);
	/* clear to black */
	w_pbox(win, 0, 0, win->width, win->height);
}

void X_Point(int x, int y)
{
	assert(win);
	w_plot(win, x, y);
}

void X_FillPoly(int n, short *points, int linecol, int fillcol)
{
	assert(win);
	w_setpattern(win, MAX_GRAYSCALES * fillcol / colors);
	w_dppoly(win, n, points);
	if (!fillcol) {
		w_setpattern(win, MAX_GRAYSCALES * linecol / colors);
		w_dpoly(win, n, points);
	}
}

void X_PutImagemap(uchar *data, int w, int h, int x, int y)
{
	BITMAP bm, *hf = NULL;

	assert(win);

	bm.type = BM_DIRECT8;
	bm.width = w;
	bm.height = h;
	bm.planes = 8;
	bm.unitsize = 1;
	bm.upl = w;

	bm.colors = colors;
	bm.palette = gray;
	bm.data = data;

	/* if we have to do color conversions, monochrome gets best
	 * results.
	 */
	if (wserver->type != BM_DIRECT8) {
		hf = w_convertBitmap(&bm, BM_PACKEDMONO, 2);
	}
	if (!hf) {
		hf = &bm;
	}
	w_putblock(hf, win, x, y);
	w_flush();
	if (hf != &bm) {
		w_freebm(hf);
	}
}
