/*
 * Bitmap lense (w) 1997 by Eero Tamminen
 *
 * Only point_copy() and main() functions are GUI / platform dependent.
 *
 * With W window system + MiNT this could be used like this:
 *	$ gcc -O lense.c -o lense -lW -lsocket
 *	$ djpeg -gray image.jpg | lense
 *
 * NOTES:
 *
 * - lense lookup array could be allocated instead of using a static
 * one from lense_mapping() (where it's calculated). User could be
 * even presented with a modifiable lense image from which mapping would
 * be created in accordance to physical laws concerning optics...
 *
 * - if point_copy() isn't inlined into bitmap_lense(), it could be `virtual'
 * function ie.  one could use pointer to a grayscale, truecolor, clipped
 * etc.  version when appropiate.
 */

#include <stdlib.h>
#include <stdio.h>
#include <Wlib.h>

/* limit effect to given circle (lense where curvature changes sign) */
#define LIMIT2CIRCLE

/* max. lense radius = needed distortion lookup table size */
#define MAX_LENSE_SIZE	512

/* value for `clipped' pixels */
#define DEFAULT_VALUE	0


static void
point_copy(BITMAP *src, int sx, int sy, BITMAP *dst, int dx, int dy)
{
	if (dx < 0 || dy < 0 || dx >= dst->width || dy >= dst->height) {
		return;
	}
	if (sx < 0 || sy < 0 || sx >= src->width || sy >= src->height) {
		*((uchar*)dst->data + dst->upl * dst->unitsize * dy + dx) =
		DEFAULT_VALUE;
		return;
	}
	*((uchar*)dst->data + dst->upl * dst->unitsize * dy + dx) =
	*((uchar*)src->data + src->upl * src->unitsize * sy + sx);
}

/* Quick integer square root using binomial theorem (from Dr. Dobbs journal) */
static unsigned long
isqrt(unsigned long N)
{
	unsigned long l2, u, v, u2, n;
	if(2 > N) {
		return N;
	}
	u = N;
	l2 = 0;
	/* 1/2 * log_2 N = highest bit in the result */
	while((u >>= 2)) {
		l2++;
	}
	u = 1L << l2;
	v = u;
	u2 = u << l2;
	while(l2--) {
		v >>= 1;
		n = (u + u + v) << l2;
		n += u2;
		if(n <= N) {
			u += v;
			u2 = n;
		}
	}
	return u;
}

/* map a source point (c,d) into destination point (a,b) */
static void
lense_point (long r, long a, long b, long *c, long *d, short *map)
{
	short h, new;

	/* point distance from center */
	h = isqrt(a*a + b*b);
	new = h - map[h];

	/* new, distorted offset(s) */
	*c = a * new / h;
	*d = b * new / h;
}

/* compose lookup table for given lense */
static short *
lense_mapping(int r, float strenght)
{
	static short map[MAX_LENSE_SIZE];
	long x;

	if (r > sizeof(map)) {
		fprintf(stderr, "error: illegal mapping size: %d\n", r);
		return NULL;
	}
	map[0] = 0;
	if (r < 1) {
		return map;
	}

#ifdef LIMIT2CIRCLE
	/* calculate distance based offset lookup table. With plain `r'
	 * division instead of `strenght' factor, offset would be r/4
	 * in the midpoint between lense center and edge. Offsets at
	 * the center and edges are always zero.
	 */

	/* `strenght' of value 2.0 or greater would make the offsets go off
	 * the lense area, thus the limitation.
	 */
	if (strenght <= -2.0 || strenght >= 2.0) {
		fprintf(stderr, "error: illegal lense strenght: %f\n", strenght);
		return NULL;
	}
	/* table normalization */
	strenght = strenght / r;

	x = --r;
	while (x >= 0) {
		map[x] = (r - x) * x * strenght + 0.5;
		x--;
	}
#else
	/* sensible values are ones in range of about -0.8 - 0.8.
	 * value 1.0 will enlarge center point to whole circle.
	 */
	strenght = strenght / r;

	x = 0;
	while (++x < r) {
		map[x] = (x * x) * strenght + 0.5;
	}
#endif

	return map;
}

/* a lense effect of given size `r' into given position (x,y) from
 * `src' to `dst' bitmap. Positive `sign' magnifies.
 */
static void
bitmap_lense (BITMAP *src, BITMAP *dst, int x, int y, int r, float strenght)
{
	long a, b, c, d, dx, dy, rq = r * r;
	short *map;

	map = lense_mapping(r, strenght);
	if (! map) {
		return;
	}

	/* center point */
	point_copy (src, x, y, dst, x, y);
	if (r < 2) {
		return;
	}

	/* middle line */
	b = 0;
	for (a = 1; a <= r; a++) {
		lense_point (r, a, b, &c, &d, map);
		point_copy (src, x - c, y, dst, x - a, y);
		point_copy (src, x + c, y, dst, x + a, y);
	}

	dy = 0;
	dx = r;
	while (dx) {

		dy++;
		while (dx*dx + dy*dy > rq) {
			dx--;
		}
		b = dy;

		/* middle point(s) */
		a = 0;
		lense_point (r, a, b, &c, &d, map);
		point_copy (src, x - c, y - d, dst, x - a, y - b);
		point_copy (src, x - c, y + d, dst, x - a, y + b);

		/* rest of the line(s) */
		for (a = 1; a <= dx; a++) {
			lense_point (r, a, b, &c, &d, map);
			point_copy (src, x - c, y - d, dst, x - a, y - b);
			point_copy (src, x - c, y + d, dst, x - a, y + b);
			point_copy (src, x + c, y - d, dst, x + a, y - b);
			point_copy (src, x + c, y + d, dst, x + a, y + b);
		}
	}
}

/* main */

int main(int argc, char *argv[])
{
	long timeout;
#ifdef LIMIT2CIRCLE
	float strenght = 1.5;
#else
	float strenght = 0.4;
#endif
	short mx, my, x, y, r, new;
	BITMAP *src, *dst;
	WEVENT *ev;
	WWIN *win;

	r = 0;
	while (++r < argc && argv[r][0] == '-' && !argv[r][2]) {
		if (argv[r][1] == 's' && r+1 < argc) {
			strenght = atof(argv[++r]);
		} else {
#ifdef LIMIT2CIRCLE
		   fprintf(stderr, "usage: %s [-s <strenght (-2.0 - 2.0)>] [PBM image]\n", *argv);
#else
		   fprintf(stderr, "usage: %s [-s <strenght (-0.6 - 0.6)>] [PBM image]\n", *argv);
#endif
		   return -1;
		}
	}

	dst = w_readpbm(argv[r]);
	if (!dst) {
		fprintf(stderr, "unable to read file %s\n", argv[r]);
		return -1;
	}
	x = dst->width;
	y = dst->height;
	src = w_allocbm(x, y, dst->type, dst->colors);
	if (!src) {
		fprintf(stderr, "unable to allocate backup image\n");
		return -1;
	}
	memcpy(src->data, dst->data, dst->height * dst->unitsize * dst->upl);

	w_init();
	win = w_create(x, y, W_MOVE | W_TITLE | EV_MOUSE);
	if (!win) {
		fprintf(stderr, "unable to create window\n");
	}

	w_setmode(win, M_INVERS);
	w_putblock(dst, win, 0, 0);
	w_open(win, UNDEF, UNDEF);

	/* let user select the distortion area + do the effect */
	timeout = -1;
	r = 1;
	for (;;) {
		ev = w_queryevent(NULL, NULL, NULL, timeout);

		/* timeout? */
		if (!ev) {
			w_querymousepos(win, &mx, &my);

			mx = ABS(mx - x);
			my = ABS(my - y);
			new = MAX(mx, my);
			/* changed radius? */
			if (new != r) {
				w_circle(win, x, y, r);
				r = new;
				w_circle(win, x, y, r);
			}
			continue;
		}

		switch(ev->type) {
			case EVENT_MPRESS:
				timeout = 40;
				x = ev->x;
				y = ev->y;
			        r = 1;
				w_circle(win, x, y, r);
				break;
			case EVENT_MRELEASE:
				timeout = -1;
				w_circle(win, x, y, r);
				bitmap_lense (src, dst, x, y, r, strenght);
				w_putblock(dst, win, 0, 0);
				break;
			default:
				w_exit();
				return 0;
		}
	}
	return 0;
}
