/*
 * programs/weyes.c, part of W
 * (C) 94-02/96 by TeSche (Torsten Scherer)
 * itschere@techfak.uni-bielefeld.de
 *
 * CHANGES:
 * - resizable
 * - may be quit with 'q' key
 * - uses ellipses instead of circles
 * - eye highlights
 * - may be put on root
 * 11/97, ++eero
 */

#include <math.h>
#include <stdio.h>
#include <Wlib.h>

#define MIN_XSIZE 80
#define MIN_YSIZE 60

typedef struct {
  short midx, midy;
  short pupx, pupy;
} EYE;


static WWIN *win;
static short rootw, rooth, winw, winh, rx, ry, rxpup, rypup;
static EYE lefteye, righteye;
static int use_root;


static void updateeye(EYE *eye)
{
  short	mx, my, dx, dy;
  float	angle;

  w_querymousepos(win, &mx, &my);
  dx = mx - eye->midx;
  dy = my - eye->midy;

  /* angle measured clockwise, zero is down */

  if (!dy) {
    angle = M_PI / 2;
  } else {
    angle = atan((float)ABS(dx) / (float)ABS(dy));
  }

  if ((dy >= 0) && (dx >= 0)) {
    angle = -angle;
  } else if ((dy < 0) && (dx < 0)) {
    angle = M_PI - angle;
  } else if ((dy < 0) && (dx >= 0)) {
    angle = -M_PI + angle;
  }

  mx -= eye->midx;
  my -= eye->midy;
  if (mx < 0)
    mx = -mx;
  if (mx > rx)
    mx = rx;
  if (my < 0)
    my = -my;
  if (my > ry)
    my = ry;

  dx = eye->midx - mx * sin(angle);
  dy = eye->midy + my * cos(angle);

  /* move the pupille */
  if ((dx != eye->pupx) || (dy != eye->pupy)) {
    w_pbox(win, eye->pupx - rxpup/2, eye->pupy - rypup/2, rxpup/3, rypup/3);
    w_pellipse(win, eye->pupx, eye->pupy, rxpup, rypup);
    eye->pupx = dx;
    eye->pupy = dy;
    w_pellipse(win, dx, dy, rxpup, rypup);
    w_pbox(win, dx - rxpup/2, dy - rypup/2, rxpup/3, rypup/3);
  }
}

static void draw_eyes(int xp, int yp)
{
  if (!use_root) {
    w_setmode(win, M_DRAW);
    w_pbox(win, 0, 0, winw, winh);
  }

  rx = winw/4;
  ry = winh/2;
  lefteye.midx = lefteye.pupx = xp + rx;
  lefteye.midy = lefteye.pupy = yp + ry;
  righteye.midx = righteye.pupx = xp + 3*rx;
  righteye.midy = righteye.pupy = yp + ry;

  rx--;
  ry--;
  w_setmode(win, M_CLEAR);
  w_pellipse(win, lefteye.midx, lefteye.midy, rx, ry);
  w_pellipse(win, righteye.midx, righteye.midy, rx, ry);

  if (use_root) {
    w_setmode(win, M_DRAW);
    w_ellipse(win, lefteye.midx, lefteye.midy, rx, ry);
    w_ellipse(win, righteye.midx, righteye.midy, rx, ry);
  }

  rxpup = rx / 4;
  rypup = ry / 4;
  rx -= (rxpup + 2);
  ry -= (rypup + 2);

  w_setmode(win, M_INVERS);
  w_pellipse(win, lefteye.pupx, lefteye.pupy, rxpup, rypup);
  w_pellipse(win, righteye.pupx, righteye.pupy, rxpup, rypup);
  w_pbox(win, lefteye.pupx - rxpup/2, lefteye.pupy - rypup/2, rxpup/3, rypup/3);
  w_pbox(win, righteye.pupx - rxpup/2, righteye.pupy - rypup/2, rxpup/3, rypup/3);
}

static void check_size(short *winw, short *winh)
{
  if (*winw < MIN_XSIZE) {
    *winw = MIN_XSIZE;
  }
  if (*winh < MIN_YSIZE) {
    *winh = MIN_YSIZE;
  }
}

static void usage(void)
{
  fprintf(stderr, "usage: weyes [-r] [-h] [-g <geometry>]\n");
}


int main(long argc, char **argv)
{
  char *prg = *argv, *geo = NULL;
  short err, xp, yp, ewinw, ewinh;
  WSERVER *wserver;
  WEVENT *ev;

  err = 0;
  while (*++argv) {
    if (**argv == '-') {
      if (!*(*argv+2)) switch (*(*argv+1)) {

        case 'h':
	  err = 1;
	  break;

	case 'g':
	  if (!(geo = *++argv)) {
	    fprintf(stderr, "option requires an argument\n");
	    argv--;
	    err = 1;
	  }
	  break;

	case 'r':
	  use_root = 1;
	  break;

	default:
	  err = 1;
	  fprintf(stderr, "illegal option: %s\n", *argv);
      } else {
	err = 1;
	fprintf(stderr, "illegal option: %s\n", *argv);
      }
    } else {
      err = 1;
    }
  }

  if (err) {
    usage();
    return -1;
  }

  if (!(wserver = w_init())) {
    fprintf(stderr, "error: %s can't connect to wserver\n", prg);
    return -1;
  }
  rootw = wserver->width;
  rooth = wserver->height;

  winw = MIN_XSIZE;
  winh = MIN_YSIZE;
  xp = UNDEF;
  yp = UNDEF;
  if (geo) {
    scan_geometry(geo, &winw, &winh, &xp, &yp);
  }
  check_size(&winw, &winh);

  if (use_root) {
    win = WROOT;
    ewinw = winw + 2;
    ewinh = winh + 2;
  } else {
    if (!(win = w_create(winw, winh, W_MOVE | W_RESIZE | EV_KEYS))) {
      fprintf(stderr, "error: %s can't create window\n", prg);
      exit(-1);
    }
    w_querywinsize(win, 1, &ewinw, &ewinh);
  }

  if (xp < 0 && xp != UNDEF) {
    xp = rootw - ewinw - xp;
  }
  if (yp < 0 && yp != UNDEF) {
    yp = rooth - ewinh - yp;
  }

  if (use_root) {
    if (xp == UNDEF || yp == UNDEF) {
      xp = 0;
      yp = 0;
    }
    draw_eyes(xp, yp);
  } else {
    draw_eyes(0, 0);
    if (w_open(win, xp, yp) < 0) {
      w_delete(win);
      fprintf(stderr, "error: %s can't open window\n", prg);
      return -1;
    }
  }

  while (42) {

    updateeye(&lefteye);
    updateeye(&righteye);

    /* watch out for `exit' event */
    ev = w_queryevent(NULL, NULL, NULL, 200);
    if(!ev)
      continue;

    switch(ev->type)
    {
      case EVENT_GADGET:
      case EVENT_KEY:
        if (ev->key != GADGET_EXIT && ev->key != 'q')
	  break;
	w_close(win);
	w_delete(win);
	return 0;

      case EVENT_RESIZE:
	winw = ev->w;
	winh = ev->h;
	check_size(&winw, &winh);
	w_move (ev->win, ev->x, ev->y);
	w_resize (ev->win, winw, winh);

	/* updates global size variables too */
	draw_eyes(0, 0);
	break;
    }
  }
}
