/*
 * server/types.h, part of W
 * (C) 94-02/96 by Torsten Scherer (TeSche)
 * itschere@techfak.uni-bielfeld.de
 *
 * definition of most (internal) data structures + some (enough?) comments
 * to help you understand them...
 */

#ifndef uchar
#define	uchar	unsigned char
#endif
#ifndef ushort
#define	ushort	unsigned short
#endif
#ifndef ulong
#define	ulong	unsigned long
#endif


/* the header of a *.wfnt font FILE(!)
 */

typedef struct {
  long magic;   /* MAGIC_WFNT */
  ushort flags;
  short height;
  uchar widths[256];
} FONTHDR;


/* how a font looks like in memory
 */

typedef struct {
  FONTHDR hdr;
  char *name;   /* the name repeated, to be able to search for it */
  ulong *data;   /* actual font data in longs */
  long offsets[256];   /* offsets in longs for each char */
  long numused;   /* how many times this font is used by clients */
} FONT;


typedef struct _client {
  /* these fields implement a socket buffer */
  char *buf;
  short inbuf;

  short sh;   /* file handle for socket */

  /* some links for the linear list */
  struct _client *prev, *next;

  short flags;   /* various flags */
  ulong raddr;   /* remote IP adress if != 0 */
  long pakets, bytes;   /* some statistics */
  short uid;   /* uid of user of this client */

  /* these fields are used for get/setblock */
  short btype, bwin, bx0, by0, bwidth, bheight, bx1, by1;
  long bsize, boffset;
  BITMAP *bbm;

  /* fonts are now on a per-client basis instead of per-window */
  FONT *font[MAXCLIENTFONTS];

  /* some more statistics */
  int openWindows, totalWindows;
} CLIENT;


/* rectangle definition, used in many places like rect.c. for clipping
 * rectangles, for speed reasons we store both x1, y1 and w, h.
 * x1 = x0+w-1, y1 = y0+w-1 and w,h >= 0 is required.
 */

typedef struct _rect{
  short x0, y0, w, h;
  short x1, y1;
  struct _rect *next;
} REC;


/* a graphic context
 */

typedef struct {
  short	drawmode;   /* M_CLEAR... */
  short	textstyle;   /* F_NORMAL... */
  FONT *font;
  ushort *pattern;   /* for dashed functions */
} GCONTEXT;


/* maximum number of 'logical' areas a window might have
 */
#define AREA_MAXAREAS 5

/* some indices to the area array for each window, if *.width is < 1 the
 * area is not used in this window, e.g. it hasn't got the specific gadget.
 * naturally all windows should have the work area...
 */

#define AREA_WORK 0
#define AREA_CLOSE 1
#define AREA_ICON 2
#define AREA_TITLE 3
#define AREA_SIZE 4


typedef struct _window {
  /* client this window belongs to */
  CLIENT *cptr;

  /* window handle that is passed to the clients */
  long id;

  /* client window pointer returned in events */
  WWIN *libPtr;

  /* each window has a pointer to its parent (only for the root window the
   * `parent' pointer is NULL).
   *
   * `childs' points to a doubly linked list of childs of that window. If
   * there are no childs then `childs' is NULL.
   *
   * the childlist is constructed via the `next' and `prev' pointers. the
   * topmost window is the first (leftmost) element of the list, the window
   * at the bottom is the last (rightmost) element of the list. for the top
   * window the `prev' pointer is NULL, for the bottom window `next' is NULL.
   */
  struct _window *next, *prev;
  struct _window *parent, *childs;

  /* a pointer to a list of rectangles that make up the visible area of the
   * window. All rectangles are stored with absolute (ie not window relative
   * coordinates). both pointers are NULL if the window is closed (ie when
   * is_open == 0).
   */
  REC *vis_recs;
  REC *vis_recs_backup;

  /* `pos' is a rectangle that makes up the whole area the window covers,
   * including the border (in absolute coords). `work' is a rectangle that
   * makes up the working area of the window, excluding the border (in
   * absolute coords). `clip' will become the user-settable clipping areas,
   * clip[0] in absolute and clip[1] in relative coordinates - but they're
   * not yet implemented.
   */
  REC pos;
  REC work;
  REC clip[2];

  /* `area' is an array of rectangles describing several logical areas of the
   * window (in coordinates relative to `pos'), like the working area (again),
   * and... well, nothing so far, but more gadgets may follow.
   */
  REC area[AREA_MAXAREAS];

  /* `is_open' is nonzero if the window is opened (ie. mapped to the screen).
   */
  int is_open;

  /* `is_hidden' is zero if the window is totally visible, ie is not hidden
   * by other windows.
   */
  int is_hidden;
  int is_hidden_backup;

  /* `is_dirty' must be set by the drawing routines to some value != 0 if the
   * window on the screen is more up to date than the backing bitmap.
   *
   * the window manager routines use it in the following way when opening,
   * closing, moving, resizing or restacking windows:
   *
   * if (is_dirty) {
   *	save parts of the window on the screen to the backing bitmap
   *    because they will be obscured by other windows soon.
   * }
   */
  int is_dirty;

  /* functions for performing backing storage and redrawing for this window.
   * the rectangle is in absolute (ie not window relative) coords and is
   * already clipped to the visibile area of the window.
   */
  int (*save) (struct _window *, REC *);
  int (*redraw) (struct _window *, REC *);

  /* bitblitting routine for moving windows fast. */
  int (*bitblk) (REC *, REC *);

  short flags;   /* window creation flags */
  char title[MAXTITLE];   /* title text */

  /* the backing store bitmap */
  BITMAP bitmap;

  /* patterns */
  ushort patbuf[16]; /* buffer for patterns */

  /* the graphic context */
  GCONTEXT gc;

  /* rectangle that must be redrawn */
  REC dirty;
} WINDOW;


typedef struct {
  short	fh;   /* filehandle */
  short	visible;   /* is it currently visible? */
  short	dx, dy;   /* where it is drawn */
  short	rx, ry;   /* where it is really */
  short	disabled;   /* whether is it disabled or not */
} MOUSE;


typedef struct {
  char title[16];
  char  command[SMALLBUF];
} ITEM;


typedef struct {
  short	items;
  short	maxlen;
  ITEM item[MAXITEMS];
} MENU;


/* the interface to the actual graphic driver
 */

typedef struct {
  BITMAP bm;
  void (*vtswitch)(void);
  void (*mouseShow)(void);
  void (*mouseHide)(void);
  void (*mouseMove)(void);
  void (*plot)   (BITMAP *bm, long x0, long y0);
  long (*test)   (BITMAP *bm, long x0, long y0);
  void (*line)   (BITMAP *bm, long x0, long y0, long xe, long ye);
  void (*hline)  (BITMAP *bm, long x0, long y0, long xe);
  void (*vline)  (BITMAP *bm, long x0, long y0, long ye);
  void (*box)    (BITMAP *bm, long x0, long y0, long width, long height);
  void (*pbox)   (BITMAP *bm, long x0, long y0, long width, long height);
  void (*dvline) (BITMAP *bm, long x0, long y0, long ye);
  void (*dhline) (BITMAP *bm, long x0, long y0, long xe);
  void (*dbox)   (BITMAP *bm, long x0, long y0, long width, long height);
  void (*dpbox)  (BITMAP *bm, long x0, long y0, long width, long height);
  void (*circ)   (BITMAP *bm, long x0, long y0, long r);
  void (*pcirc)  (BITMAP *bm, long x0, long y0, long r);
  void (*bitblk) (BITMAP *bm, long x0, long y0, long width, long height, BITMAP *bm1, long x1, long y1);
  void (*scroll) (BITMAP *bm, long x0, long y0, long width, long height, long y1);
  void (*printc) (BITMAP *bm, long x0, long y0, ulong c);
  void (*prints) (BITMAP *bm, long x0, long y0, uchar *s);
  void (*dplot)  (BITMAP *bm, long x0, long y0);
  void (*dline)  (BITMAP *bm, long x0, long y0, long xe, long ye);
  void (*dcirc)  (BITMAP *bm, long x0, long y0, long r);
  void (*dpcirc) (BITMAP *bm, long x0, long y0, long r);
  void (*poly)   (BITMAP *bm, long npts, long *pts);
  void (*dpoly)  (BITMAP *bm, long npts, long *pts);
  void (*ppoly)  (BITMAP *bm, long npts, long *pts);
  void (*dppoly) (BITMAP *bm, long npts, long *pts);
  BITMAP * (*getblock) (BITMAP *bm0, short x0, short y0, short width, short height);
  void     (*putblock) (BITMAP *bm0, short x0, short y0, short width, short height, BITMAP *bm1, short x1, short y1);
} SCREEN;
