/*
 * server/main.c, part of W
 * (C) 94-02/96 by Torsten Scherer (TeSche)
 * itschere@techfak.uni-bielefeld.de
 */

#include <sys/types.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <sys/time.h>
#include <unistd.h>
#include <signal.h>
#include <stdlib.h>
#include <sys/stat.h>
#include <string.h>
#include <sys/ioctl.h>
#include <termios.h>
#include <fcntl.h>
#ifdef __MINT__
#include <support.h>
#endif
#ifdef linux
#include <linux/kd.h>
#endif

#include "wserver.h"
#include "window.h"
#include "rect.h"

#ifdef __MINT__
long _stksize = 32000;
#endif


/*
 *	some constants
 */

#define	EFILNF		-33
#define	FLAG_NONE	0x0
#define	FLAG_EXITED	0x1


/*
 * global variables
 */

WINDOW *glob_leftmousepressed = NULL;
WINDOW *glob_rightmousepressed = NULL;
struct termios glob_termios;
long glob_savetime, glob_issaving = 0;
short glob_ineth, glob_unixh;
fd_set glob_crfd;
CLIENT *glob_clients = NULL, *glob_saveclient = 0;
long glob_bytes = 0, glob_pakets = 0;
MOUSE glob_mouse = {0, 0, 0, 0, 0, 0, 0};
SCREEN *glob_screen;
ushort DefaultPattern[16] = {
  0x5555, 0xAAAA, 0x5555, 0xAAAA, 0x5555, 0xAAAA, 0x5555, 0xAAAA,
  0x5555, 0xAAAA, 0x5555, 0xAAAA, 0x5555, 0xAAAA, 0x5555, 0xAAAA
};
short glob_uid;
short glob_pakettype;


/*
 * private variables
 */

static short is_terminating = 0;


/*
 * a two step terminator...
 */

void terminate(int sig, char *msg)
{
  CLIENT *cptr;
  EVENTP paket;

  mysignal(sig, SIG_IGN);

  /*
   * shall we just prepare exiting?
   */

  if (!is_terminating && !sig) {

    is_terminating = 1;

    /* this will tell all clients to terminate */

    cptr = glob_clients;
    while (cptr) {
      paket.len = sizeof(EVENTP);
      paket.type = PAK_EVENT;
      paket.event.type = EVENT_GADGET;
      paket.event.key = GADGET_EXIT;
      write(cptr->sh, &paket, sizeof(EVENTP));
      cptr = cptr->next;
    }

    mysignal(SIGALRM, sigroutine);
    alarm(10);

    return;
  }

  /*
   * something more urgent occured (or al clients died)
   */

  clear_scr();
#ifdef linux
  ioctl(0, KDSETMODE, KD_TEXT);
#endif
  tcsetattr(0, TCSANOW, &glob_termios);
  cursor_on();

  if (sig > 0) {
    fprintf(stderr, "fatal: Wserver received signal #%i\n", sig);
  } else if (sig < 0) {
    fprintf(stderr, "fatal: Wserver terminated with message: %s\n", msg);
  }

  close(glob_unixh);
  unlink(UNIXNAME);

  exit(sig);
}


void sigroutine(int sig)
{
  terminate(sig, "signal");
#ifdef linux
  mysignal(sig, sigroutine);
#endif
}


/*
 * place a rectangle somewhere on the screen
 *
 * x0 and y0 should hold the previous position, if there was any, and 'UNDEF'
 * (-32768) if there wasn't any. the mouse will keep its relative position to
 * the rectangle so that if you click on an already existing window and release
 * the button mithout moving the mouse, the mousepointer won't move to the
 * upper left edge of the window. if you discard the move or the rectangle is
 * bigger than the screen both x0 and y0 will contain 'UNDEF' and something
 * != 0 is returned.
 */

int get_rectangle(short width, short height,
		  short *x0, short *y0,
		  char pressed, char released)
{
  WEVENT *ev = NULL;
  short mdx, mdy;
  short wx, wy, oldwx, oldwy;
  short end;
  static GCONTEXT gc = {M_INVERS, 0, NULL, NULL};

#ifdef CHILDS_INSIDE_PARENT
  if ((width > glob_screen->bm.width) || (height > glob_screen->bm.height)) {
    *x0 = UNDEF;
    *y0 = UNDEF;
    return -1;
  }
#endif

  if ((*x0 == UNDEF) || (*y0 == UNDEF)) {
    wx = glob_mouse.rx;
    wy = glob_mouse.ry;
  } else {
    wx = *x0;
    wy = *y0;
  }

#ifdef CHILDS_INSIDE_PARENTS
  if (wx + width > glob_screen->bm.width)
    wx = glob_screen->bm.width - width;
  if (wy + height > glob_screen->bm.height)
    wy = glob_screen->bm.height - height;
#endif

  mdx = glob_mouse.rx - wx;
  mdy = glob_mouse.ry - wy;

  mouse_hide();

  gc0 = &gc;
  clip0 = &glob_rootwindow->pos;

  (*glob_screen->box)(&glob_screen->bm, wx, wy, width, height);
  (*glob_screen->box)(&glob_screen->bm, wx+1, wy+1, width-2, height-2);

  end = 0;
  while (!end)
    if (get_eventmask(EV_MOUSE, -1, NULL))
      if ((ev = event_mouse())) {

	oldwx = wx;
	oldwy = wy;

	if (ev->type == EVENT_MMOVE) {
#ifdef CHILDS_INSIDE_PARENTS
	  wx += ev->x;
	  if (wx < 0) {
	    wx = 0;
	  } else if (wx + width > glob_screen->bm.width) {
	    wx = glob_screen->bm.width - width;
	  }
	  wy += ev->y;
	  if (wy < 0) {
	    wy = 0;
	  } else if (wy + height > glob_screen->bm.height) {
	    wy = glob_screen->bm.height - height;
	  }
#else
	  /*
	   * only make sure the mouse stays inside the screen
	   * NOTE: we must ensure (otherwise SIGSEGV):
	   *  0 <= mouse.x < glob_screen->bm.width
	   *  0 <= mouse.y < glob_screen->bm.height
	   */
	  wx += ev->x;
	  if (wx + mdx < 0) {
	    wx = -mdx;
	  } else if (wx + mdx >= glob_screen->bm.width) {
	    wx = glob_screen->bm.width - mdx - 1;
	  }
	  wy += ev->y;
	  if (wy + mdy < 0) {
	    wy = -mdy;
	  } else if (wy + mdy >= glob_screen->bm.height) {
	    wy = glob_screen->bm.height - mdy - 1;
	  }
#endif
	} else {
	  if ((ev->reserved[0] & pressed) || (ev->reserved[1] & released)) {
	    end = 1;
	  }
	}

	if ((wx != oldwx) || (wy != oldwy)) {
	  (*glob_screen->box)(&glob_screen->bm,
			      oldwx, oldwy, width, height);
	  (*glob_screen->box)(&glob_screen->bm,
			      oldwx+1, oldwy+1, width-2, height-2);
	  (*glob_screen->box)(&glob_screen->bm, wx, wy, width, height);
	  (*glob_screen->box)(&glob_screen->bm,
			      wx+1, wy+1, width-2, height-2);
	}
      }

  (*glob_screen->box)(&glob_screen->bm, wx, wy, width, height);
  (*glob_screen->box)(&glob_screen->bm, wx+1, wy+1, width-2, height-2);

  /* right button = discard window? */
  if (ev->reserved[0] & BUTTON_RIGHT) {
    *x0 = UNDEF;
    *y0 = UNDEF;
    return -1;
  }

  *x0 = wx;
  *y0 = wy;

  glob_mouse.rx = wx + mdx;
  glob_mouse.ry = wy + mdy;
  mouse_show();

  glob_leftmousepressed = NULL;
  glob_rightmousepressed = NULL;

  return 0;
}


/*****************************************************************************/

/*
 * some routines dealing with sockets
 */

CLIENT *createClient(void)
{
  CLIENT *ptr;

  if (!(ptr = (CLIENT *)malloc(sizeof(CLIENT)))) {
    return NULL;
  }

  memset(ptr, 0, sizeof(CLIENT));

  if (!(ptr->buf = malloc(LARGEBUF))) {
    free(ptr);
    return NULL;
  }

  return ptr;
}


void killClient(CLIENT *cptr)
{
  short i;
  CLIENT *tmp = cptr->next;

  /* shut down socket, close windows and remove client from queue */

  shutdown(cptr->sh, 2);
  close(cptr->sh);

  /* kill all windows for a particular client */
  windowKillClient(cptr);

  for (i=0; i<MAXCLIENTFONTS; i++) {
    if (cptr->font[i]) {
      client_unloadfont(cptr, i);
    }
  }

  /* also clear the fd_set bit */
  FD_CLR(cptr->sh, &glob_crfd);

  if (cptr->prev) {
    cptr->prev->next = tmp;
  } else {
    glob_clients = tmp;
  }

  if (tmp) {
    tmp->prev = cptr->prev;
  }

  if (cptr->bbm) {
    free(cptr->bbm->data);
    free(cptr->bbm);
  }

  if (glob_saveclient == cptr) {
    glob_issaving = 0;
    glob_saveclient = 0;
  }

  free(cptr->buf);
  free(cptr);
}


void process_paket(CLIENT *cptr, PAKET *pptr)
{
  PAKET *rpptr = NULL;
  short arg1, arg2;
  static PAKET rawpaket;
  static INITRETP irpaket;
  static SRETP srpaket;
  static S3RETP s3rpaket;
  static RSTATUSP rspaket;
  static LRETP lrpaket;
  static LFONTRETP lfontrpaket;

  switch (glob_pakettype = pptr->type) {

    case PAK_NULLR:
      srpaket.len = sizeof(SRETP);
      srpaket.type = htons(PAK_SRET);
      srpaket.ret = htons(-1);
      rpptr = (PAKET *)&srpaket;
      break;

    case PAK_INIT:
      cptr->uid = ntohs(((INITP *)pptr)->uid);
      irpaket.len = sizeof(INITRETP);
      irpaket.type = htons(PAK_INITRET);
      irpaket.vmaj = htons(_WMAJ);
      irpaket.vmin = htons(_WMIN);
      irpaket.width = htons(glob_screen->bm.width);
      irpaket.height = htons(glob_screen->bm.height);
      irpaket.colors = htons(2);
      rpptr = (PAKET *)&irpaket;
      break;

    case PAK_CREATE:
      srpaket.len = sizeof(SRETP);
      srpaket.type = htons(PAK_SRET);
      srpaket.ret =
	htons(client_create(cptr,
			    ntohs(((CREATEP *)pptr)->width),
			    ntohs(((CREATEP *)pptr)->height),
			    ntohs(((CREATEP *)pptr)->flags),
			    ((CREATEP *)pptr)->libPtr));
      rpptr = (PAKET *)&srpaket;
      break;

    case PAK_CREATE2:
      srpaket.len = sizeof(SRETP);
      srpaket.type = htons(PAK_SRET);
      srpaket.ret
	= htons(client_create2(cptr, ntohs(((CREATEP *)pptr)->width),
			       ntohs(((CREATEP *)pptr)->height),
			       ntohs(((CREATEP *)pptr)->flags),
			       ntohs(((CREATEP *)pptr)->handle),
			       ((CREATEP *)pptr)->libPtr));
      rpptr = (PAKET *)&srpaket;
      break;

    case PAK_OPEN:
      srpaket.len = sizeof(SRETP);
      srpaket.type = htons(PAK_SRET);
      srpaket.ret
	= htons(client_open(cptr, ntohs(((OPENP *)pptr)->handle),
			    ntohs(((OPENP *)pptr)->x0), ntohs(((OPENP *)pptr)->y0)));
      rpptr = (PAKET *)&srpaket;
      break;

    case PAK_MOVE:
      srpaket.len = sizeof(SRETP);
      srpaket.type = htons(PAK_SRET);
      srpaket.ret
	= htons(client_move(cptr, ntohs(((MOVEP *)pptr)->handle),
			    ntohs(((MOVEP *)pptr)->x0), ntohs(((MOVEP *)pptr)->y0)));
      rpptr = (PAKET *)&srpaket;
      break;

    case PAK_CLOSE:
      srpaket.len = sizeof(SRETP);
      srpaket.type = htons(PAK_SRET);
      srpaket.ret = htons(client_close(cptr, ntohs(((CLOSEP *)pptr)->handle)));
      rpptr = (PAKET *)&srpaket;
      break;

    case PAK_DELETE:
      srpaket.len = sizeof(SRETP);
      srpaket.type = htons(PAK_SRET);
      srpaket.ret = htons(client_delete(cptr, ntohs(((DELETEP *)pptr)->handle)));
      rpptr = (PAKET *)&srpaket;
      break;

    case PAK_LOADFONT:
      lfontrpaket.len = sizeof(LFONTRETP);
      lfontrpaket.type = htons(PAK_LFONTRET);
      client_loadfont(cptr, ((LOADFONTP *)pptr)->fontname, &lfontrpaket);
      rpptr = (PAKET *)&lfontrpaket;
      break;

    case PAK_UNLOADFONT:
      srpaket.len = sizeof(SRETP);
      srpaket.type = htons(PAK_SRET);
      srpaket.ret
	= htons(client_unloadfont(cptr, ntohs(((UNLOADFONTP *)pptr)->fonthandle)));
      rpptr = (PAKET *)&srpaket;
      break;

    case PAK_QWINSZ:
      s3rpaket.len = sizeof(S3RETP);
      s3rpaket.type = htons(PAK_S3RET);
      s3rpaket.ret[2]
	= htons(client_querywinsize(cptr, ntohs(((QWINSZP *)pptr)->handle),
				    ntohs(((QWINSZP *)pptr)->effective),
				    &arg1, &arg2));
      s3rpaket.ret[1] = htons(arg2);
      s3rpaket.ret[0] = htons(arg1);
      rpptr = (PAKET *)&s3rpaket;
      break;

    case PAK_TEST:
      srpaket.len = sizeof(SRETP);
      srpaket.type = htons(PAK_SRET);
      srpaket.ret
	= htons(client_test(cptr, htons(((TESTP *)pptr)->handle),
			    ntohs(((TESTP *)pptr)->x0), ntohs(((TESTP *)pptr)->y0)));
      rpptr = (PAKET *)&srpaket;
      break;

    case PAK_QMPOS:
      s3rpaket.len = sizeof(S3RETP);
      s3rpaket.type = htons(PAK_S3RET);
      s3rpaket.ret[2]
	= htons(client_querymousepos(cptr, ntohs(((QMPOSP *)pptr)->handle),
				     &arg1, &arg2));
      s3rpaket.ret[1] = htons(arg2);
      s3rpaket.ret[0] = htons(arg1);
      rpptr = (PAKET *)&s3rpaket;
      break;

    case PAK_QWPOS:
      s3rpaket.len = sizeof(S3RETP);
      s3rpaket.type = htons(PAK_S3RET);
      s3rpaket.ret[2]
	= htons(client_querywindowpos(cptr, ntohs(((QWPOSP *)pptr)->handle),
				      ntohs(((QWPOSP *)pptr)->effective),
				      &arg1, &arg2));
      s3rpaket.ret[1] = htons(arg2);
      s3rpaket.ret[0] = htons(arg1);
      rpptr = (PAKET *)&s3rpaket;
      break;

    case PAK_QSTATUS:
      rspaket.len = sizeof(RSTATUSP);
      rspaket.type = htons(PAK_RSTATUS);
      rspaket.ret
	= htons(client_status(cptr, glob_clients,
			      ntohs(((QSTATUSP *)pptr)->index), &rspaket.status));
      rpptr = (PAKET *)&rspaket;
      break;

    case PAK_PUTBLKREQ:
      srpaket.len = sizeof(SRETP);
      srpaket.type = htons(PAK_SRET);
      srpaket.ret
	= htons(client_putblockreq(cptr, ntohs(((PUTBLKREQP *)pptr)->rwidth),
				   ntohs(((PUTBLKREQP *)pptr)->rheight),
				   ntohs(((PUTBLKREQP *)pptr)->x0),
				   ntohs(((PUTBLKREQP *)pptr)->y0),
				   ntohs(((PUTBLKREQP *)pptr)->width),
				   ntohs(((PUTBLKREQP *)pptr)->height),
				   ntohs(((PUTBLKREQP *)pptr)->handle),
				   ntohs(((PUTBLKREQP *)pptr)->x1),
				   ntohs(((PUTBLKREQP *)pptr)->y1)));
      rpptr = (PAKET *)&srpaket;
      break;

    case PAK_SSAVER:
      srpaket.len = sizeof(SRETP);
      srpaket.type = htons(PAK_SRET);
      if ((glob_saveclient && (glob_saveclient != cptr)) ||
	  (ntohs(((SSAVERP *)pptr)->seconds) < 10)) {
	srpaket.ret = htons(-1);
      } else {
	glob_saveclient = cptr;
	glob_savetime = ntohs(((SSAVERP *)pptr)->seconds);
	glob_issaving = 0;
	srpaket.ret = htons(0);   /* boah... */
      }
      rpptr = (PAKET *)&srpaket;
      break;

    case PAK_GETBLKREQ:
      lrpaket.len = sizeof(LRETP);
      lrpaket.type = htons(PAK_LRET);
      lrpaket.ret
	= htons(client_getblockreq(cptr, ntohs(((GETBLKREQP *)pptr)->handle),
				   ntohs(((GETBLKREQP *)pptr)->x0),
				   ntohs(((GETBLKREQP *)pptr)->y0),
				   ntohs(((GETBLKREQP *)pptr)->width),
				   ntohs(((GETBLKREQP *)pptr)->height)));
      rpptr = (PAKET *)&lrpaket;
      break;

    case PAK_GETBLKDATA:
      rawpaket.len = sizeof(PAKET);
      rawpaket.type = htons(PAK_RAWDATA);
      client_getblockdata(cptr, &rawpaket);
      rpptr = (PAKET *)&rawpaket;
      break;

/* pakets that don't need a return code */

    case PAK_NULL:
      break;

    case PAK_EXIT:
      cptr->flags |= FLAG_EXITED;
      break;

    case PAK_STEXTSTYLE:
      client_settextstyle(cptr, ntohs(((STEXTSTYLEP *)pptr)->handle),
			  ntohs(((STEXTSTYLEP *)pptr)->flags));
      break;

    case PAK_STITLE:
      client_settitle(cptr, ntohs(((STITLEP *)pptr)->handle),
		      ((STITLEP *)pptr)->title);
      break;

    case PAK_PLOT:
    case PAK_DPLOT:
      client_plot(cptr, ntohs(((PLOTP *)pptr)->handle),
		  ntohs(((PLOTP *)pptr)->x0), ntohs(((PLOTP *)pptr)->y0));
      break;

    case PAK_LINE:
    case PAK_DLINE:
      client_line(cptr, ntohs(((LINEP *)pptr)->handle),
		  ntohs(((LINEP *)pptr)->x0), ntohs(((LINEP *)pptr)->y0),
		  ntohs(((LINEP *)pptr)->xe), ntohs(((LINEP *)pptr)->ye));
      break;

    case PAK_HLINE:
    case PAK_DHLINE:
      client_hline(cptr, ntohs(((HVLINEP *)pptr)->handle),
		   ntohs(((HVLINEP *)pptr)->x0), ntohs(((HVLINEP *)pptr)->y0),
		   ntohs(((HVLINEP *)pptr)->e));
      break;

    case PAK_VLINE:
    case PAK_DVLINE:
      client_vline(cptr, ntohs(((HVLINEP *)pptr)->handle),
		   ntohs(((HVLINEP *)pptr)->x0), ntohs(((HVLINEP *)pptr)->y0),
		   ntohs(((HVLINEP *)pptr)->e));
      break;

    case PAK_BOX:
    case PAK_PBOX:
    case PAK_DBOX:
    case PAK_DPBOX:
      client_box(cptr, ntohs(((BOXP *)pptr)->handle),
		 ntohs(((BOXP *)pptr)->x0), ntohs(((BOXP *)pptr)->y0),
		 ntohs(((BOXP *)pptr)->width), ntohs(((BOXP *)pptr)->height));
      break;

    case PAK_BITBLK:
      client_bitblk(cptr, ntohs(((BITBLKP *)pptr)->handle),
		    ntohs(((BITBLKP *)pptr)->x0), ntohs(((BITBLKP *)pptr)->y0),
		    ntohs(((BITBLKP *)pptr)->width),
		    ntohs(((BITBLKP *)pptr)->height),
		    ntohs(((BITBLKP *)pptr)->x1), ntohs(((BITBLKP *)pptr)->y1));
      break;

    case PAK_BITBLK2:
      client_bitblk2(cptr, ntohs(((BITBLKP *)pptr)->handle),
		     ntohs(((BITBLKP *)pptr)->x0),
		     ntohs(((BITBLKP *)pptr)->y0),
		     ntohs(((BITBLKP *)pptr)->width),
		     ntohs(((BITBLKP *)pptr)->height),
		     ntohs(((BITBLKP *)pptr)->dhandle),
		     ntohs(((BITBLKP *)pptr)->x1),
		     ntohs(((BITBLKP *)pptr)->y1));
      break;

    case PAK_VSCROLL:
      client_vscroll(cptr, ntohs(((VSCROLLP *)pptr)->handle),
		     ntohs(((VSCROLLP *)pptr)->x0),
		     ntohs(((VSCROLLP *)pptr)->y0),
		     ntohs(((VSCROLLP *)pptr)->width),
		     ntohs(((VSCROLLP *)pptr)->height),
		     ntohs(((VSCROLLP *)pptr)->y1));
      break;

    case PAK_PRINTC:
      client_printc(cptr, ntohs(((PRINTCP *)pptr)->handle),
		    ntohs(((PRINTCP *)pptr)->x0),
		    ntohs(((PRINTCP *)pptr)->y0),
		    (ushort)ntohs(((PRINTCP *)pptr)->c));
      break;

    case PAK_PRINTS:
      client_prints(cptr, ntohs(((PRINTSP *)pptr)->handle),
		    ntohs(((PRINTSP *)pptr)->x0),
		    ntohs(((PRINTSP *)pptr)->y0),
		    ((PRINTSP *)pptr)->s);
      break;

    case PAK_CIRCLE:
    case PAK_PCIRCLE:
    case PAK_DCIRCLE:
    case PAK_DPCIRCLE:
      client_circle(cptr, ntohs(((CIRCLEP *)pptr)->handle),
		    ntohs(((CIRCLEP *)pptr)->x0),
		    ntohs(((CIRCLEP *)pptr)->y0),
		    ntohs(((CIRCLEP *)pptr)->r));
      break;

    case PAK_POLY:
    case PAK_PPOLY:
    case PAK_DPOLY:
    case PAK_DPPOLY:
      client_poly(cptr, ntohs(((POLYP *)pptr)->handle),
		  ntohs(((POLYP *)pptr)->numpoints), ((POLYP *)pptr)->points);
      break;

    case PAK_RAWDATA:
      client_putblockdata(cptr, pptr);
      break;

    case PAK_BEEP:
      client_beep(cptr);
      break;

    case PAK_SMODE:
      client_setmode(cptr, ntohs(((SMODEP *)pptr)->handle),
		     ntohs(((SMODEP *)pptr)->mode));
      break;

    case PAK_SFONT:
      client_setfont(cptr, ntohs(((SFONTP *)pptr)->handle),
		     ntohs(((SFONTP *)pptr)->fonthandle));
      break;

    case PAK_SPATTERN:
      client_setpattern(cptr, ntohs(((SPATTERNP *)pptr)->handle),
			ntohs(((SPATTERNP *)pptr)->pattern));
      break;

    default:
      fprintf(stderr, "wserver: received unknown paket from client 0x%08x, killing client\n", (unsigned int)cptr);
      cptr->flags |= FLAG_EXITED;
      cptr->inbuf = 0;
  }

  if (rpptr) {
    int len = rpptr->len;
    /* we don't care if the connection is broken */
    rpptr->len = htons(rpptr->len);
    write(cptr->sh, rpptr, len);
  }
}


/*****************************************************************************/

/*
 * some routines dealing with events, the first one just reports a bitmask of
 * what events are ready to be processed and the others actually process them.
 */

#ifdef __MINT__
#define	MBUFSIZE 381
#else
#define	MBUFSIZE 3
#endif

short mevents = 0;
char mbuf[MBUFSIZE], *mptr;

long get_eventmask(long wanted, long timeout, fd_set *retrfd)
{
  fd_set rfd;
  struct timeval tv, *tvp;
  int rfds;
  long events = 0;

  do {

    /* set up file descriptors */
    if (wanted & EV_CLIENT) {
      rfd = glob_crfd;
    } else {
      FD_ZERO(&rfd);
    }
    if (wanted & EV_UCONN) {
      FD_SET(glob_unixh, &rfd);
    }
#ifndef AF_UNIX_ONLY
    if ((wanted & EV_ICONN) && (glob_ineth >= 0)) {
      FD_SET(glob_ineth, &rfd);
    }
#endif
    if (wanted & EV_KEYS) {
      FD_SET(glob_kbd, &rfd);
    }
    if (wanted & EV_MOUSE) {
      if (mevents) {
	events |= EV_MOUSE;
	timeout = 0;
      } else {
	FD_SET(glob_mouse.fh, &rfd);
      }
    }

    /* set up timeout */
    if (timeout < 0) {
      tvp = NULL;
    } else {
      tv.tv_sec = timeout / 1000;
      tv.tv_usec = timeout - 1000 * tv.tv_sec;
      tvp = &tv;
    }

    /* select the descriptors */
    if ((rfds = select(FD_SETSIZE, &rfd, NULL, NULL, tvp)) < 0) {
      /* experience has shown that it is not save
       * to test the fd_sets after an error
       */
      return 0;
    }

    /* what have we got? */
    if (FD_ISSET(glob_mouse.fh, &rfd)) {
      long ret;
      if ((ret = read(glob_mouse.fh, mbuf, MBUFSIZE)) > 0) {
	events |= EV_MOUSE;
	rfds--;
	mptr = mbuf;
	mevents = ret / 3;
      }
    }
    if (FD_ISSET(glob_kbd, &rfd)) {
      events |= EV_KEYS;
      rfds--;
    }
    if (FD_ISSET(glob_unixh, &rfd)) {
      events |= EV_UCONN;
      rfds--;
    }
#ifndef AF_UNIX_ONLY
    if (glob_ineth >= 0) if (FD_ISSET(glob_ineth, &rfd)) {
      events |= EV_ICONN;
      rfds--;
    }
#endif
    if (rfds) {
      /* what remains? */
      events |= EV_CLIENT;
      if (retrfd) {
	*retrfd = rfd;
      }
    }

  } while (!timeout && !events);

  return events;
}


void event_connect(short sokh)
{
  int rlen;
  struct sockaddr_in raddr;
  short shtmp;
  CLIENT *cptr;

  /* it seems to be save to expect these to stay empty for AF_UNIX accepts()
   */
  raddr.sin_addr.s_addr = 0;

  rlen = sizeof(struct sockaddr);
  if ((shtmp = accept(sokh, (struct sockaddr *)&raddr, &rlen)) < 0) {
    /* oops, I ran out of file handles? */
    return;
  }

  if ((cptr = createClient())) {
    /* ok, we can do it */
    cptr->sh = shtmp;
    cptr->raddr = raddr.sin_addr.s_addr;
    FD_SET(shtmp, &glob_crfd);
    cptr->prev = NULL;
    cptr->next = glob_clients;
    if (glob_clients) {
      glob_clients->prev = cptr;
    }
    glob_clients = cptr;
  } else {
    /* shut it down immediately */
    shutdown(shtmp, 2);
    close(shtmp);
  }
}


void event_client(fd_set *rfd)
{
  CLIENT *next, *cptr = glob_clients;
  int len;
  PAKET *pptr;
  long ret, offset;

  while (cptr) {
    /* keep a local copy of next client, killclient might trash cptr */
    next = cptr->next;

    if (FD_ISSET(cptr->sh, rfd)) {

      if ((ret = read(cptr->sh, cptr->buf + cptr->inbuf,
		      LARGEBUF - cptr->inbuf)) < 1) {
	cptr->flags |= FLAG_EXITED;
      } else {
	offset = 0;

	/* try to extract pakets */
	glob_bytes += ret;
	cptr->inbuf += ret;
	pptr = (PAKET *)cptr->buf;

	while ((cptr->inbuf >= 4) && (cptr->inbuf >= (len = ntohs(pptr->len)))) {
	  glob_pakets++;
	  cptr->pakets++;
	  cptr->bytes += len;
	  cptr->inbuf -= len;
	  pptr->len = len;
	  pptr->type = ntohs(pptr->type);
	  process_paket(cptr, pptr);
	  offset += len;
	  pptr = (PAKET *)(((char *)pptr) + len);
	}

	/* move remaining data down to beginning of buffer */
	if (offset && cptr->inbuf) {
	  bcopy(cptr->buf + offset, cptr->buf, cptr->inbuf);
	}
      }
    }

    if (cptr->flags & FLAG_EXITED) {
      killClient(cptr);
    }

    cptr = next;
  }
}


/*
 * normally this routine assumes that a previous check has shown that mouse
 * data is available, but perhaps it's better to be a bit carefull.
 *
 * other than that, we still lack a way to simultainously report mouse
 * movement *and* button presses which is why the stuff here tries to treat
 * them seperately. perhaps I should really implement a higher level buffer
 * of mouse events...
 */

WEVENT *event_mouse(void)
{
  short dx, dy, button;
  static char lastbutton = 7;
  static WEVENT someevent;

  dx = 0;
  dy = 0;

  /* first read movement only data */

  while (mevents && ((*mptr++ & 7) == lastbutton)) {
    dx += *mptr++;
    dy -= *mptr++;
    mevents--;
  }

  if (mevents)
    mptr--;   /* adjust pointer */

  if (dx || dy) {
    /* the mouse moved */
    someevent.type = EVENT_MMOVE;
    someevent.x = dx;
    someevent.y = dy;

    return &someevent;
  }

  if (mevents && (*mptr != lastbutton)) {
    /* some button event occured */
    button = *mptr++ & 7;
    mptr++; mptr++;		/* ignore movement here */
    mevents--;

    someevent.type = EVENT_MPRESS; /* or EVENT_MRELEASE, we don't care here. */
    someevent.x = glob_mouse.rx;
    someevent.y = glob_mouse.ry;
    someevent.reserved[0] = lastbutton & ~button;
    someevent.reserved[1] = ~lastbutton & button;
    lastbutton = button;

    return &someevent;
  }

  /* obviously nothing happened */

  return NULL;
}


void event_mousemove(WEVENT *ev)
{
  short	oldx, oldy;

  oldx = glob_mouse.rx;
  oldy = glob_mouse.ry;

  glob_mouse.rx += ev->x;
  glob_mouse.ry += ev->y;

  if (glob_mouse.rx < 0)
    glob_mouse.rx = 0;
  if (glob_mouse.rx >= glob_screen->bm.width)
    glob_mouse.rx = glob_screen->bm.width - 1;
  if (glob_mouse.ry < 0)
    glob_mouse.ry = 0;
  if (glob_mouse.ry >= glob_screen->bm.height)
    glob_mouse.ry = glob_screen->bm.height - 1;

  /* move mouse, if necessary */
  if ((glob_mouse.rx != oldx) || (glob_mouse.ry != oldy)) {
    mouse_move();
  }

  /* watch out for active windows */
  w_changeActiveWindow();
}


void event_mousebutton(WEVENT *ev)
{
  short	x, y;
  EVENTP paket;
  WINDOW *win;

  paket.len = sizeof(EVENTP);
  paket.type = PAK_EVENT;

  /*
   * first deal with release events seperately as
   * they might go to different windows...
   */

  if (glob_leftmousepressed) if ((ev->reserved[1] & BUTTON_LEFT) &&
				 (glob_leftmousepressed != glob_backgroundwin)) {
    paket.event.type = EVENT_MRELEASE;
    paket.event.win = glob_leftmousepressed->libPtr;
    paket.event.key = BUTTON_LEFT;
    x = ev->x - glob_leftmousepressed->work.x0;
    y = ev->y - glob_leftmousepressed->work.y0;
    if ((x < 0) || (x >= glob_leftmousepressed->work.w) ||
	(y < 0) || (y >= glob_leftmousepressed->work.h)) {
      x = -1;
      y = -1;
    }
    paket.event.x = x;
    paket.event.y = y;
    if (write(glob_leftmousepressed->cptr->sh, &paket,
	      sizeof(EVENTP)) != sizeof(EVENTP)) {
      perror("Wserver: event_mousebutton(): write(#1)");
    }
    glob_leftmousepressed = NULL;
  }

  if (glob_rightmousepressed) if ((ev->reserved[1] & BUTTON_RIGHT) &&
				  (glob_rightmousepressed != glob_backgroundwin)) {
    paket.event.type = EVENT_MRELEASE;
    paket.event.win = glob_rightmousepressed->libPtr;
    paket.event.key = BUTTON_RIGHT;
    x = ev->x - glob_rightmousepressed->work.x0;
    y = ev->y - glob_rightmousepressed->work.y0;
    if ((x < 0) || (x >= glob_rightmousepressed->work.w) ||
	(y < 0) || (y >= glob_rightmousepressed->work.h)) {
      x = -1;
      y = -1;
    }
    paket.event.x = x;
    paket.event.y = y;
    if (write(glob_rightmousepressed->cptr->sh, &paket,
	      sizeof(EVENTP)) != sizeof(EVENTP)) {
      perror("Wserver: event_mousebutton(): write(#2)");
    }
    glob_rightmousepressed = NULL;
  }

  /*
   * can we already break here?
   */

  if (!ev->reserved[0]) {
    return;
  }

  /*
   * there was a `press' event, deal with it...
   */

  x = ev->x;
  y = ev->y;

  if ((win = window_find(x, y, 1)) != glob_backgroundwin) {

    /* action on window */

    x -= win->pos.x0;
    y -= win->pos.y0;
    ev->win = win->libPtr;

    if (rect_cont_point (&win->area[AREA_WORK], x, y)) {

      /* action in work area of window */

      if (ev->reserved[0] & BUTTON_LEFT)
	glob_leftmousepressed = win;
      if (ev->reserved[0] & BUTTON_RIGHT)
	glob_rightmousepressed = win;

      if (win->flags & EV_MOUSE) {

	/* we won't deal with the `middle' button emulation so far... */
	ev->key = ev->reserved[0] & (BUTTON_LEFT | BUTTON_RIGHT);
	ev->x -= win->work.x0;
	ev->y -= win->work.y0;
	paket.event = *ev;

	if (write(win->cptr->sh, &paket, sizeof(EVENTP)) != sizeof(EVENTP)) {
	  perror("Wserver: event_mousebutton(): write(#3)");
	}

      } else {

	if (ev->reserved[0] & BUTTON_LEFT)
	  glob_leftmousepressed = NULL;
	if (ev->reserved[0] & BUTTON_RIGHT)
	  glob_rightmousepressed = NULL;
      }

    } else if (rect_cont_point (&win->area[AREA_CLOSE], x, y)) {

      /* exit gadget clicked */

      ev->type = EVENT_GADGET;
      ev->key = GADGET_CLOSE;
      paket.event = *ev;

      if (write(win->cptr->sh, &paket, sizeof(EVENTP)) != sizeof(EVENTP)) {
	perror("Wserver: event_mousebutton(): write(#4)");
      }

      if (ev->reserved[0] & BUTTON_LEFT)
	glob_leftmousepressed = NULL;
      if (ev->reserved[0] & BUTTON_RIGHT)
	glob_rightmousepressed = NULL;

    } else if (rect_cont_point (&win->area[AREA_ICON], x, y)) {

      /* icon gadget clicked */

      ev->type = EVENT_GADGET;
      ev->key = GADGET_ICON;
      paket.event = *ev;

      if (write(win->cptr->sh, &paket, sizeof(EVENTP)) != sizeof(EVENTP)) {
	perror("Wserver: event_mousebutton(): write(#5)");
      }

      if (ev->reserved[0] & BUTTON_LEFT)
	glob_leftmousepressed = NULL;
      if (ev->reserved[0] & BUTTON_RIGHT)
	glob_rightmousepressed = NULL;

    } else {

      /* action on frame of window */

      if ((win->flags & W_MOVE) &&
	  (ev->reserved[0] & BUTTON_LEFT)) {
	x = win->pos.x0;
	y = win->pos.y0;
	if (!get_rectangle(win->pos.w, win->pos.h,
		      &x, &y, BUTTON_RIGHT, BUTTON_LEFT))
	  client_move(NULL, win->id, x, y);
      }

      if (!(win->flags & W_TOP) &&
	  (ev->reserved[0] & BUTTON_RIGHT)) {
	/* top or down the window */
	w_topDown(win);
      }
    }

  } else {

    /* action on background */

    if (ev->reserved[0] & BUTTON_LEFT) {
      menu_domenu();
    }
  }
}


/*
 * event_key(): send up to MAXKEYS of keyboard input to the active window
 */

#define MAXKEYS 8

void event_key(void)
{
  EVENTP paket[MAXKEYS];
  uchar	c[MAXKEYS];
  int max, i;

  if (ioctl(glob_kbd, FIONREAD, &max)) {
    /* how could this happen? */
    return;
  }
  if (max > MAXKEYS) {
    max = MAXKEYS;
  }
  if ((max = read(glob_kbd, c, max)) < 1) {
    /* how could this happen? */
    return;
  }

  if ((glob_activewindow == glob_backgroundwin) ||
      !(glob_activewindow->flags & EV_KEYS)) {
    /* want no keys */
    return;
  }

  for (i=0; i<max; i++) {
    paket[i].len = sizeof(EVENTP);
    paket[i].type = PAK_EVENT;
    paket[i].event.type = EVENT_KEY;
    paket[i].event.key = c[i];
    paket[i].event.win = glob_activewindow->libPtr;
  }

  write(glob_activewindow->cptr->sh, paket, max * sizeof(EVENTP));
}


/*****************************************************************************/

/*
 * guess what... :-)
 */

void mainloop(void)
{
  long events;
  fd_set rfd;
  WEVENT *ev;
  time_t now, last = time(0);

  while (42) {

    events = get_eventmask(EV_KEYS | EV_MOUSE | EV_UCONN | EV_ICONN | EV_CLIENT, 1000, &rfd);
    now = time(0);

/* check for new connection requests */
    if (events & EV_UCONN) {
      event_connect(glob_unixh);
    }
#ifndef AF_UNIX_ONLY
    if (events & EV_ICONN) {
      event_connect(glob_ineth);
    }
#endif

/* check for client pakets */
    if (events & EV_CLIENT) {
      event_client(&rfd);
    }

/* check for mouse events */
    if (events & EV_MOUSE) {
      last = now;
      if ((ev = event_mouse())) switch (ev->type) {
        case EVENT_MMOVE:
	  event_mousemove(ev);
	  break;
	case EVENT_MPRESS:
	  event_mousebutton(ev);
	  break;
      }
    }

/* update everything that may have been changed by the client or the mouse */
    windowRedrawAllIfDirty();

/* check for keys */
    if (events & EV_KEYS) {
      last = now;
      event_key();
    }

/* how about the saver? */
    if (glob_saveclient && !glob_issaving && (now - last > glob_savetime)) {
      EVENTP	paket;
      glob_issaving = 1;
      paket.len = sizeof(EVENTP);
      paket.type = PAK_EVENT;
      paket.event.type = EVENT_SAVEON;
      if (write(glob_saveclient->sh, &paket, sizeof(EVENTP))
	  != sizeof(EVENTP)) {
	perror("server: enable_ssaver: write()");
      }
    }

    if (glob_issaving && (now - last < glob_savetime)) {
      EVENTP	paket;
      glob_issaving = 0;
      paket.len = sizeof(EVENTP);
      paket.type = PAK_EVENT;
      paket.event.type = EVENT_SAVEOFF;
      if (write(glob_saveclient->sh, &paket, sizeof(EVENTP))
	  != sizeof(EVENTP)) {
	perror("server: disable_ssaver: write()");
      }
    }

/* re-enable the mouse cursor if necessary and possible */
#ifdef LAZYMOUSE
    if ((glob_mouse.rx != glob_mouse.dx) || (glob_mouse.ry != glob_mouse.dy))
#endif
      mouse_show();

/* complete exit wanted? */
    if (is_terminating && !glob_clients) {
      /* this won't return */
      terminate(0, "");
    }

/* deal with VT switches if necessary (Linux68k only) */
    if (glob_screen->vtswitch) {
      (glob_screen->vtswitch)();
    }
  }
}


/*
 * guess what... :)
 */

void main(void)
{
  mysignal(SIGTTOU, SIG_IGN);

  switch (initialize()) {
    case 0:
      break;
    case -1:
      unlink(UNIXNAME);
    default:
      exit(-1);
    }

#ifdef __MINT__
  /* we want the server to be as quick as possible */
  (void)Prenice(Pgetpid(), -50);
#endif

  /* aaaaaand... */
  mainloop();
}
