/*
 * server/init.c, part of W
 * (C) 94-01/96 by Torsten Scherer (TeSche)
 * itschere@techfak.uni-bielefeld.de
 */

#include <string.h>
#include <signal.h>
#include <unistd.h>
#include <stdlib.h>
#include <sys/wait.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netdb.h>
#include <sys/ioctl.h>
#include <fcntl.h>
#include <pwd.h>
#include <termios.h>
#ifdef __MINT__
# include <support.h>
#endif

#include "wserver.h"
#include "window.h"


/*
 *  a SIGCHLD routine, just to remove the zombies
 */

void sigchild(int arg)
{
  /* uk: ask for information about terminated children, until there is no more,
   * just for the case that several processes died at the same time.
   */
  long res;
#ifdef __MINT__
  union wait status;

  while ((res = waitpid(-1, (__WP)&status, WNOHANG)) > 0) {
#else
  int status;

  while ((res = waitpid(-1, &status, WNOHANG)) > 0) {
#endif
    /* do nothing so far */
  }
}


/*
 *
 */

void startup(struct passwd *pw)
{
  char fname[80];
  short fh;

  sprintf(fname, "%s/.wrc", pw->pw_dir);
  if (access(fname, X_OK)) {
    sprintf(fname, "%s/wrc", LIBDIR);
    if (access(fname, X_OK)) {
      return;
    }
  }

  switch (fork()) {

    case -1: /* error */
      fprintf(stderr, "Wserver (warning): Can't fork startup sequence\n");
      break;

    case 0: /* child */
      if ((fh = open("/dev/null", O_RDWR)) >= 0) {
	dup2(fh, 0);
	dup2(fh, 1);
	dup2(fh, 2);
	close(fh);
      } else {
	fprintf(stderr, "Wserver (warning): Startup sequence can't open /dev/null\n");
      }

      /* this shouldn't return */
      execl(pw->pw_shell, pw->pw_shell, fname, NULL);

      /* just to be sure the main program continues as wanted */
      _exit(-99);
      break;

    default: /* parent */
  }

  /* if this failed we can still go on */
}


/*
 *  how to init the sockets?
 */

short initunixsocket(void)
{
  short	sh;
  struct sockaddr unixladdr;

  if ((sh = socket(AF_UNIX, SOCK_STREAM, 0)) < 0) {
    perror("server: socket(AF_UNIX)");
    return -1;
  }

  unixladdr.sa_family = AF_UNIX;
  strcpy(unixladdr.sa_data, UNIXNAME);
  if (bind(sh, &unixladdr, sizeof(struct sockaddr))) {
    close(sh);
    perror("server: bind(AF_UNIX)");
    return -1;
  }

  if (listen(sh, MAXLISTENQUEUE)) {
    close(sh);
    perror("server: listen(AF_UNIX)");
    return -1;
  }

  return sh;
}


#ifndef AF_UNIX_ONLY

short initinetsocket(void)
{
  short	sh;
  struct sockaddr_in inetladdr;

  if ((sh = socket(AF_INET, SOCK_STREAM, 0)) < 0) {
    perror("server: socket(AF_INET)");
    return -1;
  }

  /* bind to anything which is local */

  inetladdr.sin_family = AF_INET;
  inetladdr.sin_addr.s_addr = INADDR_ANY;
  inetladdr.sin_port = htons(SERVERPORT);
  if (bind(sh, (struct sockaddr *)&inetladdr, sizeof(struct sockaddr))) {
    close(sh);
    perror("server: bind(AF_INET)");
    return -1;
  }

  if (listen(sh, MAXLISTENQUEUE)) {
    close(sh);
    perror("server: listen(AF_INET)");
    return -1;
  }

  return sh;
}

#endif


/*
 *  major init code
 */

short initialize(void)
{
  FILE *fp;
  char buf[256], *title, *cmd;
  char *ttyn, *ptr = NULL;
  short i;
  struct passwd *pw;
  struct termios mytermios;
  int have_exit_item = 0;

  mysignal(SIGHUP, sigroutine);
  mysignal(SIGQUIT, sigroutine);
  mysignal(SIGSTOP, SIG_IGN);
  mysignal(SIGTSTP, SIG_IGN);
  mysignal(SIGILL, sigroutine);
  mysignal(SIGCHLD, sigchild);
  mysignal(SIGPIPE, SIG_IGN);   /* can only happen if a client terminated, and we're going to deal with it somewhere else */
  mysignal(SIGTERM, sigroutine);
  mysignal(SIGINT, sigroutine);

  /*
   *  check for login name
   */

  if (!(pw = getpwuid(glob_uid = getuid()))) {
    fprintf(stderr, "error: couldn't resolve your login name\n");
    return -2;
  }

  /*
   *  check for terminal name
   */

  if (!(ttyn = ttyname(0))) {
    fprintf(stderr, "error: can't resolve tty name\n");
    return -2;
  }

  while (*ttyn) {
    if (*ttyn++ == '/') {
      ptr = ttyn;
    }
  }

  if (!ptr) {
    fprintf(stderr, "error: illegal tty name\n");
    return -2;
  }

#ifndef linux
  /* under linux things aren't that easy to decide... */

#ifndef __NetBSD__
  /* (Phx) nor under NetBSD, which is booted under /dev/ttyeX (where
     X depends on the graphics device which is used... */

  if (strcmp(ptr, "console")) {
    fprintf(stderr, "error: W cannot be started on %s\n", ptr);
    return -2;
  }
#endif
#endif

  /*
   *  check for sockets
   */

  if ((glob_unixh = initunixsocket()) < 0) {
    fprintf(stderr, "error: can't create local communication socket. if there isn't\n");
    fprintf(stderr, "       currently a Wserver running this is an error and you\n");
    fprintf(stderr, "       should retry it after erasing %s\n", UNIXNAME);
    return -2;
  }

  /* from here on we assume that no other Wserver is running */

#ifndef AF_UNIX_ONLY
  if ((glob_ineth = initinetsocket()) < 0) {
    /* this may fail due to missing network */
    fprintf(stderr, "warning: can't create any network communication socket, at least\n");
    fprintf(stderr, "         the loopback interface should be enabled\n");
  }
#endif

  /*
   *  some more initializations
   */

  if (font_init()) {
    fprintf(stderr, "error: can't load fonts\n");
    return -1;
  }

  if (kbdInit() < 0) {
    return -1;
  }

  if (mouse_init()) {
    fprintf(stderr, "error: can't find mouse, or it's busy\n");
    return -1;
  }

  cursor_off();

  if (!(glob_screen = screen_init())) {
    fprintf(stderr, "error: this screen is not supported\n");
    return -1;
  }

  /*
   * we shouldn't print anything after here
   */

  if (window_init() < 0) {
    exit(-1);
  }

  glob_mouse.rx = glob_screen->bm.width >> 1;
  glob_mouse.ry = glob_screen->bm.height >> 1;
  mouse_show();

  /* set terminal to raw mode */

  tcgetattr(0, &glob_termios);
  mytermios = glob_termios;
  makeraw (&mytermios);
  tcsetattr(0, TCSANOW, &mytermios);

  FD_ZERO(&glob_crfd);

  /*
   *  set up menu
   */

  menu.items = 0;
  menu.maxlen = 0;
  sprintf(buf, "%s/.wconfig", pw->pw_dir);
  if (!(fp = fopen(buf, "r"))) {
    sprintf(buf, "%s/wconfig", LIBDIR);
    fp = fopen(buf, "r");
  }
  if (fp) {
    while (ftgets(buf, 256, fp)) {
      if (!strncmp(buf, "menuitem=", 9)) {
	title = buf + 9;
	if ((cmd = strchr(title, ','))) {
	  *cmd++ = 0;
#ifndef REFRESH
	  if (!strcasecmp(cmd, "@refresh")) {
	    cmd = NULL;
	  }
#endif
	}
	if (cmd && (menu.items < (MAXITEMS-1))) {
	  if ((i = strlen(title)) > menu.maxlen) {
	    menu.maxlen = i;
	  }
	  strcpy(menu.item[menu.items].title, title);
	  strcpy(menu.item[menu.items].command, cmd);
	  if (!strcasecmp(menu.item[menu.items].command, "@exit"))
	    have_exit_item = 1;
	  menu.items++;
	}
      }
    }
    fclose(fp);
  }

  /* hmm, no menu defined? use default menu... */

  if (!menu.items) {
    menu.items = 5;
    menu.maxlen = 10;
    strcpy(menu.item[0].title, "top 6x6");
    strcpy(menu.item[0].command, "wterm - f wsys6x6.fnt top");
    strcpy(menu.item[1].title, "top 8x8");
    strcpy(menu.item[1].command, "wterm - f wsys8x8.fnt top");
    strcpy(menu.item[2].title, "wterm 6x6");
    strcpy(menu.item[2].command, "wterm - f wsys6x6.fnt");
    strcpy(menu.item[3].title, "wterm 8x8");
    strcpy(menu.item[3].command, "wterm - f wsys8x8.fnt");
    strcpy(menu.item[4].title, "wterm 8x16");
    strcpy(menu.item[4].command, "wterm - f wsys8x16.fnt");
  }

  /* if there is no exit item in the menu, add one... */
  if (!have_exit_item) {
    strcpy(menu.item[menu.items].title, "exit");
    strcpy(menu.item[menu.items].command, "@exit");
    if (menu.maxlen < 4) {
      menu.maxlen = 4;
    }
    menu.items++;
  }

  /*
   *  other stuff...
   */

  switch (fork()) {
    case -1:
      fprintf (stderr, "wserver: cannot execute copyright\r\n");
      break;
    case 0:
      execlp ("wcpyrgt", "wcpyrgt", NULL);
      fprintf (stderr, "wserver: cannot execute copyright\n");
      exit (1);
  }

  startup(pw);

  return 0;
}
