/*
 * server/graph/packed/line.c, part of W
 * (C) 94-02/96 by Torsten Scherer (TeSche)
 * itschere@techfak.uni-bielefeld.de
 *
 * routines dealing with lines for the Atari graphics driver
 *
 * CHANGES:
 *
 * W1R3:
 * - in the *line() functions scale delta, dx and dy by 2 to avoid
 *   losing precision and thus drawing wrong lines.
 *   NOTE: x>>1 != x/2 if x<0 and x is odd. ++kay, 1/96.
 * - draw lines concurrently from both ends, which is faster. ++eero, 1/96
 * - rewrote patterned *line() functions, added dline(), which is currently
 *   commented out due to a missing entry in the SCREEN structure.
 *   ++eero, 1/96
 * - fixed a bug in line() at three places. if (delta < 0) should be
 *   if (delta <= 0). ++kay 1/96
 * - replaced eero's line() and dline() with the old ones. drawing from both
 *   end's produces different pixel's for clipped lines :(. ++kay, 1/96
 * - added clipping. ++kay, 1/96
 * - fixed a nasty bug in dhline, `width' musn't be unsigned, TeSche 01/96
 * - fixed a patterning bug in dvline, ++kay, 2/96
 * - fixed a bug in dhline. short lines with non-aligned start were sometimes
 *   drawn one pixel to short, ++kay, 2/96
 */

#include <stdio.h>
#include "../../../lib/Wlib.h"
#include "../../config.h"
#include "../../types.h"
#include "../../pakets.h"
#include "../../proto.h"
#include "../gproto.h"
#include "../clip.h"
#include "packed.h"


/*
 * arbitrary line
 */

void FUNCTION(line)(bm, x0, y0, xe, ye)
     BITMAP *bm;
     register long x0;
     register long y0;
     register long xe;
     register long ye;
{
  register ushort *ptr, bit;
  register long	dx, yi, dy, delta, ywpl;
  long swap, ox0, oy0;

  /* we always draw from left to right */
  if (xe < x0) {
    swap = x0;
    x0 = xe;
    xe = swap;
    swap = y0;
    y0 = ye;
    ye = swap;
  }
  dx = xe - x0;	/* this is guaranteed to be positive, or zero */

  ox0 = x0;
  oy0 = y0;
  if (LINE_NEEDS_CLIPPING(x0, y0, xe, ye, clip0)) {
    if (CLIP_LINE (x0, y0, xe, ye, clip0)) {
      return;
    }
  }

#ifdef MONOCHROME
  ywpl = bm->upl << 1;
  ptr = ((ushort *)bm->data) + y0 * ywpl + (x0 >> 4);
#else
  ywpl = bm->upl;
  ptr = ((ushort *)bm->data) + y0 * ywpl + (x0 >> 4) * bm->planes;
#endif
  bit = 0x8000 >> (x0 & 15);

  yi = 1;
  if ((dy = ye - y0) < 0) {
    yi = -1;
    dy = -dy;
    ywpl = -ywpl;
  }

  /* draw the first point, there always really is one */
  switch (gc0->drawmode) {
    case M_CLEAR:
      *ptr &= ~bit;
      break;
    case M_DRAW:
      *ptr |= bit;
      break;
    case M_INVERS:
      *ptr ^= bit;
  }

  /*
   * scale delta, dy and dx by 2 so we avoid dividing and thus loosing
   * precision. kay.
   */

  switch (gc0->drawmode) {
    case M_CLEAR: /* clear a line */
    if (dx < dy) {
      /* abs (slope) > 1 */
      delta = clipped_line_delta (dy, dx, y0-oy0);
      dx += dx;
      dy += dy;
      while (y0 != ye) {
	y0 += yi;
	ptr += ywpl;
	delta -= dx;
	if (delta <= 0) {
	  delta += dy;
	  if (!(bit >>= 1)) {
#ifdef MONOCHROME
	    ptr++;
#else
	    ptr += bm->planes;
#endif
	    bit = 0x8000;
	  }
	}
	*ptr &= ~bit;
      }
    } else {
      /* abs(slope) <= 1 */
      delta = clipped_line_delta (dx, dy, x0-ox0);
      dx += dx;
      dy += dy;
      while (x0 != xe) {
	x0 += 1;
	if (!(bit >>= 1)) {
#ifdef MONOCHROME
	  ptr++;
#else
	  ptr += bm->planes;
#endif
	  bit = 0x8000;
	}
	delta -= dy;
	if (delta <= 0) {
	  delta += dx;
	  ptr += ywpl;
	}
	*ptr &= ~bit;
      }
    }
    break;

  case M_DRAW: /* draw a line */
    if (dx < dy) {
      /* abs(slope) > 1 */
      delta = clipped_line_delta (dy, dx, y0-oy0);
      dx += dx;
      dy += dy;
      while (y0 != ye) {
	y0 += yi;
	ptr += ywpl;
	delta -= dx;
	if (delta <= 0) {
	  delta += dy;
	  if (!(bit >>= 1)) {
#ifdef MONOCHROME
	    ptr++;
#else
	    ptr += bm->planes;
#endif
	    bit = 0x8000;
	  }
	}
	*ptr |= bit;
      }
    } else {
      /* abs(slope) <= 1 */
      delta = clipped_line_delta (dx, dy, x0-ox0);
      dx += dx;
      dy += dy;
      while (x0 != xe) {
	x0 += 1;
	if (!(bit >>= 1)) {
#ifdef MONOCHROME
	  ptr++;
#else
	  ptr += bm->planes;
#endif
	  bit = 0x8000;
	}
	delta -= dy;
	if (delta <= 0) {
	  delta += dx;
	  ptr += ywpl;
	}
	*ptr |= bit;
      }
    }
    break;

  case M_INVERS: /* invert a line */
    if (dx < dy) {
      /* abs(slope) > 1 */
      delta = clipped_line_delta (dy, dx, y0-oy0);
      dx += dx;
      dy += dy;
      while (y0 != ye) {
	y0 += yi;
	ptr += ywpl;
	delta -= dx;
	if (delta <= 0) {
	  delta += dy;
	  if (!(bit >>= 1)) {
#ifdef MONOCHROME
	    ptr++;
#else
	    ptr += bm->planes;
#endif
	    bit = 0x8000;
	  }
	}
	*ptr ^= bit;
      }
    } else {
      /* abs(slope) <= 1 */
      delta = clipped_line_delta (dx, dy, x0-ox0);
      dx += dx;
      dy += dy;
      while (x0 != xe) {
	x0 += 1;
	if (!(bit >>= 1)) {
#ifdef MONOCHROME
	  ptr++;
#else
	  ptr += bm->planes;
#endif
	  bit = 0x8000;
	}
	delta -= dy;
	if (delta <= 0) {
	  delta += dx;
	  ptr += ywpl;
	}
	*ptr ^= bit;
      }
    }
  }
}


/*
 * fast horizontal line
 */

void FUNCTION(hline)(bm, x0, y0, xe)
     BITMAP *bm;
     register long x0;
     register long y0;
     register long xe;
{
#ifdef MONOCHROME
  register ulong *ptr, bit;
#else
  register ushort *ptr, bit;
  register long planes = bm->planes;
#endif
  register long width, todo;
  long x;

  if (xe < x0) {
    x = x0;
    x0 = xe;
    xe = x;
  }

  if (CLIP_HLINE (x0, y0, xe, clip0)) {
    return;
  }

#ifdef MONOCHROME
  bit = (ulong)x0 & 31;
  ptr = ((ulong *)bm->data) + y0 * bm->upl + (x0 >> 5);
#else
  bit = (ushort)x0 & 15;
  ptr = ((ushort *)bm->data) + y0 * bm->upl + (x0 >> 4) * planes;
#endif
  width = xe - x0 + 1;

  switch(gc0->drawmode) {

    case M_CLEAR:
      while (width) {
#ifdef MONOCHROME
	if (!bit && (width > 31)) {
	  *ptr++ = 0x00000000;
	  width -= 32;
	} else if (!bit && (width > 15)) {
	  *(ushort *)ptr = 0x0000;
	  bit = 16;
	  width -= 16;
	} else if ((bit == 16) && (width > 15)) {
	  *(((ushort *)ptr)+1) = 0x0000;
	  bit = 0;
	  ptr++;
	  width -= 16;
	} else {
	  if ((todo = 32 - bit) > width) {
	    todo = width;
	  }
	  *ptr &= ~bfmask32[bit][todo-1];
	  if ((bit += todo) == 32) {
	    bit = 0;
	    ptr++;
	  }
	  width -= todo;
	}
#else
	if (!bit && (width > 15)) {
	  *ptr = 0x0000;
	  ptr += planes;
	  width -= 16;
	} else {
	  if ((todo = 16 - bit) > width) {
	    todo = width;
	  }
	  *ptr &= ~bfmask16[bit][todo-1];
	  if ((bit += todo) == 16) {
	    bit = 0;
	    ptr += planes;
	  }
	  width -= todo;
	}
#endif
      }
      break;

    case M_DRAW:
      while (width) {
#ifdef MONOCHROME
	if (!bit && (width > 31)) {
	  *ptr++ = 0xFFFFFFFF;
	  width -= 32;
	} else if (!bit && (width > 15)) {
	  *(ushort *)ptr = 0xFFFF;
	  bit = 16;
	  width -= 16;
	} else if ((bit == 16) && (width > 15)) {
	  *(((ushort *)ptr)+1) = 0xFFFF;
	  bit = 0;
	  ptr++;
	  width -= 16;
	} else {
	  if ((todo = 32 - bit) > width) {
	    todo = width;
	  }
	  *ptr |= bfmask32[bit][todo-1];
	  if ((bit += todo) == 32) {
	    bit = 0;
	    ptr++;
	  }
	  width -= todo;
	}
#else
	if (!bit && (width > 15)) {
	  *ptr = 0xFFFF;
	  ptr += planes;
	  width -= 16;
	} else {
	  if ((todo = 16 - bit) > width) {
	    todo = width;
	  }
	  *ptr |= bfmask16[bit][todo-1];
	  if ((bit += todo) == 16) {
	    bit = 0;
	    ptr += planes;
	  }
	  width -= todo;
	}
#endif
      }
      break;

    case M_INVERS:
      while (width) {
#ifdef MONOCHROME
	if (!bit && (width > 31)) {
	  *ptr++ ^= 0xFFFFFFFF;
	  width -= 32;
	} else if (!bit && (width > 15)) {
	  *(ushort *)ptr ^= 0xFFFF;
	  bit = 16;
	  width -= 16;
	} else if ((bit == 16) && (width > 15)) {
	  *(((ushort *)ptr)+1) ^= 0xFFFF;
	  bit = 0;
	  ptr++;
	  width -= 16;
	} else {
	  if ((todo = 32 - bit) > width) {
	    todo = width;
	  }
	  *ptr ^= bfmask32[bit][todo-1];
	  if ((bit += todo) == 32) {
	    bit = 0;
	    ptr++;
	  }
	  width -= todo;
	}
#else
	if (!bit && (width > 15)) {
	  *ptr ^= 0xFFFF;
	  ptr += planes;
	  width -= 16;
	} else {
	  if ((todo = 16 - bit) > width) {
	    todo = width;
	  }
	  *ptr ^= bfmask16[bit][todo-1];
	  if ((bit += todo) == 16) {
	    bit = 0;
	    ptr += planes;
	  }
	  width -= todo;
	}
#endif
    }
  }
}


/*
 * fast vertical line
 */

void FUNCTION(vline)(bm, x0, y0, ye)
     BITMAP *bm;
     long x0;
     long y0;
     long ye;
{
#ifdef MONOCHROME
  register ulong *ptr, bit;
#else
  register ushort *ptr, bit;
#endif
  register long height, upl;
  long y;

  if (ye < y0) {
    y = y0;
    y0 = ye;
    ye = y;
  }

  if (CLIP_VLINE (x0, y0, ye, clip0)) {
    return;
  }

  upl = bm->upl;
#ifdef MONOCHROME
  bit = 0x80000000 >> (x0 & 31);
  ptr = (ulong *)bm->data + y0 * upl + (x0 >> 5);
#else
  bit = 0x8000 >> (x0 & 15);
  ptr = (ushort *)bm->data + y0 * upl + (x0 >> 4) * bm->planes;
#endif
  height = ye - y0 + 1;

  switch(gc0->drawmode) {

    case M_CLEAR:
      while (--height >= 0) {
	*ptr &= ~bit;
	ptr += upl;
      }
      break;

    case M_DRAW:
      while (--height >= 0) {
	*ptr |= bit;
	ptr += upl;
      }
      break;

    case M_INVERS:
      while (--height >= 0) {
	*ptr ^= bit;
	ptr += upl;
      }
  }
}


/*
 * pattern / dash functions
 */


/*
 * patterned line
 */

void FUNCTION(dline)(bm, x0, y0, xe, ye)
     BITMAP *bm;
     register long x0;
     register long y0;
     register long xe;
     register long ye;
{
  register ushort *ptr, bit, *pattern = gc0->pattern;
  register long	dx, yi, dy, delta, ywpl;
  long swap, ox0, oy0;

  /* we always draw from left to right */
  if (xe < x0) {
    swap = x0;
    x0 = xe;
    xe = swap;
    swap = y0;
    y0 = ye;
    ye = swap;
  }
  dx = xe - x0;	/* this is guaranteed to be positive, or zero */

  oy0 = y0;
  ox0 = x0;
  if (LINE_NEEDS_CLIPPING (x0, y0, xe, ye, clip0)) {
    if (CLIP_LINE (x0, y0, xe, ye, clip0)) {
      return;
    }
  }

#ifdef MONOCHROME
  ywpl = bm->upl << 1;
  ptr = ((ushort *)bm->data) + y0 * ywpl + (x0 >> 4);
#else
  ywpl = bm->upl;
  ptr = ((ushort *)bm->data) + y0 * ywpl + (x0 >> 4) * bm->planes;
#endif
  bit = 0x8000 >> (x0 & 15);

  yi = 1;
  if ((dy = ye - y0) < 0) {
    yi = -1;
    dy = -dy;
    ywpl = -ywpl;
  }

  /* draw the first point, there always really is one */
  *ptr &= ~bit;
  *ptr |= (pattern[y0 & 15] & bit);

  /*
   * scale delta, dy and dx by 2 so we avoid dividing and thus loosing
   * precision. kay.
   */

  if (dx < dy) {
    /* abs (slope) > 1 */
    delta = clipped_line_delta (dy, dx, y0-oy0);
    dx += dx;
    dy += dy;
    while (y0 != ye) {
      y0 += yi;
      ptr += ywpl;
      delta -= dx;
      if (delta <= 0) {
        delta += dy;
        if (!(bit >>= 1)) {
#ifdef MONOCHROME
          ptr++;
#else
          ptr += bm->planes;
#endif
          bit = 0x8000;
        }
      }
      *ptr &= ~bit;
      *ptr |= (pattern[y0 & 15] & bit);
    }
  } else {
    /* abs(slope) <= 1 */
    delta = clipped_line_delta (dx, dy, x0-ox0);
    dx += dx;
    dy += dy;
    while (x0 != xe) {
      x0 += 1;
      if (!(bit >>= 1)) {
#ifdef MONOCHROME
        ptr++;
#else
        ptr += bm->planes;
#endif
        bit = 0x8000;
      }
      delta -= dy;
      if (delta <= 0) {
        delta += dx;
        ptr += ywpl;
      }
      *ptr &= ~bit;
      *ptr |= (pattern[y0 & 15] & bit);
    }
  }
}


/*
 * patterned vertical line
 */

void FUNCTION(dvline)(bm, x0, y0, ye)
     BITMAP *bm;
     long x0;
     register long y0;
     register long ye;
{
  register ushort *ptr, val, set_bit, clr_bit, *pattern = gc0->pattern;
  register long wpl;
  long y;

  if (ye < y0) {
    y  = y0;
    y0 = ye;
    ye = y;
  }

  if (CLIP_VLINE (x0, y0, ye, clip0)) {
    return;
  }

#ifdef MONOCHROME
  wpl = bm->upl << 1;
  ptr = ((ushort *)bm->data) + y0 * wpl + (x0 >> 4);
#else
  wpl = bm->upl;
  ptr = ((ushort *)bm->data) + y0 * wpl + (x0 >> 4) * bm->planes;
#endif

  set_bit = 0x8000 >> (x0 & 15);
  clr_bit = ~set_bit;

  while(y0 <= ye) {
    val = *ptr;
    val &= clr_bit;
    *ptr = val | (pattern[y0 & 15] & set_bit);
    ptr += wpl;
    y0++;
  }
}


/*
 * patterned horizontal line
 *
 * TeSche: if pattern were 32 bits wide this could again be speeded up
 */

void FUNCTION(dhline)(bm, x0, y0, xe)
     BITMAP *bm;
     long x0;
     long y0;
     long xe;
{
  register ushort *ptr, bits, patt;
  register short width, words;
#ifndef MONOCHROME
  register long planes = bm->planes;
#endif
  long x;

  if (xe < x0) {
    x = x0;
    x0 = xe;
    xe = x;
  }

  if (CLIP_HLINE (x0, y0, xe, clip0)) {
    return;
  }

#ifdef MONOCHROME
  ptr = ((ushort *)bm->data) + y0 * (bm->upl << 1) + (x0 >> 4);
#else
  ptr = ((ushort *)bm->data) + y0 * bm->upl + (x0 >> 4) * planes;
#endif

  width = xe - x0 + 1;		/* how much left to 'draw' */
  patt  = gc0->pattern[y0 & 15];
  bits  = (ushort) x0 & 15;

  /* non-aligned beginning */
  if (bits) {
    width -= (16 - bits);
    bits = 0xFFFF >> bits;
    /* whole line inside a short? */
    if (width < 0)
      bits &= 0xFFFF << (-width);

    *ptr &= ~bits;
#ifdef MONOCHROME
    *ptr++ |= (patt & bits);
#else
    *ptr |= (patt & bits);
    ptr += planes;
#endif

    if (width < 0)
      width = 0;
  }

  words = width >> 4;
  while (--words >= 0) {
#ifdef MONOCHROME
    *ptr++ = patt;
#else
    *ptr = patt;
    ptr += planes;
#endif
  }

  /* non-aligned end */
  if ((width &= 15) > 0) {
    bits = 0xFFFF >> width;
    *ptr &= bits;
    *ptr |= (patt & ~bits);
  }
}
