/*
 * server/graph/direct8.c, part of W
 * (C) 94-02/96 TeSche (Torsten Scherer)
 * itschere@techfak.uni-bielefeld.de
 *
 * driver for direct 8 bit color displays (CG6, VGA?)
 *
 * - only very straightforward routines, not very optimized...
 *
 * CHANGES:
 *
 * - removed the *box and *circle functions to use the generic ones instead.
 *   ++kay, 1/96
 * - added dplot, dline and changed the other d* functions to use the new
 *   patterns. ++kay, 1/96
 * - applied sign extension bug fixes to printchar/printstring. ++kay, 1/96
 * - scale delta, dx, dy by 2 in line() and dline(). ++kay, 1/96
 * - added clipping (untested). ++kay, 1/96
 * - lots of changes due to global context and clipping variables, TeSche 01/96
 * - optimized dhline & dvline, TeSche 01/96
 * - added real but slow clipping for printchar (untested), ++kay, 2/96
 * - included string.h for bcopy(), Phx 02/96
 */

#include <stdio.h>
#include <string.h>
#include "../../../lib/Wlib.h"
#include "../../config.h"
#include "../../types.h"
#include "../../pakets.h"
#include "../../proto.h"
#include "../gproto.h"
#include "../clip.h"
#include "direct8.h"


/*
 * mouse stuff
 */

static uchar direct8_mbg[16*16];

void direct8_mouseShow(void)
{
  register uchar *sptr, *ptr, *bptr = direct8_mbg;
  register ushort moval, mpval, *moptr = moutline, *mpptr = mpointer, bit;
  register int xcnt, x, y;

  sptr = ((uchar *)glob_screen->bm.data) +
    glob_mouse.dy * glob_screen->bm.upl +
      glob_mouse.dx;

  y = MIN(glob_screen->bm.height - glob_mouse.dy, 16);
  xcnt = MIN(glob_screen->bm.width - glob_mouse.dx, 16);

  while (y--) {
    bit = 0x8000;
    moval = *moptr++;
    mpval = *mpptr++;
    ptr = sptr;
    for (x=0; x<xcnt; x++) {
      *bptr++ = *ptr;
      if (moval & bit) {
	*ptr = 0x00;
      }
      if (mpval & bit) {
	*ptr = 0xff;
      }
      bit >>= 1;
      ptr++;
    }
    sptr += glob_screen->bm.upl;
  }
}


void direct8_mouseHide(void)
{
  register uchar *sptr, *ptr, *bptr = direct8_mbg;
  register int xcnt, x, y;

  sptr = ((uchar *)glob_screen->bm.data) +
    glob_mouse.dy * glob_screen->bm.upl +
      glob_mouse.dx;

  y = MIN(glob_screen->bm.height - glob_mouse.dy, 16);
  xcnt = MIN(glob_screen->bm.width - glob_mouse.dx, 16);

  while (y--) {
    ptr = sptr;
    for (x=0; x<xcnt; x++) {
      *ptr++ = *bptr++;
    }
    sptr += glob_screen->bm.upl;
  }
}


void direct8_mouseMove(void)
{
  direct8_mouseHide();
  glob_mouse.dx = glob_mouse.rx;
  glob_mouse.dy = glob_mouse.ry;
  direct8_mouseShow();
}


/*
 *
 */

void direct8_plot(register BITMAP *bm,
		  register long x0,
		  register long y0)
{
  register uchar *dst = ((uchar *)bm->data) + y0 * bm->upl + x0;

  if (CLIP_POINT (x0, y0, clip0)) {
    return;
  }

  switch (gc0->drawmode) {
    case M_CLEAR:
      *dst = 0x00;
      break;
    case M_DRAW:
      *dst = 0xff;
      break;
    case M_INVERS:
      *dst ^= 0xff;
      break;
  }
}


void direct8_dplot(register BITMAP *bm,
		   register long x0,
		   register long y0)
{
  register uchar *dst;
  ushort bit;

  if (CLIP_POINT (x0, y0, clip0)) {
    return;
  }

  bit = 0x8000 >> (x0 & 15);
  dst = ((uchar *)bm->data) + y0 * bm->upl + x0;
  if (gc0->pattern[y0 & 15] & bit)
    *dst = 0xff;
  else
    *dst = 0x00;
}


long direct8_test(register BITMAP *bm,
		  register long x0,
		  register long y0)
{
  if (CLIP_POINT (x0, y0, clip0)) {
    return -1;
  }

  return *(((uchar *)bm->data) + y0 * bm->upl + x0);
}


void direct8_line(register BITMAP *bm,
		  register long x0,
		  register long y0,
		  register long xe,
		  register long ye)
{
  register long dx, dy, delta, bpl, yi;
  register uchar *ptr;
  long swap, ox0, oy0;

  /* we always draw from left to right */
  if (xe < x0) {
    swap = x0;
    x0 = xe;
    xe = swap;
    swap = y0;
    y0 = ye;
    ye = swap;
  }

  dx = xe - x0;	/* this is guaranteed to be positive, or zero */

  yi = 1;
  bpl = bm->upl;
  if ((dy = ye-y0) < 0) {
    yi = -1;
    dy = -dy;
    bpl = -bpl;
  }

  ox0 = x0;
  oy0 = y0;
  if (LINE_NEEDS_CLIPPING(x0, y0, xe, ye, clip0)) {
    if (CLIP_LINE (x0, y0, xe, ye, clip0)) {
      return;
    }
  }

  ptr = ((uchar *)bm->data) + y0 * bm->upl + x0;

  /* draw the first point, there always really is one */
  switch (gc0->drawmode) {
    case M_CLEAR:
      *ptr = 0x00;
      break;
    case M_DRAW:
      *ptr = 0xff;
      break;
    case M_INVERS:
      *ptr ^= 0xff;
  }

  /*
   * scale delta, dx, dy by 2 to avoid inaccuracies when dividing by 2.
   * kay.
   */
  switch (gc0->drawmode) {
    case M_CLEAR: /* clear a line */
    if (dx < dy) {
      /* `steile' Linien */
      delta = clipped_line_delta (dy, dx, y0-oy0);
      dx += dx;
      dy += dy;
      while (y0 != ye) {
	y0 += yi;
	ptr += bpl;
	delta -= dx;
	if (delta <= 0) {
	  delta += dy;
	  ptr++;
	}
	*ptr = 0x00;
      }
    } else {
      /* `flat' lines */
      delta = clipped_line_delta (dx, dy, x0-ox0);
      dx += dx;
      dy += dy;
      while (x0 != xe) {
	x0++;
	ptr++;
	delta -= dy;
	if (delta <= 0) {
	  delta += dx;
	  ptr += bpl;
	}
	*ptr = 0x00;
      }
    }
    break;

    case M_DRAW: /* draw a line */
    if (dx < dy) {
      /* `steile' Linien */
      delta = clipped_line_delta (dy, dx, y0-oy0);
      dx += dx;
      dy += dy;
      while (y0 != ye) {
	y0 += yi;
	ptr += bpl;
	delta -= dx;
	if (delta <= 0) {
	  delta += dy;
	  ptr++;
	}
	*ptr = 0xff;
      }
    } else {
      /* `flat' lines */
      delta = clipped_line_delta (dx, dy, x0-ox0);
      dx += dx;
      dy += dy;
      while (x0 != xe) {
	x0++;
	ptr++;
	delta -= dy;
	if (delta <= 0) {
	  delta += dx;
	  ptr += bpl;
	}
	*ptr = 0xff;
      }
    }
    break;

    case M_INVERS: /* invert a line */
    if (dx < dy) {
      /* `steile' Linien */
      delta = clipped_line_delta (dy, dx, y0-oy0);
      dx += dx;
      dy += dy;
      while (y0 != ye) {
	y0 += yi;
	ptr += bpl;
	delta -= dx;
	if (delta <= 0) {
	  delta += dy;
	  ptr++;
	}
	*ptr ^= 0xff;
      }
    } else {
      /* `flat' lines */
      delta = clipped_line_delta (dx, dy, x0-ox0);
      dx += dx;
      dy += dy;
      while (x0 != xe) {
	x0++;
	ptr++;
	delta -= dy;
	if (delta <= 0) {
	  delta += dx;
	  ptr += bpl;
	}
	*ptr ^= 0xff;
      }
    }
  }
}


void direct8_hline(register BITMAP *bm,
		   register long x0,
		   register long y0,
		   register long xe)
{
  register long count;
  register uchar *dst;
  long swap;

  if (xe < x0) {
    swap = x0;
    x0 = xe;
    xe = swap;
  }

  if (CLIP_HLINE (x0, y0, xe, clip0)) {
    return;
  }

  count = xe - x0 + 1;
  dst = ((uchar *)bm->data) + y0 * bm->upl + x0;

  switch (gc0->drawmode) {

    case M_CLEAR:
      while (((unsigned long)dst & 3) && count) {
	*dst++ = 0x00;
	count--;
      }
      while (count & ~3) {
	*((unsigned long *)dst)++ = 0x00000000;
	count -= 4;
      }
      while (count--) {
	*dst++ = 0x00;
      }
      break;

    case M_DRAW:
      while (count && ((unsigned long)dst & 3)) {
	*dst++ = 0xff;
	count--;
      }
      while (count & ~3) {
	*((unsigned long *)dst)++ = 0xffffffff;
	count -= 4;
      }
      while (count--) {
	*dst++ = 0xff;
      }
      break;

    case M_INVERS:
      while (count && ((unsigned long)dst & 3)) {
	*dst++ ^= 0xff;
	count--;
      }
      while (count & ~3) {
	*((unsigned long *)dst)++ ^= 0xffffffff;
	count -= 4;
      }
      while (count--) {
	*dst++ ^= 0xff;
      }
  }
}


void direct8_vline(register BITMAP *bm,
		   register long x0,
		   register long y0,
		   register long ye)
{
  register long count, bpl = bm->upl;
  register uchar *dst;
  long swap;

  if (ye < y0) {
    swap = y0;
    y0 = ye;
    ye = swap;
  }

  if (CLIP_VLINE (x0, y0, ye, clip0)) {
    return;
  }

  count = ye - y0 + 1;
  dst = (uchar *)bm->data + y0 * bpl + x0;

  switch (gc0->drawmode) {

    case M_CLEAR:
      while (count--) {
	*dst = 0x00;
	dst += bpl;
      }
      break;

    case M_DRAW:
      while (count--) {
	*dst = 0xff;
	dst += bpl;
      }
      break;

    case M_INVERS:
      while (count--) {
	*dst ^= 0xff;
	dst += bpl;
      }
  }
}


void direct8_dline(register BITMAP *bm,
		   register long x0,
		   register long y0,
		   register long xe,
		   register long ye)
{
  register long dx, dy, delta, bpl, yi;
  register uchar *ptr;
  register ushort *pattern = gc0->pattern;
  long swap, ox0, oy0;

  /* we always draw from left to right */
  if (xe < x0) {
    swap = x0;
    x0 = xe;
    xe = swap;
    swap = y0;
    y0 = ye;
    ye = swap;
  }
  dx = xe - x0;	/* this is guaranteed to be positive, or zero */

  bpl = bm->upl;
  yi = 1;
  if ((dy = ye-y0) < 0) {
    yi = -1;
    dy = -dy;
    bpl = -bpl;
  }

  oy0 = y0;
  ox0 = x0;
  if (LINE_NEEDS_CLIPPING (x0, y0, xe, ye, clip0)) {
    if (CLIP_LINE (x0, y0, xe, ye, clip0)) {
      return;
    }
  }

  ptr = ((uchar *)bm->data) + y0 * bm->upl + x0;

  /* draw the first point, there always really is one */
  if (pattern[y0 & 15] & (0x8000 >> (x0 & 15)))
    *ptr = 0xff;
  else
    *ptr = 0x00;

  /*
   * scale delta, dx, dy by 2 to avoid inaccuracies when dividing by 2.
   * kay.
   */
    if (dx < dy) {
      /* `steile' Linien */
      delta = clipped_line_delta (dy, dx, y0-oy0);
      dx += dx;
      dy += dy;
      while (y0 != ye) {
	y0 += yi;
	ptr += bpl;
	delta -= dx;
	if (delta <= 0) {
	  delta += dy;
	  ptr++;
	}
	if (pattern[y0 & 15] & (0x8000 >> (x0 & 15)))
	  *ptr = 0xff;
	else
	  *ptr = 0x00;
      }
    } else {
      /* `flat' lines */
      delta = clipped_line_delta (dx, dy, x0-ox0);
      dx += dx;
      dy += dy;
      while (x0 != xe) {
	x0++;
	ptr++;
	delta -= dy;
	if (delta <= 0) {
	  delta += dx;
	  ptr += bpl;
	}
	if (pattern[y0 & 15] & (0x8000 >> (x0 & 15)))
	  *ptr = 0xff;
	else
	  *ptr = 0x00;
      }
    }
}


void direct8_dvline(BITMAP *bm,
		    long x0,
		    register long y0,
		    register long ye)
{
  register long bpl = bm->upl;
  register uchar *dst;
  register ushort bit, *pattern = gc0->pattern;
  long swap;

  if (ye < y0) {
    swap = y0;
    y0 = ye;
    ye = swap;
  }

  if (CLIP_VLINE (x0, y0, ye, clip0)) {
    return;
  }

  dst = (uchar *)bm->data + y0 * bpl + x0;
  bit = 0x8000 >> (x0 & 15);

  for ( ; y0 <= ye; ++y0, dst += bpl) {
    if (pattern[y0 & 15] & bit)
      *dst = 0xff;
    else
      *dst = 0x00;
  }
}


void direct8_dhline(BITMAP *bm,
		    register long x0,
		    long y0,
		    register long xe)
{
  register uchar *ptr;
  register ushort bit, pat;
  long	x;

  if (xe < x0) {
    x = x0;
    x0 = xe;
    xe = x;
  }

  if (CLIP_HLINE (x0, y0, xe, clip0)) {
    return;
  }

  ptr = ((uchar *)bm->data) + y0 * bm->upl + x0;
  bit = 0x8000 >> (x0 & 15);
  pat = gc0->pattern[y0 & 15];

  for ( ; x0 <= xe; ++x0) {
    if (pat & bit) {
      *ptr++ = 0xff;
    } else {
      *ptr++ = 0x00;
    }
    if (!(bit >>= 1)) {
      bit = 0x8000;
    }
  }
}


void direct8_bitblk(register BITMAP *bm,
		    register long x0,
		    register long y0,
		    register long width,
		    register long height,
		    register BITMAP *bm1,
		    register long x1,
		    register long y1)
{
  register uchar *sptr, *dptr;
  register long sbpl, dbpl;

  if (CLIP_BITBLIT (x0, y0, width, height, x1, y1, clip0, clip1)) {
    return;
  }

  sbpl = bm->upl;
  dbpl = bm1->upl;

  /* a by far not optimized check... */

  if (y1 < y0) {

    sptr = ((uchar *)bm->data) + y0 * sbpl + x0;
    dptr = ((uchar *)bm1->data) + y1 * dbpl + x1;

    while (height--) {
      bcopy(sptr, dptr, width);
      sptr += sbpl;
      dptr += dbpl;
    }

  } else {

    sptr = ((uchar *)bm->data) + (y0 + height - 1) * sbpl + x0;
    dptr = ((uchar *)bm1->data) + (y1 + height - 1) * dbpl + x1;

    while (height--) {
      bcopy(sptr, dptr, width);
      sptr -= sbpl;
      dptr -= dbpl;
    }
  }
}


void direct8_scroll(register BITMAP *bm,
		    register long x0,
		    register long y0,
		    register long width,
		    register long height,
		    register long y1)
{
  register uchar *sptr, *dptr;
#if 1
  register uchar *slptr, *dlptr;
  register long xcount;
#endif
  register long bpl;
  long x1 = x0;

  if (y0 == y1) {
    return;
  }

  if (CLIP_BITBLIT (x0, y0, width, height, x1, y1, clip0, clip0)) {
    return;
  }


  bpl = bm->upl;

  if (y1 < y0) {

    /* scrollup == top-down copy */

    sptr = (uchar *)bm->data + y0 * bpl + x0;
    dptr = (uchar *)bm->data + y1 * bpl + x0;

    while (height--) {
#if 0
      bcopy(sptr, dptr, width);
#else
      slptr = sptr;
      dlptr = dptr;
      xcount = width;
      while (xcount && ((unsigned long)slptr & 3)) {
	*dlptr++ = *slptr++;
	xcount--;
      }
      while (xcount & ~3) {
	*((unsigned long *)dlptr)++ = *((unsigned long *)slptr)++;
	xcount -= 4;
      }
      while (xcount--) {
	*dlptr++ = *slptr++;
      }
#endif
      sptr += bpl;
      dptr += bpl;
    }

  } else {

    /* scrolldown == bottom-up copy */

    sptr = (uchar *)bm->data + (y0 + height - 1) * bpl + x0;
    dptr = (uchar *)bm->data + (y1 + height - 1) * bpl + x0;

    while (height--) {
#if 0
      bcopy(sptr, dptr, width);
#else
      slptr = sptr;
      dlptr = dptr;
      xcount = width;
      while (xcount && ((unsigned long)slptr & 3)) {
	*dlptr++ = *slptr++;
	xcount--;
      }
      while (xcount & ~3) {
	*((unsigned long *)dlptr)++ = *((unsigned long *)slptr)++;
	xcount -= 4;
      }
      while (xcount--) {
	*dlptr++ = *slptr++;
      }
#endif
      sptr -= bpl;
      dptr -= bpl;
    }
  }
}


/*
 * used for clipping characters
 *
 * kay seems to hope this will fit for even the largest chars. :) TeSche
 */

static uchar tmpbm_data[64*100];
static BITMAP tmpbm = {64, 100, BM_DIRECT8, 1, 64, 8, tmpbm_data};
static REC tmpclip = {0, 0, 64, 100, 63, 99, NULL};

inline void direct8_printc(BITMAP *bm,
		    long x0,
		    long y0,
		    ulong c)
{
  register short lwidth, cwidth;
  register short cheight;
  register uchar *dptr, *dlptr;
  register ulong *cptr, clong, cbit;
  register long dbpl;
  FONT *f = gc0->font;
  long save_x0, save_y0;
  BITMAP *save_bm = NULL;

  c &= 0xff;
  cheight = f->hdr.height;
  cwidth = f->hdr.widths[c];

  /*
   * if the character needs clipping we draw it to a temporary bitmap
   * and copy it to the other bitmap.
   */
  if (BOX_NEEDS_CLIPPING (x0, y0, cwidth, cheight, clip0)) {
    if (cwidth > tmpbm.width || cheight+1 > tmpbm.height)
      return;
    save_bm = bm;
    save_x0 = x0;
    save_y0 = y0;

    bm = &tmpbm;
    x0 = 0;
    y0 = 0;
  }

  dbpl = bm->upl;
  dptr = (uchar *)bm->data + y0 * dbpl + x0;
  cbit = 0x80000000L;
  cptr = f->data + f->offsets[c];
  clong = *cptr++;

  if (!(gc0->textstyle & F_UNDERLINE)) {
    cheight++;
  }

  if (gc0->textstyle & F_INVERS) {

    while (--cheight) {
      dlptr = dptr;
      lwidth = cwidth;
      while (lwidth--) {
	if (clong & cbit)
	  *dlptr++ = 0x00;
	else
	  *dlptr++ = 0xff;
	if (!(cbit >>= 1)) {
	  cbit = 0x80000000L;
	  clong = *cptr++;
	}
      }
      dptr += dbpl;
    }
    if (gc0->textstyle & F_UNDERLINE) {
      lwidth = cwidth;
      while (lwidth--) {
	*dptr++ = 0x00;
      }
    }

  } else {

    while (--cheight) {
      dlptr = dptr;
      lwidth = cwidth;
      while (lwidth--) {
	if (clong & cbit)
	  *dlptr++ = 0xff;
	else
	  *dlptr++ = 0x00;
	if (!(cbit >>= 1)) {
	  cbit = 0x80000000L;
	  clong = *cptr++;
	}
      }
      dptr += dbpl;
    }
    if (gc0->textstyle & F_UNDERLINE) {
      lwidth = cwidth;
      while (lwidth--) {
	*dptr++ = 0xff;
      }
    }
  }
  if (save_bm) {
    REC *save_clip1 = clip1;
    clip1 = clip0;
    clip0 = &tmpclip;
    (*theScreen->bitblk) (bm, 0, 0, f->hdr.widths[c], f->hdr.height,
			  save_bm, save_x0, save_y0);
    clip0 = clip1;
    clip1 = save_clip1;
  }
}


void direct8_prints(register BITMAP *bm,
		    register long x0,
		    register long y0,
		    register uchar *s)
{
  register uchar *widths = gc0->font->hdr.widths;

  while (*s) {
    direct8_printc(bm, x0, y0, *s);
    x0 += widths[*s++];
  }
}
