/*
 * server/graph/atari.c, part of W
 * (C) 94-02/96 by Torsten Scherer (TeSche)
 * itschere@techfak.uni-bielefeld.de
 *
 * a graphic driver for the Atari TT, either MiNT or Linux68k
 */

#include <stdio.h>

#if defined(__MINT__) || defined(linux)

#include "../../lib/Wlib.h"
#include "../config.h"
#include "../types.h"
#include "../pakets.h"
#include "../proto.h"
#include "gproto.h"
#include "packed/packed.h"
#include "generic/generic.h"

#ifdef PACKEDMONO
#error is mono
#endif

#ifdef PACKEDCOLOR
#error is color
#endif


/*
 * the big init functions
 */

#ifdef linux

#include <fcntl.h>
#include <signal.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/ioctl.h>
#include <sys/mman.h>
#include <sys/types.h>
#include <linux/fb.h>
#include <linux/vt.h>
#include <linux/kd.h>

static struct fb_fix_screeninfo finfo;
static struct fb_var_screeninfo vinfo;
static int vsize;
static ushort *origScreen = NULL;
static volatile int sigVt;


/*
 *
 */

static void sigvtswitch(int sig);


static void linuxVtSwitch(void)
{
  struct sigaction sa;

  /* this routine will be called at vt-switch by the main loop when all
   * graphics has been done, therefore we needn't any kind of semaphore
   * for the screen, just copy it...
   */

  switch(sigVt) {

    case SIGUSR1:
      if (!origScreen) {
	origScreen = glob_screen->bm.data;
	if ((glob_screen->bm.data = malloc(vsize))) {
	  memcpy(glob_screen->bm.data, origScreen, vsize);
	  ioctl(0, VT_RELDISP, 1);
	}
      }
      break;

    case SIGUSR2:
      if (origScreen) {
	memcpy(origScreen, glob_screen->bm.data, vsize);
	free(glob_screen->bm.data);
	glob_screen->bm.data = origScreen;
	origScreen = NULL;
	ioctl(0, VT_ACKACQ, 1);
      }
      break;

    default:
      fprintf(stderr, "unknown signal received\n");
      exit(-1);
  }

  /* ready for further signals */

  glob_screen->vtswitch = NULL;
  sa.sa_handler = sigvtswitch;
  sa.sa_flags = 0;
  sa.sa_mask = 0;

  sigaction(SIGUSR1, &sa, NULL);
  sigaction(SIGUSR2, &sa, NULL);
}


static void sigvtswitch(int sig)
{
  struct sigaction sa;

  if ((sig == SIGUSR1) || (sig == SIGUSR2)) {

    /* ignore further signals until this one is served */

    sa.sa_handler = SIG_IGN;
    sa.sa_flags = 0;
    sigaction(SIGUSR1, &sa, NULL);
    sigaction(SIGUSR2, &sa, NULL);

    sigVt = sig;
    glob_screen->vtswitch = linuxVtSwitch;
  }
}


static int linux_screen_init(BITMAP *bm)
{
  int fh;
  caddr_t addr;
  struct vt_mode vt;

  if ((fh = open("/dev/fb0current", O_RDWR)) < 0) {
    perror("open()");
    return -1;
  }

  if (ioctl(fh, FBIOGET_FSCREENINFO, &finfo)) {
    perror("ioctl()");
    return -1;
  }
  finfo.id[15] = 0;
  switch(finfo.type) {
    case FB_TYPE_PACKED_PIXELS:
      printf("%s: packed pixels\n", finfo.id);
      break;
    case FB_TYPE_PLANES:
      printf("%s: planes\n", finfo.id);
      break;
    case FB_TYPE_INTERLEAVED_PLANES:
      printf("%s: interleaved planes\n", finfo.id);
      break;
    default:
      fprintf(stderr, "%s: unknown type\n", finfo.id);
      return -1;
  }

  ioctl(fh, FBIOGET_VSCREENINFO, &vinfo);
  printf("using %ix%i of %ix%i pixels, %i bits per pixel\n",
	 vinfo.xres, vinfo.yres,
	 vinfo.xres_virtual, vinfo.yres_virtual,
	 vinfo.bits_per_pixel);
  vsize = (vinfo.xres * vinfo.bits_per_pixel >> 3) * vinfo.yres;
  if ((addr = mmap(0, vsize, PROT_READ | PROT_WRITE, MAP_SHARED, fh, 0)) < 0) {
    perror("mmap()");
    return -1;
  }
  printf("%ik videoram mapped to 0x%08x\n", vsize >> 10,(int)addr);

#if 1
  /* sigh, I often have got to disable this code because if wserver crashes
   * under Linux68k then the screen is locked in graphic mode and I've got
   * to reboot my machine... :( TeSche
   */

  /* set up signal handlers for vtswitch */
  signal(SIGUSR1, sigvtswitch);
  signal(SIGUSR2, sigvtswitch);

  /* disable vty switches */
  vt.mode = VT_PROCESS;
  vt.waitv = 0;
  vt.relsig = SIGUSR1;
  vt.acqsig = SIGUSR2;
  if (ioctl(0, VT_SETMODE, &vt)) {
    perror("ioctl(VT_SETMODE)");
    return -1;
  }

  /* disable screensaver */
  ioctl(0, KDSETMODE, KD_GRAPHICS);
#endif

  sleep(2);

  vinfo.xoffset = 0;
  vinfo.yoffset = 0;
  ioctl(fh, FBIOPUT_VSCREENINFO, &vinfo);

  bm->width = vinfo.xres;
  bm->height = vinfo.yres;
  bm->planes = vinfo.bits_per_pixel;
  bm->data = addr;

  return 0;
}

#endif


#ifdef __MINT__

#include <osbind.h>

int mint_screen_init(BITMAP *bm)
{
  switch(Getrez()) {

    case 2:
      bm->width = 640;
      bm->height = 400;
      bm->planes = 1;
      break;

    case 6:
      bm->width = 1280;
      bm->height = 960;
      bm->planes = 1;
      break;

    case 0:
      bm->width = 320;
      bm->height = 200;
      bm->planes = 4;
      break;

    case 1:
      bm->width = 640;
      bm->height = 200;
      bm->planes = 2;
      break;

    case 4:
      bm->width = 640;
      bm->height = 480;
      bm->planes = 4;
      break;

    case 7:
      bm->width = 320;
      bm->height = 480;
      bm->planes = 8;
      break;

    default:
      return -1;
  }

  bm->data = Logbase();

  if (bm->planes == 1) {
    printf("Mint monochrome, %i * %i pixel\n", bm->width, bm->height);
  } else {
    printf("Mint color, %i * %i pixel\n", bm->width, bm->height);
  }

  return 0;
}

#endif


/*
 * the real init function
 */

#if PCOLOR
SCREEN packed_color_screen = {
  {},
  NULL, packed_color_mouseShow, packed_color_mouseHide,
  packed_color_mouseMove, packed_color_plot, packed_color_test,
  packed_color_line, packed_color_hline, packed_color_vline, generic_box,
  generic_pbox, packed_color_dvline, packed_color_dhline, generic_dbox,
  generic_dpbox, generic_circ, generic_pcirc, packed_color_bitblk,
  packed_color_scroll, packed_color_printc, packed_color_prints,
  packed_color_dplot, packed_color_dline, generic_dcirc, generic_dpcirc,
  generic_poly, generic_dpoly, generic_ppoly, generic_dppoly,
  packed_color_getblock, packed_color_putblock
};
#endif

#if PMONO
SCREEN packed_mono_screen = {
  {},
  NULL, packed_mono_mouseShow, packed_mono_mouseHide, packed_mono_mouseMove,
  packed_mono_plot, packed_mono_test, packed_mono_line, packed_mono_hline,
  packed_mono_vline, generic_box, generic_pbox, packed_mono_dvline,
  packed_mono_dhline, generic_dbox, generic_dpbox, generic_circ,
  generic_pcirc, packed_mono_bitblk, packed_mono_scroll, packed_mono_printc,
  packed_mono_prints, packed_mono_dplot, packed_mono_dline, generic_dcirc,
  generic_dpcirc, generic_poly, generic_dpoly, generic_ppoly, generic_dppoly,
  packed_mono_getblock, packed_mono_putblock
};
#endif

SCREEN *atari_init(void)
{
  int ret;
  BITMAP bm;

#if defined(linux)
  ret = linux_screen_init(&bm);
#elif defined(__MINT__)
  ret = mint_screen_init(&bm);
#else
  ret = -1;
#endif

  if (ret) {
    fprintf(stderr, "fatal: unknown graphics or system is not supported!\r\n");
    return NULL;
  }

#if PMONO
  if (bm.planes == 1) {
    if (bm.width & 31) {
      fprintf(stderr, "warning: monochrome screen width not a multiple of 32 pixels,\n");
      fprintf(stderr, "         can't use this graphic mode!\n");
    } else {
      bm.type = BM_PACKEDMONO;
      bm.unitsize = 4;
      bm.upl = bm.width >> 5;
      packed_mono_screen.bm = bm;
      return &packed_mono_screen;
    }
  }
#endif

#if PCOLOR
  bm.type = BM_PACKEDCOLOR;
  bm.unitsize = 2;
  bm.upl = (bm.width * bm.planes) >> 4;
  packed_color_screen.bm = bm;
  return &packed_color_screen;
#endif

  fprintf(stderr, "fatal: you didn't configure any graphic driver?!\r\n");

  return NULL;
}

#endif
