/*
 * server/client_ret.c, part of W
 * (C) 94-02/96 by Torsten Scherer (TeSche)
 * itschere@techfak.uni-bielefeld.de
 *
 * these are functions (called by clients) that need to return something
 */

#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include "wserver.h"
#include "window.h"
#include "rect.h"


/*
 * functions dealing with windows
 */

/* size of border hardcoded */
#define BORDERWIDTH 4

static long create(CLIENT *cptr, short wwidth, short wheight, short flags,
		   WINDOW *parent, WWIN *libPtr)
{
  long size;
  short border = 0, ewidth = wwidth, eheight = wheight;
  WINDOW *win;
  int wx0 = 0, wy0 = 0, ax = 0, awidth = 0;
  int theight = 0;

  flags &= W_FLAGMASK;

  if (flags & W_TOP) {
    /* a W_TOP window (wcpyrgt, wgone & wsaver) is a dangerous thing as it can
     * lock up the screen (concerning wgone & wsaver this is exactly what it's
     * good for), so we will only allow it for local connections and if the
     * user who is requesting it is the same who started us.
     *
     * a request for a W_TOP window may still fail later on because due to
     * the same security reasons we allow only one such window to exist at a
     * time
     */
    if (cptr->raddr || (cptr->uid != glob_uid)) {
      return 0;
    }
  }

  if (!(flags & W_TITLE)) {
    /* a window without title can't have gadgets */
    flags &= ~(W_CLOSE | W_ICON | W_SIZE);
  }

  if (flags & W_NOBORDER) {
    /* a window without border can't have anything */
    flags &= ~(W_CLOSE | W_ICON | W_SIZE | W_TITLE);
  } else {
    /* add the window border */
    ewidth += BORDERWIDTH << 1;
    eheight += BORDERWIDTH << 1;
    wx0 = wy0 = border = BORDERWIDTH;
  }

  if (flags & W_TITLE) {
    flags |= W_CLOSE;   /* W_CLOSE is a `must' */
    theight = glob_font[TITLEFONT].hdr.height;
    eheight += 4 + theight;
    wy0 += 4 + theight;
    ax = BORDERWIDTH;
    awidth = wwidth;
  }

  if (!(win = window_create(parent, flags))) {
    return 0;
  }

  /* w->pos holds position and size of the whole window. the struct is already
   * filled with 0's, so x and y are already set to 0.
   */
  win->pos.x1 = (win->pos.w = ewidth) - 1;
  win->pos.y1 = (win->pos.h = eheight) - 1;

  /* w->work holds position and size of the working area of the window (later to
   * be updated to absolute coordinates) as well as area[AREA_WORK] does (always
   * relative to w->pos.
   */
  win->work.x1 = (win->work.x0 = wx0) + (win->work.w = wwidth) - 1;
  win->work.y1 = (win->work.y0 = wy0) + (win->work.h = wheight) - 1;
  win->area[AREA_WORK] = win->work;

  /* now set up the other areas, if there're any - otherwise the fields
   * will remain set to zero
   */
  if (flags & W_CLOSE) {
    win->area[AREA_CLOSE].x0 = ax;
    win->area[AREA_CLOSE].y0 = BORDERWIDTH;
    win->area[AREA_CLOSE].w = MIN(theight, awidth);
    win->area[AREA_CLOSE].h = theight;
    win->area[AREA_CLOSE].x1 = win->area[AREA_CLOSE].x0 + win->area[AREA_CLOSE].w - 1;
    win->area[AREA_CLOSE].y1 = win->area[AREA_CLOSE].y0 + theight - 1;
    ax += win->area[AREA_CLOSE].w + BORDERWIDTH;
    awidth -= win->area[AREA_CLOSE].w + BORDERWIDTH;
  }
  if (flags & W_ICON) {
    win->area[AREA_ICON].x0 = ax;
    win->area[AREA_ICON].y0 = BORDERWIDTH;
    win->area[AREA_ICON].w = MIN(theight, awidth);
    win->area[AREA_ICON].h = theight;
    win->area[AREA_ICON].x1 = win->area[AREA_ICON].x0 + win->area[AREA_ICON].w - 1;
    win->area[AREA_ICON].y1 = win->area[AREA_ICON].y0 + theight - 1;
    ax += win->area[AREA_CLOSE].w + BORDERWIDTH;
    awidth -= win->area[AREA_ICON].w + BORDERWIDTH;
  }
  if (flags & W_TITLE) {
    win->area[AREA_TITLE].x0 = ax;
    win->area[AREA_TITLE].y0 = BORDERWIDTH;
    win->area[AREA_TITLE].w = awidth;
    win->area[AREA_TITLE].h = theight;
    win->area[AREA_TITLE].x1 = win->area[AREA_TITLE].x0 + awidth - 1;
    win->area[AREA_TITLE].y1 = win->area[AREA_TITLE].y0 + theight - 1;
  }
  if (flags & W_SIZE) {
    /* nothing yet */
  }

  /* set up the default GCONTEXT structure for this window
   */
  win->gc.drawmode = M_CLEAR;
  win->gc.textstyle = F_NORMAL;
  win->gc.font = NULL; 
  memcpy(win->patbuf, DefaultPattern, sizeof(win->patbuf));
  win->gc.pattern = win->patbuf;

  /* set up the window bitmap, aligning the width to what seems appropriate
   */
  win->bitmap = glob_screen->bm;
  win->bitmap.data = NULL;
  switch (win->bitmap.type) {
    case BM_PACKEDCOLOR:
      win->bitmap.width = (ewidth + 15) & ~15;
      win->bitmap.planes = 1;
      break;
    case BM_PACKEDMONO:
      win->bitmap.width = (ewidth + 31) & ~31;
      break;
    case BM_DIRECT8:
      win->bitmap.width = (ewidth + 3) & ~3;
      break;
  }
  win->bitmap.height = eheight;
  win->bitmap.upl = ((win->bitmap.width * win->bitmap.planes) >> 3) / win->bitmap.unitsize;

  /* this is *vitally* important! */
  win->bitmap.data = NULL;

  /* create bitmap and draw new window (if !W_CONTAINER)
   */
  if (!(flags & W_CONTAINER)) {

    size = win->bitmap.unitsize * win->bitmap.upl * eheight;
    if (!(win->bitmap.data = malloc(size))) {
      window_delete(win);
      return 0;
    }

    gc0 = &win->gc;
    clip0 = &win->pos;
    (*glob_screen->pbox)(&win->bitmap, 0, 0, ewidth, eheight);

    if (flags & W_CLOSE) {
      win->gc.drawmode = M_DRAW;
      (*glob_screen->line)(&win->bitmap, win->area[AREA_CLOSE].x0,
			   win->area[AREA_CLOSE].y0, win->area[AREA_CLOSE].x1,
			   win->area[AREA_CLOSE].y1);
      (*glob_screen->line)(&win->bitmap, win->area[AREA_CLOSE].x0,
			   win->area[AREA_CLOSE].y1, win->area[AREA_CLOSE].x1,
			   win->area[AREA_CLOSE].y0);
    }

    if (flags & W_ICON) {
      win->gc.drawmode = M_DRAW;
      (*glob_screen->pbox)(&win->bitmap,
			   win->area[AREA_ICON].x0 + (win->area[AREA_ICON].w >> 1) - 1,
			   win->area[AREA_ICON].y0 + (win->area[AREA_ICON].h >> 1) - 1,
			   2, 2);
    }

    if (flags & W_TITLE) {
      win->gc.drawmode = M_DRAW;
      (*glob_screen->dpbox)(&win->bitmap, win->area[AREA_TITLE].x0,
			  win->area[AREA_TITLE].y0, win->area[AREA_TITLE].w,
			  win->area[AREA_TITLE].h);
    }
  }

  /* final set-ups
   */
  win->gc.drawmode = M_INVERS;
  win->cptr = cptr;
  win->libPtr = libPtr;
  win->flags = flags;
  cptr->totalWindows++;
  globTotalWindows++;

  return win->id;
}


long client_create(CLIENT *cptr, short width, short height,
		   short flags, WWIN *libPtr)
{
  return create(cptr, width, height, flags,
		(flags & W_TOP) ? glob_rootwindow : glob_backgroundwin,
		libPtr);
}


long client_create2(CLIENT *cptr, short width, short height,
		    short flags, ushort handle, WWIN *libPtr)
{
  WINDOW *parent;

  if (handle < 2) {
    /* you cannot append to the root or background window this way */
    return 0;
  }

  if (!(parent = windowLookup(handle))) {
    return 0;
  }

  return create(cptr, width, height, flags, parent, libPtr);
}


short client_open(CLIENT *cptr, ushort handle, short x0, short y0)
{
  WINDOW *win, *parent;
  short px0, py0;

  if (!(win = windowLookup(handle))) {
    return -9999;
  }
  if (win->is_open) {
    return 1;
  }

  /* positions are parent work area relative */

  parent = win->parent;
  px0 = parent->work.x0;
  py0 = parent->work.y0;

  /* huh, w_open(-1,-1) doesn`t work anymore, because windows can very well
   * be at position (-1,-1). kay.
   * well, so then we use something different, eh? TeSche
   */
  if ((x0 == UNDEF) || (y0 == UNDEF)) {
    if (get_rectangle(win->pos.w, win->pos.h,
		      &x0, &y0, BUTTON_LEFT | BUTTON_RIGHT, BUTTON_NONE))
      return -1;
  } else {
    /* absolute coordinates */
    x0 += px0;
    y0 += py0;
  }

#ifdef CHILDS_INSIDE_PARENT
  if (x0 < 0) {
    x0 = 0;
  }
  if (y0 < 0) {
    y0 = 0;
  }
  if (x0 + win->pos.w > glob_screen->bm.width) {
    x0 = glob_screen->bm.width - win->pos.w;
  }
  if (y0 + win->pos.h > glob_screen->bm.height) {
    y0 = glob_screen->bm.height - win->pos.h;
  }
#endif
  /*
   * must disable mouse anyway, because window_save_contents()
   * may copy from any window... The mouse_hide() should
   * better be moved into window_save_contents() ?
   *
   * ++kay
   */
  mouse_hide ();

  /* don't care for clipping here */
  clip0 = NULL;
  clip1 = NULL;

  if (window_open(win, x0, y0) < 0) {
    return -1;
  }

  win->is_dirty = 1;

  /* otherwise we would have no frame at all */
  windowDeactivate(win);

  /* am I now the active window? */
  w_changeActiveWindow();

  return 0;
}


/*
 * this will be called from window_open() when really opening a window (ie
 * it has no closed ancestors).
 *
 * kay.
 */

void client_do_open (WINDOW *win)
{
  if (win->flags & W_NOMOUSE) {
    glob_mouse.disabled++;
  }
  win->cptr->openWindows++;
  globOpenWindows++;
}


short client_move(CLIENT *cptr, ushort handle, short x0, short y0)
{
  WINDOW *win, *parent;
  short width, height;
long ret;

  if (!(win = windowLookup(handle))) {
    return -9999;
  }
  if (!win->is_open) {
    return -1;
  }
  if (!(win->flags & W_MOVE)) {
    return -1;
  }

#ifdef CHILDS_INSIDE_PARENT
  if ((x0 < 0) || (y0 < 0)) {
    return -1;
  }
#endif

  parent = win->parent;

  /* cptr == NULL means we're called by a mouse move and have absolute
   * coordinates, otherwise they're relative to the work area of the parent
   * window.
   */

  if (cptr) {
    /* positions are parent work area relative */
    x0 += parent->work.x0;
    y0 += parent->work.y0;
  } else {
    ;
  }

  width = win->pos.w;
  height = win->pos.h;

#ifdef CHILDS_INSIDE_PARENT
  /* temporarily: a window may only lay inside the work area of its parent */
  if (x0 < parent->work.x0) {
    x0 = parent->work.x0;
  }
  if (x0 + width > parent->work.x0 + parent->work.w) {
    x0 = parent->work.x0 + parent->work.w - width;
  }
  if (y0 < parent->work.y0) {
    y0 = parent->work.y0;
  }
  if (y0 + height > parent->work.y0 + parent->work.h) {
    y0 = parent->work.y0 + parent->work.h - height;
  }
#endif

  /* disable mouse always */
  mouse_hide();

  /* don't care for clipping here */
  clip0 = NULL;
  clip1 = NULL;

  ret = window_move(win, x0, y0);

  /* did the active window change? */
  w_changeActiveWindow();

  return ret;
}


short client_close(CLIENT *cptr, ushort handle)
{
  WINDOW *win;

  if (!(win = windowLookup(handle))) {
    return -9999;
  }
  if (!win->is_open) {
    return -1;
  }

  /* disable mouse always */
  mouse_hide();

  /* don't care for clipping here */
  clip0 = NULL;
  clip1 = NULL;

  window_close(win);

  /* was I the active window? */
  w_changeActiveWindow();

  return 0;
}


/*
 * this will be called from window_close() just closing a window.
 *
 * we need such a function because closing a window causes closing all
 * its subwindows as well.
 * kay.
 */

void client_do_close (WINDOW *win)
{
  if (win->flags & W_NOMOUSE) {
    glob_mouse.disabled--;
  }

  /* if there were any mouse events pending, ignore them */
  if (glob_leftmousepressed == win)
    glob_leftmousepressed = NULL;
  if (glob_rightmousepressed == win)
    glob_rightmousepressed = NULL;

  win->cptr->openWindows--;
  globOpenWindows--;
}


short client_delete(CLIENT *cptr, ushort handle)
{
  WINDOW *win;

  if (!(win = windowLookup(handle))) {
    return -9999;
  }
  if (!win->cptr) {
    return -1;
  }

  if (win->is_open) {
    client_close(cptr, handle);
  }
  window_delete(win);
  return 0;
}


/*
 * this will be called from window_delete() just before deleting a window.
 * it deletes the backing bitmap and updates the window counters.
 *
 * we need such a function because deleting a window causes deleting all
 * its subwindows as well.
 * kay.
 */

void client_do_delete (WINDOW *win)
{
  if (win->bitmap.data) {
    free (win->bitmap.data);
    win->bitmap.data = NULL;
  }

  /* the next two if's make sure we do not deactivate a window
   * that has been deleted already. this shouldn't happen because
   * currently windows are closed (and thus deactivated) before
   * deleting, but better safe than sorry.
   */
  if (win == glob_activewindow)
    glob_activewindow = glob_backgroundwin;
  if (win == glob_activetopwin)
    glob_activetopwin = glob_backgroundwin;

  win->cptr->totalWindows--;
  globTotalWindows--;
}


/*
 * these two load and unload fonts
 */

void client_loadfont(CLIENT *cptr, char *fname, LFONTRETP *lfontrpaket)
{
  short i, cfreeh, globh;

  /* is it already loaded to this client? */
  for (i=0; i<MAXCLIENTFONTS; i++) {
    if (cptr->font[i]) {
      if (!strcmp(cptr->font[i]->name, fname)) {
	lfontrpaket->handle = htons(i);
	lfontrpaket->flags = htons(cptr->font[i]->hdr.flags);
	lfontrpaket->height = htons(cptr->font[i]->hdr.height);
	memcpy(lfontrpaket->widths, cptr->font[i]->hdr.widths, 256);
	return;
      }
    }
  }

  /* is there any free entry for this client? */
  cfreeh = - 1;
  for (i=0; i<MAXCLIENTFONTS; i++) {
    if (!cptr->font[i]) {
      cfreeh = i;
      break;
    }
  }
  if (cfreeh == -1) {
    lfontrpaket->handle = htons(-1);
    return;
  }

  /* is it loaded at all? */
  for (i=0; i<MAXFONTS; i++) {
    /*
     * XXX Mintlib PL 46 has strcasecmp().
     */
#ifdef __MINT__
    if (glob_font[i].numused && !stricmp(glob_font[i].name, fname)) {
#else
    if (glob_font[i].numused && !strcasecmp(glob_font[i].name, fname)) {
#endif
      glob_font[i].numused++;
      cptr->font[cfreeh] = &glob_font[i];
      lfontrpaket->handle = htons(cfreeh);
      lfontrpaket->flags = htons(glob_font[i].hdr.flags);
      lfontrpaket->height = htons(glob_font[i].hdr.height);
      memcpy(lfontrpaket->widths, glob_font[i].hdr.widths, 256);
      return;
    }
  }

  /* looks like we've got to load it manually */
  if ((globh = font_loadfont(fname)) < 0) {
    lfontrpaket->handle = htons(-1);
    return;
  }

  cptr->font[cfreeh] = &glob_font[globh];
  lfontrpaket->handle = htons(cfreeh);
  lfontrpaket->flags = htons(glob_font[globh].hdr.flags);
  lfontrpaket->height = htons(glob_font[globh].hdr.height);
  memcpy(lfontrpaket->widths, glob_font[globh].hdr.widths, 256);
}


/*
 * unloading a font is a bit more difficult
 */

typedef struct {
  CLIENT *cptr;
  FONT *fp;
} ARGS;

static int killFont(WINDOW *win, void *arg)
{
  if ((win->cptr == ((ARGS *)arg)->cptr) && (win->gc.font == ((ARGS *)arg)->fp)) {
    win->gc.font = NULL;
  }

  return 0;
}

short client_unloadfont(CLIENT *cptr, short fonthandle)
{
  ARGS args;

  if ((fonthandle < 0) || (fonthandle >=MAXCLIENTFONTS)) {
    return -1;
  }
  if (!cptr->font[fonthandle]) {
    return 1;
  }

  /* unload this font from all windows of that client */

  args.cptr = cptr;
  args.fp = cptr->font[fonthandle];

  wtree_traverse_pre(glob_rootwindow, &args, killFont);

  cptr->font[fonthandle] = NULL;
  if (--args.fp->numused < 1) {
    font_unloadfont(args.fp);
  }

  return 0;
}


short client_querywinsize(CLIENT *cptr, ushort handle, short effective,
			  short *width, short *height)
{
  WINDOW *win;

  if (!(win = windowLookup(handle))) {
    return -9999;
  }

  if (effective) {
    if (width)
      *width = win->pos.w;
    if (height)
      *height = win->pos.h;
  } else {
    if (width)
      *width = win->work.w;
    if (height)
      *height = win->work.h;
  }

  return 0;
}


short client_test(CLIENT *cptr, ushort handle, short x0, short y0)
{
  WINDOW *win;
  short ret;

  if (!(win = windowLookup(handle))) {
    return -9999;
  }

  if (win->flags & W_CONTAINER) {
    return -9998;
  }

  x0 += win->area[AREA_WORK].x0;
  y0 += win->area[AREA_WORK].y0;

#ifndef REFRESH
  ret = -1;
  gc0 = &win->gc;
  if (!win->is_open || win->is_hidden) {
#endif
    clip0 = &win->area[AREA_WORK];
    ret = (*glob_screen->test)(&win->bitmap, x0, y0);
#ifndef REFRESH
  } else {
    if (mouse_rcintersect(win->pos.x0 + x0, win->pos.y0 + y0, 1, 1)) {
      mouse_hide();
    }
    clip0 = &win->work;
    ret = (*glob_screen->test)(&glob_screen->bm, win->pos.x0 + x0,
			       win->pos.y0 + y0);
  }
#endif
  return ret;
}


short client_querymousepos(CLIENT *cptr, ushort handle, short *x0, short *y0)
{
  WINDOW *win;
  short mx, my;

  mx = glob_mouse.dx;
  my = glob_mouse.dy;

  if (!handle) {
    if (x0)
      *x0 = mx;
    if (y0)
      *y0 = my;
    return 0;
  }

  if (!(win = windowLookup(handle))) {
    return -9999;
  }

  if (win->cptr && win->is_open) {
    /* is it in this window? */
    if (window_find(mx, my, 0) != win) {
      return -1;
    }

    mx -= win->pos.x0;
    my -= win->pos.y0;

    /* is it really in the work area of the window? */
    if (!rect_cont_point (&win->area[AREA_WORK], mx, my))
      return -1;

    if (x0)
      *x0 = mx - win->area[AREA_WORK].x0;
    if (y0)
      *y0 = my - win->area[AREA_WORK].y0;

    return 0;
  }

  return -1;
}


short client_querywindowpos(CLIENT *cptr, ushort handle, short effective,
			    short *x0, short *y0)
{
  WINDOW *win;

  if (!(win = windowLookup(handle))) {
    return -9999;
  }

  if (win->cptr && win->is_open) {

    if (effective) {
      if (x0)
	*x0 = win->pos.x0;
      if (y0)
	*y0 = win->pos.y0;
    } else {
      if (x0)
	*x0 = win->work.x0;
      if (y0)
	*y0 = win->work.y0;
    }

    return 0;
  }

  return -1;
}
