/*
 * programs/wterm.c, part of W
 * (C) 94-02/96 by Torsten Scherer (TeSche)
 * itschere@techfak.uni-bielefeld.de
 *
 * - quite some changes for W1R1
 * - yet more changes for W1R2
 *
 * TeSche 01/96:
 * - supports W_ICON & W_CLOSE
 * - supports /etc/utmp logging for SunOS4 and Linux
 * - supports catching of console output for SunOS4
 * Phx 02/96:
 * - supports NetBSD-Amiga
 */

#include <stdio.h>
#include <signal.h>
#include <utmp.h>
#include <sys/time.h>
#include <unistd.h>
#include <stdlib.h>
#include <pwd.h>
#include <string.h>
#include <sys/ioctl.h>
#include <termios.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>

#include <Wlib.h>

#define	SMALLBUF 63
#define	LARGEBUF 1024


/*
 * some pty helper functions
 */

#ifdef __MINT__
#define NSIG __NSIG
#elif linux
#define NSIG _NSIG
#endif

#ifdef __MINT__

#include <support.h>
#include <mintbind.h>
extern int forkpty(int *master, char *pty, struct termios *t, struct winsize *w);

#else

void _write_utmp(char *line, char *user, char *host, int time)
{
  int fh, offset, isEmpty, isLine;
  struct utmp ut;

  if ((fh = open("/etc/utmp", O_RDWR)) < 0) {
    return;
  }

  /* first of all try to find an entry with the same line */

  offset = 0;
  isEmpty = -1;
  isLine = -1;

  while ((isLine < 0) && (read(fh, &ut, sizeof(ut)) == sizeof(ut))) {
    if (!ut.ut_line[0]) {
      if (isEmpty < 0) {
	isEmpty = offset;
      }
    } else {
      if (!strncmp(ut.ut_line, line, sizeof(ut.ut_line))) {
	isLine = offset;
      }
    }
    offset += sizeof(ut);
  }

  if (isLine != -1) {
    /* we've found a match */
    lseek(fh, isLine, SEEK_SET);
  } else if (isEmpty != -1) {
    /* no match found, but at least an empty entry */
    lseek(fh, isLine, SEEK_SET);
  } else {
    /* not even an empty entry found, assume we can append to the file */
  }

  if (time) {
    strncpy(ut.ut_line, line, sizeof(ut.ut_line));
    strncpy(ut.ut_name, user, sizeof(ut.ut_name));
    strncpy(ut.ut_host, host, sizeof(ut.ut_host));
    ut.ut_time = time;
#ifdef linux
    ut.ut_type = USER_PROCESS;
#endif
  } else {
    memset(&ut, 0, sizeof(ut));
#ifdef linux
    ut.ut_type = DEAD_PROCESS;
#endif
  }
  write(fh, &ut, sizeof(ut));
  close(fh);
}

int forkpty(int *master, char *pty, struct termios *t, struct winsize *w)
{
  int i, mh, sh, pid;
  char fname[64];

  /* get a free pty */
  for (i=0; i<16; i++) {
    sprintf(fname, "/dev/ptyp%x", i);
    if ((mh = open(fname, O_RDWR)) >= 0)
      break;
  }

  if (mh < 0) {
    return -1;
  }

  /* open slave */
  fname[5] = 't';
  if ((sh = open(fname, O_RDWR)) < 0) {
    close(mh);
    return -1;
  }

  /* set terminal size */
  if (w) {
    ioctl(sh, TIOCSWINSZ, w);
  }

  /* try to fork */
  switch (pid = fork()) {

    case -1: /* parent: error */
      close(sh);
      close(mh);
      return -1;

    case 0: /* child */
#if 0
      close(mh);
#endif
      setsid();
      ioctl(sh, TIOCSCTTY, 0);

      dup2(sh, 0);
      dup2(sh, 1);
      dup2(sh, 2);
      if (sh > 2) {
	close(sh);
      }
      return 0;

    default: /* parent: ok */
#if 0
      close(sh);
#endif
      *master = mh;
      strcpy(pty, fname);
      return pid;
  }

  return -1;
}

#endif


/*
 * some global variables
 */

static WWIN *win;
static short winw, winh, pid, fonw, fonh, console, cfh, dh;
static int pipeh;
static char pty[SMALLBUF];
static short cblink = 0, visualbell = 0, debug = 0;
static struct winsize winsz;
static short winchflag = 0;


/****************************************************************************/

/*
 *
 */

static WWIN *iconWin;
static int isIconified;

int iconSetup(void)
{
  WFONT *font;

  if (!(font = w_loadfont("lucidat12b.wfnt"))) {
    return -1;
  }

  if (!(iconWin = w_create(w_strlen(font, "wterm") + 2, font->height + 2,
			   W_MOVE | EV_MOUSE))) {
    w_unloadfont(font);
    return -1;
  }

  w_centerPrints(iconWin, font, "wterm");
  w_unloadfont(font);

  isIconified = 0;

  return 0;
}

void iconify(void)
{
  short x0, y0;

  if (isIconified) {
    w_querywindowpos(iconWin, 1, &x0, &y0);
    w_close(iconWin);
    w_open(win, x0, y0);
  } else {
    w_querywindowpos(win, 1, &x0, &y0);
    w_close(win);
    w_open(iconWin, x0, y0);
  }

  isIconified = 1 - isIconified;
}


/****************************************************************************/


/*
 * some common tool functions
 */

void sigpipe(int sig)
{
  /* this one musn't close the window */
  _write_utmp(pty, "", "", 0);
  kill(-pid, SIGHUP);
  _exit(sig);
}


void sigchld(int sig)
{
  _write_utmp(pty, "", "", 0);
  _exit(sig);
}


void sigquit(int sig)
{
  signal(sig, SIG_IGN);
  kill(-pid, SIGHUP);
}


/*
 * this is the wterm terminal code, almost like VT52
 */

short	escstate, curx, cury, curon, curvis;
short	savx, savy, wrap, style;
short	col, lin, colmask = 0x7f, linmask = 0x7f;


void draw_cursor()
{
  if (!curvis) {
    curvis = 1;
    w_setmode(win, M_INVERS);
    w_pbox(win, curx*fonw, cury*fonh, fonw, fonh);
  }
}


void hide_cursor()
{
  if (curvis) {
    curvis = 0;
    w_setmode(win, M_INVERS);
    w_pbox(win, curx*fonw, cury*fonh, fonw, fonh);
  }
}


void esc5(uchar c)	/* setting background color */
{
  escstate = 0;
}


void esc4(uchar c)	/* setting foreground color */
{
  escstate = 0;
}


void esc3(uchar c)	/* cursor position x axis */
{
  curx = (c - 32) & colmask;
  if (curx >= col)
    curx = col - 1;
  else if (curx < 0)
    curx = 0;
  escstate = 0;
}


void esc2(uchar c)	/* cursor position y axis */
{
  cury = (c - 32) & linmask;
  if (cury >= lin)
    cury = lin - 1;
  else if (cury < 0)
    cury = 0;
  escstate = 3;
}


void esc1(uchar c)	/* various control codes */
{
  escstate = 0;

  switch(c) {
    case 'A':/* cursor up */
      if ((cury -= 1) < 0)
	cury = 0;
      break;

    case 'B':/* cursor down */
      if ((cury += 1) >= lin)
	cury = lin - 1;
      break;

    case 'C':/* cursor right */
      if ((curx += 1) >= col)
	curx = col - 1;
      break;

    case 'D':/* cursor left */
      if ((curx -= 1) < 0)
	curx = 0;
      break;

    case 'E':/* clear screen & home */
      w_setmode(win, M_CLEAR);
      w_pbox(win, 0, 0, winw, winh);
      curx = 0;
      cury = 0;
      break;

    case 'H':/* cursor home */
      curx = 0;
      cury = 0;
      break;

    case 'I':/* reverse index */
      if ((cury -= 1) < 0) {
	cury = 0;
	w_vscroll(win, 0, 0, winw, (lin-1)*fonh, fonh);
	w_setmode(win, M_CLEAR);
	w_pbox(win, 0, 0, winw, fonh);
      }
      break;

    case 'J':/* erase to end of page */
      w_setmode(win, M_CLEAR);
      if (cury < lin-1)
	w_pbox(win, 0, (cury+1)*fonh, winw, (lin-1-cury)*fonh);
      w_pbox(win, curx*fonw, cury*fonh, (col-curx)*fonw, fonh);
      break;

    case 'K':/* erase to end of line */
      w_setmode(win, M_CLEAR);
      w_pbox(win, curx*fonw, cury*fonh, (col-curx)*fonw, fonh);
      break;

    case 'L':/* insert line */
      if (cury < lin-1) {
	w_vscroll(win, 0, cury*fonh, winw, (lin-1-cury)*fonh, (cury+1)*fonh);
      }
      w_setmode(win, M_CLEAR);
      w_pbox(win, 0, cury*fonh, winw, fonh);
      curx = 0;
      break;

    case 'M':/* delete line */
      if (cury < lin-1) {
	w_vscroll(win, 0, (cury+1)*fonh, winw, (lin-1-cury)*fonh, cury*fonh);
      }
      w_setmode(win, M_CLEAR);
      w_pbox(win, 0, (lin-1)*fonh, winw, fonh);
      curx = 0;
      break;

    case 'Y':/* position cursor */
      escstate = 2;
      break;

    case 'b':/* set foreground color */
      escstate = 4;
      break;

    case 'c':/* set background color */
      escstate = 5;
      break;

    case 'd':/* erase beginning of display */
      w_setmode(win, M_CLEAR);
      if (cury > 0)
	w_pbox(win, 0, 0, winw, cury*fonh);
      if (curx > 0)
	w_pbox(win, 0, cury*fonh, curx*fonw, fonh);
      break;

    case 'e':/* enable cursor */
      curon = 1;
      break;

    case 'f':/* disable cursor */
      curon = 0;
      break;

    case 'j':/* save cursor position */
      savx = curx;
      savy = cury;
      break;

    case 'k':/* restore cursor position */
      curx = savx;
      cury = savy;
      break;

    case 'l':/* erase entire line */
      w_setmode(win, M_CLEAR);
      w_pbox(win, 0, cury*fonh, winw, fonh);
      curx = 0;
      break;

    case 'o':/* erase beginning of line */
      w_setmode(win, M_CLEAR);
      if (curx > 0)
	w_pbox(win, 0, cury*fonh, curx*fonw, fonh);
      break;

    case 'p':/* enter reverse video mode */
      style |= F_INVERS;
      w_settextstyle(win, style);
      break;

    case 'q':/* exit reverse video mode */
      style &= ~F_INVERS;
      w_settextstyle(win, style);
      break;

    case 'v':/* enable wrap at end of line */
      wrap = 1;
      break;

    case 'w':/* disable wrap at end of line */
      wrap = 0;
      break;

/* and these are the extentions not in VT52 */

    case 'g': /* enter bold mode */
      style |= F_BOLD;
      w_settextstyle(win, style);
      break;

    case 'h': /* clear all sttributes */
      style = 0;
      w_settextstyle(win, style);
      break;

    case 'i': /* enter underline mode */
      style |= F_UNDERLINE;
      w_settextstyle(win, style);
      break;

      /* j, k and l are already used */

    case 'm': /* exit underline mode */
      style &= ~F_UNDERLINE;
      w_settextstyle(win, style);
      break;

    default: /* unknown escape sequence */
      break;
  }
}


/*
 * something to buffer plain text output
 */

short	sbufcnt = 0;
short	sbufx, sbufy;
char	sbuf[SMALLBUF+1];

void sflush()
{
  if (sbufcnt) {
    sbuf[sbufcnt] = 0;
    w_printstring(win, sbufx*fonw, sbufy*fonh, sbuf);
    sbufcnt = 0;
  }
}

void sadd(char c)
{
  if (sbufcnt == SMALLBUF) {
    sflush();
  }

  if (!sbufcnt) {
    sbufx = curx;
    sbufy = cury;
  }

  sbuf[sbufcnt++] = c;
}


/*
 * un-escaped character print routine
 */

void esc0(uchar c)
{
  switch(c) {
    case 0:
      /*
       * printing \000 on a terminal means "do nothing".
       * But since we use \000 as string terminator none
       * of the characters that follow were printed.
       *
       * perl -e 'printf("a%ca", 0);'
       *
       * said 'a' in a wterm, but should say 'aa'. This
       * bug screwed up most ncurses programs.
       *
       * kay.
       */
    break;
 
  case 7: /* bell */
      if (visualbell) {
	w_setmode(win, M_INVERS);
	w_pbox(win, 0, 0, winw, winh);
	w_test(win, 0, 0);
	w_pbox(win, 0, 0, winw, winh);
      } else {
	w_beep();
      }
      break;

    case 8: /* backspace */
      sflush();
      if (--curx < 0) {
	curx = 0;
      }
      break;

    case 9: /* tab */
      sflush();
      if ((curx = ((curx >> 3) + 1) << 3) >= col) {
	curx = col - 1;
      }
      break;

    case 10: /* line feed */
      sflush();
      if (++cury >= lin) {
	w_vscroll(win, 0, fonh, winw, (lin-1)*fonh, 0);
	w_setmode(win, M_CLEAR);
	w_pbox(win, 0, (lin-1)*fonh, winw, fonh);
	cury = lin-1;
      }
      break;

    case 13: /* carriage return */
      sflush();
      curx = 0;
      break;

    case 27: /* escape */
      sflush();
      escstate = 1;
      break;

    case 127: /* delete */
      break;

    default: /* any printable char */
      sadd(c);
      if (++curx >= col) {
	sflush();
	if (!wrap) {
	  curx = col-1;
	} else {
	  curx = 0;
	  if (++cury >= lin) {
	    w_vscroll(win, 0, fonh, winw, (lin-1)*fonh, 0);
	    w_setmode(win, M_CLEAR);
	    w_pbox(win, 0, (lin-1)*fonh, winw, fonh);
	    cury = lin-1;
	  }
	}
      }
  }
}


void printc(uchar c)
{
  switch(escstate) {
    case 0:
      esc0(c);
      break;

    case 1:
      sflush();
      esc1(c);
      break;

    case 2:
      sflush();
      esc2(c);
      break;

    case 3:
      sflush();
      esc3(c);
      break;

    case 4:
      sflush();
      esc4(c);
      break;

    case 5:
      sflush();
      esc5(c);
      break;

    default: escstate = 0;
  }
}


void init()
{
  curx = savx = 0;
  cury = savy = 0;
  wrap = 1;
  style = F_NORMAL;
  curon = 1;
  curvis = 0;
  draw_cursor();
  escstate = 0;
  w_settextstyle(win, style);
}


/*
 * general code...
 */

void prints(char *s)
{
  while (*s)
    printc((short)*s++);
}


void term()
{
  uchar buf[LARGEBUF];
  long in, l;
  WEVENT *wevent;
  fd_set rfd;
  int ready;

  while (42) {
    FD_ZERO(&rfd);
    FD_SET(pipeh, &rfd);
    if (console) {
      FD_SET(cfh, &rfd);
    }

    /* how about an event? */
    ready = 0;

    if ((wevent = w_queryevent(&rfd, NULL, NULL, 500))) {

      ready++;
      switch (wevent->type) {

        case EVENT_GADGET:
	  switch (wevent->key) {
	    case GADGET_CLOSE:
	    case GADGET_EXIT:
	      sigquit(-1);
	      break;
	    case GADGET_ICON:
	      iconify();
	      break;
	  }
	  break;

	case EVENT_KEY:
	  *buf = wevent->key & 0x7f;
	  write(pipeh, buf, 1);
	  break;

	case EVENT_MPRESS:
	  if (isIconified) {
	    iconify();
	  }
	  break;

#if 0
	  /*
	   * experimental code...
	   */

	case EVENT_MPRESS:
	  w_setmode(win, M_INVERS);
	  w_pbox(win, 0, 0, winw, winh);
	  w_test(win, 0, 0);
	  w_pbox(win, 0, 0, winw, winh);
	  {
	    int pgrp;
	    switch (winchflag = 1 - winchflag) {
	      case 0:
	        winsz.ws_col = col;
		winsz.ws_row = lin;
		break;
	      case 1:
		winsz.ws_col = col - 5;
		winsz.ws_row = lin - 5;
	    }

	    if (ioctl(pipeh, TIOCGPGRP, &pgrp)) {
	      prints("ioctl error #1\r\n");
	    }

	    if (ioctl(pipeh, TIOCSWINSZ, &winsz)) {
	      prints("ioctl error #2\r\n");
	    }
	    kill(-pgrp, SIGWINCH);
	  }
	  break;
#endif
      }
    }

    /* is there tty data to be printed? */
    if (FD_ISSET(pipeh, &rfd)) {

      ready++;

      if (!ioctl(pipeh, FIONREAD, &in)) if (in > 0) {

	if (in > LARGEBUF) {
	  in = LARGEBUF;
	}

	if ((in = read(pipeh, buf, in)) > 0) {

	  if (curon) {
	    hide_cursor();
	  }

	  for (l=0; l<in; l++) {
	    printc(buf[l]);
	    if (buf[l] == '\n')
	      printc('\r');
	  }
	  sflush();

	  if (curon) {
	    draw_cursor();
	  }

	  if (debug) {
	    char *ptr = buf;
#if 1
	    while (in-- > 0) {
	      printf("wterm: received 0x%02x\n", *ptr++);
	    }
#else
	    long cnt = in;
	    while (cnt--)
	      *ptr &= 0x7f;
	    write(dh, buf, in);
#endif
	  }
	}
      }
    }

#ifdef __MINT__
    /* in any case: watch out for console output */
    if (console && FD_ISSET(cfh, &rfd)) {

      ready++;
      if ((in = read(cfh, buf, LARGEBUF)) > 0) {

	if (curon) {
	  hide_cursor();
	}

	for (l=0; l<in; l++) {
	  if (buf[l] == '\n') {
	    printc('\r');
	  }
	  printc(buf[l]);
	}

	sflush();

	if (curon) {
	  draw_cursor();
	}
      }
    }
#endif

    /* cursor blink timeout */
    if (cblink && !ready) {
      if (curvis) {
	hide_cursor();
      } else {
	draw_cursor();
      }
    }
  }
}


volatile void usage(char *s)
{
  if (s) {
    fprintf(stderr, "error: %s\n", s);
  }
  printf("usage: wterm [-b] [-f <fontname>] [-g <geometry>] [-h] [-v] [-c] [program {args}]\n");
  exit(0);
}


void *mysignal(int signum, void *handler)
{
#ifdef __MINT__
  return signal(signum, handler);
#else
  struct sigaction sa, so;

  sa.sa_handler = handler;
#if defined(sun) || defined(__NetBSD__)
  sa.sa_mask = 0;
  sa.sa_flags = 0;
#else
  sa.sa_mask = SA_NOMASK;
  sa.sa_flags = SA_RESTART;
#endif
  sigaction(signum, &sa, &so);

  return so.sa_handler;
#endif
}


/*
 * guess what... :)
 */

void main(long argc, char **argv)
{
  WFONT *font;
  short xp, yp, ewidth, eheight;
  char *fontname, *shell = NULL, *cptr, *geometry = NULL;
  struct passwd *pw;
  char buf[80], *ptr;
  short uid;
  WSERVER *wserver;
  char thesh[128];
#ifdef __MINT__
  int fd;
#endif

  /* just in case we're started in the background */
  signal(SIGTTOU, SIG_IGN);

  /* who am I? */
  if (!(pw = getpwuid((uid = getuid())))) {
    fprintf(stderr, "error: wterm can't determine determine your login name\n");
    exit(-1);
  }

  if (!(wserver = w_init())) {
    fprintf(stderr, "error: wterm can't connect to wserver\n");
    exit(-1);
  }

  if (iconSetup() < 0) {
    fprintf(stderr, "error: wterm can't create icon window\n");
    exit(-1);
  }

  /*
   * which font shall I use?
   */

  if (wserver->width == 1280)
    fontname = "lucidat10.wfnt";
  else
    fontname = "fixed7.wfnt";

  /*
   * scan arguments...
   */

  console = 0;
  argv++;
  while (*argv && **argv=='-') switch (*(*argv+1)) {
    case 'b':
      cblink = 1;
      argv++;
      break;

    case 'c':
      console = 1;
      argv++;
      break;

    case 'd':
      debug = 1;
      argv++;
      break;

    case 'f':
      if (*++argv) {
	fontname = *argv++;
      } else {
	usage("-f option requires an argument");
      }
      break;

    case 'g':
      if (*++argv) {
	geometry = *argv++;
      } else {
	usage("-g option requires an argument");
      }
      break;

    case 'h':
      /* this will never return */
      usage("");

    case 'v':
      visualbell = 1;
      argv++;
      break;

    default:
      usage("unknown option");
  }

  /*
   * now *argv either points to a program to start or is zero
   */

  if (*argv) {
    shell = *argv;
  }
  if (!shell) {
    shell = getenv("SHELL=");
  }
  if (!shell) {
    shell = pw->pw_shell;
  }
  if (!shell) {
    shell = "/bin/sh";
  }

  if (!*argv) {
    /*
     * the '-' makes the shell think it is a login shell,
     * we leave argv[0] alone if it isn`t a shell (ie.
     * the user specified the program to run as an argument
     * to wterm.
     */
    cptr = strrchr(shell, '/');
    sprintf (thesh, "-%s", cptr ? cptr + 1 : shell);
    *--argv = thesh;
  }

  /* create and open window */

  if (!(font = w_loadfont(fontname))) {
    fprintf(stderr, "error: wterm can't load font %s\n", fontname);
    exit(-1);
  }

  fonw = font->widths[0];
  fonh = font->height;

  col = 80;
  lin = 25;
  xp = -32768;
  yp = -32768;
  if (geometry) {
    scan_geometry(geometry, &col, &lin, &xp, &yp);
    if (col < 1) {
      col = 1;
    }
    if (lin < 1) {
      lin = 1;
    }
    if (col > 0x7f)
      colmask = 0xffff;
    if (lin > 0x7f)
      linmask = 0xffff;
  }

  winh = lin * fonh;
  winw = col * fonw;

  if (!(win = w_create(winw, winh, W_MOVE | W_CLOSE | W_ICON | W_TITLE | EV_KEYS | EV_MOUSE))) {
    fprintf(stderr, "error: wterm can't create window\r\n");
    exit(-1);
  }

  w_setfont(win, font);
  w_querywinsize(win, 1, &ewidth, &eheight);

  if ((xp != -32768) && (yp != -32768)) {
    /* negative numbers are 1 too negative, that is -1 stands for `-'0 */
    if (xp < 0) {
      xp = wserver->width - ewidth + xp + 1;
    }
    if (yp < 0) {
      yp = wserver->height - eheight + yp + 1;
    }
  } else {
    xp = UNDEF;
    yp = UNDEF;
  }

  if (w_open(win, xp, yp) < 0) {
    w_unloadfont(font);
    w_close(win);
    w_delete(win);
    exit(-1);
  }

  sprintf(buf, "wterm: %s", shell);
  w_settitle(win, buf);

  /*
   * what kind of terminal do we want to emulate?
   */

  putenv("TERM=wterm");

  /*
   * this one should enable us to get rid of an /etc/termcap entry for
   * both curses and ncurses, hopefully...
   */

#ifdef __MINT__

  putenv("TERMCAP=wterm|WTerm terminal:al=\\EL:am:bc=\\ED:bl=^G:bs:cd=\\EJ:\
ce=\\EK:cl=\\EE:cm=\\EY%+ %+ :cr=^M:dl=\\EM:do=\\EB:ho=\\EH:is=\\Ev\\Eq:\
l0=F10:le=\\ED:ms:nd=\\EC:pt:re=\\EC:rc=\\Ek:rs=\\Ev\\Eq\\EE:sc=\\Ej:sr=\\EI:\
ti=\\Ev\\Ee\\Eh:up=\\EA:ve=\\Ee:vi=\\Ef:so=\\Ep:se=\\Eq:mb=\\Ei:md=\\Eg:\
mr=\\Ep:me=\\Eh:te=\\Eh:us=\\Ei:ue=\\Eh:\
kb=^H:kl=\\ED:kr=\\EC:ku=\\EA:kd=\\EB:kI=\\EI:kh=\\EE:kP=\\Ea:kN=\\Eb:k0=\\EY:\
k1=\\EP:k2=\\EQ:k3=\\ER:k4=\\ES:k5=\\ET:k6=\\EU:k7=\\EV:k8=\\EW:k9=\\EX:\
s0=\\Ey:s1=\\Ep:s2=\\Eq:s3=\\Er:s4=\\Es:s5=\\Et:s6=\\Eu:s7=\\Ev:s8=\\Ew:\
s9=\\Ex:li#25:co#80:");

#if 0
  putenv("TERMINFO=wterm|WTerm terminal:al=\\EL:am:bc=\\ED:bl=^G:bs:cd=\\EJ:\
ce=\\EK:cl=\\EE:cm=\\EY%+ %+ :cr=^M:dl=\\EM:do=\\EB:ho=\\EH:is=\\Ev\\Eq:\
l0=F10:le=\\ED:ms:nd=\\EC:pt:re=\\EC:rc=\\Ek:rs=\\Ev\\Eq\\EE:sc=\\Ej:sr=\\EI:\
ti=\\Ev\\Ee\\Eh:up=\\EA:ve=\\Ee:vi=\\Ef:so=\\Ep:se=\\Eq:mb=\\Ei:md=\\Eg:\
mr=\\Ep:me=\\Eh:te=\\Eh:us=\\Ei:ue=\\Eh:\
kb=^H:kl=\\ED:kr=\\EC:ku=\\EA:kd=\\EB:kI=\\EI:kh=\\EE:kP=\\Ea:kN=\\Eb:k0=\\EY:\
k1=\\EP:k2=\\EQ:k3=\\ER:k4=\\ES:k5=\\ET:k6=\\EU:k7=\\EV:k8=\\EW:k9=\\EX:\
s0=\\Ey:s1=\\Ep:s2=\\Eq:s3=\\Er:s4=\\Es:s5=\\Et:s6=\\Eu:s7=\\Ev:s8=\\Ew:\
s9=\\Ex:");
#endif

#elif defined(linux)

  putenv("TERMCAP=wterm|WTerm terminal:al=\\EL:am:bc=\\ED:bl=^G:bs:cd=\\EJ:\
ce=\\EK:cl=\\EE:cm=\\EY%+ %+ :cr=^M:dl=\\EM:do=\\EB:ho=\\EH:is=\\Ev\\Eq:\
l0=F10:le=\\ED:ms:nd=\\EC:pt:re=\\EC:rc=\\Ek:rs=\\Ev\\Eq\\EE:sc=\\Ej:sr=\\EI:\
ti=\\Ev\\Ee\\Eh:up=\\EA:ve=\\Ee:vi=\\Ef:so=\\Ep:se=\\Eq:mb=\\Ei:md=\\Eg:\
mr=\\Ep:me=\\Eh:te=\\Eh:us=\\Ei:ue=\\Eh:\
kb=^H:kl=\\E[D:kr=\\E[C:ku=\\E[A:kd=\\E[B:kI=\\EI:kh=\\EE:kP=\\Ea:kN=\\Eb:\
k0=\\EY:k1=\\EP:k2=\\EQ:k3=\\ER:k4=\\ES:k5=\\ET:k6=\\EU:k7=\\EV:k8=\\EW:\
k9=\\EX:s0=\\Ey:s1=\\Ep:s2=\\Eq:s3=\\Er:s4=\\Es:s5=\\Et:s6=\\Eu:s7=\\Ev:\
s8=\\Ew:s9=\\Ex:li#25:co#80:");

#if 0
  putenv("TERMINFO=wterm|WTerm terminal:al=\\EL:am:bc=\\ED:bl=^G:bs:cd=\\EJ:\
ce=\\EK:cl=\\EE:cm=\\EY%+ %+ :cr=^M:dl=\\EM:do=\\EB:ho=\\EH:is=\\Ev\\Eq:\
l0=F10:le=\\ED:ms:nd=\\EC:pt:re=\\EC:rc=\\Ek:rs=\\Ev\\Eq\\EE:sc=\\Ej:sr=\\EI:\
ti=\\Ev\\Ee\\Eh:up=\\EA:ve=\\Ee:vi=\\Ef:so=\\Ep:se=\\Eq:mb=\\Ei:md=\\Eg:\
mr=\\Ep:me=\\Eh:te=\\Eh:us=\\Ei:ue=\\Eh:\
kb=^H:kl=\\E[D:kr=\\E[C:ku=\\E[A:kd=\\E[B:kI=\\EI:kh=\\EE:kP=\\Ea:kN=\\Eb:\
k0=\\EY:k1=\\EP:k2=\\EQ:k3=\\ER:k4=\\ES:k5=\\ET:k6=\\EU:k7=\\EV:k8=\\EW:\
k9=\\EX:s0=\\Ey:s1=\\Ep:s2=\\Eq:s3=\\Er:s4=\\Es:s5=\\Et:s6=\\Eu:s7=\\Ev:\
s8=\\Ew:s9=\\Ex:");
#endif

#elif defined(sun)

  /* only very basic cursor keys so far... */

  putenv("TERMCAP=wterm|WTerm terminal:al=\\EL:am:bc=\\ED:bl=^G:bs:cd=\\EJ:\
ce=\\EK:cl=\\EE:cm=\\EY%+ %+ :cr=^M:dl=\\EM:do=\\EB:ho=\\EH:is=\\Ev\\Eq:\
l0=F10:le=\\ED:ms:nd=\\EC:pt:re=\\EC:rc=\\Ek:rs=\\Ev\\Eq\\EE:sc=\\Ej:sr=\\EI:\
ti=\\Ev\\Ee\\Eh:up=\\EA:ve=\\Ee:vi=\\Ef:so=\\Ep:se=\\Eq:mb=\\Ei:md=\\Eg:\
mr=\\Ep:me=\\Eh:te=\\Eh:us=\\Ei:ue=\\Eh:li#25:co#80:\
kb=^H:kl=\\E[D:kr=\\E[C:ku=\\E[A:kd=\\E[B:\
k1=\\E[224z:k2=\\E[225z:k3=\\E[226z:k4=\\E[227z:k5=\\E[228z:\
k6=\\E[229z:k7=\\E[230z:k8=\\E[231z:k9=\\E[232z:k0=\\E[233z:\
kN=\\E[222z:kP=\\E[216z:kh=\\E[214z:kH=\\E220z:");

#if 0
  putenv("TERMINFO=wterm|WTerm terminal:al=\\EL:am:bc=\\ED:bl=^G:bs:cd=\\EJ:\
ce=\\EK:cl=\\EE:cm=\\EY%+ %+ :cr=^M:dl=\\EM:do=\\EB:ho=\\EH:is=\\Ev\\Eq:\
l0=F10:le=\\ED:ms:nd=\\EC:pt:re=\\EC:rc=\\Ek:rs=\\Ev\\Eq\\EE:sc=\\Ej:sr=\\EI:\
ti=\\Ev\\Ee\\Eh:up=\\EA:ve=\\Ee:vi=\\Ef:so=\\Ep:se=\\Eq:mb=\\Ei:md=\\Eg:\
mr=\\Ep:me=\\Eh:te=\\Eh:us=\\Ei:ue=\\Eh:\
kb=^H:kl=\\E[D:kr=\\E[C:ku=\\E[A:kd=\\E[B:\
k1=\\E[224z:k2=\\E[225z:k3=\\E[226z:k4=\\E[227z:k5=\\E[228z:\
k6=\\E[229z:k7=\\E[230z:k8=\\E[231z:k9=\\E[232z:k0=\\E[233z:\
kN=\\E[222z:kP=\\E[216z:kh=\\E[214z:kH=\\E220z:");
#endif

#elif defined(__NetBSD__)

  putenv("TERMCAP=wterm|WTerm terminal:al=\\EL:am:bc=\\ED:bl=^G:bs:cd=\\EJ:\
ce=\\EK:cl=\\EE:cm=\\EY%+ %+ :cr=^M:dl=\\EM:do=\\EB:ho=\\EH:is=\\Ev\\Eq:\
l0=F10:le=\\ED:ms:nd=\\EC:pt:re=\\EC:rc=\\Ek:rs=\\Ev\\Eq\\EE:sc=\\Ej:sr=\\EI:\
ti=\\Ev\\Ee\\Eh:up=\\EA:ve=\\Ee:vi=\\Ef:so=\\Ep:se=\\Eq:mb=\\Ei:md=\\Eg:\
mr=\\Ep:me=\\Eh:te=\\Eh:us=\\Ei:ue=\\Eh:li#25:co#80:\
kb=^H:kl=\\E[D:kr=\\E[C:ku=\\E[A:kd=\\E[B:\
k1=\\E[224z:k2=\\E[225z:k3=\\E[226z:k4=\\E[227z:k5=\\E[228z:\
k6=\\E[229z:k7=\\E[230z:k8=\\E[231z:k9=\\E[232z:k0=\\E[233z:\
kN=\\E[222z:kP=\\E[216z:kh=\\E[214z:kH=\\E220z:");

  putenv("TERMINFO=wterm|WTerm terminal:al=\\EL:am:bc=\\ED:bl=^G:bs:cd=\\EJ:\
ce=\\EK:cl=\\EE:cm=\\EY%+ %+ :cr=^M:dl=\\EM:do=\\EB:ho=\\EH:is=\\Ev\\Eq:\
l0=F10:le=\\ED:ms:nd=\\EC:pt:re=\\EC:rc=\\Ek:rs=\\Ev\\Eq\\EE:sc=\\Ej:sr=\\EI:\
ti=\\Ev\\Ee\\Eh:up=\\EA:ve=\\Ee:vi=\\Ef:so=\\Ep:se=\\Eq:mb=\\Ei:md=\\Eg:\
mr=\\Ep:me=\\Eh:te=\\Eh:us=\\Ei:ue=\\Eh:\
kb=^H:kl=\\E[D:kr=\\E[C:ku=\\E[A:kd=\\E[B:\
k1=\\E[224z:k2=\\E[225z:k3=\\E[226z:k4=\\E[227z:k5=\\E[228z:\
k6=\\E[229z:k7=\\E[230z:k8=\\E[231z:k9=\\E[232z:k0=\\E[233z:\
kN=\\E[222z:kP=\\E[216z:kh=\\E[214z:kH=\\E220z:");

#else

#error oops, a new operating system?

#endif

  init();

  /*
   * create a pty
   */

  winsz.ws_col = col;
  winsz.ws_row = lin;
  if ((pid = forkpty(&pipeh, pty, NULL, &winsz)) < 0) {
    prints("wterm: can't create pty\r\n");
    sleep(2);
    w_unloadfont(font);
    w_close(win);
    w_delete(win);
    exit(-1);
  }

  if ((ptr = rindex(pty, '/'))) {
    strcpy(pty, ptr + 1);
  }
  
  if (!pid) {
    int i;
    for (i = getdtablesize(); --i >= 3; )
      close (i);
    /*
     * SIG_IGN are not reset on exec()
     */
    for (i = NSIG; --i >= 0; )
      signal (i, SIG_DFL);
 
    /* caution: start shell with correct user id! */
    seteuid(getuid());
    setegid(getgid());

    /* this shall not return */
    execvp(shell, argv);

    /* oops? */
    prints("wterm: can't start shell\r\n");
    sleep(3);
    w_unloadfont(font);
    w_close(win);
    w_delete(win);
    _exit(-1);
  }

#ifdef __MINT__
  /*
   * the slave line opened in openpty() may still be our ctty.
   * so loose it.
   */
  fd = open ("/dev/null", O_RDWR);
  dup2 (fd, -1);
  close (fd);
#endif

  _write_utmp(pty, pw->pw_name, "", time(0));

  /* catch some signals */
  mysignal(SIGTERM, sigquit);
  mysignal(SIGHUP, sigquit);
  mysignal(SIGINT, sigquit);
  mysignal(SIGQUIT, sigquit);
  mysignal(SIGPIPE, sigpipe);
  mysignal(SIGCHLD, sigchld);

  /* prepare to catch console output */
  if (console) {

    /* for any OS chr$(7) might cause endless loops if catched from console */
    visualbell = 1;

#ifdef __MINT__
    if ((cfh = open("/dev/xconout2", O_RDONLY)) < 0) {
      prints("wterm: can't catch console output, device busy or not present\r\n");
      console = 0;
    }
#elif defined(sun) || defined(linux)
    console = 0;                     /* data will come to normal pipe handle */
    ioctl(pipeh, TIOCCONS, 0);
#else
    prints("wterm: can't catch console output on this operating system\r\n");
    console = 0;
#endif
  }

#ifdef __NetBSD__
  prints("NetBSD\r\n\tOutput\r\n\t\tTest\r\n\r\n");
#endif
  if (debug) {
    w_trace(1);
  }

  term();
}
