/*
 * programs/wperfmon.c, part of W
 * (C) 94-02/96 by Torsten Scherer (TeSche)
 * itschere@techfak.uni-bielefeld.de
 *
 * performance monitor for W1
 */

#include <stdio.h>
#include <signal.h>
#include <unistd.h>

#include <Wlib.h>

#define FONT "fixed8.wfnt"
#define	REPS 5

int width, height;


#ifdef __MINT__

#include <mintbind.h>

long now()
{
  long stack, ret;

  stack = Super(0L);
  ret = *((long *)0x4ba) * 5;
  Super(stack);

  return ret;
}

#else

#include <sys/time.h>

long now()
{
  struct timeval tv;

  gettimeofday(&tv, NULL);

  return tv.tv_sec * 1000 + tv.tv_usec / 1000;
}

#endif


/*
 * something...
 */

WWIN *win;
short fw, fh, end;

static void sigalrm(int sig)
{
  end = 1;
}


/*
 * some tests
 */

static void test_null(short reps, long sec)
{
  short i;
  long vorher, l, suml;
  float s, sums;

  printf("testing null operation...\n");

  suml = 0;
  sums = 0;
  for (i=0; i<reps; i++) {
    vorher = now();
    l = 0;
    end = 0;
    alarm(sec);
    while (!end) {
      l++;
      w_null();
    }
    w_test(win, 0, 0);
    s = (now() - vorher) / 1000.0;
    printf("%li ops in %f seconds -> %f ops/sec\n", l, s, l / s);
    suml += l;
    sums += s;
  }

  printf("average: %f secs/op, %f ops/sec\n", sums / suml, suml / sums);
}


static void test_plot(short reps, long sec)
{
  short x, y, i;
  long vorher, l, suml;
  float s, sums;

  printf("testing plot...\n");

  suml = 0;
  sums = 0;
  for (i=0; i<reps; i++) {
    x = 0;
    y = 0;
    l = 0;
    end = 0;
    w_setmode(win, M_CLEAR);
    w_pbox(win, 0, 0, width, height);
    w_setmode(win, M_INVERS);
    w_flush();
    vorher = now();
    alarm(sec);
    while (!end) {
      w_plot(win, x, y);
      l++;
      if ((x += 13) >= width) {
	x -= width;
	if (++y == height) {
	  y = 0;
	}
      }
    }
    w_test(win, 0, 0);
    s = (now() - vorher) / 1000.0;

    printf("%li ops in %f seconds -> %f ops/sec\n", l, s, l / s);
    suml += l;
    sums += s;
  }

  printf("average: %f secs/op, %f ops/sec\n", sums / suml, suml / sums);
}


static void test_vline(short reps, long sec)
{
  short x, i;
  long vorher, l, suml;
  float s, sums;

  printf("testing vline...\n");

  suml = 0;
  sums = 0;
  for (i=0; i<reps; i++) {
    x = 0;
    l = 0;
    end = 0;
    w_setmode(win, M_CLEAR);
    w_pbox(win, 0, 0, width, height);
    w_setmode(win, M_INVERS);
    w_flush();
    vorher = now();
    alarm(sec);
    while (!end) {
      w_vline(win, x, 0, height-1);
      l++;
      if (++x == width) {
	x = 0;
      }
    }
    w_test(win, 0, 0);
    s = (now() - vorher) / 1000.0;
    printf("%li ops in %f seconds -> %f ops/sec\n", l, s, l / s);
    suml += l;
    sums += s;
  }

  printf("average: %f secs/op, %f ops/sec\n", sums / suml, suml / sums);
}


static void test_hline(short reps, long sec)
{
  short y, i;
  long vorher, l, suml;
  float s, sums;

  printf("testing hline...\n");

  suml = 0;
  sums = 0;
  for (i=0; i<reps; i++) {
    y = 0;
    l = 0;
    end = 0;
    w_setmode(win, M_CLEAR);
    w_pbox(win, 0, 0, width, height);
    w_setmode(win, M_INVERS);
    w_flush();
    vorher = now();
    alarm(sec);
    while (!end) {
      w_hline(win, 0, y, width-1);
      l++;
      if (++y == height) {
	y = 0;
      }
    }
    w_test(win, 0, 0);
    s = (now() - vorher) / 1000.0;
    printf("%li ops in %f seconds -> %f ops/sec\n", l, s, l / s);
    suml += l;
    sums += s;
  }

  printf("average: %f secs/op, %f ops/sec\n", sums / suml, suml / sums);
}


#define POLYPOINTS 9

static void test_poly(short reps, long sec)
{
  short i;
  long vorher, l, suml;
  float s, sums;
  short points[POLYPOINTS<<1] = {0, height>>1, width>>2, 0, width>>1, height>>2, (width>>2)*3, 0, width-1, height>>1, (width>>2)*3, height-1, width>>1, (height>>2)*3, width>>2, height-1, 0, height>>1};

  printf("testing ppoly...\n");

  suml = 0;
  sums = 0;
  for (i=0; i<reps; i++) {
    l = 0;
    end = 0;
    w_setmode(win, M_CLEAR);
    w_pbox(win, 0, 0, width, height);
    w_setmode(win, M_INVERS);
    w_flush();

    vorher = now();
    alarm(sec);
    while (!end) {
      w_ppoly(win, POLYPOINTS, points);
      l++;
    }
    w_test(win, 0, 0);
    s = (now() - vorher) / 1000.0;
    printf("%li ops in %f seconds -> %f ops/sec\n", l, s, l / s);
    suml += l;
    sums += s;
  }

  printf("average: %f secs/op, %f ops/sec\n", sums / suml, suml / sums);
}


static void test_line(short reps, long sec)
{
  short i, j;
  long vorher, l, suml;
  float s, sums;

  printf("testing line...\n");

  suml = 0;
  sums = 0;
  for (i=0; i<reps; i++) {
    l = 0;
    end = 0;
    w_setmode(win, M_CLEAR);
    w_pbox(win, 0, 0, width, height);
    w_setmode(win, M_INVERS);
    w_flush();
    vorher = now();
    alarm(sec);
    while (!end) {
      for (j=0; j<width; j++)
	w_line(win, j, 0, width-1-j, height-1);
      for (j=0; j<height; j++)
	w_line(win, 0, j, width-1, height-1-j);
      l += width + height;
    }
    w_test(win, 0, 0);
    s = (now() - vorher) / 1000.0;
    printf("%li ops in %f seconds -> %f ops/sec\n", l, s, l / s);
    suml += l;
    sums += s;
  }

  printf("average: %f secs/op, %f ops/sec\n", sums / suml, suml / sums);
}


static void test_circle(short reps, long sec)
{
  short x, y, i;
  long vorher, l, suml;
  float s, sums;

  printf("testing circle...\n");

  suml = 0;
  sums = 0;
  for (i=0; i<reps; i++) {
    x = 50;
    y = 50;
    l = 0;
    end = 0;
    w_setmode(win, M_CLEAR);
    w_pbox(win, 0, 0, width, height);
    w_setmode(win, M_INVERS);
    w_flush();
    vorher = now();
    alarm(sec);
    while (!end) {
      w_circle(win, x, y, 49);
      l++;
      if (++x >= width-50) {
	x = 50;
	if (++y >= height-50) {
	  y = 50;
	}
      }
    }
    w_test(win, 0, 0);
    s = (now() - vorher) / 1000.0;
    printf("%li ops in %f seconds -> %f ops/sec\n", l, s, l / s);
    suml += l;
    sums += s;
  }

  printf("average: %f secs/op, %f ops/sec\n", sums / suml, suml / sums);
}


static void test_pcircle(short reps, long sec)
{
  short x, y, i;
  long vorher, l, suml;
  float s, sums;

  printf("testing pcircle...\n");

  suml = 0;
  sums = 0;
  for (i=0; i<reps; i++) {
    x = 50;
    y = 50;
    l = 0;
    end = 0;
    w_setmode(win, M_CLEAR);
    w_pbox(win, 0, 0, width, height);
    w_setmode(win, M_INVERS);
    w_flush();
    vorher = now();
    alarm(sec);
    while (!end) {
      w_pcircle(win, x, y, 49);
      l++;
      if (++x >= width-50) {
	x = 50;
	if (++y >= height-50) {
	  y = 50;
	}
      }
    }
    w_test(win, 0, 0);
    s = (now() - vorher) / 1000.0;
    printf("%li ops in %f seconds -> %f ops/sec\n", l, s, l / s);
    suml += l;
    sums += s;
  }

  printf("average: %f secs/op, %f ops/sec\n", sums / suml, suml / sums);
}


static void test_bitblk(short reps, long sec)
{
  short i, j;
  long vorher, l, suml;
  float s, sums;

  printf("testing bitblk...\n");

  suml = 0;
  sums = 0;
  for (i=0; i<reps; i++) {
    w_setmode(win, M_CLEAR);
    w_pbox(win, 0, 0, width, height);
    w_setmode(win, M_INVERS);
    for (j=0; j<width; j++)
      w_line(win, j, 0, width-1-j, height-1);
    for (j=0; j<height; j++)
      w_line(win, 0, j, width-1, height-1-j);
    w_flush();
    l = 0;
    end = 0;
    vorher = now();
    alarm(sec);
    while (!end) {
      w_bitblk(win, 1, 0, width-1, height, 0, 0);
      l++;
    }
    w_test(win, 0, 0);
    s = (now() - vorher) / 1000.0;
    printf("%li ops in %f seconds -> %f ops/sec\n", l, s, l / s);
    suml += l;
    sums += s;
  }

  printf("average: %f secs/op, %f ops/sec\n", sums / suml, suml / sums);
}


static void test_scroll(short reps, long sec)
{
  short i, j;
  long vorher, l, suml;
  float s, sums;

  printf("testing vscroll...\n");

  suml = 0;
  sums = 0;
  for (i=0; i<reps; i++) {
    w_setmode(win, M_CLEAR);
    w_pbox(win, 0, 0, width, height);
    w_setmode(win, M_INVERS);
    for (j=0; j<width; j++)
      w_line(win, j, 0, width-1-j, height-1);
    for (j=0; j<height; j++)
      w_line(win, 0, j, width-1, height-1-j);
    w_flush();
    l = 0;
    end = 0;
    vorher = now();
    alarm(sec);
    while (!end) {
      w_vscroll(win, 0, 1, width, height-1, 0);
      l++;
    }
    w_test(win, 0, 0);
    s = (now() - vorher) / 1000.0;
    printf("%li ops in %f seconds -> %f ops/sec\n", l, s, l / s);
    suml += l;
    sums += s;
  }

  printf("average: %f secs/op, %f ops/sec\n", sums / suml, suml / sums);
}


static void test_printchar(short reps, long sec)
{
  short x, y, i;
  long vorher, l, suml;
  float s, sums;

  printf("testing character output...\n");

  suml = 0;
  sums = 0;
  for (i=0; i<reps; i++) {
    x = 0;
    y = 0;
    l = 0;
    end = 0;
    vorher = now();
    alarm(sec);
    while (!end) {
      w_printchar(win, x, y, l++ & 255);
      if ((x += fw) == width) {
	x = 0;
	if ((y += fh) == height) {
	  y = 0;
	}
      }
    }
    w_test(win, 0, 0);
    s = (now() - vorher) / 1000.0;
    printf("%li ops in %f seconds -> %f ops/sec\n", l, s, l / s);
    suml += l;
    sums += s;
  }

  printf("average: %f secs/op, %f ops/sec\n", sums / suml, suml / sums);
}


/*
 * something else...
 */

void main(void)
{
  WSERVER *wserver;
  WFONT *wfont;

#ifdef __MINT__
  signal(SIGALRM, sigalrm);
#else
  struct sigaction sa;

  sa.sa_handler = sigalrm;
  sa.sa_mask = 0;
#ifdef sun
  sa.sa_flags = 0;
#else
  sa.sa_flags = SA_RESTART;
#endif
  sigaction(SIGALRM, &sa, NULL);
#endif

  if (!(wserver = w_init())) {
    fprintf(stderr, "error: can't connect to wserver\n");
    exit(-1);
  }

  if ((wserver->width < width) || (wserver->height < height)) {
    fprintf(stderr, "error: screen size too small\n");
    exit(-1);
  }

  if (!(wfont = w_loadfont(FONT))) {
    fprintf(stderr, "error: can't load font\n");
    return;
  }

  fw = wfont->widths[0];
  fh = wfont->height;
  width = 80 * fw;
  height = 25 * fh;

  if (!(win = w_create(width, height, W_TOP | W_NOBORDER))) {
    fprintf(stderr, "error: can't create window\n");
    return;
  }

  if (w_open(win, 0, 0) < 0) {
    w_delete(win);
    fprintf(stderr, "error: can't open window\n");
    return;
  }

  w_setfont(win, wfont);

  test_null(REPS, 10);
  test_plot(REPS, 10);
  test_line(REPS, 10);
  test_vline(REPS, 10);
  test_hline(REPS, 10);
  test_poly(REPS, 10);
  test_bitblk(REPS, 10);
  test_scroll(REPS, 10);
  test_circle(REPS, 10);
  test_pcircle(REPS, 10);
  test_printchar(REPS, 10);

  w_delete(win);
}
