/*
 * programs/weyes.c, part of W
 * (C) 94-02/96 by TeSche (Torsten Scherer)
 * itschere@techfak.uni-bielefeld.de
 */

#include <math.h>
#include <stdio.h>
#include <Wlib.h>


typedef struct {
  short midx, midy;
  short pupx, pupy;
} EYE;


WWIN *win;
short winw, winh, rmax, rpup;
EYE lefteye, righteye;

void updateeye(EYE *eye)
{
  short	winx, winy, mx, my, r;
  short	dx, dy, px, py;
  float	w;

  w_querywindowpos(win, 0, &winx, &winy);
  w_querymousepos(NULL, &mx, &my);

  winx += eye->midx;
  winy += eye->midy;

  dx = mx - winx;
  dy = my - winy;

  /* angle measured clockwise, zero is down */

  if (!dy) {
    w = M_PI / 2;
  } else {
    w = atan((float)abs(dx) / abs(dy));
  }

  if ((dy >= 0) && (dx >= 0)) {
    w = -w;
  } else if ((dy < 0) && (dx < 0)) {
    w = M_PI - w;
  } else if ((dy < 0) && (dx >= 0)) {
    w = -M_PI + w;
  }

  /* calculate radius */

  if ((r = (short)sqrt(dx * dx + dy * dy)) > rmax) {
    r = rmax;
  }

  px = eye->midx - r * sin(w);
  py = eye->midy + r * cos(w);

  /* move the pupille */
  if ((px != eye->pupx) || (py != eye->pupy)) {
    w_pcircle(win, eye->pupx, eye->pupy, rpup);
    eye->pupx = px;
    eye->pupy = py;
    w_pcircle(win, eye->pupx, eye->pupy, rpup);
  }
}


void usage()
{
  fprintf(stderr, "usage: weyes [-h] [-g <geometry>]\n");
}


void main(long argc, char **argv)
{
  char *prg = *argv, *geo = NULL;
  short err, xp, yp, ewinw, ewinh;
  WEVENT *ev;
  WSERVER *wserver;

  err = 0;
  while (*++argv) {
    if (**argv == '-') {
      if (!*(*argv+2)) switch (*(*argv+1)) {

        case 'h':
	  err = 1;
	  break;

	case 'g':
	  if (!(geo = *++argv)) {
	    fprintf(stderr, "option requires an argument\n");
	    argv--;
	    err = 1;
	  }
	  break;

	default:
	  err = 1;
	  fprintf(stderr, "illegal option: %s\n", *argv);
      } else {
	err = 1;
	fprintf(stderr, "illegal option: %s\n", *argv);
      }
    } else {
      err = 1;
    }
  }

  if (err) {
    usage();
    exit(-1);
  }

  if (!(wserver = w_init())) {
    fprintf(stderr, "error: %s can't connect to wserver\n", prg);
    exit(-1);
  }

  winw = 80;
  winh = 40;
  xp = -32768;
  yp = -32768;
  if (geo) {
    scan_geometry(geo, &winw, &winh, &xp, &yp);
  }
  if (winw < 80) {
    winw = 80;
  }
  if (winh < 40) {
    winh = 40;
  }

  if (!(win = w_create(winw, winh, W_MOVE | W_TOP))) {
    fprintf(stderr, "error: %s can't create window\n", prg);
    exit(-1);
  }

  w_querywinsize(win, 1, &ewinw, &ewinh);

  if ((xp != -32768) && (yp != -32768)) {
    if (xp < 0) {
      xp = wserver->width - ewinw - xp;
    }
    if (yp < 0) {
      yp = wserver->height - ewinh - yp;
    }
  } else {
    xp = UNDEF;
    yp = UNDEF;
  }

  if (w_open(win, xp, yp) < 0) {
    w_delete(win);
    fprintf(stderr, "error: %s can't open window\n", prg);
    exit(-1);
  }

  lefteye.midx = lefteye.pupx = winw/4;
  lefteye.midy = lefteye.pupy = winh/2;
  righteye.midx = righteye.pupx = 3*winw/4;
  righteye.midy = righteye.pupy = winh/2;

  rmax = MIN(winh/2, winw/4) - 1;
  w_circle(win, lefteye.midx, lefteye.midy, rmax);
  w_circle(win, righteye.midx, righteye.midy, rmax);

  rpup = rmax / 5;
  rmax -= (rpup + 2);
  w_pcircle(win, lefteye.pupx, lefteye.pupy, rpup);
  w_pcircle(win, righteye.pupx, righteye.pupy, rpup);

  while (42) {

    updateeye(&lefteye);
    updateeye(&righteye);

    /* watch out for `exit' event */
    if ((ev = w_queryevent(NULL, NULL, NULL, 250))) {
      if ((ev->type == EVENT_GADGET) && (ev->key == GADGET_EXIT)) {
	w_close(win);
	w_delete(win);
	exit(0);
      }
    }
  }
}
