/*
 * programs/wclock.c, part of W
 * (C) 94-02/96 by Torsten Scherer (TeSche)
 * itschere@techfak.uni-bielefeld.de
 *
 * - Changed for W1R0 by TeSche 12/94
 *
 * - Changed by Eero 11/95
 *   - Uses a lookup table instead of sin & cos
 *   - calculations in fixed point instead of floats
 *   - Clock face scales to the window size
 *   - Shows only hours and minutes
 *   - Misc changes
 *
 * - adapted to W1R2 by TeSche
 */

#include <stdio.h>
#include <signal.h>
#include <time.h>
#include <Wlib.h>


typedef struct {
  short	hxe, hye;
  short	mxe, mye;
} POINTERS;


static WWIN *window;
static short midx, midy;
static short maxxr, maxyr;


/* offset and draw clock pointers */
static void draw(POINTERS *p)
{
  w_setmode(window, M_DRAW);
  w_line(window, midx, midy, midx + p->hxe, midy + p->hye);
  w_line(window, midx, midy, midx + p->mxe, midy + p->mye);
  w_pbox(window, midx - 1, midy - 1, 3, 3);
}

/* offset and clear the clock pointers */
static void clear(POINTERS *p)
{
  w_setmode(window, M_CLEAR);
  w_line(window, midx, midy, midx + p->hxe, midy + p->hye);
  w_line(window, midx, midy, midx + p->mxe, midy + p->mye);
}

/* calculate clock pointer offsets from the middle */
static void offset(short *x, short *y, short sixty)
{
  short sini[16] =			/* sin(PI / 60 * sixty) * 16000 */
  {
    0,     1672,  3327,  4944,  6508,  8000,  9405,  10706,
    11890, 12944, 13856, 14617, 15217, 15650, 15912, 16000
  };
  short index = sixty % 15;

  switch(sixty / 15)
  {
    case 3:
      *x = -sini[15-index];
      *y = -sini[index];
      break;
    case 2:
      *x = -sini[index];
      *y = sini[15-index];
      break;
    case 1:
      *x = sini[15-index];
      *y = sini[index];
      break;
    default:
      *x = sini[index];
      *y = -sini[15-index];
  }
  *x = (long)*x * maxxr / 16000;
  *y = (long)*y * maxyr / 16000;
}

/* update clock pointers */
static void update(struct tm *loc)
{
  static POINTERS p = {0, 0, 0, 0};

  /* clear old pointers */
  clear(&p);

  /* offset from the middle */
  offset(&p.hxe, &p.hye, (loc->tm_hour % 12) * 5 + (loc->tm_min + 6) / 12);
  offset(&p.mxe, &p.mye, loc->tm_min);

  /* shorten hour pointer */
  p.hxe = p.hxe * 2 / 3;
  p.hye = p.hye * 2 / 3;

  /* draw new pointers */
  draw(&p);
}

/* set globals and draw the clock 'numbers' */
static void draw_clock(short width, short height)
{
/* half_size / x offsets */
#define P_DIST   6
#define N_DIST  12	
  short i, xp, yp;  

  midx = width / 2;
  midy = height / 2;
  maxxr = midx - midx / N_DIST;
  maxyr = midy - midy / N_DIST;

  w_setmode(window, M_DRAW);

  for (i = 0; i < 60; i++)
  {
    offset(&xp, &yp, i);
    xp += midx;
    yp += midy;
    if(midx < 30 || midy < 30)
    {
      if(i % 5 == 0)
        w_plot(window, xp, yp);
    }
    else
      if (i % 5)
	w_plot(window, xp, yp);			/* minutes in-bitween */
      else
	w_pbox(window, xp - 1, yp - 1, 3, 3);	/* hours */
  }
  maxxr -= midx / P_DIST;
  maxyr -= midy / P_DIST;
}

/* parse command line arguments */
static void get_options(char *argv[], char **geo)
{
  short err = 0;

  while (*++argv)
  {
    if (**argv == '-')		/* an option? */
    {
      if (!*(*argv+2))		/* one letter? */
      {
        switch (*(*argv+1))
	{
	  case 'h':
	    err = 1;
	    break;

	  case 'g':
	    if (!(*geo = *++argv))
	    {
	      fprintf(stderr, "option requires an argument\n");
	      argv--;
	      err = 1;
	    }
	    break;

	  default:
	    err = 1;
	    fprintf(stderr, "illegal option: %s\n", *argv);
	}
      }
      else
      {
        err = 1;
	fprintf(stderr, "illegal option: %s\n", *argv);
      }
    }
    else
      err = 1;
  }
  if (err)
  {
    fprintf(stderr, "usage: wclock [-h] [-g <geometry>]\n");
    exit(-1);
  }
}

/* init the connection, open a clock window
 * and set window size and handle variables
 */
static void open_clock(char *argv[], short *width, short *height)
{
/* window style and minimum / default size */
#define PROPERTIES	(W_MOVE | EV_KEYS)
#define MIN_SIZE	30
#define DEF_SIZE	80

  WSERVER *wserver;
  char *geo = NULL;
  short ewidth, eheight, xp, yp;

  get_options(argv, &geo);

  if (!(wserver = w_init()))
  {
    fprintf(stderr, "%s: can't connect to wserver\n", *argv);
    exit(-1);
  }

  *width = -1;
  *height = -1;
  xp = -32768;
  yp = -32768;

  if (geo)
    scan_geometry(geo, width, height, &xp, &yp);

  /* limit size */
  if (*width < MIN_SIZE)
    if (*width <= 0)
      *width = DEF_SIZE;
    else
      *width = MIN_SIZE;

  if (*height < MIN_SIZE)
    if(*height <= 0)
      *height = DEF_SIZE;
    else
      *height = MIN_SIZE;

  if (!(window = w_create(*width, *height, PROPERTIES)))
  {
    fprintf(stderr, "%s: can't create window\n", *argv);
    exit(-1);
  }
  w_querywinsize(window, 1, &ewidth, &eheight);

  if ((xp != -32768) && (yp != -32768))
  {
    if (xp < 0)
      xp = wserver->width - ewidth - xp;
    if (yp < 0)
      yp = wserver->height - eheight - yp;
  }
  else
  {
    xp = UNDEF;
    yp = UNDEF;
  }

  if (w_open(window, xp, yp) < 0)
  {
    w_delete(window);
    fprintf(stderr, "%s: can't open window\n", *argv);
    exit(-1);
  }
}

static void terminate()
{
  w_close(window);
  w_delete(window);
  exit(0);
}

void main(int argc, char *argv[])
{
  short		width, height;
  WEVENT	*ev;
  struct tm	*loc;
  long		t;

  signal(SIGTTOU, SIG_IGN);

  open_clock(argv, &width, &height);
  draw_clock(width, height);

  /* set exit handlers */
  signal(SIGHUP,  terminate);
  signal(SIGINT,  terminate);
  signal(SIGQUIT, terminate);
  signal(SIGABRT, terminate);
  signal(SIGTERM, terminate);

  while(42)
  {
    t = time(0);
    loc = localtime(&t);
    update(loc);

    /* watch out for ESC / `exit' event while idling a min */
    if ((ev = w_queryevent(NULL, NULL, NULL, 60000)))
      if ((ev->type == EVENT_GADGET && ev->key == GADGET_EXIT) ||
          (ev->type == EVENT_KEY && (ev->key & 0xff) == '\033'))
        terminate();
  }
}
