/*  W-Ant
 *
 * Compile with something like (under MiNT):
 *   gcc -O2 -Wall -mbaserel ant.c -o want -lbW -lbsocket
 *
 *  (w) 1996 by Eero Tamminen, puujalka@modeemi.cs.tut.fi
 */

#include <Wlib.h>
#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <memory.h>

/* window defines */
#define WIN_PROPERTIES	(W_MOVE | W_TITLE | W_CLOSE | EV_MOUSE)
#define WIN_NAME	" W-Ant "
#define BLK_SIZE	8		/* size for an ant 'position' */
#define MIN_COLS	12
#define MIN_ROWS	12
#define DEF_COLS	40
#define DEF_ROWS	40

/* max. sequence lenght (= bits in seed variable) */
#define MAX_SEQ		32


/* prototypes */
int	parse(int argc, char *argv[],
	unsigned long *seed, unsigned char *lenght, int *rows, int *cols);
void	help(char *name);
int	walk(WWIN *win,
	unsigned long seed, unsigned char lenght, int cols, int rows);
WWIN	*win_init(int width, int height);

/* ---------------------------- Main ---------------------------------- */

int main (int argc, char *argv[])
{
  int
    cols = DEF_COLS,			/* walking area for the ant	*/
    rows = DEF_ROWS;
  unsigned char
    lenght = 6;				/* lenght of seq. (2->MAX_SEQ)	*/
  unsigned long
    seed = 38;				/* move sequence (rllrrl)	*/
  WWIN
    *win;

  if(parse(argc, argv, &seed, &lenght, &rows, &cols))
  {
    if (!(win = win_init(cols * BLK_SIZE, rows * BLK_SIZE)))
      return -1;
    if(!walk(win, seed, lenght, cols, rows))
      fprintf(stderr, "%s: unable to allocate array\n", *argv);
    w_delete(win);
  }
  else
    help(*argv);

  return 0;
}

int parse(int argc, char *argv[], unsigned long *seed, unsigned char *len,
	int *rows, int *cols)
{
  int idx = 0;
  char mark;

  while (++idx < argc)
    {
      mark = toupper(argv[idx][0]);		/* first character...	*/
      switch (mark)
	{
	case 'R':				/* new seed		*/
	case 'L':
	  *seed = 0; *len = 0;
	  while (*len < MAX_SEQ && (mark == 'R' || mark == 'L'))
	    {
	      *seed <<= 1;
	      if (mark == 'R')
		*seed |= 1;
	      (*len)++;
	      mark = toupper(argv[idx][*len]);
	    }
 	  if(*len < 2)
	    return 0;
	  break;
	case 'W':				/* screen width		*/
 	  *cols = MAX(MIN_COLS, atoi(&argv[idx][1]) & 0x1ff);
	  break;
	case 'H':				/* screen height	*/
 	  *rows = MAX(MIN_ROWS, atoi(&argv[idx][1]) & 0x1ff);
	  break;
	default:				/* unrecogniced option	*/
	  return 0;
	}
    }
  return 1;
}

void help(char *name)
{
  fprintf(stderr, "W - Ant (w) 1996 by Eero Tamminen.\n\n");
  fprintf(stderr, "An example: %s rllllr w40 h40\n", name);
  fprintf(stderr, "  rllllr  : Ant's turn sequence\n");
  fprintf(stderr, "  w40     : Set window width to 40 blocks\n");
  fprintf(stderr, "  h40     : Set window height to 40 blocks\n\n");
  fprintf(stderr, "Ant follows the rules:\n");
  fprintf(stderr, "  Block, over which ant goes, value is increased.\n");
  fprintf(stderr, "  Value starts from zero when it exceeds sequence lenght.\n");
  fprintf(stderr, "  The block value is an index into ant's turn sequence.\n");
  fprintf(stderr, "  In the sequence R = right and L = left turn.\n\n");
  fprintf(stderr, "(Minimum values are: rl w%d h%d)\n", MIN_COLS, MIN_ROWS);
}

/* --------------------- Do the Ant Walk ------------------------------ */

int walk(WWIN *win,
	unsigned long seed, unsigned char lenght, int cols, int rows)
{
  unsigned char **grid;			       /* 2-dim. array		*/
  unsigned char value, prev;
  int antx, anty, heading = 0;
  unsigned long move = 0;		       /* movement counter	*/
  WEVENT *ev = NULL;

  /* create and initialize the grid array				*/
  if(!(grid = (unsigned char **)malloc(sizeof(int) * cols)))
    return 0;
  for (antx = 0; antx < cols; antx++)
    {
      if(!(grid[antx] = (unsigned char *)malloc(sizeof(int) * rows)))
        return 0;
      for (anty = 0; anty < rows; anty++)
	grid[antx][anty] = 0;
    }
  antx = cols >> 1;				/* initial position	*/
  anty = rows >> 1;				/* at center		*/

  w_setmode(win, M_DRAW);
  do
    {
      prev = grid[antx][anty];			/* previous grid value	*/
      value = (prev + 1) % lenght;		/* new grid value	*/
      grid[antx][anty] = value;			/* store		*/

      /* 'color' according to value */
      w_setpattern(win, MAX_GRAYSCALES * value / (lenght-1));
      /* display at [antx, anty] */
      w_dpbox(win, antx*BLK_SIZE+1, anty*BLK_SIZE+1, BLK_SIZE-2, BLK_SIZE-2);

      /* turn according to seed sequence & (previous) grid value */
      if (seed & (1 << prev))
	heading++;
      else
	heading--;

      /* new ant position */
      switch (heading)
	{
	case 4:
	  heading = 0;
	case 0:
	  anty++;		/* down		*/
	  break;
	case 1:
	  antx--;		/* left		*/
	  break;
	case 2:
	  anty--;		/* up		*/
	  break;
	case -1:
	  heading = 3;
	case 3:
	  antx++;		/* right	*/
	  break;
	}

      /* check boundaries */
      if (antx >= cols)
	antx = 0;
      if (anty >= rows)
	anty = 0;
      if (antx < 0)
	antx = cols - 1;
      if (anty < 0)
	anty = rows - 1;

      if (!(++move & 15))
	ev = w_queryevent(NULL, NULL, NULL, 0);
    }
  /* flushing buffer at every move should slow it down enough... */
  while(!ev);

  /* free array alloc.	*/
  for (antx = 0; antx < cols; antx++)
    free (grid[antx]);
  free (grid);

  /* how many moves ant made before it was interrupted.. */
  return move;
}


/* ----------------------- W stuff --------------------------- */

/* initialize W connection and a output window */
WWIN *win_init(int width, int height)
{
  WSERVER *wserver;
  WWIN *win;

  if(!(wserver = w_init()))
  {
    fprintf(stderr, "can't connect to wserver\n");
    return 0;
  }

  if(!(win = w_create(width, height, WIN_PROPERTIES)))
  {
    fprintf(stderr, "can't create window\n");
    return 0;
  }
  if(w_open(win, UNDEF, UNDEF) < 0)
  {
    fprintf(stderr, "can't open window\n");
    return 0;
  }
  w_settitle(win, WIN_NAME);
  w_pbox(win, 0, 0, width, height);	/* black background */
  return win;
}
