/*
 * lib/socket.c, part of W
 * (C) 94-02/96 by Torsten Scherer (TeSche)
 * itschere@techfak.uni-bielefeld.de
 */

#include <stdio.h>
#include <unistd.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <sys/types.h>
#include <sys/time.h>
#include <signal.h>
#include <stdlib.h>
#include <netdb.h>
#include <string.h>

#include "Wlib.h"
#include "../server/config.h"
#include "../server/pakets.h"

#define	BUFSIZE		1024
#define	MAXWEVENTS	32


/*
 * some public global variables
 */

WWIN *WROOT = NULL;


/*
 * some private variables...
 */

static short _sh = -1;
static long _iptr = 0, _optr = 0;
static char _ibuf[BUFSIZE], _obuf[BUFSIZE];
static short _wevents, _wev_write, _wev_read;
static WEVENT _wevent[MAXWEVENTS];
int _trace = 0, _traceIndent = 0;


/*
 *	some private functions...
 */

char *_check_window(WWIN *ptr)
{
  if (!ptr) {
    return "no window";
  }

  if (ptr->magic != MAGIC_W) {
    return "window fault";
  }

  return NULL;
}


long _initialize()
{
  char	*wdisplay;

  if ((wdisplay = getenv("WDISPLAY"))) {

    struct hostent *he;
    struct sockaddr_in raddr;

    if (!(he = gethostbyname(wdisplay))) {
      fprintf(stderr,
	      "Wlib: can't resolve adress for server `%s'.\r\n", wdisplay);
      return -1;
    }

    if ((_sh = socket(AF_INET, SOCK_STREAM, 0)) < 0) {
      fprintf(stderr, "Wlib: can't create AF_INET socket.\r\n");
      return -1;
    }

    raddr.sin_family = AF_INET;
    raddr.sin_addr = *(struct in_addr *)he->h_addr_list[0];
    raddr.sin_port = htons(SERVERPORT);
    if (connect(_sh, (struct sockaddr *)&raddr, sizeof(struct sockaddr))) {
      fprintf(stderr, "Wlib: can't connect to server `%s' (%s).\r\n",
	      wdisplay, inet_ntoa(raddr.sin_addr));
      close(_sh);
      return -1;
    }

  } else {

    struct sockaddr raddr;

    if ((_sh = socket(AF_UNIX, SOCK_STREAM, 0)) < 0) {
      fprintf(stderr, "Wlib: can't create AF_UNIX socket.\r\n");
      return -1;
    }

    raddr.sa_family = AF_UNIX;
    strcpy(raddr.sa_data, "/tmp/wserver");
    if (connect(_sh, &raddr, sizeof(struct sockaddr))) {
      fprintf(stderr, "Wlib: can't connect to local server.\r\n");
      close(_sh);
      return -1;
    }
  }

  _wevents = _wev_write = _wev_read = 0;

  return 0;
}


void w_flush(void)
{
  long	ret;
  char	*ptr = _obuf;

  /*
   * shit, on the one hand side this is TCP and therefore
   * guaranteed to only return if either all data is written
   * or the connection is broken, but on the other hand side
   * this involves an OS call which may be interrupted by a
   * signal, so what can we do here?
   */

  while (_optr > 0) {

    if ((ret = write(_sh, ptr, _optr)) < 0) {
      perror("Wlib: _send_paket() returned");
      exit(-99);
    }

    ptr += ret;
    _optr -= ret;
  }

  /* _optr should now be zero */
}


/*
 * _send_paket(): this version will send only complete pakets. this
 * requires that each paket is smaller than our buffer, but that
 * assumption was already made earlier at other places, so what the heck.
 */

void _send_paket(PAKET *pptr)
{
  short	size = pptr->len;

  pptr->len = htons(pptr->len);
  pptr->type = htons(pptr->type);

  if (_optr + size >= BUFSIZE) {
    w_flush();
  }

  memcpy(_obuf + _optr, pptr, size);

  _optr += size;
}


/*
 * some stuff dealing with buffered WEVENTS, the push routine
 * also deals with net2host byte order problems and converting
 * between the internal and external event types
 */

static void _event_push(WEVENT *ptr)
{
  if (_wevents < MAXWEVENTS) {
    _wevents++;
    _wevent[_wev_write].win = ptr->win;
    _wevent[_wev_write].type = ntohs(ptr->type);
    _wevent[_wev_write].x = ntohs(ptr->x);
    _wevent[_wev_write].y = ntohs(ptr->y);
    _wevent[_wev_write].key = ntohl(ptr->key);
    if (++_wev_write == MAXWEVENTS) {
      _wev_write = 0;
    }
  } else {
    fprintf(stderr, "Wlib: event buffer overflow.\r\n");
  }
}


static WEVENT *_wevent_pop(void)
{
  WEVENT *ptr = NULL;

  if (_wevents) {
    _wevents--;
    ptr = &_wevent[_wev_read];
    if (++_wev_read == MAXWEVENTS) {
      _wev_read = 0;
    }
  }

  return ptr;
}


/*
 * routine for reading the socket and buffering event pakets,
 * returns when either the first non-event paket is found or the
 * buffer does not contain a complete paket. so if there's a non
 * event paket returned it may leave some other pakets in the
 * buffer. therefore there may be pakets to be processed without
 * select() saying there's anything on the socket, so be aware of
 * this!
 */

static PAKET *_scan_socket(void)
{
  long ret, again = 1;
  PAKET *pptr = NULL;
  short len, type;
  static PAKET paket;

  /* read as many bytes as possible... */

  if ((ret = read(_sh, &_ibuf[_iptr], BUFSIZE - _iptr)) < 1) {

    fprintf(stderr, "Wlib: connection broken, sending SIGPIPE to my own.\r\n");
    kill(getpid(), SIGPIPE);

    /* not reached? */
    fprintf(stderr, "Wlib: connection broken, how did we get here? DIE!!\r\n");
    exit(-999);
  }
  _iptr += ret;

  /* ...and analyse them */

  while (again) {
    if (_iptr >= 4) {
      pptr = (PAKET *)_ibuf;
      len = ntohs(pptr->len);
      type = ntohs(pptr->type);
      if (_iptr >= len) {
	if (type == PAK_EVENT) {
	  _event_push((WEVENT *)&((EVENTP *)pptr)->event);
	  _iptr -= len;
	  memcpy(_ibuf, &_ibuf[len], _iptr);
	  pptr = NULL;
	} else {
	  memcpy(&paket, pptr, len);
	  paket.len = len;
	  paket.type = type;
	  _iptr -= len;
	  memcpy(_ibuf, &_ibuf[len], _iptr);
	  pptr = &paket;
	  again = 0;
	}
      } else {
	again = 0;
	pptr = NULL;
      }
    } else {
      again = 0;
    }
  }

  return pptr;
}


/*
 * two routines for listening on the socket, one for waiting
 * for a return code and one for waiting for an event
 */

PAKET *_wait4paket(short type)
{
  fd_set mrfd;
  struct timeval tv, *tvp;
  PAKET *pptr = NULL;

  w_flush();

  while (!pptr) {
#ifdef __MINT__
    tv.tv_sec = -1;
    tv.tv_usec = 0;
    tvp = &tv;
#else
    tvp = NULL;
#endif
    FD_ZERO(&mrfd);
    FD_SET(_sh, &mrfd);
    select(FD_SETSIZE, &mrfd, NULL, NULL, &tv);
    if (FD_ISSET(_sh, &mrfd)) {
      pptr = _scan_socket();
    }
  }

  if (type != pptr->type) {
    fprintf(stderr, "Wlib: socket communication garbled, aborting.\r\n");
    w_exit();
    exit(-1);
  }

  return pptr;
}

WEVENT *_wait4event(fd_set *rfd,
		    fd_set *wfd,
		    fd_set *xfd,
		    long timeout,
		    short *numready)
{
  WEVENT *evptr = NULL;
  struct timeval tv, *tvp;
  short fds;
  PAKET *pptr;

  w_flush();

  if ((evptr = _wevent_pop())) {
    *numready = 0;
    return evptr;
  }

  tvp = &tv;
  if (timeout < 0) {
#ifdef __MINT__
    tv.tv_sec = -1;
    tv.tv_usec = 0;
#else
    tvp = NULL;
#endif
  } else {
    tv.tv_sec = timeout / 1000;
    tv.tv_usec = 1000 * (timeout - 1000 * tv.tv_sec);
  }

  /* need no local copies of the descriptor sets, w_queryevent() has set
   * up everything properly already
   */
  FD_SET(_sh, rfd);

  if ((fds = select(FD_SETSIZE, rfd, wfd, xfd, tvp)) < 0) {
    fprintf(stderr, "Wlib: select() failed in _wait4event().\r\n");
    fds = 0;
  }

  if (FD_ISSET(_sh, rfd)) {
    FD_CLR(_sh, rfd);
    if ((pptr = _scan_socket())) {
      fprintf(stderr, "Wlib: received non-event paket 0x%04x (len=%i) in _wait4event.\r\n", pptr->type, pptr->len);
    }
    fds--;
    evptr = _wevent_pop();
  }

  *numready = fds;

  return evptr;
}


/*
 *
 */

void _spaces(int arg)
{
  /* horribly inefficient... */
  while (arg > 0) {
    printf(" ");
    arg--;
  }
}
