/*
 * lib/img.c, part of Wlib, part of W
 * (C) 94-02/96 by Torsten Scherer (TeSche)
 * itschere@techfak.uni-bielefeld.de
 *
 * routines to read a monochrome IMG file as a BITMAP structure
 */

#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/stat.h>
#include "Wlib.h"


/*
 * IMG header definition
 */

typedef struct {
  ushort ver;
  ushort hdrlen;
  ushort planes;
  ushort patlen;
  ushort pixwidth;
  ushort pixheight;
  ushort scanwidth;
  ushort scanlines;
} IMGHDR;


/*
 * routines for loading a monochrome IMG picture
 */

static uchar *load(char *fname)
{
  struct stat st;
  uchar *ptr;
  short fh;

  if (stat(fname, &st)) {
    fprintf(stderr, "error: can't stat %s\n", fname);
    return NULL;
  }

  if (st.st_size < sizeof(IMGHDR)) {
    fprintf(stderr, "error: file too short, can't be an IMG picture\n");
    return NULL;
  }

  if (!(ptr = malloc(st.st_size))) {
    fprintf(stderr, "error: can't allocate memory for file\n");
    return NULL;
  }

  if ((fh = open(fname, O_RDONLY, 0)) < 0) {
    fprintf(stderr, "error: can't open file\n");
    free(ptr);
    return NULL;
  }

  if (read(fh, ptr, st.st_size) != st.st_size) {
    fprintf(stderr, "error: can't load file\n");
    close(fh);
    free(ptr);
    return NULL;
  }

  close(fh);
  return ptr;
}


static long decode(uchar *data, short scanbytes, short realbytes, ushort scanlines, ushort patlen, uchar *pic)
{
  short cnt, i, bytecnt;
  ushort j;
  uchar first, *repptr = data, repcnt = 0;

  while (scanlines--) {

    if (repcnt) {
      repcnt--;
      data = repptr;
    }

    /* is there a repetition header? */

    if (!*data && !*(data+1) && (*(data+2) == 0xff)) {
      data += 3;
      repcnt = *data++;
      repptr = data;
    }

    bytecnt = scanbytes;
    while (bytecnt) switch (first = *data++) {

      case 0: /* pattern run */
        cnt = *data++;
	for (i=0; i<cnt; i++)
	  for (j=0; j<patlen; j++)
	    *pic++ = *(data+j);
	data += patlen;
	bytecnt -= (cnt * patlen);
	break;

      case 128: /* bit string */
	cnt = *data++;
	for (i=0; i<cnt; i++)
	  *pic++ = *data++;
	bytecnt -= cnt;
	break;

      default: /* solid run */
	cnt = first & 127;
	if (first & 128)
	  for (i=0; i<cnt; i++)
	    *pic++ = 255;
	else
	  for (i=0; i<cnt; i++)
	    *pic++ = 0;
	bytecnt -= cnt;
    }

    if (scanbytes != realbytes)
      pic++;
  }

  return 0;
}


/*
 * what we're going to export...
 */

BITMAP *w_readimg(char *fname, short *width, short *height)
{
  uchar *file, *filedata, *picture;
  IMGHDR *imghdr;
  int dbpl;
  BITMAP *ret;

  if (!(ret = malloc(sizeof(BITMAP)))) {
    fprintf(stderr, "error: out of memory\n");
    return NULL;
  }

  if (!(file = load(fname))) {
    free(ret);
    return NULL;
  }

  imghdr = (IMGHDR *)file;
  if ((imghdr->ver != 1) || (imghdr->hdrlen < 8) || (imghdr->planes != 1)) {
    free(file);
    free(ret);
    fprintf(stderr, "error: this is no monochrome IMG picture\n");
    return NULL;
  }

  filedata = file + (2 * imghdr->hdrlen);

  /* enough memory to load picture word aligned */
  dbpl = ((imghdr->scanwidth + 31) & ~31) >> 3;
  if (!(picture = malloc(dbpl * imghdr->scanlines))) {
    free(file);
    free(ret);
    fprintf(stderr, "error: can't allocate memory for picture\n");
    return NULL;
  }

  ret->width = dbpl << 3;
  ret->height = imghdr->scanlines;
  ret->type = BM_PACKEDMONO;
  ret->unitsize = 4;
  ret->upl = ret->width >> 5;
  ret->planes = 1;
  ret->data = (ushort *)picture;

  if (decode(filedata, (imghdr->scanwidth + 7) >> 3, dbpl, imghdr->scanlines, imghdr->patlen, picture) < 0) {
    fprintf(stderr, "error: can't decode image data\n");
    free(picture);
    free(file);
    free(ret);
    return NULL;
  }

  free(file);

  *width = imghdr->scanwidth;
  *height = imghdr->scanlines;

  return ret;
}
