/*
 * Driver for different AM7990 LANCE based boards.
 * Currently supported:
 * PAMs EMEGA
 * PAMs VME
 * To be supported soon:
 * Riebl Card
 * TUM Board
 * These boards don't have their complete HW address in an NV memory, so we
 * need an ioctl to pass the HW address. That's not implemented yet.
 *
 * This driver has been developed and tested on a 16MHz ST. If you run in
 * on substantially faster machines a few wait()'s may be needed here and
 * there. Please contact me in case of success or failure...
 *
 * TO DOs:
 * - lance_probe() should really test if the LANCE is there. At the moment
 *   it simply crashes so its up to you only to install the driver if you
 *   really have the board...
 * - Other boards need the full 24 address bits - so handle them correctly!
 * - We need new IOCTLs to pass a hardware address for some boards which dont
 *   have the whole address in a ROM...
 *
 * On a 16MHz ST the driver reaches the following throughput:
 * ftp send:  ~ k/sec (ncftp 1.7.5 -> wu-ftpd 2.0)
 * ftp recv:  ~ k/sec (ncftp 1.7.5 <- wu-ftpd 2.0)
 * tcp send:  ~ k/sec (tcpcl -> tcpsv)
 * tcp recv:  ~ k/sec (tcpsv <- tcpcl)
 * nfs read:  ~ k/sec
 * nfs write: ~ k/sec
 *
 * 01/15/94, Kay Roemer.
 * 09/26/95, Torsten Lang.
 */

#include <string.h>
#include "config.h"
#include "netinfo.h"
#include "kerbind.h"
#include "atarierr.h"
#include "sockios.h"
#include "sockerr.h"
#include "buf.h"
#include "if.h"
#include "ifeth.h"
#include "lancemem.h"
#include "util.h"

#define offsetof(s,e) ((long)(&((s *)0)->e))

/*
 * 200 Hz timer
 */
#define HZ200 (*(volatile long *)0x4baL)

#define MIN_LEN 60

#define TIMEOUTCNT 400 /* 2s */
#define TIMEOUT	1000 /* 5s */

static PKTBUF *pkt_ring[RECVBUFFS];

void (*ihandler) (void) = 0;
static int pkt_recv;
static volatile int pkt_sent = 0;
static volatile TMD *tmd_sent;

static struct netif if_lance;

static long	lance_open	(struct netif *);
static long	lance_close	(struct netif *);
static long	lance_output	(struct netif *, BUF *, char *, short, short);
static long	lance_ioctl	(struct netif *, short, long);

static long	lance_probe	(struct netif *);
static long	lance_reset	(struct netif *);
static void	lance_install_ints (void);
static void	lance_int	(void);

static long
lance_open (nif)
	struct netif *nif;
{
	lance_reset (nif);
	return 0;
}

static long
lance_close (nif)
	struct netif *nif;
{
	return 0;
}

static long
lance_output (nif, buf, hwaddr, hwlen, pktype)
	struct netif *nif;
	BUF *buf;
	char *hwaddr;
	short hwlen, pktype;
{
	short len;
	BUF *nbuf;
	register volatile int *rdp;
	register volatile TMD *tmd;
	register long timeout;

	nbuf = eth_build_hdr (buf, nif, hwaddr, pktype);
	if (nbuf == 0) {
		++nif->out_errors;
		return ENSMEM;
	}
	len = nbuf->dend - nbuf->dstart;

	memcpy (PPKT, nbuf->dstart, len);

	if (len < MIN_LEN)
		len = MIN_LEN;
#ifdef PAMs_INTERN
	len = (len+1) & ~1;
#endif

	tmd = (TMD *)tmd_sent;

	if ((tmd->tmd1 & OWN) == OWN_CHIP) {
		buf_deref (nbuf, BUF_NORMAL);
		++nif->out_errors;
		return EINTRN;
	}

	rdp = RCP_RDP;
	pkt_sent = 0;
	if (!(*rdp & CSR0_TXON)) {
		buf_deref (nbuf, BUF_NORMAL);
		++nif->out_errors;
		return EINTRN;
	}

	tmd->tmd2 = -len;
	tmd->tmd3 = 0;
	tmd->tmd1 = (STP | ENP | OWN_CHIP);	/* start/end of packet */

	timeout = HZ200 + TIMEOUT;
	while (!pkt_sent && !(tmd->tmd1 & ERR) &&
	       !(tmd->tmd3 & ~0x3FF) && HZ200 - timeout < 0)
		;   /* wait till packet sent */

	buf_deref (nbuf, BUF_NORMAL);

	if (!pkt_sent && !(tmd->tmd1 & ERR) && !(tmd->tmd3 & ~0x3FF)) {
		++nif->out_errors;
		return ETIMEDOUT;
	}
	if ((tmd->tmd1 & ERR) || (tmd->tmd3 & ~0x3FF) || pkt_sent < 0) {
		++nif->collisions;
		return ETIMEDOUT;
	}
	return 0;	
}

static long
lance_ioctl (nif, cmd, arg)
	struct netif *nif;
	short cmd;
	long arg;
{
	switch (cmd) {
	case SIOCSIFNETMASK:
	case SIOCSIFFLAGS:
	case SIOCSIFADDR:
		return 0;

	case SIOCSIFMTU:
		/*
		 * Limit MTU to 1500 bytes. MintNet has alraedy set nif->mtu
		 * to the new value, we only limit it here.
		 */
		if (nif->mtu > ETH_MAX_DLEN)
			nif->mtu = ETH_MAX_DLEN;
		return 0;
	}
	return EINVFN;
}

static long
lance_pkt_init (void)
{
	short i;
	
	for (i = 0; i < RECVBUFFS; ++i)
		pkt_ring[i] = PPKT+(i+XMITBUFFS);
	return 0;
}

long
driver_init (void)
{
	static char message[100];

	strcpy (if_lance.name, "en");
	if_lance.unit = if_getfreeunit ("en");
	if_lance.metric = 0;
	if_lance.flags = IFF_BROADCAST;
	if_lance.mtu = 1500;
	if_lance.timer = 0;
	if_lance.hwtype = HWTYPE_ETH;
	if_lance.hwlocal.len =
	if_lance.hwbrcst.len = ETH_ALEN;

	if_lance.rcv.maxqlen = IF_MAXQ;
	if_lance.snd.maxqlen = IF_MAXQ;

	if_lance.open = lance_open;
	if_lance.close = lance_close;
	if_lance.output = lance_output;
	if_lance.ioctl = lance_ioctl;

	if_lance.timeout = 0;

	if_lance.data = 0;

	/*
	 * Tell upper layers the max. number of packets we are able to
	 * receive in fast succession.
	 */
	if_lance.maxpackets = 0;

	lance_pkt_init ();
	if (lance_probe (&if_lance) < 0) {
		c_conws ("lance: driver not installed.\r\n");
		return 1;
	}
	lance_install_ints ();

	if_register (&if_lance);

	sprintf (message, "LANCE driver v%s (en%d) (%x:%x:%x:%x:%x:%x)\r\n",
		"0.2",
		if_lance.unit,
		if_lance.hwlocal.addr[0],
		if_lance.hwlocal.addr[1],
		if_lance.hwlocal.addr[2],
		if_lance.hwlocal.addr[3],
		if_lance.hwlocal.addr[4],
		if_lance.hwlocal.addr[5]);
	c_conws (message);
	return 0;
}

static long
lance_reset (nif)
	struct netif *nif;
{
	return 0;
}

static long
lance_probe (nif)
	struct netif *nif;
{
	register volatile LNCMEM *mem;
	register volatile TMD	*md;
	register long i;
	register volatile int *rdp;
#ifdef PAMs_INTERN
	u_short volatile *eemem;
#endif
	char my_haddr[6];

	rdp = RCP_RDP;
	*(rdp+1) = CSR0;
	*rdp = CSR0_STOP;
	/*
	 * init data structures
	 */
		
#ifdef PAMs_INTERN
	eemem = (u_short *)RCP_MEMBOT;
	*(volatile char *)LANCE_EEPROM;
	memcpy (nif->hwbrcst.addr, "\377\377\377\377\377\377", ETH_ALEN);
	for (i=0 ; i<ETH_ALEN ; i++)
		nif->hwlocal.addr[i] = my_haddr[i] =
			((eemem[i<<1] & 0xf) << 4) | (eemem[(i<<1)+1] & 0xf); 
	*(volatile char *)LANCE_MEM;
#endif
	mem = (LNCMEM *)RCP_MEMBOT;
	mem->init.mode = 0;		/* mode word */
	for (i=0 ; i<ETH_ALEN ; i++)
		mem->init.haddr[i ^ 1] = my_haddr[i];

    	/*
    	 * logical address filter - receive all logical addresses
    	 */    	
	mem->init.laf[0] = 0xFFFFFFFFL;
	mem->init.laf[1] = 0xFFFFFFFFL;

	mem->init.rdrp.drp_lo = (u_short)offsetof(LNCMEM,rmd[0]);
#if 0
	/*
	 * receive ring descr. pointer
	 */
	((u_long)&(PRMD[0]));
#endif
	mem->init.rdrp.drp_hi = 0;
	mem->init.rdrp.len = RECVRLEN;
	
	mem->init.tdrp.drp_lo = (u_short)offsetof(LNCMEM,tmd);
#if 0
	/*
	 * transmit ring descr. pointer
	 */
	((u_long)&PTMD);
#endif
	mem->init.tdrp.drp_hi = 0;
	mem->init.tdrp.len = XMITRLEN;

	md = &mem->tmd;
	/*
	 * don't know address of packet to send
	 */
	md->ladr = (u_short)(PPKT);
	md->tmd1 = OWN_HOST;
	/*
	 * zero pkt size
	 */
	md->tmd2 = 0;
	md->tmd3 = 0;
	for (i=0; i< RECVBUFFS; i++) {
		(RMD *)md = mem->rmd+i;
		((RMD *)md)->ladr = (u_short)(0xFFFFL & (u_long)pkt_ring[i]);
		((RMD *)md)->rmd1 = OWN_CHIP;
		/*
		 * 2's complement (maximum size)
		 */
		((RMD *)md)->rmd2 = -MAXPKTLEN;
		((RMD *)md)->mcnt = 0;
	}
	tmd_sent = &mem->tmd;
	pkt_recv = 0;
	*(rdp+1) = CSR3;
#ifndef PAMs_INTERN
	*rdp = CSR3_BSWP;
#else
	*rdp = CSR3_BSWP | CSR3_ACON;
#endif
	*(rdp+1) = CSR2;
	*rdp = 0;
	*(rdp+1) = CSR1;
	*rdp = 0;
	*(rdp+1) = CSR0;
	*rdp = CSR0_STRT | CSR0_INIT;
	i = HZ200 + TIMEOUTCNT;
	while (!(*rdp & CSR0_IDON) && HZ200 - i < 0)
		;
	if ((*rdp & (CSR0_IDON | CSR0_ERR)) ^ CSR0_IDON) {
		*rdp = CSR0_STOP;
		ihandler = NULL;
		return(-1);
	}
	ihandler = lance_int;
	/*
	 * clear idon-bit
	 */
	*rdp = CSR0_IDON;
	/*
	 * enable interrupts
	 */
	*rdp = CSR0_INEA;
	/*
	 * RCP_RAP (=*rdp+1) must not be changed after initialition!
	 * Other routines assume that RCP_RAP is set to CSR0 which is
	 * in fact the only register that can be read when LANCE is running.
	 */
	return(0);
}

/*
 * LANCE interrupt routine
 */
void
lance_int (void)
{
	register int		i;
	register u_short	csr0;
	register int		type;

	register volatile int	*rdp;
	register volatile TMD	*md;
	register          PKTBUF *pkt;

	struct netif *nif = &if_lance;
	BUF *buf;
	int pktlen;

	/*
	 * register data port
	 */
	rdp = RCP_RDP;
	csr0 = *rdp;
	if (!(csr0 & CSR0_INTR))
		return;
	
	if (csr0 & CSR0_IDON)
		*(rdp) = CSR0_IDON | CSR0_INEA;
	else if (csr0 & CSR0_RINT) {
		*rdp = CSR0_RINT | CSR0_INEA;
		for (i=0; i<RECVBUFFS; i++ ) {
			(RMD *)md  = PRMD+i;
			if ((((RMD *)md)->rmd1 & OWN) == OWN_CHIP)
				continue;
			/*
			 * packet ok ?
			 */
			if ((STP|ENP) == (((RMD *)md)->rmd1 & (ERR|STP|ENP))) {
				pkt = pkt_ring[i];
				type = pkt->et_type;

				/*
				 * packet length without checksum
				 */
				pktlen = ((RMD *)md)->mcnt - 4;
				buf = buf_alloc (pktlen+100, 50, BUF_ATOMIC);
				if (buf == 0)
					++nif->in_errors;
				else {
					buf->dend = buf->dstart + pktlen;
					memcpy (buf->dstart, pkt, pktlen);
					if (!if_input (nif, buf, 0,
						       eth_remove_hdr (buf)))
						++nif->in_packets;
					else
						++nif->in_errors;
				}
			} else {
			  	if (((RMD *)md)->rmd1 & ERR) {
			  		++nif->in_errors;
					lance_reset (nif);
					return;
				}
			}
			/*
			 * free packet
			 */
			((RMD *)md)->mcnt = 0;
			/*
			 * give packet back to lance
			 */
			((RMD *)md)->rmd1 = OWN_CHIP;
		}
	} else if (csr0 & CSR0_TINT) {
		/*
		 * clear interrupt bit & reenable
		 */ 
		*rdp = CSR0_TINT | CSR0_INEA;
		md = (TMD *)tmd_sent;
		/*
		 * packet given back to host ?
		 */
		if ((md->tmd1 & OWN) == OWN_HOST) {
			if ((md->tmd1) & ERR) {
				pkt_sent = -1;
				++nif->out_errors;
			} else {
				pkt_sent = 1;
				++nif->out_packets;
				if (md->tmd1 & (MORE | ONE | DEF)) 
					++nif->collisions;
			}
		} else
			++nif->out_errors;
	} else if (csr0 & CSR0_ERR) {
		++nif->in_errors;
		*rdp = CSR0_CERR | CSR0_MERR | CSR0_BABL | CSR0_MISS | CSR0_INEA;
	}
}

extern void lance_hbi_int (void);
extern void lance_v5_int (void);
extern void lance_vbi_int (void);
extern long old_vbi;

/*
 * Install interrupt handler
 */
static void
lance_install_ints (void)
{
	long old_v5, old_hbl;
	short sr;

	sr = spl7 ();
	old_v5 = s_etexec ((short)LANCEIVEC, (long)lance_v5_int);
#ifdef PAMs_INTERN
	/*
         * This is a real dirty hack but seems to be necessary for the MiNT
         * environment. Even with a correct hbi handler installed I found an
         * IPL of 3 quite often. This is definitely a misbehaviour of the
         * MiNT environment since the hbi handler works under plain TOS (what
         * it is supposed to do since Atari guarantees an IPL of 0 at start
         * time of a program).
         * Since the PAMs EMEGA boards run at an unchangeable IPL of 3 it is
         * ABSOLUTELY necessary that the OS and the application programs
         * behave correct!
         * So someone with the knowledge and enough time should take a look
         * at the various MiNT sources, filesystems etc. to find the bug...
         * When these bug(s) are fixed the vbi can stay unchanged!
         */

	old_vbi = s_etexec ((short)HBI+2, (long)lance_vbi_int);

	*(char *)LANCEIVECREG = LANCEIVEC;
	old_hbl = s_etexec ((short)HBI, (long)lance_hbi_int);
#endif
	spl (sr);
}
