/*
 * This file belongs to FreeMiNT.  It's not in the original MiNT 1.12
 * distribution.  See the file Changes.MH for a detailed log of changes.
 */

/*
 * This file is dedicated to the FreeMiNT project.
 * It's not allowed to use this file for other projects without my
 * explicit permission.
 */

/*
 * begin:	1998-02-01
 * last change:	1998-09-08
 * 
 * Author: Frank Naumann - <fnaumann@cs.uni-magdeburg.de>
 * 
 * please send suggestions, patches or bug reports to me or
 * the MiNT mailing list
 * 
 * 
 * changes since last version:
 * 
 * 1998-09-25:	(v1.00)
 * 
 * - add: lcase flag to force a strlwr operation on real 8+3 names
 *        -> make_shortname/readdir/fatfs_config modified
 * - fix: some modifications to reduce overkill syncing for small
 *        fatfs_write calls in writethrough mode
 * - fix: bug in lseek (SEEK_END)
 *        negative offsets are allowed, the compendium is wrong
 * 
 * 1998-09-09:
 * 
 * - fix: bug in symlink (idiotic implementation from MagiC)
 * - some name changes related to block_IO restructurization
 * - add: val_bpb - check the xpbp
 * - fix: delete_cookie: missing check for 'cookie->links'
 * - fix: missing increment/decrement for 'cookie->links' in open/close
 * 
 * 1998-08-06:	(v0.99)
 * 
 * - new: free cluster counter for every drive (speedup dfree)
 *        -> changed dfree?? to ffree?? (fat free)
 *        -> modified nextcl (decrease freecl counter)
 *        -> modified del_chain (increase freecl counter)
 * 
 * Rainer Seitel:
 * - change: replace t_getdate()/t_gettime() by datestamp/timestamp
 * - fix: getcl12 and fixcl12 unaligned word access (68000)
 * 
 * - fix: FAT syncing if mirroring is enabled
 * - fix: missing breaks in Dcntl (FS_INFO)
 * - fix: some missing const in function declaration
 * - change: all drv variables are now of type ushort (before long)
 * - change: better hash function
 * - new: get_bpb - remove completly use of getbpb()
 *        all boot sectors are handled bye the xfs itself now
 * - fix: get_devinfo - better start lastalloc
 * - fix: next_free32 - little bug in overflow calculation
 * - fix: fixcl32 - bug in updating more than 2 FATs
 * 
 * 1998-07-23:	(v0.98)
 * 
 * - change: dskchng now support new method
 * - new: start alloc of clusters on FAT32 at cluster 16
 *        (avoid fragmentation of the root dir)
 * - fix: get_bpb: FAT32 calculation was not correct
 * 
 * Rainer Seitel:
 * - fix: readlabel/writelabel (detection of a label)
 * - fix: getxattr: ctime was incorrect set
 * - fix: chmode: directories can now be hidden
 * - fix: dpathconf (DP_XATTRFIELDS) now doesn't return DP_RDEV flag
 * - fix: functions return now konsequent E_OK 
 * 
 * - add: a little fix for A/B cache consistence
 * - add: Dcntl opcode 'FS_INFO' (experimental!)
 * 
 * 1998-07-13:	(v0.97)
 * 
 * - new: remove sync buffercache (now global in s_ync)
 * - new: writelabel create label if it doesn't exist
 * - fix: is_shortname -> double points are not detected
 * 
 * 1998-07-08:	(v0.96)
 * 
 * - new: some opcodes
 * - change: Dcntl opcodes
 * - fix: correct dostrunc on FAT
 * - fix: getcl32/fixcl32/nextcl32 - highest 4 bits are reserved
 * - new: DOS media detection (include FAT32)
 *        (own getbpb)
 * - change: adaption for new block low level I/O
 * - change: adaption for new buffer cache
 * 
 * 1998-07-01:	(v0.95)
 *
 * - fix: little bug in getxattr (CLUSTSIZE and not CLSIZE)
 * 
 * - fix: little bug in nextcl16/32 (last free clusters are not found)
 * - fix: bug in lseek
 * - new: __updatedir to mark the actual dir entry as modified
 * - change: new strategie for extend directorys
 * - new: some Dcntl's
 * - new: ISO, GEMDOS and MSDOS name modus
 *   configurable for every drive
 * 
 * - fix: update of FA_CHANGED and mdate/mtime correct
 * - add: FTRUNCATE on fatfs_ioctl (not tested)
 * - add: FUTIME on fatfs_ioctl (not tested)
 * - change: FIONREAD -> read direct the values
 * - change: pathconf (korrekt return DP_CASEINS, DP_NOTRUNC)
 * - fix: make_shortname (every long VFAT entry get a ~)
 * - fix: make_shortname (now up to 6 digits for the number after ~)
 * - fix: fatfs_close (archivbit is not set on symlinks now)
 * - fix: rename (check for existing file now)
 * 
 * - change: replace makros by inline functions
 *   (better typecheck and automatic cast)
 * - change: dir2str, str2dir (reduce a variable, more logical parameter)
 * 
 * - new: optimized search_cookie -> really speed improvement
 * - new: is_short (determine a FAT or VFAT entry)
 *   --> change: make_shortname (faster)
 *   --> change: fatfs_lookup (faster)
 * - new: faster SWAP68_W and SWAP68_L (GCC inline assembler)
 * - new: complete file name trunc (make_shortname)
 * - new: symlink is configurable for every drive
 *   --> change: fatfs_symlink, fatfs_pathconf
 * 
 * 
 * known bugs:
 * 
 * - dskchng() doesn't free the dynamically used memory for the drive
 *   (LOCKS & FILE structs) if open files are on an invalidated medium
 *   (minor)
 * 
 * todo:
 * 
 * - Fxattr() -> better values for subdirectories (nblocks, nlinks, ...)
 * - writelabel/readlabel -> update bootsector label on DOS 4.0 mediums
 * - real file locking (minor)
 * 
 * optimizations to do:
 * 
 * - __FIO -> L_BS, linear algorithm?
 * - fatfs_symlink/fatfs_readlink (minor)
 * - make_cookie
 * - fatfs_rename
 * 
 */

# include "fatfs.h"
# include "block_IO.h"

# include "global.h"

# include "biosfs.h"
# include "filesys.h"
# include "memory.h"
# include "proc.h"
# include "util.h"
# include "unicode.h"

# include "welcome.h"


/*
 * internal version
 */

# define FATFS_MAJOR	1
# define FATFS_MINOR	00
# define FATFS_STATUS	

# define FATFS_VERSION	str (FATFS_MAJOR) "." str (FATFS_MINOR) str (FATFS_STATUS) 
# define FATFS_DATE	__DATE__

# define FATFS_BOOT	\
	"\033pFAT/VFAT/FAT32 filesystem version " FATFS_VERSION "\r\n"

# define FATFS_GREET	\
	"\033q " FATFS_DATE " by Frank Naumann.\r\n\r\n"


/*
 * debugging stuff
 */

# ifdef DEBUG_INFO
# define FS_DEBUG 1
# define FS_DEBUG_COOKIE 0
# define FS_LOGFILE "u:\\ram\\fat.log"
# define FS_DUMPFILE "u:\\ram\\fshashtab.dmp"
# endif

/****************************************************************************/
/* BEGIN tools */

# define COOKIE_EQUAL(c1, c2) \
	(((c1)->index == (c2)->index) && ((c1)->dev == (c2)->dev))

/*
 * wp???_intel(): read/write 16 bit byte swapped from/to odd addresses.
 */

FASTFN ushort
WPEEK_INTEL (register uchar *ptr)
{
	register ushort c1 = *ptr++;
	register ushort c2 = *ptr;
	return ((c2 << 8) | (c1 & 0x00ff));
}

FASTFN void
WPOKE_INTEL (register uchar *ptr, register ushort value)
{
	*ptr++ = (uchar) (value & 0x00ff);
	*ptr = value >> 8;
}

/* END tools */
/****************************************************************************/

/****************************************************************************/
/* BEGIN definition part */

/*
 * filesystem
 */

static long	fatfs_root	(int drv, fcookie *fc);

static long	fatfs_lookup	(fcookie *dir, const char *name, fcookie *fc);
static DEVDRV *	fatfs_getdev	(fcookie *fc, long *devsp);
static long	fatfs_getxattr	(fcookie *fc, XATTR *xattr);

static long	fatfs_chattr	(fcookie *fc, int attrib);
static long	fatfs_chown	(fcookie *fc, int uid, int gid);
static long	fatfs_chmode	(fcookie *fc, unsigned mode);

static long	fatfs_mkdir	(fcookie *dir, const char *name, unsigned mode);
static long	fatfs_rmdir	(fcookie *dir, const char *name);
static long	fatfs_creat	(fcookie *dir, const char *name, unsigned mode, int attrib, fcookie *fc);
static long	fatfs_remove	(fcookie *dir, const char *name);
static long	fatfs_getname	(fcookie *root, fcookie *dir, char *pathname, int size);
static long	fatfs_rename	(fcookie *olddir, char *oldname, fcookie *newdir, const char *newname);

static long	fatfs_opendir	(DIR *dirh, int flags);
static long	fatfs_readdir	(DIR *dirh, char *nm, int nmlen, fcookie *);
static long	fatfs_rewinddir	(DIR *dirh);
static long	fatfs_closedir	(DIR *dirh);

static long	fatfs_pathconf	(fcookie *dir, int which);
static long	fatfs_dfree	(fcookie *dir, long *buf);
static long	fatfs_writelabel(fcookie *dir, const char *name);
static long	fatfs_readlabel	(fcookie *dir, char *name, int namelen);

static long	fatfs_symlink	(fcookie *dir, const char *name, const char *to);
static long	fatfs_readlink	(fcookie *file, char *buf, int len);
static long	fatfs_hardlink	(fcookie *fromdir, const char *fromname, fcookie *todir, const char *toname);
static long	fatfs_fscntl	(fcookie *dir, const char *name, int cmd, long arg);
static long	fatfs_dskchng	(int drv, int mode);

static long	fatfs_release	(fcookie *fc);
static long	fatfs_dupcookie	(fcookie *dst, fcookie *src);
static long	fatfs_sync	(void);

FILESYS fatfs_filesys =
{
	(FILESYS *) 0,
	/*
	 * FS_KNOPARSE		kernel shouldn't do parsing
	 * FS_CASESENSITIVE	file names are case sensitive (only for TOS-DOMAIN)
	 * FS_NOXBIT		if a file can be read, it can be executed
	 * FS_LONGPATH		filesystem understands "size" argument to "getname"
	 * FS_NO_C_CACHE	don't cache cookies for this filesystem
	 * FS_DO_SYNC		filesystem has a sync function
	 * FS_OWN_MEDIACHANGE	filesystem control self media change (dskchng)
	 */
	FS_CASESENSITIVE | FS_NOXBIT | FS_LONGPATH | FS_NO_C_CACHE | FS_DO_SYNC | FS_OWN_MEDIACHANGE,
	fatfs_root,
	fatfs_lookup, fatfs_creat, fatfs_getdev, fatfs_getxattr,
	fatfs_chattr, fatfs_chown, fatfs_chmode,
	fatfs_mkdir, fatfs_rmdir, fatfs_remove, fatfs_getname, fatfs_rename,
	fatfs_opendir, fatfs_readdir, fatfs_rewinddir, fatfs_closedir,
	fatfs_pathconf, fatfs_dfree, fatfs_writelabel, fatfs_readlabel,
	fatfs_symlink, fatfs_readlink, fatfs_hardlink, fatfs_fscntl, fatfs_dskchng,
	fatfs_release, fatfs_dupcookie,
	fatfs_sync
};

/*
 * device driver
 */

static long	fatfs_open	(FILEPTR *f);
static long	fatfs_write	(FILEPTR *f, const char *buf, long bytes);
static long	fatfs_read	(FILEPTR *f, char *buf, long bytes);
static long	fatfs_lseek	(FILEPTR *f, long where, int whence);
static long	fatfs_ioctl	(FILEPTR *f, int mode, void *buf);
static long	fatfs_datime	(FILEPTR *f, ushort *time, int rwflag);
static long	fatfs_close	(FILEPTR *f, int pid);

DEVDRV fatfs_device =
{
	fatfs_open,
	fatfs_write, fatfs_read, fatfs_lseek, fatfs_ioctl, fatfs_datime,
	fatfs_close,
	null_select, null_unselect
};


/*
 * internal global data definitions
 */

/* FAT boot sector */
typedef struct
{
	uchar	boot_jump[3];	/* Boot strap short or near jump */
	char	system_id[8];	/* Name - can be used to special case partition manager volumes */
	
	uchar	sector_size[2];	/* bytes per logical sector */
	uchar	cluster_size;	/* sectors/cluster */
	ushort	reserved;	/* reserved sectors */
	uchar	fats;		/* number of FATs */
	uchar	dir_entries[2];	/* root directory entries */
	uchar	sectors[2];	/* number of sectors */
	uchar	media;		/* media code (unused) */
	ushort	fat_length;	/* sectors/FAT */
	ushort	secs_track;	/* sectors per track */
	ushort	heads;		/* number of heads */
	ulong	hidden;		/* hidden sectors (unused) */
	ulong	total_sect;	/* number of sectors (if sectors == 0) */
	
} _F_BS;

/* FAT volume info */
typedef struct
{
	uchar	drive_number;	/* BIOS drive number */
	uchar	RESERVED;	/* Unused */
	uchar	ext_boot_sign;	/* 0x29 if fields below exist (DOS 3.3+) */
	
# define EXT_INFO		0x29
	
	uchar	vol_id[4];	/* Volume ID number */
	char	vol_label[11];	/* Volume label */
	char	fs_type[8];	/* Typically FAT12, FAT16, or FAT32 */
	
} _F_VI;

/* FAT32 boot sector */
typedef struct
{
	_F_BS	fbs;		/* normal FAT boot sector */
	
	ulong	fat32_length;	/* sectors/FAT */
	ushort	flags;		/* bit 8: fat mirroring, low 4: active fat */
	
# define FAT32_ActiveFAT_Mask	0x0f
# define FAT32_NoFAT_Mirror	0x80
	
	uchar	version[2];	/* major, minor filesystem version */
	ulong	root_cluster;	/* first cluster in root directory */
	ushort	info_sector;	/* filesystem info sector */
	ushort	backup_boot;	/* backup boot sector */
	
# define INVALID_SECTOR		0xffff
	
	ushort	RESERVED2[6];	/* Unused */
	
} _F32_BS;

/* FAT32 boot fsinfo */
typedef struct
{
	ulong	reserved1;	/* Nothing as far as I can tell */
	ulong	signature;	/* 0x61417272L */
	
# define FAT32_FSINFOSIG	0x61417272L
	
	ulong	free_clusters;	/* Free cluster count.  -1 if unknown */
	ulong	next_cluster;	/* Most recently allocated cluster. Unused under Linux. */
	ulong	reserved2[4];
	
} _FAT32_BFSINFO;

/* fat entry structure */
typedef struct
{
	char	name[11];	/* short name */
	char	attr;		/* file attribut */
	uchar	lcase;		/* used by Windows NT and Linux */
	uchar	ctime_ms;	/* creation time milliseconds */
	ushort	ctime;		/* creation time */
	ushort	cdate;		/* creation date */
	ushort	adate;		/* last access date */
	ushort	stcl_fat32;	/* the 12 upper bits for the stcl */
	ushort	time;		/* last modification time */
	ushort	date;		/* last modification date */
	ushort	stcl;		/* start cluster */
	ulong	flen;		/* file len */
	
} _DIR; /* 32 byte */

/* vfat entry structure */
typedef struct
{
	uchar	head;		/* bit 0..4: number of slot, bit 6: endofname */
	uchar	name0_4[10];	/* 5 unicode character */
	char	attr;		/* attribut (0x0f) */
	uchar	unused;		/* not used, reserved, (= 0) */
	uchar	chksum;		/* checksum short name */
	uchar	name5_10[12];	/* 6 unicode character */
	ushort	stcl;		/* start cluster (must be 0) */
	uchar	name11_12[4];	/* 2 unicode character */
	
} LDIR; /* 32 byte */

/* binary check */
/*# if (sizeof (_DIR) != 32 || sizeof (LDIR) != 32)
# error "fatfs.c: Internal error, compiler problem (sizeof (_DIR) != 32)."
# endif
*/

/*
 * file attributes
 */

# define FA_RDONLY	0x01	/* write protected */
# define FA_HIDDEN	0x02	/* hidden */
# define FA_SYSTEM	0x04	/* system */
# define FA_LABEL	0x08	/* label */
# define FA_DIR		0x10	/* subdirectory */
# define FA_CHANGED	0x20	/* archiv bit */
# define FA_VFAT	0x0f	/* VFAT entry */
# define FA_SYMLINK	0x40	/* symbolic link (MagiC like) */

# define FA_TOSVALID	(FA_RDONLY | FA_HIDDEN | FA_SYSTEM | FA_DIR | FA_CHANGED)


# define FAT_PATHMAX	129	/* include '\0' */
# define FAT_NAMEMAX	13	/* include '.' and '\0' */

/* this informations are from Microsoft */
# define VFAT_PATHMAX	260	/* include '\0' */
# define VFAT_NAMEMAX	256	/* include '\0' */


typedef struct cookie COOKIE;

struct cookie
{
	COOKIE	*next;		/* internal usage */
	char 	*name;		/* full pathname (own alloc) */
	long	links;		/* 'in use' counter */
	ushort	dev;		/* the device on which is the file */
	ushort	rdev;		/* not used at the moment */
	long	dir;		/* the start cluster of the dir */
	long	offset;		/* the offset in the dir */
	long	stcl;		/* the start cluster */
	_DIR	info;		/* the direntry */
	FILEPTR	*open;		/* linked list of opened file ptr (kernel alloc) */
	LOCK 	*locks;		/* linked list of locks on this file (own alloc) */
	char	*extra;		/* extra data, must be kmalloc'ed */
	char	slots;		/* number of VFAT slots */	
};

static COOKIE dummy_cookie =
{
	NULL, "\0", 0, 0, 0, 0, 0, 1,
	{{0}, FA_DIR, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
	NULL, NULL, NULL, 0
};

/* internal eXtended bpb */
typedef struct
{
	ulong	recsiz;		/* bytes per sector */
	ulong	clsiz;		/* sectors per cluster */
	ulong	clsizb;		/* bytes per cluster */
	
	ulong	numcl;		/* total number of clusters */
	
	ulong	rdrec;		/* root directory rec or stcl if FAT32 */
	ulong	rdlen;		/* root directory size or 0 if FAT32 */
	ulong	datrec;		/* first data sector */
	
	ulong	fsiz;		/* size of a FAT */
	ulong	fatrec;		/* startsector of !*first*! FAT */
	short	fats;		/* number of additional FATs (0 = 1 FAT, 1 = 2 FATs, ...) */
	short	ftype;		/* type of FAT */
	
/* ftype: */
# define FAT_INVALID	-1
# define FAT_TYPE_12	2
# define FAT_TYPE_16	0
# define FAT_TYPE_32	1
	
	/* special for FAT32 */
	ushort	fflag;		/* FAT flag, contain mirror or active FAT status */
	ushort	info;		/* FAT32 info sector, 0 == doesn't exist */
	ushort	version;	/* major/minor version */
	
} _x_BPB;

/* device info structure */
typedef struct
{
	DI *	di;		/* device identifikator for this drive */
	
	ulong	recsiz;		/* bytes per sector */
	ulong	clsiz;		/* sectors per cluster */
	ulong	clsizb;		/* bytes per cluster */
	ulong	fstart;		/* fat start */
	ulong	flen;		/* fat len in sectors */
	ulong	rdstart;	/* root dir start */
	ulong	rdlen;		/* root dir len in sectors  */
	ulong	rdentrys;	/* root dir entry's */
	ulong	dstart;		/* 1st cluster sector number */
	ulong	doffset;	/* clusteroffset */
	ulong	numcl;		/* total number of clusters */
	ulong	maxcl;		/* highest clnumber + 1 */
	ulong	entrys;		/* number of dir entrys in a cluster */
	
	long	(*getcl)(register long, const ushort, ulong);
	long	(*fixcl)(register long, const ushort, long);
	long	(*ffree)(const ushort);
	
	COOKIE	*root;		/* the root COOKIE for this drive */
	short	ftype;		/* the type of the fat */
	short	fat2on;		/* is there an active second FAT? */
	ulong	lastcl;		/* the last allocated cluster */
	long	freecl;		/* free cluster counter, -1 if unknown */
	
	/* FAT32 extensions */
	short	info;		/* optional info sector */
	short	info_active;	/* update info sector flag */
	short	fmirroring;	/* status of fat mirroring */
	short	actual_fat;	/* active fat if fat mirroring is disabled */
	
} DEVINFO;

/* extended open directory descriptor */
typedef struct
{
	ushort	dev;		/* device */
	ushort	rdev;		/* not used at the moment */
	long	stcl;		/* start cluster */
	long	actual;		/* actual cluster (sector number) */
	long	cl;		/* cluster number */
	long	index;		/* logical index */
	long	real_index;	/* index there normal & vfat point */
	UNIT *	u;		/* the locked UNIT */
	_DIR *	info;		/* points to the actual _DIR */
	
} oDIR;

/* extended open file descriptor */
typedef struct
{
	long	mode;		/* the file i/o mode */
	long	actual;		/* actual cluster */
	long	cl;		/* number of the actual cluster */
	ulong	flen;		/* actual file len */
	long	error;		/* holds the last error */
	
} FILE;


/*
 * internal global functions
 */


/* debugging functions */

# ifndef FS_DEBUG

# define FAT_DEBUG(x)
# define FAT_DEBUG_PRINTDIR(x)
# define FAT_DEBUG_COOKIE(x)
# define FAT_DEBUG_HASH(x)

# define FAT_DEBUG_ON
# define FAT_DEBUG_OFF

# else

# define FAT_DEBUG(x) fatfs_debug x
# define FAT_DEBUG_PRINTDIR(x) fatfs_print_dir x
# define FAT_DEBUG_HASH(x) fatfs_dump_hashtable x
# if FS_DEBUG_COOKIE
# define FAT_DEBUG_COOKIE(x) fatfs_print_cookie x
# else
# define FAT_DEBUG_COOKIE(x)
# endif

# define FAT_DEBUG_ON	fatfs_config (0, FATFS_DEBUG, ENABLE)
# define FAT_DEBUG_OFF	fatfs_config (0, FATFS_DEBUG, DISABLE)

static long	fatfs_debug_mode;
static void	fatfs_debug		(const char *s, ...);
static void	fatfs_print_dir		(const _DIR *d, ushort dev);
static void	fatfs_dump_hashtable	(void);
# if FS_DEBUG_COOKIE
static void	fatfs_print_cookie	(COOKIE *c);
# endif

# endif


/* help functions */

FASTFN char *	fullname	(const COOKIE *c, const char *name);
FASTFN long	INDEX		(const COOKIE *c);


/* cookie cache access */

FASTFN ulong	c_hash		(register const char *s);
FASTFN COOKIE *	c_lookup	(register const char *s, register ushort dev);
FASTFN void	c_install	(register COOKIE *c);
static void	c_remove	(register COOKIE *c);

static COOKIE *	get_cookie	(register const char *s);


/* FAT access functions */

static long	getcl12		(long cluster, const ushort dev, ulong n);
static long	getcl16		(long cluster, const ushort dev, ulong n);
static long	getcl32		(long cluster, const ushort dev, ulong n);

static long	fixcl12		(long cluster, const ushort dev, long nextcl);
static long	fixcl16		(long cluster, const ushort dev, long nextcl);
static long	fixcl32		(long cluster, const ushort dev, long nextcl);

static long	ffree12		(const ushort dev);
static long	ffree16		(const ushort dev);
static long	ffree32		(const ushort dev);

/* FAT utilities functions */

FASTFN long	next_free12	(register long cluster, register const ushort dev);
FASTFN long	next_free16	(register long cluster, register const ushort dev);
FASTFN long	next_free32	(register long cluster, register const ushort dev);
static long	nextcl		(register long cluster, register const ushort dev);
static long	del_chain	(long cluster, const ushort dev);


/* DIR help functions */

FASTFN void	zero_cl		(register long cl, register const ushort dev);
FASTFN void	dir2str		(register const char *src, register char *nm);
FASTFN void	str2dir		(register const char *src, register char *nm);
FASTFN short	is_short	(register const char *src, register const char *table);
FASTFN long	fat_trunc	(register char *dst, const char *src, register long len, COOKIE *dir);
FASTFN long	vfat_trunc	(register char *dst, const char *src, register long len, COOKIE *dir);
FASTFN long	make_shortname	(COOKIE *dir, const char *src, char *dst);

/* DIR low lewel acces */

FASTFN long	__opendir	(register oDIR *dir, register const long cluster, register const ushort dev);
FASTFN void	__closedir	(register oDIR *dir);
FASTFN void	__updatedir	(register oDIR *dir);
static long	__seekdir	(register oDIR *dir, register long index, short mode);
FASTFN long	__SEEKDIR	(register oDIR *dir, register long index, short mode);
FASTFN long	__readvfat	(register oDIR *dir, char *lname, long size);
static long	__nextdir	(register oDIR *dir, char *name, long size);

static long	search_cookie	(COOKIE *dir, COOKIE **found, const char *name, short mode);
static long	write_cookie	(const COOKIE *c);
static long	delete_cookie	(COOKIE *c, long mode); /* mode == 0 -> don't delete cluster chain */
static long	make_cookie	(COOKIE *dir, COOKIE **new, const char *name, int attr);


/* device check and initialization */

static long	val_bpb		(_x_BPB *xbpb);
static long	get_bpb		(DI *di, _x_BPB *xbpb);
static DEVINFO *get_devinfo	(const ushort drv);


/* special FAT32 extension */

static void	upd_fat32fats	(register const ushort dev, long reference);
static void	upd_fat32boot	(register const ushort dev);
static void	val_fat32info	(register const ushort dev);
static void	inv_fat32info	(register const ushort dev);
static void	upd_fat32info	(register const ushort dev);


/* internal device driver functions */

static long	__FUTIME	(COOKIE *c, ushort *ptr);
static long	__FTRUNCATE	(COOKIE *c, long newlen);
static long	__FIO		(FILEPTR *f, char *buf, long bytes, short mode);

/* mode values: */
# define	READ		0
# define	WRITE		1

/* END definition part */
/****************************************************************************/

/****************************************************************************/
/* BEGIN global data definition & access implementation */

/*
 * shortnames are returned in lower case
 */

# define LCASE(dev)	(__lcase_flag)
static short __lcase_flag = 1;

# ifdef FATFS_TESTING
/*
 * temporary array to describe the TEST partitions
 */

# define TEST_PARTITIONS(dev)	(__test [(dev)])
static short __test [NUM_DRIVES];
# endif

/*
 * VFAT configuration
 */

# define VFAT(dev)	(__vfat [(dev)])
static short __vfat [NUM_DRIVES];

/*
 * symbolic links configuration
 */

# define SLNK(dev)	(__slnk [(dev)])
static short __slnk [NUM_DRIVES];

/*
 * new name mode configuration
 */

# define NAME_MODE(dev)		(__name_mode.mode [(dev)])
# define DEFAULT_TABLE(dev)	(__name_mode.table [(dev)])
static struct
{
	const char *table [NUM_DRIVES];
	short mode [NUM_DRIVES];
	
} __name_mode;

/*
 * global device info array
 */

static struct
{
	DEVINFO *info [NUM_DRIVES];
	long valid [NUM_DRIVES];
	
} devinfo;

/* valid */
# define INVALID	0
# define VALID		1

/*
 * makros for easy access
 */

# define BPB(dev)	(devinfo.info[(dev)])
# define BPBVALID(dev)	(devinfo.valid[(dev)])
# define DI(dev)	(BPB (dev)->di)
# define RCOOKIE(dev)	(BPB (dev)->root)

# define SECSIZE(dev)	(BPB (dev)->recsiz)
# define CLUSTSIZE(dev)	(BPB (dev)->clsizb)
# define CLSIZE(dev)	(BPB (dev)->clsiz)
# define CLUSTER(dev)	(BPB (dev)->numcl)
# define MAXCL(dev)	(BPB (dev)->maxcl)
# define ROOT(dev)	(BPB (dev)->rdstart)
# define ROOTSIZE(dev)	(BPB (dev)->rdlen)
# define ROOTENTRYS(dev)(BPB (dev)->rdentrys)
# define CLFIRST(dev)	(BPB (dev)->dstart)
# define DOFFSET(dev)	(BPB (dev)->doffset)
# define FAT1(dev)	(BPB (dev)->fstart)
# define FATSIZE(dev)	(BPB (dev)->flen)
# define FAT2ON(dev)	(BPB (dev)->fat2on)
# define ENTRYS(dev)	(BPB (dev)->entrys)

# define FAT_TYPE(dev)	(BPB (dev)->ftype)
# define LASTALLOC(dev)	(BPB (dev)->lastcl)
# define FREECL(dev)	(BPB (dev)->freecl)
# define FAT32(dev)	(FAT_TYPE (dev) == FAT_TYPE_32)

/* special for FAT32 */
# define FAT32ista(dev)	(BPB (dev)->info)
# define FAT32info(dev)	(BPB (dev)->info_active)
# define FAT32mirr(dev)	(BPB (dev)->fmirroring)
# define FAT32prim(dev)	(BPB (dev)->actual_fat)

/* avoid root dir fragmentation */
# define FAT32_ROFF	16


# ifdef USE_INLINE_FUNCS

FASTFN long
GETCL (register long cluster, register const ushort dev, register long offset)
{
	return (offset > 0) ? (*(BPB (dev)->getcl))(cluster, dev, offset) : cluster;
}

FASTFN long
FIXCL (register long cluster, register const ushort dev, register long nextcl)
{
	return (*(BPB (dev)->fixcl))(cluster, dev, nextcl);
}

FASTFN long
DFREE (const fcookie *dir, long *buf)
{
	const ushort dev = dir->dev;
	
	if (FREECL (dev) < 0)
		FREECL (dev) = (*(BPB (dev)->ffree))(dev);
	
	*buf++ = FREECL (dev);
	*buf++ = CLUSTER (dev);
	*buf++ = SECSIZE (dev);
	*buf   = CLSIZE (dev);
	
	return E_OK;
}

FASTFN long
NEXTCL (register long cluster, register const ushort dev, register const long mode)
{
	return (mode == READ) ? GETCL (cluster, dev, 1) : nextcl (cluster, dev);
}

# else

# define GETCL(cluster, dev, offset)	(((offset) > 0) ? (*(BPB (dev)->getcl))(cluster, dev, offset) : (cluster))
# define FIXCL(cluster, dev, nextcl)	((*(BPB (dev)->fixcl))(cluster, dev, nextcl))
# define DFREE(dir, buf)		
# error Makro DFREE not right
# define NEXTCL(cluster, dev, mode)	(((mode) == READ) ? GETCL ((cluster), (dev), 1) : nextcl (cluster, dev))

# endif

/*
 * help functions
 */

FASTFN char *
fullname (const COOKIE *c, const char *name)
{
	register long len = _STRLEN (c->name);
	register char * full = kmalloc (len + _STRLEN (name) + 2);
	if (full)
	{
		(void) _STRCPY (full, c->name);
		*(full + len) = '\\';
		(void) _STRCPY (full + len + 1, name);
	}
	else
		ALERT ("fatfs.c: kmalloc fail in: fullname (%s, %s)", c->name, name);
	
	return full;
}

FASTFN long
INDEX (const COOKIE *c)
{
	return
		c->stcl ?	c->stcl
			:	c->dir + (c->offset << 16);
	
	/* "x << 16" equivalent to "x * 32 * 2048" */
}

/*
 * cookie hash table
 */

# define COOKIE_HASHBITS	8
# define COOKIE_CACHE		(1UL << COOKIE_HASHBITS)
# define COOKIE_HASHMASK	(COOKIE_CACHE - 1)

static COOKIE cookies [COOKIE_CACHE];
static COOKIE * table [COOKIE_CACHE];

/* ?
 *	prevhash = (prevhash << 4) | (prevhash >> 28);
 *	return prevhash ^ _TOUPPER (c);
 */

# if 1
FASTFN ulong
c_hash (register const char *s)
{
	register ulong hash = 0;
	
	while (*s)
	{
		hash = ((hash << 5) - hash) + _TOUPPER (*s);
		s++;
	}
	
	hash ^= (hash >> COOKIE_HASHBITS) ^ (hash >> (COOKIE_HASHBITS << 1));
	
	return hash & COOKIE_HASHMASK;
}
# else
static ushort crcTable [] = /* 16 12 5 0 */
{
	0x0000, 0x1189, 0x2312, 0x329b, 0x4624, 0x57ad, 0x6536, 0x74bf,
	0x8c48, 0x9dc1, 0xaf5a, 0xbed3, 0xca6c, 0xdbe5, 0xe97e, 0xf8f7,
	0x1081, 0x0108, 0x3393, 0x221a, 0x56a5, 0x472c, 0x75b7, 0x643e,
	0x9cc9, 0x8d40, 0xbfdb, 0xae52, 0xdaed, 0xcb64, 0xf9ff, 0xe876,
	0x2102, 0x308b, 0x0210, 0x1399, 0x6726, 0x76af, 0x4434, 0x55bd,
	0xad4a, 0xbcc3, 0x8e58, 0x9fd1, 0xeb6e, 0xfae7, 0xc87c, 0xd9f5,
	0x3183, 0x200a, 0x1291, 0x0318, 0x77a7, 0x662e, 0x54b5, 0x453c,
	0xbdcb, 0xac42, 0x9ed9, 0x8f50, 0xfbef, 0xea66, 0xd8fd, 0xc974,
	0x4204, 0x538d, 0x6116, 0x709f, 0x0420, 0x15a9, 0x2732, 0x36bb,
	0xce4c, 0xdfc5, 0xed5e, 0xfcd7, 0x8868, 0x99e1, 0xab7a, 0xbaf3,
	0x5285, 0x430c, 0x7197, 0x601e, 0x14a1, 0x0528, 0x37b3, 0x263a,
	0xdecd, 0xcf44, 0xfddf, 0xec56, 0x98e9, 0x8960, 0xbbfb, 0xaa72,
	0x6306, 0x728f, 0x4014, 0x519d, 0x2522, 0x34ab, 0x0630, 0x17b9,
	0xef4e, 0xfec7, 0xcc5c, 0xddd5, 0xa96a, 0xb8e3, 0x8a78, 0x9bf1,
	0x7387, 0x620e, 0x5095, 0x411c, 0x35a3, 0x242a, 0x16b1, 0x0738,
	0xffcf, 0xee46, 0xdcdd, 0xcd54, 0xb9eb, 0xa862, 0x9af9, 0x8b70,
	0x8408, 0x9581, 0xa71a, 0xb693, 0xc22c, 0xd3a5, 0xe13e, 0xf0b7,
	0x0840, 0x19c9, 0x2b52, 0x3adb, 0x4e64, 0x5fed, 0x6d76, 0x7cff,
	0x9489, 0x8500, 0xb79b, 0xa612, 0xd2ad, 0xc324, 0xf1bf, 0xe036,
	0x18c1, 0x0948, 0x3bd3, 0x2a5a, 0x5ee5, 0x4f6c, 0x7df7, 0x6c7e,
	0xa50a, 0xb483, 0x8618, 0x9791, 0xe32e, 0xf2a7, 0xc03c, 0xd1b5,
	0x2942, 0x38cb, 0x0a50, 0x1bd9, 0x6f66, 0x7eef, 0x4c74, 0x5dfd,
	0xb58b, 0xa402, 0x9699, 0x8710, 0xf3af, 0xe226, 0xd0bd, 0xc134,
	0x39c3, 0x284a, 0x1ad1, 0x0b58, 0x7fe7, 0x6e6e, 0x5cf5, 0x4d7c,
	0xc60c, 0xd785, 0xe51e, 0xf497, 0x8028, 0x91a1, 0xa33a, 0xb2b3,
	0x4a44, 0x5bcd, 0x6956, 0x78df, 0x0c60, 0x1de9, 0x2f72, 0x3efb,
	0xd68d, 0xc704, 0xf59f, 0xe416, 0x90a9, 0x8120, 0xb3bb, 0xa232,
	0x5ac5, 0x4b4c, 0x79d7, 0x685e, 0x1ce1, 0x0d68, 0x3ff3, 0x2e7a,
	0xe70e, 0xf687, 0xc41c, 0xd595, 0xa12a, 0xb0a3, 0x8238, 0x93b1,
	0x6b46, 0x7acf, 0x4854, 0x59dd, 0x2d62, 0x3ceb, 0x0e70, 0x1ff9,
	0xf78f, 0xe606, 0xd49d, 0xc514, 0xb1ab, 0xa022, 0x92b9, 0x8330,
	0x7bc7, 0x6a4e, 0x58d5, 0x495c, 0x3de3, 0x2c6a, 0x1ef1, 0x0f78 
};

FASTFN ulong
c_hash (register const char *s)
{
	register ulong result = 0;
	
	while (*s)
	{
		result = (crcTable [(((ulong) _TOUPPER (*s)) ^ result) & 0xff]) ^ (result >> 8);
		s++;
	}
	
	return result & COOKIE_HASHMASK;
}
# endif

FASTFN COOKIE *
c_lookup (register const char *s, register ushort dev)
{
	register COOKIE *c;
	
	FAT_DEBUG (("c_lookup: s = %s, c_hash (s) = %li\r\n", s, c_hash (s)));
	
	for (c = table [c_hash (s)]; c != NULL; c = c->next)
	{
		if ((_STRICMP (c->name, s) == 0) && dev == c->dev)
		{
			FAT_DEBUG (("c_lookup: match s = %s, c->name = %s\r\n", s, c->name));
			return c;
		}
	}
	
	return NULL;
}

FASTFN void
c_install (register COOKIE *c)
{
	register ulong hashval = c_hash (c->name);
	
	FAT_DEBUG (("c_install: c->name = %s, c_hash (c->name) = %li\r\n", c->name, c_hash (c->name)));
	
	c->next = table [hashval];
	table [hashval] = c;
}

static void
c_remove (register COOKIE *c)
{
	register ulong hashval = c_hash (c->name);
	register COOKIE **temp = & table [hashval];
	long flag = 1;
	
	while (*temp)
	{
		if (*temp == c)
		{
			*temp = c->next;
			flag = 0;
			break;
		}
		temp = &(*temp)->next;
	}
	
	if (flag)
	{
		ALERT ("fatfs.c: remove from hashtable fail in: c_remove (addr = %lx, %s)", c, c->name);
	}
	if (c->open)
	{
		ALERT ("fatfs.c: open FILEPTR detect in: c_remove (%s)", c->name);
		c->open = NULL;
	}
	if (c->locks)
	{
		ALERT ("fatfs.c: open LOCKS detect in: c_remove (%s)", c->name);
		c->locks = NULL;
	}
	if (c->extra)
	{
		kfree (c->extra);
		c->extra = NULL;
	}
	
	kfree (c->name);
	c->name = NULL;
	c->links = 0;
}

static COOKIE *
get_cookie (register const char *s)
{
	static long count = 0;
	register long i;
	
	for (i = 0; i < COOKIE_CACHE; i++)
	{
		count++; if (count == COOKIE_CACHE) count = 0;
		{
			register COOKIE *c = &(cookies[count]);
			if (c->links == 0)
			{
				if (c->name) c_remove (c);
				c->name = (char *) s;
				c->links = 1;
				c_install (c);
				return c;
			}
		}
	}
	ALERT ("fatfs.c: get_cookie: no free COOKIE found for %s", s);
	FAT_DEBUG_HASH (());
	return NULL;
}

/* END global data & access implementation */
/****************************************************************************/

/****************************************************************************/
/* BEGIN FAT part */

/*
 * makros for cluster to sector and sector to cluster calculation
 */

# ifdef USE_INLINE_FUNCS

FASTFN long
C2S (register long cluster, ushort dev)
{
	return ((cluster * CLSIZE (dev)) + DOFFSET (dev));
}
FASTFN long
S2C (register long sector, ushort dev)
{
	return ((sector - DOFFSET (dev)) / CLSIZE (dev));
}

# else

# define C2S(cluster, dev)	((cluster) * CLSIZE (dev) + DOFFSET (dev))
# define S2C(sector, dev)	(((sector) - DOFFSET (dev)) / CLSIZE (dev))

# endif

/*
 * makros for FAT validation
 */

# define FAT_VALID12(word, dev)	(((word) > 0x001) && ((word) < MAXCL (dev)))
# define FAT_VALID16(word, dev)	(((word) > 0x0001) && ((word) < MAXCL (dev)))
# define FAT_VALID32(word, dev)	(((word) > 0x0001) && ((word) < MAXCL (dev)))

# define FAT_LAST12(word)	((word) > 0xff7)
# define FAT_LAST16(word)	((word) > 0xfff7)
# define FAT_LAST32(word)	((word) > 0xffffff7)

# define FAT_BAD12(word)	(((word) > 0xfef) && ((word) < 0xff8))
# define FAT_BAD16(word)	(((word) > 0xffef) && ((word) < 0xfff8))
# define FAT_BAD32(word)	(((word) > 0xfffffef) && ((word) < 0xffffff8))

# define FAT_FREE(word)		((word) == 0)

/*
 * makros for fat values
 */

# define CLLAST12		(0xfff)
# define CLLAST16		(0xffff)
# define CLLAST32		(0xfffffff)

# define CLBAD12		(0xff7)
# define CLBAD16		(0xfff7)
# define CLBAD32		(0xffffff7)

# define CLFREE			(0)
# define CLLAST			(-1)
# define CLBAD			(-2)
# define CLILLEGAL		(-3)

/*
 * stcl access
 */

# ifdef USE_INLINE_FUNCS

# define GET32_STCL(dir)	get32_stcl (dir)
# define PUT32_STCL(dir, cl)	put32_stcl (dir, cl)

FASTFN long
get32_stcl (_DIR *dir)
{
	return (((ulong) (SWAP68_W (dir->stcl_fat32))) << 16) | SWAP68_W (dir->stcl);
}

FASTFN void
put32_stcl (_DIR *dir, long cl)
{
	dir->stcl = SWAP68_W ((ushort) cl);
	dir->stcl_fat32 = SWAP68_W ((ushort) (cl >> 16));
}

FASTFN long
GET_STCL (_DIR *dir, ushort dev)
{
	return FAT32 (dev) ? GET32_STCL (dir) : SWAP68_W (dir->stcl);
}

FASTFN void
PUT_STCL (_DIR *dir, ushort dev, long cl)
{
	if (FAT32 (dev))
		PUT32_STCL (dir, cl);
	else
		dir->stcl = SWAP68_W ((ushort) cl);
}

# else

# define GET32_STCL(dir)	((((ulong) (SWAP68_W ((dir)->stcl_fat32))) << 16) | SWAP68_W ((dir)->stcl))
# define PUT32_STCL(dir, cl)	((dir)->stcl = SWAP68_W ((ushort) (cl)), (dir)->stcl_fat32 = SWAP68_W ((ushort) ((cl) >> 16)))

# define GET_STCL(dir, dev)	FAT32 (dev) ? GET32_STCL (dir) : SWAP68_W ((dir)->stcl)
# define PUT_STCL(dir, dev, cl)	({ if (FAT32 (dev)) PUT32_STCL (dir, cl); else (dir)->stcl = SWAP68_W ((ushort) cl); })

# endif

/*
 * getcl??:
 * --------
 * get the contents of the fat entry 'cluster' [repeat n]
 * 
 * return: cluster number to be linked or
 *         negative error number:
 *          CLFREE    ... free
 *          CLLAST    ... last in linked list
 *          CLBAD     ... bad cluster
 *          CLILLEGAL ... illegal value
 *          EREADF    ... read fail
 * 
 * fixcl??:
 * --------
 * set the contents of the fat entry 'cluster'
 * to link to 'nextcl';
 * nextcl can be pos. value or CLFREE, CLLAST, CLBAD
 * 
 * ffree??:
 * --------
 * - fat free - return the number of free cluster
 * 
 * next_free??:
 * ------------
 * allocate a new cluster
 * 
 * nextcl:
 * -------
 * get the next cluster; at the end of the cluster chain is
 * a new cluster is allocated
 * 
 * del_chain:
 * ----------
 * delete the cluster chain started at cluster
 * 
 */

/*
 * getcl
 */

static long
getcl12 (long cluster, const ushort dev, ulong n)
{
	FAT_DEBUG (("getcl12: enter (cluster = %li, n = %li)\r\n", cluster, n));
	
	/* input validation */
	if (cluster < 2 || cluster >= MAXCL (dev))
	{
		FAT_DEBUG (("getcl12: leave failure (cluster out of range)\r\n"));
		return CLILLEGAL;
	}
	
	{	register const ulong entrys = SECSIZE (dev) << 1; /* we read 3 sectors */
		register ulong old_sector = 0;
		UNIT *u = NULL;
		
		do {	register const ulong sector = FAT1 (dev) + (cluster / entrys * 3);
			register const ulong offset = ((cluster % entrys) * 12) / 8;
			register long newcl;
			
			if (sector != old_sector)
			{
				register long sectors = ((sector + 3) - FAT1 (dev) - FATSIZE (dev));
				sectors = MAX (3, sectors);
				
				FAT_DEBUG (("getcl12: start = %li, sectors = %li\r\n", sector, sectors));
				
				old_sector = sector;
				u = bio.read (DI (dev), sector, SECSIZE (dev) * sectors);
				if (!u)
				{
					FAT_DEBUG (("getcl12: leave failure (can't read the fat)\r\n"));
					return EREADF;
				}
			}
			
			/* copy the entry */
			newcl = WPEEK_INTEL (u->data + offset);
			
			FAT_DEBUG (("getcl12: cluster & 1 = %li, newcl = %li\r\n", cluster & 1, newcl));
			
			/* mask out the 12 bits */
			if (cluster & 1)
				cluster = newcl >> 4;
			else
				cluster = newcl & 0x0fff;
			
			FAT_DEBUG (("getcl12: cluster = %li\r\n", cluster));
			
			/* entry valid? */
			if (!FAT_VALID12 (cluster, dev))
			{
				register long ret;
				
				FAT_FREE (cluster) ? (ret =  CLFREE) :
				FAT_LAST12 (cluster) ? (ret = CLLAST) :
				FAT_BAD12  (cluster) ? (ret = CLBAD) : (ret = CLILLEGAL);
				
				FAT_DEBUG (("getcl12: leave failure (invalid entry, ret = %li)\r\n", ret));
				return ret;
			}
		}
		while (--n);
	}
	
	FAT_DEBUG (("getcl12: leave ok (sector = %li, cluster = %li)\r\n", C2S (cluster, dev), cluster));
	return cluster;
}

static long
getcl16 (long cluster, const ushort dev, ulong n)
{
	FAT_DEBUG (("getcl16: enter (cluster = %li, n = %li)\r\n", cluster, n));
	
	/* input validation */
	if (cluster < 2 || cluster >= MAXCL (dev))
	{
		FAT_DEBUG (("getcl16: leave failure (cluster out of range)\r\n"));
		return CLILLEGAL;
	}
	
	{	register const ulong entrys = SECSIZE (dev) >> 1;
		register ulong old_sector = 0;
		UNIT *u = NULL;
		
		do {	register const ulong sector = FAT1 (dev) + cluster / entrys;
			register const ulong offset = cluster % entrys;
			
			if (sector != old_sector)
			{
				old_sector = sector;
				u = bio.read (DI (dev), sector, SECSIZE (dev));
				if (!u)
				{
					FAT_DEBUG (("getcl16: leave failure (can't read the fat)\r\n"));
					return EREADF;
				}
			}
			
			/* copy the entry */
			cluster = SWAP68_W (*(((ushort *) u->data) + offset));
			
			/* entry valid? */
			if (!FAT_VALID16 (cluster, dev))
			{
				register long ret;
				
				FAT_FREE (cluster) ? (ret =  CLFREE) :
				FAT_LAST16 (cluster) ? (ret = CLLAST) :
				FAT_BAD16  (cluster) ? (ret = CLBAD) : (ret = CLILLEGAL);
				
				FAT_DEBUG (("getcl16: leave failure (invalid entry, ret = %li)\r\n", ret));
				return ret;
			}
		}
		while (--n);
	}
	
	FAT_DEBUG (("getcl16: leave ok (sector = %li, cluster = %li)\r\n", C2S (cluster, dev), cluster));
	return cluster;
}

static long
getcl32 (long cluster, const ushort dev, ulong n)
{
	FAT_DEBUG (("getcl32: enter (cluster = %li, n = %li)\r\n", cluster, n));
	
	/* input validation */
	if (cluster < 2 || cluster >= MAXCL (dev))
	{
		FAT_DEBUG (("getcl32: leave failure (cluster out of range)\r\n"));
		return CLILLEGAL;
	}
	
	{	register const ulong entrys = SECSIZE (dev) >> 2;
		register ulong old_sector = 0;
		UNIT *u = NULL;
		
		do {	register const ulong sector = FAT32prim (dev) + cluster / entrys;
			register const ulong offset = cluster % entrys;
			
			if (sector != old_sector)
			{
				old_sector = sector;
				u = bio.read (DI (dev), sector, SECSIZE (dev));
				if (!u)
				{
					FAT_DEBUG (("getcl32: leave failure (can't read the fat)\r\n"));
					return EREADF;
				}
			}
			
			/* copy the entry */
			cluster = SWAP68_L (*(((ulong *) u->data) + offset));
			
			/* the highest 4 Bits are reserved, mask out */
			cluster &= 0x0fffffff;
			
			/* entry valid? */
			if (!FAT_VALID32 (cluster, dev))
			{
				register long ret;
				
				FAT_FREE (cluster) ? (ret =  CLFREE) :
				FAT_LAST32 (cluster) ? (ret = CLLAST) :
				FAT_BAD32  (cluster) ? (ret = CLBAD) : (ret = CLILLEGAL);
				
				FAT_DEBUG (("getcl32: leave failure (invalid entry, ret = %li)\r\n", ret));
				return ret;
			}
		}
		while (--n);
	}
	
	FAT_DEBUG (("getcl32: leave ok (sector = %li, cluster = %li)\r\n", C2S (cluster, dev), cluster));
	return cluster;
}

/*
 * fixcl
 */

static long
fixcl12 (long cluster, const ushort dev, long nextcl)
{
	FAT_DEBUG (("fixcl12: enter (cluster = %li, nextcl = %li)\r\n", cluster, nextcl));
	
	/* input validation */
	if (cluster < 2 || cluster >= MAXCL (dev))
	{
		FAT_DEBUG (("fixcl12: leave failure (cluster out of range)\r\n"));
		return CLILLEGAL;
	}
	
	switch (nextcl)
	{
		case CLLAST:	nextcl = CLLAST12;	break;
		case CLBAD:	nextcl = CLBAD12;	break;
	}
	
	{	register const ulong entrys = SECSIZE (dev) << 1; /* we read 3 sectors */
		UNIT *u = NULL;
		
		register const ulong sector = FAT1 (dev) + (cluster / entrys * 3);
		register const ulong offset = ((cluster % entrys) * 12) / 8;
		
		register long sectors = ((sector + 3) - FAT1 (dev) - FATSIZE (dev));
		sectors = MAX (3, sectors);
			
		FAT_DEBUG (("getcl12: start = %li, sectors = %li\r\n", sector, sectors));
			
		u = bio.read (DI (dev), sector, SECSIZE (dev) * sectors);
		if (u)
		{
			register long newcl;
			
			/* read the FAT entry */
			newcl = WPEEK_INTEL (u->data + offset);
			
			FAT_DEBUG (("fixcl12: newcl = %li\r\n", newcl));
			
			/* mask in the 12 used bits */
			if (cluster & 1)
				newcl = (newcl & 0x000f) | (nextcl << 4);
			else
				newcl = (newcl & 0xf000) | nextcl;
			
			FAT_DEBUG (("fixcl12: newcl = %li\r\n", newcl));
			
			/* write the entry to the first FAT */
			WPOKE_INTEL (u->data + offset, newcl);
			
			bio_MARK_MODIFIED ((&bio), u);
			
			if (FAT2ON (dev))
			{
				u = bio.read (DI (dev), sector + FATSIZE (dev), SECSIZE (dev) * sectors);
				if (u)
				{
					/* write the entry to the second FAT */
					WPOKE_INTEL (u->data + offset, newcl);
					
					bio_MARK_MODIFIED ((&bio), u);
				}
			}
			
			FAT_DEBUG (("fixcl12: leave ok (return E_OK)\r\n"));
			return E_OK;
		}
		
	}
	
	FAT_DEBUG (("fixcl12: leave failure (can't read the fat)\r\n"));
	return EREADF;
}

static long
fixcl16 (long cluster, const ushort dev, long nextcl)
{
	FAT_DEBUG (("fixcl16: enter (cluster = %li, nextcl = %li)\r\n", cluster, nextcl));
	
	/* input validation */
	if (cluster < 2 || cluster >= MAXCL (dev))
	{
		FAT_DEBUG (("fixcl16: leave failure (cluster out of range)\r\n"));
		return CLILLEGAL;
	}
	
	switch (nextcl)
	{
		case CLLAST:	nextcl = CLLAST16;	break;
		case CLBAD:	nextcl = CLBAD16;	break;
	}
	
	{	register const ulong entrys = SECSIZE (dev) >> 1;
		UNIT *u = NULL;
		
		register const ulong sector = FAT1 (dev) + cluster / entrys;
		register const ulong offset = cluster % entrys;
		
		u = bio.read (DI (dev), sector, SECSIZE (dev));
		if (u)
		{
			/* write the entry to the first FAT */
			*(((ushort *) u->data) + offset) = SWAP68_W ((ushort) nextcl);
			
			bio_MARK_MODIFIED ((&bio), u);
			
			if (FAT2ON (dev))
			{
				u = bio.read (DI (dev), sector + FATSIZE (dev), SECSIZE (dev));
				if (u)
				{
					/* write the entry to the second FAT */
					*(((ushort *) u->data) + offset) = SWAP68_W ((ushort) nextcl);
					
					bio_MARK_MODIFIED ((&bio), u);
				}
			}
			
			FAT_DEBUG (("fixcl16: leave ok (return E_OK)\r\n"));
			return E_OK;
		}
	}
	
	FAT_DEBUG (("fixcl16: leave failure (can't read the fat)\r\n"));
	return EREADF;
}

static long
fixcl32 (long cluster, const ushort dev, long nextcl)
{
	FAT_DEBUG (("fixcl32: enter (cluster = %li, nextcl = %li)\r\n", cluster, nextcl));
	
	/* input validation */
	if (cluster < 2 || cluster >= MAXCL (dev))
	{
		FAT_DEBUG (("fixcl32: leave failure (cluster out of range)\r\n"));
		return CLILLEGAL;
	}
	
	switch (nextcl)
	{
		case CLLAST:	nextcl = CLLAST32;	break;
		case CLBAD:	nextcl = CLBAD32;	break;
	}
	
	{	register const ulong entrys = SECSIZE (dev) >> 2;
		UNIT *u = NULL;
		
		register ulong sector = FAT32prim (dev) + cluster / entrys;
		register const ulong offset = cluster % entrys;
		
		u = bio.read (DI (dev), sector, SECSIZE (dev));
		if (u)
		{
			/* mask in the highest 4 bit (reserved) */
			nextcl |= *(((ulong *) u->data) + offset) & 0xf0000000;
			
			/* write the entry to the first FAT */
			*(((ulong *) u->data) + offset) = SWAP68_L ((ulong) nextcl);
			
			bio_MARK_MODIFIED ((&bio), u);
			
			if (FAT32mirr (dev))
			{
				/* update all FATs,
				 * 'sector' is in the first FAT
				 * in this case
				 */
				register long i;
				for (i = FAT2ON (dev); i; i--)
				{
					sector += FATSIZE (dev);
					u = bio.read (DI (dev), sector, SECSIZE (dev));
					if (u)
					{
						/* write the FAT entry */
						*(((ulong *) u->data) + offset) = SWAP68_L ((ulong) nextcl);
					
						bio_MARK_MODIFIED ((&bio), u);
					}
				}
			}
			
			FAT_DEBUG (("fixcl32: leave ok (return E_OK)\r\n"));
			return E_OK;
		}
	}
	
	FAT_DEBUG (("fixcl32: leave failure (can't read the fat)\r\n"));
	return EREADF;
}

/*
 * fat free
 */

static long
ffree12 (const ushort dev)
{
	long count = 0;
	FAT_DEBUG (("ffree12: enter (%c)\r\n", 'A'+dev));
	
	{	register long i;
		for (i = 2; i < MAXCL (dev); i++)
			if (!getcl16 (i, dev, 1))
				count++;
	}
	
	FAT_DEBUG (("ffree12: leave count = %li\r\n", count));
	return count;
}

static long
ffree16 (const ushort dev)
{
	long count = 0;
	FAT_DEBUG (("ffree16: enter (%c)\r\n", 'A'+dev));
# if 0
	/*
	 * use GETCL (slow)
	 */
	{	register long i;
		for (i = 2; i < MAXCL (dev); i++)
			if (!getcl16 (i, dev, 1))
				count++;
	}
# else
	/*
	 * read fat direct
	 */
	{	long entrys = SECSIZE (dev) >> 1;	/* FAT entrys per sector */
		long sector = FAT1 (dev);		/* the final startsector to read */
		long todo = FATSIZE (dev);		/* number of sectors to read */
		long cluster = 0;
		UNIT *u;
		
		do {	u = bio.read (DI (dev), sector, SECSIZE (dev));
			if (u)
			{
				/* recalc max if overflow */
				if ((cluster + entrys) > MAXCL (dev))
				{
					entrys -= (cluster + entrys) - MAXCL (dev);
				}
				/* count the free cluster */
				{	register ushort *value = (ushort *) u->data;
					register long i = 0;
					if (cluster == 0)
					{
						value++; value++;
						i = 2;
					}
					for (; i < entrys; i++)
					{
						if (!*value++)
							count++;
					}
				}
				cluster += entrys;
				sector++;
			}
		}
		while (--todo);
	}
# endif	
	
	FAT_DEBUG (("ffree16: leave count = %li\r\n", count));
	return count;
}

static long
ffree32 (const ushort dev)
{
	long count = 0;
	FAT_DEBUG (("ffree32: enter (%c)\r\n", 'A'+dev));
# if 0
	/*
	 * use GETCL (slow)
	 */
	{	register long i;
		for (i = 2; i < MAXCL (dev); i++)
			if (!getcl32 (i, dev, 1))
				count++;
	}
# else
	/*
	 * read fat direct
	 */
	{	long entrys = SECSIZE (dev) >> 2;	/* FAT entrys per sector */
		long sector = FAT32prim (dev);		/* the final startsector to read */
		long todo = FATSIZE (dev);		/* number of sectors to read */
		long cluster = 0;
		UNIT *u;
		
		do {	u = bio.read (DI (dev), sector, SECSIZE (dev));
			if (u)
			{
				/* recalc max if overflow */
				if ((cluster + entrys) > MAXCL (dev))
				{
					entrys -= (cluster + entrys) - MAXCL (dev);
				}
				/* count the free cluster */
				{	register ulong *value = (ulong *) u->data;
					register long i = 0;
					if (cluster == 0)
					{
						value++; value++;
						i = 2;
					}
					for (; i < entrys; i++)
					{
						/* the highest 4 bits are reserved */
						register long cl = SWAP68_L (*value);
						cl &= 0x0fffffff;
						
						if (!cl) count++;
						value++;
					}
				}
				cluster += entrys;
				sector++;
			}
		}
		while (--todo);
	}
# endif	
	
	FAT_DEBUG (("ffree32: leave count = %li\r\n", count));
	return count;
}

/*
 * FAT utility functions
 */

FASTFN long
next_free12 (register long cluster, register const ushort dev)
{
	register const long max = MAXCL (dev);
	register long i;
	
	FAT_DEBUG (("next_free12: enter cluster = %li, dev = %i\r\n", cluster, dev));
	
	if (cluster == 0)
	{
		cluster = LASTALLOC (dev);
	}
	
	/* search free cluster */
	for (i = 2; i < max; i++)
	{
		/* out of range? */
		if (cluster < 2 || cluster >= max)
			cluster = 2;
		
		if (getcl12 (cluster, dev, 1) != CLFREE)
		{
			cluster++;
		}
		else
		{
			/* yes, found a free cluster */
			FAT_DEBUG (("next_free12: leave ok, cluster = %li, dev = %i\r\n", cluster, dev));
			LASTALLOC (dev) = cluster;
			return cluster;
		}
	}
	
	/* disk full */
	return EACCDN;
}

FASTFN long
next_free16 (register long cluster, register const ushort dev)
{
	long entrys = SECSIZE (dev) >> 1;
	long todo = FATSIZE (dev) + 1;
	long sector;
	UNIT *u;
	register long i;
	
	FAT_DEBUG (("next_free16: enter cluster = %li\r\n", cluster));
	
	if (cluster == 0)
	{
		cluster = LASTALLOC (dev);
		FAT_DEBUG (("next_free16: use LASTALLOC = %li\r\n", cluster));
	}
	
	i = cluster % entrys;
	sector = FAT1 (dev) + cluster / entrys;
	cluster -= i;
	
	FAT_DEBUG (("next_free16: i = %li, sector = %li\r\n", i, sector));
	
	do {	u = bio.read (DI (dev), sector, SECSIZE (dev));
		if (u)
		{
			/* recalc entrys if overflow */
			if ((cluster + entrys) > MAXCL (dev))
			{
				entrys -= (cluster + entrys) - MAXCL (dev);
			}
			/* search a free cluster */
			{	register ushort *value = (ushort *) u->data;
				while (i < entrys)
				{
					if (!*(value + i))
					{
						cluster += i;
						FAT_DEBUG (("next_free16: leave ok, cluster = %li, dev = %i\r\n", cluster, dev));
						LASTALLOC (dev) = cluster;
						return cluster;
					}
					i++;
				}
			}
			cluster += entrys;
			sector++;
			i = 0;
			if (sector >= (FAT1 (dev) + FATSIZE (dev)))
			{
				cluster = 0;
				entrys = SECSIZE (dev) >> 1;
				sector = FAT1 (dev);
				i = 2;
			}
		}
	}
	while (--todo);
	
	/* disk full */
	return EACCDN;
}

FASTFN long
next_free32 (register long cluster, register const ushort dev)
{
	long entrys = SECSIZE (dev) >> 2;
	long todo = FATSIZE (dev) + 1;
	long sector;
	UNIT *u;	
	register long i;
	
	FAT_DEBUG (("next_free32: enter cluster = %li\r\n", cluster));
	
	if (cluster == 0)
	{
		cluster = LASTALLOC (dev);
		FAT_DEBUG (("next_free32: use LASTALLOC = %li\r\n", cluster));
	}
	
	i = cluster % entrys;
	sector = FAT32prim (dev) + cluster / entrys;
	cluster -= i;
	
	FAT_DEBUG (("next_free32: i = %li, sector = %li\r\n", i, sector));
	
	do {	u = bio.read (DI (dev), sector, SECSIZE (dev));
		if (u)
		{
			/* recalc entrys if overflow */
			if ((cluster + entrys) > MAXCL (dev))
			{
				entrys -= (cluster + entrys) - MAXCL (dev);
			}
			/* search a free cluster */
			{	register ulong *value = (ulong *) u->data;
				while (i < entrys)
				{
					/* the highest 4 bits are reserved */
					register long cl = SWAP68_L (*(value + i));
					cl &= 0x0fffffff;
					if (!cl)
					{
						cluster += i;
						FAT_DEBUG (("next_free32: leave ok, cluster = %li, dev = %i\r\n", cluster, dev));
						LASTALLOC (dev) = cluster;
						return cluster;
					}
					i++;
				}
			}
			cluster += entrys;
			sector++;
			i = 0;
			if (sector >= (FAT1 (dev) + FATSIZE (dev)))
			{
				cluster = 0;
				entrys = SECSIZE (dev) >> 2;
				sector = FAT32prim (dev);
				i = 2;
			}
		}
	}
	while (--todo);
	
	/* disk full */
	return EACCDN;
}

static long
nextcl (register long cluster, register const ushort dev)
{
	register long content =
		(cluster == 0) ? CLLAST : GETCL (cluster, dev, 1);
	
	if (content == 0)
		ALERT ("fatfs.c: nextcl: content = 0!");
	
	if (content == CLLAST)
	{
		/* last, alloc a new cluster */
		
		switch (FAT_TYPE (dev))
		{
			case FAT_TYPE_12:
			{
				content = next_free12 (cluster, dev);
				break;
			}
			case FAT_TYPE_16:
			{
				content = next_free16 (cluster, dev);
				break;
			}
			case FAT_TYPE_32:
			{
				content = next_free32 (cluster, dev);
				break;
			}
			default:
			{
				return EACCDN;
			}
		}
		
		if (content > 0)
		{
			register long r;
			
			r = FIXCL (content, dev, CLLAST);
			if (r) return r;
			
			/* decrease free cluster counter */
			if (!(FREECL (dev) < 0))
				FREECL (dev)--;
			
			if (cluster)
			{
				r = FIXCL (cluster, dev, content);
				if (r) return r;
			}
		}
	}
	
	return content;
}

static long
del_chain (long cluster, const ushort dev)
{
	register long next;
	
	FAT_DEBUG (("del_chain: enter\r\n"));
	
	next = GETCL (cluster, dev, 1);
	while (next > 0)
	{
		register long r;
		
		FAT_DEBUG (("del_chain: FIXCL cluster = %li, next = %li\r\n", cluster, next));
		
		r = FIXCL (cluster, dev, 0);
		if (r == 0)
		{
			/* increase free cluster counter */
			if (!(FREECL (dev) < 0))
				FREECL (dev)++;
			
			cluster = next;
			next = GETCL (cluster, dev, 1);
		}
		else
		{
			FAT_DEBUG (("del_chain: leave failure (FIXCL, cluster = %li)\r\n", cluster));
			return r;
		}
	}
	
	if (next == CLLAST)
	{
		register long r;
		
		r = FIXCL (cluster, dev, 0);
		if (r == 0)
		{
			/* increase free cluster counter */
			if (!(FREECL (dev) < 0))
				FREECL (dev)++;
		}
		
		FAT_DEBUG (("del_chain: leave ok (return FIXCL = %li) cluster = %li\r\n", r, cluster));
		return r;
	}
	
	FAT_DEBUG (("del_chain: leave failure (not at end)\r\n"));
	return ERROR;
}

/* END FAT part */
/****************************************************************************/

/****************************************************************************/
/* BEGIN DIR part */

FASTFN void
zero_cl (register long cl, register const ushort dev)
{
	UNIT *u;
	u = bio.getunit (DI (dev), C2S (cl, dev), CLUSTSIZE (dev));
	if (u)
	{
		quickzero (u->data, CLUSTSIZE (dev) >> 8);
		bio_MARK_MODIFIED ((&bio), u);
	}
}

FASTFN void
dir2str (register const char *src, register char *nm)
{
	register long i = 8;
# ifdef FS_DEBUG
	char *start = nm;
# endif
	
	FAT_DEBUG (("dir2str: enter (src = %s)\r\n", src));
	
	while (i-- && *src != ' ')
	{
		*nm++ = *src++;
	}
	src += i + 1;
	if (*src > ' ')
	{
		*nm++ = '.';
		i = 3;
		while (i-- && *src != ' ')
		{
			*nm++ = *src++;
		}
	}
	*nm = '\0';
	
	FAT_DEBUG (("dir2str: leave ok (nm = %s)\r\n", start));
}

FASTFN void
str2dir (register const char *src, register char *nm)
{
	register long i = 8;
# ifdef FS_DEBUG
	char *start = nm;
# endif
	
	FAT_DEBUG (("str2dir: enter (src = %s)\r\n", src));
	
	while (i--)
		if (*src && *src != '.')
			*nm++ = *src++;
		else
			*nm++ = ' ';
	if (*src == '.') src++;
	i = 3;
	while (i--)
		if (*src)
			*nm++ = *src++;
		else
			*nm++ = ' ';
	
	FAT_DEBUG (("str2dir: leave ok (nm = %s)\r\n", start));
}

/*
 * MS-DOS: A..Z0..9!#$%&'()-@^_`{}~ und grosse Umlaute
 */

# define MSDOS_VALID(c)		(msdos_table [(long) c])
# define MSDOS_TABLE		(msdos_table)
static const char msdos_table [256] =
{
/* 000   */ 0,
/* 001   */ 0,	/* 002   */ 0,	/* 003   */ 0,	/* 004   */ 0,	/* 005   */ 0,
/* 006   */ 0,	/* 007   */ 0,	/* 008   */ 0,	/* 009   */ 0,	/* 010   */ 0,
/* 011   */ 0,	/* 012   */ 0,	/* 013   */ 0,	/* 014   */ 0,	/* 015   */ 0,
/* 016   */ 0,	/* 017   */ 0,	/* 018   */ 0,	/* 019   */ 0,	/* 020   */ 0,
/* 021   */ 0,	/* 022   */ 0,	/* 023   */ 0,	/* 024   */ 0,	/* 025   */ 0,
/* 026   */ 0,	/* 027   */ 0,	/* 028   */ 0,	/* 029   */ 0,	/* 030   */ 0,
/* 031   */ 0,	/* 032   */ 0,	/* 033 ! */ 0,	/* 034 " */ 0,	/* 035 # */ 1,
/* 036 $ */ 1,	/* 037 % */ 1,	/* 038 & */ 1,	/* 039 ' */ 1,	/* 040 ( */ 1,
/* 041 ) */ 1,	/* 042 * */ 0,	/* 043 + */ 0,	/* 044 , */ 0,	/* 045 - */ 1,
/* 046 . */ 0,	/* 047 / */ 0,	/* 048 0 */ 1,	/* 049 1 */ 1,	/* 050 2 */ 1,
/* 051 3 */ 1,	/* 052 4 */ 1,	/* 053 5 */ 1,	/* 054 6 */ 1,	/* 055 7 */ 1,
/* 056 8 */ 1,	/* 057 9 */ 1,	/* 058 : */ 0,	/* 059 ; */ 0,	/* 060 < */ 0,
/* 061 = */ 0,	/* 062 > */ 0,	/* 063 ? */ 0,	/* 064 @ */ 1,	/* 065 A */ 1,
/* 066 B */ 1,	/* 067 C */ 1,	/* 068 D */ 1,	/* 069 E */ 1,	/* 070 F */ 1,
/* 071 G */ 1,	/* 072 H */ 1,	/* 073 I */ 1,	/* 074 J */ 1,	/* 075 K */ 1,
/* 076 L */ 1,	/* 077 M */ 1,	/* 078 N */ 1,	/* 079 O */ 1,	/* 080 P */ 1,
/* 081 Q */ 1,	/* 082 R */ 1,	/* 083 S */ 1,	/* 084 T */ 1,	/* 085 U */ 1,
/* 086 V */ 1,	/* 087 W */ 1,	/* 088 X */ 1,	/* 089 Y */ 1,	/* 090 Z */ 1,
/* 091 [ */ 0,	/* 092 \ */ 0,	/* 093 ] */ 0,	/* 094 ^ */ 1,	/* 095 _ */ 1,
/* 096 ` */ 1,	/* 097 a */ 1,	/* 098 b */ 1,	/* 099 c */ 1,	/* 100 d */ 1,
/* 101 e */ 1,	/* 102 f */ 1,	/* 103 g */ 1,	/* 104 h */ 1,	/* 105 i */ 1,
/* 106 j */ 1,	/* 107 k */ 1,	/* 108 l */ 1,	/* 109 m */ 1,	/* 110 n */ 1,
/* 111 o */ 1,	/* 112 p */ 1,	/* 113 q */ 1,	/* 114 r */ 1,	/* 115 s */ 1,
/* 116 t */ 1,	/* 117 u */ 1,	/* 118 v */ 1,	/* 119 w */ 1,	/* 120 x */ 1,
/* 121 y */ 1,	/* 122 z */ 1,	/* 123 { */ 1,	/* 124 | */ 0,	/* 125 } */ 1,
/* 126 ~ */ 1,	/* 127   */ 0,	/* 128   */ 0,	/* 129   */ 0,	/* 130   */ 0
/* 131 - 255 automatically set to 0 */
};

/*
 * GEMDOS: A..Z0..9!#$%&'()-@^_`{}~"+,;<=>[]| und grosse Umlaute
 */

# define GEMDOS_VALID(c)	(gemdos_table [(long) c])
# define GEMDOS_TABLE		(gemdos_table)
static const char gemdos_table [256] =
{
/* 000   */ 0,
/* 001   */ 0,	/* 002   */ 0,	/* 003   */ 0,	/* 004   */ 0,	/* 005   */ 0,
/* 006   */ 0,	/* 007   */ 0,	/* 008   */ 0,	/* 009   */ 0,	/* 010   */ 0,
/* 011   */ 0,	/* 012   */ 0,	/* 013   */ 0,	/* 014   */ 0,	/* 015   */ 0,
/* 016   */ 0,	/* 017   */ 0,	/* 018   */ 0,	/* 019   */ 0,	/* 020   */ 0,
/* 021   */ 0,	/* 022   */ 0,	/* 023   */ 0,	/* 024   */ 0,	/* 025   */ 0,
/* 026   */ 0,	/* 027   */ 0,	/* 028   */ 0,	/* 029   */ 0,	/* 030   */ 0,
/* 031   */ 0,	/* 032   */ 0,	/* 033 ! */ 1,	/* 034 " */ 0,	/* 035 # */ 1,
/* 036 $ */ 1,	/* 037 % */ 1,	/* 038 & */ 1,	/* 039 ' */ 1,	/* 040 ( */ 1,
/* 041 ) */ 1,	/* 042 * */ 0,	/* 043 + */ 1,	/* 044 , */ 1,	/* 045 - */ 1,
/* 046 . */ 0,	/* 047 / */ 0,	/* 048 0 */ 1,	/* 049 1 */ 1,	/* 050 2 */ 1,
/* 051 3 */ 1,	/* 052 4 */ 1,	/* 053 5 */ 1,	/* 054 6 */ 1,	/* 055 7 */ 1,
/* 056 8 */ 1,	/* 057 9 */ 1,	/* 058 : */ 0,	/* 059 ; */ 1,	/* 060 < */ 1,
/* 061 = */ 1,	/* 062 > */ 1,	/* 063 ? */ 0,	/* 064 @ */ 1,	/* 065 A */ 1,
/* 066 B */ 1,	/* 067 C */ 1,	/* 068 D */ 1,	/* 069 E */ 1,	/* 070 F */ 1,
/* 071 G */ 1,	/* 072 H */ 1,	/* 073 I */ 1,	/* 074 J */ 1,	/* 075 K */ 1,
/* 076 L */ 1,	/* 077 M */ 1,	/* 078 N */ 1,	/* 079 O */ 1,	/* 080 P */ 1,
/* 081 Q */ 1,	/* 082 R */ 1,	/* 083 S */ 1,	/* 084 T */ 1,	/* 085 U */ 1,
/* 086 V */ 1,	/* 087 W */ 1,	/* 088 X */ 1,	/* 089 Y */ 1,	/* 090 Z */ 1,
/* 091 [ */ 1,	/* 092 \ */ 0,	/* 093 ] */ 1,	/* 094 ^ */ 1,	/* 095 _ */ 1,
/* 096 ` */ 1,	/* 097 a */ 1,	/* 098 b */ 1,	/* 099 c */ 1,	/* 100 d */ 1,
/* 101 e */ 1,	/* 102 f */ 1,	/* 103 g */ 1,	/* 104 h */ 1,	/* 105 i */ 1,
/* 106 j */ 1,	/* 107 k */ 1,	/* 108 l */ 1,	/* 109 m */ 1,	/* 110 n */ 1,
/* 111 o */ 1,	/* 112 p */ 1,	/* 113 q */ 1,	/* 114 r */ 1,	/* 115 s */ 1,
/* 116 t */ 1,	/* 117 u */ 1,	/* 118 v */ 1,	/* 119 w */ 1,	/* 120 x */ 1,
/* 121 y */ 1,	/* 122 z */ 1,	/* 123 { */ 1,	/* 124 | */ 1,	/* 125 } */ 1,
/* 126 ~ */ 1,	/* 127   */ 0,	/* 128   */ 0,	/* 129   */ 0,	/* 130   */ 0
/* 131 - 255 automatically set to 0 */
};

/*
 * ISO: A..Z0..9_
 */

# define ISO_VALID(c)		(iso_table [(long) c])
# define ISO_TABLE		(iso_table)
static const char iso_table [256] =
{
/* 000   */ 0,
/* 001   */ 0,	/* 002   */ 0,	/* 003   */ 0,	/* 004   */ 0,	/* 005   */ 0,
/* 006   */ 0,	/* 007   */ 0,	/* 008   */ 0,	/* 009   */ 0,	/* 010   */ 0,
/* 011   */ 0,	/* 012   */ 0,	/* 013   */ 0,	/* 014   */ 0,	/* 015   */ 0,
/* 016   */ 0,	/* 017   */ 0,	/* 018   */ 0,	/* 019   */ 0,	/* 020   */ 0,
/* 021   */ 0,	/* 022   */ 0,	/* 023   */ 0,	/* 024   */ 0,	/* 025   */ 0,
/* 026   */ 0,	/* 027   */ 0,	/* 028   */ 0,	/* 029   */ 0,	/* 030   */ 0,
/* 031   */ 0,	/* 032   */ 0,	/* 033 ! */ 0,	/* 034 " */ 0,	/* 035 # */ 0,
/* 036 $ */ 0,	/* 037 % */ 0,	/* 038 & */ 0,	/* 039 ' */ 0,	/* 040 ( */ 0,
/* 041 ) */ 0,	/* 042 * */ 0,	/* 043 + */ 0,	/* 044 , */ 0,	/* 045 - */ 0,
/* 046 . */ 0,	/* 047 / */ 0,	/* 048 0 */ 1,	/* 049 1 */ 1,	/* 050 2 */ 1,
/* 051 3 */ 1,	/* 052 4 */ 1,	/* 053 5 */ 1,	/* 054 6 */ 1,	/* 055 7 */ 1,
/* 056 8 */ 1,	/* 057 9 */ 1,	/* 058 : */ 0,	/* 059 ; */ 0,	/* 060 < */ 0,
/* 061 = */ 0,	/* 062 > */ 0,	/* 063 ? */ 0,	/* 064 @ */ 0,	/* 065 A */ 1,
/* 066 B */ 1,	/* 067 C */ 1,	/* 068 D */ 1,	/* 069 E */ 1,	/* 070 F */ 1,
/* 071 G */ 1,	/* 072 H */ 1,	/* 073 I */ 1,	/* 074 J */ 1,	/* 075 K */ 1,
/* 076 L */ 1,	/* 077 M */ 1,	/* 078 N */ 1,	/* 079 O */ 1,	/* 080 P */ 1,
/* 081 Q */ 1,	/* 082 R */ 1,	/* 083 S */ 1,	/* 084 T */ 1,	/* 085 U */ 1,
/* 086 V */ 1,	/* 087 W */ 1,	/* 088 X */ 1,	/* 089 Y */ 1,	/* 090 Z */ 1,
/* 091 [ */ 0,	/* 092 \ */ 0,	/* 093 ] */ 0,	/* 094 ^ */ 0,	/* 095 _ */ 1,
/* 096 ` */ 0,	/* 097 a */ 1,	/* 098 b */ 1,	/* 099 c */ 1,	/* 100 d */ 1,
/* 101 e */ 1,	/* 102 f */ 1,	/* 103 g */ 1,	/* 104 h */ 1,	/* 105 i */ 1,
/* 106 j */ 1,	/* 107 k */ 1,	/* 108 l */ 1,	/* 109 m */ 1,	/* 110 n */ 1,
/* 111 o */ 1,	/* 112 p */ 1,	/* 113 q */ 1,	/* 114 r */ 1,	/* 115 s */ 1,
/* 116 t */ 1,	/* 117 u */ 1,	/* 118 v */ 1,	/* 119 w */ 1,	/* 120 x */ 1,
/* 121 y */ 1,	/* 122 z */ 1,	/* 123 { */ 0,	/* 124 | */ 0,	/* 125 } */ 0,
/* 126 ~ */ 0,	/* 127   */ 0,	/* 128   */ 0,	/* 129   */ 0,	/* 130   */ 0
/* 131 - 255 automatically set to 0 */
};

FASTFN short
is_short (register const char *src, register const char *table)
{
	register long i = _STRLEN (src);
	
	FAT_DEBUG (("is_short: enter (src = %s, len = %li)\r\n", src, i));
	
	/* verify length and beginning point */
	if (i > 12 || *src == '.')
	{
		FAT_DEBUG (("is_short: leave check 0 (src = %s)\r\n", src));
		return 0;
	}
	
	/* verify base name */
	i = 8;
	while (i-- && *src && *src != '.')
		if (table [(long) *src])
			src++;
		else
		{
			FAT_DEBUG (("is_short: leave check 1 (src = %s)\r\n", src));
			return 0;
		}
	
	/* verify extension */
	if (*src == '.')
	{
		src++;
		i = 3;
		while (i-- && *src && *src != '.')
			if (table [(long) *src])
				src++;
			else
			{
				FAT_DEBUG (("is_short: leave check 2 (src = %s)\r\n", src));
				return 0;
			}
	}
	
	/* anything left? */
	if (*src)
	{
		FAT_DEBUG (("is_short: leave check 3 (src = %s)\r\n", src));
		return 0;
	}
	
	FAT_DEBUG (("is_short: leave ok (return = 1)\r\n"));
	return TOS_SEARCH; /* shortname */
}

FASTFN long
fat_trunc (register char *dst, const char *src, register long len, COOKIE *dir)
{
	/*
	 * truncate name to 8+3 and verify that the name
	 * doesn't exist in this dir
	 * 
	 * 1: select the first eight characters
	 * 
	 * 2: if there is an extension, select its first three
	 *    characters
	 * 
	 * 3: convert letters to uppercase.
	 * 
	 * 4: convert to underscore any character
	 *    that are illegal
	 * 
	 * 5: check for existing filename
	 */
	
	register const char *table = DEFAULT_TABLE (dir->dev);
	register const char *s = src;
	register long i;
	
	/* step 1 - 4 */
	
	if (*s == '.') s++;
	i = 8;
	while (i-- && *s && *s != '.')
	{
		*dst++ = table [(long) *s] ? _TOUPPER (*s) : '_';
		s++;
	}
	
	s = src + len; s--;
	while (*s && *s != '.') s--;
	
	if (*s == '.')
	{
		*dst++ = '.';
		s++;
		for (i = 1; i < 4 && *s; i++)
		{
			*dst++ = table [(long) *s] ? _TOUPPER (*s) : '_';
			s++;
		}
	}
	
	/* step 5 */
	i = search_cookie (dir, NULL, dst, TOS_SEARCH);
	
	if (i == E_OK)
	{
		/* file exist */
		return EACCDN;
	}
	
	/* file doesn't exist */
	return E_OK;
}

FASTFN long
vfat_trunc (register char *dst, const char *src, register long len, COOKIE *dir)
{
	/*
	 * truncate name to 8+3 and verify that the name
	 * doesn't exist in this dir
	 *
	 * 1: select the first eight characters
	 * 
	 * 2: if there is an extension, select its first three
	 *    characters
	 * 
	 * 3: convert letters to uppercase.
	 * 
	 * 4: convert to underscore any character that are illegal
	 *    under the FAT fs; valid are:
	 *    character, digit and $ % ' - _ @ ~ ` ! ( ) # { }
	 * 
	 * 5: if the resulting name already exists in the same dir,
	 *    replace the last two characters with a tilde and a
	 *    unique integer (Even if the resulting name is unique,
	 *    replace the last two characters if the long filename
	 *    has embedded spaces or illegal characters.)
	 */
	
	char ext [4] = { '\0', '\0', '\0', '\0' };
	register const char *s = src;
	register char *d = dst;
	register char *bak;
	register long i;
	
	/* first zero the dst */
	for (i = 13; i; i--)
		*d++ = '\0';
	
	/* step 1 - 4 */
	
	if (*s == '.') s++;
	d = dst;
	i = 8;
	while (i-- && *s && *s != '.')
	{
		*d++ = MSDOS_VALID (*s) ? _TOUPPER (*s) : '_';
		s++;
	}
	
	bak = d - 1;
	
	s = src + len; s--;
	while (*s && *s != '.') s--;
	
	if (*s == '.')
	{
		ext [0] = *d++ = '.';
		s++;
		for (i = 1; i < 4 && *s; i++)
		{
			ext [i] = *d++ = MSDOS_VALID (*s) ? _TOUPPER (*s) : '_';
			s++;
		}
	}
	
	/* step 5 */
	i = bak - dst;
	if (i < 7)
	{
		i = 7 - i;
		bak += MIN (2, i);
		for (i = 4; i; i--)
			*(bak + i) = ext[i-1];
	}
	i = 1;
	*bak = ((char) i) + '0'; *(bak - 1) = '~';
	while (search_cookie (dir, NULL, dst, TOS_SEARCH) == E_OK)
	{
		register long j;
		if (i == 10L || i == 100L || i == 1000L || i == 10000L || i == 100000L)
		{
			j = bak - dst;
			if (j < 7)
			{
				bak++;
				for (j = 4; j; j--)
					*(bak + j) = ext[j-1];
			}
		}
		d = bak;
		j = i;
		while (j)
		{
			*d-- = ((char) (j % 10)) + '0';
			j /= 10;
		}
		*d = '~';
		i++;
	}
	
	return E_OK;
}

FASTFN long
make_shortname (COOKIE *dir, const char *src, char *dst)
{
	register const long len = _STRLEN (src);
	const short sflag = is_short (src, DEFAULT_TABLE (dir->dev));
	
# ifdef FS_DEBUG
	const char *start = dst;
# endif
	
	FAT_DEBUG (("make_shortname: enter (src = %s, len, = %li, short = %li)\r\n", src, len, sflag));
	if (sflag == 0)
	{
		long r;
		
		if (VFAT (dir->dev))
			r = vfat_trunc (dst, src, len, dir);
		else
			r = fat_trunc (dst, src, len, dir);
		
		if (r) return r;
	}
	else
	{
		/*
		 * copy the name (it's in correct 8+3 format)
		 */
		
		register const char *s = src;
		while (*s)
		{
			*dst++ = _TOUPPER (*s);
			s++;
		}
		*dst ='\0';
	}
	
	if (VFAT (dir->dev))
	{
		if (sflag == 0)
		{
			/* a really longname */
			
			FAT_DEBUG (("make_shortname: leave ok (VFAT) (s = %s, return = %li)\r\n", start, len));
			return len;
		}
		
		/* casesensitive check */
		if (LCASE (dir->dev))
		{
# if 0
			register const char *s = src;
			while (*s)
			{
				if (*s >= 'A' && *s <= 'Z')
				{
					FAT_DEBUG (("make_shortname: leave ok (casesensitive [lcase]) (s = %s, return = %li)\r\n", start, len));
					return len;
				}
				else
					s++;
			}
# else
			/* create always VFAT names
			 * to avoid problems with wrong upper/lowercase
			 * under MagiC/Windows
			 */
			FAT_DEBUG (("make_shortname: leave ok (casesensitive [lcase]) (s = %s, return = %li)\r\n", start, len));
			return len;
# endif
		}
		else
		{
			register const char *s = src;
			while (*s)
			{
				if (*s >= 'a' && *s <= 'z')
				{
					FAT_DEBUG (("make_shortname: leave ok (casesensitive) (s = %s, return = %li)\r\n", start, len));
					return len;
				}
				else
					s++;
			}
		}
		
		/* name is in 8+3 and correct upper/lower case */
	}
	
	/* else { truncated, copied and verifed before } */
	
	FAT_DEBUG (("make_shortname: leave ok (FAT) (s = %s, return = 0)\r\n", start));
	return E_OK;
}

FASTFN long
__opendir (register oDIR *dir, register const long cluster, register const ushort dev)
{
	FAT_DEBUG (("__opendir: enter\r\n"));
	
	dir->dev = dev;
	dir->rdev = dev;
	dir->stcl = cluster;
	dir->actual = 0;
	dir->cl = -1;
	dir->index = -1;
	dir->real_index = -1;
	dir->u = NULL;
	
	FAT_DEBUG (("__opendir: leave ok\r\n"));
	return E_OK;
}

FASTFN void
__closedir (register oDIR *dir)
{
	FAT_DEBUG (("__closedir: enter\r\n"));
	
	if (dir->u)
		bio.unlock (dir->u);
	
	FAT_DEBUG (("__closedir: leave ok\r\n"));
}

FASTFN void
__updatedir (register oDIR *dir)
{
	FAT_DEBUG (("__updatedir: enter\r\n"));
	
	bio_MARK_MODIFIED (&(bio), dir->u);
	
	FAT_DEBUG (("__updatedir: leave ok\r\n"));
}

static long
__seekdir (register oDIR *dir, register long index, short mode)
{
	register const ushort dev = dir->dev;
	register long offset;
	
	FAT_DEBUG (("__seekdir: enter (index = %li)\r\n", index));
	
	if (dir->stcl == 1)
	{
		/* ROOT DIR */
		
		if (index >= ROOTENTRYS (dev))
		{
			/* no more in root dir */
			
			FAT_DEBUG (("__seekdir: leave ok (no more in root, dev = %i)\r\n", dir->dev));
			return ENMFIL;
		}
		else
		{
			/* calculate the offset */
			
			register const long entrys = ENTRYS (dev) / CLSIZE (dev);
			register const long sector = index / entrys;
			
			offset = index % entrys;
			
			if (sector != dir->cl)
			{
				/* read the right unit */
				
				if (dir->u)
					bio.unlock (dir->u);
				
				dir->actual = sector;
				dir->cl = sector;
				dir->u = bio.read (DI (dev), ROOT (dev) + sector, SECSIZE (dev));
				
				if (!dir->u)
				{
					dir->cl = -1;
					dir->info = NULL;
					FAT_DEBUG (("__seekdir: leave failure (read cluster, sector = %li);\r\n", ROOT (dir->dev) + sector));
					return EREADF;
				}
				
				bio.lock (dir->u);
			}
		}
	}
	else
	{
		/* SUB DIR */
		
		register const long entrys = ENTRYS (dev);
		register const long cluster = index / entrys;
		
		offset = index % entrys;
		
		if (cluster != dir->cl)
		{
			/* read the right cluster */
			
			register long actual = GETCL (dir->stcl, dev, cluster);
			if (actual <= 0)
			{
				FAT_DEBUG (("__seekdir: failure get right cluster (error = %li)\r\n", actual));
				if (mode == READ)
				{
					FAT_DEBUG (("__seekdir: leave ENMFIL (read only)\r\n"));
					return ENMFIL;
				}
				actual = GETCL (dir->stcl, dev, cluster - 1);
				if (actual <= 0)
				{
					FAT_DEBUG (("__seekdir: can't read prev cl (error = %li)", actual));
					return ENMFIL;
				}
				actual = nextcl (actual, dev);
				if (actual <= 0)
				{
					FAT_DEBUG (("__seekdir: can't expand dir (error = %li)", actual));
					return ENMFIL;
				}
				zero_cl (actual, dev);
			}
			
			if (dir->u)
				bio.unlock (dir->u);
			
			dir->actual = actual;
			dir->cl = cluster;
			dir->u = bio.read (DI (dev), C2S (actual, dev), CLUSTSIZE (dev));
			
			if (!dir->u)
			{
				dir->cl = -1;
				dir->info = NULL;
				FAT_DEBUG (("__seekdir: leave failure (read cluster, sector = %li);\r\n", dir->actual));
				return EREADF;
			}
			
			bio.lock (dir->u);
		}
	}
	
	dir->index = dir->real_index = index;
	dir->info = ((_DIR *) dir->u->data) + offset;
	
	FAT_DEBUG (("__seekdir: leave ok\r\n"));
	return E_OK;
}

FASTFN long
__SEEKDIR (register oDIR *dir, register long index, short mode)
{
	register const long entrys = (dir->stcl > 1) ?
		ENTRYS (dir->dev) :
		ENTRYS (dir->dev) / CLSIZE (dir->dev);
	
	FAT_DEBUG (("__SEEKDIR: set index = %li\r\n", index));
	if ((index / entrys) == dir->cl)
	{
		dir->index = dir->real_index = index;
		dir->info = ((_DIR *) dir->u->data) + (index % entrys);
		return E_OK;
	}
	return __seekdir (dir, index, mode);
}

# ifdef USE_INLINE_FUNCS
FASTFN void
ldir2unicode (long slot, uchar *name, LDIR *ldir)
{
 	register long offset = slot * 13 * 2;
	register long i;
	
	for (i = 0; i < 10; i++)
		name [offset+i] = ldir->name0_4   [i];
	
	offset += 10;
	for (i = 0; i < 12; i++)
		name [offset+i] = ldir->name5_10  [i];
	
	offset += 12;
	for (i = 0; i < 4; i++)
		name [offset+i] = ldir->name11_12 [i];
}
# else
# define ldir2unicode(slot, name, ldir) \
{	register long offset = (slot) * 13 * 2;			\
	register long i;					\
								\
	for (i = 0; i < 10; i++)				\
		(name) [offset+i] = (ldir)->name0_4   [i];	\
	offset += 10;						\
	for (i = 0; i < 12; i++)				\
		(name) [offset+i] = (ldir)->name5_10  [i];	\
	offset += 12;						\
	for (i = 0; i < 4; i++)					\
		(name) [offset+i] = (ldir)->name11_12 [i];	\
}
# endif

FASTFN long
__readvfat (register oDIR *dir, char *lname, long size)
{
	LDIR *ldir = (LDIR *) dir->info;
	
	FAT_DEBUG (("__readvfat: enter (real = %li, index = %li)\r\n", dir->real_index, dir->index));
	
	if (ldir->head & 0x40)
	{
		uchar name [VFAT_NAMEMAX * 2];
		long r = 0;
		long is_long = 1;
		
		long slots = ldir->head & ~0x40;
		long slot = slots - 1;
		uchar chksum = ldir->chksum;
		
		FAT_DEBUG (("__readvfat: slots = %li, chksum = %li\r\n", slots, (long) chksum));
		
		if (lname) ldir2unicode (slot, name, ldir);
		
		while (slot > 0)
		{
			r = __SEEKDIR (dir, dir->real_index + 1, READ);
			if (r) return r;
			ldir = (LDIR *) dir->info;
			
			if (ldir->head != slot)
			{
				FAT_DEBUG (("__readvfat: error (head != slot, head = %li)\r\n", (long) ldir->head));
				is_long = 0;
				break;
			}
			if (ldir->chksum != chksum)
			{
				FAT_DEBUG (("__readvfat: error (bad chksum, slot = %li, chksum = %li)\r\n", slot, (long) ldir->chksum));
				is_long = 0;
				break;
			}
			
			slot--;
			
			if (lname) ldir2unicode (slot, name, ldir);
		}
		
		if (is_long)
		{
			register uchar sum;
			
			r = __SEEKDIR (dir, dir->real_index + 1, READ);
			if (r) return r;
			
			ldir = (LDIR *) dir->info;
			
			FAT_DEBUG_PRINTDIR ((dir->info, dir->dev));
			
			/* calculate the checksum */
			{
				register long i;
				for (sum = i = 0; i < 11; i++)
				{
					sum = (((sum & 1) << 7) | ((sum & 0xfe) >> 1))
						+ dir->info->name [i];
				}
			}
			
			if (sum == chksum)
			{
				if (lname)
				{
					/* copy Unicode string to Atari ST character string */
					register long i = slots * 13;
					
					/* range check */
					if (i >= size)
					{
						i--; /* null relative */
						i <<= 1;
						for (; i; i -= 2)
						{
							if (name [i] == '\0' && name [i+1] == '\0')
							{
# if 0
								i -= 2;
								i++;
# else /* faster */
								i >>= 1;
# endif
								break;
							}
						}
						if (i == 0 || i > size - 1)
						{
							i = size - 1;
							slots = ENAMETOOLONG;
						}
					}
					
					/* copy */
					lname [i] = '\0';
					while (i--)
					{
						register const long j = i << 1;
						lname [i] = uni2atari ((name [j]), (name [j+1]));
					}
					FAT_DEBUG (("__readvfat: (lname = %s)\r\n", lname));
				}
				FAT_DEBUG (("__readvfat: leave ok\r\n"));
				return slots;
			}
			else
			{
				FAT_DEBUG (("__readvfat: error (bad chksum, sum = %li)\r\n", (long) sum));
			}
		}
		
		/* repositionate backward (for nextdir) */
		r = __SEEKDIR (dir, dir->real_index - 1, READ);
		
		return r;
	}
	
	FAT_DEBUG (("__readvfat: leave failure (not valid head)\r\n"));
	return 0;
}

static long
__nextdir (register oDIR *dir, char *name, long size)
{
	for (;;)
	{
		register long index = dir->real_index + 1;
		register long r;
		
		r = __SEEKDIR (dir, index, READ);
		if (r) return r;
		
		if (dir->info->name[0] == 0)
		{
			/* never used */
			
			FAT_DEBUG (("__nextdir: leave failure (no more) index = %li\r\n", index));
			return ENMFIL;
		}
		else
		
		if (dir->info->name[0] == (char) 0xe5)
		{
			/* a invalid entry */
			
			FAT_DEBUG (("__nextdir: (skip invalid) index = %li\r\n", index));
			/* skip */
		}
		else
		
		if (dir->info->attr == FA_VFAT)
		{
			/* a vfat entry */
			
			r = __readvfat (dir, name, size);
			if (r > 0)
			{
				dir->index = index;
				return r;
			}
			
			if (r == ENAMETOOLONG)
			{
				FAT_DEBUG (("__nextdir: ENAMETOOLONG (cookie invalid)\r\n"));
				return r;
			}
			
			FAT_DEBUG (("__nextdir: (vfat entry, return = %li, index = %li, real = %li)\r\n", r, index, dir->real_index));
			/* failure, skip */
		}
		else
		
		if ((dir->info->attr & (FA_DIR|FA_LABEL)) == FA_LABEL)
		{
			/* the label or an invalid vfat entry */
			
			FAT_DEBUG (("__nextdir: (skip label) index = %li\r\n", index));
			/* skip */
		}
		else
		{
			/* a valid entry */
			
			if (name) dir2str (dir->info->name, name);
			return E_OK;
		}
	}
	
	FAT_DEBUG (("__nextdir: leave failure (unknown)\r\n"));
	return ENMFIL;
}


static long
search_cookie (COOKIE *dir, COOKIE **found, const char *name, short mode)
{
	oDIR odir;
	long r;
	
	FAT_DEBUG (("search_cookie: enter (name = %s)\r\n", name));
	
	/* 1. search in COOKIE cache */
	{
		COOKIE *search;
		char *temp = fullname (dir, name);
		if (!temp)
		{
			FAT_DEBUG (("search_cookie: leave failure, out of memory\r\n"));
			return ENSMEM;
		}
		
		FAT_DEBUG (("search_cookie: looking for: %s\r\n", temp));
		
		search = c_lookup (temp, dir->dev); kfree (temp);
		if (search)
		{
			if (found)
			{
				*found = search;
				search->links++;
			}
			FAT_DEBUG (("search_cookie: leave ok, found in table\r\n"));
			return E_OK;
		}
	}
	
	if (dir->extra)
	{
		if (_STRICMP (name, dir->extra) == 0)
		{
			FAT_DEBUG (("search_cookie: leave not found (extra) (name = %s)\r\n", name));
			return EFILNF;
		}
	}
	
	/* 2. search on disk */
	
	if (mode == 0) mode = is_short (name, VFAT (dir->dev) ? MSDOS_TABLE : GEMDOS_TABLE);
	
	r = __opendir (&odir, dir->stcl, dir->dev);
	if (r) return r;
	
	if (mode == TOS_SEARCH)
	{
		/* FAT search on FAT or VFAT */
		
		char fat_name [FAT_NAMEMAX];
		
		str2dir (name, fat_name);
		while ((r = __nextdir (&odir, NULL, 0)) >= 0)
		{
			if (_STRNICMP (odir.info->name, fat_name, 11) == 0)
			{
				if (found)
				{
					char *temp;
					char buf [VFAT_NAMEMAX];
					
					if (r)
					{
						long j;
						(void) __SEEKDIR (&odir, odir.index - 1, READ);
						j = __nextdir (&odir, buf, VFAT_NAMEMAX);
						if (r != j)
						{
							ALERT ("fatfs.c: filesystem inconsistent in search_cookie");
							return EFILNF;
						}
					}
					else
					{
						dir2str (odir.info->name, buf);
					}
					
					temp = fullname (dir, buf);
					if (temp == NULL)
					{
						FAT_DEBUG (("search_cookie: leave failure, out of memory\r\n"));
						return ENSMEM;
					}
					
					if (r)
					{
						*found = c_lookup (temp, dir->dev);
						if (*found)
						{
							(*found)->links++;
							kfree (temp);
							FAT_DEBUG (("search_cookie: found in table (FAT search, r = %li)\r\n", r));
							return E_OK;
						}
					}
					
					*found = get_cookie (temp);
					if (*found == NULL)
					{
						kfree (temp);
						return EFILNF;
					}
					
					(*found)->dev = dir->dev;
					(*found)->rdev = dir->dev;
					(*found)->dir = dir->stcl;
					(*found)->offset = odir.index;
					(*found)->stcl = GET_STCL (odir.info, dir->dev);
					(*found)->info = *(odir.info);
					(*found)->slots = r;
				}
				__closedir (&odir);
				FAT_DEBUG (("search_cookie: leave found (FAT search)\r\n"));
				return E_OK;
			}
		}
	}
	else
	{
		/* VFAT search */
		
		char buf [VFAT_NAMEMAX];
		
		while ((r = __nextdir (&odir, buf, VFAT_NAMEMAX)) >= 0)
		{
			if (_STRICMP (buf, name) == 0) /* always not casesensitive, yes, right */
			{
				if (found)
				{
					char *temp;
					
					temp = fullname (dir, buf);
					if (temp == NULL)
					{
						FAT_DEBUG (("search_cookie: leave failure, out of memory\r\n"));
						return ENSMEM;
					}
					
					*found = get_cookie (temp);
					if (*found == NULL)
					{
						kfree (temp);
						return EFILNF;
					}
					
					(*found)->dev = dir->dev;
					(*found)->rdev = dir->dev;
					(*found)->dir = dir->stcl;
					(*found)->offset = odir.index;
					(*found)->stcl = GET_STCL (odir.info, dir->dev);
					(*found)->info = *(odir.info);
					(*found)->slots = r;
				}
				__closedir (&odir);
				FAT_DEBUG (("search_cookie: leave found (VFAT search)\r\n"));
				return E_OK;
			}
		}
	}
	
	{
		if (dir->extra)
			kfree (dir->extra);
		
		dir->extra = kmalloc (_STRLEN (name) + 1);
		if (dir->extra)
			(void) _STRCPY (dir->extra, name);
	}
	
	__closedir (&odir);
	
	FAT_DEBUG (("search_cookie: leave failure (name = %s)\r\n", name));
	return EFILNF;
}

static long
write_cookie (const COOKIE *c)
{
	oDIR dir;
	register long r;
	
	FAT_DEBUG (("write_cookie: enter (dir = %li, offset = %li, dev = %i)\r\n", c->dir, c->offset, c->dev));
	
	if (c->dir == 0)
	{
		/* ROOT */
		
		FAT_DEBUG (("write_cookie: leave failure (root)\r\n"));
		return EACCDN;
	}
	
	/* open the dir */
	r = __opendir (&dir, c->dir, c->dev);
	
	r = __seekdir (&dir, c->offset + c->slots, READ);
	if (r < E_OK)
	{
		__closedir (&dir);
		
		FAT_DEBUG (("write_cookie: leave failure (__seekdir)\r\n"));
		return r;
	}
	
	/* copy the dirstruct */
	*dir.info = c->info;
	
	__updatedir (&dir);
	__closedir (&dir);
	
	FAT_DEBUG (("write_cookie: leave ok (return E_OK)\r\n"));
	return E_OK;
}

static long
delete_cookie (COOKIE *c, long mode)
{
	oDIR dir;
	register long r;
	
	FAT_DEBUG (("delete_cookie: enter (dir = %li, offset = %i, dev = %i)\r\n", c->dir, c->offset, c->dev));
	
	if (c->dir == 0)
	{
		/* ROOT */
		
		FAT_DEBUG (("delete_cookie: leave failure (root)\r\n"));
		return EACCDN;
	}
	
	if (c->links > 1)
	{
		FAT_DEBUG (("delete_cookie: leave failure (cookie in use)\r\n"));
		return EACCDN;
	}
	
	/* open the dir */
	r = __opendir (&dir, c->dir, c->dev);
	
	/* seek to the right dir */
	r = __seekdir (&dir, c->offset + c->slots, READ);
	if (r < E_OK)
	{
		__closedir (&dir);
		
		FAT_DEBUG (("delete_cookie: leave failure (__seekdir)\r\n"));
		return r;
	}
	
	/* invalidate the FAT entry */
	dir.info->name[0] = 0xe5;
	__updatedir (&dir);
	
	/* delete the cluster-chain */
	if (mode && c->stcl) (void) del_chain (c->stcl, c->dev);
	
	if (c->slots)
	{
		/* invalidate the VFAT entrys */
		r = c->slots;
		while (r--)
		{
			(void) __SEEKDIR (&dir, c->offset + r, READ);
			dir.info->name[0] = 0xe5;
			__updatedir (&dir);
		}
	}
	
	__closedir (&dir);
	c_remove (c);
	
	FAT_DEBUG (("delete_cookie: leave ok\r\n"));
	return E_OK;
}

FASTFN void
copy_to_vfat (LDIR *ldir, register const char *name)
{
	char unicode[26];
	register long i;
	register long slot = ldir->head;
	
	slot = (slot & 0x40) ? slot & ~0x40 : slot;
	slot--;
	name += slot * 13;
	
	FAT_DEBUG (("copy_to_vfat: enter (name = %s, ldir->head = %li)\r\n", name, slot));
	
	for (i = 0; *name && i < 25; i += 2)
	{
		atari2uni (*name, unicode + i);
		name++;
	}
	if (i < 25)
	{
		unicode[i++] = '\0';
		unicode[i++] = '\0';
		if (i < 25)
			for (; i < 26; i++)
				unicode[i] = 0xff;
	}
	
	for (i = 0; i < 10; i++)
		ldir->name0_4   [i] = unicode [i];
	for (i = 0; i < 12; i++)
		ldir->name5_10  [i] = unicode [i+10];
	for (i = 0; i < 4; i++)
		ldir->name11_12 [i] = unicode [i+22];
	
	FAT_DEBUG (("copy_to_vfat: leave ok\r\n"));
}

FASTFN long
search_free (oDIR *dir, long *pos, long slots)
{
	long r;
	
	r = __seekdir (dir, *pos, WRITE);
	while (r == 0)
	{
		if (dir->info->name[0] == (char) 0xe5)
		{
			/* a deleted entry */
			if (slots > 1)
			{
				/* we need more as one, check for more */
				register long tpos = *pos;
				register long searched = slots;
				while (searched-- && (r == 0))
				{
					r = __SEEKDIR (dir, ++tpos, WRITE);
					if (dir->info->name[0] &&
						dir->info->name[0] != (char) 0xe5)
					{
						*pos = tpos;
						break;
					}
				}
				if (((tpos - *pos) == slots) && (r == 0 || r == ENMFIL))
				{
					/* yes, we found enough deleted entrys :-) */
					/* seek to the old pos */
					r = __SEEKDIR (dir, *pos, WRITE);
					break;
				}
			} else
				/* we only need one */
				break;
		} else
		
		if (dir->info->name[0] == 0)
			/* here are the end of the dir */
			break;
		
		/* go to the next entry */
		(*pos)++;
		r = __SEEKDIR (dir, *pos, WRITE);
	}
	return r;
}

static long
make_cookie (COOKIE *dir, COOKIE **new, const char *name, int attr)
{
	oDIR odir;
	char shortname[13];
	char *full;
	long pos = 0;
	long vfat;
	long slot;
	long r;
	
	FAT_DEBUG (("make_cookie: enter (name = %s, stcl = %li, pos = %li)\r\n", name, dir->stcl, pos));
	
	if (dir->extra)
	{
		kfree (dir->extra);
		dir->extra = NULL;
	}
	
	/*
	 * validate the name and make a correkt 8+3 name
	 * if VFAT is enabled and the name is a longname
	 * 'vfat' holds the len of the longname
	 */
	
	vfat = make_shortname (dir, name, shortname);
	FAT_DEBUG (("make_cookie: (8+3 = %s, vfat = %li)\r\n", shortname, vfat));
	if (vfat < 0) return vfat; /* invalid name (no vfat) */
	
	if (vfat)
	{
		/* slots to be filled out */
		vfat = (vfat / 13) + ((vfat % 13) ? 1 : 0);
		FAT_DEBUG (("make_cookie: (slots = %li)\r\n", vfat));
	}
	
	r = __opendir (&odir, dir->stcl, dir->dev);
	r = search_free (&odir, &pos, vfat ? vfat + 1 : 1);
	
	full = fullname (dir, name);
	*new = get_cookie (full);
	
	/* if anything goes wrong we leave here */
	if (r || !full || !*new)
	{
		FAT_DEBUG (("make_cookie: leave failure (r = %li)\r\n", r));
		if (*new) c_remove (*new);
		__closedir (&odir);
		return r;
	}
	
	/* copy the short name to the entry */
	str2dir (shortname, (*new)->info.name);
	
	if (vfat)
	{
		/* a VFAT entry */
		
		/* calculate the checksum */
		register uchar chksum = 0;
		{
			register long i;
			for (chksum = i = 0; i < 11; i++)
				chksum = (((chksum & 1) << 7) | ((chksum & 0xfe) >> 1))
					+ (*new)->info.name[i];
		}
		
		FAT_DEBUG (("make_cookie: chksum = %li, pos = %li\r\n", (ulong) chksum, pos));
		
		/* now we write the VFAT entrys */
		slot = 0;
		while (slot < vfat)
		{
			register LDIR *ldir = (LDIR *) odir.info;
			
			FAT_DEBUG (("make_cookie: writing VFAT, slot = %li\r\n", slot));
			
			/* fill out the slot */
			ldir->head = vfat - slot; if (slot == 0) ldir->head |= 0x40;
			ldir->attr = FA_VFAT;
			ldir->unused = 0;
			ldir->stcl = 0;
			ldir->chksum = chksum;
			copy_to_vfat (ldir, name);
			
			__updatedir (&odir);
			
			r = __SEEKDIR (&odir, pos + slot + 1, WRITE);
			if (r)
			{
				FAT_DEBUG (("make_cookie: leave failure (__seekdir = %li)\r\n", r));
				/* remove from cache */
				c_remove (*new);
				__closedir (&odir);
				return r;
			}
			slot++;
		}
	}
	
	/* now make the FAT entry */
	
	(*new)->info.attr = attr;
	(*new)->info.time = SWAP68_W (timestamp);
	(*new)->info.date = SWAP68_W (datestamp);
	(*new)->info.stcl = 0;
	(*new)->info.stcl_fat32 = 0;
	(*new)->info.flen = 0;
	
	if (VFAT (dir->dev))
	{
		(*new)->info.lcase = 0;
		(*new)->info.ctime_ms = 0;
		(*new)->info.ctime = (*new)->info.time;
		(*new)->info.cdate = (*new)->info.date;
		(*new)->info.adate = (*new)->info.date;
	}
	else
	{
		(*new)->info.lcase = 0;
		(*new)->info.ctime_ms = 0;
		(*new)->info.ctime = 0;
		(*new)->info.cdate = 0;
		(*new)->info.adate = 0;
	}
	
	/* validate the cookie */
	
	(*new)->dev = dir->dev;
	(*new)->rdev = dir->dev;
	(*new)->dir = dir->stcl;
	(*new)->offset = pos;
	(*new)->stcl = 0;
	(*new)->slots = vfat;
	
	*(odir.info) = (*new)->info;
	
	__updatedir (&odir);
	__closedir (&odir);
	
	FAT_DEBUG (("make_cookie: leave E_OK\r\n"));
	return E_OK;
}

/* END DIR part */
/****************************************************************************/

/****************************************************************************/
/* BEGIN init & configuration part */

void
fatfs_init (void)
{
	/* internal init */
	long i;
	for (i = 0; i < NUM_DRIVES; i++)
	{
		/* VFAT extension disabled by default */
		VFAT (i) = DISABLE;
		
		/* symbolic link configuration default on */
		SLNK (i) = ENABLE;
		
		/* new name mode configuration default to GEMDOS */
		NAME_MODE (i) = GEMDOS;
		DEFAULT_TABLE (i) = GEMDOS_TABLE;
		
		/* zero devinfo */
		BPB (i) = NULL;
		BPBVALID (i) = INVALID;
	}
	
	boot_print (FATFS_BOOT);
	boot_print (FATFS_GREET);
}

long
fatfs_config (const ushort drv, const long config, const long mode)
{
	if (drv >= 0 && drv < NUM_DRIVES)
	{
		switch (config)
		{
			case FATFS_VFAT:
			{
				if (mode == ASK) return VFAT (drv);
				VFAT (drv) = mode ? ENABLE : DISABLE;
				
				if (VFAT (drv))
				{
					NAME_MODE (drv) = MSDOS;
					DEFAULT_TABLE (drv) = MSDOS_TABLE;
				}
				else
				{
					NAME_MODE (drv) = GEMDOS;
					DEFAULT_TABLE (drv) = GEMDOS_TABLE;
				}
				break;
			}
			case FATFS_VCASE:
			{
				if (mode == ASK) return LCASE (drv);
				LCASE (drv) = mode ? ENABLE : DISABLE;
				break;
			}
# ifdef FS_DEBUG
			case FATFS_DEBUG:
			{
				(void) bio.config (drv, BIO_DEBUGLOG, mode);
				fatfs_debug_mode = mode;
				break;
			}
			case FATFS_DEBUG_T:
			{
				FAT_DEBUG_HASH (());
				break;
			}
# endif
# ifdef FATFS_TESTING
			case FATFS_DRV:
			{
				if (mode == ASK) return TEST_PARTITIONS (drv);
				TEST_PARTITIONS (drv) = mode ? ENABLE : DISABLE;
				break;
			}
# endif
			default:
			{
				/* failure */
				return EINVFN;
			}
		}
		
		/* ok */
		return E_OK;
	}
	
	/* failure */
	return EINVFN;
}

/* END init & configuration part */
/****************************************************************************/

/****************************************************************************/
/* BEGIN special FAT32 extension */

static void
upd_fat32fats (register const ushort dev, long reference)
{
	long i;
	for (i = 0; i < FATSIZE (dev); i++)
	{
		UNIT *u = bio.read (DI (dev), reference + i, SECSIZE (dev));
		if (u)
		{
			long j;
			
			bio.lock (u);
			
			for (j = 0; j <= FAT2ON (dev); j++)
			{
				long start = FAT1 (dev) + j * FATSIZE (dev);
				if (start != reference)
				{
					UNIT *mirr = bio.getunit (DI (dev), start + i, SECSIZE (dev));
					if (mirr)
					{
						memcpy (mirr->data, u->data, SECSIZE (dev));
						bio_MARK_MODIFIED (&(bio), mirr);
					}
				}
			}
			
			bio.unlock (u);
		}
	}
}

static void
upd_fat32boot (register const ushort dev)
{
	UNIT *u = bio.read (DI (dev), 0, DI (dev)->pssize);
	if (u)
	{
		_F32_BS *f32bs = (_F32_BS *) u->data;
		ushort temp;
		ushort i = 0;
		
		if (!FAT32mirr (dev))
		{
			temp = FAT32prim (dev);
			temp -= FAT1 (dev);
			temp /= FATSIZE (dev);
			i |= (temp & FAT32_ActiveFAT_Mask);
			i |= FAT32_NoFAT_Mirror;
		}
		
		f32bs->flags = SWAP68_W (i);
		
		bio_MARK_MODIFIED ((&bio), u);
	}
}

static void
val_fat32info (register const ushort dev)
{
	if (FAT32info (dev))
	{
		UNIT *u = bio.read (DI (dev), FAT32ista (dev), DI (dev)->pssize);
		if (u)
		{
			_FAT32_BFSINFO *info = (_FAT32_BFSINFO *) u->data;
			
			if (SWAP68_L (info->signature) == FAT32_FSINFOSIG)
			{
				LASTALLOC (dev) = SWAP68_L (info->next_cluster);
				if (LASTALLOC (dev) < FAT32_ROFF || LASTALLOC (dev) >= MAXCL (dev))
					LASTALLOC (dev) = FAT32_ROFF;
			}
			else
				FAT32info (dev) = DISABLE;
		}
		else
			FAT32info (dev) = DISABLE;
	}
}

static void
inv_fat32info (register const ushort dev)
{
	if (FAT32info (dev))
	{
		UNIT *u = bio.read (DI (dev), FAT32ista (dev), DI (dev)->pssize);
		if (u)
		{
			_FAT32_BFSINFO *info = (_FAT32_BFSINFO *) u->data;
			
			if (SWAP68_L (info->signature) == FAT32_FSINFOSIG)
			{
				info->free_clusters = SWAP68_L (-1L);
				info->next_cluster = SWAP68_L (FAT32_ROFF);
				
				bio_MARK_MODIFIED ((&bio), u);
			}
		}
	}
}

static void
upd_fat32info (register const ushort dev)
{
	UNIT *u;
	u = bio.read (DI (dev), FAT32ista (dev), DI (dev)->pssize);
	if (u)
	{
		_FAT32_BFSINFO *info = (_FAT32_BFSINFO *) u->data;
		
		info->free_clusters = SWAP68_L (-1L);
		info->next_cluster = SWAP68_L (LASTALLOC (dev));
		
		bio_MARK_MODIFIED ((&bio), u);
	}
}

/* END special FAT32 extension */
/****************************************************************************/

/****************************************************************************/
/* BEGIN device info */

static long
val_bpb (_x_BPB *xbpb)
{
	static long max = 0;
	
	if (!max)
	{
		max = bio.config (0, BIO_MAX_BLOCK, 0);
	}
	
	FAT_DEBUG (("val_bpb: enter (max = %li)\r\n", max));
	
	if (xbpb->clsizb > max)
	{
		FAT_DEBUG (("val_bpb: leave failure (clsizb = %li)\r\n", xbpb->clsizb));
		return EGENRL;
	}
	
	/*
	 * other checks:
	 */
	
	/* fat type */
	if (xbpb->ftype != FAT_TYPE_12
		&& xbpb->ftype != FAT_TYPE_16
		&& xbpb->ftype != FAT_TYPE_32)
	{
		FAT_DEBUG (("val_bpb: leave failure (ftype = %i)\r\n", xbpb->ftype));
		return EMEDIA;
	}
	
	/* vital values */
	if (xbpb->recsiz & (512 - 1)
		|| xbpb->recsiz == 0
		|| xbpb->numcl  == 0)
	{
		FAT_DEBUG (("val_bpb: leave failure (vital data [recsiz = %li, numcl = %li])\r\n", xbpb->recsiz, xbpb->numcl));
		return EMEDIA;
	}
	
	/* root directory size */
	if (xbpb->ftype != FAT_TYPE_32 && xbpb->rdlen == 0)
	{
		FAT_DEBUG (("val_bpb: leave failure (dir size = %li)\r\n", xbpb->rdlen));
		return EMEDIA;
	}
	
	/* number of clusters */
	if (xbpb->ftype == FAT_TYPE_16 && xbpb->numcl > 65518U)
	{
		FAT_DEBUG (("val_bpb: leave failure (number of clusters = %li)\r\n", xbpb->numcl));
		return EMEDIA;
	}
	
	/* size of FAT */
	{
		ulong minfat = xbpb->numcl + 2;	/* size in cluster */
		
		switch (xbpb->ftype)
		{
			case FAT_TYPE_12:	minfat *= 12;	/* in bits */
						break;
			case FAT_TYPE_16:	minfat <<= 4;	/* in bits */
						break;
			case FAT_TYPE_32:	minfat <<= 5;	/* in bits */
						break;
		}
		minfat >>= 3; /* in bytes */
		
		minfat += xbpb->recsiz - 1L;
		minfat /= xbpb->recsiz;
		
		if (minfat > xbpb->fsiz)
		{
			FAT_DEBUG (("val_bpb: leave failure (FAT size = %li, minfat = %li)\r\n", xbpb->fsiz, minfat));
			return EMEDIA;
		}
	}
	
	FAT_DEBUG (("val_bpb: leave ok\r\n"));
	
	/* device ok */
	return E_OK;
}

static long
get_bpb (DI *di, _x_BPB *xbpb)
{
	_F_BS *fbs;
	_F32_BS *f32bs;
	_F_VI *fvi;
	
	UNIT *u;
	
	FAT_DEBUG (("get_bpb: enter\r\n"));
	
	/* read boot sector */
	u = bio.read (di, 0, di->pssize);
	if (!u)
	{
		FAT_DEBUG (("get_bpb: bio.read fail, leave EDRVNR\r\n"));
		return EDRVNR;
	}
	
	fbs = (void *) u->data;
	f32bs = (void *) u->data;
	
	xbpb->ftype = FAT_INVALID;
	
	/*
	 * step 1: check for GEM/BGM partition
	 */
	
	if ((di->id [0] == 'B' && di->id [1] == 'G' && di->id [2] == 'M')
		|| (di->id [0] == 'G' && di->id [1] == 'E' && di->id [2] == 'M'))
	{
		FAT_DEBUG (("get_bpb: GEM/BGM detected\r\n"));
		
		xbpb->ftype = FAT_TYPE_16;
		fvi = (void *) (u->data + sizeof (*fbs));
	}
	
	/*
	 * step 2: check for F32 partition
	 */
	
	if (di->id [0] == 'F' && di->id [1] == '3' && di->id [2] == '2')
	{
		FAT_DEBUG (("get_bpb: F32 detected\r\n"));
		
		xbpb->ftype = FAT_TYPE_32;
		fvi = (void *) (u->data + sizeof (*f32bs));
	}
	
	/*
	 * step 3: check for dos medium (supported signs: 0x04, 0x06, 0x0b)
	 */
	
	if (di->id [0] == '\0' && di->id [1] == 'D')
	{		
		FAT_DEBUG (("get_bpb: DOS medium detected (%x)\r\n", (int) (di->id [2])));
		
		/* check media descriptor (must be 0xf8 on harddisks) */
		if (fbs->media != 0xf8)
			ALERT ("fatfs.c: get_bpb: unknown media deskriptor (%x) on %c (ID = %x)", (int) fbs->media, 'A'+di->drv, (int) (di->id [2]));
		
		switch (di->id [2])
		{
			case 0x04:
			case 0x06:
			{
				xbpb->ftype = FAT_TYPE_16;
				fvi = (void *) (u->data + sizeof (*fbs));
				break;
			}
			case 0x0b:
			{
				xbpb->ftype = FAT_TYPE_32;
				fvi = (void *) (u->data + sizeof (*f32bs));
				break;
			}
			default:
			{
				FORCE ("fatfs.c: get_bpb: DOS partition type not supported (%x) on %c\r\n", (int) (di->id [2]), 'A'+di->drv);
				FAT_DEBUG (("get_bpb: unknown DOS partition type (%x)\r\n", (int) (di->id [2])));
				return EMEDIA;
			}
		}
	}
	
	/*
	 * step 4: check for NULL partition (A, B or unknown)
	 */
	
	if (di->id [0] == '\0' && di->id [1] == '\0' && di->id [2] == '\0')
	{
		FAT_DEBUG (("get_bpb: \\0\\0\\0 detected\r\n"));
		if (di->drv == 0 || di->drv == 1)
			xbpb->ftype = FAT_TYPE_12;
	}
	
	xbpb->recsiz = WPEEK_INTEL (fbs->sector_size);
	xbpb->clsiz = fbs->cluster_size;
	xbpb->clsizb = xbpb->recsiz * xbpb->clsiz;
	
	xbpb->rdlen = (WPEEK_INTEL (fbs->dir_entries) * 32) / xbpb->recsiz;
	
	xbpb->fatrec = SWAP68_W (fbs->reserved);
	xbpb->fats = fbs->fats;
	
	switch (xbpb->ftype)
	{
		case FAT_TYPE_12:
		case FAT_TYPE_16:
		{
			xbpb->fsiz = SWAP68_W (fbs->fat_length);
			
			xbpb->rdrec = xbpb->fatrec + xbpb->fsiz * xbpb->fats;
			xbpb->datrec = xbpb->rdrec + xbpb->rdlen;
			
			break;
		}
		case FAT_TYPE_32:
		{
			xbpb->fsiz = SWAP68_L (f32bs->fat32_length);
			
			xbpb->rdrec = SWAP68_L (f32bs->root_cluster);
			xbpb->datrec = xbpb->fatrec + xbpb->fsiz * xbpb->fats;
			
			break;
		}
		default:
		{
			FAT_DEBUG (("get_bpb: FAT type not supported (drv = %c)\r\n", 'A'+di->drv));
			return EMEDIA;
		}
	}
	
	if (WPEEK_INTEL (fbs->sectors))
		xbpb->numcl = WPEEK_INTEL (fbs->sectors);
	else
		xbpb->numcl = SWAP68_L (fbs->total_sect);
	
	xbpb->numcl -= xbpb->datrec;
	xbpb->numcl /= xbpb->clsiz;
	
	if (xbpb->ftype == FAT_TYPE_32)
	{
		xbpb->fflag = SWAP68_W (f32bs->flags);
		xbpb->info = SWAP68_W (f32bs->info_sector);
		xbpb->version = ((f32bs->version[1] << 8) | f32bs->version[0]);
		
		if (xbpb->info >= SWAP68_W (fbs->reserved))
		{
			xbpb->info = 0;
		}
	}
	
	xbpb->fats--;
	
	/* update logical sectorsize for HD I/O wrapper */
	bio.set_lshift (di, xbpb->recsiz);
	
	/* validate device informations */
	return val_bpb (xbpb);
}

static DEVINFO *
get_devinfo (const ushort drv)
{
	DI *di;
	
	_x_BPB xbpb;
	_x_BPB *t = &xbpb;
	
	long r;
	
	FAT_DEBUG (("get_devinfo: enter\r\n"));
	
	if (BPBVALID (drv) == VALID)
	{
		FAT_DEBUG (("get_devinfo: leave ok (VALID)\r\n"));
		return BPB (drv);
	}
	
	di = bio.get_di (drv);
	if (!di)
	{
		FAT_DEBUG (("get_devinfo: leave (bio.get_di fail)\r\n"));
		return NULL;
	}
	
	BPB (drv) = kmalloc (sizeof (*BPB (drv)));
	if (!BPB (drv))
	{
		bio.free_di (di);
		
		ALERT ("fatfs.c: kmalloc fail in: get_devinfo for BPB (%i)", drv);
		return NULL;
	}
	zero ((char *) BPB (drv), sizeof (*BPB (drv)));
	
	r = get_bpb (di, t);
	if (r == E_OK)
	{
		DEVINFO *d = BPB (drv);
		
		d->di		= di;
		
		d->recsiz	= t->recsiz;
		d->clsiz	= t->clsiz;
		d->clsizb	= t->clsizb;
		d->fstart	= t->fatrec;
		d->flen		= t->fsiz;
		d->rdstart	= t->rdrec;
		d->rdlen	= t->rdlen;
		d->rdentrys	= t->rdlen * (t->recsiz >> 5); /* (t->recsiz / 32); */
		d->dstart	= t->datrec;
		d->doffset	= t->datrec - 2 * t->clsiz; /* the first cluster has the number 2 */
		d->numcl	= t->numcl;
		d->maxcl	= t->numcl + 2;
		d->entrys	= t->clsizb >> 5; /* t->clsizb / 32 */;
		
		if (!d->root)
		{
			d->root = kmalloc (sizeof (*(d->root)));
			if (!d->root)
			{
				bio.free_di (di);
				BPBVALID (drv) = INVALID;
				kfree (BPB (drv)); BPB (drv) = NULL;
				
				ALERT ("fatfs.c: kmalloc fail in: get_devinfo for root (%i)", drv);
				return NULL;
			}
			*(d->root) = dummy_cookie;
			d->root->dev = drv;
		}
		
		d->freecl	= -1; /* unknown */
		d->fat2on	= t->fats;
		
		switch (t->ftype)
		{
			case FAT_TYPE_12:
			{
				d->ftype	= FAT_TYPE_12;
				
				d->getcl	= getcl12;
				d->fixcl	= fixcl12;
				d->ffree	= ffree12;
				d->lastcl	= 2;
				
				break;
			}
			case FAT_TYPE_16:
			{
				d->ftype	= FAT_TYPE_16;
				
				d->getcl	= getcl16;
				d->fixcl	= fixcl16;
				d->ffree	= ffree16;
				d->lastcl	= 2;
				
				break;
			}
			case FAT_TYPE_32:
			{
				d->ftype	= FAT_TYPE_32;
				
				d->getcl	= getcl32;
				d->fixcl	= fixcl32;
				d->ffree	= ffree32;
				d->lastcl	= FAT32_ROFF;
				
				d->root->stcl	= t->rdrec;
				PUT32_STCL (&(d->root->info), t->rdrec);
				
				d->info		= t->info;
				d->info_active	= t->info ? ENABLE : DISABLE;
				
				val_fat32info (drv);
				
				d->fmirroring	= t->fflag & FAT32_NoFAT_Mirror;
				d->actual_fat	= t->fflag & FAT32_ActiveFAT_Mask;
				
				if (d->fmirroring)	/* disabled, setup up primary FAT */
				{
					if (d->actual_fat <= t->fats)
					{
						d->fmirroring = DISABLE;
						d->actual_fat *= d->flen;
						d->actual_fat += d->fstart;
					}
				}
				else			/* enabled */
				{
					d->fmirroring = ENABLE;
				}
				
				if (d->fmirroring)	/* enabled, primary FAT = first FAT */
				{
					d->actual_fat = d->fstart;
				}
				
				break;
			}
		}
		
		FAT_DEBUG ((
			"%c -> recsize = %li x clsiz = %li -> clsizb = %li\r\n"
			"rdstart = %li, rdlen = %li\r\n"
			"dstart = %li, numcl = %li\r\n",
			'A'+drv, d->recsiz, d->clsiz, d->clsizb,
			d->rdstart, d->rdlen,
			d->dstart, d->numcl
		));
		FAT_DEBUG (("fstart = %li, flen = %li\r\n", d->fstart, d->flen));
		FAT_DEBUG (("fat2on = %d, ftype = %d\r\n", d->fat2on, d->ftype));
		
		BPBVALID (drv) = VALID;
		
		FAT_DEBUG (("get_devinfo: leave ok\r\n"));
		return d;
	}
	
	bio.free_di (di);
	BPBVALID (drv) = INVALID;
	kfree (BPB (drv)); BPB (drv) = NULL;
	
	FAT_DEBUG (("get_devinfo: leave (getbpb fail)\r\n"));
	return NULL;
}

/* END device info part */
/****************************************************************************/

/****************************************************************************/
/* BEGIN filesystem */

static long
fatfs_root (int drv, fcookie *fc)
{
	DEVINFO *a;
	
	FAT_DEBUG (("fatfs_root: enter\r\n"));
	
# ifdef FATFS_TESTING
	if (TEST_PARTITIONS (drv))
# endif
	{
		if (drv == 0 || drv == 1)
		{
			long check = drv ? 0 : 1;
			
			if (BPBVALID (check))
			{
				fatfs_dskchng (check, 1);
			}
		}
		
		a = get_devinfo (drv);
		if (a)
		{
			/* nice, make a fcookie for the kernel */
			
			fc->fs = &fatfs_filesys;
			fc->dev = drv;
			fc->aux = 0;
			fc->index = (long) RCOOKIE (drv); RCOOKIE (drv)->links++;
			
			FAT_DEBUG (("fatfs_root: drive %d active\r\n", drv));
			return E_OK;
		}
		FAT_DEBUG (("fatfs_root: get_devinfo fail\r\n"));
	}
	fc->fs = NULL;
	
	FAT_DEBUG (("fatfs_root: return EDRIVE (drive %c)\r\n", 'A'+drv));
	return EDRIVE;
}

static long
fatfs_lookup (fcookie *dir, const char *name, fcookie *fc)
{
	COOKIE *c = (COOKIE *) dir->index;
	
	FAT_DEBUG (("fatfs_lookup: enter (name = %s, dir = %s, c->dir = %li)\r\n", name, c->name, c->dir));
	
	/* 1 - itself */
	if (!name || (name[0] == '.' && name[1] == '\0'))
	{	
		c->links++;
		*fc = *dir;
		
		FAT_DEBUG (("fatfs_lookup: leave ok, (name = \".\")\r\n"));
		return E_OK;
	}
	
	/* 2 - parent dir */
	if (name[0] == '.' && name[1] == '.' && name[2] == '\0')
	{
		if (c->dir == 0)
		{
			/* no parent, ROOT */
			
			*fc = *dir;
			FAT_DEBUG (("fatfs_lookup: leave ok, EMOUNT, (name = \"..\")\r\n"));
			return EMOUNT;
		}
		if (c->dir == RCOOKIE (c->dev)->stcl)
		{
			/* parent is ROOT DIR */
			
			fc->fs = &fatfs_filesys;
			fc->dev = c->dev;
			fc->aux = 0;
			fc->index = (long) RCOOKIE (c->dev); RCOOKIE (c->dev)->links++;
			
			FAT_DEBUG (("fatfs_lookup: leave ok, ROOT DIR, (name = \"..\")\r\n"));
			return E_OK;
		}
		{	/* parent is a SUB DIR */
			
			COOKIE *search;
			char *temp = fullname (c, "\0");
			register long i;
			
			if (!temp)
			{
				FAT_DEBUG (("fatfs_lookup: leave failure, out of memory\r\n"));
				return ENSMEM;
			}
			
			i = _STRLEN (temp);
			i--;
			while (i--)
			{
				if (temp[i] == '\\')
				{
					temp[i] = '\0';
					break;
				}
			}
			
			search = c_lookup (temp, c->dev); kfree (temp);
			if (search)
			{
				fc->fs = &fatfs_filesys;
				fc->dev = c->dev;
				fc->aux = 0;
				fc->index = (long) search;
				search->links++;
				FAT_DEBUG (("fatfs_lookup: leave ok, found in table\r\n"));
				return E_OK;
			}
			
			ALERT ("fatfs.c: fatfs_lookup: not implemented \"..\" for dir: %s", c->name);
			FAT_DEBUG (("fatfs_lookup: return failure, not implemented (name = \"..\")\r\n"));
			return EFILNF;
			
			/* ??? Pure C: "Unreachable code" */
			FAT_DEBUG (("fatfs_lookup: leave ok, SUB DIR, (name = \"..\")\r\n"));
			return E_OK;
		}
	}
	
	/* 3 - normal name */
	{
		COOKIE *search;
		long r;
		
		r = search_cookie (c, &search, name, 0);
		if (r == E_OK)
		{
			fc->fs = &fatfs_filesys;
			fc->dev = c->dev;
			fc->aux = 0;
			fc->index = (long) search;
		}
		
		FAT_DEBUG (("fatfs_lookup: leave (r = %li)\r\n", r));
		return r;
	}
}

static DEVDRV *
fatfs_getdev (fcookie *fc, long *devsp)
{
	FAT_DEBUG (("fatfs_getdev: ok\r\n"));
	
	if (fc->fs == &fatfs_filesys) return &fatfs_device;
	
	FAT_DEBUG (("fatfs_getdev: leave failure\r\n"));
	*devsp = EINVFN;
	return NULL;
}

static long
fatfs_getxattr (fcookie *fc, XATTR *xattr)
{
# if 0
	long cluster;
# endif
	COOKIE *c = (COOKIE *) fc->index;
	
	FAT_DEBUG (("fatfs_getxattr: enter\r\n"));
	FAT_DEBUG_COOKIE ((c));
	
	xattr->mode	= (c->info.attr & FA_DIR) ?
		(S_IFDIR | DEFAULT_DIRMODE) : (S_IFREG | DEFAULT_MODE);
	
	xattr->mode	&= ~(S_IROTH | S_IWOTH | S_IXOTH);
	
	if (c->info.attr & FA_RDONLY)
		xattr->mode &= ~(S_IWUSR | S_IWGRP | S_IWOTH);
	
	if (c->info.attr == FA_SYMLINK)
		xattr->mode = S_IFLNK;
	
	xattr->index	= INDEX (c);
	xattr->dev	= c->dev;
	xattr->rdev	= c->rdev;
	xattr->nlink	= 1;
	xattr->uid	= 0;
	xattr->gid	= 0;
# if 0
	/* ???: Das sollte die Unterverzeichnisgre werden. Mir ist nicht klar, wie
	 * ich auf das Wurzelverzeichnis abfrage. Das braucht ja fr FAT12 und FAT16
	 * eine Sonderbehandlung. */
	xattr->blksize	= CLUSTSIZE (fc->dev);
	if (c->info.attr & FA_DIR)
	{
		xattr->nlink = 2;  /* minimum for subdirectories */
		cluster = c->stcl;
		xattr->nblocks = 0;
		do
			xattr->nblocks++;
		while ((cluster = GETCL (cluster, fc->dev, 1)) > 0);
		xattr->size = xattr->nblocks * xattr->blksize;
	}
	else
	{
		xattr->size	= SWAP68_L (c->info.flen);
		xattr->nblocks	= (xattr->size + xattr->blksize - 1) / xattr->blksize;
	}
# else
	xattr->size	= SWAP68_L (c->info.flen);
	xattr->blksize	= CLUSTSIZE (fc->dev);
	xattr->nblocks	= (xattr->size + xattr->blksize - 1) / xattr->blksize;
	if (!xattr->nblocks && (c->info.attr & FA_DIR))
		xattr->nblocks = 1;	/* no dir takes 0 blocks... */
# endif
	xattr->mtime	= SWAP68_W (c->info.time);
	xattr->mdate	= SWAP68_W (c->info.date);
	xattr->atime	= xattr->mtime; 
	xattr->adate	= c->info.adate ? SWAP68_W (c->info.adate) : xattr->mdate;
	xattr->ctime	= c->info.ctime ? SWAP68_W (c->info.ctime) : xattr->mtime;
	xattr->cdate	= c->info.cdate ? SWAP68_W (c->info.cdate) : xattr->mdate;
	xattr->attr	= c->info.attr & FA_TOSVALID;
	
	FAT_DEBUG (("fatfs_getxattr: return ok\r\n"));
	return E_OK;
}

static long
fatfs_chattr (fcookie *fc, int attrib)
{
	COOKIE *c = (COOKIE *) fc->index;
	long r;
	
	FAT_DEBUG (("fatfs_chattr: enter\r\n"));
	FAT_DEBUG_COOKIE ((c));
	
	if (c->dir)
	{
		/* only the lowest 8 bits ar used */
		c->info.attr = attrib & FA_TOSVALID;
		r = write_cookie (c);
		if (r == E_OK)
		{
			bio_SYNC_DRV ((&bio), DI (c->dev));
			FAT_DEBUG (("fatfs_chattr: leave ok\r\n"));
			return E_OK;
		}
		FAT_DEBUG (("fatfs_chattr: leave failure (write_cookie = %li)\r\n", r));
		return r;
	}
	return EACCDN;
}

static long
fatfs_chown (fcookie *fc, int uid, int gid)
{
	UNUSED (fc); UNUSED (uid); UNUSED (gid);
	FAT_DEBUG (("fatfs_chown: not supported\r\n"));
	
	/* not supported by fat & vfat */
	return EINVFN;
}

static long
fatfs_chmode (fcookie *fc, unsigned mode)
{
	COOKIE *c = (COOKIE *) fc->index;
	char new;
	
	FAT_DEBUG (("fatfs_chmode: enter\r\n"));
	FAT_DEBUG_COOKIE ((c));
	
	if (c->dir)
	{
		if (!(mode & S_IWUSR))
			new = c->info.attr | FA_RDONLY;
		else
			new = c->info.attr & ~FA_RDONLY;
		
		if (c->info.attr & FA_DIR)
		{
			if (mode & S_ISUID) /* hidden directory */
				new |= FA_HIDDEN;
			else
				new &= ~FA_HIDDEN;
		}
		
		if (new != c->info.attr)
		{
			long r;
			
			c->info.attr = new;
			r = write_cookie (c);
			if (r)
			{
				FAT_DEBUG (("fatfs_chmode: leave failure (write_cookie = %li)\r\n", r));
				return r;
			}
			
			bio_SYNC_DRV ((&bio), DI (c->dev));
		}
		
		FAT_DEBUG (("fatfs_chmode: leave ok\r\n"));
		return E_OK;
	}
	return EACCDN;
}

static long
fatfs_mkdir (fcookie *dir, const char *name, unsigned mode)
{
	COOKIE *c = (COOKIE *) dir->index;
	COOKIE *new = NULL;
	long r;
	
	UNUSED (mode);
	
	FAT_DEBUG (("fatfs_mkdir: enter\r\n"));
	
	/* check if dir exist */
	r = search_cookie (c, NULL, name, 0);
	if (r == E_OK)
	{
		FAT_DEBUG (("fatfs_mkdir: leave failure (dir exist)\r\n"));
		return EACCDN;
	}
	
	r = make_cookie (c, &new, name, FA_DIR);
	if (r == E_OK)
	{
		long stcl = r = nextcl (0, c->dev);
		if (r > 0)
		{
			PUT_STCL (&(new->info), c->dev, stcl);
			new->stcl = stcl;
			r = write_cookie (new); new->links--;
			if (r == E_OK)
			{
				UNIT *u = bio.getunit (DI (c->dev), C2S (stcl, c->dev), CLUSTSIZE (c->dev));
				if (u)
				{
					register const ushort date = datestamp;
					register const ushort time = timestamp;
					register _DIR *info = (_DIR *) u->data;
					register long j;
					
					quickzero (u->data, CLUSTSIZE (c->dev) >> 8);
					
					info->name[0] = '.';
					for (j = 1; j < 11; j++) info->name[j] = ' ';
					info->attr = FA_DIR;
					info->date = SWAP68_W (date);
					info->time = SWAP68_W (time);
					PUT_STCL (info, c->dev, stcl);
					
					info++;
					info->name[0] = info->name[1] = '.';
					for (j = 2; j < 11; j++) info->name[j] = ' ';
					info->attr = FA_DIR;
					info->date = SWAP68_W (date);
					info->time = SWAP68_W (time);
					if (new->dir > 1)
						PUT_STCL (info, c->dev, new->dir);
					
					bio_MARK_MODIFIED ((&bio), u);
					bio_SYNC_DRV ((&bio), DI (c->dev));
					
					FAT_DEBUG (("fatfs_mkdir: leave ok (return bio_MARK_MODIFIED)\r\n"));
					return E_OK;
					
				}
				else /* if (ptr) */
				{
					FAT_DEBUG (("fatfs_mkdir: leave failure (read)\r\n"));
					r = EREADF;
				}
				
			}
			else /* if (r == E_OK) */
			{
				FAT_DEBUG (("fatfs_mkdir: leave failure (write_cookie = %li)\r\n", r));
			}
			
		}
		else /* if (r > 0) */
		{
			FAT_DEBUG (("fatfs_mkdir: leave failure (nextcl = %li)\r\n", r));
		}
		
		(void) delete_cookie (new, 1);
		
	}
	else /* if (r == E_OK) */
	{
		FAT_DEBUG (("fatfs_mkdir: leave failure (make_cookie = %li)\r\n", r));
	}
	
	return r;
}

static long
fatfs_rmdir (fcookie *dir, const char *name)
{
	DIR dirh;
	long r;
	
	FAT_DEBUG (("fatfs_rmdir: enter\r\n"));
	
	r = fatfs_lookup (dir, name, &(dirh.fc));
	if (r == E_OK)
	{
		FAT_DEBUG (("fatfs_rmdir: found: dev = %i\r\n", dirh.fc.dev));
		/* Ordner muss leer sein, ansonsten EACCDN */
		{	dirh.flags = 0;
			r = fatfs_opendir (&dirh, dirh.flags);
			if (r == E_OK)
			{
				char buf [VFAT_NAMEMAX];
				fcookie file;
				long count = 0;
				while (fatfs_readdir (&dirh, buf, VFAT_NAMEMAX, &file) == 0)
				{
					fatfs_release (&file);
					c_remove ((COOKIE *) file.index);
					count++;
					if (count > 2)
					{
						(void) fatfs_closedir (&dirh);
						FAT_DEBUG (("fatfs_rmdir: leave failure (not free)\r\n"));
						return EACCDN;
					}
				}
				(void) fatfs_closedir (&dirh);
			}
			else
			{
				/* failure while opening */
				FAT_DEBUG (("fatfs_rmdir: leave failure (fatfs_opendir = %li)\r\n", r));
				return r;
			}
		}
		r = delete_cookie ((COOKIE *) dirh.fc.index, 1);
		if (r == 0)
		{
			bio_SYNC_DRV ((&bio), DI (dir->dev));
			FAT_DEBUG (("fatfs_rmdir: leave ok\r\n"));
			return E_OK;
		}
		else
		{
			FAT_DEBUG (("fatfs_rmdir: leave failure (delete_cookie = %li)\r\n", r));
		}
	}
	else
	{
		FAT_DEBUG (("fatfs_rmdir: leave failure (fatfs_lookup = %li)\r\n", r));
	}
	
	return r;
}

static long
fatfs_creat (fcookie *dir, const char *name, unsigned mode, int attrib, fcookie *fc)
{
	COOKIE *c = (COOKIE *) dir->index;
	COOKIE *new = NULL;
	long r;
	
	UNUSED (mode);
	
	FAT_DEBUG (("fatfs_creat: enter\r\n"));
	
/*-- kernel do this
	r = search_cookie (c, NULL, name, 0);
	if (r == E_OK)
	{
		FAT_DEBUG (("fatfs_creat: leave failure (file exist)\r\n"));
		return EACCDN;
	}
*/
	
	r = make_cookie (c, &new, name, attrib);
	fc->fs = &fatfs_filesys;
	fc->dev = c->dev;
	fc->aux = 0;
	fc->index = (long) new;
	
	bio_SYNC_DRV ((&bio), DI (c->dev));
	
	FAT_DEBUG (("fatfs_creat: leave ok (return make_cookie = %li)\r\n", r));
	return r;
}

static long
fatfs_remove (fcookie *dir, const char *name)
{
	COOKIE *file;
	long r;
	
	FAT_DEBUG (("fatfs_remove: enter\r\n"));
	
	r = search_cookie ((COOKIE *) dir->index, &file, name, 0);
	if (r == E_OK)
	{
		r = delete_cookie (file, 1);
		if (r == E_OK)
		{
			bio_SYNC_DRV ((&bio), DI (dir->dev));
			FAT_DEBUG (("fatfs_remove: leave ok\r\n"));
			return E_OK;
		}
		else
		{
			FAT_DEBUG (("fatfs_remove: leave failure (delete_cookie)\r\n"));
		}
	}
	else
	{
		FAT_DEBUG (("fatfs_remove: leave failure (search_cookie)\r\n"));
	}
	return r;
}

static long
fatfs_getname (fcookie *root, fcookie *dir, char *pathname, int size)
{
	char *r = ((COOKIE *) root->index)->name;
	char *d = ((COOKIE *) dir->index)->name;
	long i, j;
	
	FAT_DEBUG (("fatfs_getname: enter\r\n"));
	FAT_DEBUG (("fatfs_getname: root = %s, dir = %s\r\n", r, d));
	
	if (size <= 0)
	{
		FAT_DEBUG (("fatfs_getname: leave failure (ENAMETOOLONG)\r\n"));
		return ENAMETOOLONG;
	}
	
	*pathname = '\0';
	
	if (COOKIE_EQUAL (root, dir))
	{
		FAT_DEBUG (("fatfs_getname: leave ok, equal\r\n"));
		return E_OK;
	}
	
	i = strlen (r);
	j = strlen (d);
	if (j > i)
	{
		if ((j - i) < size)
		{
			strcpy (pathname, d + i);
			
			FAT_DEBUG (("fatfs_getname: leave pathname = %s\r\n", pathname));
			return E_OK;
		}
		
		FAT_DEBUG (("fatfs_getname: leave failure (ENAMETOOLONG)\r\n"));
		return ENAMETOOLONG;
	}
	
	FAT_DEBUG (("fatfs_getname: leave failure (EINTRN)\r\n"));
	return EINTRN;
}

static long
fatfs_rename (fcookie *olddir, char *oldname, fcookie *newdir, const char *newname)
{
	COOKIE *oldd = (COOKIE *) olddir->index;
	COOKIE *newd = (COOKIE *) newdir->index;
	COOKIE *old;
	COOKIE *new;
	long r;
	
	FAT_DEBUG (("fatfs_rename: enter (old = %s, new = %s)\r\n", oldname, newname));
	
	/* on same device? */
	if (oldd->dev != newd->dev)
	{
		FAT_DEBUG (("fatfs_rename: leave failure (cross device rename)\r\n"));
		return EACCDN;
	}
	
	/* check if file exist */
	r = search_cookie (newd, NULL, newname, 0);
	if (r == E_OK)
	{
		/* check for rename same file (casepreserving) */
		if (!(COOKIE_EQUAL (olddir, newdir) && (_STRICMP (oldname, newname) == 0)))
		{
			FAT_DEBUG (("fatfs_rename: leave failure (file exist)\r\n"));
			return EACCDN;
		}
	}
	
	r = search_cookie (oldd, &old, oldname, 0);
	if (r) return r;
	
	if (old->links > 1)
	{
		FAT_DEBUG (("fatfs_rename: leave failure (cookie in use)\r\n"));
		old->links--;
		return EACCDN;
	}
	
	r = make_cookie (newd, &new, newname, old->info.attr);
	if (r) return r;
	
	new->info.lcase		= old->info.lcase;
	new->info.ctime_ms	= old->info.ctime_ms;
	new->info.ctime		= old->info.ctime;
	new->info.cdate		= old->info.cdate;
	new->info.adate		= old->info.adate;
	new->info.stcl_fat32	= old->info.stcl_fat32;
	new->info.time		= old->info.time;
	new->info.date		= old->info.date;
	new->info.stcl		= old->info.stcl;
	new->info.flen		= old->info.flen;
	
	new->stcl = old->stcl;
	
	r = write_cookie (new);
	if (r) return r;
	
	r = delete_cookie (old, 0);
	if (r) return r;
	
	new->links--;
	bio_SYNC_DRV ((&bio), DI (new->dev));
	
	FAT_DEBUG (("fatfs_rename: leave ok\r\n"));
	return E_OK;
}

static long
fatfs_opendir (DIR *dirh, int flags)
{
	COOKIE *c = (COOKIE *) dirh->fc.index;
	oDIR *dir = (oDIR *) dirh->fsstuff;
	
	UNUSED (flags);
	
	FAT_DEBUG (("fatfs_opendir: enter (c->dir = %li, c->offset = %li)\r\n", c->dir, c->offset));
	FAT_DEBUG_COOKIE ((c));
	
	dirh->index = 0;
	
	FAT_DEBUG (("fatfs_opendir: leave return (__opendir) (stcl = %li)\r\n", c->stcl));
	return __opendir (dir, c->stcl, c->dev);
}

static long
fatfs_readdir (DIR *dirh, char *nm, int nmlen, fcookie *fc)
{
	oDIR *dir = (oDIR *) dirh->fsstuff;
	COOKIE *c = (COOKIE *) dirh->fc.index;
	COOKIE *new;
	char *name;
	register long r;
	
	FAT_DEBUG (("fatfs_readdir: enter\r\n"));
	
	if (dirh->flags & TOS_SEARCH)
	{
		FAT_DEBUG (("fatfs_readdir: TOS_SEARCH\r\n"));
	}
	else
	{
		nm += 4;
		nmlen -= 4;
	}
	
	r = __nextdir (dir, nm, nmlen);
	if (r < 0) return r;
	
	name = fullname (c, nm);
	if (name)
	{
		new = c_lookup (name, dir->dev);
		if (!new)
		{
			new = get_cookie (name);
			if (new)
			{
				new->dev = dir->dev;
				new->rdev = dir->dev;
				new->dir = c->stcl;
				new->offset = dir->index;
				new->stcl = GET_STCL (dir->info, dir->dev);
				new->info = *(dir->info);
				new->slots = r;
			}
		}
		else
		{
			kfree (name);
			new->links++;
		}
		if (new)
		{
			fc->fs = &fatfs_filesys;
			fc->dev = dir->dev;
			fc->aux = 0;
			fc->index = (long) new;
			
			dirh->index = dir->index;
			
			if (r && ((dirh->flags & TOS_SEARCH) || !VFAT (dir->dev)))
			{
				/* return TOS name */
				FAT_DEBUG (("fatfs_readdir: TOS_SEARCH, make TOS_NAME\r\n"));
				dir2str (dir->info->name, nm);
			}
# if 0
			else if (!r && LCASE (dir->dev))
				_STRLWR (nm);
# else
			if (VFAT (dir->dev))
			{
				if (!r && LCASE (dir->dev))
					_STRLWR (nm);
			}
			else
			{
				if (curproc->domain != DOM_TOS)
					_STRLWR (nm);
			}
# endif
			
			if ((dirh->flags & TOS_SEARCH) == 0)
			{
				*(long *) (nm - 4) = INDEX (new);
			}
			
			FAT_DEBUG_COOKIE ((new));
			FAT_DEBUG (("fatfs_readdir: leave ok (nm = %s)\r\n", nm));
			return E_OK;
		}
		
		FAT_DEBUG (("fatfs_readdir: leave failure (get_cookie)\r\n"));
	}
	
	FAT_DEBUG (("fatfs_readdir: leave failure (out of memory)\r\n"));
	return ENSMEM;
}

static long
fatfs_rewinddir (DIR *dirh)
{
	register long r;

	FAT_DEBUG (("fatfs_rewinddir: enter\r\n"));
	
	(void) fatfs_closedir (dirh);
	r = fatfs_opendir (dirh, dirh->flags);
	
	FAT_DEBUG (("fatfs_rewinddir: leave ok\r\n"));
	return r;
}

static long
fatfs_closedir (DIR *dirh)
{
	register oDIR *dir = (oDIR *) dirh->fsstuff;
	
	FAT_DEBUG (("fatfs_closedir: enter\r\n"));
	
	dirh->index = 0;
	__closedir (dir);
	
	FAT_DEBUG (("fatfs_closedir: leave ok\r\n"));
	return E_OK;
}

static long
fatfs_pathconf (fcookie *dir, int which)
{
	FAT_DEBUG (("fatfs_pathconf: enter\r\n"));
	
	switch (which)
	{
		case DP_INQUIRE:	return DP_XATTRFIELDS;
		case DP_IOPEN:		return UNLIMITED;
		case DP_MAXLINKS:	return 1;
		case DP_PATHMAX:	return VFAT (dir->dev) ? VFAT_PATHMAX - 1 : FAT_PATHMAX - 1;
		case DP_NAMEMAX:	return VFAT (dir->dev) ? VFAT_NAMEMAX - 1 : FAT_NAMEMAX - 1;
		case DP_ATOMIC:		return CLUSTSIZE (dir->dev);
		case DP_TRUNC:		return VFAT (dir->dev) ? DP_NOTRUNC : DP_DOSTRUNC;
		case DP_CASE:		return VFAT (dir->dev) ? DP_CASEINSENS : DP_CASECONV;
		case DP_MODEATTR:	return (FA_TOSVALID | DP_FT_DIR | DP_FT_REG | (SLNK (dir->dev) ? DP_FT_LNK : 0));
		case DP_XATTRFIELDS:	return (DP_INDEX | DP_DEV | DP_BLKSIZE | DP_SIZE | DP_NBLOCKS | DP_MTIME |
						(VFAT (dir->dev) ? (DP_ATIME | DP_CTIME) : 0));
	}
	
	FAT_DEBUG (("fatfs_pathconf: leave failure\r\n"));	
	return EINVFN;
}

static long
fatfs_dfree (fcookie *dir, long *buf)
{
	return DFREE (dir, buf);
}

static long
fatfs_writelabel (fcookie *dir, const char *name)
{
	oDIR odir;
	long index = 0;
	long i, r;
	
	FAT_DEBUG (("fatfs_writelabel: enter (name = %s)\r\n", name));
	
	r = __opendir (&odir, RCOOKIE (dir->dev)->stcl, dir->dev);
	r = __seekdir (&odir, index++, READ);
	while ((r == E_OK)
		&& (((odir.info->attr & (FA_DIR | FA_LABEL)) != FA_LABEL)
			|| (odir.info->attr == FA_VFAT)))
	{
		r = __SEEKDIR (&odir, index++, READ);
	}
	
	if (r)
	{
		/* no label found, create one */
		COOKIE *new;
		char label[13];
		
		register const char *src = name;
		register char *dst = label;
		
		/* close open directory */
		__closedir (&odir);
		
		/* create base name */
		i = 8;
		while (i-- && *src && *src != '.')
		{
			if (DEFAULT_TABLE (dir->dev) [(long) *src])
				*dst++ = _TOUPPER (*src);
			else
				*dst = '_';
			src++;
		}
		
		/* create extension */
		if (*src)
		{
			*dst++ = '.';
			if (*src == '.') src++;
			i = 3;
			while (i-- && *src && *src != '.')
			{
				if (DEFAULT_TABLE (dir->dev) [(long) *src])
					*dst++ = _TOUPPER (*src);
				else
					*dst = '_';
				src++;
			}
		}
		
		/* terminate new label */
		*dst = '\0';
		
		FAT_DEBUG (("fatfs_writelabel: create (label = %s)\r\n", label));
		
		r = make_cookie (RCOOKIE (dir->dev), &new, label, FA_LABEL);
		if (r)
		{
			FAT_DEBUG (("fatfs_writelabel: leave failure make_cookie (%li)\r\n", r));
			return EACCDN;
		}
		new->links--;
		
		bio_SYNC_DRV ((&bio), DI (dir->dev));
		
		FAT_DEBUG (("fatfs_writelabel: leave ok (E_OK)\r\n"));
		return E_OK;
	}
	
	/* copy name (no character check?) */
	for (i = 0; i < 11 && name[i] != '\0'; i++)
		odir.info->name[i] = name[i];
	
	/* fill out with spaces */
	for (; i < 11; i++)
		odir.info->name[i] = ' ';
	
	__updatedir (&odir);
	__closedir (&odir);
	
	bio_SYNC_DRV ((&bio), DI (dir->dev));
	
	FAT_DEBUG (("fatfs_writelabel: leave ok (E_OK) (label = %s, index = %li)\r\n", name, odir.index));
	return E_OK;
}

static long
fatfs_readlabel (fcookie *dir, char *name, int namelen)
{
	oDIR odir;
	long index = 0;
	long i, r;
	
	FAT_DEBUG (("fatfs_readlabel: enter (namelen = %i)\r\n", namelen));
	
	r = __opendir (&odir, RCOOKIE (dir->dev)->stcl, dir->dev);
	r = __seekdir (&odir, index++, READ);
	while ((r == E_OK)
		&& (((odir.info->attr & (FA_DIR | FA_LABEL)) != FA_LABEL)
			|| (odir.info->attr == FA_VFAT)))
	{
		r = __SEEKDIR (&odir, index++, READ);
	}
	
	if (r)
	{
		__closedir (&odir);
		
		FAT_DEBUG (("fatfs_readlabel: leave failure\r\n"));
		return EFILNF;
	}
	
	FAT_DEBUG (("fatfs_readlabel: label found (index = %li)\r\n", odir.index));
	
	for (i = 0; odir.info->name[i] != ' ' && namelen && i < 11; namelen--, i++)
		name[i] = odir.info->name[i];
	
	if (namelen == 0)
	{
		FAT_DEBUG (("fatfs_readlabel: leave failure (namelen == 0)\r\n"));
		return ENAMETOOLONG;
	}
	
	name[i] = '\0';
	
	__closedir (&odir);
	
	FAT_DEBUG (("fatfs_readlabel: leave ok (label = %s, namelen = %i, index = %li)\r\n", name, namelen, odir.index));
	return E_OK;
}

static long
fatfs_symlink (fcookie *dir, const char *name, const char *to)
{
	FILEPTR f;
	short linklen;
	long r;
	
	if (!SLNK (dir->dev)) return EINVFN;
	
	FAT_DEBUG (("fatfs_symlink: enter (dir = %s)\r\n", ((COOKIE *) dir->index)->name));
	FAT_DEBUG (("fatfs_symlink: (name = %s, to = %s)\r\n", name, to));
	
/*--???
	r = search_cookie ((COOKIE *) dir->index, NULL, name, 0);
	if (r == E_OK)
	{
		FAT_DEBUG (("fatfs_symlink: leave failure (file exist)\r\n"));
		return EACCDN;
	}
*/
	
	r = fatfs_creat (dir, name, 0, FA_SYMLINK, &(f.fc));
	if (r)
	{
		FAT_DEBUG (("fatfs_symlink: leave failure (creat = %li)\r\n", r));
		return r;
	}
	
	f.flags = O_TRUNC | O_WRONLY;
	f.links = 0;
	r = fatfs_open (&f);
	if (r)
	{
		FAT_DEBUG (("fatfs_symlink: leave failure (open = %li)\r\n", r));
		delete_cookie ((COOKIE *) f.fc.index, 1);
		return r;
	}
	
	linklen = strlen (to) + 1;
	if (linklen & 1) linklen += 1;
	
	(void) fatfs_write (&f, (char *) &linklen, 2);
	(void) fatfs_write (&f, to, linklen - 1);
	(void) fatfs_write (&f, "\0", 1);
	(void) fatfs_close (&f, 0);
	
	(void) fatfs_release (&(f.fc));
	
	FAT_DEBUG (("fatfs_symlink: leave ok\r\n"));
	return E_OK;
}

static long
fatfs_readlink (fcookie *file, char *buf, int len)
{
	COOKIE *c = (COOKIE *) file->index;
	FILEPTR f;
	short linklen;
	long r;
	
	FAT_DEBUG (("fatfs_readlink: enter (dir = %s)\r\n", c->name));
	
	if (c->info.attr != FA_SYMLINK)
	{
		FAT_DEBUG (("fatfs_readlink: leave failure (not a symlink)\r\n"));
		return EACCDN;  /* EINVAL on HP-UX */
	}
	
	(void) fatfs_dupcookie (&(f.fc), file);
	
	f.flags = O_RDONLY;
	f.links = 0;
	r = fatfs_open (&f);
	if (r == E_OK)
	{
		r = fatfs_read (&f, (char *) &linklen, 2);
		if (r != 2)
			r = (r < 0) ? r : ERROR;  /* read error */
		else if (linklen >= len)
			r = ENAMETOOLONG;
		else
		{
			r = fatfs_read (&f, buf, MIN (len, linklen));
			if (r == linklen)
			{
				buf[--len] = '\0';
				r = E_OK;
			}
			else
				r = (r < 0) ? r : ERROR;  /* read error */
		}
		
		(void) fatfs_close (&f, 0);
	}
	else
	{
		FAT_DEBUG (("fatfs_readlink: failure (open = %li)\r\n", r));
	}
	
	(void) fatfs_release (&(f.fc));
	
	FAT_DEBUG (("fatfs_readlink: leave return = %li\r\n", r));
	return r;
}

static long
fatfs_hardlink (fcookie *fromdir, const char *fromname, fcookie *todir, const char *toname)
{
	FAT_DEBUG (("fatfs_hardlink: invalid function\r\n"));
	
	return EINVFN;
}

static long
fatfs_fscntl (fcookie *dir, const char *name, int cmd, long arg)
{
	FAT_DEBUG (("fatfs_fscntl: enter (name = %s, cmd = %i, arg = %li)\r\n", name, cmd, arg));
	
	switch (cmd)
	{
		case MX_KER_XFSNAME:
		{
			strcpy ((char *) arg, "vfat-xfs");
			return E_OK;
		}
		case FS_INFO:
		{
			struct fs_info *info = (struct fs_info *) arg;
			if (info)
			{
				char *dst = info->type_asc;
				
				strcpy (info->name, "vfat-xfs");
				info->version = ((long) FATFS_MAJOR << 16) | FATFS_MINOR;
				info->type = 0;
				
				if (VFAT (dir->dev))
				{
					info->type |= (1L << 16);
					*dst++ = 'v';
				}
				
				switch (FAT_TYPE (dir->dev))
				{
					case FAT_TYPE_12:
					{
						info->type |= 0;
						strcpy (dst, "fat12");
						
						break;
					}
					case FAT_TYPE_16:
					{
						info->type |= 1;
						strcpy (dst, "fat16");
						
						break;
					}
					case FAT_TYPE_32:
					{
						info->type |= 2;
						strcpy (dst, "fat32");
						
						break;
					}
				}
			}
			return E_OK;
		}
		case FS_USAGE:
		{
			break;
		}
		case VFAT_CNFDFLN:
		{
			long drv = 1;
			long i;
			for (i = 0; i < NUM_DRIVES; i++)
			{
				if (arg & drv)
					VFAT (i) = ENABLE;
				else
					VFAT (i) = DISABLE;
				drv <<= 1;
			}
			return E_OK;
		}
		case VFAT_CNFLN:
		{
			return fatfs_config (dir->dev, FATFS_VFAT, arg);
		}
		case V_CNTR_SLNK:
		{
			if (arg == ASK) return SLNK (dir->dev);
			SLNK (dir->dev) = arg ? ENABLE : DISABLE;
			break;
		}
		case DL_SETCHAR:
		case V_CNTR_MODE:
		{
			if (VFAT (dir->dev)) return EINVFN;
			switch (arg)
			{
				case ASK:
				{
					return NAME_MODE (dir->dev);
				}
				case GEMDOS:
				{
					NAME_MODE (dir->dev) = GEMDOS;
					DEFAULT_TABLE (dir->dev) = GEMDOS_TABLE;
					break;
				}
				case ISO:
				{
					NAME_MODE (dir->dev) = ISO;
					DEFAULT_TABLE (dir->dev) = ISO_TABLE;
					break;
				}
				case MSDOS:
				{
					NAME_MODE (dir->dev) = MSDOS;
					DEFAULT_TABLE (dir->dev) = MSDOS_TABLE;
					break;
				}
				default:
				{
					return EINVFN;
				}
			}
			break;
		}
		case V_CNTR_FAT32:
		{
			struct control_FAT32 *f32 = (struct control_FAT32 *) arg;
			
			if (!FAT32 (dir->dev)) return EINVFN;
			
			if (f32->mode == 0)
			{
				if (FAT32mirr (dir->dev))
					f32->mirr = 0;
				else
				{
					f32->mirr = FAT32prim (dir->dev);
					f32->mirr -= FAT1 (dir->dev);
					f32->mirr /= FATSIZE (dir->dev);
					f32->mirr++;
				}
				f32->fats = FAT2ON (dir->dev) + 1;
				f32->info = 0;
				f32->info |= FAT32info (dir->dev) ? FAT32_info_active : 0;
				f32->info |= FAT32ista (dir->dev) ? FAT32_info_exist : 0;
			}
			else
			{
				/*
				 * setup FAT stuff
				 */
				
				long old_mirr = FAT32mirr (dir->dev);
				long old_fat = FAT32prim (dir->dev);
				
				if (f32->mirr)
				{
					f32->mirr--;
					if (f32->mirr >= 0 && f32->mirr <= FAT2ON (dir->dev))
					{
						f32->mirr *= FATSIZE (dir->dev);
						f32->mirr += FAT1 (dir->dev);
						
						FAT32prim (dir->dev) = f32->mirr;
						FAT32mirr (dir->dev) = DISABLE;
					}
					else
						FAT32mirr (dir->dev) = ENABLE;
				}
				else
					FAT32mirr (dir->dev) = ENABLE;
				
				if (FAT32mirr (dir->dev))
					FAT32prim (dir->dev) = FAT1 (dir->dev);
				
				/* anything changed? */
				if (old_mirr != FAT32mirr (dir->dev) || old_fat != FAT32prim (dir->dev))
				{
					/* update FAT's if neccessary */
					if (old_mirr == DISABLE)
						upd_fat32fats (dir->dev, old_fat);
					
					/* set up new boot sector */
					upd_fat32boot (dir->dev);
				}
				
				/*
				 * setup INFO sector
				 */
				
				if (FAT32ista (dir->dev))
				{
					if (f32->info & FAT32_info_active)
						FAT32info (dir->dev) = ENABLE;
					else
					{
						FAT32info (dir->dev) = DISABLE;
						inv_fat32info (dir->dev);
					}
				}
			}
			return E_OK;
		}
		case V_CNTR_WP:
		{
			return EINVFN;
		}
		case V_CNTR_WB:
		{
			return bio.config (dir->dev, BIO_WB, arg);
		}
		case FUTIME:
		{
			COOKIE *c;
			long r = search_cookie ((COOKIE *) dir->index, &c, name, 0);
			if (r == E_OK)
			{
				r = __FUTIME (c, (ushort *) arg);
				c->links--;
			}
			FAT_DEBUG (("fatfs_fscntl: leave (FUTIME) (r = %li)\r\n", r));
			return r;
		}
		case FTRUNCATE:
		{
			COOKIE *c;
			long r = search_cookie ((COOKIE *) dir->index, &c, name, 0);
			if (r == E_OK)
			{
				r = __FTRUNCATE (c, arg);
				c->links--;
			}
			FAT_DEBUG (("fatfs_fscntl: leave (FTRUNCATE) (r = %li)\r\n", r));
			return r;
		}
	}
	
	FAT_DEBUG (("fatfs_fscntl: invalid function\r\n"));
	return EINVFN;
}

static long
fatfs_dskchng (int drv, int mode)
{
	long change = 1;
	
	FAT_DEBUG (("fatfs_dskchng: enter (drv = %c, mode = %i)\r\n", 'A'+drv, mode));
	
	if (mode == 0)
		change = BIO_DSKCHNG (DI (drv));
	
	if (change == 0)
	{
		/* no change */
		FAT_DEBUG (("fatfs_dskchng: leave no change\r\n"));
		return change;
	}
	
	FAT_DEBUG (("fatfs_dskchng: invalidate drv (change = %li)\r\n", change));
	
	/* I hope this isn't a failure */
	bio.sync_drv (DI (drv));
	
	/* invalid all cookies */
	{
		register long i;
		for (i = 0; i < COOKIE_CACHE; i++)
		{
			register COOKIE *c = &(cookies[i]);
			if (c->dev == drv)
			{
				if (c->name)
					c_remove (c);
			}
		}
	}
	
	/* free the DI (also invalidate cache units) */
	bio.free_di (DI (drv));
	
	/* invalidate the BPB */
	BPBVALID (drv) = INVALID;
	
	/* free the dynamically allocated memory */
	kfree (RCOOKIE (drv)); RCOOKIE (drv) = NULL;
	kfree (BPB (drv)); BPB (drv) = NULL;
	
	FAT_DEBUG (("fatfs_dskchng: leave (change = %li)\r\n", change));
	return change;
}

static long
fatfs_release (fcookie *fc)
{
	register COOKIE *c = (COOKIE *) fc->index;
	
	FAT_DEBUG (("fatfs_release: enter (c->name = %s, c->links = %li, c->dev = %i)\r\n", c->name, c->links, c->dev));
	
	if (c->links)
		c->links--;
	
	FAT_DEBUG (("fatfs_release: leave ok (c->links = %li)\r\n", c->links));
	return E_OK;
}

static long
fatfs_dupcookie (fcookie *dst, fcookie *src)
{
	register COOKIE *c = (COOKIE *) src->index;
	
	FAT_DEBUG (("fatfs_dupcookie: ok (c->name = %s, c->links = %li, c->dev = %i)\r\n", c->name, c->links, c->dev));
	
	c->links++;
	*dst = *src;
	return E_OK;
}

static long
fatfs_sync (void)
{
	FAT_DEBUG (("fatfs_sync: ok\r\n"));
	
	/* update FAT32 info sectors */
	/* if (FAT32info (dev)) upd_fat32info (dev); */
	
	/* buffer cache automatically synced */
	return E_OK;
}

/* END filesystem */
/****************************************************************************/

/****************************************************************************/
/* BEGIN device driver */

/*
 * internal
 */

static long
__FUTIME (COOKIE *c, ushort *ptr)
{
	long r;
	
	FAT_DEBUG (("__FUTIME [%s]: enter\r\n", c->name));
	
	/* VFAT adate */
	if (VFAT (c->dev))
	{
		if (ptr)
			c->info.adate = SWAP68_W (ptr[1]);
		else
			c->info.adate = SWAP68_W (datestamp);
	}
	
	/* mtime/mdate */
	if (ptr)
	{
		c->info.time = SWAP68_W (ptr[2]);
		c->info.date = SWAP68_W (ptr[3]);
	}
	else
	{
		c->info.time = SWAP68_W (timestamp);
		c->info.date = SWAP68_W (datestamp);
	}
	
	FAT_DEBUG (("__FUTIME: leave return write_cookie\r\n"));
	
	/* write and leave */
	r = write_cookie (c);
	
	bio_SYNC_DRV ((&bio), DI (c->dev));
	return r;
}

static long
__FTRUNCATE (COOKIE *c, long newlen)
{
	const long oldlen = SWAP68_L (c->info.flen);
	long cl = newlen / CLUSTSIZE (c->dev);
	long actual;
	long r;
	
	FAT_DEBUG (("__FTRUNCATE [%s]: enter (newlen = %li)\r\n", c->name, newlen));
	
	/* range check */
	if (newlen > oldlen) return ERANGE;
	
	/* avoid simple case */
	if (newlen == oldlen) return E_OK;
	
	if (newlen && (newlen % CLUSTSIZE (c->dev) == 0))
	{
		/* correct cluster boundary */
		cl--;
	}
	
	/* search the new last cluster */
	actual = GETCL (c->stcl, c->dev, cl);
	if (actual <= 0)
	{
		/* bad clustered or read error */
		FAT_DEBUG (("__FTRUNCATE: leave failure, bad clustered (error = %li)\r\n", actual));
		return actual;
	}
	
	/* truncate cluster chain and update last cluster */
	r = GETCL (actual, c->dev, 1);
	if (r > 0)
	{
		(void) del_chain (r, c->dev);
		(void) FIXCL (actual, c->dev, CLLAST);
	}
	
	/* write new file len */
	c->info.flen = SWAP68_L (newlen);
	
	FAT_DEBUG (("__FTRUNCATE: leave return write_cookie\r\n"));
	
	/* write and leave */
	r = write_cookie (c);
	bio_SYNC_DRV ((&bio), DI (c->dev));
	return r;
}

static long
__FIO (FILEPTR *f, char *buf, long bytes, short mode)
{
	COOKIE *c = (COOKIE *) f->fc.index;
	FILE *ptr = (FILE *) f->devinfo;
	const ushort dev = c->dev;
	long actual = ptr->actual;
	long temp = ptr->flen - f->pos;
	long todo;
	long offset;
	long data;
	
	FAT_DEBUG (("__FIO [%s]: enter (bytes = %li, mode: %s)\r\n", c->name, bytes, (mode == READ) ? "READ" : "WRITE"));
	FAT_DEBUG (("__FIO: f->pos = %li, ptr->actual = %li\r\n", f->pos, ptr->actual));
	
	if (ptr->error < 0)
	{
		register long i = ptr->error;
		ptr->error = 0;
		FAT_DEBUG (("__FIO: ERROR (value = %li)\r\n", i));
		return i;
	}
	
	if (bytes <= 0)
	{
		FAT_DEBUG (("__FIO: ERROR (bytes = %li, return 0)\r\n", bytes));
		return 0;
	}
	
	if (mode == READ)
	{
		if (temp <= 0)
		{
			FAT_DEBUG (("__FIO: leave, file end (left = %li)\r\n", temp));
			return 0;
		}
		bytes = MIN (bytes, temp);
	}
	todo = bytes;
	
	if (c->stcl == 0)
	{
		/* 
		 * no first cluster,
		 * here only writing, if reading we leave before (while flen == 0)
		 */
		actual = nextcl (0, dev);
		if (actual <= 0)
		{
			FAT_DEBUG (("__FIO: leave failure (nextcl = %li)\r\n", actual));
			return EACCDN;
		}
		c->stcl = ptr->actual = actual;
		PUT_STCL (&(c->info), dev, actual);
	}
	
	while (todo > 0)
	{
		temp = f->pos / CLUSTSIZE (dev);
		if (temp > ptr->cl)
		{
			/* get next cluster */
			
			FAT_DEBUG (("__FIO: temp - ptr->cl = %li\r\n", temp - ptr->cl));
			
			actual = NEXTCL (actual, dev, mode);
			if (actual <= 0)
			{
				/* bad clustered */
				ptr->error = actual;
				FAT_DEBUG (("__FIO: leave failure, bad clustered (return = %li)\r\n", bytes - todo));
				break;
			}
			
			ptr->actual = actual;
			ptr->cl++;
		}
		
		/* offset */
		offset = f->pos % CLUSTSIZE (dev);
		
		if ((todo >= CLUSTSIZE (dev)) && (offset == 0))
		{
			register long cls = 1;
			data = CLUSTSIZE (dev);
			
			FAT_DEBUG (("__FIO: CLUSTER (todo = %li, pos = %li)\r\n", todo, f->pos));
			
			if (todo - data > CLUSTSIZE (dev))
			{
				register long oldcl = ptr->actual;
				register long newcl = NEXTCL (oldcl, dev, mode);
				
				/* linear read/write optimization */
				while ((newcl > 0) && ((newcl - oldcl) == 1))
				{
					data += CLUSTSIZE (dev);
					cls++;
					
					ptr->actual++;
					ptr->cl++;
					
					if (todo - data > CLUSTSIZE (dev))
					{
						oldcl = newcl;
						newcl = NEXTCL (oldcl, dev, mode);
					}
					else
						break;
				}
			}
			
			FAT_DEBUG (("__FIO: CLUSTER (data = %li, cluster = %li)\r\n", data, cls));
			
			/* read/write direct */
			if (mode == READ)
			{
				ptr->error = bio.l_read (DI (dev), C2S (actual, dev), cls, CLUSTSIZE (dev), buf);
			}
			else
			{
				ptr->error = bio.l_write (DI (dev), C2S (actual, dev), cls, CLUSTSIZE (dev), buf);
			}
			
			if (ptr->error)
			{
				FAT_DEBUG (("__FIO: leave failure, read/write direct\r\n"));
				break;
			}
			
			actual = ptr->actual;
		}
		else
		{
			UNIT *u;
			
			FAT_DEBUG (("__FIO: BYTES (todo = %li, pos = %li)\r\n", todo, f->pos));
			
			data = CLUSTSIZE (dev) - offset;
			data = MIN (todo, data);
			
			/* read the unit */
			u = bio.read (DI (dev), C2S (ptr->actual, dev), CLUSTSIZE (dev));
			if (!u)
			{
				ptr->error = EREADF;
				FAT_DEBUG (("__FIO: leave failure, read unit (return = %li, dev = %i)\r\n", bytes - todo, dev));
				break;
			}
			
			if (mode == READ)
			{
				/* copy */
				
				memcpy (buf, (u->data + offset), data);
			}
			else
			{
				/* copy and write */
				
				memcpy ((u->data + offset), buf, data);
				bio_MARK_MODIFIED ((&bio), u);
			}
		}
		
		buf += data;
		todo -= data;
		f->pos += data;
	}
	
	if (mode == WRITE)
	{
		if (f->pos > ptr->flen)
		{
			ptr->flen = f->pos;
			c->info.flen = SWAP68_L (f->pos);
		}
		if (c->info.attr != FA_SYMLINK
			&& !(c->info.attr & FA_CHANGED))
		{
			c->info.attr |= FA_CHANGED;
		}
		c->info.time = SWAP68_W (timestamp);
		c->info.date = SWAP68_W (datestamp);
		ptr->error = write_cookie (c);
		
		if (!BIO_WB_CHECK (DI (dev)))
		{
			if (bytes >= CLUSTSIZE (dev)
				&& f->pos % CLUSTSIZE (dev) == 0)
			{
				bio.sync_drv (DI (dev));
			}
		}
	}
	
	FAT_DEBUG (("__FIO: leave ok (todo = %li, processed = %li, pos = %li)\r\n", todo, bytes - todo, f->pos));
	return (bytes - todo);
}

/*
 * external
 */

static long
fatfs_open (FILEPTR *f)
{
	COOKIE *c = (COOKIE *) f->fc.index;
	FILE *ptr;
	
	FAT_DEBUG (("fatfs_open [%s]: enter\r\n", c->name));
	
	if (c->open)
		if (denyshare (c->open, f))
		{
			FAT_DEBUG (("fatfs_open: file sharing denied\r\n"));
			return EACCDN;
		}
	
	if (c->info.attr & FA_LABEL || c->info.attr & FA_DIR)
	{
		FAT_DEBUG (("fatfs_open: leave failure, not a valid file\r\n"));
		return EACCDN;
	}
	
	ptr = kmalloc (sizeof (*ptr));
	if (!ptr)
	{
		ALERT ("fatfs.c: kmalloc fail in: fatfs_open (%s)", c->name);
		FAT_DEBUG (("fatfs_open: leave failure, memory out for FILE struct\r\n"));
		return ENSMEM;
	}
	
	ptr->mode = f->flags;
	ptr->actual = c->stcl;
	ptr->cl = 0;
	ptr->flen = SWAP68_L (c->info.flen);
	
	ptr->error = 0;
	
	if (ptr->mode & O_TRUNC && (ptr->flen || c->stcl))
	{
		long r;
		
		FAT_DEBUG (("fatfs_open: truncate file to 0 bytes (flen = %li, stcl = %li)\r\n", ptr->flen, c->stcl));
		
		ptr->flen = 0;
		c->info.flen = 0;
		c->info.stcl = c->info.stcl_fat32 = 0;
		
		r = write_cookie (c);
		if (r)
		{
			kfree (ptr);
			FAT_DEBUG (("fatfs_open: leave failure (write_cookie = %li)\r\n", r));
			return r;
		}
		
		FAT_DEBUG (("fatfs_open: del_chain\r\n"));
		if (c->stcl)
		{
			(void) del_chain (c->stcl, c->dev);
			c->stcl = 0;
		}
		
		bio_SYNC_DRV ((&bio), DI (c->dev));
	}
	
	if ((ptr->mode & O_RWMODE) == O_EXEC)
	{
		ptr->mode = (ptr->mode ^ O_EXEC) | O_RDONLY;
	}
	
	f->pos = 0;
	f->next = c->open;
	c->open = f;
	c->links++;
	
	f->devinfo = (long) ptr;
	
	FAT_DEBUG (("fatfs_open: leave ok\r\n"));
	return E_OK;
}

static long
fatfs_write (FILEPTR *f, const char *buf, long bytes)
{
	FAT_DEBUG (("fatfs_write: enter (bytes = %li)\r\n", bytes));
	
	if ((((FILE *) f->devinfo)->mode & O_RWMODE) == O_RDONLY)
	{
		FAT_DEBUG (("fatfs_write: leave failure (bad mode)\r\n"));
		return EACCDN;
	}
	
	FAT_DEBUG (("fatfs_write: leave return __FIO ()\r\n"));
	return __FIO (f, (char *) buf, bytes, WRITE);
}

static long
fatfs_read (FILEPTR *f, char *buf, long bytes)
{
	FAT_DEBUG (("fatfs_read: enter (bytes = %li)\r\n", bytes));

	if ((((FILE *) f->devinfo)->mode & O_RWMODE) == O_WRONLY)
	{
		FAT_DEBUG (("fatfs_read: leave failure (bad mode)\r\n"));
		return EACCDN;
	}
	
	FAT_DEBUG (("fatfs_read: leave return __FIO ()\r\n"));
	return __FIO (f, buf, bytes, READ);
}

static long
fatfs_lseek (FILEPTR *f, long where, int whence)
{
	register COOKIE *c = (COOKIE *) f->fc.index;
	register FILE *ptr = (FILE *) f->devinfo;
	
	FAT_DEBUG (("fatfs_lseek [%s]: enter (where = %li, whence = %i)\r\n", c->name, where, whence));
	
	switch (whence)
	{
		case SEEK_SET:					break;
		case SEEK_CUR:	where += f->pos;		break;
		case SEEK_END:	where = ptr->flen + where;	break;
		default:	return EINVFN;			break;
	}
	
	if (where > ptr->flen || where < 0)
	{
		FAT_DEBUG (("fatfs_lseek: leave failure ERANGE (where = %li)\r\n", where));
		return ERANGE;
	}
	
	if (where == 0)
	{
		f->pos = 0;
		ptr->cl = 0;
		ptr->actual = c->stcl;
		FAT_DEBUG (("fatfs_lseek: leave ok (where = %li)\r\n", where));
		return 0;
	}
	
	{	/* calculate and set the new actual cluster and position */
		
		long actual = 0;
		register long cl = where / CLUSTSIZE (c->dev);
		
		if ((where % CLUSTSIZE (c->dev) == 0) && (where == ptr->flen))
		{
			/* correct cluster boundary */
			cl--;
		}
		
		if (cl != ptr->cl)
		{
			if (cl > ptr->cl)
				actual = GETCL (ptr->actual, c->dev, cl - ptr->cl);
			else
				actual = GETCL (c->stcl, c->dev, cl);
			if (actual <= 0)
			{
				/* bad clustered or read error */
				ptr->error = actual;
				
				FAT_DEBUG (("fatfs_lseek: leave failure, bad clustered (ptr->error = %li)\r\n", ptr->error));
				return EACCDN;
			}
			ptr->cl = cl;
			ptr->actual = actual;
		}
		f->pos = where;
	}
	
	FAT_DEBUG (("fatfs_lseek: leave ok (f->pos = %li)\r\n", f->pos));
	return where;
}

/*
 * the following function based on 'tos_ioctl' from 'tosfs.c'
 * please notice the right copyright
 */

static long
fatfs_ioctl (FILEPTR *f, int mode, void *buf)
{
	COOKIE *c = (COOKIE *) f->fc.index;
	FILE *ptr = (FILE *) f->devinfo;
	
	FAT_DEBUG (("fatfs_ioctl [%s]: enter (mode = %i)\r\n", c->name, mode));
	
	switch (mode)
	{
		case FIONREAD:
		{
			*(long *) buf = ptr->flen - f->pos;
			return E_OK;
		}
		case FIONWRITE:
		{
			*(long *) buf = 1;
			return E_OK;
		}
		case FUTIME:
		{
			return __FUTIME (c, buf);
		}
		case FTRUNCATE:
		{
			long r;
			
			if ((f->flags & O_RWMODE) == O_RDONLY)
				return EACCDN;
			
			r = __FTRUNCATE (c, *(long *) buf);
			if (r == E_OK)
			{
				long pos = f->pos;
				(void) fatfs_lseek (f, 0, SEEK_SET);
				(void) fatfs_lseek (f, pos, SEEK_SET);
			}
			
			return r;
		}
		case FIOEXCEPT:
		{
			*(long *) buf = 0;
			return E_OK;
		}
		case F_SETLK:
		case F_SETLKW:
		case F_GETLK:
		{
			struct flock *fl = (struct flock *) buf;
			
			LOCK t;
			LOCK *lck;
			
			t.l = *fl;
			
			switch (t.l.l_whence)
			{
				case SEEK_SET:
				{
					break;
				}
				case SEEK_CUR:
				{
					long r = fatfs_lseek (f, 0L, SEEK_CUR);
					t.l.l_start += r;
					break;
				}
				case SEEK_END:
				{
					long r = fatfs_lseek (f, 0L, SEEK_CUR);
					t.l.l_start = fatfs_lseek (f, t.l.l_start, SEEK_END);
					(void) fatfs_lseek (f, r, SEEK_SET);
					break;
				}
				default:
				{
					FAT_DEBUG (("fatfs_ioctl: invalid value for l_whence\r\n"));
					return EINVFN;
				}
			}
			
			if (t.l.l_start < 0) t.l.l_start = 0;
			t.l.l_whence = 0;
			
			if (mode == F_GETLK)
			{
				lck = denylock (c->locks, &t);
				if (lck)
					*fl = lck->l;
				else
					fl->l_type = F_UNLCK;
				
				return E_OK;
			}
			
			if (t.l.l_type == F_UNLCK)
			{
				/* try to find the lock */
				LOCK **lckptr = &(c->locks);
				
				lck = *lckptr;
				while (lck)
				{
					if (lck->l.l_pid == curproc->pid
						&& lck->l.l_start == t.l.l_start
						&& lck->l.l_len == t.l.l_len)
					{
						/* found it -- remove the lock */
						*lckptr = lck->next;
						FAT_DEBUG (("fatfs_ioctl: unlocked %s: %ld + %ld", c->name, t.l.l_start, t.l.l_len));
						/* (void) fatfs_lock (f, 1, t.l.l_start, t.l.l_len); */
						
						/* wake up anyone waiting on the lock */
						wake (IO_Q, (long) lck);
						kfree (lck);
						
						return E_OK;
					}
					
					lckptr = &(lck->next);
					lck = lck->next;
				}
				
				return ENSLOCK;
			}
			
			FAT_DEBUG (("fatfs_ioctl: lock %s: %ld + %ld", c->name, t.l.l_start, t.l.l_len));
			
			do {
				long r;
				
				/* see if there's a conflicting lock */
				while ((lck = denylock (c->locks, &t)) != 0)
				{
					FAT_DEBUG (("fatfs_ioctl: lock conflicts with one held by %d", lck->l.l_pid));
					if (mode == F_SETLKW)
					{
						/* sleep a while */
						sleep (IO_Q, (long) lck);
					}
					else
						return ELOCKED;
				}
				
				/* if not, add this lock to the list */
				lck = kmalloc (sizeof (*lck));
				if (!lck)
				{
					ALERT ("fatfs.c: kmalloc fail in: fatfs_ioctl (%s)", c->name);
					return ENSMEM;
				}
				
				r = E_OK; /* fatfs_lock (f, 0, t.l.l_start, t.l.l_len); */
				if (r)
				{
					kfree (lck);
					if (mode == F_SETLKW && r == ELOCKED)
					{
						yield ();
						lck = NULL;
					}
					else
						return r;
				}
			}
			while (!lck);
			
			lck->l = t.l;
			lck->l.l_pid = curproc->pid;
			lck->next = c->locks;
			c->locks = lck;
			
			/* mark the file as being locked */
			f->flags |= O_LOCK;
			return E_OK;
		}
	}
	
	FAT_DEBUG (("fatfs_ioctl: return EINVFN\r\n"));
	return EINVFN;
}

static long
fatfs_datime (FILEPTR *f, ushort *time, int rwflag)
{
	COOKIE *c = (COOKIE *) f->fc.index;
	register FILE *ptr = (FILE *) f->devinfo;
	
	FAT_DEBUG (("fatfs_datime [%s]: enter\r\n", c->name));
	
	if (rwflag)
	{
		/* set the date/time */
		
		c->info.time = SWAP68_W (time[0]);
		c->info.date = SWAP68_W (time[1]);
		
		if (VFAT (c->dev))
		{
			c->info.adate = c->info.date;
		}
		
		ptr->error = write_cookie (c);
		bio_SYNC_DRV ((&bio), DI (c->dev));
	}
	else
	{
		/* read the date/time */
		
		time[0] = SWAP68_W (c->info.time);
		time[1] = SWAP68_W (c->info.date);
	}
	
	FAT_DEBUG (("fatfs_datime: leave ok\r\n"));
	return E_OK;
}

static long
fatfs_close (FILEPTR *f, int pid)
{
	COOKIE *c = (COOKIE *) f->fc.index;
	
	FAT_DEBUG (("fatfs_close [%s]: enter (f->links = %i)\r\n", c->name, f->links));
	
/*
 * the following code based on 'tos_close' from 'tosfs.c'
 * please notice the right copyright
 *
 * start
 */
	/* if a lock was made, remove any locks of the process */
	if (f->flags & O_LOCK)
	{
		LOCK *lock;
		LOCK **oldlock;
		
		FAT_DEBUG (("fat_close: remove lock (pid = %i)", pid));
		
		oldlock = &c->locks;
		lock = *oldlock;
		
		while (lock)
		{
			if (lock->l.l_pid == pid)
			{
				*oldlock = lock->next;
				/* (void) fatfs_lock (f, 1, lock->l.l_start, lock->l.l_len); */
				wake (IO_Q, (long) lock);
				kfree (lock);
			}
			else
			{
				oldlock = &lock->next;
			}
			lock = *oldlock;
		}
	}
/* end */
	
	if (f->links <= 0)
	{
		FILE *ptr = (FILE *) f->devinfo;
		
		/* free the extra info */
		kfree (ptr);
		
		/* remove the FILEPTR from the linked list */
		{
			register FILEPTR **temp = &c->open;
			long flag = 1;
			while (*temp)
			{
				if (*temp == f)
				{
					*temp = f->next;
					f->next = NULL;
					flag = 0;
					break;
				}
				temp = &(*temp)->next;
			}
			if (flag)
			{
				ALERT ("fatfs.c: remove open FILEPTR fail in: fatfs_close (%s)", c->name);
			}
		}
		
		c->links--;
	}
	
	bio_SYNC_DRV ((&bio), DI (c->dev));
	
	FAT_DEBUG (("fatfs_close: leave ok\r\n"));
	return E_OK;
}

/* END device driver */
/****************************************************************************/

/****************************************************************************/
/* BEGIN debug infos */

# ifdef FS_DEBUG

# include <stdarg.h>
# include "dosfile.h"

static void
fatfs_debug (const char *fmt, ...)
{
	static char line [SPRINTF_MAX];
	if (fatfs_debug_mode)
	{
		va_list args;
		int foo;
		FILEPTR *f;
		
		va_start(args, fmt);
		f = do_open (FS_LOGFILE, (O_WRONLY|O_CREAT|O_APPEND), 0, (XATTR *)0);
		if (f)
		{
			foo = vksprintf (line, fmt, args);
			(void) (*f->dev->lseek)(f, 0, SEEK_END);
			(*f->dev->write)(f, line, _STRLEN (line));
			do_close (f);
		}
		else
		{
			foo = vksprintf (line, fmt, args);
			FORCE (line);
		}
		va_end(args);
	}
}

static void
fatfs_print_dir (const _DIR *d, ushort dev)
{
	char attr[80] = "\0";
	static char stime[20];
	static char sdate[20];
	ushort time = SWAP68_W (d->time), date = SWAP68_W (d->date);
	long i = 0;
	
	if (d->attr & FA_RDONLY)
	{
		ksprintf (attr+i, "READONLY  ");
		i += 10;
	}
	if (d->attr & FA_HIDDEN)
	{
		ksprintf (attr+i, "HIDDEN    ");
		i += 10;
	}
	if (d->attr & FA_SYSTEM)
	{
		ksprintf (attr+i, "SYSTEM    ");
		i += 10;
	}
	if (d->attr & FA_LABEL)
	{
		ksprintf (attr+i, "LABEL     ");
		i += 10;
	}
	if (d->attr & FA_DIR)
	{
		ksprintf (attr+i, "DIR       ");
		i += 10;
	}
	if (d->attr & FA_CHANGED)
	{
		ksprintf (attr+i, "CHANGED   ");
		i += 10;
	}
	ksprintf (stime, "%d:%d,%d",
		((time >> 11) & 31),
		((time >> 5) & 63),
		((time & 31) << 1));
	ksprintf (sdate, "%d.%d.19%d",
		 (date & 31),
		 ((date >> 5) & 15),
		 (80 + ((date >> 9) & 127)));
	
	FAT_DEBUG ((
		"---\r\n"
		"name: %s, attr: %s\r\n"
		"time: %s, date: %s\r\n"
		"stcl: %u, flen: %lu\r\n",
		d->name, attr,
		stime, sdate,
		SWAP68_W (d->stcl), SWAP68_L (d->flen)
	));
}

static void
fatfs_dump_hashtable (void)
{
	static char line [SPRINTF_MAX];
	short foo;
	FILEPTR *f;
	
	f = do_open (FS_DUMPFILE, (O_WRONLY|O_CREAT|O_TRUNC), 0, (XATTR *)0);
	if (f)
	{
		long i;
		(*f->dev->write)(f, "BPBs:\r\n", 7);
		for (i = 0; i < NUM_DRIVES; i++)
		{
			foo = ksprintf (line, "nr: %li\tvalid = %li\tdrv = %i", i, (long) BPBVALID (i), BPBVALID (i) ? DI(i)->drv : -1);
			(*f->dev->write)(f, line, _STRLEN (line));
			(*f->dev->write)(f, "\r\n", 2);
		}
		(*f->dev->write)(f, "cookies:\r\n", 10);
		for (i = 0; i < COOKIE_CACHE; i++)
		{
			foo = ksprintf (line, "nr: %li\tlinks = %li\tdev = %i\tname = %s", i, cookies[i].links, cookies[i].dev, cookies[i].name);
			(*f->dev->write)(f, line, _STRLEN (line));
			(*f->dev->write)(f, "\r\n", 2);
		}
		(*f->dev->write)(f, "table:\r\n", 8);
		for (i = 0; i < COOKIE_CACHE; i++)
		{
			COOKIE *temp = table [i];
			foo = ksprintf (line, "nr: %li\tptr = %lx", i, temp);
			(*f->dev->write)(f, line, _STRLEN (line));
			for (; temp; temp = temp->next)
			{
				foo = ksprintf (line, "\r\n\tnext = %lx\tlinks = %li\tdev = %i\tname = %s\r\n", temp->next, temp->links, temp->dev, temp->name);
				(*f->dev->write)(f, line, _STRLEN (line));	
			}
			(*f->dev->write)(f, "\r\n", 2);
		}
		do_close (f);
	}
}

# if FS_DEBUG_COOKIE
static void
fatfs_print_cookie (COOKIE *c)
{
	FAT_DEBUG (("c->name = %s\r\n", c->name));
	FAT_DEBUG (("c->links = %li, c->dev = %i\r\n", c->links, c->dev));
	FAT_DEBUG (("c->dir = %li, c->offset = %li, c->stcl = %li\r\n", c->dir, c->offset, c->stcl));
	FAT_DEBUG (("c->slots = %li\r\n", (long) c->slots));
	FAT_DEBUG_PRINTDIR ((&(c->info), c->dev));
}
# endif /* FS_DEBUG_COOKIE */

# endif /* FS_DEBUG */

/* END debug infos */
/****************************************************************************/
