/*
 * This file belongs to FreeMiNT.  It's not in the original MiNT 1.12
 * distribution.  See the file Changes.MH for a detailed log of changes.
 */

/*
 * This file is dedicated to the FreeMiNT project.
 * It's not allowed to use this file for other projects without my
 * explicit permission.
 */

/*
 * begin:	1998-02
 * last change: 1998-09-10
 * 
 * Author: Frank Naumann - <fnaumann@cs.uni-magdeburg.de>
 * 
 * please send suggestions or bug reports to me or
 * the MiNT mailing list
 * 
 * 
 * changes since last version:
 * 
 * 1998-09-22:
 * 
 * - fix: bio_wbq_remove clear now wb_nex/wb_free
 * 
 * 1998-09-08:
 * 
 * - new: removed all 'const' DI
 * - new: add XHReserve/XHLock support for removable medias
 * 
 * 1998-09-02:	(revision 3)
 * 
 * - Warning: Interface really changed!
 * 
 * - change: optimised l_write for small transfers
 * - merged hdio.c and bcache.c into block_IO.c
 *   this simplify a lot and allows the integration of additional features
 *   - add writeback queues, dirty UNITs will be written back
 *     in sorted order now, UNITs will also be blocked
 *   - add seperate UNIT hash tables for each device
 * 
 * - change: getunit/read -> remove lock parameter
 * - change: l_read/l_write -> works blockorientated
 *                          -> self optimising data transfers
 * 
 * 1998-07-13:	(revision 2)
 * 
 * - Warning: Interface changed!
 * - new: bio_lookup -> check if a UNIT is in cache
 * - new: bio_getunit -> same as bio_read but without reading
 *        (useful for only written units)
 * - new: bio_lock -> lock a unit after read
 * - new: prototype for bio_pre_read (not implemented yet)
 * - fix: l_read/l_write check for existing cache entries
 *        and sync if necessary
 * 
 * known bugs:
 * 
 * - nothing
 * 
 * todo:
 * 
 * - bio_pre_read
 * 
 */

# include "block_IO.h"
# include "global.h"

# include "bios.h"
# include "memory.h"
# include "pun.h"
# include "xhdi.h"
# include "util.h"


# define BLOCK_IO_VERS	3		/* internal version */

# define DEFAULT	128L		/* default cache in kB */

# define HASHBITS	8		/* size of UNIT hashtable */
# define HASHSIZE	(1UL << HASHBITS)
# define HASHMASK	(HASHSIZE - 1)

# define MIN_BLOCK	8192L		/* minimal block size */
# define CHUNK_SIZE	512L		/* minimal chunk size */
# define CHUNK_SHIFT	9		/* shift value */

# define UNLOCK		0
# define LOCK		1

# define WB_BUFFER	(1024L * 32)	/* 32 kb writeback buffer (static) */

/*
 * debugging stuff
 */

# ifdef DEBUG_INFO
# define BLOCK_IO_DEBUG 1
# define BIO_LOGFILE "u:\\ram\\fat.log"
# define BIO_DUMPFILE "u:\\ram\\block_IO.dmp"
# endif

# ifndef BLOCK_IO_DEBUG
#  define BIO_DEBUG(x)
#  define BIO_DEBUG_CACHE(x)
# else
#  define BIO_DEBUG(x) bio_debug x
#  define BIO_DEBUG_CACHE(x) bio_dump_cache x
   static long debug_mode;
   static void bio_debug (const char *s, ...);
   static void bio_dump_cache (void);
# endif

/****************************************************************************/
/* BEGIN definition part */

static long	bio_config	(const ushort dev, const long config, const long mode);

/* DI management */
static DI *	bio_get_di	(ushort drv);
static DI *	bio_res_di	(ushort drv);
static void	bio_free_di	(DI *di);

/* physical/logical calculation init */
static void	bio_set_pshift	(DI *di, ulong physical);
static void	bio_set_lshift	(DI *di, ulong logical);

/* cached block I/O */
static UNIT *	bio_lookup	(DI *di, long sector, long blocksize);
static UNIT *	bio_getunit	(DI *di, long sector, long blocksize);
static UNIT *	bio_read	(DI *di, long sector, long blocksize);
static long	bio_write	(UNIT *u);
static long	bio_l_read	(DI *di, long sector, long blocks, long blocksize, void *buf);
static long	bio_l_write	(DI *di, long sector, long blocks, long blocksize, void *buf);

/* optional feature */
static void	bio_pre_read	(DI *di, long *sector, long blocks, long blocksize);

/* synchronization */
static void	bio_lock	(UNIT *u);
static void	bio_unlock	(UNIT *u);

/* update functions */
static void	bio_mark_modified(UNIT *u);
static void	bio_sync_drv	(DI *di);

/* cache management */
static long	bio_validate	(DI *di, long maxblocksize);
static void	bio_invalidate	(DI *di);

BIO bio =
{
	BLOCK_IO_VERS, 0, bio_config,
	bio_get_di, bio_res_di, bio_free_di,
	bio_set_pshift, bio_set_lshift,
	bio_lookup, bio_getunit, bio_read, bio_write, bio_l_read, bio_l_write,
	bio_pre_read,
	bio_lock, bio_unlock,
	bio_mark_modified, bio_sync_drv,
	bio_validate, bio_invalidate
};

/* cache block */
struct cbl
{
	char	*data;		/* ptr to the data */
	UNIT	**active;	/* array of the used UNITS */
	short	*used;		/* array of the UNIT positions */
	ulong	stat;		/* access statistic */
	short	lock;		/* locked unit counter */
	short	free;		/* free chunks */
};


/*
 * internal prototypes
 */


/* rwabs wrapper */

static long	rwabs_log	(DI *di, ushort rw, void *buf, ulong size, ulong recno);
static long	rwabs_log_lrec	(DI *di, ushort rw, void *buf, ulong size, ulong recno);
/*static long	rwabs_phy	(DI *di, ushort rw, void *buf, ulong size, ulong recno);*/
/*static long	rwabs_phy_lrec	(DI *di, ushort rw, void *buf, ulong size, ulong recno);*/
static long	rwabs_xhdi	(DI *di, ushort rw, void *buf, ulong size, ulong recno);


/* dskchng wrapper */

static long	dskchng_bios	(DI *di);
static long	dskchng_xhdi	(DI *di);


/* */

FASTFN void	bio_xhdi_lock	(register DI *di);
FASTFN void	bio_xhdi_unlock	(register DI *di);


/* cache help functions */

FASTFN long	bio_get_chunks	(register long size);
FASTFN void	bio_update_stat	(register UNIT *u);


/* cache hash table functions */

FASTFN ulong	bio_hash	(register const long sector);
FASTFN UNIT *	bio_hash_lookup	(register const long sector, register const long size, register UNIT **table);
FASTFN void	bio_hash_install(register UNIT *u);
FASTFN void	bio_hash_remove	(register UNIT *u);


/* writeback queue functions */

FASTFN void	bio_wbq_insert	(register UNIT *u);
FASTFN void	bio_wbq_remove	(register UNIT *u);

FASTFN void	bio_wb_writeout	(DI *di, void *buffer, long size, long sector);
FASTFN void	bio_wb_unit	(register UNIT *u);
FASTFN void	bio_wb_queue	(DI *di);


/* cache unit management functions */

FASTFN void	bio_unit_remove	(register UNIT *u);
static UNIT *	bio_unit_get	(DI *di, long sector, long size);

/* END definition part */
/****************************************************************************/

/****************************************************************************/
/* BEGIN rwabs wrapper */

static long
rwabs_log (DI *di, ushort rw, void *buf, ulong size, ulong recno)
{
	ulong n = size >> (di->pshift + di->lshift);
	if (n > 65535UL)
	{
		ALERT ("hdio.c: rwabs_log: n to large (drv = %i)", di->drv);
		return ESECNF;
	}
	return rwabs (rw, buf, n, recno, di->drv, 0L);
}

static long
rwabs_log_lrec (DI *di, ushort rw, void *buf, ulong size, ulong recno)
{
	ulong n = size >> (di->pshift + di->lshift);
	if (n > 65535UL)
	{
		ALERT ("hdio.c: rwabs_log_lrec: n to large (drv = %i)", di->drv);
		return ESECNF;
	}
	return rwabs (rw, buf, n, -1, di->drv, recno);
}

# if 0
static long
rwabs_phy (DI *di, ushort rw, void *buf, ulong size, ulong recno)
{
	ulong n = size >> di->pshift;
	recno <<= di->lshift;
	if (!n || (recno + n) > di->size)
	{
		ALERT ("hdio.c: rwabs_phy: access outside partition (drv = %i)", di->drv);
		return ESECNF;
	}
	if (n > 65535UL)
	{
		ALERT ("hdio.c: rwabs_phy: n to large (drv = %i)", di->drv);
		return ESECNF;
	}
	return rwabs (rw | 8, buf, n, (recno + di->start), di->drv, 0L);
}

static long
rwabs_phy_lrec (DI *di, ushort rw, void *buf, ulong size, ulong recno)
{
	ulong n = size >> di->pshift;
	recno <<= di->lshift;
	if (!n || (recno + n) > d->size)
	{
		ALERT ("hdio.c: rwabs_phy_lrec: access outside partition (drv = %i)", di->drv);
		return ESECNF;
	}
	if (n > 65535UL)
	{
		ALERT ("hdio.c: rwabs_phy_lrec: n to large (drv = %i)", di->drv);
		return ESECNF;
	}
	return rwabs (rw | 8, buf, n, -1, di->drv, (recno + di->start));
}
# endif

static long
rwabs_xhdi (DI *di, ushort rw, void *buf, ulong size, ulong recno)
{
	register ulong n = size >> di->pshift;
	recno <<= di->lshift;
	if (!n || (recno + n) > di->size)
	{
		ALERT ("hdio.c: rwabs_xhdi: access outside partition (drv = %i)", di->drv);
		return ESECNF;
	}
	if (n > 65535UL)
	{
		ALERT ("hdio.c: rwabs_xhdi: n to large (drv = %i)", di->drv);
		return ESECNF;
	}
	return XHReadWrite (di->major, di->minor, rw, recno + di->start, n, buf);
}

/* END rwabs wrapper */
/****************************************************************************/

/****************************************************************************/
/* BEGIN dskchng wrapper */

static long
dskchng_bios (DI *di)
{
	long r;
	
	r = mediach (di->drv);
	if (r) (void) getbpb (di->drv);
	
	return r;
}

static long
dskchng_xhdi (DI *di)
{
	long r;
	
	r = mediach (di->drv);
	if (r) (void) getbpb (di->drv);
	
	return r;
}

/* END dskchng wrapper */
/****************************************************************************/

/****************************************************************************/
/* BEGIN  */

FASTFN void
bio_xhdi_lock (register DI *di)
{
	if (!di->key)
	{
		di->key = XHReserve (di->major, di->minor, LOCK, 0);
		XHLock (di->major, di->minor, LOCK, di->key);
	}
}

FASTFN void
bio_xhdi_unlock (register DI *di)
{
	if (di->key)
	{
		XHLock (di->major, di->minor, UNLOCK, di->key);
		XHReserve (di->major, di->minor, UNLOCK, di->key);
		di->key = 0;
	}
}

/* END  */
/****************************************************************************/

/****************************************************************************/
/* BEGIN cache help functions */

FASTFN long
bio_get_chunks (register long size)
{
	return (size + (CHUNK_SIZE - 1)) >> CHUNK_SHIFT;
}

FASTFN void
bio_update_stat (register UNIT *u)
{
	u->cbl->stat = u->stat = c20ms;
}

/* END cache help functions */
/****************************************************************************/

/****************************************************************************/
/* BEGIN cache hash table */

/*
 * unit hash manipulation
 */

FASTFN ulong
bio_hash (register const long sector)
{
	register ulong hash;
	
	hash = sector;
	hash = hash + (hash >> HASHBITS) + (hash >> (HASHBITS << 1));
	
	return hash  & HASHMASK;
}

FASTFN UNIT *
bio_hash_lookup (register const long sector, register const long size, register UNIT **table)
{
	register UNIT *u;
	
	for (u = table [bio_hash (sector)]; u; u = u->next)
	{
		if (u->sector == sector)
		{
			/* failure of the xfs */
			assert (size <= u->size);
			
			bio_update_stat (u);
			return u;
		}
	}
	
	return NULL;
}

FASTFN void
bio_hash_install (register UNIT *u)
{
	register UNIT **n = &(u->di->table [bio_hash (u->sector)]);
	
	u->next = *n;
	*n = u;
	
	bio_update_stat (u);
}

FASTFN void
bio_hash_remove (register UNIT *u)
{
	register UNIT **n = &(u->di->table [bio_hash (u->sector)]);
	
	while (*n)
	{
		if (u == *n)
		{
			/* remove from table */
			*n = (*n)->next;
			
			return;
		}
		n = &((*n)->next);
	}
}

/* END cache hash table */
/****************************************************************************/

/****************************************************************************/
/* BEGIN writeback queue functions */


/*
 * Attention: must be an elementar and list save insert operation
 * 
 * - wb_next/wb_prev must be NULL
 */

FASTFN void
bio_wbq_insert (register UNIT *u)
{
	if (!u->dirty)
	{
		register UNIT *queue = u->di->wb_queue;
		
		u->dirty = 1;
		
		if (queue)
		{
			register UNIT *old = NULL;
			
			while (queue && queue->sector < u->sector)
			{
				old = queue;
				queue = queue->wb_next;
			}
			
			if (queue)
			{
				if (queue->wb_prev)
				{
					queue->wb_prev->wb_next = u;
					u->wb_prev = queue->wb_prev;
				}
				
				u->wb_next = queue;
				queue->wb_prev = u;
				
				/* head check */
				if (u->di->wb_queue == queue)
				{
					 u->di->wb_queue = u;
				}
			}
			else
			{
				/* last element */
				old->wb_next = u;
				u->wb_prev = old;
			}
		}
		else
		{
			/* empty list*/
			u->di->wb_queue = u;
		}
	}
}


/*
 * Attention: must be an elementar and list save remove operation
 * 
 * - set wb_next/wb_prev to NULL
 */

FASTFN void
bio_wbq_remove (register UNIT *u)
{
	if (u->dirty)
	{
		if (u->wb_next)
			u->wb_next->wb_prev = u->wb_prev;
		
		if (u->wb_prev)
			u->wb_prev->wb_next = u->wb_next;
		else
		{
			/* u is first element -> correct start */
			u->di->wb_queue = u->wb_next;
		}
		
		u->wb_next = NULL;
		u->wb_prev = NULL;
		u->dirty = 0;
	}
}


/*
 * write out size bytes in buffer to sector
 * on device DI
 */

FASTFN void
bio_wb_writeout (DI *di, void *buffer, long size, long sector)
{
	register long r;
	
	r = BIO_RWABS (di, 1, buffer, size, sector);
	if (r)
	{
		ALERT ("block_IO.c: bio_wb_writeout: RWABS fail (ignored, return = %li))", r);
	}
	else
	{
		BIO_DEBUG (("bio_wb_writeout: (sector %li, bytes %li) written back!\r\n", sector, size));
	}
}

/*
 * writeback one cache UNIT
 * 
 * Attention: must be a list save writeback operation
 * 
 * - first remove element from writeback queue
 * - then writeback
 */

FASTFN void
bio_wb_unit (register UNIT *u)
{
	if (u->dirty)
	{
		bio_wbq_remove (u);
		bio_wb_writeout (u->di, u->data, u->size, u->sector);
	}
}


/*
 * writeback a complete queue
 * 
 * - writeback queue is invalidated
 * - then goes through the list
 * - all wb_next/wb_prev pointer from elements are set to NULL
 */

FASTFN void
bio_wb_queue (DI *di)
{
# if 0
	static char buffer [WB_BUFFER]; /* this is ok here */
# endif
	register UNIT *queue = di->wb_queue;	/* start point */
	
	/* invalidate queue */
	di->wb_queue = NULL;
	
	/* go through writeback queue */
	while (queue)
	{
		/* save next element */
		register UNIT *next = queue->wb_next;
		
# if 0
		/* calculate offset to next sector */
		register long incr = queue->size >> (di->pshift + di->lshift);
# endif
		
		/* invalidate element on wb queue */
		queue->wb_prev = NULL;
		queue->wb_next = NULL;
		queue->dirty = 0;
		
# if 0 /* to expensive */
		if (next
			&& ((queue->sector + incr) == next->sector)
			&& ((queue->size + next->size) <= buffersize))
		{
			register long sector = queue->sector;
			register long size = queue->size;
			
			quickmovb (buffer, queue->data, size);
			
			do {
				queue = next;
				next = queue->wb_next;
				
				/* invalidate element on wb queue */
				queue->wb_prev = NULL;
				queue->wb_next = NULL;
				queue->dirty = 0;
				
				quickmovb (buffer + size, queue->data, queue->size);
				size += queue->size;
				
				incr = queue->size >> (di->pshift + di->lshift);
			}
			while (next
				&& ((queue->sector + incr) == next->sector)
				&& ((size + next->size) <= buffersize));
			
			bio_wb_writeout (di, buffer, size, sector);
		}
		else
# endif
			bio_wb_writeout (di, queue->data, queue->size, queue->sector);
		
		queue = next;
	}
}

/* END writeback queue functions */
/****************************************************************************/

/****************************************************************************/
/* BEGIN cache unit management */

/*
 * block cache
 */

static struct
{
	long	max_size;	/* max. blocksize */
	long	chunks;		/* number of chunks in each block */
	long	count;		/* number of blocks in cache */
	CBL	*blocks;	/* ptr array to the cache blocks */
	
} cache;

FASTFN void
bio_unit_remove (register UNIT *u)
{
	const long chunks = bio_get_chunks (u->size);
	register short *used = u->cbl->used + u->pos;
	register long n;
	
	*(u->cbl->active + u->pos) = NULL;
	
	/* clear block datas */
	for (n = chunks; n; n--, used++)
	{
		*used = 0;
	}
	
	/* remove from hash table */
	bio_hash_remove (u);
	
	/* writeback if dirty; always a list save operation */
	bio_wb_unit (u);
	
	/* remove any lock */
	if (u->lock) u->cbl->lock -= u->lock;
	
	/* correct n */
	u->cbl->free += chunks;
	
	/* free the memory */
	kfree (u);
}

static UNIT *
bio_unit_get (DI *di, long sector, long size)
{
	const long n = bio_get_chunks (size);
	UNIT *new;
	
	register long found = -1;
	
	BIO_DEBUG (("bio_unit_get: enter (size = %li)\r\n", size));
	
	/* failure of the xfs */
	assert (size <= cache.max_size);
	
	{	register CBL *b = cache.blocks;
		register const long ni = n;
		register ulong min = 4294967295UL;
		register long i;
		
		for (i = cache.count; i; i--, b++)
		{
			if (b->lock == 0)
			{
				if (ni < b->free)
				{
					found = cache.count - i;
					break;
				}
				{	register ulong temp = b->stat;
					if (temp < min)
					{
						min = temp;
						found = cache.count - i;
					}
				}
			}
		}
	}
	
	if (found < 0)
	{
		ALERT ("block_IO.c: bio_unit_get: no free unit in cache, abort");
		return NULL;
	}
	
	BIO_DEBUG (("bio_unit_get: use CBL %li\r\n", found));
	
	new = kmalloc (sizeof (*new));
	if (new)
	{
		CBL * const b = cache.blocks + found;
		short *actual_used = b->used;
		register const long end = (cache.chunks == n) ? 0 : cache.chunks - n + 1;
		register long long min_cost = 4294967295UL;
		long i;
		
		found = 0;
		
		for (i = end; i; i--, actual_used++)
		{
			register short *used = actual_used;
			register short old_used = 0;
			register long cost = 0;
			register long j;
			
			for (j = n; j; j--, used++)
			{
				if (*used)
				{
					if (*used != old_used)
					{
						register UNIT *u = b->active [*used - 1];
						old_used = *used;
						cost += u->size;
						cost -= (c20ms - u->stat);
						if (u->dirty) cost <<= 1;
					}
				}
				else
					cost -= CHUNK_SIZE;
			}
			
			if (cost < min_cost)
			{
				min_cost = cost;
				found = end - i;
			}
		}
		
		{	register short *used = b->used + found;
			found++;
			for (i = n; i; i--, used++)
			{
				if (*used) bio_unit_remove (b->active [*used - 1]);
				*used = found;
			}
			found--;
		}
		
		new->data = b->data + (found << CHUNK_SHIFT);
		new->next = NULL;
		new->wb_prev = NULL;
		new->wb_next = NULL;
		new->cbl = b;
		new->di = di;
		new->sector = sector;
		new->size = size;
		new->stat = 0;
		new->pos = found;
		new->dirty = 0;
		new->lock = 0;
		
		bio_hash_install (new);
		
		b->free -= n;
		*(b->active + found) = new;
		
		return new;
	}
	
	BIO_DEBUG (("bio_unit_get: leave can't get free UNIT (kmalloc (i) fail)\r\n"));
	ALERT ("block_IO.c: bio_unit_get: kmalloc (%li) fail, out of memory?", sizeof (*new));
	
	return NULL;
}

/* END cache unit management */
/****************************************************************************/

/****************************************************************************/
/* BEGIN global data */

static DI bio_di [NUM_DRIVES];

/* END global data */
/****************************************************************************/

/****************************************************************************/
/* BEGIN init & configuration */

void
init_block_IO (void)
{
	long i;
	
	long m_stat;
	long m_block;
	char *data;
	
	/* validate AHDI minimum version */
	PUN_INFO *pun = get_pun ();
	if (!pun)
	{
		FATAL ("AHDI 3.0 not supported on this system.");
	}
	
	/* initalize XHDI interface */
	XHDI_init ();
	
	/* initalize DI array */
	for (i = 0; i < NUM_DRIVES; i++)
	{
		DI *di = & bio_di [i];
		
		di->next	= NULL;
		di->table	= NULL;
		di->wb_queue	= NULL;
		
		*((ushort *) &(di->drv)) = i; /* to avoid warning */
		di->major	= 0;
		di->minor	= 0;
		di->mode	= 0;
		
		di->start	= 0;
		di->size	= 0;
		di->pssize	= 0;
		
		di->lshift	= 0;
		di->pshift	= 0;
		
		/* default I/O routines */
		di->rwabs	= rwabs_log;
		di->dskchng	= dskchng_bios;
		
		di->valid	= 0;
		di->lock	= DISABLE;
		di->key		= 0;
	}
	
	
	/*
	 * initalize cache structures
	 * initalize default cache
	 */
	
	cache.max_size = pun->max_sect_siz * 2;
	cache.max_size = MAX (cache.max_size, MIN_BLOCK);
	cache.count = (DEFAULT * 1024L) / cache.max_size;
	cache.chunks = cache.max_size >> CHUNK_SHIFT;
	
	m_stat = cache.chunks;
	m_stat *= (sizeof (UNIT *) + sizeof (short *));
	
	m_block = m_stat;
	m_block += sizeof (CBL);
	
	cache.blocks = kmalloc (m_block * cache.count);
	data = kmalloc (cache.count * cache.max_size);
	
	if (!cache.blocks || !data)
	{
		FATAL ("Not enough RAM for initial hd cache.");
	}
	
	{	char *c = (char *) (cache.blocks + cache.count);
		long i;
		for (i = 0; i < cache.count; i++)
		{
			cache.blocks [i].data = data; data += cache.max_size;
			cache.blocks [i].active = (UNIT **) c;
			cache.blocks [i].used = (short *) (c + cache.chunks * sizeof (UNIT *));
			cache.blocks [i].lock = 0;
			cache.blocks [i].free = cache.chunks;
			cache.blocks [i].stat = 0;
			c += m_stat;
			
			/* initialize block */
			{
				long j;
				for (j = 0; j < cache.chunks; j++)
				{
					cache.blocks [i].active [j] = NULL;
					cache.blocks [i].used [j] = 0;
				}
			}
		}
	}
}

void
bio_set_cache_size (long size)
{
	long count;
	long m_stat;
	long m_block;
	CBL *blocks;
	char *data;
	
	count = (size * 1024L) / cache.max_size;
	
	m_stat = cache.chunks;
	m_stat *= (sizeof (UNIT *) + sizeof (short *));
	
	m_block = m_stat;
	m_block += sizeof (CBL);
	
	blocks = kmalloc (m_block * (cache.count + count));
	data = kmalloc (count * cache.max_size);
	
	if (!blocks || !data)
	{
		if (blocks) kfree (blocks);
		if (data) kfree (data);
		
		ALERT ("block_IO.c: Not enough RAM for additional buffer cache.");
	}
	
	{	char *c = (char *) (blocks + cache.count + count);
		long i;
		for (i = 0; i < cache.count + count; i++)
		{
			if (i < cache.count)
			{
				blocks [i].data = cache.blocks [i].data;
				blocks [i].active = (UNIT **) c;
				blocks [i].used = (short *) (c + cache.chunks * sizeof (UNIT *));
				blocks [i].lock = cache.blocks [i].lock;
				blocks [i].free = cache.blocks [i].free;
				blocks [i].stat = cache.blocks [i].stat;
				c += m_stat;
				
				/* initialize block */
				{
					long j;
					for (j = 0; j < cache.chunks; j++)
					{
						blocks [i].active [j] = cache.blocks [i].active [j];
						blocks [i].used [j] = cache.blocks [i].used [j];
					}
				}
			}
			else
			{
				blocks [i].data = data; data += cache.max_size;
				blocks [i].active = (UNIT **) c;
				blocks [i].used = (short *) (c + cache.chunks * sizeof (UNIT *));
				blocks [i].lock = 0;
				blocks [i].free = cache.chunks;
				blocks [i].stat = 0;
				c += m_stat;
				
				/* initialize block */
				{
					long j;
					for (j = 0; j < cache.chunks; j++)
					{
						blocks [i].active [j] = NULL;
						blocks [i].used [j] = 0;
					}
				}
			}
		}
	}
	
	/* free old information block array */
	kfree (cache.blocks);
	
	/* set up new information block array */
	cache.blocks = blocks;
	cache.count += count;
}

static long
bio_config (const ushort drv, const long config, const long mode)
{
	if (drv >= 0 && drv < NUM_DRIVES)
	{
		switch (config)
		{
			case BIO_WP:
			{
				DI *di = & bio_di [drv];
				
				if (mode == ASK) return BIO_WP_CHECK (di);
				if (mode)
					di->mode |= BIO_WP;
				else
					di->mode &= ~BIO_WP;
				break;
			}
			case BIO_WB:
			{
				DI *di = & bio_di [drv];
				
				if (mode == ASK) return BIO_WB_CHECK (di);
				if (mode)
					di->mode |= BIO_WB;
				else
					di->mode &= ~BIO_WB;
				break;
			}
			case BIO_MAX_BLOCK:
			{
				return cache.max_size;
			}
# ifdef BLOCK_IO_DEBUG
			case BIO_DEBUGLOG:
			{
				debug_mode = mode;
				break;
			}
			case BIO_DEBUG_T:
			{
				BIO_DEBUG_CACHE (());
				break;
			}
# endif
			default:
			{
				return EINVFN;
			}
		}
	}
	return E_OK;
}

/* END init & configuration */
/****************************************************************************/

/****************************************************************************/
/* BEGIN DI management */

static DI *
bio_get_di (ushort drv)
{
	UNIT **table;
	DI *di;
	
	if (drv < 0 || drv >= NUM_DRIVES) return NULL;
	
	di = & bio_di [drv];
	if (di->lock) return NULL;
	
	table = kmalloc (HASHSIZE * sizeof (*table));
	if (!table)
	{
		ALERT ("block_IO.c: kmalloc fail in bio_get_di (%c:), out of memory?", 'A'+drv);
		return NULL;
	}
	else
		zero ((void *) table, HASHSIZE * sizeof (*table));
	
	/*
	 * default values
	 */
	
	di->next	= NULL;
	di->table	= table;
	di->wb_queue	= NULL;
	
	di->major = 0;
	di->minor = 0;
	di->mode &= ~BIO_LRECNO;
	
	di->start = 0;
	di->size = 0;
	di->pssize = 0;
	di->lshift = 0;
	di->pshift = 0;
	
	/* default I/O routines */
	di->rwabs = rwabs_log;
	di->dskchng = dskchng_bios;
	
	di->valid = 0;
	di->id[0] = '\0';
	di->id[1] = '\0';
	di->id[2] = '\0';
	di->id[3] = '\0';
	di->key	= 0;
	
	/* ok, check for a valid XHDI drive, use it by default */
	if (XHDI_installed >= 0x110)
	{
		ulong pssize;
		ulong flags;
		long i;
		
		i = XHInqDev2 (drv, &(di->major), &(di->minor), &(di->start), NULL, &(di->size), di->id);
		if (i == E_OK)
			i = XHInqTarget2 (di->major, di->minor, &pssize, &flags, NULL, 0);
		
		if (i == E_OK)
		{
			di->rwabs = rwabs_xhdi;
			di->dskchng = dskchng_xhdi;
			
			bio_set_pshift (di, pssize);
			bio_set_lshift (di, pssize);
			
			if (flags & XH_TARGET_RESERVED)
			{
				return NULL;
			}
			
			di->mode |= BIO_LRECNO;
			di->valid = 1;
			di->lock = ENABLE;
			
			if (flags & XH_TARGET_REMOVABLE)
			{
				di->mode |= BIO_REMOVABLE;
			}
			
			return di;
		}
	}
	
	/* fall back to BIOS */
	{
		_BPB *bpb;
		ulong pssize;
		
		bpb = (_BPB *) getbpb (drv); /* this check implies FAT devices */
		if (bpb)
		{
			PUN_INFO *pun = get_pun ();
			if (pun && (drv >= 16 || !(pun->pun[drv] & 8)))
			{
				di->rwabs = rwabs_log_lrec;
				di->mode |= BIO_LRECNO;
			}
			
			pssize = 512; /* hmm? */
			
			bio_set_pshift (di, pssize);
			bio_set_lshift (di, pssize);
			
			di->valid = 1;
			di->lock = ENABLE;
			
			return di;
		}
	}
	
	return NULL;
}

static DI *
bio_res_di (ushort drv)
{
	UNIT **table;
	DI *di;
	
	if (drv < 0 || drv >= NUM_DRIVES) return NULL;
	
	di = & bio_di [drv];
	if (di->lock) return NULL;
	
	table = kmalloc (HASHSIZE * sizeof (*table));
	if (!table)
	{
		ALERT ("block_IO.c: kmalloc fail in bio_res_di (%c:), out of memory?", 'A'+drv);
		return NULL;
	}
	else
		zero ((void *) table, HASHSIZE * sizeof (*table));
	
	di->valid = 1;
	di->lock = ENABLE;
	
	return di;
}

static void
bio_free_di (DI *di)
{
	bio_invalidate (di);
	
	kfree (di->table);
	
	di->valid = 0;
	di->lock = DISABLE;
	
	if (di->mode & BIO_REMOVABLE)
	{
		bio_xhdi_unlock (di);
	}
}

/* END DI management */
/****************************************************************************/

/****************************************************************************/
/* BEGIN physical/logical calculation init */

static void
bio_set_pshift (DI *di, ulong physical)
{
	di->pssize = physical;
	di->pshift = 0;
	
	while (physical > 1)
	{
		di->pshift++;
		physical >>= 1;
	}
}

static void
bio_set_lshift (DI *di, ulong logical)
{
	if (logical < di->pssize)
		FATAL ("block_IO.c: bio_set_lshift: logical < pssize!");
	
	logical /= di->pssize;
	di->lshift = 0;
	
	while (logical > 1)
	{
		di->lshift++;
		logical >>= 1;
	}
}

/* END physical/logical calculation init */
/****************************************************************************/

/****************************************************************************/
/* BEGIN cached block I/O */

static UNIT *
bio_lookup (DI *di, long sector, long blocksize)
{
	return bio_hash_lookup (sector, blocksize, di->table);
}

static UNIT *
bio_getunit (DI *di, long sector, long blocksize)
{
	UNIT *u;
	
	BIO_DEBUG (("bio_getunit: entry (sector = %li, drv = %i, size = %li)\r\n", sector, di->drv, blocksize));
	
	u = bio_hash_lookup (sector, blocksize, di->table);
	if (!u)
	{
		u = bio_unit_get (di, sector, blocksize);
	}
	
	BIO_DEBUG (("bio_getunit: leave %s\r\n", u ? "ok" : "failure"));
	return u;
}

static UNIT *
bio_read (DI *di, long sector, long blocksize)
{
	UNIT *u;
	long i;
	
	BIO_DEBUG (("bio_read: entry (sector = %li, drv = %i, size = %li)\r\n", sector, di->drv, blocksize));
	
	u = bio_hash_lookup (sector, blocksize, di->table);
	if (!u)
	{
		u = bio_unit_get (di, sector, blocksize);
		if (u)
		{
			i = BIO_RWABS (di, 0, u->data, blocksize, sector);
			if (i)
			{
				BIO_DEBUG (("bio_read: rwabs fail (ret = %li)\r\n", i));
				
				bio_unit_remove (u);
				u = NULL;
			}
		}
	}
	
	BIO_DEBUG (("bio_read: leave %s\r\n", u ? "ok" : "failure"));
	return u;
}

static long
bio_write (UNIT *u)
{
	BIO_DEBUG (("bio_write: entry\r\n"));
	
	if (BIO_WB_CHECK (u->di))
	{
		bio_wbq_insert (u); /* always a list save operation */
		
		BIO_DEBUG (("bio_write: leave ok (writeback)\r\n"));
		return E_OK;
	}
	else
	{
		long r;
		
		r = BIO_RWABS (u->di, 1, u->data, u->size, u->sector);
		
		BIO_DEBUG (("bio_write: leave ok (writethrough = %li)\r\n", r));
		return r;
	}
}

static long
bio_l_read (DI *di, long sector, long blocks, long blocksize, void *buf)
{
	register const long incr = blocksize >> (di->pshift + di->lshift);
	
	register long tstart = sector;
	register long tblocks = 0;
	register long r = E_OK;
	
	BIO_DEBUG (("bio_l_read: entry (sector = %li, drv = %i, size = %li, incr = %li)\r\n", sector, di->drv, blocks * blocksize, incr));
	
	/* failure of the xfs */
	assert (incr > 0);
	
	while (blocks)
	{
		UNIT *u;
		
		u = bio_hash_lookup (sector, blocksize, di->table);
		if (u)
		{
			if (tblocks)
			{
				register const long size = tblocks * blocksize;
				
				r = BIO_RWABS (di, 0, buf, size, tstart);
				
				tblocks = 0;
				buf += size;
				
				if (r) break;
			}
			
			quickmovb (buf, u->data, blocksize);
			buf += blocksize;
		}
		else
		{
			if (tblocks == 0) tstart = sector;
			tblocks++;
		}
		
		sector += incr;
		blocks--;
	}
	
	if (tblocks)
		r = BIO_RWABS (di, 0, buf, tblocks * blocksize, tstart);
	
	if (r)
		BIO_DEBUG (("bio_l_read: leave failure, rwabs fail (ret = %li)\r\n", r));
	else
		BIO_DEBUG (("bio_l_read: leave ok\r\n"));
	
	return r;
}

static long
bio_large_write (DI *di, long sector, long size, void *buf)
{
	register UNIT **table = di->table;
	register ulong end = sector + (size >> (di->pshift + di->lshift));
	register long i;
	
	BIO_DEBUG (("bio_large_write: entry (sector = %li, drv = %i, size = %li\r\n", sector, di->drv, size));
	
	/* synchronisize cache with direct transfer
	 * -> remove entries in range: sector <= xxx < end
	 */
	for (i = 0; i < HASHSIZE; i++)
	{
		register UNIT *u;
		
		for (u = table [i]; u; u = u->next)
		{
			if (u->sector >= sector && u->sector < end)
			{
				/* overwritten by linear transfer;
				 * always a list save operation
				 */
				bio_wbq_remove (u);
				bio_unit_remove (u);
			}
		}
	}
	
	/* cache ready, do linear transfer */
	i = BIO_RWABS (di, 1, buf, size, sector);
	if (i)
		BIO_DEBUG (("bio_large_write: leave failure, rwabs fail (ret = %li)\r\n", i));
	else
		BIO_DEBUG (("bio_large_write: leave ok\r\n"));
	
	return i;
}

FASTFN long
bio_small_write (DI *di, long sector, long blocks, long blocksize, void *buf)
{
	register const long incr = blocksize >> (di->pshift + di->lshift);
	long r = E_OK;
	
	BIO_DEBUG (("bio_small_write: entry (sector = %li, drv = %i, blocks = %li\r\n", sector, di->drv, blocks));
	
	while (blocks)
	{
		UNIT *u;
		
		u = bio_getunit (di, sector, blocksize);
		if (!u)
		{
			/* not enough memory?
			 * -> fall back to direct transfer
			 */
			r = bio_large_write (di, sector, blocks * blocksize, buf);
			break;
		}
		
		quickmovb (u->data, buf, blocksize);
		buf += blocksize;
		
		bio_MARK_MODIFIED (&(bio), u);
		
		sector += incr;
		blocks--;
	}
	
	return r;
}

static long
bio_l_write (DI *di, long sector, long blocks, long blocksize, void *buf)
{
	register long size = blocks * blocksize;
	register long i;
	
	BIO_DEBUG (("bio_l_write: entry (sector = %li, drv = %i, size = %li\r\n", sector, di->drv, size));
	
	if (size > cache.max_size)
		i = bio_large_write (di, sector, size, buf);
	else
		i = bio_small_write (di, sector, blocks, blocksize, buf);
	
	return i;
}

/* END cached block I/O */
/****************************************************************************/

/****************************************************************************/
/* BEGIN optional feature */

static void
bio_pre_read (DI *di, long *sector, long blocks, long blocksize)
{
	UNUSED (di); UNUSED (sector); UNUSED (blocks); UNUSED (blocksize);
	
	BIO_DEBUG (("bio_pre_read: leave not implemented\r\n"));
}

/* END optional feature */
/****************************************************************************/

/****************************************************************************/
/* BEGIN synchronization */

static void
bio_lock (UNIT *u)
{
	register DI *di = u->di;
	
	BIO_DEBUG (("bio_lock: sector = %li, drv = %i\r\n", u->sector, u->di->drv));
	
	u->lock++;
	u->cbl->lock++;
	
	if (di->mode & BIO_REMOVABLE)
	{
		bio_xhdi_lock (di);
		di->lock++;
	}
}

static void
bio_unlock (UNIT *u)
{
	register DI *di = u->di;
	
	BIO_DEBUG (("bio_unlock: sector = %li, drv = %i\r\n", u->sector, u->di->drv));
	
	if (u->lock)
	{
		u->lock--;
		u->cbl->lock--;
	}
	
	if (di->mode & BIO_REMOVABLE)
	{
		di->lock--;
		
		if (di->lock == ENABLE && !di->wb_queue)
		{
			bio_xhdi_unlock (di);
		}
	}
}

/* END synchronization */
/****************************************************************************/

/****************************************************************************/
/* BEGIN update functions */

static void
bio_mark_modified (UNIT *u)
{
	register DI *di = u->di;
	
	bio_wbq_insert (u); /* always a list save operation */
	
	BIO_DEBUG (("bio_mark_modified: sector = %li, drv = %i\r\n", u->sector, u->di->drv));
	
	if (di->mode & BIO_REMOVABLE)
	{
		bio_xhdi_lock (di);
	}
}

static void
bio_sync_drv (DI *di)
{
	BIO_DEBUG (("bio_sync_drv: sync %c:\r\n", 'A' + di->drv));
	
	/* writeback queue */
	bio_wb_queue (di);
	
	BIO_DEBUG (("bio_sync_drv: wb_queue on %c: flushed.\r\n", 'A' + di->drv));
	
	if ((di->mode & BIO_REMOVABLE) && (di->lock == ENABLE))
	{
		bio_xhdi_unlock (di);
	}
}

void
bio_sync_all (void)
{
	register long i;
	
	for (i = 0; i < NUM_DRIVES; i++)
	{
		register DI *di = &(bio_di [i]);
		
		if (di->valid)
		{
			BIO_DEBUG (("bio_sync_all: sync %c:\r\n", 'A' + di->drv));
			
			/* writeback queue */
			bio_wb_queue (di);
			
			if ((di->mode & BIO_REMOVABLE) && (di->lock == ENABLE))
			{
				bio_xhdi_unlock (di);
			}
		}
	}
	
	BIO_DEBUG (("bio_sync_all: all wb_queues flushed.\r\n"));
}

/* END update functions */
/****************************************************************************/

/****************************************************************************/
/* BEGIN cache management */

static long
bio_validate (DI *di, long maxblocksize)
{
	/* at the moment only check the maximum blocksize */
	
	long r = E_OK;
	
	BIO_DEBUG (("bio_validate: entry (di->drv = %i, max = %li)\r\n", di->drv, maxblocksize));
	
	if (maxblocksize > cache.max_size)
	{
		r = ENSMEM;
	}
	
	return r;
}

static void
bio_invalidate (DI *di)
{
	/* invalid all cache units for drv */
	
	register UNIT **table = di->table;
	register long i;
	
	BIO_DEBUG (("bio_invalidate: entry (di->drv = %i)\r\n", di->drv));
	
	/* invalidate writeback queue */
	di->wb_queue = NULL;
	
	/* remove all hashtable entries */
	for (i = 0; i < HASHSIZE; i++)
	{
		register UNIT *u;
		
		for (u = table [i]; u; u = u->next)
		{
			if (u->dirty)
			{
				/* never writeback */
				u->dirty = 0;
				
				/* inform user */
				ALERT ("block_IO.c: bio_invalidate: cache unit not written back (%li, %li)!", u->sector, u->size);
			}
			
			/* remove from table */
			bio_unit_remove (u);
		}
	}
	
	BIO_DEBUG (("bio_invalidate: leave ok, all units are invalidated\r\n"));
}

/* END cache management */
/****************************************************************************/

/****************************************************************************/
/* BEGIN debug infos */

# ifdef BLOCK_IO_DEBUG

# include <stdarg.h>
# include "dosfile.h"
# define out_device 2 /* console */

static void
bio_debug (const char *fmt, ...)
{
	static char line [SPRINTF_MAX];
	if (debug_mode)
	{
		va_list args;
		int foo;
		FILEPTR *f;
		
		va_start (args, fmt);
		f = do_open(BIO_LOGFILE, (O_WRONLY|O_CREAT|O_APPEND), 0, (XATTR *)0);
		if (f)
		{
			foo = vksprintf (line, fmt, args);
			(void) (*f->dev->lseek)(f, 0, 2);
			(*f->dev->write)(f, line, _STRLEN (line));
			do_close (f);
		}
		else
		{
			foo = vksprintf (line, fmt, args);
			FORCE (line);
		}
		va_end (args);
	}
}

static void
bio_dump_cache (void)
{
	static char line [SPRINTF_MAX];
	short foo;
	FILEPTR *f;
	
	f = do_open (BIO_DUMPFILE, (O_WRONLY|O_CREAT|O_TRUNC), 0, (XATTR *) 0);
	if (f)
	{
		CBL *b = cache.blocks;
		long i;
		
		for (i = 0; i < NUM_DRIVES; i++)
		{
			register UNIT **table = bio_di [i].table;
			register long j;
			
			if (table)
			{
		
				(*f->dev->write)(f, "table:\r\n", 8);
				for (j = 0; j < HASHSIZE; j++)
				{
					UNIT *t = table [j];
					foo = ksprintf (line, "nr: %li\tptr = %lx", j, t);
					(*f->dev->write)(f, line, _STRLEN (line));
					for (; t; t = t->next)
					{
						foo = ksprintf (line, "\r\n\thnext = %lx\tlock = %i\tdirty = %i"
								"\tsector = %li\tdev = %i\r\n",
								t->next, t->lock, t->dirty, t->sector, t->di->drv
						);
						(*f->dev->write)(f, line, _STRLEN (line));
					}
					(*f->dev->write)(f, "\r\n", 2);
				}
			}
		}
		foo = ksprintf (line, "blocks:\t(max_size = %li)\r\n", cache.max_size);
		(*f->dev->write)(f, line, _STRLEN (line));
		for (i = 0; i < cache.count; i++)
		{
			long j;
			foo = ksprintf (line, "buffer = %lx, bufer->stat = %li, lock = %i, free = %i\r\n", b[i].data, b[i].stat, b[i].lock, b[i].free);
			(*f->dev->write)(f, line, _STRLEN (line));
			for (j = 0; j < cache.chunks; j++)
			{
				foo = ksprintf (line, "\tused = %i\tstat = %li\tactive = %lx\r\n", b[i].used[j], b[i].active[j] ? b[i].active[j]->stat : -1, b[i].active[j]);
				(*f->dev->write)(f, line, _STRLEN (line));
			}
			(*f->dev->write)(f, "\r\n", 2);
		}
		do_close (f);
	}
}
# endif

/* END debug infos */
/****************************************************************************/
