/* @(#)ssystem.c mint/freemint
 * $Id$
 * by jerry g geiger
 *  - jerry@zedat.fu-berlin.de  or jerry@merlin.abacus.de
 * almost completely rewritten by Draco, draco@mi.com.pl, Warszawa, 4.XII.1997. 
 * 
 * 

 General purpose: access vital system variables and constants without need
 to switch to Supervisor mode. Prototype:

 long ARGS_ON_STACK s_system(int mode, ulong arg1, ulong arg2)

 Available modes:

 mode		arg1	arg2	return	comment
 ----		----	----	------	-------
  -1		NULL	NULL	0	-1 if does not
 OSNAME		NULL	NULL	MiNT	$4d694e54 for MiNT
 OSXNAME	NULL	NULL	Free	$46726565 = FreeMiNT
 OSVERSION	NULL	NULL	vers	$010e0662 = 1.14.6 beta
 TOSHEADER	0L	NULL	vers	returns TOS version number in low word
 OS_BUILD_DATE	NULL	NULL	date	$040c07dc = 4.XII.1997.
 OS_BUILD_TIME	NULL	NULL	time	$04090a16 = Thu 9:10:22 am
 COMPILE_TYPE	NULL	NULL	code	code see below	
 FEATURES	NULL	NULL	fcode	fcode see below
 GETCOOKIE	tag	NULL	value	-1 if no such tag
 GETCOOKIE     	slot	NULL	tag	get tag from slot
 SETCOOKIE	tag	value	0	-1 on error
 GET_LVAL	addr	NULL	value	returns lword from specified addr
 GET_WVAL	addr	NULL	value	returns word from speified addr
 GET_BVAL	addr	NULL	value	returns byte from specified addr
 SET_LVAL	addr	value	0
 SET_WVAL	addr	value	0
 SET_BVAL	addr	value	0
 TIOCMGET	addr	NULL	short	dirty hack for MiNTLibs' ioctl.c

 Comments:

 mode -1 returns 0 if the kernel supports Ssystem(). Any programs going to
         use this call should use mode -1 prior to any further Ssystem calls.

 Kernel information functions:

 OSNAME	identifies operating system type.
 OSXNAME identifies exact kernel. A value of $46726565 identifies FreeMiNT
         if prior OSNAME call has identified MiNT.
 OSVERSION returns three parts of version number (major, minor and patchlevel)
         in three higher bytes. Low bytes contains $62 for beta or $00
         otherwise.
 OS_BUILD_DATE returns day, month and year of kernel compilation.
 OS_BUILD_TIME returns day of week (1 for monday etc), hour, minute and
         second.
 COMPILE_TYPE returns what CPU the kernel has been compiled for. The low word
         contains CPU identification in the same format as in cookie jar.
         The high word identifies FPU, currently NULL (kernel does not need
         FPU to run).
 FEATURES returns a state of kernel features:
 	bit 0 is set if memory protection is enabled
 	bit 1 is set if virtual memory manager is enabled
 	bits 2-3 contain security level number
 
 If Ssystem() exists, these calls above will always succeed.
 
 Cookie Jar management functions:
 
 GETCOOKIE returns cookie value for the specified tag or -1 if such a tag
 	does not exist. If the higher word of arg1 is zero, then GETCOOKIE
 	will assume it as a slot number, not a tag id. In this case a
 	corresponding tag id will be returned or -1 if the slot does not exist
 	(the cookie jar does not contain enough slots). If the whole arg1
 	is zero, the call will return a value for NULL cookie or -1 if the
 	cookie jar does not exist at all.
 
 SETCOOKIE requires euid root, returns EACCDN otherwise. If allowed, the
 	specified tag of the specified value will be placed in the cookie
 	jar and a zero will be returned. If the specified tag id already
 	exists, its value will be replaced with the new one. The call
 	returns ENSMEM if it was impossible to place a cookie. Specifically,
 	to avoid placing bad entries, if any byte of new tag id contains a
 	zero, such a tag will be refused.

 System variables maganement:

 GET_LVAL returns a long word fetched from the specified 16 bit address.
 GET_WVAL returns a word from the specified 16 bit address
 GET_BVAL returns a byte from the specified 16 bit address
 SET_LVAL places the specified longword at the specified address
 SET_WVAL places the specified word at the specified address
 SET_BVAL places the specified byte at the specified address
 TOSHEADER allows the first 4k of TOS code to be accessed for read.
	  Always a whole longword is returned. 

 All SET_xxxx functions require euid root, otherwise no action will be
 performed and EACCDN will be returned. Both stacked arguments are always
 longword regardless of the data size. The high word of the arg1 (address)
 must be always zero, otherwise the call returns EACCDN. Also, for LWORD
 and WORD operations, the bit 0 of the address will be reset to zero. This
 proves this call will not cause an address error on 68000 machines.
 
 GET_xxxx functions return 0 if the specified address is outside the allowed
 area ($08-$ffff actually).

 TIME/DATE functions not implemented for now (Jerry?).

 */

#include "version.h"
#include "cookie.h"
#include "mint.h"
#include "ssystem.h"
#include "buildtime.h"
#include "ioctl.h"

extern int vm_in_use; /* from main.c */
extern int no_mem_prot;
extern int secure_mode;

/* definitions in version.h */
static long MiNT_version = (ulong)MAJ_VERSION<<24|(ulong)MIN_VERSION<<16|(ulong)PAT_LEVEL<<8|(ulong)BETA_IDENT;

/* definitions in automatically created buildtime.h */
static long MiNT_date = (ulong)BUILD_DAY<<24|(ulong)BUILD_MONTH<<16|(ulong)BUILD_YEAR;
static long MiNT_time = (ulong)BUILD_DAY_OF_WEEK<<24|(ulong)BUILD_HOUR<<16|(ulong)BUILD_MIN<<8|(ulong)BUILD_SEC;

/* jerry:
 * find cookie cookiep->tag and return it's value in 
 * cookiep->value, return 0
 * return -1 if not found
 */

static long get_cookie(ulong arg1)
{
	COOKIE *cjar;
	int slotnum = 0;  /* number of already taken slots */

	cjar = *CJAR;     /* for compatibility. */

	/* If arg1 == 0, we return the value of NULL slot */

	if (arg1 == 0) {
		while(cjar->tag)
			cjar++;
		return(cjar->value);
	}

	/* if the high word of arg1 is zero, this is the slot number
	   to look at. The first slot is number 1. */

	if ((arg1 & 0xffff0000) == 0) {
		while(cjar->tag) {
			cjar++;
			slotnum++;
		}
		slotnum++;
		if (arg1 > slotnum)
			return(-1);
		cjar = *CJAR;
		slotnum = 1;
		while(slotnum != arg1) {
			slotnum++;
			cjar++;
		}
		return(cjar->tag);
	}	

	/* all other values of arg1 mean tag id to search for */
	
	while(cjar->tag) {
		if(cjar->tag == arg1)
			return(cjar->value);
		cjar++;
	}
	return(-1);
}


/* 
 * add cookie cookiep->tag to cookie list or change it's value
 * if already existing
 * 
 */

static long set_cookie(ulong arg1, ulong arg2)
{
	COOKIE * cjar; int n = 0;
	cjar = *CJAR;	/* for compatibility. 	*/

	/* 0x0000xxxx feature of GETCOOKIE may be confusing, so
	   prevent users from using slotnumber HERE :) */

	if ((arg1>>24 == 0)||((arg1&0x00ff0000) == 0)||((arg1&0x0000ff00) == 0)||((arg1&0x000000ff) == 0))
		return(-1);	/* Bad tag id */

	while(cjar->tag) {
		n++;
		if(cjar->tag == arg1) {
			cjar->value = arg2;
			return(0);
		}
		cjar++;
	}
	n++;
	if(n <  cjar->value) {
		n =  cjar->value;
		cjar->tag = arg1;
		cjar->value = arg2;
		cjar++;
		cjar->tag = 0L;
		cjar->value = n;
		return(0);
	}
	return(ENSMEM);	/* LIST exhausted :-)	*/
}

long ARGS_ON_STACK s_system(int mode, ulong arg1, ulong arg2)
{
	int isroot = 0;		/* check euid -> 1, ruid --> 2	*/
	unsigned long *lpointer;
	unsigned int  *wpointer;
	unsigned char *bpointer;
	unsigned long *sysbase;
	short *mfp;

	if(curproc->euid == 0)
		isroot = 1;
	if(curproc->ruid == 0)
		isroot = 2;

	if(mode == -1)	/* yes this call exists	*/
		return(0L);
	
	switch(mode) 
	{
	/* Kernel information */
		case OSNAME:
			return (0x4d694e54L) /* 'MiNT' */ ;break;
		case OSXNAME:
			return VERS_IDENT; break;
		case OSVERSION:
			return MiNT_version; break;
		case TOSHEADER:
			sysbase = *((long **)(0x4f2L));
			arg1 = arg1 & 0x00000ffe;	/* 4k allowed */
			return sysbase[arg1];
			break;
		case OS_BUILD_DATE:
			return MiNT_date; ;break;
		case OS_BUILD_TIME:
			return MiNT_time; ;break;
		case COMPILE_TYPE:
#ifdef CPU060
			return (0x0000003cL);	/* 060 kernel */
#endif
#ifdef CPU040
			return (0x00000028L);	/* 040 kernel */
#endif
#ifdef ONLY030
			return (0x0000001eL);	/* 030 kernel */
#endif
			break;			/* generic 68000 */
		case FEATURES:
			return (((!no_mem_prot)&0x01)|((vm_in_use<<1)&0x02)|(secure_mode<<2));
			break;
	/* GEMDOS variables */
		case GET_LVAL:
			arg1 = arg1 & 0xfffffffe;
			lpointer = (unsigned long *)arg1;
			if (arg1<0x08 || arg1>0xfffc)
				return (0L);
			return (*lpointer);
			break;
		case GET_WVAL:
			arg1 = arg1 & 0xfffffffe;
			wpointer = (unsigned int *)arg1;
			if (arg1<0x08 || arg1>0xfffe)
				return (0L);
			return (*wpointer);
			break;
		case GET_BVAL:
			bpointer = (unsigned char *)arg1;
			if (arg1<0x08 || arg1>0xffff)
				return (0L);
			return (*bpointer);
			break;
		case SET_LVAL:
			if (isroot == 0)
				return EACCDN;
			arg1 = arg1 & 0xfffffffe;
			lpointer = (unsigned long*)arg1;
			if (arg1<0x08 || arg1>0xfffc)
				return EACCDN;
			*lpointer = arg2;
			break;
		case SET_WVAL:
			if (isroot == 0)
				return EACCDN;
			arg1 = arg1 & 0xfffffffe;
			wpointer = (unsigned int*)arg1;
			if (arg1<0x08 || arg1>0xfffe)
				return EACCDN;
			*wpointer = arg2;
			break;
		case SET_BVAL:
			if (isroot == 0)
				return EACCDN;
			bpointer = (unsigned char*)arg1;
			if (arg1<0x08 || arg1>0xffff)
				return EACCDN;
			*bpointer = arg2;
			break;
	/* Cookie Jar functions */
		case GETCOOKIE:
			return get_cookie(arg1);
			break;
		case SETCOOKIE:
			if (isroot == 0)
				return EACCDN;
			return set_cookie(arg1, arg2);
			break;
	/* Hack (dirty one) for MiNTLibs */
		case TIOCMGET:
			mfp = (short *)arg1;
			return ((*mfp)&0x00ff);
			break;			
		default:
			return EINVFN;	/* invalid function number */
			break;						
	}
	return(0L);
}

/* eof ssystem.c	*/
