/* This file is part of 'minixfs' Copyright 1991,1992,1993 S.N.Henson */

# include "bitmap.h"
# include "global.h"


static long	alloc_bit	(ushort *buf, long num, long last);
static long	free_bit	(ushort *buf, long bitnum);

FASTFN int	bitcount	(register unsigned int wrd);


/* This routine is used for allocating both free inodes and free zones 
 * Search a bitmap for a zero , then return its bit number and change it
 * to a one ...... but without exceeding 'num' bits 
 */

static long
alloc_bit (ushort *buf, register long num, register long last)
{
	register const long end = num >> 4;
	register long i;
	register long j;
	register long k;
	
	k = 1;
	for (i = last >> 4; (i <= end) && (buf[i] == 65535U); i++);
	
	if (i > end)
		return 0;
	else
	{
		for(j = 0; j < 16; j++)
		{
			if (!(buf[i] & k))
			{
				long free;
				free = i * 16 + j;
				if (free >= num) return 0;
				buf[i] |= k;
				return free;
			}
			k <<= 1;
		}
	}
	
	ALERT ("minixfs: alloc_bit: This can't happen !");
	return 0;
}

/* zero a bit of a bitmap return 0 if already zero */

static long
free_bit (ushort *buf, register long bitnum)
{
	register long index = bitnum >> 4;
	register ushort bit = 1 << (bitnum & 15);
	register long ret;

	ret = buf[index] & bit;
	buf[index] &= ~bit;
	
	return ret;
}


FASTFN int
bitcount (register unsigned int wrd)
{
	register int count = 0;
	register ushort i;
	
	for (i = 1; i != 0; i <<= 1)
		if (wrd & i)
			count++;
		
	return count;
}

long
count_bits (ushort *buf, long num)
{
	register const long end = num >> 4; /* num/16 */
	register long count = 0;
	register long i;
	
	for (i = 0; i < end; i++)
	{
		if (buf[i] == 65535U) count += 16;
		else count += bitcount (buf[i]);
	}
	count += bitcount ((ushort)(buf[end] & ((1L << (num % 16)) -1L)));
	return count;
}


long
alloc_zone (int drive)
{
	long save;

	super_info *psblk = super_ptr[drive];

	if (!(save = alloc_bit (psblk->zbitmap, psblk->sblk.s_zones - psblk->sblk.s_firstdatazn + 1, psblk->zlast)))
		return 0;
	
	psblk->zdirty = 1; /* Mark zone bitmap as dirty */
	if (save > psblk->zlast)
		psblk->zlast = save;
	
	return (save + psblk->sblk.s_firstdatazn - 1);
}

/* Release a zone */
int
free_zone (long int zone, int drive)
{
	super_info *psblk = super_ptr[drive];
	long save, ret;
	
	save = zone + 1 - psblk->sblk.s_firstdatazn;
	ret = free_bit (psblk->zbitmap, save);
	
	/* Mark zone bitmap as dirty */
	psblk->zdirty = 1;
	if (save < psblk->zlast)
		psblk->zlast = save;
	
	if (!ret) ALERT ("Drive %d zone %ld freeing already free zone !", drive, zone);
	return ret;
}


ushort
alloc_inode (int drive)
{
	super_info *psblk = super_ptr[drive];
	ushort save;	

	if (!(save = alloc_bit (psblk->ibitmap, psblk->sblk.s_ninodes + 1L, psblk->ilast)))
		return 0;
	
	/* Mark inode bitmap as dirty */
	psblk->idirty = 1;
	if (save > psblk->ilast)
		psblk->ilast = save;
	
	return save;
}

/* Release an inode */
int
free_inode (unsigned int inum, int drive)
{
	super_info *psblk = super_ptr[drive];
	register long ret;
	
	ret = free_bit (psblk->ibitmap, inum);
	if (inum < psblk->ilast)
		psblk->ilast = inum;
	
	/* Mark inode bitmap as dirty */
	psblk->idirty = 1;
	
	if (!ret) ALERT ("Drive %d inode %d, freeing already free inode!", drive, inum);
	return ret;
}

