/* This file is part of 'minixfs' Copyright 1991,1992,1993 S.N.Henson */

# include "inode.h"
# include "global.h"

# include "bitmap.h"
# include "cache.h"
# include "zone.h"


/* Inode routines */

/*
 * Read an inode from the cache
 */

int
read_inode (unsigned int num, d_inode *rip, int drv)
{
	bufr *tmp;
	super_info *psblk = super_ptr[drv];
	num -= 1;
	
	tmp = cget_block (num / psblk->ipb + psblk->ioff, drv, &icache);
	*rip = tmp->binode[num % psblk->ipb];
	
	return 0;
}

/*
 * Write out an inode
 */

int
write_inode (unsigned int num, d_inode *rip, int drv)
{
	cache *tmp;
	super_info *psblk = super_ptr[drv];
	num -= 1;
	
	tmp = cache_get (num / psblk->ipb + psblk->ioff, drv, &icache);
	tmp->buffer->binode[num%psblk->ipb] = *rip;
	tmp->status = 2;
	
	return 0;
}

/* 'get' an inode from the cache, return a pointer to the inode and
 * if the pointer 'flag' is non-zero, return a pointer to the 'status' 
 * flag as well. Need two versions for this as well.
 */

d_inode
*get_inode (unsigned int inum, int drive, int **flag)
{
	cache *tmp;
	super_info *psblk = super_ptr[drive];
	inum -= 1;
	
	tmp = cache_get ((inum >> L_IPB2) + psblk->ioff, drive, &icache);

	if (flag)
		*flag = &tmp->status;

	return &tmp->buffer->binode[inum & (INODES_PER_BLOCK2-1)] ;
}

/* Truncate an inode to 'count' zones, this is used by unlink() as well as
 * (f)truncate() . Bit tricky this , we have to note which blocks to free,
 * and free indirection/double indirection blocks too but iff all the blocks
 * inside them are free too. We also need to keep count of how much to leave 
 * alone, sparse files complicate this a bit .... so do 2 fs versions ....
 */

void
trunc_inode (d_inode *rip, int drive, long int count, int zap)
/* count - number of blocks to leave */
/* zap - flag to alter inode */
{
	int i,j;
	bufr *tmp = NULL;
	char some,dirty;
	super_info *psblk = super_ptr[drive];
	cache_control *control = (IS_DIR ((*rip)) || IS_SYM ((*rip))) ? &syscache : &usrcache;
	cache *p, *q;
	
	/* Handle zones in inode first */
	if(count<psblk->dzpi)
	{
		for(i=count;i<psblk->dzpi;i++) {
			if(rip->i_zone[i]) {
			/* remove zones from cache too so they can't end up
			   in both ones when later reallocated... also may
			   save a few needless writes.	-nox */
				if( (p=in_cache(rip->i_zone[i],drive,control)) )
					p->status=0;
				free_zone(rip->i_zone[i],drive);
			}
			if(zap)rip->i_zone[i]=0;
		}
		count=0;
	}
	else count-=psblk->dzpi;
/* Handle indirect zone */
	if(count< psblk->zpind) {
		some=0;
		dirty=0;
		if(rip->i_zone[7]) {
			tmp=(q=cache_get(rip->i_zone[7],drive,&syscache))->buffer;
			for(i=0;i<psblk->zpind;i++) {
				if(PIND(tmp,i)) {
					if(count)some=1;
					else {
						if( (p=in_cache(PIND(tmp,i),drive,control)) )
							p->status=0;
						free_zone(PIND(tmp,i),drive);
						if(zap)PIND(tmp,i)=0;
						dirty=1;
					}
				}	
				if(count)count--;
			}
			if(!some) {
				q->status=0;
				free_zone(rip->i_zone[7],drive);
				if(zap)rip->i_zone[7]=0;
			}
			else if(dirty) 
# if 1
			/* a bit faster :) */
				q->status=2;
# else
				write_zone(rip->i_zone[7],tmp,drive,&syscache);
# endif
		}
	}
	else count-=psblk->zpind;
	/* Handle double indirect ... */
	if (count < (long) psblk->zpind * psblk->zpind) {
	    if(rip->i_zone[8]) {
		some=0;
		dirty=0;
		read_zone(rip->i_zone[8],&temp,drive,&syscache);
		for(i=0;i<psblk->zpind;i++) {
			if(IND(temp,i)) {
			    char lsome,ldirty; /* local some,dirty for inds */
			    lsome=0;
			    ldirty=0;
			    tmp=(q=cache_get(IND(temp,i),drive,&syscache))->buffer;
			    for(j=0;j<psblk->zpind;j++) {
				if(PIND(tmp,j)) {
					if(count) { 
						some=1;
						lsome=1;
					}
					else {
						if( (p=in_cache(PIND(tmp,j),drive,control)) )
							p->status=0;
						free_zone(PIND(tmp,j),drive);
						if(zap)PIND(tmp,j)=0;
						ldirty=1;
					}
				}
				if(count)count--;
			    }
			    if(!lsome) {
				q->status=0;
				free_zone(IND(temp,i),drive);
				if(zap)IND(temp,i)=0;
				dirty=1;
			    }
			    else if(ldirty)
# if 1
				q->status=2;
# else
				write_zone(IND(temp,i),tmp,drive,&syscache);
# endif
			}
			else 
			{
				if(count>=psblk->zpind)count-=psblk->zpind;
				else count=0;
			}
		}
		if(!some) {
			if( (p=in_cache(rip->i_zone[8],drive,&syscache)) )
				p->status=0;
			free_zone(rip->i_zone[8],drive);
			if(zap)rip->i_zone[8]=0;
		}
		else if(dirty)write_zone(rip->i_zone[8],&temp,drive,&syscache);
	    }
	}
	else
	    count -= (long) psblk->zpind * psblk->zpind;
	/* Handle triple indirect ... */
	if (rip->i_zone[9])
	  {
	    some = 0;
	    dirty = 0;
	    read_zone (rip->i_zone[9], &temp, drive, &syscache);
	    for (i = 0; i < psblk->zpind; i++)
	      {
		if (IND (temp, i))
		  {
		    bufr temp2;
		    char lsome, ldirty; /* local some, dirty for inds */
		    lsome = 0;
		    ldirty = 0;
		    read_zone (IND (temp, i), &temp2, drive, &syscache);
		    for (j = 0; j < psblk->zpind; j++)
		      {
			if (IND (temp2, j))
			  {
			    char lsome1, ldirty1;
			    int k;
			    bufr *tmp2;
			    cache *r;
			    lsome1 = 0;
			    ldirty1 = 0;
			    r = cache_get (IND (temp2, j), drive, &syscache);
			    tmp2 = r->buffer;
			    for (k = 0; k < psblk->zpind; k++)
			      {
				if (PIND ( tmp2, k))
				  {
				    if (count)
				      {
					some = 1;
					lsome = 1;
					lsome1 = 1;
				      }
				    else
				      {
					p = in_cache (PIND ( tmp2, k),
						      drive, control);
					if (p)
					  p->status = 0;
					free_zone (PIND ( tmp2, k),
						   drive);
					if (zap)
					  PIND ( tmp2, k) = 0;
					ldirty1 = 1;
				      }
				  }
				if (count)
				  count--;
			      }
			    if (!lsome1)
			      {
				r->status = 0;
				free_zone (IND (temp2, j), drive);
				if (zap)
				  PIND ( tmp, j) = 0;
				ldirty = 1;
			      }
			    else if (ldirty1)
			      r->status = 2;
			  }
			else
			  {
			    if (count >= psblk->zpind)
			      count -= psblk->zpind;
			    else
			      count = 0;
			  }
		      }
		    if (!lsome)
		      {
			q = in_cache(IND (temp, i), drive, &syscache);
			if (q)
			  q->status = 0;
			free_zone (IND (temp, i), drive);
			if (zap)
			  IND (temp, i) = 0;
			dirty = 1;
		      }
		    else if (ldirty)
		      write_zone(IND (temp, i), &temp2, drive, &syscache);
		  }
		else 
		  {
		    if (count >= (long) psblk->zpind * psblk->zpind)
		      count -= (long) psblk->zpind * psblk->zpind;
		    else
		      count = 0;
		  }
	      }
	    if (!some)
	      {
		p = in_cache (rip->i_zone[9], drive, &syscache);
		if (p)
		  p->status = 0;
		free_zone (rip->i_zone[9], drive);
		if (zap)
		  rip->i_zone[9] = 0;
	      }
	    else if (dirty)
	      write_zone (rip->i_zone[9], &temp, drive, &syscache);
	  }
}

/* Inode version of (f)truncate, truncates a file to size 'length'
 */

long
itruncate (unsigned int inum, int drive, long int length)
{
	long count;
	d_inode rip;
	FILEPTR *p;
	
	read_inode (inum,&rip,drive);
	
	/* Regulars only , clever directory compaction stuff later ... */
	if (!IS_REG (rip)) return EACCDN;
	
	/* If file smaller than 'length' nothing to do */
	if (rip.i_size <= length)
	{
# if 0
		rip.i_size = length;
# endif
		return 0;
	}
	count = (length + 1023) / 1024;
	
	/* invalidate f_cache zones */
	for (p = firstptr; p; p = p->next)
		if ((drive == p->fc.dev) && (inum == p->fc.index) && ((f_cache *) p->devinfo)->lzone > count)
			((f_cache *) p->devinfo)->lzone = 0;
	
	trunc_inode (&rip, drive, count, 1);
	rip.i_size = length;	
	
	write_inode (inum, &rip, drive);
	if (cache_mode) l_sync();
	
	return 0;
}


typedef struct inode
{
	struct inode *next;
	ulong	nr;
	ushort	drv;
	ushort	links;
	d_inode	rip;
	
} INODE;

/* cookie cache access */

FASTFN ulong	c_hash		(register long nr, register ushort drv);
FASTFN INODE *	c_lookup	(register long nr, register ushort drv);
FASTFN void	c_install	(register INODE *i);
FASTFN void	c_remove	(register INODE *i);

static INODE *	get_cookie	(register long nr, register ushort drv);

/*
 * cookie hash table
 */

# define INODE_CACHE 511
static INODE inodes  [INODE_CACHE];	/* ~ 40 kb like default before */
static INODE * table [INODE_CACHE];

FASTFN ulong
c_hash (register long nr, register ushort drv)
{
	register ulong hashval = nr & (((ulong) drv) << 16);
	return hashval % INODE_CACHE;
}

FASTFN INODE *
c_lookup (register long nr, register ushort drv)
{
	register INODE *c;
	
	for (c = table [c_hash (nr, drv)]; c != NULL; c = c->next)
	{
		if (nr == c->nr && drv == c->drv)
		{
			return c;
		}
	}
	
	return NULL;
}

FASTFN void
c_install (register INODE *c)
{
	register ulong hashval = c_hash (c->nr, c->drv);
	
	c->next = table [hashval];
	table [hashval] = c;
}

FASTFN void
c_remove (register INODE *c)
{
	register ulong hashval = c_hash (c->nr, c->drv);
	register INODE **temp = & table [hashval];
	long flag = 1;
	
	while (*temp)
	{
		if (*temp == c)
		{
			*temp = c->next;
			flag = 0;
			break;
		}
		temp = &(*temp)->next;
	}
	
	if (flag)
	{
		ALERT ("minixfs.c: remove from hashtable fail in: c_remove (addr = %lx, %li)", c, c->nr);
	}
	
	c->next = NULL;
	c->nr = 0;
	c->drv = 0;
	c->links = 0;
}

static INODE *
get_cookie (register long nr, register ushort drv)
{
	static long count = 0;
	register long i;
	for (i = 0; i < INODE_CACHE; i++)
	{
		count++; if (count == INODE_CACHE) count = 0;
		{
			register INODE *c = &(inodes[count]);
			if (c->links == 0)
			{
				c->nr = nr;
				c->drv = drv;
				c->links = 1;
				c_install (c);
				return c;
			}
		}
	}
	ALERT ("minixfs.c: get_cookie: no free INODE found for %li", nr);
	return NULL;
}
