#define exit(x) abort() 	      /* So bad assert() causes core dump */
/*

	Xsunclock

	Designed and implemented by John Walker in November of 1988.
	OpenWindows version implemented by John Walker in January of 1992.

	OpenWindows Version

    The algorithm used to calculate the position of the Sun is given in
    Chapter 18 of:

    "Astronomical  Formulae for Calculators" by Jean Meeus, Third Edition,
    Richmond: Willmann-Bell, 1985.  This book can be obtained from:

       Willmann-Bell
       P.O. Box 35025
       Richmond, VA  23235
       USA
       Phone: (804) 320-7016

    This program was written by:

	John Walker
	kelvin@fourmilab.ch
	http://www.fourmilab.ch/

    This  program is in the public domain: "Do what thou wilt shall be the
    whole of the law".  I'd appreciate  receiving  any  bug  fixes  and/or
    enhancements,  which  I'll  incorporate  in  future  versions  of  the
    program.  Please leave the original attribution information intact	so
    that credit and blame may be properly apportioned.

    Revision history:

	1.0  12/21/88  Initial SunView version.
	      8/24/89  Finally got around to submitting.

*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <sys/types.h>
#include <sys/timeb.h>
#include <assert.h>

#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/Shell.h>
#include <Xol/OpenLook.h>
#include <Xol/ControlAre.h>
#include <Xol/Stub.h>
#include <Xol/StaticText.h>
/* #include <Xol/RectButton.h> */
#include <Xol/FooterPane.h>
#include <Xol/Form.h>
#include <Xol/Menu.h>
#include <Xol/MenuButton.h>
#include <Xol/OblongButt.h>
#include <Xol/PopupWindo.h>
#include <Xol/ScrollingL.h>
#include <Xol/Caption.h>

#include "elements.h"

#define TITLE \
        "XSunclock   by John Walker, kelvin@fourmilab.ch   v2.1"

#include "xsunclock_icon.h"

static char map_icon_work[sizeof map_icon_bits];

static char icon_label_last[40] = "";

#define CLOSED_SECS	1	      /* Update interval when tool closed */
#define OPEN_SECS	1	      /* Update interval when open */

#ifndef max
#define max(a,b) ((a) > (b) ? (a) : (b))		  /* Maximum value */
#endif
#define abs(x) ((x) < 0 ? (-(x)) : x)			  /* Absolute value */
#define sgn(x) (((x) < 0) ? -1 : ((x) > 0 ? 1 : 0))	  /* Extract sign */
#define dtr(x) ((x) * (PI / 180.0))			  /* Degree->Radian */
#define rtd(x) ((x) / (PI / 180.0))			  /* Radian->Degree */
#define fixangle(a) ((a) - 360.0 * (floor((a) / 360.0)))  /* Fix angle	  */

#define V      (void)
#define EOS    '\0'

#define PI 3.14159265358979323846

#define TERMINC  100		      /* Circle segments for terminator */

#define PROJINT  (60 * 10)	      /* Frequency of seasonal recalculation
					 in seconds. */

static Boolean iconic = FALSE;	      /* Iconic ? */

static char *wdname[] = {	      /* Week day names */
        "Sun", "Mon", "Tue", "Wed", "Thu", "Fri", "Sat"
};

static Widget toplevel; 	      /* Top level application shell */
static Widget canvas;		      /* Canvas for drawing moon image */
static Widget ltime, utime;	      /* Local and universal time */
static Pixmap p, p1;		      /* Pixmaps swapped into icon */
static XGCValues gcv;		      /* Dummy values for graphics context */
static GC gc, xgc, igc, cgc, gc1;     /* Graphics context */
static XtAppContext app;	      /* Application context */
static XtIntervalId tid = NULL;       /* Active timeout interval ID */

static Dimension mheight = 320, mwidth = 640;	  /* Map height and width */

static int xdots, ydots;	      /* Screen size */
static int onoon = -1;		      /* Previous pixel longitude of noon */
static int renoon = TRUE;	      /* Recompute noon ? */
static short *wtab = NULL, *wtab1 = NULL, *wtabs;
static int wtabsize;		      /* Width table size */
static struct timeb tbtp;

static int fdate = FALSE,	      /* Show simulated date if true */
	   idir = 1,		      /* Simulated time increment/decrement */
	   animate = FALSE;	      /* Animate simulated time display ? */
static int lincr = 0;		      /* Animation mode time increment */
static long cctime;		      /* Time to display (real or simulated) */
static long lctime = 0; 	      /* Last full calculation time */

static struct satellite sat;	      /* Current satellite arguments */
static int satdisp = FALSE;	      /* True if satellite to be shown */
#ifdef TRAILS
static int needtrail = TRUE;	      /* Need satellite orbit trail ? */
#endif
static int lastsatx = -1, lastsaty;   /* Last satellite position */
static int tickoff = FALSE;	      /* Satellite blinker */

struct satce {			      /* Satellite chain entry */
    struct satce *snext;
    OlListToken token;
    struct satellite s;
};
static struct satce *satchain = NULL, /* Satellite list */
		    *satlast = NULL;
static int numsats = 0; 	      /* Number of satellites */
static OlListToken lasttoken = NULL;  /* Current satellite list token */
static void (*satTouchItem)();	      /* Satellite list touch item routine */
static Widget fsat;		      /* Satellite tracking status */
static Widget satIname, satIyear, satIlaunch, satIpiece, /* Satellite info */
	      satIincl, satIecc, satIrevday,
	      satIlat, satIlong, satIalt, satIorbit,
	      satIeclipse;

static char *menutags[] = {
    "Animate",
    "Step",
    "Real time",

    NULL
};
static Widget mbutton[3];	      /* Popup menu buttons */

static char *itags[] = {
    "Minute",
    "Hour",
    "Day",
    "Week",
    "Month",
    "Year",
    NULL
};

static char *dtags[] = {
    "Forward",
    "Backward",
    NULL
};

static long dvalues[] = {
    1,
    -1
};

/*  Forward functions  */

static void ringgg();
static double jtime(), gmst();
static void drawterm(), sunpos(), projillum(),
	    moveterm(), outdiff(), xspan(), ispan(), undrawsat();

extern long time();
extern int fprintf();
extern char *sprintf();
extern char *icongeom();
extern void mapdraw();

/*  ACTSAT  --	Activate display of satellites.  */

static void doactsat(s)
  struct satellite *s;
{
    struct satce *se;

    se = (struct satce *) malloc(sizeof(struct satce));
    se->snext = NULL;
    se->s = *s;
    if (satchain == NULL) {
	satchain = se;
    } else {
	satlast->snext = se;
    }
    satlast = se;
    numsats++;
}

static void actsat(arg)
  char *arg;
{
    FILE *fp = NULL;
    int saterr, satline;
    char *home = getenv("HOME");
    char homedir[512];

    /* Search for the satellite definition files in all the usual
       places:

	    $HOME/.Satellites
	    /usr/local/Satellites
	    <execution_path>/Satellites

    */

    if (home != NULL) {
	strcpy(homedir, home);
        strcat(homedir, "/.Satellites");
        fp = fopen(homedir, "r");
    }

    if (fp == NULL) {
        fp = fopen("/usr/local/Satellites", "r");
    }
    if (fp == NULL) {
	strcpy(homedir, arg);
        home = strrchr(homedir, '/');
	if (home == NULL) {
            strcpy(homedir, ".");
	} else {
	    *home = EOS;
	}
        strcat(homedir, "/Satellites");
        fp = fopen(homedir, "r");
    }
    satdisp = FALSE;
    numsats = 0;
    if (fp == NULL) {
	return;
    }

    insat(fp, doactsat, &saterr, &satline);
    if (saterr != 0) {
printf("Error %d processing line %d of satellite elements.\n", saterr, satline);
    }

    /* Sort the satellites into alphabetical order.  This is done with
       a bubble sort, which is excused only since there aren't  a  lot
       of  satellites  in the first place and the fact that we only do
       it once, after first loading the database. */

    if (numsats > 0) {
	int i, j;
	struct satce *se = satchain;
	struct satce **sat = (struct satce **) malloc(numsats *
				sizeof(struct satce *));

	for (i = 0; i < numsats; i++) {
	    sat[i] = se;
	    se = se->snext;
	}

	for (i = 0; i < numsats - 1; i++) {
	    for (j = i + 1; j < numsats; j++) {
		if (strcasecmp(sat[i]->s.satname, sat[j]->s.satname) > 0) {
		    se = sat[j];
		    sat[j] = sat[i];
		    sat[i] = se;
		}
	    }
	}
	satchain = sat[0];
	for (i = 0; i < numsats - 1; i++) {
	    sat[i]->snext = sat[i + 1];
	}
	sat[numsats - 1]->snext = NULL;
	free(sat);
    }
    fclose(fp);
}

/*  PICKSAT  --  User has chosen satellite.  */

static void picksat(w, client_data, call_data)
  Widget w;
  XtPointer client_data, call_data;
{
    OlListToken token = (OlListToken) call_data;
    OlListItem *newitem = OlListItemPointer(token), *lastitem;
    struct satce *se = (struct satce *) newitem->user_data;
    char ebuf[40];

    if (lasttoken == token) {	      /* Deselect current choice */
	newitem->attr &= ~OL_LIST_ATTR_CURRENT;
	(*satTouchItem)(w, token);
	undrawsat();
	lasttoken = NULL;
    } else {

	if (lasttoken) {
	    lastitem = OlListItemPointer(lasttoken);

	    if (lastitem->attr & OL_LIST_ATTR_CURRENT) {
		lastitem->attr &= ~OL_LIST_ATTR_CURRENT;
	    }
	    undrawsat();
	    (*satTouchItem)(w, lasttoken);
	}
	newitem->attr |= OL_LIST_ATTR_CURRENT;
	sat = se->s;
	satdisp = TRUE;

	(*satTouchItem)(w, token);
	lasttoken = token;
    }

    if (satdisp) {
        sprintf(ebuf, "Tracking %s", sat.satname);
    } else {
	ebuf[0] = EOS;
    }
    XtVaSetValues(fsat,
	XtNstring, (XtArgVal) ebuf,
	NULL);
#define setSatLabel(sname, svalue) XtVaSetValues(sname, \
            XtNstring, (XtArgVal) (satdisp ? svalue : ""), NULL)
    setSatLabel(satIname, sat.satname);
    sprintf(ebuf, "%d", sat.intdes.launchyear + 1900 +
			(sat.intdes.launchyear < 57 ? 100 : 0));
    setSatLabel(satIyear, ebuf);
    sprintf(ebuf, "%d", sat.intdes.launchno);
    setSatLabel(satIlaunch, ebuf);
    setSatLabel(satIpiece, sat.intdes.pieceno);
    sprintf(ebuf, "%.2f", sat.inclination);
    setSatLabel(satIincl, ebuf);
    sprintf(ebuf, "%.2f", sat.eccentricity);
    setSatLabel(satIecc, ebuf);
    sprintf(ebuf, "%.2f", sat.meanmotion);
    setSatLabel(satIrevday, ebuf);

    setSatLabel(satIlat, "");
    setSatLabel(satIlong, "");
    setSatLabel(satIalt, "");
    setSatLabel(satIorbit, "");
    setSatLabel(satIeclipse, "");
}

/*  SATPOP  --	Create satellite popup widget.	*/

static Widget satpop(parent)
  Widget parent;
{
    Widget popup, upper, lower, footer, con, sl, t1;
    Arg wargs[10];
    OlListToken (*additem)();
    void (*deleteitem)(), (*updateview)(), (*viewitem)();
    OlListItem item;
    struct satce *se;
    short n = 0;

    popup = XtVaCreatePopupShell("satpop", popupWindowShellWidgetClass, parent,
        XtNtitle, "Satellite Tracking",
	NULL);
    XtVaGetValues(popup,
	XtNupperControlArea, &upper,
	XtNlowerControlArea, &lower,
	XtNfooterPanel,      &footer,
	NULL);

    XtSetArg(wargs[0], XtNviewHeight, 10);
    XtSetArg(wargs[1], XtNselectable, FALSE);
    sl = XtCreateManagedWidget("satlist", scrollingListWidgetClass,
	lower, wargs, 2);

    XtSetArg(wargs[0], XtNapplAddItem, &additem);
    XtSetArg(wargs[1], XtNapplUpdateView, &updateview);
    XtSetArg(wargs[2], XtNapplTouchItem, &satTouchItem);
    XtGetValues(sl, wargs, 3);

    XtAddCallback(sl, XtNuserMakeCurrent, picksat, NULL);

    con = XtCreateManagedWidget("control", controlAreaWidgetClass,
				lower, NULL, 0);
    XtVaSetValues(con,
	XtNlayoutType, (XtArgVal) OL_FIXEDCOLS,
	XtNcenter, (XtArgVal) FALSE,
	XtNalignCaptions, (XtArgVal) TRUE,
	XtNvSpace, (XtArgVal) 0,
	XtNmeasure, (XtArgVal) 1,
	NULL);

#define creSatLabel(sname, slabel) t1 = XtCreateManagedWidget(slabel, \
            captionWidgetClass, con, NULL, 0);                        \
            sname = XtVaCreateManagedWidget("sattext",                \
            staticTextWidgetClass, t1, XtNstrip, FALSE,               \
            XtNstring, "                        ", NULL)
    creSatLabel(satIname,    "Satellite:");
    creSatLabel(satIyear,    "Launch year:");
    creSatLabel(satIlaunch,  "Launch number:");
    creSatLabel(satIpiece,   "Piece:");
    creSatLabel(satIincl,    "Inclination:");
    creSatLabel(satIecc,     "Eccentricity:");
    creSatLabel(satIrevday,  "Rev/day:");
    creSatLabel(t1,          "");

    creSatLabel(satIlat,     "Latitude:");
    creSatLabel(satIlong,    "Longitude:");
    creSatLabel(satIalt,     "Altitude:");
    creSatLabel(satIorbit,   "Orbit number:");
    creSatLabel(satIeclipse, "Eclipsed?");
#undef creSatLabel

    (*updateview)(sl, FALSE);

    se = satchain;
    while (se != NULL) {
	item.label_type = OL_STRING;
	item.attr = n++;
	item.label = XtNewString(se->s.satname);
	item.mnemonic = NULL;
	item.user_data = (XtPointer) se;
	se->token = (*additem)(sl, 0, 0, item);

	se = se->snext;
    }
    (*updateview)(sl, TRUE);

    return popup;
}

/*  POPSAT  --	Pop up the satellite window when selected from the menu.  */

static void popsat(w, client_data, call_data)
  Widget w;
  XtPointer client_data, call_data;
{
    Widget popup = (Widget) client_data;

    XtPopup(popup, XtGrabNone);
}

/*  UPDTIME  --  Update time based on current increment and mode.  */

static void updtime()
{
    struct tm *ct;

    switch (lincr) {
	case 0: 		  /* Minute */
	    cctime += 60 * idir;
	    break;

	case 1: 		  /* Hour */
	    cctime += 60 * 60 * idir;
	    break;

	case 2: 		  /* Day */
	    cctime += (24L * 60L * 60L) * idir;
	    break;

	case 3: 		  /* Week */
	    cctime += (7L * 24L * 60L * 60L) * idir;
	    break;

	case 4: 		  /* Month */
	    ct = gmtime(&cctime);
	    if (idir > 0 && ct->tm_mon >= 11) {
		ct->tm_mon = 0;
		ct->tm_year++;
		if (ct->tm_year > 136) {
		    ct->tm_year = 70;
		    lctime = 0;
		}
	    } else if (idir < 0 && ct->tm_mon <= 0) {
		ct->tm_mon = 11;
		ct->tm_year--;
		if (ct->tm_year < 70) {
		    ct->tm_year = 136;
		    lctime = 0;
		}
	    } else {
		ct->tm_mon += idir;
	    }
	    cctime = timegm(ct);
	    break;

	case 5: 		  /* Year */
	    ct = gmtime(&cctime);
	    ct->tm_year += idir;
	    if (ct->tm_year > 136) {
		ct->tm_year = 70;
		lctime = 0;
	    } else if (ct->tm_year < 70) {
		ct->tm_year = 136;
		lctime = 0;
	    }
	    cctime = timegm(ct);
	    break;
    }
    if (cctime < 0) {
	cctime = lctime = 0;
    }
}

/*  DRAWSAT  --  Draw satellite icon at given latitude and longitude. */

static void drawsat(lat, lon)
  double lat, lon;
{
    int ilat, ilon;

#define SatSize 10
    assert(!iconic);
    if (tickoff) {
	if (lastsatx >= 0) {
	    XFillArc(XtDisplay(canvas), XtWindow(canvas), xgc,
		lastsatx - (SatSize / 2 + 1), lastsaty - (SatSize / 2 + 1),
		SatSize + 1, SatSize + 1, 0, 64 * 360);
	    lastsatx = -1;
	}
    } else {
	if (lon > 180) {
	    lon -= 360;
	} else if (lon < -180) {
	    lon += 360;
	}
	ilat = ydots - (lat + 90) * (ydots / 180.0);
	ilon = xdots - (180 + lon) * (xdots / 360.0);
	XFillArc(XtDisplay(canvas), XtWindow(canvas), xgc,
	    ilon - (SatSize / 2 + 1), ilat - (SatSize / 2 + 1),
	    SatSize + 1, SatSize + 1, 0, 64 * 360);
	lastsatx = ilon;
	lastsaty = ilat;
    }
    tickoff = !tickoff;
}

/*  UNDRAWSAT --  Clear satellite if displayed.  */

static void undrawsat()
{
    if (satdisp && tickoff) {
	drawsat(0.0, 0.0);
    }
    assert(tickoff == FALSE);
    assert(lastsatx < 0);
    satdisp = FALSE;
}

#ifdef TRAILS

/*  DRAWTRAIL  --  Draw segment of satellite orbit trail.  */

static void drawtrail(trailphase, now, lat, lon)
  int *trailphase;
  double *now;
  double lat, lon;
{
    int ilat, ilon;
    static int lilat, lilon;

    if (lon > 180) {
	lon -= 360;
    } else if (lon < -180) {
	lon += 360;
    }
    ilat = ydots - (lat + 90) * (ydots / 180.0);
    ilon = xdots - (180 + lon) * (xdots / 360.0);
    if (abs(*trailphase) != 2) {
	XDrawLine(XtDisplay(canvas), XtWindow(canvas), xgc,
	    lilon, lilat, ilon, ilat);
    }
    lilat = ilat;
    lilon = ilon;
    if (abs(*trailphase) == 2) {
	*trailphase /= 2;
    }
    *now += *trailphase * (0.01 / sat.meanmotion);
}
#endif


/*  UPDSAT  --	Update satellite position.  */

static void updsat(ct, dotrail)
  struct tm *ct;
  int dotrail;
{
    double now, smaxis, raanp, perigeep, reforbit, epday,
	   avgmotion, currmotion, currorbit, meana, truea,
	   satx, saty, satz, radius, satvx, satvy, satvz,
	   ssplat, ssplong, sspheight, inow;
    long iorbitnum, liorbitnum;
    char ebuf[40];
    static int initialised = FALSE;
    int trailphase = 2;

    extern double Kepler();
    extern long GetDayNum();

    assert(satdisp);
    now = GetDayNum(ct->tm_year + 1900, ct->tm_mon + 1, ct->tm_mday);
    now += (ct->tm_hour / 24.0) + (ct->tm_min / (24.0 * 60)) +
	   (ct->tm_sec / (24.0 * 60.0 * 60.0));
    if (!initialised) {
	InitOrbitRoutines(now);
	initialised = TRUE;
    }
    epday = sat.epochday + GetDayNum(sat.epochyear, 1, 0);
    smaxis = 331.25 * exp(2.0 * log((24 * 60.0) / sat.meanmotion) / 3.0);
    GetPrecession(smaxis, sat.eccentricity, dtr(sat.inclination),
	&raanp, &perigeep);
    reforbit = dtr(sat.meananomaly) / (2 * PI) + sat.revno;
    inow = now;

    while (TRUE) {
	avgmotion = sat.meanmotion + (now - epday) * (sat.meanmotiond1 / 2);
	currmotion = sat.meanmotion + (now - epday) * sat.meanmotiond1;
	smaxis = 331.25 * exp(2.0 * log((24 * 60.0) / currmotion) / 3.0);
	currorbit = reforbit + (now - epday) * avgmotion;
	iorbitnum = currorbit;

	meana = (currorbit - iorbitnum) * (2 * PI);
	truea = Kepler(meana, sat.eccentricity);

	GetSatPosition(epday, dtr(sat.rascendnode), dtr(sat.argperigee),
	    smaxis, dtr(sat.inclination), sat.eccentricity, raanp, perigeep,
	    now, truea, &satx, &saty, &satz, &radius, &satvx, &satvy, &satvz);

	GetSubSatPoint(satx, saty, satz, now, &ssplat, &ssplong, &sspheight);
	if (!dotrail) {
	    drawsat(rtd(ssplat), rtd(ssplong));

            sprintf(ebuf, "%.2f %c", abs(rtd(ssplat)), ssplat < 0 ? 'S' : 'N');
	    setSatLabel(satIlat, ebuf);
            sprintf(ebuf, "%.2f %c", rtd(ssplong) > 180 ?
					 360 - rtd(ssplong) : rtd(ssplong),
                                     rtd(ssplong) > 180 ? 'E' : 'W');
	    setSatLabel(satIlong, ebuf);
            sprintf(ebuf, "%.2f km", sspheight);
	    setSatLabel(satIalt, ebuf);
            sprintf(ebuf, "%.2f", currorbit);
	    setSatLabel(satIorbit, ebuf);
            setSatLabel(satIeclipse, Eclipsed(satx, saty, satz, radius, now) ? "Yes" : "");
	    break;
	}
#ifdef TRAILS
	if (abs(trailphase) == 2) {
	    liorbitnum = iorbitnum;
	}
	drawtrail(&trailphase, &now, rtd(ssplat), rtd(ssplong));
	if (liorbitnum != iorbitnum) {
	    if (trailphase < 0) {
		break;
	    }
	    now = inow;
	    trailphase = -2;
	}
#endif
    }
}

/*  UPDIMAGE  --  Update current displayed image.  */

static void updimage(istimer)
  int istimer;
{
    int xl;
    struct tm *ct;
    char tbuf[80];
    double jt, sunra, sundec, sunrv, sunlong, gt;
    struct tm lt;
    static int lisec = 61;	      /* Last iconic seconds */

if (!XtIsRealized(canvas)) {
    return;
}

    if (iconic) {
	xdots = map_icon_width;
	ydots = map_icon_height;
    } else {
	xdots = mwidth;
	ydots = mheight;
    }

    /* For  some  crazy reason, you can  actually resize the window so
       that the height of the graphics area goes negative.  Of course,
       since  a  Dimension is declared as an unsigned short, you can't
       even test for a negative  number  to  detect  this  case.   The
       following piece of adhocracy detects the negative dimension and
       refuses to proceed further trying to  draw  in  a  void	window
       which  will  almost  certainly  bring  us  to perdition further
       along. */

    if (xdots > 60000 || ydots > 60000) {
printf("Idiot negative sized window!!!\n");
	return;
    }

    /* If this is a full repaint of the window, force complete
       recalculation. */

    if (!istimer) {
	lctime = 0;
	onoon = -1;
	lastsatx = -1;
	lisec = 61;
    }

    if (fdate) {
	if (animate) {
	    updtime();
	}
    } else {
	V time(&cctime);
    }
    lt = *localtime(&cctime);

    /* Special  case  to  reduce overhead  whilst iconic: if we're
       only showing the icon, update the  display  only  once  per
       minute,	detected  by  the  fact  that  the current seconds
       reading is less than that of the  last  update.	 The  icon
       shows  only  hours  and	minutes, and is sufficiently small
       that once-a-minute updates are plenty to keep  the  picture
       in sync.  */

    if (iconic && !fdate && (lt.tm_sec > lisec))
	return;

    ct = gmtime(&cctime);

    jt = jtime(ct);
    sunpos(jt, FALSE, &sunra, &sundec, &sunrv, &sunlong);
    gt = gmst(jt);

    /* Projecting the illumination curve  for the current seasonal
       instant is costly.  If we're running in real time, only  do
       it every PROJINT seconds.  */

assert(cctime >= 0);
    if (fdate || !istimer ||
	(cctime < lctime) || ((cctime - lctime) > PROJINT)) {
	projillum(wtab, xdots, ydots, sundec);
	wtabs = wtab;
	wtab = wtab1;
	wtab1 = wtabs;
	lctime = cctime;
    }

    sunlong = fixangle(180.0 + (sunra - (gt * 15)));
    xl = sunlong * (xdots / 360.0);

    if (satdisp && !iconic) {
#ifdef TRAILS
	if (needtrail) {
	    updsat(ct, TRUE);
	    needtrail = FALSE;
	}
#endif
	updsat(ct, FALSE);
    }

    /* If the subsolar point has moved at least one pixel, update
       the illuminated area on the screen.  */

    if (fdate || !istimer || renoon || (onoon != xl)) {
	moveterm(iconic ? ispan : xspan, wtab1, xl, wtab, onoon, xdots, ydots);
	onoon = xl;
	renoon = FALSE;

	if (iconic) {
	    Pixmap np;

	    /* The following silly little dance where we double buffer
	       the icon in two pixmaps and swap them  back  and  forth
               has  nothing  to  do  with  efficiency.  Rather, it's a
               work-around for the fact that X doesn't update the icon
	       unless  you  pass  it it different address for the icon
	       pixmap than the one it had before. */

	    np = XCreateBitmapFromData(XtDisplay(toplevel),
				       RootWindowOfScreen(XtScreen(toplevel)),
				       map_icon_work, map_icon_width,
				       map_icon_height);
	    XCopyPlane(XtDisplay(toplevel), np, p1, gc1, 0, 0,
		map_icon_width, map_icon_height, 0, 0, 1);
	    XFreePixmap(XtDisplay(toplevel), np);
	    XtVaSetValues(toplevel,
		XtNiconPixmap, (XtArgVal) p1,
		NULL);

	    np = p;
	    p = p1;
	    p1 = np;
	}
    }

    if (iconic) {

	/* Display time in icon label */

        V sprintf(tbuf, "%d:%02d %s %s %d/%d",
	    lt.tm_hour, lt.tm_min,
#ifdef USE_TIMEZONE
	    timezone(tbtp.timezone, lt.tm_isdst),
#else
	    lt.tm_zone,
#endif
	    wdname[lt.tm_wday],
	    lt.tm_mon + 1, lt.tm_mday);
	if (strcmp(tbuf, icon_label_last) != 0) {
	    strcpy(icon_label_last, tbuf);
	    XtVaSetValues(toplevel,
		XtNiconName, tbuf,
		NULL);
	}
	lisec = lt.tm_sec;

    } else {

	/* Display time in open window */

        V sprintf(tbuf, " %02d:%02d:%02d %s %s %02d/%02d/%02d",
	    lt.tm_hour, lt.tm_min, lt.tm_sec,
#ifdef USE_TIMEZONE
	    timezone(tbtp.timezone, lt.tm_isdst),
#else
	    lt.tm_zone,
#endif
	    wdname[lt.tm_wday],
	    lt.tm_mon + 1, lt.tm_mday, (lt.tm_year % 100));

	XtVaSetValues(ltime,
	    XtNstring, (XtArgVal) tbuf,
	    NULL);

        V sprintf(tbuf, " %02d:%02d:%02d UTC %02d/%02d/%02d",
	    ct->tm_hour, ct->tm_min, ct->tm_sec,
	    ct->tm_mon + 1, ct->tm_mday, (ct->tm_year % 100));

	XtVaSetValues(utime,
	    XtNstring, (XtArgVal) tbuf,
	    NULL);
    }
}

/*  MAPVEC  --	Draw a vector in the map window.  */

static void mapvec(x1, y1, x2, y2)
  int x1, y1, x2, y2;
{
    XDrawLine(XtDisplay(canvas), XtWindow(canvas), igc,
	x1, y1, x2, y2);
}

/*  EXPOSE  --	Graphics area repaint procedure.  */

/* ARGSUSED */
static void expose(w, xevent, region)
  Widget w;
  XEvent *xevent;
  Region *region;
{
    int i;

    for (i = 0; i < wtabsize; i++) {
	wtab[i] = wtab1[i] = -1;
    }
    XFillRectangle(XtDisplay(w), XtWindow(w), gc, 0, 0, mwidth, mheight);

    mapdraw(mwidth, mheight, mapvec);
    updimage(FALSE);
}

/*  RESIZE  --	Graphics area resize procedure.  */

/* ARGSUSED */
static void resize(w)
  Widget w;
{
    int i;
    Arg wargs[10];

    XtSetArg(wargs[0], XtNheight, &mheight);
    XtSetArg(wargs[1], XtNwidth, &mwidth);
    XtGetValues(canvas, wargs, 2);

    wtabsize = max(mheight, map_icon_height);

    if (wtab != NULL) {
	free(wtab);
	free(wtab1);
    }
    wtab = (short *) malloc((unsigned int) wtabsize * sizeof(short));
    wtab1 = (short *) malloc((unsigned int) wtabsize * sizeof(short));
    for (i = 0; i < wtabsize; i++) {
	wtab[i] = wtab1[i] = -1;
    }
    xdots = mwidth;
    ydots = mheight;
}

/*  EPROC  --  Intercept expose and UnmapNotify events to discover when
	       the window is iconised and restored to full size.  There
               must be a better way to do this, but I'll be darned if I
	       can figger it out!  */

/* ARGSUSED */
static void eproc(w, client_data, event, ctd)
  Widget w;
  XtPointer client_data;
  XEvent *event;
  Boolean *ctd;
{
    Boolean liconic = iconic;

    if (event->type == Expose) {
	iconic = FALSE;
    } else if (event->type == UnmapNotify) {
	iconic = TRUE;
    }

    /* If iconic  state  changed  and  there's  an  unexpired  timeout
       pending, cancel the timeout and invoke ringgg() to register the
       correct timeout for the new visibility state. */

    if (liconic != iconic) {
	if (iconic) {
	    int i;

	    memcpy(map_icon_work, map_icon_bits, sizeof map_icon_bits);
	    for (i = 0; i < wtabsize; i++) {
		wtab[i] = wtab1[i] = -1;
	    }
	    updimage(FALSE);
	}
	ringgg(toplevel, NULL);
    }
}

/*  BUSYBUTTON	--  Set busy status of a menu button.  */

static void busybutton(n, status)
  int n, status;
{
    XtVaSetValues(mbutton[n],
	XtNbusy, (XtArgVal) status,
	NULL);
}

/*  PUSHED  --	Handle button press in menu.  */

/* ARGSUSED */
static void pushed(w, client_data, call_data)
  Widget w;
  XtPointer client_data, call_data;
{
    int i;

    switch ((int) client_data) {

	case 1: 		      /* Animate */
	    animate = fdate = TRUE;
	    ringgg(toplevel, NULL);
/*	    updimage(TRUE); */
	    break;

	case 2: 		      /* Step */
	    animate = FALSE;
	    fdate = TRUE;
	    updtime();
	    updimage(TRUE);
	    client_data = (XtPointer) 100;
	    break;

	case 3: 		      /* Real time */
	    animate = fdate = FALSE;
	    lctime = 0; 	      /* Force recalculation */
	    renoon = TRUE;	      /* Mark noon position invalid */
	    updimage(TRUE);
	    break;
    }
    for (i = 1; i <= 3; i++) {
	busybutton(i - 1, i == ((int) client_data));
    }
}

/*  INTSET  --	Set interval of increment.  */

/* ARGSUSED */
static void intset(w, client_data, call_data)
  Widget w;
  XtPointer client_data, call_data;
{
    lincr = (int) client_data;
}

/*  DIRSET  --	Set direction of increment.  */

/* ARGSUSED */
static void dirset(w, client_data, call_data)
  Widget w;
  XtPointer client_data, call_data;
{
    idir = (long) client_data;
}

/*  MAIN  --  Main program.  */

int main(argc, argv)
  int argc;
  char *argv[];
{
    int c;
    Widget dirmenu, dirpane, foot, form, interval, ipane, menu, pane;
    char *igerr;
    Arg wargs[10];

    OlToolkitInitialize((XtPointer) NULL);
    toplevel = XtAppInitialize(&app, "XSunclock",
			       (XrmOptionDescList) NULL,
			       0, &argc, argv, (String *) NULL,
			       (ArgList) NULL, 0);
    p = XCreatePixmap(XtDisplay(toplevel),
		      RootWindowOfScreen(XtScreen(toplevel)),
		      map_icon_width,
		      map_icon_height,
		      1);
    p1 = XCreatePixmap(XtDisplay(toplevel),
		       RootWindowOfScreen(XtScreen(toplevel)),
		       map_icon_width,
		       map_icon_height,
		       1);

    XtVaSetValues(toplevel,
	XtNtitle, (XtArgVal) TITLE,
	XtNiconPixmap, (XtArgVal) p,
	NULL);

    igerr = icongeom(toplevel, &argc, argv);
    if (igerr != NULL) {
        fprintf(stderr, "Error: %s\n", igerr);
	return 2;
    }

    XtAddEventHandler(toplevel, ExposureMask | StructureNotifyMask,
		      TRUE, eproc, NULL);

    foot = XtCreateManagedWidget("footer", footerPanelWidgetClass,
	toplevel, NULL, 0);

    canvas = XtCreateManagedWidget("canvas", stubWidgetClass, foot,
	NULL, 0);

    XtVaSetValues(canvas,
	XtNheight, 320,
	XtNwidth, 640,
	XtNexpose, expose,
	XtNresize, resize,
	NULL);

    form = XtCreateManagedWidget("form", formWidgetClass, foot, NULL, 0);
    XtVaSetValues(form,
	XtNxAddWidth, (XtArgVal) TRUE,
	NULL);

    ltime = XtCreateManagedWidget("ltime", staticTextWidgetClass, form,
	NULL, 0);

    XtVaSetValues(ltime,
        XtNstring, (XtArgVal) "Local time",
	NULL);

    fsat = XtCreateManagedWidget("fsat", staticTextWidgetClass, form,
	NULL, 0);

    XtVaSetValues(fsat,
	XtNalignment, (XtArgVal) OL_CENTER,
        XtNstring, (XtArgVal) "",
	XtNxRefWidget, (XtArgVal) ltime,
	XtNxAttachRight, (XtArgVal) TRUE,
	XtNxVaryOffset, (XtArgVal) TRUE,
	NULL);

    utime = XtCreateManagedWidget("utime", staticTextWidgetClass, form,
	NULL, 0);

    XtVaSetValues(utime,
        XtNstring, (XtArgVal) "Universal time",
	XtNxRefWidget, (XtArgVal) fsat,
	XtNxAttachRight, (XtArgVal) TRUE,
	XtNxVaryOffset, (XtArgVal) TRUE,
	NULL);

    XtSetArg(wargs[0], XtNpushpin, (XtArgVal) OL_OUT);

    menu = XtCreatePopupShell("Xsunclock", menuShellWidgetClass, canvas,
	wargs, 1);

    XtSetArg(wargs[0], XtNmenuPane, &pane);
    XtGetValues(menu, wargs, 1);
    for (c = 0; menutags[c] != NULL; c++) {
	mbutton[c] = XtCreateManagedWidget(menutags[c],
	    oblongButtonWidgetClass, pane, NULL, 0);
	XtAddCallback(mbutton[c], XtNselect, pushed, (XtPointer) (c + 1));
    }
    busybutton(2, TRUE);

    interval = XtCreateManagedWidget("Interval", menuButtonWidgetClass,
	pane, NULL, 0);
    XtSetArg(wargs[0], XtNmenuPane, &ipane);
    XtGetValues(interval, wargs, 1);
    for (c = 0; itags[c] != NULL; c++) {
	Widget button = XtCreateManagedWidget(itags[c],
	    oblongButtonWidgetClass, ipane, NULL, 0);
	XtAddCallback(button, XtNselect, intset, (XtPointer) c);
    }

    dirmenu = XtCreateManagedWidget("Direction", menuButtonWidgetClass,
	pane, NULL, 0);
    XtSetArg(wargs[0], XtNmenuPane, &dirpane);
    XtGetValues(dirmenu, wargs, 1);
    for (c = 0; dtags[c] != NULL; c++) {
	Widget button = XtCreateManagedWidget(dtags[c],
	    oblongButtonWidgetClass, dirpane, NULL, 0);
	XtAddCallback(button, XtNselect, dirset, (XtPointer) dvalues[c]);
    }

    actsat(argv[0]);
    if (satchain != NULL) {
        Widget button = XtCreateManagedWidget("Satellite",
	    oblongButtonWidgetClass, pane, NULL, 0), spop;

	spop = satpop(button);
	XtAddCallback(button, XtNselect, popsat, spop);

    }

    XtSetArg(wargs[0], XtNforeground, &gcv.foreground);
    XtSetArg(wargs[1], XtNbackground, &gcv.background);

    XtGetValues(canvas, wargs, 2);

    gcv.function = GXcopy;
    gcv.line_style = LineSolid;
    gc = XtGetGC(toplevel, GCForeground | GCBackground | GCFunction |
			   GCLineStyle, &gcv);

    /* Graphics context for writing into icon pixmaps. */
    gc1 = XCreateGC(XtDisplay(toplevel), p, GCForeground | GCBackground | GCFunction, &gcv);

    gcv.function = GXcopyInverted;
    igc = XtGetGC(toplevel, GCForeground | GCBackground | GCFunction |
			   GCLineStyle, &gcv);

    gcv.function = GXclear;
    cgc = XtGetGC(toplevel, GCForeground | GCBackground | GCFunction |
			   GCLineStyle, &gcv);

    gcv.function = GXxor;
    gcv.foreground = gcv.foreground ^ gcv.background;
    xgc = XtGetGC(toplevel, GCForeground | GCBackground | GCFunction |
			   GCLineStyle, &gcv);

    ftime(&tbtp);


    iconic = TRUE;
    memcpy(map_icon_work, map_icon_bits, sizeof map_icon_bits);
    ringgg(toplevel, NULL);
    XtRealizeWidget(toplevel);

    XtAppMainLoop(app);

    return 0;
}

/*  ROP  --  Perform raster op on icon.  */

static void rop(sx, sy, lx)
  int sx, sy, lx;
{
    int rowoff = sy * (map_icon_width / 8), i;

    assert(sy >= 0);
    assert(sy < map_icon_height);
    assert(sx >= 0);
    assert(((sx + lx) - 1) < map_icon_width);
    for (i = sx; i < (sx + lx); i++) {
	map_icon_work[rowoff + (i / 8)] ^= 1 << (i & 7);
    }
}

/*  RINGGG  --	Update	status	on  interval  timer  ticks  and redraw
		window if needed.  */

static void ringgg(client_data, id)
  XtPointer client_data;
  XtIntervalId *id;
{
    /* If we were called preemptively, cancel the pending unexpired
       timeout. */

    if (id == NULL && tid != NULL) {
	XtRemoveTimeOut(tid);
    }

    tid = XtAppAddTimeOut(app, animate ? 150 :
	(id == NULL ? 10 : ((iconic ? CLOSED_SECS : OPEN_SECS) * 1000)),
	ringgg, (Widget) client_data);

    updimage(TRUE);
}


/*  JDATE  --  Convert internal GMT date and time to Julian day
	       and fraction.  */

static long jdate(t)
  struct tm *t;
{
    long c, m, y;

    y = t->tm_year + 1900;
    m = t->tm_mon + 1;
    if (m > 2) {
	m = m - 3;
    } else {
	m = m + 9;
	y--;
    }
    c = y / 100L;		      /* Compute century */
    y -= 100L * c;
    return t->tm_mday + (c * 146097L) / 4 + (y * 1461L) / 4 +
	(m * 153L + 2) / 5 + 1721119L;
}

/* JTIME --    Convert internal GMT  date  and	time  to  astronomical
	       Julian  time  (i.e.   Julian  date  plus  day fraction,
	       expressed as a double).	*/

static double jtime(t)
  struct tm *t;
{
    return (jdate(t) - 0.5) + 
	(((long) t->tm_sec) +
	 60L * (t->tm_min + 60L * t->tm_hour)) / 86400.0;
}

/*  KEPLER  --	Solve the equation of Kepler.  */

static double kepler(m, ecc)
  double m, ecc;
{
    double e, delta;
#define EPSILON 1E-6

    e = m = dtr(m);
    do {
	delta = e - ecc * sin(e) - m;
	e -= delta / (1 - ecc * cos(e));
    } while (abs(delta) > EPSILON);
    return e;
}

/*  SUNPOS  --	Calculate position of the Sun.	JD is the Julian  date
		of  the  instant for which the position is desired and
		APPARENT should be nonzero if  the  apparent  position
		(corrected  for  nutation  and aberration) is desired.
                The Sun's co-ordinates are returned  in  RA  and  DEC,
		both  specified  in degrees (divide RA by 15 to obtain
		hours).  The radius vector to the Sun in  astronomical
                units  is returned in RV and the Sun's longitude (true
		or apparent, as desired) is  returned  as  degrees  in
		SLONG.	*/

static void sunpos(jd, apparent, ra, dec, rv, slong)
  double jd;
  int apparent;
  double *ra, *dec, *rv, *slong;
{
    double t, t2, t3, l, m, e, ea, v, theta, omega,
	   eps;

    /* Time, in Julian centuries of 36525 ephemeris days,
       measured from the epoch 1900 January 0.5 ET. */

    t = (jd - 2415020.0) / 36525.0;
    t2 = t * t;
    t3 = t2 * t;

    /* Geometric mean longitude of the Sun, referred to the
       mean equinox of the date. */

    l = fixangle(279.69668 + 36000.76892 * t + 0.0003025 * t2);

    /* Sun's mean anomaly. */

    m = fixangle(358.47583 + 35999.04975*t - 0.000150*t2 - 0.0000033*t3);

    /* Eccentricity of the Earth's orbit. */

    e = 0.01675104 - 0.0000418 * t - 0.000000126 * t2;

    /* Eccentric anomaly. */

    ea = kepler(m, e);

    /* True anomaly */

    v = fixangle(2 * rtd(atan(sqrt((1 + e) / (1 - e))  * tan(ea / 2))));

    /* Sun's true longitude. */

    theta = l + v - m;

    /* Obliquity of the ecliptic. */

    eps = 23.452294 - 0.0130125 * t - 0.00000164 * t2 + 0.000000503 * t3;

    /* Corrections for Sun's apparent longitude, if desired. */

    if (apparent) {
       omega = fixangle(259.18 - 1934.142 * t);
       theta = theta - 0.00569 - 0.00479 * sin(dtr(omega));
       eps += 0.00256 * cos(dtr(omega));
    }

    /* Return Sun's longitude and radius vector */

    *slong = theta;
    *rv = (1.0000002 * (1 - e * e)) / (1 + e * cos(dtr(v)));

    /* Determine solar co-ordinates. */

    *ra =
	fixangle(rtd(atan2(cos(dtr(eps)) * sin(dtr(theta)), cos(dtr(theta)))));
    *dec = rtd(asin(sin(dtr(eps)) * sin(dtr(theta))));
}

/*  GMST  --  Calculate Greenwich Mean Siderial Time for a given
	      instant expressed as a Julian date and fraction.	*/

static double gmst(jd)
  double jd;
{
    double t, theta0;

    /* Time, in Julian centuries of 36525 ephemeris days,
       measured from the epoch 1900 January 0.5 ET. */

    t = ((floor(jd + 0.5) - 0.5) - 2415020.0) / 36525.0;

    theta0 = 6.6460656 + 2400.051262 * t + 0.00002581 * t * t;

    t = (jd + 0.5) - (floor(jd + 0.5));

    theta0 += (t * 24.0) * 1.002737908;

    theta0 = (theta0 - 24.0 * (floor(theta0 / 24.0)));

    return theta0;
}

/*  PROJILLUM  --  Project illuminated area on the map.  */

static void projillum(wtab, xdots, ydots, dec)
  short *wtab;
  int xdots, ydots;
  double dec;
{
    int i, ftf = TRUE, ilon, ilat, lilon, lilat, xt;
    double m, x, y, z, th, lon, lat, s, c;

    /* Clear unoccupied cells in width table */

    for (i = 0; i < ydots; i++) {
	wtab[i] = -1;
    }

    /* Build transformation for declination */

    s = sin(-dtr(dec));
    c = cos(-dtr(dec));

    /* Increment over a semicircle of illumination */

    for (th = -(PI / 2); th <= PI / 2 + 0.001; th += PI / TERMINC) {

	/* Transform the point through the declination rotation. */

	x = -s * sin(th);
	y = cos(th);
	z = c * sin(th);

	/* Transform the resulting co-ordinate through the
	   map projection to obtain screen co-ordinates. */

	lon = (y == 0 && x == 0) ? 0.0 : rtd(atan2(y, x));
	lat = rtd(asin(z));

	ilat = ydots - (lat + 90) * (ydots / 180.0);
	ilon = lon * (xdots / 360.0);

	if (ftf) {

	    /* First time.  Just save start co-ordinate. */

	    lilon = ilon;
	    lilat = ilat;
	    ftf = FALSE;
	} else {

	    /* Trace out the line and set the width table. */

	    if (lilat == ilat) {
		wtab[(ydots - 1) - ilat] = 2 * (ilon == 0 ? 1 : ilon);
	    } else {
		m = ((double) (ilon - lilon)) / (ilat - lilat);
		for (i = lilat; i != ilat; i += sgn(ilat - lilat)) {
		    xt = lilon + floor((m * (i - lilat)) + 0.5);
		    wtab[(ydots - 1) - i] = 2 * (xt == 0 ? 1 : xt);
		}
	    }
	    lilon = ilon;
	    lilat = ilat;
	}
    }

    /* Now tweak the widths to generate full illumination for
       the correct pole. */

    if (dec < 0.0) {
	ilat = ydots - 1;
	lilat = -1;
    } else {
	ilat = 0;
	lilat = 1;
    }

    for (i = ilat; i != ydots / 2; i += lilat) {
	if (wtab[i] != -1) {
	    while (TRUE) {
		wtab[i] = xdots;
		if (i == ilat) {
		    break;
		}
		i -= lilat;
	    }
	    break;
	}
    }
}

/*  XSPAN  --  Complement a span of pixels.  Called with line in which
	       pixels are contained, leftmost pixel in the  line,  and
	       the   number   of   pixels   to	 complement.   Handles
	       wrap-around at the right edge of the screen.  */

static void xspan(pline, leftp, npix)
  int pline, leftp, npix;
{
    assert(npix <= xdots);
    assert(pline >= 0 && pline < ydots);
    leftp = leftp % xdots;

    if ((leftp + npix) > xdots) {
	XDrawLine(XtDisplay(canvas), XtWindow(canvas), xgc, 
	    leftp, pline, (xdots - 1), pline);
	XDrawLine(XtDisplay(canvas), XtWindow(canvas), xgc,
	    0, pline, ((leftp + npix) - (xdots + 1)),
	    pline);
    } else {
	XDrawLine(XtDisplay(canvas), XtWindow(canvas), xgc,
	    leftp, pline, leftp + (npix - 1), pline);
    }
}

/*  ISPAN  --  Complement a span of pixels.  Called with line in which
	       pixels are contained, leftmost pixel in the  line,  and
	       the   number   of   pixels   to	 complement.   Handles
	       wrap-around at the right edge of the screen.  */

static void ispan(pline, leftp, npix)
  int pline, leftp, npix;
{
    assert(npix <= xdots);
    assert(pline >= 0 && pline < ydots);
    leftp = leftp % xdots;

    if ((leftp + npix) > xdots) {
	rop(leftp, pline, ((xdots - 1) - (leftp)) + 1);
	rop(0, pline, ((leftp + npix) - (xdots)));
    } else {
	rop(leftp, pline, npix);
    }
}

/*  MOVETERM  --  Update illuminated portion of the globe.  */

static void moveterm(cspan, wtab, noon, otab, onoon, xdots, ydots)
  void (*cspan)();
  short *wtab, *otab;
  int noon, onoon, xdots, ydots;
{
    int i, ol, oh, nl, nh;

    for (i = 0; i < ydots; i++) {

	/* If line is off in new width table but is set in
	   the old table, clear it. */

	if (wtab[i] < 0) {
	    if (otab[i] >= 0) {
		(*cspan)(i, ((onoon - (otab[i] / 2)) + xdots) % xdots,
		   otab[i]);
	    }
	} else {

	    /* Line is on in new width table.  If it was off in
	       the old width table, just draw it. */

	    if (otab[i] < 0) {
	       (*cspan)(i, ((noon - (wtab[i] / 2)) + xdots) % xdots,
		  wtab[i]);
	    } else {

		/* If both the old and new spans were the entire
                   screen, they're equivalent. */

		if ((otab[i] == wtab[i]) && (wtab[i] == xdots)) {
		    continue;
		}

		/* The line was on in both the old and new width
		   tables.  We must adjust the difference in the
		   span.  */

		ol =  ((onoon - (otab[i] / 2)) + xdots) % xdots;
		oh = (ol + otab[i]) - 1;
		nl =  ((noon - (wtab[i] / 2)) + xdots) % xdots;
		nh = (nl + wtab[i]) - 1;

		/* If spans are disjoint, erase old span and set
		   new span. */

		if (oh < nl || nh < ol) {
		    (*cspan)(i, ol, (oh - ol) + 1);
		    (*cspan)(i, nl, (nh - nl) + 1);
		} else {
		    /* Clear portion(s) of old span that extend
		       beyond end of new span. */
		    if (ol < nl) {
			(*cspan)(i, ol, nl - ol);
			ol = nl;
		    }
		    if (oh > nh) {
			(*cspan)(i, nh + 1, oh - nh);
			oh = nh;
		    }
		    /* Extend existing (possibly trimmed) span to
		       correct new length. */
		    if (nl < ol) {
			(*cspan)(i, nl, ol - nl);
		    }
		    if (nh > oh) {
			(*cspan)(i, oh + 1, nh - oh);
		    }
		}
	    }
	}
	otab[i] = wtab[i];
    }
}
