/*
 * xsky - an interactive sky atlas
 *
 * Copyright 1992-5, Terry R. Friedrichsen
 *
 * This program may be copied and redistributed, in whole or in part,
 * as long as you don't try to make any money from the sale or redis-
 * tribution of the program or any part of the program, or pretend
 * that you wrote the program or any of its parts unless specifically
 * credited by the original author.
 *
 * You are free to make use of this software in your own programs, as
 * long as you credit the original author where it is due.
 */

/*
 * WARRANTY:
 * xsky was written as a learning project and as a demonstration of
 * X Window System programming.  xsky doesn't do anything; it is not
 * merchantable, and it is not fit for any purpose whatsoever.  In
 * fact, don't use xsky at all; it's free, and you're getting what
 * you paid for.
 */

#include <stdio.h>
#include <ctype.h>

#include <math.h>
#include <stdlib.h>

#include <unistd.h>

#include <string.h>

#include <X11/Intrinsic.h>

#include "ybs.h"

#include "skydefs.h"

#include "pos.h"
#include "catalog.h"
#include "draw.h"
#include "catreader.h"

#include "starnames.h"

struct ybs_node {
  double ra_rad;
  double dec_rad;
  float mag;
  long filepos;
  shortint hr_number;
  smallint bayer_index;
  smallint bayer_super;
  shortint flamsteed_number;
  smallint constellation_index;
  char spectrum;
};

/* static function prototypes */
static FILE *read_ybs PROTOTYPE((void));
static void calculate_ybs_field PROTOTYPE((struct display *,
					                struct cat_header *));
static boolean check_mag_limit PROTOTYPE((int,double));
static boolean find_ybs_star PROTOTYPE((char *));
static void get_ybs_pos PROTOTYPE((int,double *,double *));
static void draw_ybs_star PROTOTYPE((Widget,int,int,int,int,int,int,int));
static boolean ps_draw_ybs_star PROTOTYPE((struct display *,int,int,int));
static boolean display_ybs PROTOTYPE((FILE *,int));
static void format_ybs_name PROTOTYPE((int,char *,IDfont *));
static boolean find_star_by_hr PROTOTYPE((int));
static boolean find_star_by_bayer PROTOTYPE((int,int));
static boolean find_star_by_flamsteed PROTOTYPE((int,int));
static void fill_ybs_entry PROTOTYPE((char *,int));
static int mag_compare PROTOTYPE((struct ybs_node *,struct ybs_node *));

/* external function prototypes */
extern char *build_filespec PROTOTYPE((char *,char *));
extern boolean read_binary_database PROTOTYPE((char *,void *,int *));
extern void init_precess PROTOTYPE((double,double));
extern void hs_heapsort PROTOTYPE((char [],size_t,size_t,int (*)()));
extern boolean write_binary_database PROTOTYPE((char *,void *,size_t,size_t));
extern void precess PROTOTYPE((double,double,double *,double *));
extern void read_starname_table PROTOTYPE((void));
extern void update_display_pos PROTOTYPE((double,double));
extern void format_ybs_display PROTOTYPE((char *));
extern int find_const_abbrev PROTOTYPE((char *));
extern int find_constellation PROTOTYPE((char *));
extern int find_greek_letter PROTOTYPE((char *));
extern double ra_pos_to_rad PROTOTYPE((struct ra_pos));
extern double dec_pos_to_rad PROTOTYPE((struct dec_pos));
extern void add_to_display_list PROTOTYPE((struct display *,
					           struct cat_header *,
					                  int,double,double));
extern void draw_star PROTOTYPE((Widget,int,int,int,StarColor));
extern boolean ps_draw_star PROTOTYPE((struct display *,double,double,double));
extern int find_star_by_name PROTOTYPE((char *));

extern char greek_letter[];

static struct ybs_node ybs_array[MAX_YBS];

static int ybs_array_max;

/* table of star names */
extern struct starname starnames[];

extern float mag_limit;

#define YBS_DATA_FILE   "ybs.dat"
#define YBS_BINARY_FILE "ybs.bin"

static FILE *ybs_fd;

/* YBS record buffer */
static char ybs_rec[YBS_RECLEN + 1 + 1];



/* initialize the catalog data structure */

void init_ybs(catalog)

struct cat_header *catalog;

{
  catalog->read_data = read_ybs;
  catalog->calculate_field = calculate_ybs_field;
  catalog->mag_limit_check = check_mag_limit;
  catalog->find_object = find_ybs_star;
  catalog->get_object_pos = get_ybs_pos;
  catalog->draw_object = draw_ybs_star;
  catalog->ps_draw_object = ps_draw_ybs_star;
  catalog->display_data = display_ybs;
  catalog->format_name = format_ybs_name;
  catalog->menu_name = YBS_MENU_NAME;
  catalog->catalog_name = YBS_CATALOG_NAME;

  return;
}



/* read in the YBS catalog file, extracting needed info */

static FILE *read_ybs()

{
  char *filebuff;
  char *bin_filebuff;
  int ybs_entry;

/* get the path to the YBS data */
  filebuff = build_filespec("YBS_PATH",YBS_DATA_FILE);

/* open the YBS data file */
  if ((ybs_fd = fopen(filebuff,"r")) == (FILE *)NULL) {
    printf("sky:  can't open YBS data file %s\n",filebuff);
    perror("sky");
    exit(1);
  }

/* get the path to the YBS binary file */
  bin_filebuff = build_filespec("YBS_BIN_PATH",YBS_BINARY_FILE);

/* see if we can read the binary file */
  if (access(bin_filebuff,R_OK) == 0)
    /* we can; try to use the pre-built database */
    if (read_binary_database(bin_filebuff,ybs_array,&ybs_array_max)) {
      /* success; free the filename buffers */
      free((void *)filebuff);
      free((void *)bin_filebuff);

      /* read in the star name table */
      read_starname_table();

      /* return the fd of the open data file */
      return(ybs_fd);
    }

/* prior to reading the YBS catalog, build the rotation matrix for precessing
 * the stellar coordinates to the common equinox from the given epoch */
  if (YBS_EPOCH != COMMON_EQUINOX)
    init_precess(YBS_EPOCH,COMMON_EQUINOX);

/* read the YBS catalog, saving the information necessary for the display */
  ybs_entry = 0;
  while (TRUE) {
    ybs_array[ybs_entry].filepos = ftell(ybs_fd);
    if (fgets(ybs_rec,sizeof(ybs_rec),ybs_fd) == (char *)NULL)
      break;
    /* if no position is given, skip the entry */
    if (strncmp(&ybs_rec[POS_START],"               ",15) == EQUAL)
      continue;
    fill_ybs_entry(ybs_rec,ybs_entry++);
  }

/* leave the file open for later use */
/*  (void)fclose(ybs_fd);*/

/* remember the last ybs entry we filled */
  ybs_array_max = ybs_entry - 1;

/* now sort the ybs array in decreasing order of magnitude */
  hs_heapsort((void *)ybs_array,(size_t)ybs_entry,
	                         (size_t)sizeof(struct ybs_node),mag_compare);

/* attempt to save the binary database */
  if (access(bin_filebuff,W_OK) == 0)
    /* it's writeable, so let's try */
    if (! write_binary_database(bin_filebuff,(void *)ybs_array,
				            (size_t)sizeof(struct ybs_node),
			                                   (size_t)ybs_entry))
      /* dump of binary database failed; be sure the file is gone */
      (void)unlink(bin_filebuff);

/* free the filename buffers */
  free((void *)filebuff);
  free((void *)bin_filebuff);

/* read in the star name table */
  read_starname_table();

  return(ybs_fd);
}



/* return TRUE if the magnitude of the star meets the limiting magnitude */

static boolean check_mag_limit(array_pos,limit)

int array_pos;
float limit;

{
  return((boolean)(ybs_array[array_pos].mag <= limit));
}



/* compare two array entries for magnitude - compare routine for heapsort */

static int mag_compare(ybs_entry_1,ybs_entry_2)

struct ybs_node *ybs_entry_1, *ybs_entry_2;

{
/* major key is the magnitude */
  if (ybs_entry_1->mag < ybs_entry_2->mag)
    return(-1);
  else if (ybs_entry_1->mag > ybs_entry_2->mag)
    return(1);
  else
    return(0);

/* NOTREACHED */
}



/* given an object's index, return its right ascension and declination */

static void get_ybs_pos(array_pos,ra_rad,dec_rad)

int array_pos;
double *ra_rad, *dec_rad;

{
  *ra_rad = ybs_array[array_pos].ra_rad;
  *dec_rad = ybs_array[array_pos].dec_rad;

  return;
}



/* given a YBS record, make a data node for it */

static void fill_ybs_entry(record,ybs_entry)

char *record;
int ybs_entry;

{
  int reclen;
  char field[11];
  struct ra_pos ra_pos;
  struct dec_pos dec_pos;

/* get the length of the record for GET_FIELD */
  reclen = strlen(record) - 1;

/* get the HR number */
  GET_FIELD(field,record,0,4);
  ybs_array[ybs_entry].hr_number = atoi(field);

/* get the Flamsteed number */
  GET_FIELD(field,record,FLAMSTEED_START,3);
  ybs_array[ybs_entry].flamsteed_number = (shortint)atoi(field);

/* get the Bayer letter index */
  GET_FIELD(field,record,BAYER_START,2);
  ybs_array[ybs_entry].bayer_index = (smallint)atoi(field);

/* get the Bayer superscript */
  GET_FIELD(field,record,BAYER_SUPER_START,2);
  ybs_array[ybs_entry].bayer_super = (smallint)atoi(field);

/* get the constellation abbreviation */
  GET_FIELD(field,record,CONSTELLATION_NAME,3);
  if (strcmp(field,"   ") != EQUAL)
    ybs_array[ybs_entry].constellation_index =
                                           (smallint)find_const_abbrev(field);
  if (ybs_array[ybs_entry].constellation_index == (smallint)-1)
    printf("constellation abbreviation %s was not found for HR %d\n",field,
	                                      ybs_array[ybs_entry].hr_number);

/* get the RA in pieces */
  GET_FIELD(field,record,RA_START,2);
  ra_pos.hours = (smallint)atoi(field);

  GET_FIELD(field,record,RA_START + 2,2);
  ra_pos.minutes = (smallint)atoi(field);

  GET_FIELD(field,record,RA_START + 4,2);
  ra_pos.seconds = (smallint)atoi(field);

  GET_FIELD(field,record,RA_START + 7,1);
  ra_pos.thousandths = (shortint)atoi(field) * 100;

/* similarly, get the declination in pieces */
  dec_pos.sign = (short)1;
  GET_FIELD(field,record,DEC_START,1);
  if (field[0] == '-')
    dec_pos.sign = (short)-1;

  GET_FIELD(field,record,DEC_START + 1,2);
  dec_pos.degrees = (smallint)atoi(field);

  GET_FIELD(field,record,DEC_START + 3,2);
  dec_pos.minutes = (smallint)atoi(field);

  GET_FIELD(field,record,DEC_START + 5,2);
  dec_pos.seconds = (smallint)atoi(field);

  dec_pos.hundredths = (smallint)0;

/* compute the RA and declination in radians */
  ybs_array[ybs_entry].ra_rad = ra_pos_to_rad(ra_pos);
  ybs_array[ybs_entry].dec_rad = dec_pos_to_rad(dec_pos);

/* precess them to the common equinox */
  if (YBS_EPOCH != COMMON_EQUINOX)
    precess(ybs_array[ybs_entry].ra_rad,ybs_array[ybs_entry].dec_rad,
	          &ybs_array[ybs_entry].ra_rad,&ybs_array[ybs_entry].dec_rad);

/* get the magnitude */
  GET_FIELD(field,record,MAG_START,5);
  ybs_array[ybs_entry].mag = atof(field);

/* get the spectral class */
  ybs_array[ybs_entry].spectrum = record[SPECTRUM_START + 2];

  return;
}



/* construct a list of those stars which appear in the current display */

static void calculate_ybs_field(display,catalog)

struct display *display;
struct cat_header *catalog;

{
  int i;

/* loop through all YBS entries, checking for visibility */
  for (i = 0; i <= ybs_array_max; i++)
    /* check this star and add it to the list if visible */
    add_to_display_list(display,catalog,i,ybs_array[i].ra_rad,
			                                ybs_array[i].dec_rad);

#ifdef DEBUG
  printf("calculate_ybs_field returning\n");
#endif
  return;
}



/* draw a YBS star on the sky */

static void draw_ybs_star(sky_widget,array_pos,obj_x,obj_y,ul_x,ul_y,
		                                                 width,height)

Widget sky_widget;
int array_pos;
int obj_x, obj_y;
int ul_x, ul_y;
int width, height;

{
  int size;
  int safety_margin;
  StarColor color;

/* be sure it is bright enough to display */
  if (! check_mag_limit(array_pos,mag_limit))
    return;

  /* calculate size based on magnitude */
  size = STARSIZE(ybs_array[array_pos].mag);

  /* draw the object even if it is only partially in the box */
  safety_margin = 1;
  if (((obj_x + (size + 1) / 2) >= ul_x) &&
               ((obj_x - (size + 1) / 2 - safety_margin) <= (ul_x + width)) &&
               ((obj_y + (size + 1) / 2 + safety_margin) >= ul_y) &&
               ((obj_y - (size + 1) / 2 - safety_margin) <= (ul_y + height)))
    ;
  else
    return;

  /* assign a color based on the spectral class */
  switch (ybs_array[array_pos].spectrum) {
  case 'W':
  case 'O':
    color = O;
    break;
  case 'B':
    color = B;
    break;
  case 'A':
    color = A;
    break;
  case 'F':
    color = F;
    break;
  case 'G':
    color = G;
    break;
  case 'K':
    color = K;
    break;
  case 'M':
  case 'C':
  case 'N':
  case 'S':
    color = M;
    break;
  case 'p':        /* at least one star is just "pec" */
    color = A;     /* make unknowns white */
    break;
  default:
    printf("array pos = %d\n",array_pos);
    printf("star %d has unrecognized spectral class %c\n",
	                                      ybs_array[array_pos].hr_number,
	                                      ybs_array[array_pos].spectrum);
  }

/* draw the star with correct location, size, and color */
  draw_star(sky_widget,obj_x,obj_y,size,color);

  return;
}



/* draw one YBS star on the PostScript display */

static boolean ps_draw_ybs_star(display,array_pos,x,y)

struct display *display;
int array_pos;
int x, y;

{
  int size;

/* be sure it is bright enough to display */
  if (! check_mag_limit(array_pos,mag_limit))
    return(TRUE);

/* calculate size based on magnitude */
  size = STARSIZE(ybs_array[array_pos].mag);

/* draw the star in the correct position with the correct size */
  return(ps_draw_star(display,(double)x,(double)y,(double)size));
}



/* display the catalog data for the given star */

static boolean display_ybs(ybs_fd,ybs_array_pos)

FILE *ybs_fd;
int ybs_array_pos;

{
  int i;

/* get info from the YBS catalog */
  if (fseek(ybs_fd,ybs_array[ybs_array_pos].filepos,SEEK_SET) == -1)
    return(FALSE);
  if (fgets(ybs_rec,sizeof(ybs_rec),ybs_fd) == (char *)NULL)
    return(FALSE);

/* fill out the (possibly short) record with blanks */
  for (i = strlen(ybs_rec) - 1; i < YBS_RECLEN; i++)
    ybs_rec[i] = ' ';

/* format the record into an informational display */
  format_ybs_display(ybs_rec);

  return(TRUE);
}



/* if the name in the buffer is a recognizeable format, attempt to find it */

static boolean find_ybs_star(buffer)

char *buffer;

{
  char *ptr;
  int constellation;
  int hr_number;
  int i;

/*
 * algorithm:
 *
 * 1) if the first two letters are HR or hr, look for HR number
 * 2) if the first three letters are YBS or ybs, look for HR number
 * 3) if the buffer after the first token is a constellation, then (4) or (5)
 * 4) if the first token is a greek letter, look for Bayer designation
 * 5) if the first token is numeric, look for Flamsteed designation
 * 6) check for a name match
 */

  if ((strncmp(buffer,"HR",2) == EQUAL) || strncmp(buffer,"hr",2) == EQUAL)
    /* try the name as a Henry Russell number */
    return(find_star_by_hr(atoi(&buffer[2])));
  else if ((strncmp(buffer,"YBS",3) == EQUAL) ||
	                                     strncmp(buffer,"ybs",3) == EQUAL)
    /* try the name as a Henry Russell number */
    return(find_star_by_hr(atoi(&buffer[3])));
  else
    ;

/* advance to the second token */
  ptr = buffer;
  while ((! isspace(*ptr)) && (*ptr != '\0'))
    ptr++;
  /* if we found a space, advance to the start of the next token */
  if (isspace(*ptr))
    ptr++;

/* if there is more than one token, check to see if it is a constellation */
  if (*ptr == '\0')
    /* this must be a star name */
    if ((hr_number = find_star_by_name(buffer)) == -1)
      return(FALSE);
    else
      return(find_star_by_hr(hr_number));
  else {
    /* if the rest of the buffer is not a constellation, must be a star name */
    if ((constellation = find_constellation(ptr)) == -1)
      if ((hr_number = find_star_by_name(buffer)) == -1)
	return(FALSE);
      else
        return(find_star_by_hr(hr_number));

    /* plant a nul at the end of the first token */
    *--ptr = '\0';

    /* in case this is a Bayer designation, find the Greek letter index */
    i = find_greek_letter(buffer);

    /* repair the buffer before we forget */
    *ptr = ' ';

    /* was a Greek letter index found? */
    if (i == -1) {
      /* is the first token all numeric? */
      ptr = buffer;
      while (isdigit(*ptr))
	ptr++;
      if (*ptr == ' ')
	/* yes; this is a Flamsteed designation */
	return(find_star_by_flamsteed(atoi(buffer),constellation));
      else
	/* no; we don't recognize this type of designation */
	return(FALSE);
    }
    else
      /* yes; this is a Bayer designation */
      return(find_star_by_bayer(i,constellation));
  }

/* NOTREACHED */
}



/* find the HR star in the buffer and update the display coordinates */

static boolean find_star_by_hr(ybs_no)

int ybs_no;

{
  int i;

/* loop through the array looking for this number */
  for (i = 0; i <= ybs_array_max; i++)
    if (ybs_array[i].hr_number == ybs_no)
      break;

/* return FALSE if we didn't find it */
  if (i > ybs_array_max)
    return(FALSE);
  else {
    /* update the display coordinates */
    update_display_pos(ybs_array[i].ra_rad,ybs_array[i].dec_rad);
    return(TRUE);
  }

/* NOTREACHED */
}



/* find a star given its greek letter and constellation index */

static boolean find_star_by_bayer(greek,constellation)

int greek;
int constellation;

{
  int i;

/* make the Greek letter index 1-relative */
  greek++;

/* search the YBS catalog for a matching star */
  for (i = 0; i <= ybs_array_max; i++)
    if ((ybs_array[i].bayer_index == greek) &&
	                  (ybs_array[i].constellation_index == constellation))
      break;

/* if we found a match, move to that star */
  if (i > ybs_array_max)
    /* no match found; fail */
    return(FALSE);
  else {
    /* update the display coordinates */
    update_display_pos(ybs_array[i].ra_rad,ybs_array[i].dec_rad);
    return(TRUE);
  }

/* NOTREACHED */
}



/* find a star given its Flamsteed designation */

static boolean find_star_by_flamsteed(flamsteed_no,constellation)

int flamsteed_no;
int constellation;

{
  int i;

/* search the YBS catalog for a matching star */
  for (i = 0; i <= ybs_array_max; i++)
    if ((ybs_array[i].flamsteed_number == flamsteed_no) &&
	                  (ybs_array[i].constellation_index == constellation))
      break;

/* if we found a match, move to that star */
  if (i > ybs_array_max)
    /* no match found; fail */
    return(FALSE);
  else {
    /* update the display coordinates */
    update_display_pos(ybs_array[i].ra_rad,ybs_array[i].dec_rad);
    return(TRUE);
  }

/* NOTREACHED */
}



/* format a nice name for a star */

static void format_ybs_name(ybs_pos,buffer,font)

int ybs_pos;
char *buffer;
IDfont *font;

{
  char field[11];
  int reclen;
  int num;
  int i;

/* prefer Bayer designation over Flamsteed */
  if (ybs_array[ybs_pos].bayer_index > 0) {
    /* change to Greek font and format as a Bayer designation */
    *font = GREEK;
    sprintf(buffer,"%c",greek_letter[ybs_array[ybs_pos].bayer_index - 1]);
    /* is there a superscript? */
    if (ybs_array[ybs_pos].bayer_super > 0)
      /* yes, add the superscript */
      sprintf(&buffer[strlen(buffer)],"-%d",ybs_array[ybs_pos].bayer_super);
  }
  else if (ybs_array[ybs_pos].flamsteed_number > 0)
    /* format as a Flamsteed designation */
    sprintf(buffer,"%d",ybs_array[ybs_pos].flamsteed_number);
  else {
    /* neither Bayer nor Flamsteed - work harder for other designations */
    if (fseek(ybs_fd,ybs_array[ybs_pos].filepos,SEEK_SET) == -1)
      return;
    if (fgets(ybs_rec,sizeof(ybs_rec),ybs_fd) == (char *)NULL)
      return;

    /* try an HD number next */
    reclen = strlen(ybs_rec) - 1;
    GET_FIELD(field,ybs_rec,HD_START,6);
    if ((num = atoi(field)) != 0) {
      if (num >= 100000)
	sprintf(buffer,"HDE %d",num);
      else
	sprintf(buffer,"HD %d",num);
      return;
    }

    /* check BD next */
    GET_FIELD(field,ybs_rec,BD_START,8);
    if (strncmp(field,"        ",8) != 0) {
      if (field[1] == ' ')
	field[1] = '0';
      for (i = 0; i < 5; i++)
	if (field[3 + i] == ' ')
	  field[3 + i] = '0';
      sprintf(buffer,"BD %.3sd %.5s",&field[0],&field[3]);
      return;
    }

    /* finally, use the HR number */
    sprintf(buffer,"HR %d",ybs_array[ybs_pos].hr_number);
  }

  return;
}
