/*
 * xsky - an interactive sky atlas
 *
 * Copyright 1992-4, Terry R. Friedrichsen
 *
 * This program may be copied and redistributed, in whole or in part,
 * as long as you don't try to make any money from the sale or redis-
 * tribution of the program or any part of the program, or pretend
 * that you wrote the program or any of its parts unless specifically
 * credited by the original author.
 *
 * You are free to make use of this software in your own programs, as
 * long as you credit the original author where it is due.
 */

/*
 * WARRANTY:
 * xsky was written as a learning project and as a demonstration of
 * X Window System programming.  xsky doesn't do anything; it is not
 * merchantable, and it is not fit for any purpose whatsoever.  In
 * fact, don't use xsky at all; it's free, and you're getting what
 * you paid for.
 */

/* this routine formats and displays a YBS record */

#include <stdio.h>
#include <string.h>

#include <ctype.h>

#include "ybs.h"

#include "skydefs.h"

#include "catreader.h"

#include "format.h"

/* static function prototypes */
static void display_pos PROTOTYPE((char *));

/* external function prototypes */
extern int find_const_abbrev PROTOTYPE((char *));
extern char *find_starnames PROTOTYPE((int));
extern void format_starnames PROTOTYPE((char *,int));

extern char *greek[];

extern char *constellations_abbrev[];
extern char *constellations_latin[];

/* table of blank YBS entries */
static int ybs_blank[] = {
                92, 95, 182, 1057, 1841, 2472, 2496, 3515, 3671,
                6309, 6515, 7189, 7539, 8296, 0 };

#define MAX_YBS_ROW   70
#define NUM_YBS_ROWS  23



void format_ybs_display(record)

char *record;

{
  int reclen;
  char field[5];
  int hr_num;
  char *starlist;
  int i, j;
  char const_buffer[4];
  int con;
  int super;
  int desig_count;
  boolean ir_flag;
  boolean mag_flag;
  boolean numeric_flag;
  char bayer_buffer[3];
  int rad_vel;

/* start at the beginning of the display */
  idx = 0;
  row = 0;
  row_start = 0;
  max_row = MAX_YBS_ROW;

/* get the record length */
  reclen = strlen(record);

/* display the catalog number */
  GET_FIELD(field,record,CAT_START,4);
  hr_num = atoi(field);
  sprintf(&obj_info[idx],"HR %d",hr_num);
  idx += strlen(&obj_info[idx]);

  /* count the designations to control length of the first line */
  desig_count = 1;

/* see if this is one of the blank records */
  i = 0;
  while (ybs_blank[i] != 0) {
    if (hr_num == ybs_blank[i])
      break;
    i++;
  }

/* handle blank records specially */
  if (ybs_blank[i] != 0) {
    /* print the name */
    advance_to_tab();
    sprintf(&obj_info[idx],"    %.9s",&record[NAME_START]);
    idx += strlen(&obj_info[idx]);
    /* and the variable designation, if any */
    if (strncmp(&record[VARIABLE_START],"         ",9) != EQUAL) {
      advance_to_tab();
      sprintf(&obj_info[idx],"    %.9s",&record[VARIABLE_START]);
      idx += strlen(&obj_info[idx]);
    }
    return;
  }

/* if there's a constellation name, try for Flamsteed and Bayer designations */
  if (strncmp(&record[CONSTELLATION_NAME],"   ",3) != EQUAL) {
    /* get the constellation abbreviation index */
    strncpy(const_buffer,&record[CONSTELLATION_NAME],3);
    const_buffer[3] = '\0';
    con = find_const_abbrev(const_buffer);

    if (con == -1)
      printf("no match for constellation found in record %s\n",record);
    else {
      /* display Flamsteed designation, if provided */
      if (strncmp(&record[FLAMSTEED_START],"   ",3) != EQUAL) {
	add_space(4);
	sprintf(&obj_info[idx],"%.3s %s",&record[FLAMSTEED_START],
		                                   constellations_latin[con]);
	idx += strlen(&obj_info[idx]);

	/* count the designations */
	desig_count++;
      }

      /* display Bayer designation, if provided */
      if (strncmp(&record[BAYER_START],"  ",2) != EQUAL) {
	/* grab the Bayer letter index */
	GET_FIELD(field,record,BAYER_START,2);
	i = atoi(field);

	/* print the Bayer designation */
	add_space(4);
	sprintf(&obj_info[idx],"%s",greek[i - 1]);
	idx += strlen(&obj_info[idx]);

	/* get the Bayer superscript number */
	if (strncmp(&record[BAYER_SUPER_START],"  ",2) != EQUAL) {
	  GET_FIELD(field,record,BAYER_SUPER_START,2);
	  super = atoi(field);
	  sprintf(&obj_info[idx],"-%d",super);
	  idx += strlen(&obj_info[idx]);
	}

	/* finally, tack on the constellation */
	sprintf(&obj_info[idx]," %s",constellations_latin[con]);
	idx += strlen(&obj_info[idx]);

	/* count the designations */
	desig_count++;
      }
    }
  }

/* display the BD number */
  if (record[BD_START + 1] == ' ')
    record[BD_START + 1] = '0';
  for (i = 0; i < 5; i++)
    if (record[BD_START + 3 + i] == ' ')
      record[BD_START + 3 + i] = '0';
  add_space(4);
  sprintf(&obj_info[idx],"BD %.3sd %.5s",
	                             &record[BD_START],&record[BD_START + 3]);
  idx += strlen(&obj_info[idx]);

  /* count the designations */
  if (++desig_count == 4) {
    next_line();
    advance_to_tab();
  }

/* display the HD number */
  ljust(&record[HD_START],6);
  if (desig_count < 4)
    add_space(4);
  if (record[HD_START + 5] == ' ')
    sprintf(&obj_info[idx],"HD %.5s",&record[HD_START]);
  else
    sprintf(&obj_info[idx],"HDE %.6s",&record[HD_START]);
  idx += strlen(&obj_info[idx]);

/* check to see if we have star names for this star */
  if ((starlist = find_starnames(hr_num)) != (char *)NULL) {
    /* space down a little */
    blank_line();

    /* put out the star name list header */
    add_string("Common names:  ");

    /* put in the formatted star name list */
    format_starnames(starlist,MAX_YBS_ROW);
  }

/* space down some */
  blank_line();

/* identify infrared sources */
  ir_flag = FALSE;
  if (record[INFRARED_START] == 'I') {
    add_string("Infrared source");
    advance_to_tab();
    advance_to_tab();
    ir_flag = TRUE;
  }

/* display multiple-component data */
  if (record[DOUBLE_START] != ' ') {
    /* do a bunch of stuff for multiple-star systems */
    if (record[DOUBLE_START] == 'D')
      add_string("Occultation binary");
    else if (record[DOUBLE_START] == 'S')
      add_string("Speckle binary");
    else if (record[DOUBLE_START] == 'A')
      add_string("Astrometric binary");
    else
      ;

/* output the ADS number, if any */
    if (strncmp(&record[ADS_START],"    ",4) != EQUAL) {
      sprintf(&obj_info[idx],"    ADS %.4s",&record[ADS_START]);
      idx += strlen(&obj_info[idx]);
    }

/* output the identified components */
    if (strncmp(&record[MULTIPLE_START],"  ",2) != EQUAL) {
      advance_to_tab();
      sprintf(&obj_info[idx],"HR number represents component(s) %.2s",
	                                             &record[MULTIPLE_START]);
      idx += strlen(&obj_info[idx]);
    }

/* give the magnitude difference and separation, if specified */
    if (strncmp(&record[DELTA_MAG_START],"          ",10) != EQUAL) {
      blank_line();
      mag_flag = FALSE;
      if (strncmp(&record[DELTA_MAG_START],"    ",4) != EQUAL) {
	sprintf(&obj_info[idx],"Magnitude difference:  %.4s",
		                                    &record[DELTA_MAG_START]);
	idx += strlen(&obj_info[idx]);
	mag_flag = TRUE;
      }
      if (strncmp(&record[SEPARATION_START],"      ",6) != EQUAL) {
	if (mag_flag)
	  advance_to_tab();
	sprintf(&obj_info[idx],"Separation:  %.4s\"%.2s",
		                               &record[SEPARATION_START],
		                               &record[SEPARATION_START + 4]);
	idx += strlen(&obj_info[idx]);
      }
    }

/* put in some blankness */
    blank_line();
  }
  else
/* if this was an infrared source, we need a line terminator and blank line */
    if (ir_flag)
      blank_line();

/* display the position */
  display_pos(&record[POS_START]);

/* now display the magnitude */
  mag_flag = FALSE;
  if (strncmp(&record[MAG_START],"     ",5) != EQUAL) {
    next_line();
    add_string("Mag:  ");
    sprintf(&obj_info[idx],"%.5s",&record[MAG_START]);
    idx += strlen(&obj_info[idx]);
    mag_flag = TRUE;
  }

/* display variability information */
  if (record[VARIABLE_START] != ' ') {
    if (mag_flag)
      add_string(" - ");
    add_string("variable ");
/* there are basically 5 types of designation here:  two or three digits
 * followed by a constellation are a Bayer designation, V folowed by all
 * numeric followed by a constellation is an ordinal designation, all numeric
 * is a CSV designation, one or two letters followed by a constellation are
 * an Argelander designation, and VAR or VAR? for unnamed (suspected)
 * variables.  additionally, there is one lower-case letter followed by
 * a constellation. */
    i = 1;
    while (i < 9)
      /* find the first blank in the field */
      if (record[VARIABLE_START + i] == ' ')
	break;
      else
	i++;

    /* drop in a nul to mark the end of the first token */
    record[VARIABLE_START + i++] = '\0';

    /* find the start of the second token */
    while (i < 9)
      if (record[VARIABLE_START + i] != ' ')
	break;
      else
	i++;

    /* check to see if the first token is all numeric */
    j = 0;
    do {
      if (! isdigit(record[VARIABLE_START + j]))
	break;
      else
	j++;
    } while (record[VARIABLE_START + j] != '\0');

    /* if all numeric, set a flag */
    if (record[VARIABLE_START + j] == '\0')
      numeric_flag = TRUE;
    else
      numeric_flag = FALSE;

    /* now i points to the start of the second token, if any */
    if (i >= 9)
      /* no second token; either CSV or unnamed */
      if (strcmp(&record[VARIABLE_START],"VAR?") == EQUAL)
	sprintf(&obj_info[idx],"unnamed (suspected)");
      else if (strcmp(&record[VARIABLE_START],"VAR") == EQUAL)
	sprintf(&obj_info[idx],"unnamed");
      else
	/* if all numeric, this must be a CSV designation */
	if (numeric_flag)
	  sprintf(&obj_info[idx],"CSV %s",&record[VARIABLE_START]);
	else
	  ;
    else {
      /* second token exists; it must be a constellation */
      strncpy(const_buffer,&record[VARIABLE_START + i],3);
      const_buffer[3] = '\0';
      con = find_const_abbrev(const_buffer);
      if (con == -1)
	printf("bad constellation abbrev in VARIABLE field - record = %s\n",
	                                                              record);
      else
	if (numeric_flag) {
	  /* translate number to Bayer designation */
	  strncpy(bayer_buffer,&record[VARIABLE_START],2);
	  bayer_buffer[2] = '\0';
	  j = atoi(bayer_buffer);

	  /* watch out for a possible subscript */
	  if (isdigit(record[VARIABLE_START + 2]))
	    sprintf(&obj_info[idx],"%s-%c %s",greek[j - 1],
		                                   &record[VARIABLE_START + 2],
		                                   constellations_latin[con]);
	  else
	    sprintf(&obj_info[idx],"%s %s",greek[j - 1],
		                                   constellations_latin[con]);
	}
	else
	  /* handle Argelander or ordinal */
	  sprintf(&obj_info[idx],"%s %s",&record[VARIABLE_START],
		                                   constellations_latin[con]);
    }
    idx += strlen(&obj_info[idx]);
    blank_line();
  }
  else
    if (mag_flag)
/* space down some more */
      blank_line();

/* display color indices, if any */
  if (strncmp(&record[B_V_START],"               ",15) != EQUAL) {
    add_string("Color indices:");
    next_line();
    advance_to_tab();
    if (strncmp(&record[B_V_START],"     ",5) != EQUAL) {
      sprintf(&obj_info[idx],"B - V:  %.5s     ",&record[B_V_START]);
      idx += strlen(&obj_info[idx]);
    }
    if (strncmp(&record[U_B_START],"     ",5) != EQUAL) {
      sprintf(&obj_info[idx],"U - B:  %.5s     ",&record[U_B_START]);
      idx += strlen(&obj_info[idx]);
    }
    if (strncmp(&record[R_I_START],"     ",5) != EQUAL) {
      sprintf(&obj_info[idx],"R - I:  %.5s",&record[R_I_START]);
      idx += strlen(&obj_info[idx]);
    }
/* space down some more */
    blank_line();
  }

/* add the spectral class */
  sprintf(&obj_info[idx],"Spectral classification:  %.20s",
	                                             &record[SPECTRUM_START]);
  idx += strlen(&obj_info[idx]);
  blank_line();

/* list the proper motions */
  sprintf(&obj_info[idx],"Proper motion - RA:  %.2s\"%.4s",
	                                 &record[PROPER_MOTION_RA_START],
	                                 &record[PROPER_MOTION_RA_START + 2]);
  idx += strlen(&obj_info[idx]);
  advance_to_tab();
  sprintf(&obj_info[idx],"Dec:  %.2s\"%.4s",
	                                &record[PROPER_MOTION_DEC_START],
	                                &record[PROPER_MOTION_DEC_START + 2]);
  idx += strlen(&obj_info[idx]);

/* output parallax and radial velocity, if any */
  if (strncmp(&record[PARALLAX_START],"         ",9) != EQUAL) {
    /* space down some more */
    blank_line();
    if (strncmp(&record[PARALLAX_START],"     ",5) != EQUAL) {
      sprintf(&obj_info[idx],"Parallax:  %.1s0\"%.4s",
	                                         &record[PARALLAX_START],
	                                         &record[PARALLAX_START + 1]);
      idx += strlen(&obj_info[idx]);
      advance_to_tab();
      advance_to_tab();
    }

    if (strncmp(&record[RADIAL_VEL_START],"    ",4) != EQUAL) {
      /* scan the radial velocity magnitude */
      GET_FIELD(field,record,RADIAL_VEL_START + 1,3);
      rad_vel = atoi(field);
      sprintf(&obj_info[idx],"Radial velocity:  %.1s%d km/sec",
	                                   &record[RADIAL_VEL_START],rad_vel);
      idx += strlen(&obj_info[idx]);
    }
  }

/* fill the display out to the correct number of lines */
  while (row < NUM_YBS_ROWS)
    next_line();
  obj_info[idx] = '\0';

/* and go back */
  return;
}



static void display_pos(posptr)

char *posptr;

{
/* identify the epoch */
  add_string("Position (2000.0)");

/* fix up the RA minutes and seconds */
  if (posptr[2] == ' ')
    posptr[2] = '0';
  if (posptr[4] == ' ')
    posptr[4] = '0';

/* display the right ascension */
  advance_to_tab();
  sprintf(&obj_info[idx]," RA:   %.2sh %.2sm %.2ss%.2s",&posptr[0],&posptr[2],
	                                               &posptr[4],&posptr[6]);
  idx += strlen(&obj_info[idx]);
  next_line();

/* fix up the declination minutes and seconds */
  if (posptr[11] == ' ')
    posptr[11] = '0';
  if (posptr[13] == ' ')
    posptr[13] = '0';

/* and display the declination */
  advance_to_tab();
  advance_to_tab();
  advance_to_tab();
  sprintf(&obj_info[idx],"Dec:  %.3sd %.2s' %.2s\"",&posptr[8],&posptr[11],
	                                             &posptr[13],&posptr[15]);
  idx += strlen(&obj_info[idx]);
  next_line();

  return;
}
