/*
 * xsky - an interactive sky atlas
 *
 * Copyright 1992-6, Terry R. Friedrichsen
 *
 * This program may be copied and redistributed, in whole or in part,
 * as long as you don't try to make any money from the sale or redis-
 * tribution of the program or any part of the program, or pretend
 * that you wrote the program or any of its parts unless specifically
 * credited by the original author.
 *
 * You are free to make use of this software in your own programs, as
 * long as you credit the original author where it is due.
 */

/*
 * WARRANTY:
 * xsky was written as a learning project and as a demonstration of
 * X Window System programming.  xsky doesn't do anything; it is not
 * merchantable, and it is not fit for any purpose whatsoever.  In
 * fact, don't use xsky at all; it's free, and you're getting what
 * you paid for.
 */

#include <stdio.h>
#include <ctype.h>

#include <math.h>
#include <stdlib.h>

#include <unistd.h>

#include <string.h>

#include <X11/Intrinsic.h>

#include "wds.h"

#include "skydefs.h"

#include "pos.h"
#include "catalog.h"
#include "draw.h"
#include "catreader.h"

struct wds_node {
  double ra_rad;
  double dec_rad;
  int wds_number;
  float mag;
  long filepos;
  char spectrum;
};

/* static function prototypes */
static FILE *read_wds PROTOTYPE((void));
static void calculate_wds_field PROTOTYPE((struct display *,
					                struct cat_header *));
static boolean check_mag_limit PROTOTYPE((int,double));
static boolean find_wds_star PROTOTYPE((char *));
static void get_wds_pos PROTOTYPE((int,double *,double *));
static void draw_wds_star PROTOTYPE((Widget,int,int,int,int,int,int,int));
static boolean ps_draw_wds_star PROTOTYPE((struct display *,int,int,int));
static boolean display_wds PROTOTYPE((FILE *,int));
static void format_wds_name PROTOTYPE((int,char *,IDfont *));
static int mag_compare PROTOTYPE((struct wds_node *,struct wds_node *));
static void fill_wds_entry PROTOTYPE((char *,int));

/* external function prototypes */
extern char *build_filespec PROTOTYPE((char *,char *));
extern boolean read_binary_database PROTOTYPE((char *,void *,int *));
extern void init_precess PROTOTYPE((double,double));
extern void hs_heapsort PROTOTYPE((char [],size_t,size_t,int (*)()));
extern boolean write_binary_database PROTOTYPE((char *,void *,size_t,size_t));
extern double ra_pos_to_rad PROTOTYPE((struct ra_pos));
extern double dec_pos_to_rad PROTOTYPE((struct dec_pos));
extern void precess PROTOTYPE((double,double,double *,double *));
extern void add_to_display_list PROTOTYPE((struct display *,
					           struct cat_header *,
					                  int,double,double));
extern void draw_star PROTOTYPE((Widget,int,int,int,StarColor));
extern boolean ps_draw_star PROTOTYPE((struct display *,double,double,double));
extern void format_wds_display PROTOTYPE((char *,int));
extern void update_display_pos PROTOTYPE((double,double));

static struct wds_node wds_array[MAX_WDS];

static int wds_array_max;

extern float mag_limit;

/* AFAIK, these are the three possible ways the CD-ROM file name might be
 * represented by the operating system */
#define WDS_DATA_FILE_1   "WDS.DAT;1"
#define WDS_DATA_FILE_2   "WDS.DAT"
#define WDS_DATA_FILE_3   "wds.dat"

#define WDS_BINARY_FILE   "wds.bin"

/* double star record buffer */
static char wds_rec[WDS_RECLEN + 2 + 1];



/* initialize the catalog data structure */

void init_wds(catalog)

struct cat_header *catalog;

{
  catalog->read_data = read_wds;
  catalog->calculate_field = calculate_wds_field;
  catalog->mag_limit_check = check_mag_limit;
  catalog->find_object = find_wds_star;
  catalog->get_object_pos = get_wds_pos;
  catalog->draw_object = draw_wds_star;
  catalog->ps_draw_object = ps_draw_wds_star;
  catalog->display_data = display_wds;
  catalog->format_name = format_wds_name;
  catalog->menu_name = WDS_MENU_NAME;
  catalog->catalog_name = WDS_CATALOG_NAME;

  return;
}



/* read in the double star catalog file, extracting needed info */

static FILE *read_wds()

{
  char *filebuff;
  FILE *wds_fd;
  char *bin_filebuff;
  int wds_entry;

/* build the path to the double star data file and open it */
  filebuff = build_filespec("WDS_CD_PATH",WDS_DATA_FILE_1);
  if ((wds_fd = fopen(filebuff,"r")) == (FILE *)NULL) {
    free((void *)filebuff);
    filebuff = build_filespec("WDS_CD_PATH",WDS_DATA_FILE_2);
    if ((wds_fd = fopen(filebuff,"r")) == (FILE *)NULL) {
      free((void *)filebuff);
      filebuff = build_filespec("WDS_CD_PATH",WDS_DATA_FILE_3);
      if ((wds_fd = fopen(filebuff,"r")) == (FILE *)NULL) {
	printf("sky:  can't open double star data file %s\n",filebuff);
	perror("sky");
	exit(1);
      }
    }
  }

/* get the path to the double star binary file */
  bin_filebuff = build_filespec("WDS_BIN_PATH",WDS_BINARY_FILE);

/* see if we can read the binary file */
  if (access(bin_filebuff,R_OK) == 0)
    /* we can; try to use the pre-built database */
    if (read_binary_database(bin_filebuff,(void *)wds_array,&wds_array_max)) {
      /* success; free the filename buffers */
      free((void *)filebuff);
      free((void *)bin_filebuff);

      /* return the fd of the open data file */
      return(wds_fd);
    }

/* prior to reading the double star catalog, build the rotation matrix for
 * precessing the stellar coordinates to the common equinox from the given
 * epoch */
  if (WDS_EPOCH != COMMON_EQUINOX)
    init_precess(WDS_EPOCH,COMMON_EQUINOX);

/* read the double star catalog, saving information necessary for display */
  wds_entry = 0;
  while (TRUE) {
    wds_array[wds_entry].filepos = ftell(wds_fd);
    wds_array[wds_entry].wds_number = wds_entry + 1;
    if (fgets(wds_rec,sizeof(wds_rec),wds_fd) == (char *)NULL)
      break;
    fill_wds_entry(wds_rec,wds_entry++);
  }

/* leave the file open for later use */
/*  (void)fclose(wds_fd);*/

/* remember the last double star entry we filled */
  wds_array_max = wds_entry - 1;

/* now sort the double star array in decreasing order of magnitude */
  hs_heapsort((void *)wds_array,(size_t)wds_entry,
	                         (size_t)sizeof(struct wds_node),mag_compare);

/* attempt to save the binary database */
  if (access(bin_filebuff,W_OK) == 0)
    /* it's writeable, so let's try */
    if (! write_binary_database(bin_filebuff,(void *)wds_array,
				            (size_t)sizeof(struct wds_node),
				                           (size_t)wds_entry))
      /* dump of binary database failed; be sure the file is gone */
      (void)unlink(bin_filebuff);

/* free the filename buffers */
  free((void *)filebuff);
  free((void *)bin_filebuff);

  return(wds_fd);
}



/* compare two array entries for magnitude - compare routine for heapsort */

static int mag_compare(wds_entry_1,wds_entry_2)

struct wds_node *wds_entry_1, *wds_entry_2;

{
/* major key is the magnitude */
  if (wds_entry_1->mag < wds_entry_2->mag)
    return(-1);
  else if (wds_entry_1->mag > wds_entry_2->mag)
    return(1);
  else
    return(0);

/* NOTREACHED */
}



/* return TRUE if the magnitude of the star meets the limiting magnitude */

static boolean check_mag_limit(array_pos,limit)

int array_pos;
float limit;

{
  return((boolean)(wds_array[array_pos].mag <= limit));
}



/* given an object's index, return its right ascension and declination */

static void get_wds_pos(array_pos,ra_rad,dec_rad)

int array_pos;
double *ra_rad, *dec_rad;

{
  *ra_rad = wds_array[array_pos].ra_rad;
  *dec_rad = wds_array[array_pos].dec_rad;

  return;
}



/* given a double star record, make a data node for it */

static void fill_wds_entry(record,wds_entry)

char *record;
int wds_entry;

{
  int reclen;
  char field[11];
  struct ra_pos ra_pos;
  struct dec_pos dec_pos;
  char mag_buff[4 + 1];

/* get the length of the record for GET_FIELD */
  reclen = WDS_RECLEN;

/* get the RA in pieces */
  GET_FIELD(field,record,POS_RA_START,2);
  ra_pos.hours = (smallint)atoi(field);

  GET_FIELD(field,record,POS_RA_START + 2,2);
  ra_pos.minutes = (smallint)atoi(field);

  GET_FIELD(field,record,POS_RA_START + 4,1);
  ra_pos.seconds = (smallint)atoi(field) * 6;

  ra_pos.thousandths = (shortint)0;

/* similarly, get the declination in pieces */
  GET_FIELD(field,record,POS_DEC_START,1);
  dec_pos.sign = (short)1;
  if (field[0] == '-')
    dec_pos.sign = (short)-1;

  GET_FIELD(field,record,POS_DEC_START + 1,2);
  dec_pos.degrees = (smallint)atoi(field);

  GET_FIELD(field,record,POS_DEC_START + 3,2);
  dec_pos.minutes = (smallint)atoi(field);

  dec_pos.seconds = (smallint)0;
  dec_pos.hundredths = (smallint)0;

/* compute the RA and declination in radians */
  wds_array[wds_entry].ra_rad = ra_pos_to_rad(ra_pos);
  wds_array[wds_entry].dec_rad = dec_pos_to_rad(dec_pos);

/* precess them to the common equinox */
  if (WDS_EPOCH != COMMON_EQUINOX)
    precess(wds_array[wds_entry].ra_rad,wds_array[wds_entry].dec_rad,
	          &wds_array[wds_entry].ra_rad,&wds_array[wds_entry].dec_rad);

/* get the magnitude */
  GET_FIELD(field,record,MAG_PRIMARY_START,3);
  if (strcmp(field,"   ") == EQUAL)
    wds_array[wds_entry].mag = 99.99;
  else {
    strncpy(mag_buff,field,2);
    mag_buff[2] = '.';
    strcpy(&mag_buff[3],&field[2]);
    wds_array[wds_entry].mag = atof(mag_buff);
  }

/* get the spectral class */
  GET_FIELD(field,record,SPECTRUM_START,4);
  wds_array[wds_entry].spectrum = field[0];
  if (wds_array[wds_entry].spectrum == ' ')
    wds_array[wds_entry].spectrum = field[2];
  /* watch out for the one with a spectral class of '0' */
  if (wds_array[wds_entry].spectrum == '0')
    /* speculate that O was meant instead */
    wds_array[wds_entry].spectrum = 'O';

  return;
}



/* construct a list of those stars which appear in the current display */

static void calculate_wds_field(display,catalog)

struct display *display;
struct cat_header *catalog;

{
  int i;

/* loop through all double star entries, checking for visibility */
  for (i = 0; i <= wds_array_max; i++)
    /* check this star and add it to the list if visible */
    add_to_display_list(display,catalog,i,wds_array[i].ra_rad,
                                                       wds_array[i].dec_rad);

#ifdef DEBUG
  printf("calculate_wds_field returning\n");
#endif
  return;
}



/* draw a double star on the sky */

static void draw_wds_star(sky_widget,array_pos,obj_x,obj_y,ul_x,ul_y,
		                                                 width,height)

Widget sky_widget;
int array_pos;
int obj_x, obj_y;
int ul_x, ul_y;
int width, height;

{
  int size;
  StarColor color;

/* be sure it is bright enough to display */
  if (! check_mag_limit(array_pos,mag_limit))
    return;

  /* calculate size based on magnitude */
  size = STARSIZE(wds_array[array_pos].mag);

  /* draw the object even if it is only partially in the box */
  if (((obj_x + (size + 1) / 2) >= ul_x) &&
                               ((obj_x - (size + 1) / 2) <= (ul_x + width)) &&
                               ((obj_y + (size + 1) / 2) >= ul_y) &&
                               ((obj_y - (size + 1) / 2) <= (ul_y + height)))
    ;
  else
    return;

  /* assign a color based on the spectral class */
  switch (wds_array[array_pos].spectrum) {
  case 'W':
  case 'O':
    color = O;
    break;
  case 'B':
    color = B;
    break;
  case 'A':
    color = A;
    break;
  case 'F':
    color = F;
    break;
  case 'G':
    color = G;
    break;
  case 'K':
    color = K;
    break;
  case 'M':
  case 'C':
  case 'R':
  case 'N':
  case 'S':
    color = M;
    break;
  default:
    printf("WDS star has unrecognized spectral class %c\n",
	                                      wds_array[array_pos].spectrum);
  case 'P':        /* some stars are peculiar */
  case 'D':        /* some I have no idea of the meaning */
  case ' ':        /* and some are just not given */
    color = A;     /* make unknowns white */
    break;
  }

/* draw the star with correct location, size, and color */
  draw_star(sky_widget,obj_x,obj_y,size,color);

  return;
}



/* draw one double star on the PostScript display */

static boolean ps_draw_wds_star(display,array_pos,x,y)

struct display *display;
int array_pos;
int x, y;

{
  int size;

/* be sure it is bright enough to display */
  if (! check_mag_limit(array_pos,mag_limit))
    return(TRUE);

/* calculate size based on magnitude */
  size = STARSIZE(wds_array[array_pos].mag);

/* draw the star in the correct position with the correct size */
  return(ps_draw_star(display,(double)x,(double)y,(double)size));
}



/* display the catalog data for the given star */

static boolean display_wds(wds_fd,wds_array_pos)

FILE *wds_fd;
int wds_array_pos;

{
/* get info from the double star catalog */
  if (fseek(wds_fd,wds_array[wds_array_pos].filepos,SEEK_SET) == -1)
    return(FALSE);
  if (fgets(wds_rec,sizeof(wds_rec),wds_fd) == (char *)NULL)
    return(FALSE);

/* format the record into an informational display */
  format_wds_display(wds_rec,wds_array[wds_array_pos].wds_number);

  return(TRUE);
}



/* find the double star in the buffer and update the display coordinates */

static boolean find_wds_star(buffer)

char *buffer;

{
  int wds_no;
  int i;

/* be sure this is a WDS designation */
  if ((strncmp(buffer,"WDS",3) == EQUAL) || strncmp(buffer,"wds",3) == EQUAL)
    buffer = &buffer[3];
  else
    return(FALSE);

/* get the WDS number */
  wds_no = atoi(buffer);

/* loop through the array looking for this number */
  for (i = 0; i <= wds_array_max; i++)
    if (wds_array[i].wds_number == wds_no)
      break;

/* return failure if we didn't find it */
  if (i > wds_array_max)
    return(FALSE);
  else {
    /* update the display coordinates */
    update_display_pos(wds_array[i].ra_rad,wds_array[i].dec_rad);
    return(TRUE);
  }

/* NOTREACHED */
}



/* format a nice name for a star */

/* ARGSUSED */

static void format_wds_name(wds_pos,buffer,font)

int wds_pos;
char *buffer;
IDfont *font;

{
/* use the WDS number */
  sprintf(buffer,"WDS %d",wds_array[wds_pos].wds_number);

  return;
}
