/*
 * xsky - an interactive sky atlas
 *
 * Copyright 1992-6, Terry R. Friedrichsen
 *
 * This program may be copied and redistributed, in whole or in part,
 * as long as you don't try to make any money from the sale or redis-
 * tribution of the program or any part of the program, or pretend
 * that you wrote the program or any of its parts unless specifically
 * credited by the original author.
 *
 * You are free to make use of this software in your own programs, as
 * long as you credit the original author where it is due.
 */

/*
 * WARRANTY:
 * xsky was written as a learning project and as a demonstration of
 * X Window System programming.  xsky doesn't do anything; it is not
 * merchantable, and it is not fit for any purpose whatsoever.  In
 * fact, don't use xsky at all; it's free, and you're getting what
 * you paid for.
 */

#include <stdio.h>

#include <string.h>
#include <ctype.h>

#include "skydefs.h"

#include "greek.h"

#include "constell.h"

/* static function prototypes */
static int const_abbrev_compar PROTOTYPE((void *,void *,int));
static char *strsub PROTOTYPE((char *,char *,int,int));

/* external function prototypes */
extern int binsearch PROTOTYPE((void *,int,void *,int (*)()));

/* global function prototypes */
int find_const_abbrev PROTOTYPE((char *));


/* match the buffer with a constellation name or abbreviation, returning a
 * zero-relative index into the constellation list, or -1 on failure */

int find_constellation(buffer)

char *buffer;

{
  char *ptr;
  int i;

/* skip leading blanks */
  while (isspace(*buffer))
    buffer++;
  ptr = buffer;

/* loop through the rest of the words in the buffer */
  while (*ptr != '\0') {
    /* upcase the first character of the word */
    *ptr = toupper(*ptr);
    ptr++;

    /* downcase the remainder of the word */
    while (! isspace(*ptr = tolower(*ptr)))
      if (*ptr == '\0')
	break;
      else
	ptr++;

    /* skip to the start of the next word */
    while (isspace(*ptr))
      ptr++;
  }

/* find the index of the Latin genitive form of the constellation name */
  for (i = 0; i < NUM_CONSTELLATIONS; i++)
    if (strcmp(buffer,constellations_latin[i]) == EQUAL)
      break;

  /* if we didn't find a match, search the constellation abbreviations */
  if (i >= NUM_CONSTELLATIONS)
    i = find_const_abbrev(buffer);

/* return the constellation index */
  return(i);
}



/* search the constellation abbreviation table, returning the zero-relative
 * index into the table, or -1 on failure */

int find_const_abbrev(buffer)

char *buffer;

{
  int abbrev_idx;

/* skip leading blanks */
  while (isspace(*buffer))
    buffer++;

/* find the matching constellation abbreviation */
  abbrev_idx = binsearch((void *)constellations_abbrev,
			                  NUM_CONSTELLATIONS - 1,
			                  (void *)buffer,const_abbrev_compar);

#ifdef DEBUG
  if (abbrev_idx == -1)
    printf("no constellation found to match %s\n",buffer);
#endif

  return(abbrev_idx);

/* NOTREACHED */
}



/* comparison routine for binary search to find constellation abbreviation */

static int const_abbrev_compar(table,key,idx)

void *table;
void *key;
int idx;

{
  return(strcasecmp((char *)key,((char **)table)[idx]));
}



/* return the zero-relative index of the Greek letter in the buffer, or -1
 * on failure */

int find_greek_letter(letter)

char *letter;

{
  int i;

/* search the Greek letter table */
  for (i = 0; i < NUM_GREEK_LETTERS; i++)
    if (strcmp(letter,greek[i]) == EQUAL)
      break;

/* if we didn't find a match, return -1 */
  if (i >= NUM_GREEK_LETTERS)
    i = -1;

  return(i);
}



/* see if the given token is in the given delimited list, with a case-
 * insensitive comparison */

static char temp_list[2000];


boolean find_tok(list,token,delim)

char *list;
char *token;
char *delim;

{
  char *tok;
  char *templist;
  int i;

/* copy the list since strtok destroys it */
  strcpy(temp_list,list);

/* loop through the list, looking for this token */
  templist = temp_list;
  do {
    /* get the next token from the list */
    if ((tok = strtok(templist,delim)) == (char *)NULL)
      break;

    /* skip leading white space */
    while (isspace(*tok))
      tok++;

    /* and trim trailing white space */
    i = strlen(tok) - 1;
    while (isspace(tok[i]))
      if (--i < 0)
	break;
    tok[++i] = '\0';

    /* null out the pointer so strtok keeps going */
    templist = (char *)NULL;
  } while (strcasecmp(tok,token) != EQUAL);

/* return success or failure */
  return(! (tok == (char *)NULL));
}



/*
 * strnstr checks to see if string b is a substring of the first n
 * characters of string a, returning a pointer to the match, or NULL
 * if no match.
 *
 * note that since n specifies the length of the longer string, a, the
 * shorter string, b, MUST be null-terminated.
 */

char *strnstr(a,b,n)

char *a, *b;
int n;

{
/* call the generic search routine with the appropriate pointers and lengths */
  return(strsub(a,b,n,(int)strlen(b)));
}



static char *strsub(a,b,i,j)

char *a, *b;
int i, j;

{
  short match;
  char *x, *y, *s;

/* start the search at the beginning of the longer string */
  s = a;

/* loop through most characters of the longer string.  stop looping
 * when there aren't enough characters left in the longer string to
 * accomodate all the characters in the shorter string.  is that clear
 * at all, or should I have tried English?
 * note that, the first time through, this test ensures that the
 * substring being searched for is not longer than the string we
 * are looking in.
 */
  while ((a + i - s) >= j) {	/* enough characters left? */
    /* assume we will find a matching substring */
    match = TRUE;
    /* reset the pointer to the beginning of the shorter string */
    x = b;
    /* set the pointer to where to begin searching in the longer string */
    y = s;

/* loop through all the characters in the shorter string */
    while (*x)
/* stop at the first sign of a non-matching character */
      if (*x++ != *y++) {
	/* flag that this substring does not match */
	match = FALSE;
	break;
      }

/* did the substring match? */
    if (match)
/* yes, return pointer to position in longer string at which match was found */
      return(s);
    else
/* no, try to find a match one character farther down the longer string */
      s++;
  }

/* if we fall out, nothing matched.  return a bogus character position */
  return(NULL);
}
