/*
 * xsky - an interactive sky atlas
 *
 * Copyright 1992-4, Terry R. Friedrichsen
 *
 * This program may be copied and redistributed, in whole or in part,
 * as long as you don't try to make any money from the sale or redis-
 * tribution of the program or any part of the program, or pretend
 * that you wrote the program or any of its parts unless specifically
 * credited by the original author.
 *
 * You are free to make use of this software in your own programs, as
 * long as you credit the original author where it is due.
 */

/*
 * WARRANTY:
 * xsky was written as a learning project and as a demonstration of
 * X Window System programming.  xsky doesn't do anything; it is not
 * merchantable, and it is not fit for any purpose whatsoever.  In
 * fact, don't use xsky at all; it's free, and you're getting what
 * you paid for.
 */

/*
 * The user catalog is intended to allow the user to add temporary
 * objects to the sky display; in particular, it is not really well-
 * suited to augmenting the available catalogs with new objects.  The
 * idea is that the user will wish to enter positions for, say, a
 * comet or a planet or an asteroid, so as to be able to prepare a
 * finder chart for use at the telescope.
 *
 * The format of the user catalog is:
 *
 * catalog name
 * equinox
 * user object number\RA, dec\label\description
 *
 * The catalog name is up to 80 characters long, for chart display.
 * The equinox is the equinox of the positions given in the user
 * catalog.  The position will be precessed to xsky's common equinox.
 * The user object number must be a decimal integer; it's value and
 * meaning are up to the user to decide.  The RA and declination are
 * in the form nnh nnm nns.nnn, nnd nn' nn".nn where leading zeroes
 * not be specified and the seconds may have fewer decimal places or
 * be omitted altogether.  The label is the label that xsky will place
 * on the object in the display; this will usually be an object name
 * or a date. The description field can be used to contain the rest
 * of the relevant information.  The description can be up to 400 char-
 * acters long; it will be broken into 80-column lines for the object
 * information display.
 */

#include <stdio.h>

#include <stdlib.h>

#include <unistd.h>

#include <string.h>

#include <X11/Intrinsic.h>

#include "skydefs.h"

#include "pos.h"
#include "catalog.h"
#include "draw.h"
#include "catreader.h"

struct user_node {
  double ra_rad;
  double dec_rad;
  long filepos;
  shortint user_number;
};

/* static function prototypes */
static FILE *read_user PROTOTYPE((void));
static void calculate_user_field PROTOTYPE((struct display *,
					                struct cat_header *));
static boolean find_user_obj PROTOTYPE((char *));
static void get_user_pos PROTOTYPE((int,double *,double *));
static void draw_user_obj PROTOTYPE((Widget,int,int,int,int,int,int,int));
static boolean ps_draw_user_obj PROTOTYPE((struct display *,int,int,int));
static boolean display_user PROTOTYPE((FILE *,int));
static void format_user_name PROTOTYPE((int,char *,IDfont *));
static int fill_user_entry PROTOTYPE((char *,int));
static void format_user_display PROTOTYPE((char *));

/* external function prototypes */
extern char *build_filespec PROTOTYPE((char *,char *));
extern void init_precess PROTOTYPE((double,double));
extern boolean get_ra PROTOTYPE((char *,struct ra_pos *));
extern boolean get_dec PROTOTYPE((char *,struct dec_pos *));
extern double ra_pos_to_rad PROTOTYPE((struct ra_pos));
extern double dec_pos_to_rad PROTOTYPE((struct dec_pos));
extern void precess PROTOTYPE((double,double,double *,double *));
extern void add_to_display_list PROTOTYPE((struct display *,
					           struct cat_header *,
					                  int,double,double));
extern void pos_rad_to_X PROTOTYPE((struct display *,double,double,
				                                int *,int *));
extern void draw_object PROTOTYPE((Widget,int,int,int,ObjShape));
extern boolean ps_draw_object PROTOTYPE((struct display *,double,double,
					                    double,ObjShape));
extern void update_display_pos PROTOTYPE((double,double));

static char user_catalog_name[80 + 1 + 1];

#define USER_MENU_NAME      "User"

#define USER_RECLEN         2048

#define MAX_USER            200

static struct user_node user_array[MAX_USER];

static int user_array_max;

#define USER_DATA_FILE   "usercat.dat"

static FILE *user_fd;

static long usercat_obj_filepos;

static float user_epoch;

/* user record buffer */
static char user_rec[USER_RECLEN + 1 + 1];



void init_user(catalog)

struct cat_header *catalog;

{
  catalog->read_data = read_user;
  catalog->calculate_field = calculate_user_field;
  catalog->mag_limit_check = NULLBOOLFUNCTION;
  catalog->find_object = find_user_obj;
  catalog->get_object_pos = get_user_pos;
  catalog->draw_object = draw_user_obj;
  catalog->ps_draw_object = ps_draw_user_obj;
  catalog->display_data = display_user;
  catalog->format_name = format_user_name;
  catalog->menu_name = USER_MENU_NAME;
  catalog->catalog_name = user_catalog_name;

  return;
}



/* return TRUE if a readable user catalog exists */

boolean check_user_cat()

{
  char *filebuff;
  boolean retval;

/* get the path to the user data */
  filebuff = build_filespec("USER_PATH",USER_DATA_FILE);

/* determine readability of the file */
  retval = (access(filebuff,R_OK) == 0);

/* free the filename buffer */
  free((void *)filebuff);

/* and go back */
  return(retval);
}



/* read in the user catalog file, extracting needed info */

static FILE *read_user()

{
  char *filebuff;
  int user_entry;

/* get the path to the user data */
  filebuff = build_filespec("USER_PATH",USER_DATA_FILE);

/* open the user data file */
  if ((user_fd = fopen(filebuff,"r")) == (FILE *)NULL) {
#ifdef DEBUG
    printf("sky:  can't open user data file %s\n",filebuff);
    perror("sky");
    exit(1);
#else
    return((FILE *)NULL);
#endif
  }

/* read the catalog name */
  if (fgets(user_catalog_name,sizeof(user_catalog_name),user_fd) ==
                                                               (char *)NULL) {
    printf("sky:  can't read user data file %s\n",filebuff);
    perror("sky");
    exit(1);
  }
  /* remove the trailing newline */
  user_catalog_name[strlen(user_catalog_name) - 1] = '\0';

/* read the epoch from the user catalog */
  if (fgets(user_rec,sizeof(user_rec),user_fd) == (char *)NULL) {
    printf("sky:  can't read user data file %s\n",filebuff);
    perror("sky");
    exit(1);
  }
  user_epoch = atof(user_rec);

/* we are now positioned at the first object record - save the position */
  usercat_obj_filepos = ftell(user_fd);

/* prior to reading the user catalog, build the rotation matrix for precessing
 * the object coordinates to the common equinox from the given epoch */
  if (user_epoch != COMMON_EQUINOX)
    init_precess(user_epoch,COMMON_EQUINOX);

/* read the user catalog, saving the information necessary for the display */
  user_entry = 0;
  while (TRUE) {
    user_array[user_entry].filepos = ftell(user_fd);
    if (fgets(user_rec,sizeof(user_rec),user_fd) == (char *)NULL)
      break;
    user_entry = fill_user_entry(user_rec,user_entry);
  }

/* leave the file open for later use */
/*  (void)fclose(user_fd);*/

/* remember the last user entry we filled */
  user_array_max = user_entry - 1;

/* free the filename buffer */
  free((void *)filebuff);

  return(user_fd);
}



/* given an object's index, return its right ascension and declination */

static void get_user_pos(array_pos,ra_rad,dec_rad)

int array_pos;
double *ra_rad, *dec_rad;

{
  *ra_rad = user_array[array_pos].ra_rad;
  *dec_rad = user_array[array_pos].dec_rad;

  return;
}



/* given a user record, make a data node for it */

static int fill_user_entry(record,user_entry)

char *record;
int user_entry;

{
  char *ptr;
  struct ra_pos ra_pos;
  struct dec_pos dec_pos;

/* get the user object number */
  ptr = strtok(record,"\\");
  if (strlen(ptr) == 0)
    /* do sequential numbering automatically for the user */
    user_array[user_entry].user_number = user_entry;
  else
    user_array[user_entry].user_number = atoi(ptr);

/* crack the RA and dec strings */
  ptr = strtok((char *)NULL,",");
  if (! get_ra(ptr,&ra_pos)) {
    printf("user object number %d has malformed RA\n",
	                                   user_array[user_entry].user_number);
    return(user_entry);
  }

  ptr = strtok((char *)NULL,"\\");
  if (! get_dec(ptr,&dec_pos)) {
    printf("user object number %d has malformed declination\n",
	                                   user_array[user_entry].user_number);
    return(user_entry);
  }

/* compute the RA and declination in radians */
  user_array[user_entry].ra_rad = ra_pos_to_rad(ra_pos);
  user_array[user_entry].dec_rad = dec_pos_to_rad(dec_pos);

/* precess them to the common equinox */
  if (user_epoch != COMMON_EQUINOX)
    precess(user_array[user_entry].ra_rad,user_array[user_entry].dec_rad,
	       &user_array[user_entry].ra_rad,&user_array[user_entry].dec_rad);

  return(++user_entry);
}



/* construct a list of those objects which appear in the current display */

static void calculate_user_field(display,catalog)

struct display *display;
struct cat_header *catalog;

{
  int i;

/* loop through all user entries, checking for visibility */
  for (i = 0; i <= user_array_max; i++)
    /* check this star and add it to the list if visible */
    add_to_display_list(display,catalog,i,user_array[i].ra_rad,
                                                       user_array[i].dec_rad);

#ifdef DEBUG
  printf("calculate_user_field returning\n");
#endif
  return;
}



/* draw a user object on the sky */

/* ARGSUSED */

static void draw_user_obj(sky_widget,array_pos,obj_x,obj_y,ul_x,ul_y,
		                                                 width,height)

Widget sky_widget;
int array_pos;
int obj_x, obj_y;
int ul_x, ul_y;
int width, height;

{
 /* draw the object even if it is only partially in the box */
  if (((obj_x + (OBJSIZE + 1) / 2) >= ul_x) &&
                            ((obj_x - (OBJSIZE + 1) / 2) <= (ul_x + width)) &&
                            ((obj_y + (OBJSIZE + 1) / 2) >= ul_y) &&
                            ((obj_y - (OBJSIZE + 1) / 2) <= (ul_y + height)))
    ;
  else
    return;

  /* draw an x on the spot */
  draw_object(sky_widget,obj_x,obj_y,0,XMARK);

  return;
}



/* draw one user object on the PostScript display */

/* ARGSUSED */

static boolean ps_draw_user_obj(display,array_pos,x,y)

struct display *display;
int array_pos;
int x, y;

{
  return(ps_draw_object(display,(double)x,(double)y,(double)0,XMARK));
}



/* display the catalog data for the given object */

static boolean display_user(user_fd,user_array_pos)

FILE *user_fd;
int user_array_pos;

{
  int i;

/* get info from the user catalog */
  if (fseek(user_fd,user_array[user_array_pos].filepos,SEEK_SET) == -1)
    return(FALSE);
  if (fgets(user_rec,sizeof(user_rec),user_fd) == (char *)NULL)
    return(FALSE);

/* fill out the (possibly short) record with blanks */
  for (i = strlen(user_rec) - 1; i < USER_RECLEN; i++)
    user_rec[i] = ' ';

/* format the record into an informational display */
  format_user_display(user_rec);

  return(TRUE);
}



/* find the user object in the buffer and update the display coordinates */

static boolean find_user_obj(buffer)

char *buffer;

{
  int i, j;
  shortint usernum;
  char *label;
  char *ptr;

/* be sure this is a user designation */
  if ((strncmp(buffer,"user",4) == EQUAL) ||
                                       (strncmp(buffer,"USER",4) == EQUAL)) {
    buffer = &buffer[4];

    /* get the index of the user object */
    i = atoi(buffer);

    /* find this user object number */
    for (j = 0; j <= user_array_max; j++)
      if (user_array[j].user_number == i)
	break;

    /* update the display coordinates if found */
    if (j <= user_array_max) {
      update_display_pos(user_array[j].ra_rad,user_array[j].dec_rad);
      return(TRUE);
    }
    else
      /* otherwise, give up */
      return(FALSE);
  }
  else {
    /* we're the last resort - look through the user's labels for a match */
    if (fseek(user_fd,usercat_obj_filepos,SEEK_SET) == -1)
      return(FALSE);
    while (fgets(user_rec,sizeof(user_rec),user_fd) != (char *)NULL) {
      /* save the user number for this record */
      usernum = (shortint)atoi(strtok(user_rec,"\\"));

      /* find the label field */
      label = strtok((char *)NULL,"\\");
      label = strtok((char *)NULL,"\\");

      /* skip this record if the label is blank */
      ptr = label;
      while (isspace(*ptr))
	ptr++;
      if (*ptr == '\0')
	continue;

      /* see if it matches what we're looking for (case-insensitive compare) */
      if (strcasecmp(label,buffer) == EQUAL) {
	/* it matches; find the entry with the matching user number */
	for (i = 0; i <= user_array_max; i++)
	  if (user_array[i].user_number == usernum) {
	    update_display_pos(user_array[i].ra_rad,user_array[i].dec_rad);
	    return(TRUE);
	  }

	/* if we fall out of the loop, no user number matched (this can happen
	 * only if one of the user records was bogus and therefore ignored) */
	break;
      }
    }

    /* if we fall out of the loop, none of the labels matched */
    return(FALSE);
  }

/* NOTREACHED */
}



/* format a nice name for the object */

/* ARGSUSED */

static void format_user_name(user_pos,buffer,font)

int user_pos;
char *buffer;
IDfont *font;

{
  char *ptr;

/* get the name from the user record */
  if (fseek(user_fd,user_array[user_pos].filepos,SEEK_SET) == -1)
    return;
  if (fgets(user_rec,sizeof(user_rec),user_fd) == (char *)NULL)
    return;

/* use the third token from the record */
  ptr = strtok(user_rec,"\\");
  ptr = strtok((char *)NULL,"\\");
  ptr = strtok((char *)NULL,"\\");
  if (ptr == (char *)NULL)
    /* if unavailable, use the user record number */
    sprintf(buffer,"User %d",user_array[user_pos].user_number);
  else
    strcpy(buffer,ptr);

  return;
}



/* this routine formats and displays a user record */

#include "format.h"

#define MAX_USER_ROW   80
#define NUM_USER_ROWS  12



static void format_user_display(record)

char *record;

{
  char *posptr;
  char *ptr;
  char *ptr2;

/* start at the beginning of the display */
  idx = 0;
  row = 0;
  row_start = 0;
  max_row = MAX_USER_ROW;

/* display the catalog number */
  sprintf(&obj_info[idx],"User object %s",strtok(record,"\\"));
  idx += strlen(&obj_info[idx]);

/* save a pointer to the position */
  posptr = strtok((char *)NULL,"\\");

/* add the label to the info display */
  sprintf(&obj_info[idx]," - %s",strtok((char *)NULL,"\\"));
  idx += strlen(&obj_info[idx]);
  blank_line();

/* display the position */
  add_string("Position:  ");
  add_string(posptr);
  blank_line();

/* point to the description */
  ptr = strtok((char *)NULL,"\\");

/* put in a label */
  add_string("Description:");
  blank_line();

/* break the description into < MAX_USER_ROW pieces */
  while (TRUE) {
    /* find the place to end this row */
    ptr2 = &ptr[min(strlen(ptr),MAX_USER_ROW)];
    /* special handling for the last piece */
    if (*ptr2 == '\0') {
      add_string(ptr);
      break;
    }
    else {
      /* move backwards to white space */
      while (! isspace(*ptr2))
	ptr2--;
      /* plant a nul and advance to the next printing character */
      *ptr2++ = '\0';
      while (isspace(*ptr2))
	ptr2++;
      /* put out this piece of the description */
      add_string(ptr);
      /* advance to the next line */
      next_line();
      /* and advance to the next piece */
      ptr = ptr2;
    }
  }

/* fill the display out to the correct number of lines */
  while (row < NUM_USER_ROWS)
    next_line();
  obj_info[idx] = '\0';

/* and go back */
  return;
}
