/*
 * xsky - an interactive sky atlas
 *
 * Copyright 1995, Terry R. Friedrichsen
 *
 * This program may be copied and redistributed, in whole or in part,
 * as long as you don't try to make any money from the sale or redis-
 * tribution of the program or any part of the program, or pretend
 * that you wrote the program or any of its parts unless specifically
 * credited by the original author.
 *
 * You are free to make use of this software in your own programs, as
 * long as you credit the original author where it is due.
 */

/*
 * WARRANTY:
 * xsky was written as a learning project and as a demonstration of
 * X Window System programming.  xsky doesn't do anything; it is not
 * merchantable, and it is not fit for any purpose whatsoever.  In
 * fact, don't use xsky at all; it's free, and you're getting what
 * you paid for.
 */

#include <stdio.h>
#include <ctype.h>

#include <math.h>
#include <stdlib.h>

#include <unistd.h>

#include <string.h>

#include <X11/Intrinsic.h>

#include "ugc.h"

#include "skydefs.h"

#include "pos.h"
#include "catalog.h"
#include "draw.h"
#include "catreader.h"

struct ugc_node {
  double ra_rad;
  double dec_rad;
  long filepos;
  shortint ugc_num;
};

/* static function prototypes */
static FILE *read_ugc PROTOTYPE((void));
static void calculate_ugc_field PROTOTYPE((struct display *,
					                struct cat_header *));
static boolean find_ugc_obj PROTOTYPE((char *));
static void get_ugc_pos PROTOTYPE((int,double *,double *));
static void draw_ugc_obj PROTOTYPE((Widget,int,int,int,int,int,int,int));
static boolean ps_draw_ugc_obj PROTOTYPE((struct display *,int,int,int));
static boolean display_ugc PROTOTYPE((FILE *,int));
static void format_ugc_name PROTOTYPE((int,char *,IDfont *));
static void fill_ugc_entry PROTOTYPE((char *,int));

/* external function prototypes */
extern char *build_filespec PROTOTYPE((char *,char *));
extern boolean read_binary_database PROTOTYPE((char *,void *,int *));
extern void init_precess PROTOTYPE((double,double));
extern boolean write_binary_database PROTOTYPE((char *,void *,size_t,size_t));
extern double ra_pos_to_rad PROTOTYPE((struct ra_pos));
extern double dec_pos_to_rad PROTOTYPE((struct dec_pos));
extern void precess PROTOTYPE((double,double,double *,double *));
extern void add_to_display_list PROTOTYPE((struct display *,
					           struct cat_header *,
				                          int,double,double));
extern void draw_object PROTOTYPE((Widget,int,int,int,ObjShape));
extern boolean ps_draw_object PROTOTYPE((struct display *,double,double,
					                    double,ObjShape));
extern void format_ugc_display PROTOTYPE((char *));
extern void update_display_pos PROTOTYPE((double,double));

static struct ugc_node ugc_array[MAX_UGC];

static int ugc_array_max;

/* AFAIK, these are the three possible ways the CD-ROM file name might be
 * represented by the operating system */
#define UGC_DATA_FILE_1   "UGC.DAT;1"
#define UGC_DATA_FILE_2   "UGC.DAT"
#define UGC_DATA_FILE_3   "ugc.dat"

#define UGC_BINARY_FILE   "ugc.bin"

/* UGC record buffer */
static char ugc_rec[UGC_RECLEN + 1 + 1];



void init_ugc(catalog)

struct cat_header *catalog;

{
  catalog->read_data = read_ugc;
  catalog->calculate_field = calculate_ugc_field;
  catalog->mag_limit_check = NULLBOOLFUNCTION;
  catalog->find_object = find_ugc_obj;
  catalog->get_object_pos = get_ugc_pos;
  catalog->draw_object = draw_ugc_obj;
  catalog->ps_draw_object = ps_draw_ugc_obj;
  catalog->display_data = display_ugc;
  catalog->format_name = format_ugc_name;
  catalog->menu_name = UGC_MENU_NAME;
  catalog->catalog_name = UGC_CATALOG_NAME;

  return;
}



/* read in the UGC catalog file, extracting needed info */

static FILE *read_ugc()

{
  char *filebuff;
  FILE *ugc_fd;
  char *bin_filebuff;
  int ugc_entry;

/* build the path to the UGC data file and open it */
  filebuff = build_filespec("UGC_CD_PATH",UGC_DATA_FILE_1);
  if ((ugc_fd = fopen(filebuff,"r")) == (FILE *)NULL) {
    free((void *)filebuff);
    filebuff = build_filespec("UGC_CD_PATH",UGC_DATA_FILE_2);
    if ((ugc_fd = fopen(filebuff,"r")) == (FILE *)NULL) {
      free((void *)filebuff);
      filebuff = build_filespec("UGC_CD_PATH",UGC_DATA_FILE_3);
      if ((ugc_fd = fopen(filebuff,"r")) == (FILE *)NULL) {
	printf("sky:  can't open UGC data file %s\n",filebuff);
	perror("sky");
	exit(1);
      }
    }
  }

/* get the path to the UGC binary file */
  bin_filebuff = build_filespec("UGC_BIN_PATH",UGC_BINARY_FILE);

/* see if we can read the binary file */
  if (access(bin_filebuff,R_OK) == 0)
    /* we can; try to use the pre-built database */
    if (read_binary_database(bin_filebuff,ugc_array,&ugc_array_max)) {
      /* success; free the filename buffers */
      free((void *)filebuff);
      free((void *)bin_filebuff);

      /* return the fd of the open data file */
      return(ugc_fd);
    }

/* prior to reading the UGC catalog, build the rotation matrix for precessing
 * the object coordinates to the common equinox from the given epoch */
  if (UGC_EPOCH != COMMON_EQUINOX)
    init_precess(UGC_EPOCH,COMMON_EQUINOX);

/* read the UGC catalog, saving the information necessary for the display */
  ugc_entry = 0;
  while (TRUE) {
    ugc_array[ugc_entry].filepos = ftell(ugc_fd);
    if (fgets(ugc_rec,sizeof(ugc_rec),ugc_fd) == (char *)NULL)
      break;
    fill_ugc_entry(ugc_rec,ugc_entry++);
  }

/* leave the file open for later use */
/*  (void)fclose(ugc_fd);*/

/* remember the last ugc_array entry we filled */
  ugc_array_max = ugc_entry - 1;

/* attempt to save the binary database */
  if (access(bin_filebuff,W_OK) == 0)
    /* it's writeable, so let's try */
    if (! write_binary_database(bin_filebuff,(void *)ugc_array,
				            (size_t)sizeof(struct ugc_node),
				                           (size_t)ugc_entry))
      /* dump of binary database failed; be sure the file is gone */
      (void)unlink(bin_filebuff);

/* free the filename buffers */
  free((void *)filebuff);
  free((void *)bin_filebuff);

  return(ugc_fd);
}



/* given an object's index, return its right ascension and declination */

static void get_ugc_pos(array_pos,ra_rad,dec_rad)

int array_pos;
double *ra_rad, *dec_rad;

{
  *ra_rad = ugc_array[array_pos].ra_rad;
  *dec_rad = ugc_array[array_pos].dec_rad;

  return;
}



/* given a UGC record, make a data node for it */

static void fill_ugc_entry(record,ugc_entry)

char *record;
int ugc_entry;

{
  int reclen;
  char field[15];
  struct ra_pos ra_pos;
  struct dec_pos dec_pos;

/* get the length of the record for GET_FIELD */
  reclen = strlen(record) - 1;

/* get the UGC number */
  GET_FIELD(field,record,UGC_NUM_START,UGC_NUM_LENGTH);
  ugc_array[ugc_entry].ugc_num = atoi(field);

/* get the RA in pieces */
  GET_FIELD(field,record,POS_RA_START,2);
  ra_pos.hours = (smallint)atoi(field);

  GET_FIELD(field,record,POS_RA_START + 2,2);
  ra_pos.minutes = (smallint)atoi(field);

  GET_FIELD(field,record,POS_RA_START + 4,1);
  ra_pos.seconds = (smallint)(atoi(field) * 60 / 10);

  ra_pos.thousandths = 0;

/* similarly, get the declination in pieces */
  dec_pos.sign = (short)1;
  if (record[POS_DEC_START] == '-')
    dec_pos.sign = (short)-1;

  GET_FIELD(field,record,POS_DEC_START + 1,2);
  dec_pos.degrees = (smallint)atoi(field);

  GET_FIELD(field,record,POS_DEC_START + 3,2);
  dec_pos.minutes = (smallint)atoi(field);

  dec_pos.seconds = dec_pos.hundredths = 0;

/* compute the RA and declination in radians */
  ugc_array[ugc_entry].ra_rad = ra_pos_to_rad(ra_pos);
  ugc_array[ugc_entry].dec_rad = dec_pos_to_rad(dec_pos);

/* precess them to the common equinox */
  if (UGC_EPOCH != COMMON_EQUINOX)
    precess(ugc_array[ugc_entry].ra_rad,ugc_array[ugc_entry].dec_rad,
	          &ugc_array[ugc_entry].ra_rad,&ugc_array[ugc_entry].dec_rad);

  return;
}



/* construct a list of those objects which appear in the current display */

static void calculate_ugc_field(display,catalog)

struct display *display;
struct cat_header *catalog;

{
  int i;

/* loop through all UGC entries, checking for visibility */
  for (i = 0; i <= ugc_array_max; i++)
    /* check this object and add it to the list if visible */
    add_to_display_list(display,catalog,i,ugc_array[i].ra_rad,
                                                       ugc_array[i].dec_rad);

#ifdef DEBUG
  printf("calculate_ugc_field returning\n");
#endif
  return;
}



/* draw a UGC object on the sky */

/* ARGSUSED */

static void draw_ugc_obj(sky_widget,array_pos,obj_x,obj_y,ul_x,ul_y,
		                                                 width,height)

Widget sky_widget;
int array_pos;
int obj_x, obj_y;
int ul_x, ul_y;
int width, height;

{
 /* draw the object even if it is only partially in the box */
  if (((obj_x + (OBJSIZE + 1) / 2) >= ul_x) &&
                            ((obj_x - (OBJSIZE + 1) / 2) <= (ul_x + width)) &&
                            ((obj_y + (OBJSIZE + 1) / 2) >= ul_y) &&
                            ((obj_y - (OBJSIZE + 1) / 2) <= (ul_y + height)))
    ;
  else
    return;

  /* for the moment, draw a circle for anything */
  draw_object(sky_widget,obj_x,obj_y,OBJSIZE,CIRCLE);

  return;
}



/* draw one UGC object on the PostScript display */

/* ARGSUSED */

static boolean ps_draw_ugc_obj(display,array_pos,x,y)

struct display *display;
int array_pos;
int x, y;

{
  return(ps_draw_object(display,(double)x,(double)y,(double)OBJSIZE,CIRCLE));
}



/* display the catalog data for the given object */

static boolean display_ugc(ugc_fd,ugc_array_pos)

FILE *ugc_fd;
int ugc_array_pos;

{
/* get info from the UGC catalog */
  if (fseek(ugc_fd,ugc_array[ugc_array_pos].filepos,SEEK_SET) == -1)
    return(FALSE);
  if (fgets(ugc_rec,sizeof(ugc_rec),ugc_fd) == (char *)NULL)
    return(FALSE);

/* format the record into an informational display */
  format_ugc_display(ugc_rec);

  return(TRUE);
}



/* find the UGC object in the buffer and update the display coordinates */

static boolean find_ugc_obj(buffer)

char *buffer;

{
  int ugc_no;
  int i;

/* be sure this is a UGC designation */
  if ((strncmp(buffer,"UGC",3) == EQUAL) || strncmp(buffer,"ugc",3) == EQUAL)
    buffer = &buffer[3];
  else
    return(FALSE);

/* get the UGC number */
  ugc_no = atoi(buffer);

/* loop through the array looking for this number */
  for (i = 0; i <= ugc_array_max; i++)
    if (ugc_array[i].ugc_num == ugc_no)
      break;

/* return failure if we didn't find it */
  if (i > ugc_array_max)
    return(FALSE);
  else {
    /* update the display coordinates */
    update_display_pos(ugc_array[i].ra_rad,ugc_array[i].dec_rad);
    return(TRUE);
  }

/* NOTREACHED */
}



/* format a nice name for the object */

/* ARGSUSED */

static void format_ugc_name(ugc_pos,buffer,font)

int ugc_pos;
char *buffer;
IDfont *font;

{
/* use the UGC number */
  sprintf(buffer,"UGC %d",ugc_array[ugc_pos].ugc_num);

  return;
}
