/*
 * xsky - an interactive sky atlas
 *
 * Copyright 1992-4, Terry R. Friedrichsen
 *
 * This program may be copied and redistributed, in whole or in part,
 * as long as you don't try to make any money from the sale or redis-
 * tribution of the program or any part of the program, or pretend
 * that you wrote the program or any of its parts unless specifically
 * credited by the original author.
 *
 * You are free to make use of this software in your own programs, as
 * long as you credit the original author where it is due.
 */

/*
 * WARRANTY:
 * xsky was written as a learning project and as a demonstration of
 * X Window System programming.  xsky doesn't do anything; it is not
 * merchantable, and it is not fit for any purpose whatsoever.  In
 * fact, don't use xsky at all; it's free, and you're getting what
 * you paid for.
 */

#include <stdio.h>
#include <stdlib.h>

#include <string.h>

#include "skydefs.h"

#include "starnames.h"

#include "format.h"

/* external function prototypes */
extern char *build_filespec PROTOTYPE((char *,char *));

#define STARNAME_FILE   "starnames.dat"

/* table of star names */
struct starname starnames[NUM_STARNAMES];

static boolean star_names_read = FALSE;



/* read in the table of star names */

void read_starname_table()

{
  char *filebuff;
  FILE *name_fd;
  char inbuf[MAX_STARNAME_REC + 1 + 1];
  int i;

/* don't bother if we've already done this */
  if (star_names_read)
    return;

/* build the file path */
  filebuff = build_filespec("STARNAME_PATH",STARNAME_FILE);

/* open the star names file */
  if ((name_fd = fopen(filebuff,"r")) == (FILE *)NULL) {
    printf("sky:  error opening star name file %s\n",filebuff);
    perror("sky");
    exit(1);
  }

/* loop through the file, reading in star name table entries */
  i = 0;
  while (fgets(inbuf,sizeof(inbuf),name_fd) != NULL) {
    /* remove the newline at the end */
    inbuf[strlen(inbuf) - 1] = '\0';

/* save the HR number of this star */
    starnames[i].hr_number = atoi(inbuf);

/* save the star name list */
    starnames[i].namebuff = (char *)malloc(strlen(&inbuf[5]) + 1);
    strcpy(starnames[i++].namebuff,&inbuf[5]);
  }

/* be neat */
  (void)fclose(name_fd);

/* free the filename buffer */
  free((void *)filebuff);

  return;
}



/* find the star name list for this star */

char *find_starnames(hrnum)

int hrnum;

{
  int i;

/* search the star name database to find the given HR number */
  for (i = 0; i < NUM_STARNAMES; i++)
    if (starnames[i].hr_number == hrnum)
      break;

/* return the star name buffer, if found */
  if (i >= NUM_STARNAMES)
    return((char *)NULL);
  else
    return(starnames[i].namebuff);

/* NOTREACHED */
}



/* format the star name list */

void format_starnames(namebuff,row_length)

char *namebuff;
int row_length;

{
  int i, oldi;
  char oldc;

/* begin at the beginning of the name buffer */
  i = oldi = 0;

/* loop handling star names */
  do {
/* find the end of the next star name */
    while ((namebuff[i] != ';') && (namebuff[i] != '.'))
      i++;

/* will it fit on this line? */
    if (((idx - row_start) + (i - oldi) + 1) >= row_length) {
      /* no, go to next line */
      next_line();
      advance_tabs(2);
    }

/* temporarily end the star name with a nul */
    oldc = namebuff[i];
    namebuff[i] = '\0';

/* put out this star name */
    add_string(&namebuff[oldi]);

/* replace the old terminator character */
    namebuff[i] = oldc;

/* is this the end of the star list? */
    if (namebuff[i] == '.')
      /* yes; we're done here */
      break;
    else {
      /* advance to the next star name */
      while (isspace(namebuff[++i]))   ;
      oldi = i;
    }

/* add a delimiter for the next name */
    add_string(", ");

  } while (TRUE);

  return;
}



/* find the star named in the buffer and return its HR number */

int find_star_by_name(buffer)

char *buffer;

{
  int i;
  char *ptr1, *ptr2;
  char c;
  boolean notfirst;

/* skip leading white space */
  while (isspace(*buffer))
    buffer++;

/* first do a check on the primary name only */
  for (i = 0; i < NUM_STARNAMES; i++) {
    /* find the end of this star name */
    ptr1 = starnames[i].namebuff;
    while ((*ptr1 != ';') && (*ptr1 != '.'))
      ptr1++;

    /* save the character we found and replace it with a nul */
    c = *ptr1;
    *ptr1 = '\0';

    /* check to see if this star name is the one we're looking for */
    if (strcasecmp(buffer,starnames[i].namebuff) == EQUAL) {
      /* yes; replace the delimiter character at the end of the star name */
      *ptr1 = c;

      /* and return this HR number */
      return(starnames[i].hr_number);
    }
    else
      /* replace the delimiter character at the end of the star name */
      *ptr1 = c;
  }

/* loop through the star name table, looking for any match */
  for (i = 0; i < NUM_STARNAMES; i++) {
    /* begin with the first name for this star */
    ptr1 = starnames[i].namebuff;

    /* flag that this is the first name for this star (already checked) */
    notfirst = FALSE;

    do {
      /* find the end of this star name */
      ptr2 = ptr1;
      while ((*ptr2 != ';') && (*ptr2 != '.'))
	ptr2++;

      /* save the character we found and replace it with a nul */
      c = *ptr2;
      *ptr2 = '\0';

      /* check to see if this star name is the one we're looking for */
      if (notfirst && (strcasecmp(buffer,ptr1) == EQUAL)) {
	/* yes; replace the delimiter character at the end of the star name */
	*ptr2 = c;

	/* and return this HR number */
	return(starnames[i].hr_number);
      }
      else
	/* no; replace the delimiter character and try the next star name */
	if ((*ptr2++ = c) == '.')
	  /* no more names for this star */
	  break;
	else {
	  /* advance to the beginning of the next star name */
	  ptr1 = ptr2;
	  while (isspace(*ptr1))
	    ptr1++;

	  /* flag that this is not the first name */
	  notfirst = TRUE;
	}
    } while (TRUE);
  }

/* no star name matched; tell the caller we didn't find the star */
  return(-1);
}
