/*
 * xsky - an interactive sky atlas
 *
 * Copyright 1992-6, Terry R. Friedrichsen
 *
 * This program may be copied and redistributed, in whole or in part,
 * as long as you don't try to make any money from the sale or redis-
 * tribution of the program or any part of the program, or pretend
 * that you wrote the program or any of its parts unless specifically
 * credited by the original author.
 *
 * You are free to make use of this software in your own programs, as
 * long as you credit the original author where it is due.
 */

/*
 * WARRANTY:
 * xsky was written as a learning project and as a demonstration of
 * X Window System programming.  xsky doesn't do anything; it is not
 * merchantable, and it is not fit for any purpose whatsoever.  In
 * fact, don't use xsky at all; it's free, and you're getting what
 * you paid for.
 */

#include <stdio.h>
#include <ctype.h>

#include <math.h>
#include <string.h>

#include <stdlib.h>

#include <sys/types.h>
#include <sys/stat.h>

#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>

#include <X11/Shell.h>
#include <X11/Core.h>

#include <X11/cursorfont.h>

#include <X11/Xaw/Command.h>
#include <X11/Xaw/Form.h>
#include <X11/Xaw/Text.h>
#include <X11/Xaw/AsciiText.h>
#include <X11/Xaw/Dialog.h>
#include <X11/Xaw/MenuButton.h>
#include <X11/Xaw/SimpleMenu.h>
#include <X11/Xaw/SmeBSB.h>
#include <X11/Xaw/Scrollbar.h>
#include <X11/Xaw/Box.h>
#include <X11/Xaw/Toggle.h>

#include "skydefs.h"
#include "catalog.h"
#include "pos.h"
#include "draw.h"

#include "sky.h"

/* static function prototypes */
static void printusage PROTOTYPE((char *));
static void initialize_display PROTOTYPE((void));
static void read_toggle_values PROTOTYPE((void));
static void buildmaindisplay PROTOTYPE((void));
static Widget makepanelbutton PROTOTYPE((int,Widget,Widget,XtCallbackProc,
					     XtPointer,Widget *,Dimension *));
static Widget makepaneltoggle PROTOTYPE((int,Widget,Widget,XtCallbackProc,
					     XtPointer,Widget *,Dimension *));
static void makecenterfield PROTOTYPE((Widget,Widget));
static void makepointerpos PROTOTYPE((Widget,Widget *,Dimension *));
static void makescrollbars PROTOTYPE((Widget));
static void makezoomslider PROTOTYPE((void));
static void setup_wm_actions PROTOTYPE((void));
static void setup_colors PROTOTYPE((void));
static void setup_cursors PROTOTYPE((void));
static void setup_graphics PROTOTYPE((void));
static void define_night_cursors PROTOTYPE((void));
static void define_night_cursor PROTOTYPE((Widget,unsigned int));
static Boolean buildobjdisplay PROTOTYPE((XtPointer));
static Boolean buildfinddisplay PROTOTYPE((XtPointer));
static Boolean builduserlabeldisplay PROTOTYPE((XtPointer));
static Boolean buildcatalogmenu PROTOTYPE((XtPointer));
static Boolean buildchartdialog PROTOTYPE((XtPointer));
static Boolean buildinfodisplay PROTOTYPE((XtPointer));
static Boolean make_big_initial PROTOTYPE((XtPointer));
static Widget maketextfield PROTOTYPE((Widget,Widget *,XFontStruct *,Pixel,
				                          char *,char *,int));
static void PrepareInfo PROTOTYPE((Widget,XtPointer,XtPointer));
static void sky_expose PROTOTYPE((Widget,XtPointer,XEvent *,Boolean *));
static void handle_grid_button PROTOTYPE((Widget,XtPointer,XtPointer));
static void handle_bound_button PROTOTYPE((Widget,XtPointer,XtPointer));
static void toggle_ids PROTOTYPE((Widget,XtPointer,XtPointer));
static void erase_ids PROTOTYPE((Widget,XtPointer,XtPointer));
static void popup_catmenu PROTOTYPE((Widget,XtPointer,XtPointer));
static void popup_find PROTOTYPE((Widget,XtPointer,XtPointer));
static void popup_userlabel PROTOTYPE((Widget,XtPointer,XtPointer));
static void popup_chart PROTOTYPE((Widget,XtPointer,XtPointer));
static void popup_shell PROTOTYPE((Widget,XtPointer,XtPointer));
static void ps_draw_sky PROTOTYPE((struct display *,char *));
static void clone_state PROTOTYPE((struct display *));
static void save_state PROTOTYPE((void));
static void Undo PROTOTYPE((Widget,XtPointer,XtPointer));
static void Quit PROTOTYPE((Widget,XtPointer,XtPointer));
static void set_normal_cursor PROTOTYPE((void));
static void set_wait_cursor PROTOTYPE((void));
static void set_textbox_cursor PROTOTYPE((void));
static void set_rubberband_cursor PROTOTYPE((void));
static void posbox_position PROTOTYPE((void));
static void cancel_dpyinfo PROTOTYPE((Widget,XtPointer,XEvent *,Boolean *));
static void DpyInfoAcked PROTOTYPE((Widget,XtPointer,XtPointer));
static void DataAcked PROTOTYPE((Widget,XtPointer,XtPointer));
static void DataAckEvent PROTOTYPE((Widget,XtPointer,XEvent *,Boolean *));
static Boolean draw_grid_lines_work PROTOTYPE((XtPointer));
static Boolean draw_boundary_lines_work PROTOTYPE((XtPointer));
static void cancel_find PROTOTYPE((Widget,XtPointer,XEvent *,Boolean *));
static void cancel_userlabel PROTOTYPE((Widget,XtPointer,XEvent *,Boolean *));
static void FindObject PROTOTYPE((Widget,XtPointer,XtPointer));
static void CreateChart PROTOTYPE((Widget,XtPointer,XtPointer));
static void CancelChart PROTOTYPE((Widget,XtPointer,XtPointer));
static void cancel_chart PROTOTYPE((Widget,XtPointer,XEvent *,Boolean *));
static void PositionUserLabel PROTOTYPE((Widget,XtPointer,XtPointer));
static void HandleTopStruct PROTOTYPE((Widget,XtPointer,XEvent *,Boolean *));
static void HandleFormStruct PROTOTYPE((Widget,XtPointer,XEvent *,Boolean *));
static void HandleSkyStruct PROTOTYPE((Widget,XtPointer,XEvent *,Boolean *));
static void handle_pointer_entry PROTOTYPE((Widget,XtPointer,XEvent *,
						                  Boolean *));
static void handle_pointer_motion PROTOTYPE((Widget,XtPointer,XEvent *,
						                  Boolean *));
static void HandleSkyButton PROTOTYPE((Widget,XtPointer,XEvent *,Boolean *));
static void HandleSkyButtonPress PROTOTYPE((Widget,XtPointer,XEvent *,
						                  Boolean *));
static void HandleSkyButtonRelease PROTOTYPE((Widget,XtPointer,XEvent *,
							          Boolean *));
static void handle_button1_release PROTOTYPE((XtPointer,
					             XButtonReleasedEvent *));
static void handle_button2_release PROTOTYPE((XtPointer,
					             XButtonReleasedEvent *));
static void displayobjinfo PROTOTYPE((XButtonPressedEvent *));
static void add_constellation_label PROTOTYPE((int,int));
static void add_object_label PROTOTYPE((int,int));
static void add_user_label PROTOTYPE((int,int));
static void position_label PROTOTYPE((struct id_node *));
static void change_position_display PROTOTYPE((struct display *,int,int));
static void update_position_display PROTOTYPE((struct display *));
static void handle_rubberband PROTOTYPE((int,int));
static void draw_rubberband_box PROTOTYPE((Widget,struct rubberband *));
static void update_rubberband_box PROTOTYPE((struct display *,
					        struct rubberband *,int,int));
static void change_aspect PROTOTYPE((struct rubberband *));
static void change_scale PROTOTYPE((struct rubberband *));
static void remove_label PROTOTYPE((int,int));
static void move_label PROTOTYPE((int,int));
static void handle_recenter_event PROTOTYPE((XButtonPressedEvent *));
static void handle_button1_motion PROTOTYPE((Widget,XtPointer,XEvent *,
						                  Boolean *));
static void handle_button2_motion PROTOTYPE((Widget,XtPointer,XEvent *,
						                  Boolean *));
static void update_center_text PROTOTYPE((void));
static void update_sky PROTOTYPE((struct display *,int));
static void update_scrollbar_thumbs PROTOTYPE((void));
static void clear_sky PROTOTYPE((struct display *));
static void update_scale PROTOTYPE((void));
static void update_zoom_thumb PROTOTYPE((double));
static void display_zoom PROTOTYPE((Widget,XtPointer,XtPointer));
static void inc_scroll PROTOTYPE((Widget,XtPointer,XtPointer));
static void inc_zoom PROTOTYPE((Widget,XtPointer,XtPointer));

/* external function prototypes */
extern boolean init_boundaries PROTOTYPE((void));
extern void catalog_init PROTOTYPE((void));
extern void bad_init_file PROTOTYPE((char *));
extern void build_id_list PROTOTYPE((FILE *));
extern boolean find_tok PROTOTYPE((char *,char *,char *));
extern void draw_box PROTOTYPE((Widget,int,int,int,int));
extern void draw_enclosure PROTOTYPE((Widget,XPoint *,int));
extern void draw_id PROTOTYPE((Widget,struct id_node *));
extern void finalize_label_pos PROTOTYPE((struct display *,struct id_node *,
					                            int,int));
extern void draw_marked_ids PROTOTYPE((Widget));
extern void add_id_to_list PROTOTYPE((struct id_node *));
extern void draw_id_texts PROTOTYPE((Widget,int,int,int,int));
extern void remove_this_id PROTOTYPE((Widget,struct id_node *));
extern void remove_all_ids PROTOTYPE((void));
extern struct id_node *find_id_by_position PROTOTYPE((int,int));
extern struct obj_node *find_object PROTOTYPE((int,int,struct cat_header **));
extern char *identify_constellation PROTOTYPE((struct display *,int,int));
extern struct id_node *get_object_id PROTOTYPE((int,int));
extern boolean find_object_by_name PROTOTYPE((char *));
extern struct id_node *get_userlabel_id PROTOTYPE((int,int,char *,IDfont));
extern void setup_label_move PROTOTYPE((Widget,struct id_node *));
extern void calculate_field PROTOTYPE((struct display *));
extern void draw_sky PROTOTYPE((Widget,int,int,int,int));
extern boolean ps_write_file PROTOTYPE((Widget,struct display *,char *));
extern void X_to_pos_rad PROTOTYPE((struct display *,int,int,
				                          double *,double *));
extern void draw_grid_lines PROTOTYPE((Widget,struct display *,
				                            int,int,int,int));
extern void draw_boundary_lines PROTOTYPE((Widget,struct display *,
					                    int,int,int,int));
extern void update_center_position PROTOTYPE((struct display *,char *));
#ifdef ROUND_DISPLAY_POS
extern void round_ra PROTOTYPE((struct ra_pos *));
extern void round_dec PROTOTYPE((struct dec_pos *));
#endif
extern struct ra_pos rad_to_ra PROTOTYPE((double));
extern struct dec_pos rad_to_dec PROTOTYPE((double));
extern boolean calculate_position PROTOTYPE((struct display *,char *));
extern char *build_filespec PROTOTYPE((char *,char *));

/* global function prototypes */
void clear_area PROTOTYPE((Widget,int,int,int,int));

/* magic option for "build binary databases only" */
#define MAGIC_OPTION   "-bin"

/* getopt() option-processing externals */
extern int optind, opterr;
extern char *optarg;

/* default initial display scale */
static float default_scale = INITIAL_PIXELS_PER_DEGREE;

/* default initial width and height */
static int default_width = INITIAL_WIDTH;
static int default_height = INITIAL_HEIGHT;

/* default initial magnitude limit */
static float default_maglim = INITIAL_MAGNITUDE;

/* default maximum slider magnification */
static int default_slidermax = DEFAULT_SLIDERMAX;

/* note the space on the end for a fudge factor */
#define MAX_POS_STRING   "00h 00m 00s.000, -00d 00' 00\".00 "

/* maximum strings for right ascension and declination */
#define MAX_RA_STRING    "00h 00m 00s.000 "
#define MAX_DEC_STRING   "-00d 00' 00\".00 "

/* colors for the spectral classes */
#define DEFAULT_O_COLOR   "Blue"
#define DEFAULT_B_COLOR   "DeepSkyBlue"
#define DEFAULT_A_COLOR   "White"
#define DEFAULT_F_COLOR   "NavajoWhite3"
#define DEFAULT_G_COLOR   "Yellow"
#define DEFAULT_K_COLOR   "Orange"
#define DEFAULT_M_COLOR   "Red"

/* fonts and buffer for object identification text */
#define MAX_FONTNAME   100
#define LATIN_ID_FONTNAME   "helvetica10"
XFontStruct *latinidfontstr;
static char *latin_id_font = (char *)NULL;
static char latin_id_font_buffer[MAX_FONTNAME];

#define LATIN_CON_FONTNAME   "helvetica14"
XFontStruct *latinconfontstr;
static char *latin_con_font = (char *)NULL;
static char latin_con_font_buffer[MAX_FONTNAME];

#define GREEK_ID_FONTNAME   "symbol12"
XFontStruct *greekidfontstr;
static char *greek_font = (char *)NULL;
static char greek_font_buffer[MAX_FONTNAME];

/* size of a Latin x for object drawing */
int x_height;
int x_width;

/* structure for retrieval of application resources */
typedef struct _instance_variables {
  String pos;
  float scale;
  int width;
  int height;
  float maglim;
  int slidermax;
  Boolean downunder;
  Boolean use_scrollbars;
  Boolean use_slider;
  Boolean use_posdisplay;
  String latinidfont;
  String latinconfont;
  String greekfont;
  String ocolor, bcolor, acolor, fcolor, gcolor, kcolor, mcolor;
  Boolean autopopdown;
  String firstcat;
} instance_variable_rec;

static instance_variable_rec iv;

#define GETIVSTRING(resource_name,iv_element,default)                     \
  {                                                                       \
  resource_name,                                                          \
  XtCString,                                                              \
  XtRString, sizeof(String),                                              \
  XtOffsetOf(instance_variable_rec,iv_element),                           \
  XtRString, (XtPointer)default                                           \
  }

#define GETIVINT(resource_name,iv_element,default)                        \
  {                                                                       \
  resource_name,                                                          \
  XtCValue,                                                               \
  XtRInt, sizeof(int),                                                    \
  XtOffsetOf(instance_variable_rec,iv_element),                           \
  XtRInt, (XtPointer)default                                              \
  }

#define GETIVFLOAT(resource_name,iv_element,default)                      \
  {                                                                       \
  resource_name,                                                          \
  XtCValue,                                                               \
  XtRFloat, sizeof(float),                                                \
  XtOffsetOf(instance_variable_rec,iv_element),                           \
  XtRFloat, (XtPointer)default                                            \
  }

#define GETIVBOOLEAN(resource_name,iv_element,default)                    \
  {                                                                       \
  resource_name,                                                          \
  XtCValue,                                                               \
  XtRBoolean, sizeof(Boolean),                                            \
  XtOffsetOf(instance_variable_rec,iv_element),                           \
  XtRImmediate, (XtPointer)default                                        \
  }

static XtResource resources[] = {
  GETIVSTRING("displaypos",pos,INITIAL_POSITION),
  GETIVFLOAT("displayscale",scale,&default_scale),
  GETIVINT("displaywidth",width,&default_width),
  GETIVINT("displayheight",height,&default_height),
  GETIVFLOAT("displaymaglim",maglim,&default_maglim),
  GETIVINT("slidermax",slidermax,&default_slidermax),
  GETIVBOOLEAN("downunder",downunder,False),
  GETIVBOOLEAN("scrollbars",use_scrollbars,False),
  GETIVBOOLEAN("zoomslider",use_slider,True),
  GETIVBOOLEAN("posdisplay",use_posdisplay,True),
  GETIVSTRING("displaylatinidfont",latinidfont,LATIN_ID_FONTNAME),
  GETIVSTRING("displaylatinconfont",latinconfont,LATIN_CON_FONTNAME),
  GETIVSTRING("displaygreekfont",greekfont,GREEK_ID_FONTNAME),
  GETIVSTRING("displaycolorO",ocolor,DEFAULT_O_COLOR),
  GETIVSTRING("displaycolorB",bcolor,DEFAULT_B_COLOR),
  GETIVSTRING("displaycolorA",acolor,DEFAULT_A_COLOR),
  GETIVSTRING("displaycolorF",fcolor,DEFAULT_F_COLOR),
  GETIVSTRING("displaycolorG",gcolor,DEFAULT_G_COLOR),
  GETIVSTRING("displaycolorK",kcolor,DEFAULT_K_COLOR),
  GETIVSTRING("displaycolorM",mcolor,DEFAULT_M_COLOR),
  GETIVBOOLEAN("autopopdown",autopopdown,True),
  GETIVSTRING("displayfirst",firstcat,INITIAL_CATALOG)
};

/* structure describing the current sky display state */
static struct display display;

/* initial size of the display */
int initial_width, initial_height;

/* list of catalog header structures */
extern struct cat_header *cat_list_head;

/* initial list of active catalogs */
static char catlist[INITIAL_CATLIST_LEN + 1];

/* atoms for handling window manager messages */
#define DELETE_ATOM   0
#define SAVE_ATOM     1
static Atom wm_atoms[2];

/* add a quit action */
static XtActionProc wm_msg_action PROTOTYPE((Widget,XEvent *,
						        String *,Cardinal *));
static XtActionsRec wm_msg_actions[] = {
  {"wm_msg",(XtActionProc)wm_msg_action}
};
static String wm_msg_actions_trans = "<Message>WM_PROTOCOLS: wm_msg()";

/* add an action to pop down the catalog selection menu */
static XtActionProc popdown_cat_menu PROTOTYPE((Widget,XEvent *,
						        String *,Cardinal *));
static XtActionsRec obj_actions[] = {
  {"popdownObjMenu",(XtActionProc)popdown_cat_menu}
};

/* add an action to pop down the object finder display */
static XtActionProc popdown_find_display PROTOTYPE((Widget,XEvent *,
						        String *,Cardinal *));
static XtActionsRec find_actions[] = {
  {"popdownFindDisplay",(XtActionProc)popdown_find_display}
};

/* add an action to find an object when <return> key is pressed */
static XtActionProc find_object_in_text PROTOTYPE((Widget,XEvent *,
						        String *,Cardinal *));
static XtActionsRec find_text_actions[] = {
  {"findObject",(XtActionProc)find_object_in_text}
};

/* add an action to pop down the user label display */
static XtActionProc popdown_userlabel_display PROTOTYPE((Widget,XEvent *,
						        String *,Cardinal *));
static XtActionsRec userlabel_actions[] = {
  {"popdownUserLabelDisplay",(XtActionProc)popdown_userlabel_display}
};

/* add an action to position a label when <return> key is pressed */
static XtActionProc position_userlabel PROTOTYPE((Widget,XEvent *,
						        String *,Cardinal *));
static XtActionsRec userlabel_text_actions[] = {
  {"positionUserLabel",(XtActionProc)position_userlabel}
};

/* add an action to update the display center when <return> key is pressed */
static XtActionProc display_center_edited PROTOTYPE((Widget,XEvent *,
						        String *,Cardinal *));
static XtActionsRec center_text_actions[] = {
  {"updateCenter",(XtActionProc)display_center_edited}
};

/* add an action to update the display from edited Info display text */
static XtActionProc DpyInfoEdited PROTOTYPE((Widget,XEvent *,
					                String *,Cardinal *));
static XtActionsRec info_text_actions[] = {
  {"updateInfo",(XtActionProc)DpyInfoEdited}
};

/* define an action routine for <return> in the chart text widget */
static XtActionProc chart_text_return PROTOTYPE((Widget,XEvent *,
						        String *,Cardinal *));
static XtActionsRec chart_text_actions[] = {
  {"chart_text_return",(XtActionProc)chart_text_return}
};

/* set up <return> keypresses to invoke the chart_text_return action */
static String chart_text_trans = "<Key>Return:  chart_text_return()";

/* flag to indicate whether the ConfigureNotify event handlers are installed */
static boolean config_flag = FALSE;

/* buffer to hold the information display */
static char info_display[350];

/* buffers to hold the editable display position */
static char pos_buffer[50], old_pos_buffer[50];

/* buffers to hold the editable limiting magnitude and display scale */
static char mag_buffer[10];
static char scale_buffer[15];

/* buffers to hold the continuously-updating pointer position */
static char ra_buffer[sizeof(MAX_RA_STRING)];
static char dec_buffer[sizeof(MAX_DEC_STRING)];

/* buffer to hold name of object to find */
#define MAXOBJNAME   50
static char object_buffer[MAXOBJNAME + 1];

/* buffer to hold a user-supplied label */
#define MAXUSERLABEL   MAX_ID_STRING
static char userlabel_buffer[MAXUSERLABEL + 1];

/* flag indicating we're positioning a user-supplied label */
static boolean user_label_flag = FALSE;

/* buffer to hold the object information display */
char obj_info[2400];

/* the default color map and red and black RGB values for night mode */
static Colormap default_colormap;
static XColor red_rgb, black_rgb;

/* colors for graphics contexts */
static XrmValue blackstring = {
  sizeof("Black"),(caddr_t)"Black"
  };

unsigned long blackpixelval;

static XrmValue blackpixel = {
  sizeof(blackpixelval),(caddr_t)&blackpixelval
  };

static XrmValue whitestring = {
  sizeof("White"),(caddr_t)"White"
  };

unsigned long whitepixelval;

static XrmValue whitepixel = {
  sizeof(whitepixelval),(caddr_t)&whitepixelval
  };

static XrmValue redstring = {
  sizeof("Red"),(caddr_t)"Red"
  };

unsigned long redpixelval;

static XrmValue redpixel = {
  sizeof(redpixelval),(caddr_t)&redpixelval
  };

/* declare some star colors and graphics contexts for each */
#define NUMSTARCOLORS   7
static char *starcolors[NUMSTARCOLORS] = {
  (char *)NULL,
  (char *)NULL,
  (char *)NULL,
  (char *)NULL,
  (char *)NULL,
  (char *)NULL,
  (char *)NULL
  };

static unsigned long colorpixelval[NUMSTARCOLORS];

/* graphics context for drawing various things */
GC star_gcs[NUMSTARCOLORS];
GC mono_star_gc;
GC obj_gc;
GC text_gc;
GC grid_gc;

/* list header for list of visual identifications */
extern struct id_node *id_list;

/* window manager save initialization file */
static boolean initflag = FALSE;
static char *initfile;
static FILE *init_fd;

/* user-specified magnitude limit */
static boolean opt_mag_limit_flag = FALSE;
static float opt_mag_limit;

/* user-specified display scale */
static boolean opt_display_scale_flag = FALSE;
static float opt_display_scale;

/* user-specified initial position */
static char *opt_pos = (char *)NULL;

/* user-specified initial catalog list */
static char *opt_catlist = (char *)NULL;

/* user-specified Southern Hemisphere flag */
static boolean opt_downunder_flag = FALSE;
static boolean opt_downunder = FALSE;

/* user-specified scrollbar flag */
static Boolean opt_scrollbar_flag = False;
static Boolean opt_scrollbar = False;

/* user-specified magnification slider flag */
static Boolean opt_slider_flag = False;
static Boolean opt_slider = False;

/* user-specified magnification slider maximum */
static Boolean opt_slidermax_flag = False;
static int opt_slidermax = -1;

/* user-specified pointer position display flag */
static Boolean opt_posdisplay_flag = False;
static Boolean opt_posdisplay = False;

/* TRUE if black & white or greyscale display */
boolean mono_flag = FALSE;

/* TRUE if night mode */
boolean night_mode_flag = FALSE;

/* True if autopopdown of popup displays is enabled */
static Boolean autopopdown;

/* remember if user specified -L option */
static boolean big_flag = FALSE;

/* lowest magnitude to display */
float mag_limit = 8.0;

/* True if scrollbar use is enabled */
static Boolean use_scrollbars;

/* True if zoom slider use is enabled */
static Boolean use_slider;

/* maximum zoom slider magnification */
static int slidermax;

/* True if pointer position display is enabled */
static Boolean use_posdisplay;

/* truth values for command-line options which take a flag */
#define FLAG_ON_VALUE    "on"
#define FLAG_OFF_VALUE   "off"

/* define a few cursors */
static Cursor normal_cursor;
static Cursor wait_cursor;
static Cursor rubberband_cursor;
static Cursor invisible_cursor;

static Cursor night_shell_cursor;
static Cursor night_text_cursor;

/* flag for display of grid lines */
Boolean draw_grid = False;

/* flag for display of constellation boundaries */
static boolean have_boundaries = FALSE;
Boolean draw_bound = False;

/* flag for display of object IDs */
Boolean draw_ids = True;

/* right ascension and declination line intervals (RA interval is in
 * thousandths of a second of time, dec interval is in hundredths of
 * a second of arc) */
extern int ra_line_interval;
extern int dec_line_interval;

/* flag that we're drawing in PostScript, not on the display */
boolean ps_draw_flag;

/* initial vertical separation of panel buttons */
#define PANELBUTTONSEP   10

/* declare the widgets globally so callback routines can find them easily */
static Widget toplevel, form;
static Widget sky_widget, vert_scroll, horiz_scroll, zoomslider;
static Widget menuinfo, menugrid, menubound, menucat, menuhide, menuerase,
                       menufind, menuuserlabel, menuchart, menuundo, menuquit;
static Widget centerbox, centerlabel, centertext;
static Widget posbox, poslabel, pos_ratext, pos_dectext;
static Widget infoshell, infoform, infolabel, displaymag,
                                                    displayscale, ack, cancel;
static Widget catmenushell, catbox, catlabel;
static Widget chartdialogshell, chartdialog;
static Widget zoomin, zoomout;
static Widget findshell, findbox, findlabel, findtext, findack, findcancel;
static Widget userlabelshell, userlabelbox, userlabellabel, userlabeltext,
                                                userlabelack, userlabelcancel;



main(argc,argv)

int argc;
char *argv[];

{
  int c;
  int n;
  XSetWindowAttributes attr;
  Arg sky_args[20];
  struct cat_header *catalog;

/* initialize and read the list of catalogs */
  catalog_init();

/* initialize the constellation boundary database */
  have_boundaries = init_boundaries();

/* magic:  if the argument count is exactly two and the second argument is
 * exactly the magic option, just exit after having read the catalogs (and
 * thus built and saved the binary dumps); avoid all contact with X */
  if ((argc == 2) && (strcmp(argv[1],MAGIC_OPTION) == EQUAL))
    exit(0);

/* initialize the Intrinsics, creating top-level shell widget */
  toplevel = XtInitialize(argv[0],
			  "XSky",
			  (XrmOptionDescRec *)NULL,
			  (Cardinal)0,
#if XtSpecificationRelease > 4
			  &argc,
#else
			  (Cardinal *)&argc,
#endif
			  argv);
#ifdef DEBUG
  printf("XtInitialize returned\n");
#endif

/* check for any options of our own */
  while ((c = getopt(argc,argv,"hLNbum:a:f:g:c:s:p:I:P:S:Z:O:B:A:F:G:K:M:"))
	                                                              != EOF)
    switch ((char)c) {
    case 'L':     /* make the display fill the screen */
      big_flag = TRUE;
      break;
    case 'N':     /* night mode (red on black background) */
      night_mode_flag = TRUE;
      break;
    case 'a':     /* font for constellation area identifications */
      latin_con_font = optarg;
      break;
    case 'b':     /* black-and-white even if color display */
      mono_flag = TRUE;
      break;
    case 'c':     /* initial catalog list */
      opt_catlist = optarg;
      break;
    case 'f':     /* font for object identifications */
      latin_id_font = optarg;
      break;
    case 'g':     /* Greek font for object identifications */
      greek_font = optarg;
      break;
    case 'h':
      printusage(argv[0]);
      break;
    case 'p':     /* initial display position */
      opt_pos = optarg;
      break;
    case 'u':     /* orient display for Southern Hemisphere */
      opt_downunder_flag = TRUE;
      opt_downunder = TRUE;
      break;
    case 'm':     /* set limiting lower magnitude */
      if (optarg == (char *)NULL)
	printusage(argv[0]);
      else {
	opt_mag_limit = atof(optarg);
	opt_mag_limit_flag = TRUE;
      }
      break;
    case 's':     /* set display scale */
      if (optarg == (char *)NULL)
	printusage(argv[0]);
      else {
	opt_display_scale = atof(optarg);
	opt_display_scale_flag = TRUE;
      }
      break;
    case 'I':     /* window manager save initialization file name */
      initflag = TRUE;
      initfile = optarg;
      break;
    case 'P':     /* pointer position display on/off */
      opt_posdisplay_flag = True;
      if (strcasecmp(optarg,FLAG_ON_VALUE) == EQUAL)
	/* user is turning on the pointer position display */
	opt_posdisplay = True;
      else
	opt_posdisplay = False;
      break;
    case 'S':     /* scrollbars on/off */
      opt_scrollbar_flag = True;
      if (strcasecmp(optarg,FLAG_ON_VALUE) == EQUAL)
	/* user is turning on the scrollbars */
	opt_scrollbar = True;
      else
	opt_scrollbar = False;
      break;
    case 'Z':     /* use magnification (zoom) slider */
      opt_slider_flag = True;
      /* the Z option takes the values "on", "off", or a number.  "on" means
       * turn the zoom slider on with the default maximum value.  "off" (ac-
       * tually any non-numeric other than "on") means turn the zoom slider
       * off.  a numeric value means turn the zoom slider on using the value
       * as the zoom slider maximum magnification. */
      if (strcasecmp(optarg,FLAG_ON_VALUE) == EQUAL) {
	/* turn on the zoom slider with the default maximum */
	opt_slider = True;
	opt_slidermax_flag = False;
      }
      else if (strspn(optarg,"0123456789") == strlen(optarg)) {
	/* all numeric - turn on the slider and capture the maximum */
	opt_slider = True;
	opt_slidermax_flag = True;
	opt_slidermax = atoi(optarg);
      }
      else {
	/* turn off the zoom slider */
	opt_slider = False;
	opt_slidermax_flag = False;
      }
      break;
    case 'O':
      starcolors[O] = optarg;
      break;
    case 'B':
      starcolors[B] = optarg;
      break;
    case 'A':
      starcolors[A] = optarg;
      break;
    case 'F':
      starcolors[F] = optarg;
      break;
    case 'G':
      starcolors[G] = optarg;
      break;
    case 'K':
      starcolors[K] = optarg;
      break;
    case 'M':
      starcolors[M] = optarg;
      break;
    default:
      printusage(argv[0]);
      break;
    }

/* handle conflict between night mode and mono mode */
  if (mono_flag && night_mode_flag) {
    fprintf(stderr,"b&w mode and night mode are incompatible\n");
    exit(1);
  }

/* if necessary, open the window manager save initialization file */
  if (initflag)
    if ((init_fd = fopen(initfile,"r")) == (FILE *)NULL) {
      fprintf(stderr,"initialization file %s not found\n",initfile);
      exit(1);
    }

/* register the object menu actions */
  XtAddActions(obj_actions,XtNumber(obj_actions));

/* register the object finder display actions */
  XtAddActions(find_actions,XtNumber(find_actions));
  XtAddActions(find_text_actions,XtNumber(find_text_actions));

/* register the user label display actions */
  XtAddActions(userlabel_actions,XtNumber(userlabel_actions));
  XtAddActions(userlabel_text_actions,XtNumber(userlabel_text_actions));

/* register the display center text actions */
  XtAddActions(center_text_actions,XtNumber(center_text_actions));

/* register the Info text actions */
  XtAddActions(info_text_actions,XtNumber(info_text_actions));

/* fetch the application-specific resources */
  XtGetApplicationResources(toplevel,(XtPointer)&iv,resources,
			                           XtNumber(resources),
			                           (ArgList)NULL,(Cardinal)0);

/* initialize the display structure to the startup position and size */
  initialize_display();

/* if necessary, read the toggle status values */
  if (initflag)
    read_toggle_values();

/* build the main display panel */
  buildmaindisplay();

/* get the values of black and white */
  if (! XtConvertAndStore(sky_widget,XtRString,&blackstring,
			                                XtRPixel,&blackpixel))
    /* that failed; set a default and hope it is OK */
    blackpixelval = 0;

  if (! XtConvertAndStore(sky_widget,XtRString,&whitestring,
			                                XtRPixel,&whitepixel))
    /* that failed; set a default and hope it is OK */
    whitepixelval = 1;

/* get the value of red for night mode */
  if (night_mode_flag) {
    if (! XtConvertAndStore(sky_widget,XtRString,&redstring,
			                                  XtRPixel,&redpixel))
      /* that failed; set a default and live with it */
      redpixelval = 1;
  }

/* set the show flag for each catalog in the initial display list */
  catalog = cat_list_head;
  while (catalog != (struct cat_header *)NULL) {
    catalog->show_flag = find_tok(catlist,catalog->menu_name,",");
    catalog = catalog->next_cat;
  }

/* build the lists of displayable objects */
  calculate_field(&display);
#ifdef DEBUG
  printf("catalogs read and initial field calculated\n");
#endif

/* realize the widgets */
  XtRealizeWidget(toplevel);
#ifdef DEBUG
  printf("XtRealizeWidget returned\n");
#endif

/* set up to handle some window manager messages */
  setup_wm_actions();

/* establish the location of the pointer position display, if necessary */
  if (use_posdisplay)
    posbox_position();

/* make the sky display's background black, border and foreground white
 * (except that in night mode, we want the border and foreground to be
 * red) */
  n = 0;
  XtSetArg(sky_args[n],XtNbackground,blackpixelval);    n++;
  if (night_mode_flag) {
    XtSetArg(sky_args[n],XtNforeground,redpixelval);    n++;
    XtSetArg(sky_args[n],XtNborderColor,redpixelval);   n++;
  }
  else {
    XtSetArg(sky_args[n],XtNforeground,whitepixelval);  n++;
    XtSetArg(sky_args[n],XtNborderColor,whitepixelval); n++;
  }
  XtSetValues(sky_widget,sky_args,n);

/* set the sky display bit gravity - we'd really like to use CenterGravity,
 * but this fails for vertical resizes under vtwm (though it works fine with
 * mwm) */
#ifdef CG
  attr.bit_gravity = CenterGravity;
#else
  attr.bit_gravity = ForgetGravity;
#endif
  XChangeWindowAttributes(XtDisplay(sky_widget),XtWindow(sky_widget),
			                                  CWBitGravity,&attr);

/* set up some color stuff */
  setup_colors();

/* set up the cursors we want */
  setup_cursors();

/* create the graphics contexts */
  setup_graphics();

/* build the popup shells for the object information */
  catalog = cat_list_head;
  while (catalog != (struct cat_header *)NULL) {
    (void)XtAddWorkProc((XtWorkProc)buildobjdisplay,(XtPointer)catalog);
    catalog = catalog->next_cat;
  }

/* build the object finder display */
  (void)XtAddWorkProc((XtWorkProc)buildfinddisplay,(XtPointer)NULL);

/* build the user label display */
  (void)XtAddWorkProc((XtWorkProc)builduserlabeldisplay,(XtPointer)NULL);

/* build the catalog selection popup shell */
  (void)XtAddWorkProc((XtWorkProc)buildcatalogmenu,(XtPointer)NULL);

/* build the star chart PostScript filename input dialog */
  (void)XtAddWorkProc((XtWorkProc)buildchartdialog,(XtPointer)NULL);

/* start a work procedure to build the popup shell for the info display */
  (void)XtAddWorkProc((XtWorkProc)buildinfodisplay,(XtPointer)NULL);

/* if the command line calls for a large display, do it */
  if (big_flag)
    (void)XtAddWorkProc((XtWorkProc)make_big_initial,(XtPointer)NULL);

/* if necessary, get the object ID label list from the initialization file */
  if (initflag)
    build_id_list(init_fd);

/* be neat and close the initialization file */
  if (initflag)
    (void)fclose(init_fd);

#ifdef DEBUG
  printf("entering main loop\n");
#endif
/* enter the event loop */
  XtMainLoop();

/* NOTREACHED */
}



/* set up the initial sky display */

static void initialize_display()

{
/* initially, there are no state structures */
  display.old_state = (struct state *)NULL;

/* clear the display undo flags */
  display.undo_resize_flag = FALSE;

/* flag that we're not doing a redraw or a rubberband */
  display.redraw_flag = FALSE;
  display.rubberband_flag = FALSE;

/* flag that we're not placing or moving an ID */
  display.place_id_flag = FALSE;
  display.move_id_flag = FALSE;

/* save the Southern Hemisphere orientation flag */
  if (iv.downunder)
    display.downunder_flag = TRUE;
  else
    display.downunder_flag = FALSE;
  /* override the downunder flag with the command-line value */
  if (opt_downunder_flag)
    display.downunder_flag = opt_downunder;

/* save the display autopopdown flag */
  autopopdown = iv.autopopdown;

/* if we're initializing from a window manager save file, the save file
 * specifies the initial position and the current position, in radians */
  if (initflag) {
    /* read the initial and current center positions from the file */
    if (fscanf(init_fd,"%*s %lf %lf\n",&display.initial_ra_rad,
	                                       &display.initial_dec_rad) != 2)
      bad_init_file("bad initial position");
    if (fscanf(init_fd,"%*s %lf %lf\n",&display.center_ra_rad,
	                                        &display.center_dec_rad) != 2)
      bad_init_file("bad center position");
    /* update the center position fields from the radian values */
    update_center_position(&display,pos_buffer);
  }
  else {
/* otherwise, the initial center position comes from the resource file or
 * command-line option */
    strcpy(pos_buffer,(char *)iv.pos);
    /* override the initial display position with the command-line value */
    if (opt_pos != (char *)NULL)
      strcpy(pos_buffer,opt_pos);
#ifdef DEBUG
    printf("initial position buffer = %s\n",pos_buffer);
#endif
    if (! calculate_position(&display,pos_buffer)) {
      /* supplied position is bad; use a known good initial position */
      strcpy(pos_buffer,INITIAL_POSITION);
      (void)calculate_position(&display,pos_buffer);

      /* and ring a bell to signify our displeasure */
      XBell(XtDisplay(sky_widget),0);
    }
#ifdef DEBUG
    printf("display center = %lf, %lf radians\n",display.center_ra_rad,
	                                             display.center_dec_rad);
#endif

/* remember the initial display position forever for the scrollbars */
    display.initial_ra_rad = display.center_ra_rad;
    display.initial_dec_rad = display.center_dec_rad;
  }

/* save the current position buffer as the old position buffer */
  strcpy(old_pos_buffer,pos_buffer);

/* save the initial active catalog list */
  strcpy(catlist,(char *)iv.firstcat);
  /* override the initial display position with the command-line value */
  if (opt_catlist != (char *)NULL)
    strcpy(catlist,opt_catlist);
#ifdef DEBUG
  printf("initial catalog list = %s\n",catlist);
#endif

/* save the initial display scale (display scale is degrees per pixel) */
  display.scale = 1.0 / iv.scale;
  /* override the display scale with the command-line value */
  if (opt_display_scale_flag)
    display.scale = 1.0 / opt_display_scale;
#ifdef DEBUG
  printf("initial display scale = %f degrees per pixel",display.scale);
  printf(" or %f pixels per degree\n",1.0 / display.scale);
#endif

/* set the initial width and height */
  display.width = iv.width;
  display.height = iv.height;

/* save the initial size forever */
  initial_width = display.width;
  initial_height = display.height;

#ifdef DEBUG
  printf("display dimensions %d x %d pixels\n",display.width,display.height);
#endif

/* compute the new display width and height in radians */
  display.width_rad = (display.width * display.scale) / DEGREES_PER_RAD;
  display.height_rad = (display.height * display.scale) / DEGREES_PER_RAD;
#ifdef DEBUG
  printf("display dimensions = %lf, %lf radians\n",display.width_rad,
	                                                 display.height_rad);
#endif

/* set the initial limiting magnitude */
  if (opt_mag_limit_flag)
    /* use the command-line value */
    mag_limit = opt_mag_limit;
  else
    /* use the resource value */
    mag_limit = iv.maglim;
  sprintf(mag_buffer,"%.1f\n",mag_limit);

/* set the scrollbar use flag */
  if (opt_scrollbar_flag)
    /* use the command-line value */
    use_scrollbars = opt_scrollbar;
  else
    /* use the resource value */
    use_scrollbars = iv.use_scrollbars;

/* set the zoom slider use flag and maximum magnification value */
  if (opt_slider_flag)
    /* use the command-line value */
    use_slider = opt_slider;
  else
    /* use the resource value */
    use_slider = iv.use_slider;

  if (opt_slidermax_flag)
    slidermax = opt_slidermax;
  else
    slidermax = iv.slidermax;

/* get the pointer position display flag */
  if (opt_posdisplay_flag)
    /* use the command-line value */
    use_posdisplay = opt_posdisplay;
  else
    /* use the resource value */
    use_posdisplay = iv.use_posdisplay;

/* set the spectral class colors for the display */
  if (starcolors[O] == (char *)NULL)
    starcolors[O] = (char *)iv.ocolor;
  if (starcolors[B] == (char *)NULL)
    starcolors[B] = (char *)iv.bcolor;
  if (starcolors[A] == (char *)NULL)
    starcolors[A] = (char *)iv.acolor;
  if (starcolors[F] == (char *)NULL)
    starcolors[F] = (char *)iv.fcolor;
  if (starcolors[G] == (char *)NULL)
    starcolors[G] = (char *)iv.gcolor;
  if (starcolors[K] == (char *)NULL)
    starcolors[K] = (char *)iv.kcolor;
  if (starcolors[M] == (char *)NULL)
    starcolors[M] = (char *)iv.mcolor;

  return;
}



/* read the values of the various main menu toggles */

static void read_toggle_values()

{
  char value[10];

/* first is the RA/dec grid on/off value */
  if (fscanf(init_fd,"%*s %s\n",value) != 1)
    bad_init_file("bad grid flag");
  if (strcmp(value,FLAG_ON_VALUE) == EQUAL)
    draw_grid = True;
  else
    draw_grid = False;

/* next is the constellation boundaries on/off value */
  if (fscanf(init_fd,"%*s %s\n",value) != 1)
    bad_init_file("bad boundary flag");
  if (strcmp(value,FLAG_ON_VALUE) == EQUAL)
    draw_bound = True;
  else
    draw_bound = False;

/* last is the object labels on/off value */
  if (fscanf(init_fd,"%*s %s\n",value) != 1)
    bad_init_file("bad label flag");
  if (strcmp(value,FLAG_ON_VALUE) == EQUAL)
    draw_ids = True;
  else
    draw_ids = False;

  return;
}



/* build the main display panel for the application */

static void buildmaindisplay()

{
  int n;
  Arg widget_args[15];
  int button_num;
  Widget longest;
  Dimension longlength;

/* set up the shell to allow resizing */
  n = 0;
  XtSetArg(widget_args[n],XtNallowShellResize,True);                     n++;
  XtSetValues(toplevel,widget_args,n);

/* create a form widget to act as a container */
  form = XtCreateManagedWidget("form",          /* widget name */
			       formWidgetClass, /* widget class */
			       toplevel,        /* parent widget */
			       (Arg *)NULL,     /* argument list */
			       0);              /* argument list size */

/* add the info button */
  button_num = 0;
  longlength = 0;
  menuinfo = makepanelbutton(button_num++,(Widget)NULL,form,
			                     PrepareInfo,(XtPointer)NULL,
			                                &longest,&longlength);

/* add the catalog menu button */
  menucat = makepanelbutton(button_num++,menuinfo,form,popup_catmenu,
			                         (XtPointer)&catmenushell,
			                                &longest,&longlength);

/* add the zoom buttons */
  zoomin = makepanelbutton(button_num++,menucat,form,
			                     display_zoom,(XtPointer)1,
			                                &longest,&longlength);
  zoomout = makepanelbutton(button_num++,zoomin,form,
			                     display_zoom,(XtPointer)-1,
			                                &longest,&longlength);

/* add the grid button */
  menugrid = makepaneltoggle(button_num++,zoomout,form,
			            handle_grid_button,(XtPointer)NULL,
			                                &longest,&longlength);
  /* if grid drawing is initially turned on, set the button state to on */
  if (draw_grid)
    XtVaSetValues(menugrid,XtNstate,True,(String *)NULL);

/* add the constellation-boundary button */
  menubound = makepaneltoggle(button_num++,menugrid,form,
			            handle_bound_button,(XtPointer)NULL,
			                                &longest,&longlength);
  /* if we have boundaries, check initial state of boundary drawing */
  if (have_boundaries)
    /* if boundary drawing is initially on, set the button state to on */
    if (draw_bound)
      XtVaSetValues(menubound,XtNstate,True,(String *)NULL);
    else
      ;
  else
    /* no boundaries; desensitize the button */
    XtSetSensitive(menubound,False);

/* add buttons to hide and erase IDs */
  menuhide = makepaneltoggle(button_num++,menubound,form,
			                     toggle_ids,(XtPointer)NULL,
			                                &longest,&longlength);
  /* if ID drawing is initially turned off, set the button state to on */
  if (! draw_ids)
    XtVaSetValues(menuhide,XtNstate,True,(String *)NULL);

  menuerase = makepanelbutton(button_num++,menuhide,form,
			                     erase_ids,(XtPointer)NULL,
			                                &longest,&longlength);

/* add the find button */
  menufind = makepanelbutton(button_num++,menuerase,form,popup_find,
			                            (XtPointer)&findshell,
			                                &longest,&longlength);

/* add the user label button */
  menuuserlabel = makepanelbutton(button_num++,menufind,form,popup_userlabel,
				                  (XtPointer)&userlabelshell,
			                                &longest,&longlength);

/* add the chart button */
  menuchart = makepanelbutton(button_num++,menuuserlabel,form,popup_chart,
			                       (XtPointer)&chartdialogshell,
			                                &longest,&longlength);

/* add the undo button */
  menuundo = makepanelbutton(button_num++,menuchart,form,Undo,(XtPointer)NULL,
                                                        &longest,&longlength);

/* add the quit button */
  menuquit = makepanelbutton(button_num++,menuundo,form,Quit,(XtPointer)NULL,
                                                        &longest,&longlength);

/* add the pointer position display, if necessary */
  if (use_posdisplay)
    makepointerpos(form,&longest,&longlength);

/* add the text field for the display center and get it updated */
  makecenterfield(form,longest);
  update_center_text();

/* build the argument list for the sky widget creation */
  n = 0;
  XtSetArg(widget_args[n],XtNwidth,display.width);                        n++;
  XtSetArg(widget_args[n],XtNheight,display.height);                      n++;
  XtSetArg(widget_args[n],XtNfromHoriz,longest);                          n++;
  XtSetArg(widget_args[n],XtNhorizDistance,5);                            n++;
  XtSetArg(widget_args[n],XtNfromVert,centerbox);                         n++;
  XtSetArg(widget_args[n],XtNvertDistance,5);                             n++;
  XtSetArg(widget_args[n],XtNresizable,True);                             n++;
  XtSetArg(widget_args[n],XtNleft,XtChainLeft);                           n++;
  XtSetArg(widget_args[n],XtNright,XtChainRight);                         n++;
  XtSetArg(widget_args[n],XtNtop,XtChainTop);                             n++;
  XtSetArg(widget_args[n],XtNbottom,XtChainBottom);                       n++;

/* create the core widget on which to draw the sky */
  sky_widget = XtCreateManagedWidget("sky",
				     coreWidgetClass,
				     form,
				     widget_args,n);

/* add an event handler to field expose events in the sky widget */
  XtAddEventHandler(sky_widget,ExposureMask,False,sky_expose,(XtPointer)NULL);

/* add a dummy event handler to cause implicit grab to give us button release
 * events (this is a real hack ...) */
  XtAddEventHandler(sky_widget,ButtonPressMask | ButtonReleaseMask,False,
		                             HandleSkyButton,(XtPointer)NULL);

/* add an event handler to field pointer button presses in the sky widget */
  XtAddEventHandler(sky_widget,ButtonPressMask,False,
		                        HandleSkyButtonPress,(XtPointer)NULL);

/* if we're using the pointer position display, pay attention to the pointer */
  if (use_posdisplay) {
/* add an event handler to field pointer entrances to the sky widget */
    XtAddEventHandler(sky_widget,EnterWindowMask,False,
		                        handle_pointer_entry,(XtPointer)NULL);

/* add an event handler to field pointer motion in the sky widget */
    XtAddEventHandler(sky_widget,PointerMotionMask,False,
		                       handle_pointer_motion,(XtPointer)NULL);
  }

/* add scrollbars if desired */
  if (use_scrollbars)
    makescrollbars(longest);

/* add a zoom slider if desired */
  if (use_slider)
    makezoomslider();

  return;
}



/* make a panel button for the main display */

static Widget makepanelbutton(i,prev_widget,parent,callback,param,
			                                   longest,longlength)

int i;
Widget prev_widget;
Widget parent;
XtCallbackProc callback;
XtPointer param;
Widget *longest;
Dimension *longlength;

{
  char buff[15];
  int n;
  Arg widget_args[5];
  Widget w;
  Dimension length;

/* create the panel button name */
  sprintf(buff,"panelbutton%d",i);

/* add the panel button */
  n = 0;
  XtSetArg(widget_args[n],XtNjustify,XtJustifyLeft);                      n++;
  XtSetArg(widget_args[n],XtNright,XtChainLeft);                          n++;
  if (prev_widget != (Widget)NULL) {
    XtSetArg(widget_args[n],XtNfromVert,prev_widget);                     n++;
    XtSetArg(widget_args[n],XtNvertDistance,PANELBUTTONSEP);              n++;
  }
  w = XtCreateManagedWidget(buff,commandWidgetClass,parent,widget_args,n);

  /* add the callback routine, if any */
  if (callback != (XtCallbackProc)NULL)
    XtAddCallback(w,XtNcallback,callback,param);

  /* get the button width and update the longest button */
  n = 0;
  XtSetArg(widget_args[n],XtNwidth,&length);                              n++;
  XtGetValues(w,widget_args,n);

  if (length > *longlength) {
    *longlength = length;
    *longest = w;
  }

  return(w);
}



/* make a panel toggle for the main display */

static Widget makepaneltoggle(i,prev_widget,parent,callback,param,
			                                   longest,longlength)

int i;
Widget prev_widget;
Widget parent;
XtCallbackProc callback;
XtPointer param;
Widget *longest;
Dimension *longlength;

{
  char buff[15];
  int n;
  Arg widget_args[5];
  Widget w;
  Dimension length;

/* create the panel button name */
  sprintf(buff,"panelbutton%d",i);

/* add the info button */
  n = 0;
  XtSetArg(widget_args[n],XtNjustify,XtJustifyLeft);                      n++;
  XtSetArg(widget_args[n],XtNright,XtChainLeft);                          n++;
  if (prev_widget != (Widget)NULL) {
    XtSetArg(widget_args[n],XtNfromVert,prev_widget);                     n++;
    XtSetArg(widget_args[n],XtNvertDistance,PANELBUTTONSEP);              n++;
  }
  w = XtCreateManagedWidget(buff,toggleWidgetClass,parent,widget_args,n);

  /* add the callback routine, if any */
  if (callback != (XtCallbackProc)NULL)
    XtAddCallback(w,XtNcallback,callback,param);

  /* get the button width and update the longest button */
  n = 0;
  XtSetArg(widget_args[n],XtNwidth,&length);                              n++;
  XtGetValues(w,widget_args,n);

  if (length > *longlength) {
    *longlength = length;
    *longest = w;
  }

  return(w);
}



/* create a labelled text field for the display center position */

static void makecenterfield(parent,leftwidget)

Widget parent;
Widget leftwidget;

{
  int n;
  Arg widgetargs[10];
  Pixel background;
  XFontStruct *font;
  int dummy;
  XCharStruct overall_size;

/* create a box widget to act as a container */
  n = 0;
  XtSetArg(widgetargs[n],XtNfromHoriz,leftwidget);                       n++;
  XtSetArg(widgetargs[n],XtNhorizDistance,5);                            n++;
  XtSetArg(widgetargs[n],XtNorientation,XtorientHorizontal);             n++;
  XtSetArg(widgetargs[n],XtNbottom,XtChainTop);                          n++;
  XtSetArg(widgetargs[n],XtNborderWidth,0);                              n++;
  XtSetArg(widgetargs[n],XtNleft,XtChainLeft);                           n++;
  XtSetArg(widgetargs[n],XtNright,XtChainLeft);                          n++;
  centerbox = XtCreateManagedWidget("centerbox",boxWidgetClass,parent,
				                        (Arg *)widgetargs,n);

/* get the background color of the box, and use it for invisible borders */
  n = 0;
  XtSetArg(widgetargs[n],XtNbackground,&background);                     n++;
  XtGetValues(centerbox,widgetargs,n);

/* create a label widget to identify the text field */
  n = 0;
  XtSetArg(widgetargs[n],XtNborderColor,background);                     n++;
  centerlabel = XtCreateManagedWidget("centerlabel",labelWidgetClass,
				             centerbox,(Arg *)widgetargs,n);

/* get the font the label uses, and use it for the text widget */
  n = 0;
  XtSetArg(widgetargs[n],XtNfont,&font);                                 n++;
  XtGetValues(centerlabel,widgetargs,n);

/* figure out the correct length for the text widgets */
  XTextExtents(font,MAX_POS_STRING,strlen(MAX_POS_STRING),
	                                 &dummy,&dummy,&dummy,&overall_size);

/* create the text widget to hold the editable field value */
  n = 0;
  XtSetArg(widgetargs[n],XtNfont,font);                                  n++;
  XtSetArg(widgetargs[n],XtNuseStringInPlace,True);                      n++;
  XtSetArg(widgetargs[n],XtNeditType,XawtextEdit);                       n++;
  XtSetArg(widgetargs[n],XtNfromHoriz,centerlabel);                      n++;
  XtSetArg(widgetargs[n],XtNhorizDistance,10);                           n++;
  XtSetArg(widgetargs[n],XtNwidth,overall_size.width);                   n++;
  XtSetArg(widgetargs[n],XtNlength,sizeof(MAX_POS_STRING));              n++;
  XtSetArg(widgetargs[n],XtNstring,pos_buffer);                          n++;
  centertext = XtCreateManagedWidget("centertext",asciiTextWidgetClass,
				              centerbox,(Arg *)widgetargs,n);

  return;
}



/* build the continuously-updated pointer position display */

static void makepointerpos(parent,longest,longlength)

Widget parent;
Widget *longest;
Dimension *longlength;

{
  int n;
  Arg widgetargs[10];
  XFontStruct *font;
  int dummy;
  XCharStruct overall_size;

/* create a box widget to act as a container */
  n = 0;
  XtSetArg(widgetargs[n],XtNorientation,XtorientVertical);               n++;
  XtSetArg(widgetargs[n],XtNborderWidth,(Dimension)0);                   n++;
  posbox = XtCreateWidget("posbox",boxWidgetClass,parent,(Arg *)widgetargs,n);

/* create a label widget to identify the text field */
  n = 0;
  XtSetArg(widgetargs[n],XtNborderWidth,0);                              n++;
  poslabel = XtCreateManagedWidget("poslabel",labelWidgetClass,
				                 posbox,(Arg *)widgetargs,n);

/* get the font the label uses, and use it for the text widgets */
  n = 0;
  XtSetArg(widgetargs[n],XtNfont,&font);                                 n++;
  XtGetValues(poslabel,widgetargs,n);

/* figure out the correct length for the right ascension text widget */
  XTextExtents(font,MAX_RA_STRING,strlen(MAX_RA_STRING),
	                                 &dummy,&dummy,&dummy,&overall_size);

/* update the longest left-side widget */
  if ((Dimension)overall_size.width > *longlength) {
    *longlength = (Dimension)overall_size.width;
    *longest = posbox;
  }

/* create the text widget to hold the field value */
  n = 0;
  XtSetArg(widgetargs[n],XtNfont,font);                                  n++;
  XtSetArg(widgetargs[n],XtNuseStringInPlace,True);                      n++;
  XtSetArg(widgetargs[n],XtNeditType,XawtextRead);                       n++;
  XtSetArg(widgetargs[n],XtNfromVert,poslabel);                          n++;
  XtSetArg(widgetargs[n],XtNvertDistance,5);                             n++;
  XtSetArg(widgetargs[n],XtNwidth,(Dimension)overall_size.width);        n++;
  XtSetArg(widgetargs[n],XtNlength,(int)sizeof(MAX_RA_STRING));          n++;
  XtSetArg(widgetargs[n],XtNstring,ra_buffer);                           n++;
  XtSetArg(widgetargs[n],XtNdisplayCaret,False);                         n++;
  pos_ratext = XtCreateManagedWidget("pos_ratext",asciiTextWidgetClass,
				                 posbox,(Arg *)widgetargs,n);

/* figure out the correct length for the declination text widget */
  XTextExtents(font,MAX_DEC_STRING,strlen(MAX_DEC_STRING),
	                                 &dummy,&dummy,&dummy,&overall_size);

/* create the text widget to hold the field value */
  n = 0;
  XtSetArg(widgetargs[n],XtNfont,font);                                  n++;
  XtSetArg(widgetargs[n],XtNuseStringInPlace,True);                      n++;
  XtSetArg(widgetargs[n],XtNeditType,XawtextRead);                       n++;
  XtSetArg(widgetargs[n],XtNfromVert,pos_ratext);                        n++;
  XtSetArg(widgetargs[n],XtNvertDistance,5);                             n++;
  XtSetArg(widgetargs[n],XtNwidth,(Dimension)overall_size.width);        n++;
  XtSetArg(widgetargs[n],XtNlength,(int)sizeof(MAX_DEC_STRING));         n++;
  XtSetArg(widgetargs[n],XtNstring,dec_buffer);                          n++;
  XtSetArg(widgetargs[n],XtNdisplayCaret,False);                         n++;
  pos_dectext = XtCreateManagedWidget("pos_dectext",asciiTextWidgetClass,
				                 posbox,(Arg *)widgetargs,n);

  return;
}



/* make the scrollbars for the main display */

static void makescrollbars(longest)

Widget longest;

{
  int n;
  Arg widget_args[15];

/* add vertical and horizontal scrollbars to move the sky display */
  n = 0;
  XtSetArg(widget_args[n],XtNlength,display.height);                      n++;
  XtSetArg(widget_args[n],XtNorientation,XtorientVertical);               n++;
  XtSetArg(widget_args[n],XtNfromHoriz,sky_widget);                       n++;
  XtSetArg(widget_args[n],XtNhorizDistance,0);                            n++;
  XtSetArg(widget_args[n],XtNfromVert,centerbox);                         n++;
  XtSetArg(widget_args[n],XtNvertDistance,5);                             n++;
  XtSetArg(widget_args[n],XtNleft,XtChainRight);                          n++;
  XtSetArg(widget_args[n],XtNright,XtChainRight);                         n++;
  XtSetArg(widget_args[n],XtNtop,XtChainTop);                             n++;
  XtSetArg(widget_args[n],XtNbottom,XtChainBottom);                       n++;
  vert_scroll = XtCreateManagedWidget("vscroll",
				      scrollbarWidgetClass,
				      form,
				      widget_args,n);

  /* add a callback for incremental scrolling */
  XtAddCallback(vert_scroll,XtNscrollProc,inc_scroll,(XtPointer)NULL);

  n = 0;
  XtSetArg(widget_args[n],XtNlength,display.width);                       n++;
  XtSetArg(widget_args[n],XtNorientation,XtorientHorizontal);             n++;
  XtSetArg(widget_args[n],XtNfromVert,sky_widget);                        n++;
  XtSetArg(widget_args[n],XtNvertDistance,0);                             n++;
  XtSetArg(widget_args[n],XtNfromHoriz,longest);                          n++;
  XtSetArg(widget_args[n],XtNhorizDistance,5);                            n++;
  XtSetArg(widget_args[n],XtNleft,XtChainLeft);                           n++;
  XtSetArg(widget_args[n],XtNright,XtChainRight);                         n++;
  XtSetArg(widget_args[n],XtNtop,XtChainBottom);                          n++;
  XtSetArg(widget_args[n],XtNbottom,XtChainBottom);                       n++;
  horiz_scroll = XtCreateManagedWidget("hscroll",
				       scrollbarWidgetClass,
				       form,
				       widget_args,n);

  /* add a callback for incremental scrolling */
  XtAddCallback(horiz_scroll,XtNscrollProc,inc_scroll,(XtPointer)NULL);

/* set up the thumb positions and sizes */
  update_scrollbar_thumbs();

  return;
}



/* make the zoom slider for the main display */

static void makezoomslider()

{
  float shown;
  XtArgVal *l_shown;
  int n;
  Arg widget_args[15];

/* compute the length of the slider's thumb */
  shown = (float)7 / display.height;
  l_shown = (XtArgVal *)&shown;

/* set up slider bar */
  n = 0;
  XtSetArg(widget_args[n],XtNlength,display.height);                      n++;
  XtSetArg(widget_args[n],XtNorientation,XtorientVertical);               n++;
  XtSetArg(widget_args[n],XtNshown,*l_shown);                             n++;
  if (use_scrollbars) {
    XtSetArg(widget_args[n],XtNfromHoriz,vert_scroll);                    n++;
    XtSetArg(widget_args[n],XtNhorizDistance,2);                          n++;
  }
  else {
    XtSetArg(widget_args[n],XtNfromHoriz,sky_widget);                     n++;
    XtSetArg(widget_args[n],XtNhorizDistance,0);                          n++;
  }
  XtSetArg(widget_args[n],XtNfromVert,centerbox);                         n++;
  XtSetArg(widget_args[n],XtNvertDistance,5);                             n++;
  XtSetArg(widget_args[n],XtNleft,XtChainRight);                          n++;
  XtSetArg(widget_args[n],XtNright,XtChainRight);                         n++;
  XtSetArg(widget_args[n],XtNtop,XtChainTop);                             n++;
  XtSetArg(widget_args[n],XtNbottom,XtChainBottom);                       n++;
  zoomslider = XtCreateManagedWidget("zoomslide",
				      scrollbarWidgetClass,
				      form,
				      widget_args,n);

  /* add a callback for incremental scrolling */
  XtAddCallback(zoomslider,XtNscrollProc,inc_zoom,(XtPointer)NULL);

/* set the initial position of the slider thumb */
  update_zoom_thumb((double)display.scale);

  return;
}



/* create popup catalog submenu for sky program */

/* ARGSUSED */

static Boolean buildcatalogmenu(client_data)

XtPointer client_data;

{
  int n;
  Arg menuargs[5];
  Pixel background;
  int button_num;
  struct cat_header *catalog;
  char namebuff[20];

/* create a popup shell to hold the menu */
  catmenushell = XtCreatePopupShell("catmenu",transientShellWidgetClass,
				                      toplevel,(Arg *)NULL,0);

/* put a box in the transient shell */
  catbox = XtCreateManagedWidget("catbox",boxWidgetClass,catmenushell,
				                               (Arg *)NULL,0);

/* get the background color of the box, and use it for invisible borders */
  n = 0;
  XtSetArg(menuargs[n],XtNbackground,&background);                        n++;
  XtGetValues(catbox,menuargs,n);

/* put a label in the box */
  n = 0;
  XtSetArg(menuargs[n],XtNborderColor,background);                        n++;
  catlabel = XtCreateManagedWidget("catlabel",labelWidgetClass,catbox,
				                           (Arg *)menuargs,n);

/* initialize the button number */
  button_num = 0;

/* create a selection button for each available catalog */
  catalog = cat_list_head;
  while (catalog != (struct cat_header *)NULL) {
    /* create the button name */
    sprintf(namebuff,"catbutton%d",button_num++);

    /* create the menu item widget with the given label */
    n = 0;
    XtSetArg(menuargs[n],XtNlabel,catalog->menu_name);                    n++;
    catalog->catselect = XtCreateManagedWidget(namebuff,toggleWidgetClass,
					                   catbox,menuargs,n);

    /* set the initial state according to the flag value */
    n = 0;
    if (catalog->show_flag) {
      XtSetArg(menuargs[n],XtNstate,True);                                n++;
    }
    else {
      XtSetArg(menuargs[n],XtNstate,False);                               n++;
    }
    XtSetValues(catalog->catselect,menuargs,n);

    /* step to the next catalog */
    catalog = catalog->next_cat;
  }

  return(True);
}



/* create popup dialog for star chart PostScript filename entry */

/* ARGSUSED */

static Boolean buildchartdialog(client_data)

XtPointer client_data;

{
  int n;
  Arg chartargs[5];
  Widget chartlabel;
  XFontStruct *font;
  int dummy;
  XCharStruct overall_size;
  Widget charttext;

/* create a popup shell to hold the dialog widget */
  n = 0;
  XtSetArg(chartargs[n],XtNallowShellResize,True);                        n++;
  chartdialogshell = XtCreatePopupShell("chartshell",
					       transientShellWidgetClass,
				                        toplevel,chartargs,n);

/* put a dialog in the transient shell */
  n = 0;
  XtSetArg(chartargs[n],XtNlabel,"Chart filename");                       n++;
  XtSetArg(chartargs[n],XtNvalue,DEFAULT_PS_OUTPUT_FILE);                 n++;
  chartdialog = XtCreateManagedWidget("chartdialog",dialogWidgetClass,
				                chartdialogshell,chartargs,n);

/* get the font the label widget uses */
  chartlabel = XtNameToWidget(chartdialog,"label");
  XtVaGetValues(chartlabel,XtNfont,&font,(String *)NULL);

/* figure out the correct initial length for the text widget */
  XTextExtents(font,"Chart filename",strlen("Chart filename"),
	                                  &dummy,&dummy,&dummy,&overall_size);

/* size the embedded text widget and make it resizable */
  n = 0;
  XtSetArg(chartargs[n],XtNresizable,True);                               n++;
  XtSetArg(chartargs[n],XtNwidth,overall_size.width);                     n++;
  charttext = XtNameToWidget(chartdialog,"value");
  XtSetValues(charttext,chartargs,n);

/* add Create and Cancel buttons to the dialog */
  XawDialogAddButton(chartdialog,"Create",CreateChart,chartdialogshell);
  XawDialogAddButton(chartdialog,"Cancel",CancelChart,chartdialogshell);

/* set things up so that a return in the text widget acts like a create */
  XtAddActions(chart_text_actions,XtNumber(chart_text_actions));
  XtOverrideTranslations(charttext,XtParseTranslationTable(chart_text_trans));

/* add an event handler to detect when the pointer leaves the window */
  XtAddEventHandler(chartdialogshell,LeaveWindowMask,False,
		                                cancel_chart,(XtPointer)NULL);

  return(True);
}



/* work procedure to create a popup object selection display */

/* ARGSUSED */

static Boolean buildfinddisplay(client_data)

XtPointer client_data;

{
  int n;
  Arg widgetargs[10];
  Pixel background;

/* create a popup shell to hold the display widgets */
  findshell = XtCreatePopupShell("findshell",transientShellWidgetClass,
			                             toplevel,(Arg *)NULL,0);

/* create a box widget to act as a container */
  findbox = XtCreateManagedWidget("findbox",boxWidgetClass,findshell,
				                              (Arg *)NULL,0);

/* get the background color of the box, and use it for invisible borders */
  n = 0;
  XtSetArg(widgetargs[n],XtNbackground,&background);                     n++;
  XtGetValues(findbox,widgetargs,n);

/* put a label in the box */
  n = 0;
  XtSetArg(widgetargs[n],XtNborderColor,background);                     n++;
  findlabel = XtCreateManagedWidget("findlabel",labelWidgetClass,findbox,
				                        (Arg *)widgetargs,n);

/* put in a text widget to hold the editable information */
  n = 0;
  XtSetArg(widgetargs[n],XtNuseStringInPlace,True);                      n++;
  XtSetArg(widgetargs[n],XtNeditType,XawtextEdit);                       n++;
  XtSetArg(widgetargs[n],XtNfromVert,findlabel);                         n++;
  XtSetArg(widgetargs[n],XtNvertDistance,5);                             n++;
  XtSetArg(widgetargs[n],XtNwidth,200);                                  n++;
  XtSetArg(widgetargs[n],XtNlength,MAXOBJNAME);                          n++;
  XtSetArg(widgetargs[n],XtNstring,object_buffer);                       n++;
  findtext = XtCreateManagedWidget("findtext",asciiTextWidgetClass,findbox,
				                               widgetargs,n);

/* initialize the object selection buffer */
  object_buffer[0] = '\0';

/* add a command widget to allow the user to tell us he's done playing */
  findack = XtCreateManagedWidget("findack",commandWidgetClass,findbox,
				                               (Arg *)NULL,0);

/* add a command widget to allow the user to cancel alterations */
  findcancel = XtCreateManagedWidget("cancel",commandWidgetClass,findbox,
				                               (Arg *)NULL,0);

/* add an event handler to detect when the pointer leaves the window */
  XtAddEventHandler(findshell,LeaveWindowMask,False,cancel_find,
		                                             (XtPointer)NULL);

/* add the callbacks necessary for the command buttons */

  /* what do to when the user hits the "find" button */
  XtAddCallback(findack,XtNcallback,FindObject,(XtPointer)NULL);

  /* what do to when the user hits the "cancel" button */
  XtAddCallback(findcancel,XtNcallback,FindObject,(XtPointer)-1);

  return(True);
}



/* work procedure to create a popup user label display */

/* ARGSUSED */

static Boolean builduserlabeldisplay(client_data)

XtPointer client_data;

{
  int n;
  Arg widgetargs[10];
  Pixel background;

/* create a popup shell to hold the display widgets */
  userlabelshell = XtCreatePopupShell("userlabelshell",
				              transientShellWidgetClass,
			                             toplevel,(Arg *)NULL,0);

/* create a box widget to act as a container */
  userlabelbox = XtCreateManagedWidget("userlabelbox",boxWidgetClass,
				               userlabelshell,(Arg *)NULL,0);

/* get the background color of the box, and use it for invisible borders */
  n = 0;
  XtSetArg(widgetargs[n],XtNbackground,&background);                     n++;
  XtGetValues(userlabelbox,widgetargs,n);

/* put a label in the box */
  n = 0;
  XtSetArg(widgetargs[n],XtNborderColor,background);                     n++;
  userlabellabel = XtCreateManagedWidget("userlabellabel",labelWidgetClass,
					   userlabelbox,(Arg *)widgetargs,n);

/* put in a text widget to hold the editable information */
  n = 0;
  XtSetArg(widgetargs[n],XtNuseStringInPlace,True);                      n++;
  XtSetArg(widgetargs[n],XtNeditType,XawtextEdit);                       n++;
  XtSetArg(widgetargs[n],XtNfromVert,userlabellabel);                    n++;
  XtSetArg(widgetargs[n],XtNvertDistance,5);                             n++;
  XtSetArg(widgetargs[n],XtNwidth,200);                                  n++;
  XtSetArg(widgetargs[n],XtNlength,MAX_ID_STRING);                       n++;
  XtSetArg(widgetargs[n],XtNstring,userlabel_buffer);                    n++;
  userlabeltext = XtCreateManagedWidget("userlabeltext",
					        asciiTextWidgetClass,
					          userlabelbox,widgetargs,n);

/* initialize the user label buffer */
  userlabel_buffer[0] = '\0';

/* add a command widget to allow the user to tell us he's done playing */
  userlabelack = XtCreateManagedWidget("userlabelack",commandWidgetClass,
				                  userlabelbox,(Arg *)NULL,0);

/* add a command widget to allow the user to cancel label generation */
  userlabelcancel = XtCreateManagedWidget("cancel",commandWidgetClass,
				                  userlabelbox,(Arg *)NULL,0);

/* add an event handler to detect when the pointer leaves the window */
  XtAddEventHandler(userlabelshell,LeaveWindowMask,False,
		                            cancel_userlabel,(XtPointer)NULL);

/* add the callbacks necessary for the command buttons */

  /* what do to when the user hits the "label" button */
  XtAddCallback(userlabelack,XtNcallback,PositionUserLabel,(XtPointer)NULL);

  /* what do to when the user hits the "cancel" button */
  XtAddCallback(userlabelcancel,XtNcallback,PositionUserLabel,(XtPointer)-1);

  return(True);
}



/* build the popup display for the info button */

/* ARGSUSED */

static Boolean buildinfodisplay(client_data)

XtPointer client_data;

{
  int n;
  Arg widgetargs[20];
  XFontStruct *labelfontstr;
  Pixel background;
  Widget formwidget;
  XCharStruct overall_size;
  int dummy;

/* create a popup shell to hold the display widgets */
  infoshell = XtCreatePopupShell("displaydata",transientShellWidgetClass,
			                             toplevel,(Arg *)NULL,0);

/* create a form widget to act as a container */
  infoform = XtCreateManagedWidget("form",formWidgetClass,infoshell,
				                              (Arg *)NULL,0);

/* get the background color of the form, and use it for invisible borders */
  n = 0;
  XtSetArg(widgetargs[n],XtNbackground,&background);                     n++;
  XtGetValues(infoform,widgetargs,n);

/* put in a label widget to display the fixed information */
  n = 0;
  XtSetArg(widgetargs[n],XtNborderColor,background);                     n++;
  infolabel = XtCreateManagedWidget("label",labelWidgetClass,infoform,
				                        (Arg *)widgetargs,n);

/* get the font the label uses, and use it for the text widgets */
  n = 0;
  XtSetArg(widgetargs[n],XtNfont,&labelfontstr);                         n++;
  XtGetValues(infolabel,widgetargs,n);

/* while we're doing text stuff, get and save the width and height of an x */
  XTextExtents(labelfontstr,"x",1,&dummy,&dummy,&dummy,&overall_size);
  x_height = overall_size.ascent;
  x_width = overall_size.width;

/* create the editable text fields for the display parameters */

  /* start with no left widget */
  formwidget = (Widget)NULL;

  /* create the panel for the magnitude limit field */
  displaymag = maketextfield(infoform,&formwidget,labelfontstr,background,
			                      "mag. limit",mag_buffer,
			                              sizeof(mag_buffer) - 1);

  /* create the panel for the display scale field */
  displayscale = maketextfield(infoform,&formwidget,labelfontstr,background,
			                      "scale (pix/deg)",scale_buffer,
			                            sizeof(scale_buffer) - 1);

/* add a command widget to allow the user to tell us he's done playing */
  n = 0;
  XtSetArg(widgetargs[n],XtNfromVert,formwidget);                         n++;
  XtSetArg(widgetargs[n],XtNvertDistance,5);                              n++;
  ack = XtCreateManagedWidget("ack",commandWidgetClass,infoform,widgetargs,n);

/* add a command widget to allow the user to cancel alterations */
  XtSetArg(widgetargs[n],XtNfromHoriz,ack);                               n++;
  XtSetArg(widgetargs[n],XtNhorizDistance,5);                             n++;
  cancel = XtCreateManagedWidget("cancel",commandWidgetClass,infoform,
			                                        widgetargs,n);

/* add an event handler to detect when the pointer leaves the window */
  if (autopopdown)
    XtAddEventHandler(infoshell,LeaveWindowMask,False,cancel_dpyinfo,
		                                             (XtPointer)NULL);

/* add the callbacks necessary for the command buttons */

  /* what do to when the user hits the "acknowledge" button */
  XtAddCallback(ack,XtNcallback,DpyInfoAcked,(XtPointer)NULL);

  /* what do to when the user hits the "cancel" button */
  XtAddCallback(cancel,XtNcallback,DpyInfoAcked,(XtPointer)-1);

  return(True);
}



/* create one labelled text field for the info display */

static Widget maketextfield(parent,leftwidget,font,bordercolor,
			                                label,buffer,buflen)

Widget parent;
Widget *leftwidget;
XFontStruct *font;
Pixel bordercolor;
char *label;
char *buffer;
int buflen;

{
  int n;
  Arg widgetargs[10];
  Widget paramform;
  Widget paramlabel;
  int dummy;
  XCharStruct overall_size;
  Widget paramtext;

/* create a form widget to act as a container */
  n = 0;
  XtSetArg(widgetargs[n],XtNfromVert,infolabel);                         n++;
  XtSetArg(widgetargs[n],XtNvertDistance,5);                             n++;
  if (*leftwidget != (Widget)NULL) {
    XtSetArg(widgetargs[n],XtNfromHoriz,*leftwidget);                    n++;
    XtSetArg(widgetargs[n],XtNhorizDistance,10);                         n++;
  }
  paramform = XtCreateManagedWidget("paramform",formWidgetClass,parent,
				                        (Arg *)widgetargs,n);

/* create a label widget to identify the text field */
  n = 0;
  XtSetArg(widgetargs[n],XtNfont,font);                                  n++;
  XtSetArg(widgetargs[n],XtNlabel,label);                                n++;
  XtSetArg(widgetargs[n],XtNborderColor,bordercolor);                    n++;
  paramlabel = XtCreateManagedWidget("paramlabel",labelWidgetClass,
				              paramform,(Arg *)widgetargs,n);

/* figure out the length of one space for the text widget */
  XTextExtents(font," ",1,&dummy,&dummy,&dummy,&overall_size);

/* create the text widget to hold the editable field value */
  n = 0;
  XtSetArg(widgetargs[n],XtNfont,font);                                  n++;
  XtSetArg(widgetargs[n],XtNuseStringInPlace,True);                      n++;
  XtSetArg(widgetargs[n],XtNeditType,XawtextEdit);                       n++;
  XtSetArg(widgetargs[n],XtNfromVert,paramlabel);                        n++;
  XtSetArg(widgetargs[n],XtNvertDistance,5);                             n++;
  XtSetArg(widgetargs[n],XtNwidth,overall_size.width * buflen);          n++;
  XtSetArg(widgetargs[n],XtNstring,buffer);                              n++;
  XtSetArg(widgetargs[n],XtNlength,buflen);                              n++;
  paramtext = XtCreateManagedWidget("paramtext",asciiTextWidgetClass,
				              paramform,(Arg *)widgetargs,n);

/* return the containing form widget */
  *leftwidget = paramform;

/* return the text widget as the function value */
  return(paramtext);
}



/* work procedure to build a popup display for the object info information */

static Boolean buildobjdisplay(catptr)

XtPointer catptr;

{
  int n;
  Arg widgetargs[5];
  struct cat_header *catalog = (struct cat_header *)catptr;

/* create the popup shell widget to hold the display */
  n = 0;
  XtSetArg(widgetargs[n],XtNallowShellResize,True);                      n++;
  catalog->datashell = XtCreatePopupShell("datashell",
					           transientShellWidgetClass,
			                              toplevel,widgetargs,n);

/* put a dialog widget in the popup shell to display the data */
  n = 0;
  XtSetArg(widgetargs[n],XtNresize,True);                                n++;
  catalog->datadialog = XtCreateManagedWidget("datadialog",dialogWidgetClass,
				              catalog->datashell,widgetargs,n);

/* add a command widget to allow the user to tell us he's done reading */
  catalog->dataack = XtCreateManagedWidget("ack",commandWidgetClass,
			                    catalog->datadialog,(Arg *)NULL,0);

/* what do to when the user hits the "ack" button */
  XtAddCallback(catalog->dataack,XtNcallback,DataAcked,
		                                (XtPointer)catalog->datashell);

/* add an event handler to fake an ack when the window gets obscured */
  if (autopopdown)
    XtAddEventHandler(catalog->datashell,VisibilityChangeMask,False,
		                                   DataAckEvent,(XtPointer)-1);

  return(True);
}



/* set up to intercept and handle some window manager messages */

static void setup_wm_actions()

{
  Display *dpy;

/* set up a quit action */
  XtAddActions(wm_msg_actions,XtNumber(wm_msg_actions));

/* tell the shell to perform quit action on intercepted WM protocol message */
  XtOverrideTranslations(toplevel,
			       XtParseTranslationTable(wm_msg_actions_trans));

/* create atoms for WM_DELETE_WINDOW and WM_SAVE_YOURSELF */
  dpy = XtDisplay(toplevel);
  wm_atoms[DELETE_ATOM] = XInternAtom(dpy,"WM_DELETE_WINDOW",False);
  wm_atoms[SAVE_ATOM] = XInternAtom(dpy,"WM_SAVE_YOURSELF",False);

/* use the atom to intercept the message */
  (void)XSetWMProtocols(dpy,XtWindow(toplevel),wm_atoms,2);

  return;
}



/* set up some color things */

static void setup_colors()

{
  Display *display;

/* get the display */
  display = XtDisplay(toplevel);

/* get the default colormap for the display */
  default_colormap = DefaultColormap(display,DefaultScreen(display));

/* if we're in night mode, get the RGB values for red and black */
  if (night_mode_flag) {
    if (! XParseColor(display,default_colormap,"red",&red_rgb)) {
      printf("Cannot find color \"red\" in color database\n");
      exit(0);
    }

    if (! XParseColor(display,default_colormap,"black",&black_rgb)) {
      printf("Cannot find color \"black\" in color database\n");
      exit(0);
    }
  }
    
  return;
}



/* set up the cursors */

static void setup_cursors()

{
  Display *sky_display;
  Window sky_window;
  Pixmap cursor_pixmap;
  XtGCMask gcmask;
  XGCValues gcvalues;
  GC zero_gc;
  XColor invisible_color;

/* get the display and the window */
  sky_display = XtDisplay(sky_widget);
  sky_window = XtWindow(sky_widget);

/* create the cursors we will use */
  normal_cursor = XCreateFontCursor(sky_display,XC_trek);
  wait_cursor = XCreateFontCursor(sky_display,XC_watch);
  rubberband_cursor = XCreateFontCursor(sky_display,XC_crosshair);

/* create an invisible cursor, too (boy, is this hard!) */

  /* create a 1 x 1 pixmap, deep enough for the screen */
  cursor_pixmap = XCreatePixmap(sky_display,(Drawable)sky_window,1,1,1);

  /* create a graphics context that will draw zeroes */
  gcmask = GCFunction;
  gcvalues.function = GXclear;
  zero_gc = XCreateGC(sky_display,(Drawable)cursor_pixmap,gcmask,&gcvalues);

  /* draw a zero into the cursor pixmap */
  XDrawPoint(sky_display,(Drawable)cursor_pixmap,zero_gc,0,0);

  /* set up an XColor with arbitrary pixel values */
  invisible_color.red = 0;
  invisible_color.green = 0;
  invisible_color.blue = 0;

  /* make a cursor out of this pixmap */
  invisible_cursor = XCreatePixmapCursor(sky_display,
					     cursor_pixmap,cursor_pixmap,
					     &invisible_color,&invisible_color,
					                                  0,0);

/* free up what we can */
  XFreeGC(sky_display,zero_gc);

/* in night mode, change the cursor colors to red on black */
  if (night_mode_flag) {
    /* set a red foreground and black background for each cursor */
    XRecolorCursor(sky_display,normal_cursor,&red_rgb,&black_rgb);
    XRecolorCursor(sky_display,wait_cursor,&red_rgb,&black_rgb);
    XRecolorCursor(sky_display,rubberband_cursor,&red_rgb,&black_rgb);

    /* make night-mode text and shell cursors for the popups */
    night_shell_cursor = XCreateFontCursor(sky_display,XC_top_left_arrow);
    XRecolorCursor(sky_display,night_shell_cursor,&red_rgb,&black_rgb);

    night_text_cursor = XCreateFontCursor(sky_display,XC_xterm);
    XRecolorCursor(sky_display,night_text_cursor,&red_rgb,&black_rgb);

    /* change the cursors for the other main display widgets, too */
    define_night_cursors();
  }

/* start with the "waiting" cursor, and let the expose events set normal */
  set_wait_cursor();

  return;
}



/* reset the normal cursor in the sky widget */

static void set_normal_cursor()

{
/* set the normal cursor in the sky widget window */
  XDefineCursor(XtDisplay(sky_widget),XtWindow(sky_widget),normal_cursor);

  return;
}



/* set the "waiting" cursor in the sky widget */

static void set_wait_cursor()

{
/* set the wait cursor in the sky widget window */
  XDefineCursor(XtDisplay(sky_widget),XtWindow(sky_widget),wait_cursor);

  return;
}



/* set the "text-box" cursor in the sky widget */

static void set_textbox_cursor()

{
/* set the wait cursor in the sky widget window */
  XDefineCursor(XtDisplay(sky_widget),XtWindow(sky_widget),invisible_cursor);

  return;
}



/* set the "rubber-band" cursor in the sky widget */

static void set_rubberband_cursor()

{
/* set the rubber-band cursor in the sky widget window */
  XDefineCursor(XtDisplay(sky_widget),XtWindow(sky_widget),rubberband_cursor);

  return;
}



/* define night cursors for the main display widgets that need it */

static void define_night_cursors()

{
/* change the color of the top level shell's cursor */
  define_night_cursor(toplevel,XC_top_left_arrow);

/* change the centertext cursor */
  define_night_cursor(centertext,XC_xterm);

/* if necessary, change the position display text field cursors */
  if (use_posdisplay) {
    define_night_cursor(pos_ratext,XC_xterm);
    define_night_cursor(pos_dectext,XC_xterm);
  }

/* if necessary, change the scrollbar cursors */
  if (use_scrollbars) {
    define_night_cursor(vert_scroll,XC_sb_v_double_arrow);
    define_night_cursor(horiz_scroll,XC_sb_h_double_arrow);
  }

/* if necessary, change the zoom slider cursor */
  if (use_slider)
    define_night_cursor(zoomslider,XC_xterm);

  return;
}



/* define a night-mode cursor for the given widget */

static void define_night_cursor(w,type)

Widget w;
unsigned int type;

{
  Display *display;
  Cursor cursor;

/* get the display */
  display = XtDisplay(w);

/* create the cursor */
  cursor = XCreateFontCursor(display,type);

/* set a red foreground and black background for each cursor */
  XRecolorCursor(display,cursor,&red_rgb,&black_rgb);

/* use this cursor for this window */
  XDefineCursor(display,XtWindow(w),cursor);

  return;
}



/* set up the graphics contexts needed to draw and identify objects */

static void setup_graphics()

{
  Display *sky_display;
  Window sky_window;
  XWindowAttributes attr;
  int i;
  XColor rgbdef;
  XtGCMask gcmask;
  XGCValues gcvalues;

/* get the display and the window */
  sky_display = XtDisplay(sky_widget);
  sky_window = XtWindow(sky_widget);

/* get attributes of the sky window */
  XGetWindowAttributes(sky_display,sky_window,&attr);
  if (((attr.visual)->class == GrayScale) ||
                                       ((attr.visual)->class == StaticGray)) {
    mono_flag = TRUE;

    /* can't do night mode on a mono display */
    if (night_mode_flag) {
      printf("Cannot do night mode without a color display\n");
      exit(0);
    }
  }

/* for a color display, get the pixel values for each desired color name */
  if (! mono_flag)
    for (i = 0; i < NUMSTARCOLORS; i++) {
      if (! XParseColor(sky_display,default_colormap,starcolors[i],&rgbdef)) {
	printf("Cannot find color \"%s\" in color database\n",starcolors[i]);
	exit(0);
      }

      /* allocate a color cell in the color map */
      if (! XAllocColor(sky_display,default_colormap,&rgbdef)) {
	printf("Cannot allocate color cell\n");
	exit(0);
      }

      /* save the pixel value for this color */
      colorpixelval[i] = (int)rgbdef.pixel;
    }

/* create the graphics context for each star color */
  gcmask = GCForeground | GCBackground | GCArcMode;
  gcvalues.background = blackpixelval;
  gcvalues.arc_mode = ArcPieSlice;
  if (mono_flag) {
    /* monochrome monitor */
    gcvalues.foreground = whitepixelval;
    mono_star_gc = XCreateGC(sky_display,sky_window,gcmask,&gcvalues);
  }
  else
    for (i = 0; i < NUMSTARCOLORS; i++) {
      if (night_mode_flag)
	gcvalues.foreground = redpixelval;
      else
	gcvalues.foreground = colorpixelval[i];
      star_gcs[i] = XCreateGC(sky_display,sky_window,gcmask,&gcvalues);
    }

/* create the graphics context for the object box */
  gcmask = GCForeground | GCBackground;
  gcvalues.background = blackpixelval;
  if (night_mode_flag)
    gcvalues.foreground = redpixelval;
  else
    gcvalues.foreground = whitepixelval;
  obj_gc = XCreateGC(sky_display,sky_window,gcmask,&gcvalues);

/* for the moment, the object box GC will do fine for the grid line GC */
  grid_gc = obj_gc;

/* create the graphics context for the text box */
  gcmask = GCForeground | GCBackground | GCFunction;
  gcvalues.background = blackpixelval;
  if (night_mode_flag)
    gcvalues.foreground = redpixelval ^ blackpixelval;
  else
    gcvalues.foreground = whitepixelval ^ blackpixelval;
  gcvalues.function = GXxor;
  text_gc = XCreateGC(sky_display,sky_window,gcmask,&gcvalues);

/* if we're initializing from a window manager save file, the save file
 * specifies the font names */
  if (initflag) {
    /* read the Latin ID font name */
    if (fscanf(init_fd,"%*s %s\n",latin_id_font_buffer) != 1)
      bad_init_file("bad Latin ID font name");
    else
      latin_id_font = latin_id_font_buffer;

    /* read the Latin constellation font name */
    if (fscanf(init_fd,"%*s %s\n",latin_con_font_buffer) != 1)
      bad_init_file("bad Latin constellation font name");
    else
      latin_con_font = latin_con_font_buffer;

    /* read the Greek font name */
    if (fscanf(init_fd,"%*s %s\n",greek_font_buffer) != 1)
      bad_init_file("bad Greek font name");
    else
      greek_font = greek_font_buffer;
  }
  else {
/* no initialization file; if a font was given on the command line, use it,
 * otherwise, grab the one from the X resources */
    if (latin_id_font == (char *)NULL)
      latin_id_font = (char *)iv.latinidfont;
    if (latin_con_font == (char *)NULL)
      latin_con_font = (char *)iv.latinconfont;
    if (greek_font == (char *)NULL)
      greek_font = (char *)iv.greekfont;
  }

/* load the fonts we'll use for the ID text */
  if ((latinidfontstr = XLoadQueryFont(sky_display,latin_id_font)) ==
                                                        (XFontStruct *)NULL) {
    fprintf(stderr,"cannot load font %s\n",latin_id_font);
    exit(1);
  }

  if ((latinconfontstr = XLoadQueryFont(sky_display,latin_con_font)) ==
                                                        (XFontStruct *)NULL) {
    fprintf(stderr,"cannot load font %s\n",latin_con_font);
    exit(1);
  }

  if ((greekidfontstr = XLoadQueryFont(sky_display,greek_font)) ==
                                                        (XFontStruct *)NULL) {
    fprintf(stderr,"cannot load font %s\n",greek_font);
    exit(1);
  }

  return;
}



/* calculate and set the location of the pointer position display */

static void posbox_position()

{
  Position sy;
  Dimension sh;
  Position mqy;
  Dimension mqh;
  Position px;
  Dimension ph;
  int box_y_dist;
  int n;
  Arg widgetargs[10];

/* if we're using scrollbars, get the y position and height of the horizontal
 * scrollbar; otherwise, get the position and height of the sky display */
  if (use_scrollbars)
    XtVaGetValues(horiz_scroll,XtNy,&sy,XtNheight,&sh,(String *)NULL);
  else
    XtVaGetValues(sky_widget,XtNy,&sy,XtNheight,&sh,(String *)NULL);

/* get the y position and height of the quit button */
  XtVaGetValues(menuquit,XtNy,&mqy,XtNheight,&mqh,(String *)NULL);

/* get the x position and height of the pointer location display box */
  XtVaGetValues(posbox,XtNx,&px,XtNheight,&ph,(String *)NULL);

/* compute the distance from the quit button to the position box */
  box_y_dist = (sy + sh) - (mqy + mqh) - ph;

/* set the required constraints for the widget */
  n = 0;
  XtSetArg(widgetargs[n],XtNfromVert,menuquit);                          n++;
  XtSetArg(widgetargs[n],XtNvertDistance,box_y_dist);                    n++;
  XtSetArg(widgetargs[n],XtNtop,XtChainBottom);                          n++;
  XtSetArg(widgetargs[n],XtNbottom,XtChainBottom);                       n++;
  XtSetArg(widgetargs[n],XtNleft,XtChainLeft);                           n++;
  XtSetArg(widgetargs[n],XtNright,XtChainLeft);                          n++;
  XtSetValues(posbox,widgetargs,n);

/* manage the widget to get it mapped */
  XtManageChild(posbox);

  return;
}



/* this is the sky expose event action code */

/* ARGSUSED */

static void sky_expose(w,client_data,event,cont_flag)

Widget w;
XtPointer client_data;
XEvent *event;
Boolean *cont_flag;

{
  Region expose_region;
  XRectangle rectangle;
  Dimension form_width, form_height;
  Position toplevel_x, toplevel_y;
  
#ifdef EXPOSE_DEBUG
  printf("expose event count = %d\n",((XExposeEvent *)event)->count);
  if (display.redraw_flag)
    printf("pending exposure events will be combined\n");
#endif

/* set the "waiting" cursor */
  set_wait_cursor();

/* if this is a resize, collect all expose events into a single region */
  if (display.redraw_flag) {
    /* create a region for combining the exposures */
    expose_region = XCreateRegion();

/* add all pending expose events to this region */
    do {
      XtAddExposureToRegion(event,expose_region);
    } while (XCheckTypedWindowEvent(XtDisplay(w),XtWindow(w),Expose,event));

    /* convert the region to an enclosing rectangle */
    XClipBox(expose_region,&rectangle);

    /* get rid of the region */
    XDestroyRegion(expose_region);

    /* reset the flag for the next redraw event */
    display.redraw_flag = FALSE;
  }
  else {
/* if not a resize, just fill in the rectangle from this expose event */
    rectangle.x = ((XExposeEvent *)event)->x;
    rectangle.y = ((XExposeEvent *)event)->y;
    rectangle.width = ((XExposeEvent *)event)->width;
    rectangle.height = ((XExposeEvent *)event)->height;
  }

#ifdef EXPOSE_DEBUG
  printf("expose at (%d,%d) for %d x %d\n",rectangle.x,rectangle.y,
	                                   rectangle.width,rectangle.height);
#endif

/* flag that we are not drawing in PostScript */
  ps_draw_flag = FALSE;

/* now redraw the sky */
  draw_sky(w,(int)rectangle.x,(int)rectangle.y,
	                          (int)rectangle.width,(int)rectangle.height);

/* draw the grid and boundary lines only on the last expose event */
  if (((XExposeEvent *)event)->count == 0) {
    if (draw_grid)
      draw_grid_lines(w,&display,(int)rectangle.x,(int)rectangle.y,
		                  (int)rectangle.width,(int)rectangle.height);

    if (draw_bound)
      draw_boundary_lines(w,&display,(int)rectangle.x,(int)rectangle.y,
			          (int)rectangle.width,(int)rectangle.height);

    /* also draw the ID texts last */
    if (draw_ids)
      draw_id_texts(w,(int)rectangle.x,(int)rectangle.y,
		                  (int)rectangle.width,(int)rectangle.height);
    }

/* restore the normal cursor */
  if (display.place_id_flag)
    set_textbox_cursor();
  else
    set_normal_cursor();

/* now that the initial exposure is over, add event handlers to field
 * configuration changes in the toplevel, form, and sky widgets */
  if (! config_flag) {
    XtAddEventHandler(toplevel,StructureNotifyMask,False,HandleTopStruct,
		                                             (XtPointer)NULL);
    XtAddEventHandler(form,StructureNotifyMask,False,HandleFormStruct,
		                                             (XtPointer)NULL);

    /* capture the initial size and position of the form */
    XtVaGetValues(toplevel,XtNwidth,&form_width,XtNheight,&form_height,
		                                              (String *)NULL);
    XtTranslateCoords(toplevel,(Position)0,(Position)0,
		                                     &toplevel_x,&toplevel_y);
    display.form_width = (int)form_width;
    display.form_height = (int)form_height;

    XtAddEventHandler(w,StructureNotifyMask,False,HandleSkyStruct,
		                                             (XtPointer)NULL);

    /* flag that the configuration events are now handled */
    config_flag = TRUE;
  }

  return;
}



/* put out the PostScript code to draw the displayed sky */

static void ps_draw_sky(display,ps_filename)

struct display *display;
char *ps_filename;

{
#ifdef DEBUG
  printf("drawing objects in area %d,%d\n",display->width,display->height);
  printf("writing to file %s\n",ps_filename);
#endif

/* flag that we are drawing in PostScript */
  ps_draw_flag = TRUE;

/* write the PostScript commands into a file */
  if (! ps_write_file(sky_widget,display,ps_filename))
    XBell(XtDisplay(sky_widget),0);

  return;
}



/* draw right ascension and declination lines on the display */

/* ARGSUSED */

static void handle_grid_button(menu_widget,client_data,call_data)

Widget menu_widget;
XtPointer client_data;
XtPointer call_data;

{
  int n;
  Arg buttonargs[5];

/* get the new state of the toggle button */
  n = 0;
  XtSetArg(buttonargs[n],XtNstate,&draw_grid);                            n++;
  XtGetValues(menu_widget,buttonargs,n);

#ifdef GRID_DEBUG
  printf("grid menu option called to turn grid flag ");
  if (draw_grid)
    printf("on\n");
  else
    printf("off\n");
#endif

/* set the "waiting" cursor */
  set_wait_cursor();

/* draw or erase grid lines, as necessary */
  if (draw_grid)
    /* set up a work procedure to do the calculations and draw the grid */
    (void)XtAddWorkProc((XtWorkProc)draw_grid_lines_work,(XtPointer)&display);
  else
    /* grid lines are not currently eraseable - clear the display and redraw */
    clear_sky(&display);

  return;
}



/* work procedure to draw grid lines on the sky display */

static Boolean draw_grid_lines_work(client_data)

XtPointer client_data;

{
  struct display *display = (struct display *)client_data;

/* draw the lines on the entire display */
  draw_grid_lines(sky_widget,display,0,0,(int)display->width,
		                                        (int)display->height);

/* restore the normal cursor */
  set_normal_cursor();

  return(True);
}



/* draw constellation boundaries on the display */

/* ARGSUSED */

static void handle_bound_button(menu_widget,client_data,call_data)

Widget menu_widget;
XtPointer client_data;
XtPointer call_data;

{
  int n;
  Arg buttonargs[5];

/* get the new state of the toggle button */
  n = 0;
  XtSetArg(buttonargs[n],XtNstate,&draw_bound);                           n++;
  XtGetValues(menu_widget,buttonargs,n);

#ifdef BOUND_DEBUG
  printf("bound menu option called to turn bound flag ");
  if (draw_bound)
    printf("on\n");
  else
    printf("off\n");
#endif

/* set the "waiting" cursor */
  set_wait_cursor();

/* draw or erase boundary lines, as necessary */
  if (draw_bound)
    /* set up a work procedure to do the calculations and draw boundaries */
    (void)XtAddWorkProc((XtWorkProc)draw_boundary_lines_work,
			                                 (XtPointer)&display);
  else
    /* boundary lines are not eraseable - clear the display and redraw */
    clear_sky(&display);

  return;
}



/* work procedure to draw constellation boundary lines on the sky display */

static Boolean draw_boundary_lines_work(client_data)

XtPointer client_data;

{
  struct display *display = (struct display *)client_data;

/* draw the lines on the entire display */
  draw_boundary_lines(sky_widget,display,0,0,(int)display->width,
		                                        (int)display->height);

/* restore the normal cursor */
  set_normal_cursor();

  return(True);
}



/* hide the object ID texts */

/* ARGSUSED */

static void toggle_ids(menu_widget,client_data,call_data)

Widget menu_widget;
XtPointer client_data;
XtPointer call_data;

{
  int n;
  Arg buttonargs[5];

/* get the new state of the toggle button */
  n = 0;
  XtSetArg(buttonargs[n],XtNstate,&draw_ids);                             n++;
  XtGetValues(menu_widget,buttonargs,n);

/* reverse the sense of the flag, which is opposite that of the toggle */
  draw_ids = ! draw_ids;

/* redraw the sky */
  clear_sky(&display);

  return;
}



/* erase all object ID texts */

/* ARGSUSED */

static void erase_ids(menu_widget,client_data,call_data)

Widget menu_widget;
XtPointer client_data;
XtPointer call_data;

{
/* if the user can't see the IDs, don't let him erase them */
  if (! draw_ids) {
    XBell(XtDisplay(sky_widget),0);
    return;
  }

/* clear the ID list */
  remove_all_ids();

/* redraw the sky */
  clear_sky(&display);

  return;
}



/* recalculate the display lists and redraw the sky display */

static void update_sky(display,clearflag)

struct display *display;
int clearflag;

{
/* set the "waiting" cursor */
  set_wait_cursor();

/* build the new object list */
  calculate_field(display);

/* clear the display if requested */
  if ((Boolean)clearflag)
    clear_sky(display);

/* update the scrollbar thumbs */
  update_scrollbar_thumbs();

/* change pointer position display to reflect its new location on the sky */
  update_position_display(display);

  return;
}



/* unconditionally redraw the sky display */

static void clear_sky(display)

struct display *display;

{
/* clear the entire sky display and generate an expose event */
  clear_area(sky_widget,0,0,display->width,display->height);

  return;
}



/* clear a specified area of a widget */

void clear_area(w,x,y,width,height)

Widget w;
int x, y;
int width, height;

{
  XClearArea(XtDisplay(w),XtWindow(w),x,y,width,height,True);

  return;
}



/* undo button callback function */

/* ARGSUSED */

static void Undo(w,client_data,call_data)

Widget w;
XtPointer client_data;
XtPointer call_data;

{
  struct state *oldstate;

/* if there is no undo information, complain */
  if (display.old_state == (struct state *)NULL) {
    XBell(XtDisplay(sky_widget),0);
    return;
  }
  else {
    /* remove the old state from the list */
    oldstate = display.old_state;
    display.old_state = display.old_state->next;
  }

/* if the shell size has changed, reset it */
  if ((oldstate->changeflags & SIZECHANGE) != 0)
    /* change the size through the form */
    if ((oldstate->shell_width == display.form_width) &&
	                      (oldstate->shell_height == display.form_height))
      ;
    else {
      XtVaSetValues(form,XtNwidth,(Dimension)oldstate->shell_width,
		            XtNheight,(Dimension)oldstate->shell_height,
		                                              (String *)NULL);
      display.undo_resize_flag = TRUE;
    }

/* if the center position has changed, reset it */
  if ((oldstate->changeflags & CENTERCHANGE) != 0) {
    display.center_ra_rad = oldstate->center_ra_rad;
    display.center_dec_rad = oldstate->center_dec_rad;
  }

/* if the magnitude limit has changed, reset it */
  if ((oldstate->changeflags & MAGCHANGE) != 0)
    mag_limit = oldstate->mag_limit;

/* if the display scale has changed, reset it */
  if ((oldstate->changeflags & SCALECHANGE) != 0)
    display.scale = oldstate->scale;

/*
 * at this point, a number of possibilities need to be handled.  generally
 * speaking, only one of the change flags will be set.  there are, however,
 * three exceptions:  both text fields in the info display could be editted
 * at once, leading to a simultaneous magnitude limit and display scale change,
 * and rubberbanding changes the shell size, center position, and display
 * scale simultaneously, except in full-screen mode, in which case the shell
 * size does not change.  there are therefore seven cases to handle:  the four
 * possible individual state changes, simultaneous info display editting, and
 * rubberbanding in the normal and full-screen-mode cases.
 */
  switch (oldstate->changeflags) {
  case SCALECHANGE:                   /* scale change only */
    update_scale();
    break;

  case CENTERCHANGE:                  /* center position change only */
    update_center_text();
    update_sky(&display,True);
    break;

  case MAGCHANGE:                     /* magnitude limit change only */
    clear_sky(&display);
    break;

  case SIZECHANGE:                   /* size change only */
    break;

  case SCALECHANGE | MAGCHANGE:       /* simultaneous info display field mod */
    /* update_scale() will automatically take care of the mag limit change */
    update_scale();
    break;

  case SIZECHANGE | CENTERCHANGE | SCALECHANGE:     /* rubberbanding */
    /* just update the center; resize will take care of the scale change */
    update_center_text();

    /* flag that a reverse rubberbanding is in progress */
    display.rubberband_flag = TRUE;

    break;

  case CENTERCHANGE | SCALECHANGE:     /* rubberbanding (full-screen mode) */
    /* update the center let the scale update change the sky */
    update_center_text();
    update_scale();

    break;

  default:
#ifdef DEBUG
    printf("unexpected change flag combination %X\n",oldstate->changeflags);
#endif
    break;
  }

/* free the old state structure */
  free((void *)oldstate);

  return;
}



/* action on receiving window manager protocol messages */

/* ARGSUSED */

static XtActionProc wm_msg_action(w,event,params,num_params)

Widget w;
XEvent *event;
String *params;
Cardinal *num_params;

{
  XClientMessageEvent *msgevent = (XClientMessageEvent *)event;

/* figure out which message we got from the window manager */
  if ((Atom)(msgevent->data.l[0]) == wm_atoms[DELETE_ATOM])
    /* WM_DELETE_WINDOWN - simply pretend the quit button was clicked */
    Quit(menuquit,(XtPointer)NULL,(XtPointer)NULL);
  else if ((Atom)(msgevent->data.l[0]) == wm_atoms[SAVE_ATOM])
    /* WM_SAVE_YOURSELF - save the current application state */
    save_state();
  else
    return;

/* NOTREACHED */
}



/* quit button callback function */

/* ARGSUSED */

static void Quit(w,client_data,call_data)

Widget w;
XtPointer client_data;
XtPointer call_data;

{
  struct cat_header *catalog;

/* close all open catalog files */
  catalog = cat_list_head;
  while (catalog != (struct cat_header *)NULL) {
    if (catalog->file_fd != (FILE *)NULL)
      (void)fclose(catalog->file_fd);

    catalog = catalog->next_cat;
  }

/* and exit */
  exit(0);
}



/* display information window popdown (implied "cancel" button) */

/* ARGSUSED */

static void cancel_dpyinfo(w,client_data,event,cont_flag)

Widget w;
XtPointer client_data;
XEvent *event;
Boolean *cont_flag;

{
/* simply call the cancel button's callback */
  DpyInfoAcked(w,(XtPointer)-1,(XtPointer)NULL);

  return;
}



/* info acknowlege button callback function */

/* ARGSUSED */

static void DpyInfoAcked(w,client_data,call_data)

Widget w;
XtPointer client_data;
XtPointer call_data;

{
  float new_mag_limit;
  boolean mag_change_flag;
  float new_display_scale;
  boolean scale_change_flag;

/* pop down the shell */
  XtPopdown(infoshell);

/* resensitize the menu info button */
  XtSetSensitive(menuinfo,True);

/* if the client data is -1, ignore the edited strings and take no action */
  if (client_data == (XtPointer)-1)
    return;

/* see if the limiting magnitude should change */
  new_mag_limit = atof(mag_buffer);
  mag_change_flag = mag_limit != new_mag_limit;

/* see if the scale should change */
  new_display_scale = 1.0 / atof(scale_buffer);
  scale_change_flag = display.scale != new_display_scale;

/* if either value changed, clone the current state */
  if (mag_change_flag || scale_change_flag)
    clone_state(&display);
  else
    /* otherwise, there's nothing to do here */
    return;

/* get the new magnitude limit of the display */
  if (mag_change_flag) {
    /* flag the limiting magnitude change */
    display.old_state->changeflags |= MAGCHANGE;

    /* store the new magnitude limit */
    mag_limit = new_mag_limit;
  }

/* if the display scale has changed, update it */
  if (scale_change_flag) {
    /* flag the display scale change */
    display.old_state->changeflags |= SCALECHANGE;

    /* update the display scale */
    display.scale = new_display_scale;

    /* this will automagically handle a limiting magnitude change, if any */
    update_scale();
  }
  else if (mag_change_flag)
/* clear away the old objects and redraw if the limiting magnitude changed */
    clear_sky(&display);
  else
    ;

  return;
}



/* object info acknowlege button callback function */

/* ARGSUSED */

static void DataAcked(w,client_data,call_data)

Widget w;
XtPointer client_data;
XtPointer call_data;

{
/* pop down the shell */
  XtPopdown((Widget)client_data);

  return;
}



/* object info fake ack event handler */

/* ARGSUSED */

static void DataAckEvent(w,client_data,event,cont_flag)

Widget w;
XtPointer client_data;
XEvent *event;
Boolean *cont_flag;

{
/* pop down the shell only if this is a visibility obscuration event */
  if (((XVisibilityEvent *)event)->state != VisibilityUnobscured)
    XtPopdown(w);

  return;
}



/* handle editing of the display center position text */

/* ARGSUSED */

static XtActionProc display_center_edited(w,event,params,num_params)

Widget w;
XEvent *event;
String *params;
Cardinal *num_params;

{
  double old_ra, old_dec;

/* do nothing if the position buffer contents didn't actually change */
  if (strcmp(pos_buffer,old_pos_buffer) == EQUAL)
    return;

/* save the current center position temporarily */
  old_ra = display.center_ra_rad;
  old_dec = display.center_dec_rad;

#ifdef DEBUG
  printf("recalculating position from buffer %s\n",pos_buffer);
#endif

/* calculate the new center position */
  if (calculate_position(&display,pos_buffer)) {
    /* position change successful; clone the current display state */
    clone_state(&display);

    /* calculate_position() already updated the display center position */
    display.old_state->changeflags = CENTERCHANGE;
    display.old_state->center_ra_rad = old_ra;
    display.old_state->center_dec_rad = old_dec;

/* update the display center position and the sky display */
    update_center_text();
    update_sky(&display,True);
  }
  else {
    /* ring a bell to indicate we were unhappy about the supplied position */
    XBell(XtDisplay(sky_widget),0);

    /* and restore the old position */
    strcpy(pos_buffer,old_pos_buffer);
    XtVaSetValues(centertext,XtNstring,(String)pos_buffer,(String *)NULL);
  }

  return;
}



/* handle editing of the Info display texts */

/* ARGSUSED */

static XtActionProc DpyInfoEdited(w,event,params,num_params)

Widget w;
XEvent *event;
String *params;
Cardinal *num_params;

{
/* simply convert this into an acknowledge button activate callback */
  DpyInfoAcked((Widget)NULL,(XtPointer)NULL,(XtPointer)NULL);

  return;
}



/* object location find/cancel button callback function */

/* ARGSUSED */

static void FindObject(w,client_data,call_data)

Widget w;
XtPointer client_data;
XtPointer call_data;

{
  double old_ra, old_dec;

/* pop down the shell */
  XtPopdown(findshell);

/* resensitize the menu find button */
  XtSetSensitive(menufind,True);

/* if the client data is -1, ignore the edited string and take no action */
  if (client_data == (XtPointer)-1)
    return;

/* save the current display position */
  old_ra = display.center_ra_rad;
  old_dec = display.center_dec_rad;

/* find the object and update the display structure */
  if (find_object_by_name(object_buffer)) {
    /* display position has changed; clone the current display state */
    clone_state(&display);

    /* find_object_by_name() already updated the display center position */
    display.old_state->changeflags = CENTERCHANGE;
    display.old_state->center_ra_rad = old_ra;
    display.old_state->center_dec_rad = old_dec;

/* update the display center position and the sky display */
    update_center_text();
    update_sky(&display,True);
  }
  else {
    /* ring a bell to indicate we couldn't find the requested object */
    XBell(XtDisplay(sky_widget),0);
  }

  return;
}



/* find the object in response to a <return> in the text field */

/* ARGSUSED */

static XtActionProc find_object_in_text(w,event,params,num_params)

Widget w;
XEvent *event;
String *params;
Cardinal *num_params;

{
/* just pretend the button was clicked */
  FindObject((Widget)NULL,(XtPointer)0,(XtPointer)NULL);

  return;
}



/* object location window popdown (implied "cancel" button) */

/* ARGSUSED */

static void cancel_find(w,client_data,event,cont_flag)

Widget w;
XtPointer client_data;
XEvent *event;
Boolean *cont_flag;

{
/* simply call the cancel button's callback */
  FindObject(w,(XtPointer)-1,(XtPointer)NULL);

  return;
}



/* pop down the object finder display */

/* ARGSUSED */

static XtActionProc popdown_find_display(w,event,params,num_params)

Widget w;
XEvent *event;
String *params;
Cardinal *num_params;

{
/* pop down the finder display shell */
  XtPopdown(findshell);

  return;
}



/* this is the action code for <return> in the chart dialog text widget */

/* ARGSUSED */

static XtActionProc chart_text_return(w,event,params,num_params)

Widget w;
XEvent *event;
String *params;
Cardinal *num_params;

{
/* simply call the chart create button callback function */
  CreateChart(w,(XtPointer)NULL,(XtPointer)NULL);

  return;
}



/* PostScript chart creation create button callback function */

/* ARGSUSED */

static void CreateChart(w,client_data,call_data)

Widget w;
XtPointer client_data;
XtPointer call_data;

{
  char *chart_file;

/* pop down the shell */
  XtPopdown(chartdialogshell);

/* resensitize the menu info button */
  XtSetSensitive(menuchart,True);

/* get the filename entered by the user */
  chart_file = XawDialogGetValueString(chartdialog);

/* go create the PostScript star chart */
  ps_draw_sky(&display,chart_file);

  return;
}



/* PostScript chart creation cancel button callback function */

/* ARGSUSED */

static void CancelChart(w,client_data,call_data)

Widget w;
XtPointer client_data;
XtPointer call_data;

{
/* pop down the shell */
  XtPopdown(chartdialogshell);

/* resensitize the menu info button */
  XtSetSensitive(menuchart,True);

  return;
}



/* chart creation window popdown (implied "cancel" button) */

/* ARGSUSED */

static void cancel_chart(w,client_data,event,cont_flag)

Widget w;
XtPointer client_data;
XEvent *event;
Boolean *cont_flag;

{
/* simply call the cancel button's callback */
  CancelChart(w,(XtPointer)NULL,(XtPointer)NULL);

  return;
}



/* info button callback function */

/* ARGSUSED */

static void PrepareInfo(w,client_data,call_data)

Widget w;
XtPointer client_data;
XtPointer call_data;

{
  float display_width, display_height;

/* start out the info display with special mode info */
  if (mono_flag)
    strcpy(info_display,"monochrome mode; ");
  else if (night_mode_flag)
    strcpy(info_display,"night mode; ");
  else
    info_display[0] = '\0';

/* calculate and format the display size */
  display_width = min(display.width * display.scale,360.0);
  display_height = min(display.height * display.scale,360.0);
  if ((display_width < (1.0 / 60)) || (display_height < (1.0 / 60))) {
    display_width *= 3600;
    display_height *= 3600;
    sprintf(&info_display[strlen(info_display)],
	      "The display is %.3g seconds wide by %.3g seconds high      \n",
	                                                      display_width,
	                                                      display_height);
  }
  else if ((display_width < 1.0) || (display_height < 1.0)) {
    display_width *= 60;
    display_height *= 60;
    sprintf(&info_display[strlen(info_display)],
	      "The display is %.3g minutes wide by %.3g minutes high      \n",
	                                                      display_width,
	                                                      display_height);
  }
  else   
    sprintf(&info_display[strlen(info_display)],
	      "The display is %.3g degrees wide by %.3g degrees high      \n",
	                                                      display_width,
	                                                      display_height);

/* if there is a grid showing, give the grid line intervals */
  if (draw_grid) {
    if (ra_line_interval < 1000)
      sprintf(&info_display[strlen(info_display)],
	                               "RA lines are 0.%.03ds apart, ",
	                                                    ra_line_interval);
    else if (ra_line_interval < (60 * 1000))
      sprintf(&info_display[strlen(info_display)],
	                                      "RA lines are %.3gs apart, ",
	                                      (float)ra_line_interval / 1000);
    else if (ra_line_interval < (60 * 60 * 1000))
      sprintf(&info_display[strlen(info_display)],
	                               "RA lines are %.3gm apart, ",
	                               (float)ra_line_interval / (60 * 1000));
    else
      sprintf(&info_display[strlen(info_display)],
	                          "RA lines are %.3gh apart, ",
	                          (float)ra_line_interval / (60 * 60 * 1000));
    if (dec_line_interval < 100)
      sprintf(&info_display[strlen(info_display)],
	                               "dec. lines are 0.%.03d\" apart\n",
	                                                   dec_line_interval);
    else if (dec_line_interval < (60 * 100))
      sprintf(&info_display[strlen(info_display)],
	                                      "dec. lines are %.3g\" apart\n",
	                                      (float)dec_line_interval / 100);
    else if (dec_line_interval < (60 * 60 * 100))
      sprintf(&info_display[strlen(info_display)],
	                               "dec. lines are %.3g' apart\n",
	                               (float)dec_line_interval / (60 * 100));
    else
      sprintf(&info_display[strlen(info_display)],
	                          "dec. lines are %.3gd apart\n",
	                          (float)dec_line_interval / (60 * 60 * 100));
  }
  else
    strcat(info_display,"\n");

/* initialize the editable magnitude buffer */
  sprintf(mag_buffer,"%.1f\n",mag_limit);

/* initialize the editable scale buffer */
  sprintf(scale_buffer,"%.2f",1 / display.scale);

/* set the label of the label widget to the info string */
  XtVaSetValues(infolabel,XtNlabel,(String)info_display,(String *)NULL);

/* set the values of the magnitude and scale text widgets */
  XtVaSetValues(displaymag,XtNstring,(String)mag_buffer,(String *)NULL);
  XtVaSetValues(displayscale,XtNstring,(String)scale_buffer,(String *)NULL);

  /* pop up the display */
  popup_shell(menuinfo,(XtPointer)&infoshell,(XtPointer)NULL);

/* in night mode, change to friendlier cursor colors */
  if (night_mode_flag) {
    XDefineCursor(XtDisplay(infoshell),XtWindow(infoshell),night_shell_cursor);
    XDefineCursor(XtDisplay(displaymag),XtWindow(displaymag),
			                                    night_text_cursor);
    XDefineCursor(XtDisplay(displayscale),XtWindow(displayscale),
			                                    night_text_cursor);
  }

  return;
}



/* handle structure changes in the toplevel widget - we do this because the
 * vtwm window manager, on a resize, can deliver an expose event to the shell
 * for the old size before telling the shell to change to the new size.  this
 * causes the sky widget to get an Expose for the old size before getting the
 * ConfigureNotify for the new size.  that Expose is useless to us, and in
 * fact causes extra work, so the code below attempts to remove it. */

/* ARGSUSED */

static void HandleTopStruct(w,client_data,event,cont_flag)

Widget w;
XtPointer client_data;
XEvent *event;
Boolean *cont_flag;

{
  XConfigureEvent *ev = (XConfigureEvent *)event;
  XEvent expose_event;

/* ignore all Structure events except ConfigureNotify */
  if (ev->type != ConfigureNotify)
    return;

#ifdef CONFIGURE_DEBUG
  printf("top config at (%d, %d) for %d x %d occurred\n",ev->x,ev->y,
	                                                ev->width,ev->height);
#endif

/* remove an expose event for the sky widget if this is a resize */
  if ((ev->width != display.form_width) ||
                                        (ev->height != display.form_height)) {
#ifdef CONFIGURE_DEBUG
      printf("top config is resize\n");
#endif

      /* find and remove a sky expose, if any (fixup for vtwm) */
      XCheckTypedWindowEvent(XtDisplay(sky_widget),XtWindow(sky_widget),
			                                Expose,&expose_event);
    }

  return;
}



/* handle structure changes in the form widget - we don't always seem to get
 * a ConfigureNotify for the toplevel shell when one would logically expect
 * one, so we use this event handler to keep track of the toplevel size.
 * this way, we're sure we have the correct size on hand.  since this rou-
 * tine is available, we also keep track of resizing here for the purposes
 * of invalidating the undo function. */

/* ARGSUSED */

static void HandleFormStruct(w,client_data,event,cont_flag)

Widget w;
XtPointer client_data;
XEvent *event;
Boolean *cont_flag;

{
  XConfigureEvent *ev = (XConfigureEvent *)event;

/* ignore all Structure events except ConfigureNotify */
  if (ev->type != ConfigureNotify)
    return;

#ifdef CONFIGURE_DEBUG
  printf("form config at (%d, %d) for %d x %d occurred\n",
	                                          ev->x,ev->y,
	                                          ev->width,ev->height);
#endif

/* if this is a resize, save the state for undo */
  if ((display.form_width == ev->width) && (display.form_height == ev->height))
    ;
  else {
#ifdef CONFIGURE_DEBUG
    printf("form config is resize\n");
#endif
    /* don't save state if we're in an undo or rubberband */
    if (display.undo_resize_flag || display.rubberband_flag)
      display.undo_resize_flag = FALSE;
    else {
/* this is a resize; save the previous display state */
      clone_state(&display);

/* flag that the shell values are changing */
      display.old_state->changeflags = SIZECHANGE;
    }

/* save the new form size */
    display.form_width = ev->width;
    display.form_height = ev->height;
  }

  return;
}



/* handle sky display size changes - both window manager resizes and
 * rubber-banding */

/* ARGSUSED */

static void HandleSkyStruct(w,client_data,event,cont_flag)

Widget w;
XtPointer client_data;
XEvent *event;
Boolean *cont_flag;

{
  XConfigureEvent *ev = (XConfigureEvent *)event;

/* ignore all Structure events except ConfigureNotify */
  if (ev->type != ConfigureNotify)
    return;

#ifdef CONFIGURE_DEBUG
  printf("sky widget configuration change to %d x %d\n",ev->width,ev->height);
#endif

/* if this isn't a resize, just ignore it */
  if ((display.width == ev->width) && (display.height == ev->height))
    return;

#ifdef CONFIGURE_DEBUG
    printf("sky config is resize\n");
#endif

/* plug in the new width and height */
  display.width = ev->width;
  display.height = ev->height;

/* compute the new display width and height in radians */
  display.width_rad = (display.width * display.scale) / DEGREES_PER_RAD;
  display.height_rad = (display.height * display.scale) / DEGREES_PER_RAD;

/* if this is a rubberband, set a flag so the expose event handler will
 * combine pending exposures (we need to redraw the entire display anyway) */
  if (display.rubberband_flag) {
    display.redraw_flag = TRUE;

    /* turn off the rubberband-in-progress flag */
    display.rubberband_flag = FALSE;

    /* update the display to reflect the size change */
    update_sky(&display,True);
  }
  else
/* this is a window manager resize; just recalculate the field and let the
 * expose events do their thing */
    update_sky(&display,False);

  return;
}



/* dummy event handler so implicit grabs give us button release events */

/* ARGSUSED */

static void HandleSkyButton(w,client_data,event,cont_flag)

Widget w;
XtPointer client_data;
XEvent *event;
Boolean *cont_flag;

{
  return;
}



/* handle button presses in the sky widget */

/* ARGSUSED */

static void HandleSkyButtonPress(w,client_data,event,cont_flag)

Widget w;
XtPointer client_data;
XEvent *event;
Boolean *cont_flag;

{
  XButtonPressedEvent *ev = (XButtonPressedEvent *)event;

/* ignore all button presses during rubberbanding or ID placement */
  if (display.rubberband_flag || display.place_id_flag)
    return;

/* check button presses for legal combinations */
  if (ev->button == Button1)
    /* if button 1 was pressed, ... */
    if ((ev->state & (ShiftMask | ControlMask)) == (ShiftMask | ControlMask))
      /* control-shifted - simulate third button (recenter the display) */
      handle_recenter_event(ev);
    else if ((ev->state & ShiftMask) == ShiftMask)
      /* shifted - handle rubberbanding */
      handle_rubberband(ev->x,ev->y);
    else
      /* unmodified - display star information */
      displayobjinfo(ev);
  else if (ev->button == Button2) 
    /* if button 2 was pressed, ... */
    if ((ev->state & (ShiftMask | ControlMask)) == (ShiftMask | ControlMask))
      /* control-shifted - move the selected label */
      move_label(ev->x,ev->y);
    else if ((ev->state & ShiftMask) == ShiftMask)
      /* shifted - remove a label */
      remove_label(ev->x,ev->y);
    else if ((ev->state & ControlMask) == ControlMask)
      /* control-pressed - add constellation label */
      add_constellation_label(ev->x,ev->y);
    else
      /* unmodified - add object or user label */
      if (user_label_flag)
	add_user_label(ev->x,ev->y);
      else
	add_object_label(ev->x,ev->y);
  else if (ev->button == Button3)
/* if button 3 was pressed, recenter the display */
    handle_recenter_event(ev);
  else
    /* silently ignore any other button-press events */
    ;

  return;
}



/* handle button releases in the sky widget */

/* ARGSUSED */

static void HandleSkyButtonRelease(w,client_data,event,cont_flag)

Widget w;
XtPointer client_data;
XEvent *event;
Boolean *cont_flag;

{
  XButtonReleasedEvent *ev = (XButtonReleasedEvent *)event;

/* dispatch according to the button released */
  if ((ev->button == Button1) && display.rubberband_flag)
    handle_button1_release(client_data,ev);
  else if ((ev->button == Button2) && display.place_id_flag)
    handle_button2_release(client_data,ev);
  else
    /* ignore any other button-release events */
    ;

  return;
}



/* handle the release of button 1 - final position of rubberband box */

static void handle_button1_release(client_data,event)

XtPointer client_data;
XButtonReleasedEvent *event;

{
  struct rubberband *rb = (struct rubberband *)client_data;

/* remove event handlers */
  XtRemoveEventHandler(sky_widget,Button1MotionMask,True,
		                        handle_button1_motion,(XtPointer)rb);
  XtRemoveEventHandler(sky_widget,ButtonReleaseMask,True,
		                        HandleSkyButtonRelease,(XtPointer)rb);

/* remove the rubberband box */
  draw_rubberband_box(sky_widget,rb);

/* if the pointer didn't move, ignore this rubberband attempt */
  if ((rb->points[0].x == rb->points[2].x) &&
                                       (rb->points[0].y == rb->points[2].y)) {
    /* flag no rubberband in progress */
    display.rubberband_flag = FALSE;
 
    /* free the rubberband structure */
    free((void *)rb);

    return;
  }
  
/* update to the final button position */
  update_rubberband_box(&display,rb,event->x,event->y);

/* in the Southern Hemisphere, compensate for south up */
  if (display.downunder_flag) {
    rb->midpoint.x = display.width - rb->midpoint.x;
    rb->midpoint.y = display.height - rb->midpoint.y;
  }

/* save the previous display state */
  clone_state(&display);

/* flag that the center position is changing */
  display.old_state->changeflags = CENTERCHANGE;

/* center the display on the center of the rubberband box */
  X_to_pos_rad(&display,(int)rb->midpoint.x,(int)rb->midpoint.y,
	                      &display.center_ra_rad,&display.center_dec_rad);
  update_center_text();

/* match the aspect ratio of the rubberband box, but if we're in full-screen
 * mode, just rescale to approximate the rubberband */
  if (big_flag)
    change_scale(rb);
  else
    change_aspect(rb);

/* free the rubberband structure */
  free((void *)rb);

/* in full-screen mode, we have to do some stuff that would ordinarily be
 * handled by the toplevel shell update (see HandleSkyStruct()) */
  if (big_flag) {
    /* tell the expose event handler that this is a total redraw */
    display.redraw_flag = TRUE;

    /* turn off rubberband-in-progress */
    display.rubberband_flag = FALSE;

    /* update the sky scale */
    update_scale();
  }

  return;
}



/* change the aspect ratio of the display to match the rubberband box */

static void change_aspect(rb)

struct rubberband *rb;

{
  Dimension sky_curr_width, sky_curr_height;
  double rb_east_west, rb_north_south;
  double rb_aspect;
  Dimension new_sky_width, new_sky_height;
  Dimension toplevel_curr_width, toplevel_curr_height;
  Dimension new_toplevel_width, new_toplevel_height;
  Display *dpy;
  int screen;
  Dimension dpy_width, dpy_height;
  Dimension desired_sky_width, desired_sky_height;
  float reduction;
  int n;
  Arg widgetargs[5];

/* get the current width and height of the sky display */
  XtVaGetValues(sky_widget,XtNwidth,&sky_curr_width,
		                   XtNheight,&sky_curr_height,(String *)NULL);

/* compute the aspect ratio of the rubberband box */
  rb_east_west = sqrt((double)
		            ((rb->points[1].x - rb->points[rb->closeidx].x) *
			     (rb->points[1].x - rb->points[rb->closeidx].x) +
		             (rb->points[1].y - rb->points[rb->closeidx].y) *
		             (rb->points[1].y - rb->points[rb->closeidx].y)));
  rb_north_south = sqrt((double)
			      ((rb->points[rb->faridx].x - rb->points[1].x) *
			       (rb->points[rb->faridx].x - rb->points[1].x) +
		               (rb->points[rb->faridx].y - rb->points[1].y) *
		               (rb->points[rb->faridx].y - rb->points[1].y)));
  rb_aspect = rb_east_west / rb_north_south;

#ifdef RB_DEBUG
  printf("box aspect ratio = %lf / %lf = %lf\n",rb_east_west,rb_north_south,
	                                                           rb_aspect);
  printf("rubber band started at (%d,%d) and ended at (%d,%d)\n",
                                        rb->points[0].x,rb->points[0].y,
	                                     rb->points[2].x,rb->points[2].y);
  printf("current display = %d x %d\n",sky_curr_width,sky_curr_height);
#endif

/* calculate a new size for the display, using the rubber-band box aspect
 * ratio, preserving the current area */
  new_sky_width = sqrt((sky_curr_width * sky_curr_height) * rb_aspect) + 0.5;
  new_sky_height = sqrt((sky_curr_width * sky_curr_height) / rb_aspect) + 0.5;

#ifdef RB_DEBUG
  printf("new display size = %d x %d\n",new_sky_width,new_sky_height);
  printf("old display scale = %f pixels per degree\n",1 / display.scale);
#endif

/* flag that the display scale is changing */
  display.old_state->changeflags |= SCALECHANGE;

/* change the display scale by the expansion factor */
  display.scale /= new_sky_width / rb_east_west;

/* flag that the toplevel shell is changing */
  display.old_state->changeflags |= SIZECHANGE;

/* get the current size of the top level widget */
  XtVaGetValues(form,XtNwidth,&toplevel_curr_width,
		              XtNheight,&toplevel_curr_height,(String *)NULL);

#ifdef RB_DEBUG
  printf("current top level size = %d x %d\n",toplevel_curr_width,
	                                                toplevel_curr_height);
#endif

/* compute the new top level dimensions */
  new_toplevel_width = toplevel_curr_width - sky_curr_width + new_sky_width;
  new_toplevel_height = toplevel_curr_height -
                                           sky_curr_height + new_sky_height;
#ifdef RB_DEBUG
  printf("preliminary new display scale = %f\n",1 / display.scale);
  printf("preliminary new toplevel size = %d x %d\n",new_toplevel_width,
	                                                 new_toplevel_height);
#endif

/* get the screen dimensions, allowing for the window manager title bar */
  dpy = XtDisplay(toplevel);
  screen = DefaultScreen(dpy);
  dpy_width = DisplayWidth(dpy,screen);
  dpy_height = DisplayHeight(dpy,screen) - WM_TITLEHEIGHT;
#ifdef RB_DEBUG
  printf("screen size = %d x %d\n",dpy_width,dpy_height);
#endif

/* if the new top level size exceeds the screen size, reduce sky to fit */
  if ((new_toplevel_width > dpy_width) || (new_toplevel_height > dpy_height)) {
    if (new_toplevel_width > dpy_width)
      desired_sky_width = new_sky_width - (new_toplevel_width - dpy_width);
    else
      desired_sky_width = new_sky_width;
    if (new_toplevel_height > dpy_height)
      desired_sky_height = new_sky_height - (new_toplevel_height - dpy_height);
    else
      desired_sky_height = new_sky_height;
    
    /* calculate the needed reduction value */
    reduction = min((float)desired_sky_width / new_sky_width,
		                   (float)desired_sky_height / new_sky_height);
#ifdef RB_DEBUG
    printf("reduction value = %f\n",reduction);
#endif

    /* change the display scale correspondingly */
    display.scale /= reduction;

    /* and recalculate the final size of the toplevel widget */
    new_toplevel_width = toplevel_curr_width - sky_curr_width +
                                             new_sky_width * reduction + 0.5;
    new_toplevel_height = toplevel_curr_height - sky_curr_height +
                                             new_sky_height * reduction + 0.5;

#ifdef RB_DEBUG
    printf("new display scale = %f pixels per degree\n",1 / display.scale);
    printf("new top level size = %d x %d\n",new_toplevel_width,
	                                                 new_toplevel_height);
#endif
  }

#if 0
/* recalculate the toplevel origin so it stays on the screen */
  if ((toplevel_x + new_toplevel_width) > dpy_width)
    toplevel_x -= (toplevel_x + new_toplevel_width) - dpy_width;
  if ((toplevel_y + new_toplevel_height) > dpy_height)
    toplevel_y -= (toplevel_y + new_toplevel_height) - dpy_height;
#ifdef RB_DEBUG
  printf("new toplevel position = (%d, %d)\n",toplevel_x,toplevel_y);
#endif
#endif

/* resize the top level window through the form container */
  n = 0;
  XtSetArg(widgetargs[n],XtNwidth,new_toplevel_width);                    n++;
  XtSetArg(widgetargs[n],XtNheight,new_toplevel_height);                  n++;
  XtSetValues(form,widgetargs,n);

#if 0
/* move the top level window to keep things on the screen */
  n = 0;
  XtSetArg(widgetargs[n],XtNx,toplevel_x);                                n++;
  XtSetArg(widgetargs[n],XtNy,toplevel_y);                                n++;
  XtSetValues(toplevel,widgetargs,n);
#endif

/* update the zoom thumb for the new display scale */
  update_zoom_thumb((double)display.scale);

  return;
}



/* change the scale of the display to approximate the rubberband box */

static void change_scale(rb)

struct rubberband *rb;

{
  Dimension sky_curr_width, sky_curr_height;
  double rb_east_west, rb_north_south;
  float expansion;

/* get the current width and height of the sky display */
  XtVaGetValues(sky_widget,XtNwidth,&sky_curr_width,
		                   XtNheight,&sky_curr_height,(String *)NULL);

/* compute the size of the rubberband box */
  rb_east_west = sqrt((double)
		            ((rb->points[1].x - rb->points[rb->closeidx].x) *
			     (rb->points[1].x - rb->points[rb->closeidx].x) +
		             (rb->points[1].y - rb->points[rb->closeidx].y) *
		             (rb->points[1].y - rb->points[rb->closeidx].y)));
  rb_north_south = sqrt((double)
			      ((rb->points[rb->faridx].x - rb->points[1].x) *
			       (rb->points[rb->faridx].x - rb->points[1].x) +
		               (rb->points[rb->faridx].y - rb->points[1].y) *
		               (rb->points[rb->faridx].y - rb->points[1].y)));

#ifdef RB_DEBUG
  printf("box size = %lf x %lf\n",rb_east_west,rb_north_south);
  printf("rubber band started at (%d,%d) and ended at (%d,%d)\n",
                                        rb->points[0].x,rb->points[0].y,
	                                     rb->points[2].x,rb->points[2].y);
  printf("current display = %d x %d\n",sky_curr_width,sky_curr_height);
#endif

/* calculate an expansion factor for the display as the smaller of the
 * expansion factors necessary to fit the rubberband box into the current
 * display size */
  expansion = min((float)sky_curr_width / rb_east_west,
		                     (float)sky_curr_height / rb_north_south);

#ifdef RB_DEBUG
  printf("display expansion factor = %f\n",expansion);
#endif

/* flag that the display scale is changing */
  display.old_state->changeflags |= SCALECHANGE;

/* change the display scale by the expansion factor */
  display.scale /= expansion;

  return;
}



/* handle the release of button 2 - final position of ID text */

static void handle_button2_release(client_data,event)

XtPointer client_data;
XButtonReleasedEvent *event;

{
  struct id_node *id = (struct id_node *)client_data;

/* remove event handlers */
  XtRemoveEventHandler(sky_widget,Button2MotionMask,True,
		                        handle_button2_motion,(XtPointer)id);
  XtRemoveEventHandler(sky_widget,ButtonReleaseMask,True,
		                        HandleSkyButtonRelease,(XtPointer)id);

/* flag that the ID placement action is completed */
  display.place_id_flag = FALSE;

/* remove the locator box */
  draw_box(sky_widget,id->ul_x,id->ul_y,id->lr_x,id->lr_y);

/* set up the final position of the label in the display */
  finalize_label_pos(&display,id,event->x,event->y);

/* redraw any ID strings marked as having been stepped on */
  draw_marked_ids(sky_widget);

/* draw the ID string and add it to the ID display list */
  draw_id(sky_widget,id);
  add_id_to_list(id);

/* clear the flag in case this was a user label */
  user_label_flag = FALSE;

/* if this was a label move, warp the pointer back to its expected location */
  if (display.move_id_flag) {
    XWarpPointer(XtDisplay(sky_widget),XtWindow(sky_widget),
		                             XtWindow(sky_widget),0,0,0,0,
		                             id->ul_x + display.ptr_x_offset,
		                             id->ul_y + display.ptr_y_offset);
    display.move_id_flag = FALSE;
  }

/* and restore the normal cursor */
  set_normal_cursor();

  return;
}



/* display information about the object clicked on */

static void displayobjinfo(event)

XButtonPressedEvent *event;

{
  struct obj_node *obj;
  struct cat_header *catalog;
  Position x, y;

/* find the closest object; if none close enough, just ignore the event */
  if ((obj = find_object(event->x,event->y,&catalog)) ==
                                                    (struct obj_node *)NULL) {
    XBell(XtDisplay(sky_widget),0);
    return;
  }

  /* calculate the position of the popup shell */
  XtTranslateCoords(sky_widget,0,0,&x,&y);

/* find, format, and display this object's info */
  (*catalog->display_data)(catalog->file_fd,obj->array_pos);

  /* set the label of the label widget to the info string */
  XtVaSetValues(catalog->datadialog,XtNlabel,(String)obj_info,(String *)NULL);

  /* move the popup shell into position */
  XtVaSetValues(catalog->datashell,XtNx,x,XtNy,y,(String *)NULL);

  XtPopup(catalog->datashell,XtGrabNonexclusive);

/* in night mode, change to friendlier cursor colors */
  if (night_mode_flag)
    XDefineCursor(XtDisplay(catalog->datashell),XtWindow(catalog->datashell),
		                                          night_shell_cursor);

  return;
}



/* handle pointer entry to the sky widget */

/* ARGSUSED */

static void handle_pointer_entry(w,client_data,event,cont_flag)

Widget w;
XtPointer client_data;
XEvent *event;
Boolean *cont_flag;

{
  int x, y;

/* get the coordinates of the event */
  x = ((XEnterWindowEvent *)event)->x;
  y = ((XEnterWindowEvent *)event)->y;

/* the entry position likes to have a negative number in one or the other
 * coordinate; I presume this is due to the border width */
  if (x < 0)
    x = 0;
  if (y < 0)
    y = 0;

/* change the position display to reflect the pointer entry */
  change_position_display(&display,x,y);

  return;
}



/* handle pointer motion in the sky widget */

/* ARGSUSED */

static void handle_pointer_motion(w,client_data,event,cont_flag)

Widget w;
XtPointer client_data;
XEvent *event;
Boolean *cont_flag;

{
/* extract the pointer position and change the pointer position display */
  change_position_display(&display,((XMotionEvent *)event)->x,
			                          ((XMotionEvent *)event)->y);

  return;
}



/* compute the RA and dec of the given pointer position and change display */

static void change_position_display(display,x,y)

struct display *display;
int x, y;

{
  double ra_rad, dec_rad;
  struct ra_pos ra_pos;
  struct dec_pos dec_pos;

/* see if we're configured for the Southern Hemisphere */
  if (display->downunder_flag) {
    /* yes; hammer the X coordinates for inverted, mirror-image display */
    x = display->width - x;
    y = display->height - y;
  }

/* compute the sky position of the pointer */
  X_to_pos_rad(display,x,y,&ra_rad,&dec_rad);

/* convert the position in radians to sexigesimal */
  ra_pos = rad_to_ra(ra_rad);
  dec_pos = rad_to_dec(dec_rad);

#ifdef ROUND_DISPLAY_POS
/* round the display position for neatness */
  round_ra(&ra_pos);
  round_dec(&dec_pos);
#endif

/* update the RA and dec position buffers */
  sprintf(ra_buffer,"%02dh %02dm %02ds.%.03d",ra_pos.hours,ra_pos.minutes,
	                                   ra_pos.seconds,ra_pos.thousandths);

  sprintf(dec_buffer,"%c%02dd %02d' %02d\".%.02d",dec_pos.sign < 0 ? '-' : '+',
	                                                   dec_pos.degrees,
	                                                   dec_pos.minutes,
	                                                   dec_pos.seconds,
	                                                   dec_pos.hundredths);

/* update the pointer position display */
  XtVaSetValues(pos_ratext,XtNstring,(String)ra_buffer,(String *)NULL);
  XtVaSetValues(pos_dectext,XtNstring,(String)dec_buffer,(String *)NULL);

  return;
}



/* get the current pointer position and update the pointer position display */

static void update_position_display(display)

struct display *display;

{
  Window dummyw;
  int dummyi;
  int x, y;
  unsigned int dummyui;

/* get the current pointer position */
  if (XQueryPointer(XtDisplay(sky_widget),XtWindow(sky_widget),
		              &dummyw,&dummyw,&dummyi,&dummyi,&x,&y,&dummyui))

    /* see if the pointer is within the sky display */
    if (((x >= -1) && (x <= display->width)) &&
	                              ((y >= -1) && (y <= display->height))) {
      /* handle negative coordinates (due to border width?) */
      if (x < 0)
	x = 0;
      if (y < 0)
	y = 0;

      /* update the position display to correspond to the pointer location */
      change_position_display(display,x,y);
    }

  return;
}



/* begin rubber-band of display region */

static void handle_rubberband(x,y)

int x, y;

{
  struct rubberband *rb;
  int i;

/* flag that we're doing a rubberband action */
  display.rubberband_flag = TRUE;

/* get a rubberband structure for this rubber-banding */
  rb = (struct rubberband *)malloc(sizeof(struct rubberband));

/* fill in the initial point list */
  for (i = 0; i < NUM_ENCLOSURE_POINTS; i++) {
    rb->points[i].x = (short)x;
    rb->points[i].y = (short)y;
  }
  rb->midpoint.x = (short)x;
  rb->midpoint.y = (short)y;

/* register for pointer motion events under button 1 */
  XtAddEventHandler(sky_widget,Button1MotionMask,False,
		                        handle_button1_motion,(XtPointer)rb);
/* register for button release events */
  XtAddEventHandler(sky_widget,ButtonReleaseMask,False,
		                        HandleSkyButtonRelease,(XtPointer)rb);

/* set rubber-banding cursor */
  set_rubberband_cursor();

/* draw the initial box */
  draw_rubberband_box(sky_widget,rb);

  return;
}



/* handle motion in sky widget with button 1 pressed (area rubberband) */

/* ARGSUSED */

static void handle_button1_motion(w,client_data,event,cont_flag)

Widget w;
XtPointer client_data;
XEvent *event;
Boolean *cont_flag;

{
  struct rubberband *rb = (struct rubberband *)client_data;

/* redraw the current rubberband box to erase it */
  draw_rubberband_box(w,rb);

/* update the pointer position */
  update_rubberband_box(&display,rb,
			((XMotionEvent *)event)->x,((XMotionEvent *)event)->y);

/* draw the new rubberband box */
  draw_rubberband_box(w,rb);

  return;
}



/* handle motion in sky widget with button 2 pressed (text box movement) */

/* ARGSUSED */

static void handle_button2_motion(w,client_data,event,cont_flag)

Widget w;
XtPointer client_data;
XEvent *event;
Boolean *cont_flag;

{
  struct id_node *id = (struct id_node *)client_data;

/* redraw the current text box to erase it */
  draw_box(w,id->ul_x,id->ul_y,id->lr_x,id->lr_y);

/* update the position to the new location */
  id->ul_x = ((XMotionEvent *)event)->x;
  id->ul_y = ((XMotionEvent *)event)->y;

/* draw the new text box */
  draw_box(w,id->ul_x,id->ul_y,id->lr_x,id->lr_y);

  return;
}



/* update the rubberband box to the new pointer position */

static void update_rubberband_box(display,rb,x,y)

struct display *display;
struct rubberband *rb;
int x, y;

{
  double xc, yc;
  double x1, y1;
  double x3, y3;
  double midx, midy;
  double closex, closey;
  double farx, fary;
  double oslope;
  double nslope, nint;
  double intx, inty;
  double x2, y2;
  double x4, y4;

/* update the X position of the opposite corner of the rectangle */
  rb->points[2].x = (short)x;
  rb->points[2].y = (short)y;
#ifdef RB_DEBUG
  printf("X coords of initial point = (%d, %d)\n",rb->points[0].x,
	                                                     rb->points[0].y);
  printf("X coords of pointer = (%d, %d)\n",rb->points[2].x,rb->points[2].y);
#endif

/* pick which pole to use for the pole-centered Cartesian coordinates */
  if (display->downunder_flag) {
/* compute the y-flipped X coordinates of the south celestial pole */
    xc = (double)display->width / 2;
    yc = ((90 + RAD_TO_DEG(display->center_dec_rad)) / display->scale) +
                                                  (double)display->height / 2;
#ifdef RB_DEBUG
    printf("y-flipped X coordinates of the SCP = (%lf, %lf)\n",xc,yc);
#endif
  }
  else {
/* compute the y-flipped X coordinates of the north celestial pole */
    xc = (double)display->width / 2;
    yc = ((90 - RAD_TO_DEG(display->center_dec_rad)) / display->scale) +
                                                  (double)display->height / 2;
#ifdef RB_DEBUG
    printf("y-flipped X coordinates of the NCP = (%lf, %lf)\n",xc,yc);
#endif
  }

/* convert the initial point and the current pointer position to pole-centered
 * Cartesian coordinates */
  x1 = rb->points[0].x - xc;
  y1 = (display->height - rb->points[0].y) - yc;
  x3 = rb->points[2].x - xc;
  y3 = (display->height - rb->points[2].y) - yc;
#ifdef RB_DEBUG
  printf("polar cartesian coords of initial point = (%lf, %lf)\n",x1,y1);
  printf("polar cartesian coords of pointer = (%lf, %lf)\n",x3,y3);
#endif

/* step 1:  find midpoint of line between the given points (center of box) */
  midx = (x1 + x3) / 2;
  midy = (y1 + y3) / 2;

  /* save the X coordinates of the new box center point */
  rb->midpoint.x = midx + xc + 0.5;
  rb->midpoint.y = display->height - (midy + yc + 0.5);
#ifdef RB_DEBUG
  printf("X coords of box midpoint = (%d, %d)\n",rb->midpoint.x,
	                                                  rb->midpoint.y);
#endif

/* step 2:  find the closer point to the pole-centered origin */
  if ((x1 * x1 + y1 * y1) < (x3 * x3 + y3 * y3)) {
    closex = x1;
    closey = y1;
    farx = x3;
    fary = y3;
    rb->closeidx = 0;
    rb->faridx = 2;
  }
  else {
    closex = x3;
    closey = y3;
    farx = x1;
    fary = y1;
    rb->closeidx = 2;
    rb->faridx = 0;
  }

/* step 3:  handle bogon cases */

  /* case a:  center of box is on a coordinate axis */
  if ((midx == 0) || (midy == 0)) {
    /* point 1 is (farx,closey) */
    rb->points[1].x = farx + xc + 0.5;
    rb->points[1].y = display->height - (closey + yc + 0.5);

    /* point 3 is (closex,fary) */
    rb->points[3].x = closex + xc + 0.5;
    rb->points[3].y = display->height - (fary + yc + 0.5);

#ifdef RB_DEBUG
    printf("bogon case\n");
    {
      int i;

      for (i = 0; i < NUM_ENCLOSURE_POINTS; i++)
	printf("point %d = (%d, %d)\n",i,rb->points[i].x,rb->points[i].y);
      printf("\n");
    }
#endif

    return;
  }

/* step 4:  calculate slope of line from midpoint to origin */
  oslope = midy / midx;

/* step 5:  calculate slope and y-intercept of line normal to the origin line
 *	    passing through the closer point */
  nslope = -1 / oslope;
  nint = closey - nslope * closex;

/* step 6:  calculate intersection of origin and normal lines - the formulae
 *	    are y = (m2 * b1 - m1 * b2) / (m2 - m1) and x = (y - b2) / m2.
 *	    note, however, that the y-intercept of the origin line is 0 by
 *	    definition, so if we pick the origin line as line 2, we can
 *	    simplify quite a bit */
  inty = (oslope * nint) / (oslope - nslope);
  intx = inty / oslope;

/* step 7:  calculate second point of rectangle (along normal line twice the
 *	    distance from close point to intersection) */
  x2 = closex + 2 * (intx - closex);
  y2 = nslope * x2 + nint;

  /* convert to X coordinates */
  rb->points[1].x = x2 + xc + 0.5;
  rb->points[1].y = display->height - (y2 + yc + 0.5);

/* step 8:  the fourth point is on a line parallel to the origin line from
 *	    the close point.  calculate the fourth point by adding the x
 *	    and y differences between the far point and the third point to
 *	    the close point */
  x4 = closex + (farx - x2);
  y4 = closey + (fary - y2);
    
  /* convert to X coordinates */
  rb->points[3].x = x4 + xc + 0.5;
  rb->points[3].y = display->height - (y4 + yc + 0.5);

#ifdef RB_DEBUG
  printf("close point = (%lf, %lf)\n",closex,closey);
  printf("midpoint = (%lf, %lf)\n",midx,midy);
  printf("slope of origin line = %lf\n",oslope);
  printf("slope, intercept of normal line = %lf, %lf\n",nslope,nint);
  printf("intersection = (%lf, %lf)\n",intx,inty);

  printf("first point = (%lf, %lf)\n",closex,closey);
  printf("second point = (%lf, %lf)\n",x2,y2);
  printf("third point = (%lf, %lf)\n",farx,fary);
  printf("fourth point = (%lf, %lf)\n",x4,y4);

  {
    int i;

    for (i = 0; i < NUM_ENCLOSURE_POINTS; i++)
      printf("point %d = (%d, %d)\n",i,rb->points[i].x,rb->points[i].y);
    printf("\n\n");
  }

#endif

  return;
}



/* draw a rubberband box */

static void draw_rubberband_box(w,rb)

Widget w;
struct rubberband *rb;

{
/* draw the box from its list of points */
  draw_enclosure(w,rb->points,NUM_ENCLOSURE_POINTS);

  return;
}



/* remove the label clicked on */

static void remove_label(x,y)

int x, y;

{
  struct id_node *id;

/* if the user can't see the IDs, don't let him remove them */
  if (! draw_ids) {
    XBell(XtDisplay(sky_widget),0);
    return;
  }

/* find the ID enclosing (x,y) */
  id = find_id_by_position(x,y);

/* sound a bell if we didn't find an enclosing ID */
  if (id == (struct id_node *)NULL)
    XBell(XtDisplay(sky_widget),0);
  else {
    /* erase this ID and remove it from the ID list, freeing the ID node */
    remove_this_id(sky_widget,id);
    free((void *)id);
  }

  return;
}



/* return the extent values of given text in a given font */

void get_text_extents(text,font,width,ascent,descent)

char *text;
IDfont font;
short *width;
short *ascent, *descent;

{
  int dummy;
  XCharStruct overall_size;

/* calculate the size of the identifying text */
  if (font == LATIN_ID)
    XTextExtents(latinidfontstr,text,strlen(text),&dummy,&dummy,&dummy,
	                                                       &overall_size);
  else if (font == LATIN_CON)
    XTextExtents(latinconfontstr,text,strlen(text),&dummy,&dummy,&dummy,
	                                                       &overall_size);
  else if (font == GREEK)
    XTextExtents(greekidfontstr,text,strlen(text),&dummy,&dummy,&dummy,
	                                                       &overall_size);
  else
    /* default to Latin ID font */
    XTextExtents(latinidfontstr,text,strlen(text),&dummy,&dummy,&dummy,
	                                                       &overall_size);

/* give the relevant values to the caller */
  *width = overall_size.width;
  *ascent = overall_size.ascent;
  *descent = overall_size.descent;

  return;
}



/* recenter the display at the point of the button click */

static void handle_recenter_event(event)

XButtonPressedEvent *event;

{
  int x, y;

/* get the coordinates of the event */
  x = event->x;
  y = event->y;

/* see if we're configured for the Southern Hemisphere */
  if (display.downunder_flag) {
    /* yes; hammer the X coordinates for inverted, mirror-image display */
    x = display.width - x;
    y = display.height - y;
  }

/* save the current display state */
  clone_state(&display);

/* flag that the center position is changing */
  display.old_state->changeflags = CENTERCHANGE;

/* update the display center RA and declination */
  X_to_pos_rad(&display,x,y,&display.center_ra_rad,&display.center_dec_rad);
  update_center_text();

/* update the sky display */
  update_sky(&display,True);

  return;
}



/* recalculate and set the center position text field */

static void update_center_text()

{
/* update the display center */
  update_center_position(&display,pos_buffer);

/* save the new position buffer as the old position buffer */
  strcpy(old_pos_buffer,pos_buffer);

/* set the display center position into the center text widget */
  XtVaSetValues(centertext,XtNstring,(String)pos_buffer,(String *)NULL);

  return;
}



/* handle changing the display to a new scale */

static void update_scale()

{
/* compute the new display width and height in radians */
  display.width_rad = (display.width * display.scale) / DEGREES_PER_RAD;
  display.height_rad = (display.height * display.scale) / DEGREES_PER_RAD;
#ifdef ZOOM_DEBUG
  printf("new display dimensions = %lf, %lf radians\n",display.width_rad,
	                                                 display.height_rad);
#endif

/* update the sky display */
  update_sky(&display,True);

/* and update the zoom slider thumb */
  update_zoom_thumb((double)display.scale);

  return;
}



/* update the zoom slider thumb */

static void update_zoom_thumb(scale)

double scale;

{
  Dimension slidersize;
  Dimension minthumb;
  float floatpos;

/* if we're not using a zoom slider, this routine is a no-op */
  if (! use_slider)
    return;

/* ensure that we don't try to drive the thumb off the top of the slider */
  if ((1.0 / scale) > slidermax)
    scale = 1.0 / slidermax;

#ifdef ZOOM_DEBUG
  printf("updating thumb position to scale %lf pixels per degree\n",1 / scale);
#endif

/* get the height of the slider scrollbar and the minimum thumb size */
  XtVaGetValues(zoomslider,XtNwidth,&slidersize,
		                    XtNminimumThumb,&minthumb,(String *)NULL);

/* given a scale in degrees per pixel, reset the zoom slider thumb */

  /*
   * details of the following calculation:  scale is in degrees per pixel;
   * we invert to get pixels per degree.  as discussed in inc_zoom(), the
   * slider represents the logarithm of the display scale, so we compute
   * the slider position as a fraction using the logarithms of the scale
   * and slider maximum value.  we then subtract that from one to compen-
   * sate for the fact that X measures vertical scrollbars from the top,
   * while we want to measure from the bottom.
   */
  floatpos = 1.0 - (log(1.0 / scale) / log((double)slidermax));
#ifdef ZOOM_DEBUG
  printf("updating thumb position to %f\n",floatpos);
#endif

/* move the thumb to the new position (length of thumb unchanged) */
  XawScrollbarSetThumb(zoomslider,floatpos,-1.0);

  return;
}



/* update the position and length of the scrollbar thumbs */

static void update_scrollbar_thumbs()

{
  Dimension scrollsize;
  int position;
  float pos;
  float length;
  int sign;

/* this function is a no-op if scrollbars are turned off */
  if (! use_scrollbars)
    return;

/* get the length of the RA scrollbar */
  XtVaGetValues(horiz_scroll,XtNwidth,&scrollsize,(String *)NULL);

/* map the current RA onto the scrollbar */

  /*
   * this is a lot less hairy than it looks - the mapping is a linear one,
   * so we can use the usual y = mx + b.  we know that we want 2 * PI units
   * to map onto the scrollbar size, so the slope of this line (m) is just
   * (- (scrollsize / (2 * PI))) (the slope is negative because RA increases
   * to the left; if we're in downunder mode, the slope becomes positive since
   * RA increases to the right).  b = y - mx, and we know that when x is the
   * initial RA, we want y to be (scrollsize / 2).  substituting that x and y
   * into the formula for b yields the second term of the formula below after
   * some simplification.
   */
  if (display.downunder_flag)
    sign = 1;
  else
    sign = -1;
  position = (sign * (scrollsize / (2 * PI))) * display.center_ra_rad +
                  (scrollsize / 2) * (1 - sign * display.initial_ra_rad / PI);

/* remap to within scrollbar range */
  if (position < 0)
    position += scrollsize;
  else if (position > scrollsize)
    position -= scrollsize;

/* convert position to a percentage */
  pos = (float)position / scrollsize;

/* calculate the length of the thumb */
  length = (display.width * display.scale) / 360;

/* set the new position and length */
  XawScrollbarSetThumb(horiz_scroll,pos,length);

/* get the length of the dec scrollbar */
  XtVaGetValues(vert_scroll,XtNheight,&scrollsize,(String *)NULL);

/* map the current declination onto the scrollbar */

  /*
   * once again, we need to do a linear map onto the scrollbar length.  this
   * time, we want PI units to map onto the scrollbar size, so the slope of
   * this line is just (scrollsize / PI) (the slope is negative in the nor-
   * mal orientation because we want +90 to map to scrollbar position 0 and
   * -90 to map to the maximum scrollbar position; in downunder mode, we want
   * the same slope but in the opposite direction, so the slope is positive).
   * again, b = y - mx, and we know that when x is 0 (on the celestial equa-
   * tor), we want y to be (scrollsize / 2).  therefore, b = (scrollsize / 2).
   * Note that, unlike the RA case, an out-of-range value cannot occur here.
   */
  position = (sign * (scrollsize / PI)) * display.center_dec_rad +
                                                             (scrollsize / 2);

/* calculate the new thumb percentage */
  pos = ((float)position / scrollsize);

/* calculate the length of the thumb - an argument can be made here that the
 * 360 below should really be 180, since the scrollbar represents an angle of
 * PI.  however, at minimum magnification, the vertical extent of the display
 * is still 360 degrees. */
  length = (display.height * display.scale) / 360;

/* set the new position and length */
  XawScrollbarSetThumb(vert_scroll,pos,length);

  return;
}



/* work procedure to handle -L (make big) option */

/* ARGSUSED */

static Boolean make_big_initial(client_data)

XtPointer client_data;

{
  Display *display;
  int screen;
  int n;
  Arg widgetargs[5];

/* change the toplevel widget's location and size to fill the display */
  display = XtDisplay(toplevel);
  screen = DefaultScreen(display);

  n = 0;
  XtSetArg(widgetargs[n],XtNx,0);                                        n++;
  XtSetArg(widgetargs[n],XtNy,0);                                        n++;
  XtSetArg(widgetargs[n],XtNwidth,DisplayWidth(display,screen));         n++;
  XtSetArg(widgetargs[n],XtNheight,
	               DisplayHeight(display,screen) - WM_TITLEHEIGHT);  n++;
  XtSetValues(toplevel,widgetargs,n);

  return(True);
}



/* pop down the catalog selection menu */

/* ARGSUSED */

static XtActionProc popdown_cat_menu(w,event,params,num_params)

Widget w;
XEvent *event;
String *params;
Cardinal *num_params;

{
  boolean change_flag;
  struct cat_header *catalog;
  int n;
  Arg buttonargs[5];
  Boolean buttonstate;

/* pop down the object selection menu */
  XtPopdown(catmenushell);

/* see if anything actually changed in the object display */
  change_flag = FALSE;
  catalog = cat_list_head;
  while (catalog != (struct cat_header *)NULL) {
    /* get the current state of the toggle button */
    n = 0;
    XtSetArg(buttonargs[n],XtNstate,&buttonstate);                        n++;
    XtGetValues(catalog->catselect,buttonargs,n);

    /* make a note of any change */
    if (buttonstate != catalog->show_flag) {
      change_flag = TRUE;
      catalog->show_flag = buttonstate;
    }

    /* step to the next catalog */
    catalog = catalog->next_cat;
  }

/* if anything changed, recalculate display lists and redraw the sky */
  if (change_flag) {
    set_wait_cursor();
    calculate_field(&display);
    clear_sky(&display);
  }

  return;
}



/* pop up the catalog menu display */

static void popup_catmenu(shell,client_data,call_data)

Widget shell;
XtPointer client_data;
XtPointer call_data;

{
/* first, pop up the find shell */
  popup_shell(shell,client_data,call_data);

/* in night mode, change to friendlier cursor colors */
  if (night_mode_flag)
    XDefineCursor(XtDisplay(catmenushell),XtWindow(catmenushell),
		                                          night_shell_cursor);

  return;
}



/* pop up the find display */

static void popup_find(shell,client_data,call_data)

Widget shell;
XtPointer client_data;
XtPointer call_data;

{
/* first, pop up the find shell */
  popup_shell(shell,client_data,call_data);

/* in night mode, change to friendlier cursor colors */
  if (night_mode_flag) {
    XDefineCursor(XtDisplay(findshell),XtWindow(findshell),night_shell_cursor);
    XDefineCursor(XtDisplay(findtext),XtWindow(findtext),night_text_cursor);
  }

  return;
}



/* pop up the user label display */

static void popup_userlabel(shell,client_data,call_data)

Widget shell;
XtPointer client_data;
XtPointer call_data;

{
/* first, pop up the userlabel shell */
  popup_shell(shell,client_data,call_data);

/* in night mode, change to friendlier cursor colors */
  if (night_mode_flag) {
    XDefineCursor(XtDisplay(userlabelshell),XtWindow(userlabelshell),
		                                          night_shell_cursor);
    XDefineCursor(XtDisplay(userlabeltext),XtWindow(userlabeltext),
		                                           night_text_cursor);
  }

  return;
}



/* pop up the chart display */

static void popup_chart(shell,client_data,call_data)

Widget shell;
XtPointer client_data;
XtPointer call_data;

{
/* first, pop up the userlabel shell */
  popup_shell(shell,client_data,call_data);

/* in night mode, change to friendlier cursor colors */
  if (night_mode_flag) {
    Widget textwidget;

    /* get the dialog text widget */
    textwidget = XtNameToWidget(chartdialog,"value");  

    XDefineCursor(XtDisplay(chartdialogshell),XtWindow(chartdialogshell),
		                                          night_shell_cursor);
    XDefineCursor(XtDisplay(textwidget),XtWindow(textwidget),
		                                           night_text_cursor);
  }

  return;
}



/* pop up the given display */

/* ARGSUSED */

static void popup_shell(button_widget,client_data,call_data)

Widget button_widget;
XtPointer client_data;
XtPointer call_data;

{
  Position x, y;
  Dimension width;
  Widget shell_widget = *(Widget *)client_data;

/* get the position of the button widget */
  XtVaGetValues(button_widget,XtNx,&x,XtNy,&y,(String *)NULL);

/* convert to root coordinates */
  XtTranslateCoords(toplevel,x,y,&x,&y);

/* get the width of the button widget */
  XtVaGetValues(button_widget,XtNwidth,&width,(String *)NULL);

/* place the popup just off the end of the button */
  XtVaSetValues((Widget)shell_widget,XtNx,x + width,XtNy,y,(String *)NULL);

/* pop up */
  XtPopup((Widget)shell_widget,XtGrabExclusive);

  return;
}



/* user label acknowlege/cancel button callback function */

/* ARGSUSED */

static void PositionUserLabel(w,client_data,call_data)

Widget w;
XtPointer client_data;
XtPointer call_data;

{
/* pop down the shell */
  XtPopdown(userlabelshell);

/* resensitize the menu info button */
  XtSetSensitive(menuuserlabel,True);

/* if the client data is -1, ignore the edited string and take no action */
  if (client_data == (XtPointer)-1)
    return;

/* flag that we're handling a user-supplied label */
  user_label_flag = TRUE;

  return;
}



/* position the user label in response to a <return> in the text field */

/* ARGSUSED */

static XtActionProc position_userlabel(w,event,params,num_params)

Widget w;
XEvent *event;
String *params;
Cardinal *num_params;

{
/* just pretend the button was clicked */
  PositionUserLabel((Widget)NULL,(XtPointer)0,(XtPointer)NULL);

  return;
}



/* user label window popdown (implied "cancel" button) */

/* ARGSUSED */

static void cancel_userlabel(w,client_data,event,cont_flag)

Widget w;
XtPointer client_data;
XEvent *event;
Boolean *cont_flag;

{
/* simply call the cancel button's callback */
  PositionUserLabel(w,(XtPointer)-1,(XtPointer)NULL);

  return;
}



/* pop down the user label display (normal popdown) */

/* ARGSUSED */

static XtActionProc popdown_userlabel_display(w,event,params,num_params)

Widget w;
XEvent *event;
String *params;
Cardinal *num_params;

{
/* pop down the user label shell */
  XtPopdown(userlabelshell);

  return;
}



/* emit an identification for the selected constellation */

static void add_constellation_label(x,y)

int x, y;

{
  char *constellation_name;
  struct id_node *id;

/* if ID texts are not visible, don't allow the user to create any */
  if (! draw_ids) {
    XBell(XtDisplay(sky_widget),0);
    return;
  }

/* get the constellation name */
  if ((constellation_name = identify_constellation(&display,x,y)) ==
                                                               (char *)NULL) {
    XBell(XtDisplay(sky_widget),0);
    return;
  }

/* get an ID node for this label, pretending it's a user label */
  id = get_userlabel_id(x,y,constellation_name,LATIN_CON);

/* allow the user to position the label */
  position_label(id);

  return;
}



/* emit an identification for the selected object */

static void add_object_label(x,y)

int x, y;

{
  struct id_node *id;

/* if ID texts are not visible, don't allow the user to create any */
  if (! draw_ids) {
    XBell(XtDisplay(sky_widget),0);
    return;
  }

/* get the catalog ID of the object */
  if ((id = get_object_id(x,y)) == (struct id_node *)NULL) {
    XBell(XtDisplay(sky_widget),0);
    return;
  }

/* allow the user to position the label */
  position_label(id);

  return;
}



/* emit a user label */

static void add_user_label(x,y)

int x, y;

{
  struct id_node *id;

/* if ID texts are not visible, don't allow the user to create any */
  if (! draw_ids) {
    XBell(XtDisplay(sky_widget),0);
    return;
  }

/* get an ID node for the user's label */
    if (strlen(userlabel_buffer) > 0)
      id = get_userlabel_id(x,y,userlabel_buffer,LATIN_ID);
    else {
      XBell(XtDisplay(sky_widget),0);
      return;
    }

/* allow the user to position the label */
  position_label(id);

  return;
}



/* move the label clicked on */

static void move_label(x,y)

int x, y;

{
  struct id_node *id;

/* if the user can't see the IDs, don't let him move them */
  if (! draw_ids) {
    XBell(XtDisplay(sky_widget),0);
    return;
  }

/* find the ID enclosing (x,y) */
  id = find_id_by_position(x,y);

/* sound a bell if we didn't find an enclosing ID */
  if (id == (struct id_node *)NULL)
    XBell(XtDisplay(sky_widget),0);
  else {
    /* flag that we're doing a label move */
    display.move_id_flag = TRUE;

    /* prepare this ID for the move */
    setup_label_move(sky_widget,id);

    /* remember the offset of the pointer within the label */
    display.ptr_x_offset = x - id->ul_x;
    display.ptr_y_offset = y - id->ul_y;

    /* warp the pointer to the exact label position */
    XWarpPointer(XtDisplay(sky_widget),XtWindow(sky_widget),
		              XtWindow(sky_widget),0,0,0,0,id->ul_x,id->ul_y);

/* allow the user to reposition the label */
    position_label(id);
  }

  return;
}



/* set up to allow positioning of the given label */

static void position_label(id)

struct id_node *id;

{
/* set a flag indicating that we're placing an ID label */
  display.place_id_flag = TRUE;

/* register for pointer motion events under button 2 */
  XtAddEventHandler(sky_widget,Button2MotionMask,False,
		                        handle_button2_motion,(XtPointer)id);

/* register for button release events */
  XtAddEventHandler(sky_widget,ButtonReleaseMask,False,
		                        HandleSkyButtonRelease,(XtPointer)id);

/* set the text box positioning cursor */
  set_textbox_cursor();

/* draw the initial box */
  draw_box(sky_widget,id->ul_x,id->ul_y,id->lr_x,id->lr_y);

  return;
}



/* perform a display zoom */

/* ARGSUSED */

static void display_zoom(w,client_data,call_data)

Widget w;
XtPointer client_data;
XtPointer call_data;

{
#ifdef ZOOM_DEBUG
  printf("current display scale = %f pixels / degree\n",1.0 / display.scale);
#endif

/* remember the current display state */
  clone_state(&display);

/* flag that the display scale is changing */
  display.old_state->changeflags = SCALECHANGE;

/* calculate the new scale of the display, changing by 20% up or down */
  display.scale -= (int)client_data * (display.scale * 0.2);
#ifdef ZOOM_DEBUG
  printf("new display scale = %f pixels / degree\n",1.0 / display.scale);
#endif

/* go handle the change to the new scale */
  update_scale();

  return;
}



/* handle an incremental scroll on the zoom slider */

/* ARGSUSED */

static void inc_zoom(w,client_data,position)

Widget w;
XtPointer client_data;
XtPointer position;

{
  Dimension slidersize;
  Dimension minthumb;
  int pos;
  float floatpos;

/* get the height of the zoom slider and the minimum thumb size */
  XtVaGetValues(w,XtNheight,&slidersize,
		                    XtNminimumThumb,&minthumb,(String *)NULL);

/* extract position as a pixel count */
  pos = abs((int)position);

/* compute the new position as a fraction of the slider size */
  floatpos = (float)pos / slidersize;

/* calculate the new scale of the display */

  /*
   * here's the scoop:  we want equal movements of the slider, at various
   * positions along its length, to provide equal magnification *factors*
   * of the display.  this means that the size of a movement which takes
   * the display from a scale of 200 pixels per degree to 400 pixels per
   * degree (and thus provides magnification factor two) should be the
   * same size as a movement which takes the display from a scale of 1000
   * pixels per degree to 2000 pixels per degree (in particular, *not*
   * from 1000 pixels per degree to 1200 pixels per degree).  as Napier
   * so aptly demonstrated with his invention of the slide rule (you re-
   * member slide rules, don't you?), one does this with logarithms.
   *
   * so what we do is interpret the new slider position as a point along
   * a scale which runs from 0 to the log of the slider's maximum value.
   * (since the minimum magnification is 1 pixel per degree and the log
   * of 1 is 0, this establishes 0 as the minimum value of the slider.)
   * we then take the antilogarithm of that as the new display scale.
   *
   * the only tricky part of this is that X considers the *top* of a ver-
   * tical scrollbar to be zero, rather than the bottom, so we want to
   * invert that.
   */

/* remember the current display state */
  clone_state(&display);

/* flag that the display scale is changing */
  display.old_state->changeflags = SCALECHANGE;

/* calculate the new display scale as the antilog of the scaled position,
 * inverting pixels per degree into degrees per pixel */
#ifdef ZOOM_DEBUG
  printf("new slider position = %f\n",floatpos);
  printf("slider max = %d\n",slidermax);
#endif
  display.scale = 1 / exp((1.0 - floatpos) * log((double)slidermax));
#ifdef ZOOM_DEBUG
  printf("new display scale = %f pixels per degree\n",1 / display.scale);
#endif

/* go handle the change to the new scale */
  update_scale();

  return;
}



/* handle an incremental scroll on the display scrollbars */

/* ARGSUSED */

static void inc_scroll(w,client_data,position)

Widget w;
XtPointer client_data;
XtPointer position;

{
  Dimension width, height;
  int scrollsize;
  int pos;
  double distance_rad;

/* get the width and height of the affected scrollbar */
  XtVaGetValues(w,XtNwidth,&width,XtNheight,&height,(String *)NULL);

/* get the length of the scrollbar */
  if (w == vert_scroll)
    scrollsize = height;
  else if (w == horiz_scroll)
    scrollsize = width;
  else
    return;

/* remember the current display state */
  clone_state(&display);

/* flag that the display center is changing */
  display.old_state->changeflags = CENTERCHANGE;

/* extract the thumb position as a pixel count */
  pos = abs((int)position);

/* calculate the new center point of the display */

  /*
   * here's the scoop:  horizontal scrolling moves in RA only, and vertical
   * scrolling moves in dec only.  while the RA centerpoint may be modified,
   * by definition, the vertical scrollbar centerpoint is always 0d 0'; this
   * is to avoid ambiguity.  this particular restriction also means that the
   * vertical scrollbar only spans 180 degrees while the horizontal scrollbar
   * spans 360 degrees.  we calculate the distance from the center point,
   * convert that to radians, and then subtract from (or add to, if we're in
   * downunder mode) the current position.  we then normalize that result to
   * the range 0 <= x < 2 * pi for RA, but (for no particular reason) we han-
   * dle the normalization of the declination in the rad_to_dec conversion
   * routine.
   */

  /* modify the RA or dec, as appropriate */
  if (w == vert_scroll) {
    /* update the center point */
    distance_rad = ((double)(pos - (scrollsize / 2)) / scrollsize) * PI;
    /* if we're in downunder mode, south is at the top of the display */
    if (display.downunder_flag)
      display.center_dec_rad = display.initial_dec_rad + distance_rad;
    else
      display.center_dec_rad = display.initial_dec_rad - distance_rad;
  }
  else if (w == horiz_scroll) {
    /* update the center point */
    distance_rad = ((double)(pos - (scrollsize / 2)) / scrollsize) * 2 * PI;
    if (display.downunder_flag)
      display.center_ra_rad = display.initial_ra_rad + distance_rad;
    else
      display.center_ra_rad = display.initial_ra_rad - distance_rad;
    if (display.center_ra_rad < 0)
      display.center_ra_rad += 2 * PI;
    else if (display.center_ra_rad >= (2 * PI))
      display.center_ra_rad -= 2 * PI;
    else
      ;
  }

/* update the display position */
  update_center_text();

/* update the sky display */
  update_sky(&display,True);

  return;
}



/* change the center position of the display */

void update_display_pos(ra_rad,dec_rad)

double ra_rad, dec_rad;

{
  display.center_ra_rad = ra_rad;
  display.center_dec_rad = dec_rad;

  return;
}



/* make a copy of the given display state for possible undo */

static void clone_state(display)

struct display *display;

{
  struct state *oldstate;

/* get a state structure */
  oldstate = (struct state *)malloc(sizeof(struct state));

/* zero the change flags */
  oldstate->changeflags = 0;

/* save the magnitude limit and display scale */
  oldstate->mag_limit = mag_limit;
  oldstate->scale = display->scale;

/* save the current display center position */
  oldstate->center_ra_rad = display->center_ra_rad;
  oldstate->center_dec_rad = display->center_dec_rad;

/* save the current shell size and position */
  oldstate->shell_width = display->form_width;
  oldstate->shell_height = display->form_height;

/* finally, link it into the list */
  oldstate->next = display->old_state;
  display->old_state = oldstate;

  return;
}



/* save the current application state */

#define XSKY_SAVE_FILE   ".xskyinit"
#define XSKY_CMD_FILE    ".xskycmd"

static char geom_buff[4 + 1 + 4 + 1];
static char scale_buff[14 + 1];
static char pflag_buff[3 + 1];
static char sflag_buff[3 + 1];
static char zval_buff[11 + 1];
static char mag_buff[14 + 1];

#define MAXCOLORLEN    30

static char colornames[NUMSTARCOLORS][MAXCOLORLEN + 1];

static char cat_buff[INITIAL_CATLIST_LEN + 1];

#define NUM_CMD_ARGS   34

static char *cmdargs[NUM_CMD_ARGS + 1] = {
  "xsky",
  "-I", (char *)NULL,
  "-geometry", geom_buff,
  "-s", scale_buff,
  "-P", pflag_buff,
  "-S", sflag_buff,
  "-Z", zval_buff,
  "-m", mag_buff,
  "-O", colornames[O],
  "-B", colornames[B],
  "-A", colornames[A],
  "-F", colornames[F],
  "-G", colornames[G],
  "-K", colornames[K],
  "-M", colornames[M],
  "-c", cat_buff,
  (char *)NULL,                /* possible -b option */
  (char *)NULL,                /* possible -n option */
  (char *)NULL,                /* possible -u option */
  (char *)NULL
  };

#define I_OPTION_VAL      2
#define GEOM_OPTION_VAL   4
#define LS_OPTION_VAL     6
#define P_OPTION_VAL      8
#define US_OPTION_VAL    10
#define Z_OPTION_VAL     12
#define M_OPTION_VAL     14
#define COLOR_OPTION_VAL 16
#define C_OPTION_VAL     30
#define OTHER_OPTIONS    31


static void save_state()

{
/*
 * gosh, there's just tons and tons of state that needs to be saved:
 *
 *  command line:
 *     display size (X -geometry option) *
 *     display scale (-s option) *
 *     scrollbars active (-S option) *
 *     zoom slider active (-Z option) *
 *     pointer position display active (-P option) *
 *     black-and-white mode (-b option) *
 *     night mode (-n option) *
 *     downunder flag (-u option)
 *     limiting magnitude (-m option)
 *     colors of all spectral classes (-OBAFGKM options)
 *     active catalogs (-c option)
 *
 *  initialization file:
 *     initial center position (don't use -p option)
 *     current center position (no option)
 *     RA/dec grid on/off (no option)
 *     boundaries on/off (no option)
 *     labels on/off (no option)
 *     object label font (-f option - but read from file instead)
 *     Greek letter font (-g option - but read from file instead)
 *     object labels
 *     user labels
 *
 *     * = must be on command line
 *
 *     note that we don't care about the initial -L option; we specify the
 *     current size with -geometry
 */
  char *filebuff;
  FILE *state_fd;
  Dimension shell_width, shell_height;
  struct cat_header *catalog;
  int option_idx;
  int num_cmd_args;
  struct id_node *id;
  char idbuff[MAX_ID_STRING + 1];
  char *buffp;
  FILE *cmd_fd;
  int i;

/* open the state file */
  filebuff = build_filespec("HOME",XSKY_SAVE_FILE);
  if ((state_fd = fopen(filebuff,"w")) == (FILE *)NULL) {
    XBell(XtDisplay(sky_widget),0);
    return;
  }

/* specify the path of the initialization file */
  if (cmdargs[I_OPTION_VAL] != (char *)NULL)
    /* free the old file buffer to avoid a small memory leak */
    free((void *)cmdargs[I_OPTION_VAL]);
  cmdargs[I_OPTION_VAL] = filebuff;

/* get our current size and specify as X -geometry option */
  XtVaGetValues(toplevel,XtNwidth,&shell_width,XtNheight,&shell_height,
		                                              (String *)NULL);
  sprintf(geom_buff,"%dx%d",(int)shell_width,(int)shell_height);

/* specify the display scale as the -s option */
  sprintf(scale_buff,"%f",1 / display.scale);

/* specify the pointer position and scrollbar display flags */
  sprintf(pflag_buff,"%s",use_scrollbars ? FLAG_ON_VALUE : FLAG_OFF_VALUE);
  sprintf(sflag_buff,"%s",use_posdisplay ? FLAG_ON_VALUE : FLAG_OFF_VALUE);

/* specify the zoom slider maximum value */
  sprintf(zval_buff,"%d",slidermax);

/* specify the magnitude limit as the -m option */
  sprintf(mag_buff,"%f",mag_limit);

/* specify the star color options */
  sprintf(colornames[O],"%s",starcolors[O]);
  sprintf(colornames[B],"%s",starcolors[B]);
  sprintf(colornames[A],"%s",starcolors[A]);
  sprintf(colornames[F],"%s",starcolors[F]);
  sprintf(colornames[G],"%s",starcolors[G]);
  sprintf(colornames[K],"%s",starcolors[K]);
  sprintf(colornames[M],"%s",starcolors[M]);

/* specify the active catalog list */

  /* initialize the catalog list */
  strcpy(cat_buff,"");

  /* loop through all known catalogs */
  catalog = cat_list_head;
  while (catalog != (struct cat_header *)NULL) {
    /* if we're showing this catalog, put it in the list */
    if (catalog->show_flag)
      sprintf(&cat_buff[strlen(cat_buff)],"%s,",catalog->menu_name);

    /* step to the next catalog */
    catalog = catalog->next_cat;
  }

  /* remove the trailing comma, if any */
  if (strlen(cat_buff) > 0)
    cat_buff[strlen(cat_buff) - 1] = '\0';

/* specify the monochrome option */
  option_idx = OTHER_OPTIONS;
  if (mono_flag)
    cmdargs[option_idx++] = "-b";

/* specify the night-mode option */
  if (night_mode_flag)
    cmdargs[option_idx++] = "-n";

/* specify the downunder option */
  if (display.downunder_flag)
    cmdargs[option_idx++] = "-u";

/* set the number of command-line arguments */
  num_cmd_args = option_idx;

/* dump the initial center position */
  fprintf(state_fd,"initial %.12lf %.12lf\n",
	                      display.initial_ra_rad,display.initial_dec_rad);
  fprintf(state_fd,"current %.12lf %.12lf\n",
	                      display.center_ra_rad,display.center_dec_rad);

/* dump some non-option flags */
  fprintf(state_fd,"grid %s\n",draw_grid ? FLAG_ON_VALUE : FLAG_OFF_VALUE);
  fprintf(state_fd,"boundaries %s\n",draw_bound ?
	                                      FLAG_ON_VALUE : FLAG_OFF_VALUE);
  fprintf(state_fd,"labels %s\n",draw_ids ? FLAG_ON_VALUE : FLAG_OFF_VALUE);

/* dump the font names */
  fprintf(state_fd,"latin %s\n",latin_id_font);
  fprintf(state_fd,"greek %s\n",greek_font);

/* dump the ID label list */

  /* loop through the ID list, emitting catalog and user labels */
  id = id_list;
  while (id != (struct id_node *)NULL) {
    /* turn spaces in the ID string into '$' */
    strcpy(idbuff,id->id_buffer);
    buffp = idbuff;
    while (*buffp != '\0') {
      if (*buffp == ' ')
	*buffp = '$';
      buffp++;
    }
    
    /* all the news that's fit to print about an ID */
    fprintf(state_fd,
	    "%s %d %.12lf %.12lf %d %d %d %d %d %d %d %d %s %s %s %s\n",
	    id->obj_catalog == (struct cat_header *)NULL ?
	                                "User" : (id->obj_catalog)->menu_name,
	    id->obj_index,
	    id->ra_rad,id->dec_rad,
	    id->obj_x,id->obj_y,
	    id->x,id->y,
	    id->ul_x,id->ul_y,
	    id->lr_x,id->lr_y,
	    id->font == GREEK ? GREEK_ID_FONT :
	                          (LATIN_ID ? LATIN_ID_FONT : LATIN_CON_FONT),
	    id->show_flag ? "True" : "False",
	    id->redraw_flag ? "True" : "False",
	    idbuff);

    /* step to the next ID node */
    id = id->next;
  }

/* be neat and close the file */
  if (fclose(state_fd) == EOF) {
    XBell(XtDisplay(sky_widget),0);
    return;
  }

/* set the command that will restart us in the saved state */
  XSetCommand(XtDisplay(toplevel),XtWindow(toplevel),cmdargs,num_cmd_args);

/* save the command in an executable file for those without session managers */

  /* open the command file */
  filebuff = build_filespec("HOME",XSKY_CMD_FILE);
  if ((cmd_fd = fopen(filebuff,"w")) == (FILE *)NULL) {
    XBell(XtDisplay(sky_widget),0);
    return;
  }

  /* dump the command line */
  for (i = 0; i < num_cmd_args; i++)
    fprintf(cmd_fd,"%s ",cmdargs[i]);
  fprintf(cmd_fd,"\n");

  /* be neat and close the file */
  if (fclose(cmd_fd) == EOF) {
    XBell(XtDisplay(sky_widget),0);
    return;
  }

  /* set the protection so the command file is executable (ignore errors) */
  (void)chmod(filebuff,
	       (mode_t)(S_IRWXU | (S_IRGRP | S_IXGRP) | (S_IROTH | S_IXOTH)));

  /* free the filename buffer */
  free((void *)filebuff);

  return;
}



/* here on non-comprehension of initialization file */

void bad_init_file(s)

char *s;

{
  fprintf(stderr,"malformed initialization file - %s\n",s);
  exit(1);
}



/* display the command format */

static void printusage(prog)

char *prog;

{
  printf("usage:  %s [-hLbcmfgspSZOBAFGKM]\n",prog);
  exit(0);
}



#if 0

/* dump the display structure for debugging */

static void dump_display(display)

struct display *display;

{
  printf("\n\n");
  printf("display is (%d x %d) or (%lf x %lf)\n",
	                              display->width,display->height,
	                              display->width_rad,display->height_rad);
  printf("display scale = %f\n",display->scale);
  printf("downunder = %s, redraw = %s, rubberband = %s, undo_resize = %s\n",
	                        display->downunder_flag ? "true" : "false",
	                        display->redraw_flag ? "true" : "false",
	                        display->rubberband_flag ? "true" : "false",
	                        display->place_id_flag ? "true" : "false",
	                        display->move_id_flag ? "true" : "false",
	                        display->undo_resize_flag ? "true" : "false");
  printf("ra pos = %d %d %d %d, dec pos = %d %d %d %d %d\n",
	                                         display->ra_pos.hours,
	                                         display->ra_pos.minutes,
	                                         display->ra_pos.seconds,
	                                         display->ra_pos.thousandths,
	                                         display->dec_pos.sign,
	                                         display->dec_pos.degrees,
	                                         display->dec_pos.minutes,
	                                         display->dec_pos.seconds,
	                                         display->dec_pos.hundredths);
  printf("display center = (%lf, %lf)\n",
	                      display->center_ra_rad,display->center_dec_rad);
  printf("initial display center = (%lf, %lf)\n",
	                    display->initial_ra_rad,display->initial_dec_rad);
  printf("display limits = %lf, %lf, %lf, %lf\n",
	                        display->left_ra_rad,display->right_ra_rad,
	                        display->top_dec_rad,display->bottom_dec_rad);
  printf("form size = %d x %d\n",display->form_width,display->form_height);
  printf("pointer offset = (%d, %d)\n",
	                         display->ptr_x_offset,display->ptr_y_offset);

  return;
}

#endif
