/*
 * xsky - an interactive sky atlas
 *
 * Copyright 1992-6, Terry R. Friedrichsen
 *
 * This program may be copied and redistributed, in whole or in part,
 * as long as you don't try to make any money from the sale or redis-
 * tribution of the program or any part of the program, or pretend
 * that you wrote the program or any of its parts unless specifically
 * credited by the original author.
 *
 * You are free to make use of this software in your own programs, as
 * long as you credit the original author where it is due.
 */

/*
 * WARRANTY:
 * xsky was written as a learning project and as a demonstration of
 * X Window System programming.  xsky doesn't do anything; it is not
 * merchantable, and it is not fit for any purpose whatsoever.  In
 * fact, don't use xsky at all; it's free, and you're getting what
 * you paid for.
 */

#include <stdio.h>
#include <ctype.h>

#include <math.h>
#include <stdlib.h>

#include <unistd.h>

#include <string.h>

#include <X11/Intrinsic.h>

#include "sao.h"

#include "skydefs.h"

#include "pos.h"
#include "catalog.h"
#include "draw.h"
#include "catreader.h"

struct sao_node {
  double ra_rad;
  double dec_rad;
  int sao_number;
  float mag;
  long filepos;
  char spectrum;
};

/* static function prototypes */
static FILE *read_sao PROTOTYPE((void));
static void calculate_sao_field PROTOTYPE((struct display *,
					                struct cat_header *));
static boolean check_mag_limit PROTOTYPE((int,double));
static boolean find_sao_star PROTOTYPE((char *));
static void get_sao_pos PROTOTYPE((int,double *,double *));
static void draw_sao_star PROTOTYPE((Widget,int,int,int,int,int,int,int));
static boolean ps_draw_sao_star PROTOTYPE((struct display *,int,int,int));
static boolean display_sao PROTOTYPE((FILE *,int));
static void format_sao_name PROTOTYPE((int,char *,IDfont *));
static void fill_sao_entry PROTOTYPE((char *,int));
static int mag_compare PROTOTYPE((struct sao_node *,struct sao_node *));

/* external function prototypes */
extern char *build_filespec PROTOTYPE((char *,char *));
extern boolean read_binary_database PROTOTYPE((char *,void *,int *));
extern void init_precess PROTOTYPE((double,double));
extern void hs_heapsort PROTOTYPE((char [],size_t,size_t,int (*)()));
extern boolean write_binary_database PROTOTYPE((char *,void *,size_t,size_t));
extern void precess PROTOTYPE((double,double,double *,double *));
extern void add_to_display_list PROTOTYPE((struct display *,
					           struct cat_header *,
					                  int,double,double));
extern void draw_star PROTOTYPE((Widget,int,int,int,StarColor));
extern boolean ps_draw_star PROTOTYPE((struct display *,double,double,double));
extern void update_display_pos PROTOTYPE((double,double));
extern void format_sao_display PROTOTYPE((char *));

static struct sao_node sao_array[MAX_SAO];

static int sao_array_max;

extern float mag_limit;

#define SAO_DATA_FILE   "sao.dat"
#define SAO_BINARY_FILE "sao.bin"

/* SAO record buffer */
static char sao_rec[SAO_RECLEN + 1 + 1];



/* initialize the catalog data structure */

void init_sao(catalog)

struct cat_header *catalog;

{
  catalog->read_data = read_sao;
  catalog->calculate_field = calculate_sao_field;
  catalog->mag_limit_check = check_mag_limit;
  catalog->find_object = find_sao_star;
  catalog->get_object_pos = get_sao_pos;
  catalog->draw_object = draw_sao_star;
  catalog->ps_draw_object = ps_draw_sao_star;
  catalog->display_data = display_sao;
  catalog->format_name = format_sao_name;
  catalog->menu_name = SAO_MENU_NAME;
  catalog->catalog_name = SAO_CATALOG_NAME;

  return;
}



/* read in the SAO catalog file, extracting needed info */

static FILE *read_sao()

{
  char *filebuff;
  FILE *sao_fd;
  char *bin_filebuff;
  int sao_entry;

/* get the path to the SAO data */
  filebuff = build_filespec("SAO_PATH",SAO_DATA_FILE);

/* open the SAO data file */
  if ((sao_fd = fopen(filebuff,"r")) == (FILE *)NULL) {
    printf("sky:  can't open SAO data file %s\n",filebuff);
    perror("sky");
    exit(1);
  }

/* get the path to the SAO binary file */
  bin_filebuff = build_filespec("SAO_BIN_PATH",SAO_BINARY_FILE);

/* see if we can read the binary file */
  if (access(bin_filebuff,R_OK) == 0)
    /* we can; try to use the pre-built database */
    if (read_binary_database(bin_filebuff,sao_array,&sao_array_max)) {
      /* success; free the filename buffers */
      free((void *)filebuff);
      free((void *)bin_filebuff);

      /* return the fd of the open data file */
      return(sao_fd);
    }

/* prior to reading the SAO catalog, build the rotation matrix for precessing
 * the stellar coordinates to the common equinox from the given epoch */
  if (SAO_EPOCH != COMMON_EQUINOX)
    init_precess(SAO_EPOCH,COMMON_EQUINOX);

/* read the SAO catalog, saving the information necessary for the display */
  sao_entry = 0;
  while (TRUE) {
    sao_array[sao_entry].filepos = ftell(sao_fd);
    if (fgets(sao_rec,sizeof(sao_rec),sao_fd) == (char *)NULL)
      break;
    fill_sao_entry(sao_rec,sao_entry++);
  }

/* leave the file open for later use */
/*  (void)fclose(sao_fd);*/

/* remember the last sao entry we filled */
  sao_array_max = sao_entry - 1;

/* now sort the sao array in decreasing order of magnitude */
  hs_heapsort((void *)sao_array,(size_t)sao_entry,
	                         (size_t)sizeof(struct sao_node),mag_compare);

/* attempt to save the binary database */
  if (access(bin_filebuff,W_OK) == 0)
    /* it's writeable, so let's try */
    if (! write_binary_database(bin_filebuff,(void *)sao_array,
				            (size_t)sizeof(struct sao_node),
				                           (size_t)sao_entry))
      /* dump of binary database failed; be sure the file is gone */
      (void)unlink(bin_filebuff);

/* free the filename buffers */
  free((void *)filebuff);
  free((void *)bin_filebuff);

  return(sao_fd);
}



/* compare two array entries for magnitude - compare routine for heapsort */

static int mag_compare(sao_entry_1,sao_entry_2)

struct sao_node *sao_entry_1, *sao_entry_2;

{
/* major key is the magnitude */
  if (sao_entry_1->mag < sao_entry_2->mag)
    return(-1);
  else if (sao_entry_1->mag > sao_entry_2->mag)
    return(1);
  else
    return(0);

/* NOTREACHED */
}



/* return TRUE if the magnitude of the star meets the limiting magnitude */

static boolean check_mag_limit(array_pos,limit)

int array_pos;
float limit;

{
  return((boolean)(sao_array[array_pos].mag <= limit));
}



/* given an object's index, return its right ascension and declination */

static void get_sao_pos(array_pos,ra_rad,dec_rad)

int array_pos;
double *ra_rad, *dec_rad;

{
  *ra_rad = sao_array[array_pos].ra_rad;
  *dec_rad = sao_array[array_pos].dec_rad;

  return;
}



/* given an SAO record, make a data node for it */

static void fill_sao_entry(record,sao_entry)

char *record;
int sao_entry;

{
  int reclen;
  char field[11];
  struct ra_pos ra_pos;
  struct dec_pos dec_pos;
  double ra_pos_to_rad();
  double dec_pos_to_rad();

/* get the length of the record for GET_FIELD */
  reclen = strlen(record) - 1;

/* get the SAO number */
  GET_FIELD(field,record,0,6);
  sao_array[sao_entry].sao_number = atoi(field);

/* get the RA in pieces */
  GET_FIELD(field,record,POS_RA_START,2);
  ra_pos.hours = (smallint)atoi(field);

  GET_FIELD(field,record,POS_RA_START + 2,2);
  ra_pos.minutes = (smallint)atoi(field);

  GET_FIELD(field,record,POS_RA_START + 4,2);
  ra_pos.seconds = (smallint)atoi(field);

  GET_FIELD(field,record,POS_RA_START + 6,3);
  ra_pos.thousandths = (shortint)atoi(field);

/* similarly, get the declination in pieces */
  GET_FIELD(field,record,POS_DEC_START,1);
  dec_pos.sign = (short)1;
  if (field[0] == '-')
    dec_pos.sign = (short)-1;

  GET_FIELD(field,record,POS_DEC_START + 1,2);
  dec_pos.degrees = (smallint)atoi(field);

  GET_FIELD(field,record,POS_DEC_START + 3,2);
  dec_pos.minutes = (smallint)atoi(field);

  GET_FIELD(field,record,POS_DEC_START + 5,2);
  dec_pos.seconds = (smallint)atoi(field);

  GET_FIELD(field,record,POS_DEC_START + 7,2);
  dec_pos.hundredths = (smallint)atoi(field);

/* compute the RA and declination in radians */
  sao_array[sao_entry].ra_rad = ra_pos_to_rad(ra_pos);
  sao_array[sao_entry].dec_rad = dec_pos_to_rad(dec_pos);

/* precess them to the common equinox */
  if (SAO_EPOCH != COMMON_EQUINOX)
    precess(sao_array[sao_entry].ra_rad,sao_array[sao_entry].dec_rad,
	          &sao_array[sao_entry].ra_rad,&sao_array[sao_entry].dec_rad);

/* get the magnitude */
  GET_FIELD(field,record,MAG_VIS_START,3);
  sao_array[sao_entry].mag = (float)atoi(field) / 10.0;

/* get the spectral class */
  GET_FIELD(field,record,SPECTRUM_START,1);
  sao_array[sao_entry].spectrum = field[0];

  return;
}



/* construct a list of those stars which appear in the current display */

static void calculate_sao_field(display,catalog)

struct display *display;
struct cat_header *catalog;

{
  int i;

/* loop through all SAO entries, checking for visibility */
  for (i = 0; i <= sao_array_max; i++)
    /* check this star and add it to the list if visible */
    add_to_display_list(display,catalog,i,sao_array[i].ra_rad,
                                                       sao_array[i].dec_rad);

#ifdef DEBUG
  printf("calculate_sao_field returning\n");
#endif
  return;
}



/* draw an SAO star on the sky */

static void draw_sao_star(sky_widget,array_pos,obj_x,obj_y,ul_x,ul_y,
		                                                 width,height)

Widget sky_widget;
int array_pos;
int obj_x, obj_y;
int ul_x, ul_y;
int width, height;

{
  int size;
  StarColor color;

/* be sure it is bright enough to display */
  if (! check_mag_limit(array_pos,mag_limit))
    return;

  /* calculate size based on magnitude */
  size = STARSIZE(sao_array[array_pos].mag);

  /* draw the object even if it is only partially in the box */
  if (((obj_x + (size + 1) / 2) >= ul_x) &&
                               ((obj_x - (size + 1) / 2) <= (ul_x + width)) &&
                               ((obj_y + (size + 1) / 2) >= ul_y) &&
                               ((obj_y - (size + 1) / 2) <= (ul_y + height)))
    ;
  else
    return;

  /* assign a color based on the spectral class */
  switch (sao_array[array_pos].spectrum) {
  case 'W':
  case 'O':
    color = O;
    break;
  case 'B':
    color = B;
    break;
  case 'A':
    color = A;
    break;
  case 'F':
    color = F;
    break;
  case 'G':
    color = G;
    break;
  case 'K':
    color = K;
    break;
  case 'M':
  case 'C':
  case 'R':
  case 'N':
  case 'S':
    color = M;
    break;
  default:
    printf("star %d has unrecognized spectral class %c\n",
	                                      sao_array[array_pos].sao_number,
	                                      sao_array[array_pos].spectrum);
  case 'P':        /* some stars are peculiar */
  case '+':        /* some have composite spectra */
  case ' ':        /* and some are just not given */
    color = A;     /* make unknowns white */
    break;
  }

/* draw the star with correct location, size, and color */
  draw_star(sky_widget,obj_x,obj_y,size,color);

  return;
}



/* draw one SAO star on the PostScript display */

static boolean ps_draw_sao_star(display,array_pos,x,y)

struct display *display;
int array_pos;
int x, y;

{
  int size;

/* be sure it is bright enough to display */
  if (! check_mag_limit(array_pos,mag_limit))
    return(TRUE);

/* calculate size based on magnitude */
  size = STARSIZE(sao_array[array_pos].mag);

/* draw the star in the correct position with the correct size */
  return(ps_draw_star(display,(double)x,(double)y,(double)size));
}



/* display the catalog data for the given star */

static boolean display_sao(sao_fd,sao_array_pos)

FILE *sao_fd;
int sao_array_pos;

{
  int i;

/* get info from the SAO catalog */
  if (fseek(sao_fd,sao_array[sao_array_pos].filepos,SEEK_SET) == -1)
    return(FALSE);
  if (fgets(sao_rec,sizeof(sao_rec),sao_fd) == (char *)NULL)
    return(FALSE);

/* fill out the (possibly short) record with blanks */
  for (i = strlen(sao_rec) - 1; i < SAO_RECLEN; i++)
    sao_rec[i] = ' ';

/* format the record into an informational display */
  format_sao_display(sao_rec);

  return(TRUE);
}



/* find the SAO star in the buffer and update the display coordinates */

static boolean find_sao_star(buffer)

char *buffer;

{
  int sao_no;
  int i;

/* be sure this is an SAO designation */
  if ((strncmp(buffer,"SAO",3) == EQUAL) || strncmp(buffer,"sao",3) == EQUAL)
    buffer = &buffer[3];
  else
    return(FALSE);

/* get the SAO number */
  sao_no = atoi(buffer);

/* loop through the array looking for this number */
  for (i = 0; i <= sao_array_max; i++)
    if (sao_array[i].sao_number == sao_no)
      break;

/* return failure if we didn't find it */
  if (i > sao_array_max)
    return(FALSE);
  else {
    /* update the display coordinates */
    update_display_pos(sao_array[i].ra_rad,sao_array[i].dec_rad);
    return(TRUE);
  }

/* NOTREACHED */
}



/* format a nice name for a star */

/* ARGSUSED */

static void format_sao_name(sao_pos,buffer,font)

int sao_pos;
char *buffer;
IDfont *font;

{
/* use the SAO number */
  sprintf(buffer,"SAO %d",sao_array[sao_pos].sao_number);

  return;
}
