/*
 * xsky - an interactive sky atlas
 *
 * Copyright 1992-6, Terry R. Friedrichsen
 *
 * This program may be copied and redistributed, in whole or in part,
 * as long as you don't try to make any money from the sale or redis-
 * tribution of the program or any part of the program, or pretend
 * that you wrote the program or any of its parts unless specifically
 * credited by the original author.
 *
 * You are free to make use of this software in your own programs, as
 * long as you credit the original author where it is due.
 */

/*
 * WARRANTY:
 * xsky was written as a learning project and as a demonstration of
 * X Window System programming.  xsky doesn't do anything; it is not
 * merchantable, and it is not fit for any purpose whatsoever.  In
 * fact, don't use xsky at all; it's free, and you're getting what
 * you paid for.
 */

#include <stdio.h>
#include <ctype.h>

#include <math.h>
#include <stdlib.h>

#include <unistd.h>

#include <string.h>

#include <X11/Intrinsic.h>

#include "rngc.h"

#include "skydefs.h"

#include "pos.h"
#include "catalog.h"
#include "draw.h"
#include "catreader.h"

/* uncomment the next line to suppress display of nonexistent RNGC objects */
/* #define SUPPRESS_NONEXISTENT */

struct rngc_node {
  double ra_rad;
  double dec_rad;
  long filepos;
  shortint rngc_number;
  char subletter;
  smallint objtype;
};

/* static function prototypes */
static FILE *read_rngc PROTOTYPE((void));
static void calculate_rngc_field PROTOTYPE((struct display *,
					                struct cat_header *));
static boolean find_rngc_obj PROTOTYPE((char *));
static void get_rngc_pos PROTOTYPE((int,double *,double *));
static void draw_rngc_obj PROTOTYPE((Widget,int,int,int,int,int,int,int));
static boolean ps_draw_rngc_obj PROTOTYPE((struct display *,int,int,int));
static boolean display_rngc PROTOTYPE((FILE *,int));
static void format_rngc_name PROTOTYPE((int,char *,IDfont *));
static void fill_rngc_entry PROTOTYPE((char *,int));
static int rngc_compar PROTOTYPE((void *,void *,int));

/* external function prototypes */
extern void read_desc_table PROTOTYPE((void));
extern void read_ngc_name_table PROTOTYPE((void));
extern char *build_filespec PROTOTYPE((char *,char *));
extern boolean read_binary_database PROTOTYPE((char *,void *,int *));
extern void init_precess PROTOTYPE((double,double));
extern boolean write_binary_database PROTOTYPE((char *,void *,size_t,size_t));
extern double ra_pos_to_rad PROTOTYPE((struct ra_pos));
extern double dec_pos_to_rad PROTOTYPE((struct dec_pos));
extern void precess PROTOTYPE((double,double,double *,double *));
extern void add_to_display_list PROTOTYPE((struct display *,
					           struct cat_header *,
					                  int,double,double));
extern void draw_object PROTOTYPE((Widget,int,int,int,ObjShape));
extern boolean ps_draw_object PROTOTYPE((struct display *,double,double,
					                    double,ObjShape));
extern void format_rngc_display PROTOTYPE((char *));
extern int binsearch PROTOTYPE((void *,int,void *,int (*)()));
extern void update_display_pos PROTOTYPE((double,double));
extern boolean find_ngc_by_name PROTOTYPE((char *,boolean (*)()));
extern boolean find_messier_obj PROTOTYPE((char *,boolean (*)()));
extern int find_messier_number PROTOTYPE((int));

static struct rngc_node rngc_array[RNGC_NUMREC];
static int rngc_array_max;

#define RNGC_DATA_FILE   "rngc.dat"
#define RNGC_BINARY_FILE "rngc.bin"

/* RNGC record buffer */
static char rngc_rec[RNGC_RECLEN + 1 + 1];



/* initialize the catalog data structure */

void init_rngc(catalog)

struct cat_header *catalog;

{
  catalog->read_data = read_rngc;
  catalog->calculate_field = calculate_rngc_field;
  catalog->mag_limit_check = NULLBOOLFUNCTION;
  catalog->find_object = find_rngc_obj;
  catalog->get_object_pos = get_rngc_pos;
  catalog->draw_object = draw_rngc_obj;
  catalog->ps_draw_object = ps_draw_rngc_obj;
  catalog->display_data = display_rngc;
  catalog->format_name = format_rngc_name;
  catalog->menu_name = RNGC_MENU_NAME;
  catalog->catalog_name = RNGC_CATALOG_NAME;

  return;
}



/* read in the RNGC catalog, extracting needed info */

static FILE *read_rngc()

{
  char *filebuff;
  FILE *rngc_fd;
  char *bin_filebuff;
  int rngc_entry;

/* read in the table of RNGC description abbreviations */
  read_desc_table();

/* read in the table of common NGC object names */
  read_ngc_name_table();

/* get the path to the RNGC data */
  filebuff = build_filespec("RNGC_PATH",RNGC_DATA_FILE);

/* open the RNGC data file */
  if ((rngc_fd = fopen(filebuff,"r")) == (FILE *)NULL) {
    printf("sky:  can't open RNGC data file %s\n",filebuff);
    perror("sky");
    exit(1);
  }

/* get the path to the RNGC binary file */
  bin_filebuff = build_filespec("RNGC_BIN_PATH",RNGC_BINARY_FILE);

/* see if we can read the binary file */
  if (access(bin_filebuff,R_OK) == 0)
    /* we can; try to use the pre-built database */
    if (read_binary_database(bin_filebuff,rngc_array,&rngc_array_max)) {
      /* success; free the filename buffers */
      free((void *)filebuff);
      free((void *)bin_filebuff);

      /* return the fd of the open data file */
      return(rngc_fd);
    }

/* prior to reading the RNGC catalog, build the rotation matrix for precessing
 * the object coordinates to the common equinox from the given epoch */
  if (RNGC_EPOCH != COMMON_EQUINOX)
    init_precess(RNGC_EPOCH,COMMON_EQUINOX);

/* read the RNGC catalog, saving the information necessary for the display */
  rngc_entry = 0;
  while (TRUE) {
    rngc_array[rngc_entry].filepos = ftell(rngc_fd);
    if (fgets(rngc_rec,sizeof(rngc_rec),rngc_fd) == (char *)NULL)
      break;
    fill_rngc_entry(rngc_rec,rngc_entry++);
  }

/* leave the file open for later use */
/*  (void)fclose(rngc_fd);*/

/* remember the last rngc entry we filled */
  rngc_array_max = rngc_entry - 1;

/* attempt to save the binary database */
  if (access(bin_filebuff,W_OK) == 0)
    /* it's writeable, so let's try */
    if (! write_binary_database(bin_filebuff,(void *)rngc_array,
				           (size_t)sizeof(struct rngc_node),
				                          (size_t)rngc_entry))
      /* dump of binary database failed; be sure the file is gone */
      (void)unlink(bin_filebuff);

/* free the filename buffers */
  free((void *)filebuff);
  free((void *)bin_filebuff);

  return(rngc_fd);
}



/* given an object's index, return its right ascension and declination */

static void get_rngc_pos(array_pos,ra_rad,dec_rad)

int array_pos;
double *ra_rad, *dec_rad;

{
  *ra_rad = rngc_array[array_pos].ra_rad;
  *dec_rad = rngc_array[array_pos].dec_rad;

  return;
}



/* given an RNGC record, make a data node for it */

static void fill_rngc_entry(record,rngc_entry)

char *record;
int rngc_entry;

{
  int reclen;
  char field[11];
  struct ra_pos ra_pos;
  struct dec_pos dec_pos;

/* get the length of the record for GET_FIELD */
  reclen = strlen(record) - 1;

/* get the RNGC number */
  GET_FIELD(field,record,CAT_START,5);
  rngc_array[rngc_entry].rngc_number = atoi(field);

/* get the RNGC sub-letter */
  GET_FIELD(field,record,SUBLETTER_START,1);
  rngc_array[rngc_entry].subletter = field[0];

/* get the object type */
  GET_FIELD(field,record,TYPE_START,2);
  rngc_array[rngc_entry].objtype = atoi(field);

/* get the RA in pieces */
  GET_FIELD(field,record,RA_START,2);
  ra_pos.hours = (smallint)atoi(field);

  GET_FIELD(field,record,RA_START + 2,2);
  ra_pos.minutes = (smallint)atoi(field);

  GET_FIELD(field,record,RA_START + 5,1);
  ra_pos.seconds = (smallint)(atoi(field) * (60 / 10));

  ra_pos.thousandths = (shortint)0;

/* similarly, get the declination in pieces */
  GET_FIELD(field,record,DEC_START,1);
  dec_pos.sign = (short)1;
  if (field[0] == '-')
    dec_pos.sign = (short)-1;

  GET_FIELD(field,record,DEC_START + 1,2);
  dec_pos.degrees = (smallint)atoi(field);

  GET_FIELD(field,record,DEC_START + 3,2);
  dec_pos.minutes = (smallint)atoi(field);

  dec_pos.seconds = (smallint)0;
  dec_pos.hundredths = (smallint)0;

/* compute the RA and declination in radians */
  rngc_array[rngc_entry].ra_rad = ra_pos_to_rad(ra_pos);
  rngc_array[rngc_entry].dec_rad = dec_pos_to_rad(dec_pos);

/* precess them to the common equinox */
  if (RNGC_EPOCH != COMMON_EQUINOX)
    precess(rngc_array[rngc_entry].ra_rad,rngc_array[rngc_entry].dec_rad,
	       &rngc_array[rngc_entry].ra_rad,&rngc_array[rngc_entry].dec_rad);

  return;
}



/* construct a list of those objects which appear in the current display */

static void calculate_rngc_field(display,catalog)

struct display *display;
struct cat_header *catalog;

{
  int i;
#ifdef SUPPRESS_NONEXISTENT
  int objtype;
#endif

/* loop through all RNGC entries, checking for visibility */
  for (i = 0; i <= rngc_array_max; i++) {
#ifdef SUPPRESS_NONEXISTENT
    /* if the object is nonexistent, just ignore it */
    objtype = rngc_array[i].objtype;
    if (objtype >= 10)
      objtype /= 10;
    if (objtype == NONEXISTENT)
      continue;
#endif

/* check this object and add it to the list if visible */
    add_to_display_list(display,catalog,i,rngc_array[i].ra_rad,
                                                       rngc_array[i].dec_rad);
  }

#ifdef DEBUG
  printf("calculate_rngc_field returning\n");
#endif
  return;
}



/* draw an RNGC object on the sky */

static void draw_rngc_obj(sky_widget,array_pos,obj_x,obj_y,ul_x,ul_y,
		                                                 width,height)

Widget sky_widget;
int array_pos;
int obj_x, obj_y;
int ul_x, ul_y;
int width, height;

{
  int objtype;
  int safety_margin;

/* get the major object type */
  objtype = rngc_array[array_pos].objtype;
  if (objtype >= 10)
    objtype /= 10;

#ifdef DEBUG
  printf("draw_obj called to draw figure type %d at %d,%d\n",objtype,
	                                                         obj_x,obj_y);
#endif

#ifdef SUPPRESS_NONEXISTENT
/* if this is a non-existent object, suppress the display */
  if (objtype == NONEXISTENT)
    return;
#endif

/* figure a safety margin - XClearArea() seems to clear a tad too much when
 * erasing IDs */
  if (objtype == GALAXY)
    safety_margin = 4;
  else if ((objtype == OPEN_CLUSTER) || (objtype == GLOBULAR_CLUSTER))
    safety_margin = 3;
  else
    safety_margin = 2;

 /* draw the object even if it is only partially in the box */
  if (((obj_x + (OBJSIZE + 1) / 2 + safety_margin) >= ul_x) &&
            ((obj_x - (OBJSIZE + 1) / 2 - safety_margin) <= (ul_x + width)) &&
            ((obj_y + (OBJSIZE + 1) / 2 + safety_margin) >= ul_y) &&
            ((obj_y - (OBJSIZE + 1) / 2 - safety_margin) <= (ul_y + height)))
    ;
  else
    return;

/* draw a different shape for different types of objects */
  if (objtype == GALAXY)
    /* draw an ellipse for a galaxy */
    draw_object(sky_widget,obj_x,obj_y,OBJSIZE,ELLIPSE);
  else if ((objtype == OPEN_CLUSTER) || (objtype == GLOBULAR_CLUSTER))
    /* draw a circle for open and globular clusters */
    draw_object(sky_widget,obj_x,obj_y,OBJSIZE,CIRCLE);
  else
    /* for the moment, draw a square for anything else */
    draw_object(sky_widget,obj_x,obj_y,OBJSIZE,SQUARE);

  return;
}



/* draw one RNGC object on the PostScript display */

static boolean ps_draw_rngc_obj(display,array_pos,x,y)

struct display *display;
int array_pos;
int x, y;

{
  int objtype;
  ObjShape shape;

/* get the major object type */
  objtype = rngc_array[array_pos].objtype;
  if (objtype >= 10)
    objtype /= 10;

#ifdef SUPPRESS_NONEXISTENT
/* if this is a non-existent object, suppress the display */
  if (objtype == NONEXISTENT)
    return(TRUE);
#endif

/* draw the object */
  if (objtype == GALAXY)
    /* draw an ellipse for a galaxy */
    shape = ELLIPSE;
  else if ((objtype == OPEN_CLUSTER) || (objtype == GLOBULAR_CLUSTER))
    /* draw a circle for open and globular clusters */
    shape = CIRCLE;
  else
    /* for the moment, draw a square for anything else */
    shape = SQUARE;

/* draw the object in the correct position with the correct size */
  return(ps_draw_object(display,(double)x,(double)y,(double)OBJSIZE,shape));
}



/* display the catalog data for the given object */

static boolean display_rngc(rngc_fd,rngc_array_pos)

FILE *rngc_fd;
int rngc_array_pos;

{
  int i;

/* get info from the RNGC catalog */
  if (fseek(rngc_fd,rngc_array[rngc_array_pos].filepos,SEEK_SET) == -1)
      return(FALSE);
  if (fgets(rngc_rec,RNGC_RECLEN + 2,rngc_fd) == (char *)NULL)
      return(FALSE);

/* fill out the (possibly short) record with blanks */
  for (i = strlen(rngc_rec) - 1; i < RNGC_RECLEN; i++)
    rngc_rec[i] = ' ';

/* format the record into an informational display */
  format_rngc_display(rngc_rec);

  return(TRUE);
}



/* find the RNGC object in the buffer and update the display coordinates */

static boolean find_rngc_obj(buffer)

char *buffer;

{
  int rngc_no;
  char rngc_subletter;
  int rngc_idx;

/* check to see if this is an RNGC designation and step to the number */
  if ((strncmp(buffer,"NGC",3) == EQUAL) || strncmp(buffer,"ngc",3) == EQUAL)
    buffer = &buffer[3];
  else if ((strncmp(buffer,"RNGC",4) == EQUAL) ||
	                                   strncmp(buffer,"rngc",4) == EQUAL)
    buffer = &buffer[4];
  else if ((strncmp(buffer,"IC",2) == EQUAL) ||
	                                     strncmp(buffer,"ic",2) == EQUAL)
    /* find_ngc_by_name() called us with an IC object */
    return(FALSE);
  else if (find_ngc_by_name(buffer,find_rngc_obj))
    /* this was a recognized object name */
    return(TRUE);
  else
    /* try a Messier designation */
    return(find_messier_obj(buffer,find_rngc_obj));

  /* skip white space */
  while (isspace(*buffer))
    buffer++;

/* get the RNGC number */
  rngc_no = atoi(buffer);

/* get the letter suffix, if any */

  /* skip decimal digits */
  while (isdigit(*buffer))
    buffer++;

  /* skip more white space */
  while (isspace(*buffer))
    buffer++;

  /* grab the letter */
  if (isalpha(*buffer))
    rngc_subletter = toupper(*buffer);
  else
    rngc_subletter = ' ';

/* find the RNGC number */
  rngc_idx = binsearch((void *)rngc_array,rngc_array_max,(void *)&rngc_no,
		                                                 rngc_compar);

  /* return FALSE if we didn't find the RNGC object */
  if (rngc_idx == -1)
    return(FALSE);

/* back up to the beginning of this RNGC number */
  while (rngc_idx > 0)
    if (rngc_array[rngc_idx - 1].rngc_number == rngc_no)
      rngc_idx--;
    else
      break;

/* search for the RNGC number with the matching subletter */
  while ((rngc_idx <= rngc_array_max) &&
	                        (rngc_array[rngc_idx].rngc_number == rngc_no))
    if (rngc_subletter == rngc_array[rngc_idx].subletter)
      break;
    else
      rngc_idx++;

/* if we have a match, update the display center */
  if ((rngc_no == rngc_array[rngc_idx].rngc_number) &&
                         (rngc_subletter == rngc_array[rngc_idx].subletter)) {
    /* update the display coordinates */
    update_display_pos(rngc_array[rngc_idx].ra_rad,
		                                rngc_array[rngc_idx].dec_rad);
    return(TRUE);
  }
  else
    return(FALSE);

/* NOTREACHED */
}



/* comparison routine for binary search to find RNGC number in rngc_array */

static int rngc_compar(table,key,idx)

void *table;
void *key;
int idx;

{
  return(*(int *)key - ((struct rngc_node *)table)[idx].rngc_number);
}



/* format a nice name for the RNGC object */

/* ARGSUSED */

static void format_rngc_name(rngc_pos,buffer,font)

int rngc_pos;
char *buffer;
IDfont *font;

{
  int rngc_no;
  int messier_no;

/* get the RNGC number */
  rngc_no = rngc_array[rngc_pos].rngc_number;

/* see if this is a Messier object */
  if ((messier_no = find_messier_number(rngc_no)) == -1)
    /* this is not a Messier object */
    sprintf(buffer,"NGC %d",rngc_no);
  else
    /* use the Messier designation */
    sprintf(buffer,"M %d",messier_no);

  return;
}
