/*
 * xsky - an interactive sky atlas
 *
 * Copyright 1992-5, Terry R. Friedrichsen
 *
 * This program may be copied and redistributed, in whole or in part,
 * as long as you don't try to make any money from the sale or redis-
 * tribution of the program or any part of the program, or pretend
 * that you wrote the program or any of its parts unless specifically
 * credited by the original author.
 *
 * You are free to make use of this software in your own programs, as
 * long as you credit the original author where it is due.
 */

/*
 * WARRANTY:
 * xsky was written as a learning project and as a demonstration of
 * X Window System programming.  xsky doesn't do anything; it is not
 * merchantable, and it is not fit for any purpose whatsoever.  In
 * fact, don't use xsky at all; it's free, and you're getting what
 * you paid for.
 */

/* this routine formats and displays an RNGC record */

#include <stdio.h>
#include <ctype.h>
#include <string.h>

#include "rngc.h"

#include "skydefs.h"

#include "format.h"

/* static function prototypes */
static void display_type PROTOTYPE((char *));
static void display_pos PROTOTYPE((char *));

/* external function prototypes */
extern int find_messier_number PROTOTYPE((int));
extern char *find_ngc_common_names PROTOTYPE((int));
extern void display_desc PROTOTYPE((int,int,char *));

struct rngc_type {
  char typecode;
  char *typeval;
};

/* table of RNGC type code expansions */
static struct rngc_type type_table[] = {
  {'0',"unverified"},
  {'1',"open cluster"},
  {'2',"globular cluster"},
  {'3',"diffuse nebula"},
  {'4',"planetary nebula"},
  {'5',"galaxy"},
  {'6',"associated with nebulosity"},
  {'7',"nonexistent"},
  {'8',"in Large Magellanic Cloud"},
  {'9',"in Small Magellanic Cloud"}
};

#define MAX_RNGC_ROW   70
#define NUM_RNGC_ROWS  16



void format_rngc_display(record)

char *record;

{
  int rngc_no;
  int i;
  char *name;

/* start at the beginning of the display */
  idx = 0;
  row = 0;
  row_start = 0;
  max_row = MAX_RNGC_ROW;

/* convert the catalog number to binary */
  ljust(&record[CAT_START],5);
  rngc_no = atoi(&record[CAT_START]);

/* display the catalog number */
  sprintf(&obj_info[idx],"NGC %d%c",rngc_no,record[SUBLETTER_START]);
  idx += strlen(&obj_info[idx]);
  advance_to_tab();

/* add the Messier number, if found */
  if ((i = find_messier_number(rngc_no)) != -1) {
    sprintf(&obj_info[idx],"M %d",i);
    idx += strlen(&obj_info[idx]);
    advance_to_tab();
  }

/* now display the object type */
  display_type(&record[TYPE_START]);
  next_line();

/* show the common name, if any */
  if ((name = find_ngc_common_names(rngc_no)) != (char *)NULL) {
    advance_to_tab();
    add_string(name);
    next_line();
  }

/* space down some */
  next_line();

/* display the position */
  display_pos(&record[POS_START]);

/* now display the magnitude and description */
  blank_line();
  if (strncmp(&record[MAG_START],"    ",4) != EQUAL) {
    sprintf(&obj_info[idx],"Mag:  %.4s",&record[MAG_START]);
    idx += strlen(&obj_info[idx]);
  }

/* add in the description, if any */
  if (record[DESC_START] != ' ') {
    blank_line();
    next_line();
    /* update row and column to end of RNGC description */
    display_desc(rngc_no,record[TYPE_START + 1] - '0',&record[DESC_START]);
    idx += strlen(&obj_info[idx]);
  }

/* fill the display out to the correct number of lines */
  while (row < NUM_RNGC_ROWS)
    next_line();
  obj_info[idx] = '\0';

/* and go back */
  return;
}



static void display_pos(posptr)

char *posptr;

{
/* identify the epoch */
  sprintf(&obj_info[idx],"Position (1975.0)");
  idx += strlen(&obj_info[idx]);

/* fix the minutes field of the right ascension */
  if (posptr[2] == ' ')
    posptr[2] = '0';

/* display the right ascension */
  advance_to_tab();
  sprintf(&obj_info[idx]," RA:   %.2sh %.2sm.%.1s",&posptr[0],
							&posptr[2],&posptr[5]);
  idx += strlen(&obj_info[idx]);
  next_line();

/* fix the arcminutes field of the declination */
  if (posptr[9] == ' ')
    posptr[9] = '0';

/* and display the declination */
  advance_to_tab();
  advance_to_tab();
  advance_to_tab();
  sprintf(&obj_info[idx],"Dec:  %.3sd %.2s'",&posptr[6],&posptr[9]);
  idx += strlen(&obj_info[idx]);
  next_line();

  return;
}



static void display_type(typeptr)

char *typeptr;

{
  int lower;
  int i, j;

/* display the type, lowest code first */
  lower = (typeptr[0] < typeptr[1] ? 0 : 1);

/* process both possible type codes */
  for (j = 0; j < 2; j++) {
/* nothing to display if non-digit */
    if (isdigit(typeptr[lower])) {
/* find the type code to display */
      i = 0;
      while (i < 10) {
	if (typeptr[lower] == type_table[i].typecode) {
	  sprintf(&obj_info[idx],"%s ",type_table[i].typeval);
	  idx += strlen(&obj_info[idx]);
	  break;
	}

/* nope; try the next type code */
	i++;
      }
    }

/* now do the higher-valued code */
    lower = 1 - lower;
  }

  return;
}
