/*
 * xsky - an interactive sky atlas
 *
 * Copyright 1992-5, Terry R. Friedrichsen
 *
 * This program may be copied and redistributed, in whole or in part,
 * as long as you don't try to make any money from the sale or redis-
 * tribution of the program or any part of the program, or pretend
 * that you wrote the program or any of its parts unless specifically
 * credited by the original author.
 *
 * You are free to make use of this software in your own programs, as
 * long as you credit the original author where it is due.
 */

/*
 * WARRANTY:
 * xsky was written as a learning project and as a demonstration of
 * X Window System programming.  xsky doesn't do anything; it is not
 * merchantable, and it is not fit for any purpose whatsoever.  In
 * fact, don't use xsky at all; it's free, and you're getting what
 * you paid for.
 */

/* this routine prints out English for the new-style RNGC object
 * descriptions.  the algoritm is to grab each piece delimited
 * by commas, then match the longest possible piece with the
 * RNGC description code table.  that piece's translation is
 * then output and the piece is discarded. */

#include <stdio.h>
#include <ctype.h>

#include <stdlib.h>

#include <string.h>

#include "rngc.h"

#include "skydefs.h"

#include "rngc_desc.h"

#include "format.h"

/* static function prototypes */
static void read_nf_table PROTOTYPE((void));
static int find_longest_match PROTOTYPE((char *,int *,struct key_desc *));

/* external function prototypes */
extern char *build_filespec PROTOTYPE((char *,char *));
extern int binsearch PROTOTYPE((void *,int,void *,int (*)()));

/* table of description abbreviations and translations */
static struct key_desc desc_table[MAXNUMDESCS + 2 + 1];
static int max_desc;

/* index of the NF description abbreviation for NF ambiguity resolution */
static int nf_desc_idx;

/* index of the alternative NF description */
static int not_found_idx;

/* index of the S description abbreviation for NF ambiguity resolution */
static int s_desc_idx;

/* index of the alternative S description if NF ambiguity */
static int sulentic_idx;

/* indices of alternative S descriptions if south/spiral ambiguity */
static int spiral_idx;
static int south_idx;

/* table of RNGC objects which need alternate expansions */
static int ambig_table[NUM_AMBIG_OBJS + 1];
static int max_ambig_idx;



#ifdef STANDALONE

void read_desc_table PROTOTYPE((void));
void display_desc PROTOTYPE((int,int,char *));

char obj_info[2400];



main(argc,argv)

int argc;
char *argv[];

{
  int find_num;
  FILE *rngc_fd;
  char rngc_rec[RNGC_RECLEN + 1];
  int rngc_num;
  int objtype;

/* handle arguments */
  if (argc > 1) {
    if (strspn(argv[1],"0123456789") == strlen(argv[1]))
      find_num = atoi(argv[1]);
    else
      find_num = 0;
  }

/* read in the description table */
  read_desc_table();

/* open the RNGC data file */
  if ((rngc_fd = fopen("rngc.dat","r")) == (FILE *)NULL) {
    printf("can't open rngc.dat\n");
    perror("");
    exit(1);
  }

/* build the description on a single line */
  max_row = 2400;

/* if there is an argument, and it is not all numeric, presume it is an RNGC
 * description to translate */
  if ((argc > 1) && (find_num == 0)) {
    idx = row = row_start = 0;
    display_desc(-1,-1,argv[1]);
    printf("%s\n",obj_info);

    exit(0);
  }

/* loop through the entire RNGC file */
  while (fgets(rngc_rec,sizeof(rngc_rec),rngc_fd) != (char *)NULL) {
    /* get the RNGC number and the object type */
    rngc_num = atoi(rngc_rec);
    objtype = rngc_rec[TYPE_START + 1] - '0';

    /* if this isn't the RNGC number we want, skip it */
    if ((find_num > 0) && (rngc_num < find_num))
      continue;

    /* if we've passed the RNGC number we're after, quit */
    if ((find_num > 0) && (rngc_num > find_num))
      break;

    /* if this record has no RNGC description, skip it */
    if ((strlen(rngc_rec) - 1) <= DESC_START)
      continue;

    /* build the description string and emit it */
    idx = row = row_start = 0;
    display_desc(rngc_num,objtype,&rngc_rec[DESC_START]);
    printf("%d\t%s\n",rngc_num,obj_info);
  }

/* be neat and close the file */
  (void)fclose(rngc_fd);

  exit(0);

/* NOTREACHED */
}

#endif



void display_desc(rngc_num,objtype,record)

int rngc_num;
int objtype;
char *record;

{
/* record is a pointer to the start of the RNGC description text */
  int i, j;
  char descbuff[MAXDESCLEN + 1];
  int descidx;
  short nomatch_flag;

/* strip trailing whitespace from the record */
  i = strlen(record) - 1;
  while (isspace(record[--i]))
    if (i == 0) {
      i--;
      break;
    }
  record[++i] = '\0';

#ifndef STANDALONE
/* output a header */
  add_string("Revised NGC description:  ");

/* show the actual raw description string */
  add_string(record);

/* position for start of expanded description text */
  blank_line();
  advance_to_tab();
#endif

/* assume we will find matches */
  nomatch_flag = FALSE;

/* loop through the description buffer */
  i = 0;
  while (TRUE) {
    /* skip any white space to get to descriptions */
    while (isspace(record[i]))
      i++;

    /* stop if we reach the null */
    if (record[i] == '\0')
      break;

/* copy a description piece until a comma, null, or whitespace */
    j = 0;
    while ((record[i] != ',') && (record[i] != '\0') && (! isspace(record[i])))
      descbuff[j++] = record[i++];
    /* end the description buffer with a null */
    descbuff[j] = '\0';

/* now loop through the description, piecing out the code words */
    j = 0;
    while (j < strlen(descbuff)) {
/* now match the largest piece possible from the description table */
      descidx = find_longest_match(descbuff,&j,desc_table);

/* if the abbreviation we're expanding is NF, check for possible alternate
 * interpretation */
      if (descidx == nf_desc_idx)
	if ((objtype == NONEXISTENT) || (binsearch((void *)ambig_table,
						       max_ambig_idx,
						       (void *)&rngc_num,
						       NULLINTFUNCTION) != -1))
	  descidx = not_found_idx;

/* if the abbreviation we're expanding is S, check for possible alternate
 * interpretations */
      if (descidx == s_desc_idx)
	if ((objtype == NONEXISTENT) || (binsearch((void *)ambig_table,
						       max_ambig_idx,
						       (void *)&rngc_num,
						       NULLINTFUNCTION) != -1))
	  descidx = sulentic_idx;
        else
	  /* if the delimiter is ',', presume spiral */
	  if (record[i] == ',')
	    descidx = spiral_idx;
          else
	    /* otherwise, presume south except for 5960, 6327, and 7260 */
	    if ((rngc_num == 5960) || (rngc_num == 6327) || (rngc_num == 7260))
	      descidx = spiral_idx;
            else
	      descidx = south_idx;

/* advance to the next line, if necessary (leaving room for a delimiter) */
      if (! FIT(desc_table[descidx].desc_string))
	next_line();

/* put out the expansion of this piece of the description */
      if ((descidx == max_desc) && nomatch_flag)
	/* no, and neither did the last one; print with no preceding space */
	;
      else if (desc_table[descidx].desc_string[0] == '?')
	/* also, don't space in front of a question mark */
	;
      else
	/* precede the piece of description by a space */
	add_string(" ");
      add_string(desc_table[descidx].desc_string);

/* set nomatch flag according to whether this piece matched or not */
      nomatch_flag = (descidx == max_desc);
    }

/* put out the delimiter unless it is a blank or a null */
    if (record[i] == '\0')
      /* we are done when we see the null */
      break;
    else if (record[i] == ' ')
      if (nomatch_flag) {
	/* put out the delimiter anyway */
	sprintf(&obj_info[idx],"%c",record[i]);
	idx += strlen(&obj_info[idx]);
      }
      else
	;
    else {
      sprintf(&obj_info[idx],"%c",record[i]);
      idx += strlen(&obj_info[idx]);
    }

/* step past the delimiter to the next piece of the description */
    i++;
  }

  return;
}



/* buffer for reading auxiliary data files */

static char inbuf[BUFSIZ + 1];


void read_desc_table()

{
  char *filebuff;
  FILE *desc_fd;
  int desc;
  int i;

/* build the file path */
  filebuff = build_filespec("RNGC_PATH",RNGC_DESC_FILE);

/* open the descriptor file */
  if ((desc_fd = fopen(filebuff,"r")) == (FILE *)NULL) {
    printf("sky:  error opening RNGC descriptor file %s\n",filebuff);
    perror("sky");
    exit(1);
  }

/* loop through the file, reading in descriptor table entries */
  desc = 0;
  while (fgets(inbuf,BUFSIZ,desc_fd) != NULL) {
    /* remove the newline at the end */
    inbuf[strlen(inbuf) - 1] = '\0';

/* copy the description abbreviation until white space */
    i = 0;
    while (! isspace(inbuf[i])) {
      desc_table[desc].desc_name[i] = inbuf[i];
      i++;
    }

    /* be sure the abbreviation ends with a null */
    desc_table[desc].desc_name[i] = '\0';

/* save the index of the NF description, so we can evade a gross ambiguity */
    if (strcmp(desc_table[desc].desc_name,"NF") == EQUAL)
      nf_desc_idx = desc;

/* save the index of the S description, so we can evade a gross ambiguity */
    if (strcmp(desc_table[desc].desc_name,"S") == EQUAL)
      s_desc_idx = desc;

/* skip to the description string */
    while (isspace(inbuf[i]))
      i++;

/* get some space for the description string */
    desc_table[desc].desc_string = (char *)malloc(strlen(&inbuf[i]) + 1);

/* copy in the description string */
    strcpy(desc_table[desc].desc_string,&inbuf[i]);

/* step to the next table entry */
    desc++;
  }

/* be neat */
  (void)fclose(desc_fd);

/* free the filename buffer */
  free((void *)filebuff);

/* add an alternate description for NF */
  strcpy(desc_table[desc].desc_name,"nf");
  desc_table[desc].desc_string = (char *)malloc(strlen("not found") + 1);
  strcpy(desc_table[desc].desc_string,"not found");
  not_found_idx = desc++;

/* add an alternate description for S */
  strcpy(desc_table[desc].desc_name,"s");
  desc_table[desc].desc_string = (char *)malloc(strlen("- Sulentic") + 1);
  strcpy(desc_table[desc].desc_string,"- Sulentic");
  sulentic_idx = desc++;

/* add another alternate description for S */
  strcpy(desc_table[desc].desc_name,"s");
  desc_table[desc].desc_string = (char *)malloc(strlen("spiral") + 1);
  strcpy(desc_table[desc].desc_string,"spiral");
  spiral_idx = desc++;

/* add another alternate description for S */
  strcpy(desc_table[desc].desc_name,"s");
  desc_table[desc].desc_string = (char *)malloc(strlen("south") + 1);
  strcpy(desc_table[desc].desc_string,"south");
  south_idx = desc++;

/* make the last description be some question marks */
  strcpy(desc_table[desc].desc_name,"\\");
  desc_table[desc].desc_string = (char *)malloc(strlen("???") + 1);
  strcpy(desc_table[desc].desc_string,"???");

/* remember the number of descriptions read */
  if ((max_desc = desc) >= MAXNUMDESCS) {
    printf("sky:  too many descriptions in RNGC description table\n");
    exit(1);
  }

/* also read the table of NF description objects */
  read_nf_table();

  return;
}



/* read the table of RNGC numbers which need modified NF expansion */

static void read_nf_table()

{
  char *filebuff;
  FILE *ambig_fd;
  int i;

/* build the file path */
  filebuff = build_filespec("RNGC_PATH",RNGC_AMBIG_FILE);

/* open the descriptor file */
  if ((ambig_fd = fopen(filebuff,"r")) == (FILE *)NULL) {
    printf("sky:  error opening RNGC index file %s\n",filebuff);
    perror("sky");
    exit(1);
  }

/* loop through the file, reading in RNGC object numbers */
  i = 0;
  while (fgets(inbuf,BUFSIZ,ambig_fd) != NULL)
    ambig_table[i++] = atoi(inbuf);

/* save the maximum ambiguous-description table entry */
  max_ambig_idx = --i;

/* be neat */
  (void)fclose(ambig_fd);

/* free the filename buffer */
  free((void *)filebuff);

  return;
}



static int find_longest_match(buff,ind,table)

char *buff;
int *ind;
struct key_desc *table;

{
  int max_match, max_match_idx;
  char *ptr;
  int desc;
  int i;

/* make the match length impossibly small */
  max_match = -1;
  max_match_idx = -1;

/* begin at the beginning */
  desc = 0;
  while (desc <= max_desc) {

/* point to the start of the description */
    ptr = &buff[*ind];

/* match characters for a while */
    i = strlen(table[desc].desc_name);
    if (strncmp(ptr,table[desc].desc_name,(size_t)i) == EQUAL)
      if (i > max_match) {

/* the new match is bigger; save the length and the index */
	max_match = i;
	max_match_idx = desc;
      }

/* step to the next description */
    if (table[++desc].desc_name[0] == '\\')
      break;
  }

/* if no match was found, update by one to bypass a bogus piece of description
 * and set up the unrecognized character as the output */
  if (max_match_idx == -1) {
    table[desc].desc_string[0] = buff[(*ind)++];
    table[desc].desc_string[1] = '\0';

    /* make the matching index be this one */
    max_match_idx = desc;
  }
  else
/* a match was found; update the index by the 1-relative length */
    *ind += max_match;

/* return the index to the relevant description table entry */
  return(max_match_idx);
}
