/*
 * xsky - an interactive sky atlas
 *
 * Copyright 1992-5, Terry R. Friedrichsen
 *
 * This program may be copied and redistributed, in whole or in part,
 * as long as you don't try to make any money from the sale or redis-
 * tribution of the program or any part of the program, or pretend
 * that you wrote the program or any of its parts unless specifically
 * credited by the original author.
 *
 * You are free to make use of this software in your own programs, as
 * long as you credit the original author where it is due.
 */

/*
 * WARRANTY:
 * xsky was written as a learning project and as a demonstration of
 * X Window System programming.  xsky doesn't do anything; it is not
 * merchantable, and it is not fit for any purpose whatsoever.  In
 * fact, don't use xsky at all; it's free, and you're getting what
 * you paid for.
 */

#include <stdio.h>
#include <ctype.h>

#include <math.h>
#include <stdlib.h>

#include <unistd.h>

#include <string.h>

#include <X11/Intrinsic.h>

#include "qso.h"

#include "skydefs.h"

#include "pos.h"
#include "catalog.h"
#include "draw.h"
#include "catreader.h"

struct qso_node {
  double ra_rad;
  double dec_rad;
  long filepos;
  shortint desig_ra;
  shortint desig_dec;
};

/* static function prototypes */
static FILE *read_qso PROTOTYPE((void));
static void calculate_qso_field PROTOTYPE((struct display *,
					                struct cat_header *));
static boolean find_qso_obj PROTOTYPE((char *));
static void get_qso_pos PROTOTYPE((int,double *,double *));
static void draw_qso_obj PROTOTYPE((Widget,int,int,int,int,int,int,int));
static boolean ps_draw_qso_obj PROTOTYPE((struct display *,int,int,int));
static boolean display_qso PROTOTYPE((FILE *,int));
static void format_qso_name PROTOTYPE((int,char *,IDfont *));
static void fill_qso_entry PROTOTYPE((char *,int));

/* external function prototypes */
extern char *build_filespec PROTOTYPE((char *,char *));
extern boolean read_binary_database PROTOTYPE((char *,void *,int *));
extern void init_precess PROTOTYPE((double,double));
extern boolean write_binary_database PROTOTYPE((char *,void *,size_t,size_t));
extern double ra_pos_to_rad PROTOTYPE((struct ra_pos));
extern double dec_pos_to_rad PROTOTYPE((struct dec_pos));
extern void precess PROTOTYPE((double,double,double *,double *));
extern void add_to_display_list PROTOTYPE((struct display *,
					           struct cat_header *,
				                          int,double,double));
extern void draw_object PROTOTYPE((Widget,int,int,int,ObjShape));
extern boolean ps_draw_object PROTOTYPE((struct display *,double,double,
					                    double,ObjShape));
extern void format_qso_display PROTOTYPE((char *));
extern void update_display_pos PROTOTYPE((double,double));

static struct qso_node qso_array[MAX_QSO];

static int qso_array_max;

#define QSO_DATA_FILE   "qso.dat"
#define QSO_BINARY_FILE "qso.bin"

/* QSO record buffer */
static char qso_rec[QSO_RECLEN + 1 + 1];



void init_qso(catalog)

struct cat_header *catalog;

{
  catalog->read_data = read_qso;
  catalog->calculate_field = calculate_qso_field;
  catalog->mag_limit_check = NULLBOOLFUNCTION;
  catalog->find_object = find_qso_obj;
  catalog->get_object_pos = get_qso_pos;
  catalog->draw_object = draw_qso_obj;
  catalog->ps_draw_object = ps_draw_qso_obj;
  catalog->display_data = display_qso;
  catalog->format_name = format_qso_name;
  catalog->menu_name = QSO_MENU_NAME;
  catalog->catalog_name = QSO_CATALOG_NAME;

  return;
}



/* read in the QSO catalog file, extracting needed info */

static FILE *read_qso()

{
  char *filebuff;
  FILE *qso_fd;
  char *bin_filebuff;
  int qso_entry;

/* get the path to the QSO data */
  filebuff = build_filespec("QSO_PATH",QSO_DATA_FILE);

/* open the QSO data file */
  if ((qso_fd = fopen(filebuff,"r")) == (FILE *)NULL) {
    printf("sky:  can't open QSO data file %s\n",filebuff);
    perror("sky");
    exit(1);
  }

/* get the path to the QSO binary file */
  bin_filebuff = build_filespec("QSO_BIN_PATH",QSO_BINARY_FILE);

/* see if we can read the binary file */
  if (access(bin_filebuff,R_OK) == 0)
    /* we can; try to use the pre-built database */
    if (read_binary_database(bin_filebuff,qso_array,&qso_array_max)) {
      /* success; free the filename buffers */
      free((void *)filebuff);
      free((void *)bin_filebuff);

      /* return the fd of the open data file */
      return(qso_fd);
    }

/* prior to reading the QSO catalog, build the rotation matrix for precessing
 * the object coordinates to the common equinox from the given epoch */
  if (QSO_EPOCH != COMMON_EQUINOX)
    init_precess(QSO_EPOCH,COMMON_EQUINOX);

/* read the QSO catalog, saving the information necessary for the display */
  qso_entry = 0;
  while (TRUE) {
    qso_array[qso_entry].filepos = ftell(qso_fd);
    if (fgets(qso_rec,sizeof(qso_rec),qso_fd) == (char *)NULL)
      break;
    fill_qso_entry(qso_rec,qso_entry++);
  }

/* leave the file open for later use */
/*  (void)fclose(qso_fd);*/

/* remember the last qso_array entry we filled */
  qso_array_max = qso_entry - 1;

/* attempt to save the binary database */
  if (access(bin_filebuff,W_OK) == 0)
    /* it's writeable, so let's try */
    if (! write_binary_database(bin_filebuff,(void *)qso_array,
				            (size_t)sizeof(struct qso_node),
				                           (size_t)qso_entry))
      /* dump of binary database failed; be sure the file is gone */
      (void)unlink(bin_filebuff);

/* free the filename buffers */
  free((void *)filebuff);
  free((void *)bin_filebuff);

  return(qso_fd);
}



/* given an object's index, return its right ascension and declination */

static void get_qso_pos(array_pos,ra_rad,dec_rad)

int array_pos;
double *ra_rad, *dec_rad;

{
  *ra_rad = qso_array[array_pos].ra_rad;
  *dec_rad = qso_array[array_pos].dec_rad;

  return;
}



/* given a QSO record, make a data node for it */

static void fill_qso_entry(record,qso_entry)

char *record;
int qso_entry;

{
  int reclen;
  char field[11];
  struct ra_pos ra_pos;
  struct dec_pos dec_pos;

/* get the length of the record for GET_FIELD */
  reclen = strlen(record) - 1;

/* get the QSO number */
  GET_FIELD(field,record,CAT_START,4);
  qso_array[qso_entry].desig_ra = atoi(field);
  GET_FIELD(field,record,CAT_START + 5,3);
  qso_array[qso_entry].desig_dec = atoi(field);

/* get the RA in pieces */
  GET_FIELD(field,record,POS_RA_START,2);
  ra_pos.hours = (smallint)atoi(field);

  GET_FIELD(field,record,POS_RA_START + 2,2);
  ra_pos.minutes = (smallint)atoi(field);

  GET_FIELD(field,record,POS_RA_START + 4,2);
  ra_pos.seconds = (smallint)atoi(field);

  GET_FIELD(field,record,POS_RA_START + 7,3);
  ra_pos.thousandths = (shortint)atoi(field) * 10;

/* similarly, get the declination in pieces */
  dec_pos.sign = (short)1;
  if (record[POS_DEC_START] == '-')
    dec_pos.sign = (short)-1;

  GET_FIELD(field,record,POS_DEC_START + 1,2);
  dec_pos.degrees = (smallint)atoi(field);

  GET_FIELD(field,record,POS_DEC_START + 3,2);
  dec_pos.minutes = (smallint)atoi(field);

  GET_FIELD(field,record,POS_DEC_START + 5,2);
  dec_pos.seconds = (smallint)atoi(field);

  GET_FIELD(field,record,POS_DEC_START + 8,2);
  dec_pos.hundredths = (smallint)atoi(field);

/* compute the RA and declination in radians */
  qso_array[qso_entry].ra_rad = ra_pos_to_rad(ra_pos);
  qso_array[qso_entry].dec_rad = dec_pos_to_rad(dec_pos);

/* precess them to the common equinox */
  if (QSO_EPOCH != COMMON_EQUINOX)
    precess(qso_array[qso_entry].ra_rad,qso_array[qso_entry].dec_rad,
	          &qso_array[qso_entry].ra_rad,&qso_array[qso_entry].dec_rad);

  return;
}



/* construct a list of those objects which appear in the current display */

static void calculate_qso_field(display,catalog)

struct display *display;
struct cat_header *catalog;

{
  int i;

/* loop through all QSO entries, checking for visibility */
  for (i = 0; i <= qso_array_max; i++)
    /* check this object and add it to the list if visible */
    add_to_display_list(display,catalog,i,qso_array[i].ra_rad,
                                                       qso_array[i].dec_rad);

#ifdef DEBUG
  printf("calculate_qso_field returning\n");
#endif
  return;
}



/* draw a QSO object on the sky */

/* ARGSUSED */

static void draw_qso_obj(sky_widget,array_pos,obj_x,obj_y,ul_x,ul_y,
		                                                 width,height)

Widget sky_widget;
int array_pos;
int obj_x, obj_y;
int ul_x, ul_y;
int width, height;

{
 /* draw the object even if it is only partially in the box */
  if (((obj_x + (OBJSIZE + 1) / 2) >= ul_x) &&
                            ((obj_x - (OBJSIZE + 1) / 2) <= (ul_x + width)) &&
                            ((obj_y + (OBJSIZE + 1) / 2) >= ul_y) &&
                            ((obj_y - (OBJSIZE + 1) / 2) <= (ul_y + height)))
    ;
  else
    return;

  /* for the moment, draw a circle for anything */
  draw_object(sky_widget,obj_x,obj_y,OBJSIZE,CIRCLE);

  return;
}



/* draw one QSO object on the PostScript display */

/* ARGSUSED */

static boolean ps_draw_qso_obj(display,array_pos,x,y)

struct display *display;
int array_pos;
int x, y;

{
  return(ps_draw_object(display,(double)x,(double)y,(double)OBJSIZE,CIRCLE));
}



/* display the catalog data for the given object */

static boolean display_qso(qso_fd,qso_array_pos)

FILE *qso_fd;
int qso_array_pos;

{
  int i;

/* get info from the QSO catalog */
  if (fseek(qso_fd,qso_array[qso_array_pos].filepos,SEEK_SET) == -1)
    return(FALSE);
  if (fgets(qso_rec,sizeof(qso_rec),qso_fd) == (char *)NULL)
    return(FALSE);

/* fill out the (possibly short) record with blanks */
  for (i = strlen(qso_rec) - 1; i < QSO_RECLEN; i++)
    qso_rec[i] = ' ';

/* format the record into an informational display */
  format_qso_display(qso_rec);

  return(TRUE);
}



/* find the QSO object in the buffer and update the display coordinates */

static boolean find_qso_obj(buffer)

char *buffer;

{
  int ra_qso_no, dec_qso_no;
  int i;

/* be sure this is a QSO designation */
  if ((strncmp(buffer,"QSO",3) == EQUAL) || strncmp(buffer,"qso",3) == EQUAL)
    buffer = &buffer[3];
  else
    return(FALSE);

/* get the right ascension portion of the QSO number */
  ra_qso_no = atoi(buffer);

/* advance to the separating dash */
  while (*buffer++ != '-')
    if (*buffer == '\0')
      return(FALSE);

/* get the declination portion of the QSO number */
  dec_qso_no = atoi(buffer);

/* loop through the array looking for this number */
  for (i = 0; i <= qso_array_max; i++)
    if ((qso_array[i].desig_ra == ra_qso_no) &&
	                               (qso_array[i].desig_dec == dec_qso_no))
      break;

/* return failure if we didn't find it */
  if (i > qso_array_max)
    return(FALSE);
  else {
    /* update the display coordinates */
    update_display_pos(qso_array[i].ra_rad,qso_array[i].dec_rad);
    return(TRUE);
  }

/* NOTREACHED */
}



/* format a nice name for the object */

/* ARGSUSED */

static void format_qso_name(qso_pos,buffer,font)

int qso_pos;
char *buffer;
IDfont *font;

{
/* use the QSO number */
  sprintf(buffer,"QSO %.04d-%.03d",qso_array[qso_pos].desig_ra,
	                                        qso_array[qso_pos].desig_dec);

  return;
}
