/*
 * xsky - an interactive sky atlas
 *
 * Copyright 1992-6, Terry R. Friedrichsen
 *
 * This program may be copied and redistributed, in whole or in part,
 * as long as you don't try to make any money from the sale or redis-
 * tribution of the program or any part of the program, or pretend
 * that you wrote the program or any of its parts unless specifically
 * credited by the original author.
 *
 * You are free to make use of this software in your own programs, as
 * long as you credit the original author where it is due.
 */

/*
 * WARRANTY:
 * xsky was written as a learning project and as a demonstration of
 * X Window System programming.  xsky doesn't do anything; it is not
 * merchantable, and it is not fit for any purpose whatsoever.  In
 * fact, don't use xsky at all; it's free, and you're getting what
 * you paid for.
 */

#include <stdio.h>

#include <string.h>

#include <X11/Intrinsic.h>

#include "skydefs.h"
#include "catalog.h"
#include "draw.h"
#include "pos.h"

/* static function prototypes */
static boolean ps_write_file_work PROTOTYPE((Widget,struct display *));
static boolean ps_setup PROTOTYPE((void));
static boolean ps_begin_header PROTOTYPE((void));
static boolean ps_put_header PROTOTYPE((char *));
static boolean ps_end_header PROTOTYPE((void));
static boolean ps_begin_drawing PROTOTYPE((int,int));
static boolean ps_draw_id PROTOTYPE((struct id_node *,int));
static boolean ps_end_drawing PROTOTYPE((void));

/* external function prototypes */
extern void draw_grid_lines PROTOTYPE((Widget,struct display *,
				                            int,int,int,int));
extern void draw_boundary_lines PROTOTYPE((Widget,struct display *,
					                    int,int,int,int));

/* global function prototypes */
boolean ps_put PROTOTYPE((char *));
void ps_newpath PROTOTYPE((void));
void ps_endpath PROTOTYPE((void));

/* list of catalog header structures */
extern struct cat_header *cat_list_head;

/* lowest magnitude to display */
extern float mag_limit;

/* right ascension and declination line intervals (RA interval is in
 * thousandths of a second of time, dec interval is in hundredths of
 * a second of arc) */
extern int ra_line_interval;
extern int dec_line_interval;

/* flag for new path for grid line */
static boolean ps_new_path;

/* buffer for generated PostScript lines */
static char ps_buffer[200];

/* count of the number of header lines generated */
static int headerlines;

/* flag for display of object IDs */
extern Boolean draw_ids;

/* list header for list of visual identifications */
extern struct id_node *id_list;

/* flag for display of grid lines */
extern Boolean draw_grid;

/* flag for display of constellation boundaries */
extern Boolean draw_bound;

/* PostScript output file */
static FILE *psfile;



/* handle writing the PostScript output file */

boolean ps_write_file(sky_widget,display,file)

Widget sky_widget;
struct display *display;
char *file;

{
  boolean popen_flag;
  boolean retval;

/* if the first character is '!', open a pipe to a command */
  if (file[0] == '!')
    if ((psfile = popen(&file[1],"w")) == (FILE *)NULL)
      return(FALSE);
    else
      /* flag that we've done a popen() */
      popen_flag = TRUE;
  else
/* otherwise, open an output file */
    if ((psfile = fopen(file,"w")) == (FILE *)NULL)
      return(FALSE);
    else
      /* flag that we have not done a popen() */
      popen_flag = FALSE;

/* write out the PostScript commands */
  retval = ps_write_file_work(sky_widget,display);

/* close the output */
  if (popen_flag)
    (void)pclose(psfile);
  else
    (void)fclose(psfile);

  return(retval);
}



/* write PostScript commands to draw the sky display image */

static boolean ps_write_file_work(sky_widget,display)

Widget sky_widget;
struct display *display;

{
  struct cat_header *catalog;
  char local_pos_buffer[50];
  struct obj_node *obj;
  struct id_node *id;

/* put out the PostScript function definitions */
  if (! ps_setup())
    return(FALSE);

/* start out the PostScript chart header */
  if (! ps_begin_header())
    return(FALSE);

/* put out the header lines */
  catalog = cat_list_head;
  while (catalog != (struct cat_header *)NULL) {
    if (catalog->show_flag) {
      sprintf(ps_buffer,"Catalog:  %s",catalog->catalog_name);
      if (! ps_put_header(ps_buffer))
	return(FALSE);
    }

    /* step to the next catalog */
    catalog = catalog->next_cat;
  }

/* output other relevant chart values */
  if (! ps_put_header("Equinox:  2000.0"))
    return(FALSE);

  sprintf(ps_buffer,"Limiting magnitude:  %.2f",mag_limit);
  if (! ps_put_header(ps_buffer))
    return(FALSE);

/* build the right ascension position in the buffer */
  sprintf(local_pos_buffer,"%2dh %02dm %02ds",display->ra_pos.hours,
	                                            display->ra_pos.minutes,
	                                            display->ra_pos.seconds);

/* add it to the PostScript buffer */
  sprintf(ps_buffer,"Chart center:  RA = %s, ",local_pos_buffer);

/* build the declination position in the buffer */
  if (display->dec_pos.sign == (short)-1)
    strcpy(local_pos_buffer,"-");
  else
    local_pos_buffer[0] = '\0';
  sprintf(&local_pos_buffer[strlen(local_pos_buffer)],"%2d~ %02d' %02d\"",
	                                            display->dec_pos.degrees,
	                                            display->dec_pos.minutes,
	                                            display->dec_pos.seconds);

/* and add it to the PostScript buffer */
  sprintf(&ps_buffer[strlen(ps_buffer)]," dec = %s",local_pos_buffer);

  /* output the chart position line */
  if (! ps_put_header(ps_buffer))
    return(FALSE);

/* if grid lines are being drawn, give the grid line separations */
  if (draw_grid) {
    if (ra_line_interval < 1000)
      sprintf(ps_buffer,"Right ascension lines are 0.%.03ds apart, ",
	                                                    ra_line_interval);
    else if (ra_line_interval < (60 * 1000))
      sprintf(ps_buffer,"Right ascension lines are %.3gs apart, ",
	                                      (float)ra_line_interval / 1000);
    else if (ra_line_interval < (60 * 60 * 1000))
      sprintf(ps_buffer,"Right ascension lines are %.3gm apart, ",
	                               (float)ra_line_interval / (60 * 1000));
    else
      sprintf(ps_buffer,"Right ascension lines are %.3gh apart, ",
	                          (float)ra_line_interval / (60 * 60 * 1000));

    if (dec_line_interval < 100)
      sprintf(&ps_buffer[strlen(ps_buffer)],
	                             "declination lines are 0.%.03d\" apart\n",
	                                                   dec_line_interval);
    else if (dec_line_interval < (60 * 100))
      sprintf(&ps_buffer[strlen(ps_buffer)],
	                               "declination lines are %.3g\" apart\n",
	                                      (float)dec_line_interval / 100);
    else if (dec_line_interval < (60 * 60 * 100))
      sprintf(&ps_buffer[strlen(ps_buffer)],
	                               "declination lines are %.3g' apart\n",
	                               (float)dec_line_interval / (60 * 100));
    else
      sprintf(&ps_buffer[strlen(ps_buffer)],
	                          "declination lines are %.3g~ apart\n",
	                          (float)dec_line_interval / (60 * 60 * 100));

    if (! ps_put_header(ps_buffer))
      return(FALSE);
  }

/* signal the end of the header information */
  if (! ps_end_header())
    return(FALSE);

/* set up to begin drawing the chart */
  if (! ps_begin_drawing(display->width,display->height))
    return(FALSE);

/* draw objects from all catalogs */
  catalog = cat_list_head;
  while (catalog != (struct cat_header *)NULL) {

/* if the catalog is active, draw all the objects in its display list */
    if (catalog->show_flag) {
      obj = catalog->obj_list_head;
      while (obj != (struct obj_node *)NULL) {

/* construct and emit the PostScript commands to draw the object */
	if (! (*catalog->ps_draw_object)(display,obj->array_pos,
					             (int)obj->x,(int)obj->y))
	  return(FALSE);

/* step to the next object */
	obj = obj->next;
      }
    }

/* step to the next catalog */
    catalog = catalog->next_cat;
  }

/* output the identification strings */
  if (draw_ids) {
    id = id_list;
    while (id != (struct id_node *)NULL) {
      if (! ps_draw_id(id,(Dimension)display->height))
	return(FALSE);
      id = id->next;
    }
  }

/* draw the grid lines and consellation boundaries over the top of it all */
  if (draw_grid)
    draw_grid_lines(sky_widget,display,0,0,display->width,display->height);
  if (draw_bound)
    draw_boundary_lines(sky_widget,display,0,0,display->width,display->height);

/* end the drawing and return the final success or failure */
  return(ps_end_drawing());
}



/* output the PostScript function definitions */

static boolean ps_setup()

{
/* start out the PostScript file in a standard way */
  if (! ps_put("%!PS-Adobe-2.0\n"))
    return(FALSE);

/* define a function to draw an ellipse */
  if (! ps_put("/ellipse {\n"))
    return(FALSE);
  if (! ps_put("    factor div 2 div\n"))
    return(FALSE);
  if (! ps_put("    gsave\n"))
    return(FALSE);
  if (! ps_put("    3 1 roll 2 mul 3 -1 roll 1 .5 scale\n"))
    return(FALSE);
  if (! ps_put("    newpath 0 360 arc closepath\n"))
    return(FALSE);
  if (! ps_put("    .3 setlinewidth 0 setgray stroke\n"))
    return(FALSE);
  if (! ps_put("    grestore\n"))
    return(FALSE);
  if (! ps_put("} bind def\n"))
    return(FALSE);

/* define a function to draw a circle */
  if (! ps_put("/circle {\n"))
    return(FALSE);
  if (! ps_put("    factor div 2 div\n"))
    return(FALSE);
  if (! ps_put("    newpath 0 360 arc closepath\n"))
    return(FALSE);
  if (! ps_put("    gsave .3 setlinewidth 0 setgray stroke grestore\n"))
    return(FALSE);
  if (! ps_put("} bind def\n"))
    return(FALSE);

/* define a function to draw a box for object location */
  if (! ps_put("/box {\n"))
    return(FALSE);
  if (! ps_put("    factor div\n"))
    return(FALSE);
  if (! ps_put("    3 1 roll\n"))
    return(FALSE);
  if (! ps_put("    newpath moveto\n"))
    return(FALSE);
  if (! ps_put("    dup neg 0 exch rlineto\n"))
    return(FALSE);
  if (! ps_put("    dup 0 rlineto\n"))
    return(FALSE);
  if (! ps_put("    dup 0 exch rlineto\n"))
    return(FALSE);
  if (! ps_put("    dup neg 0 rlineto\n"))
    return(FALSE);
  if (! ps_put("    closepath\n"))
    return(FALSE);
  if (! ps_put("    gsave .3 setlinewidth 0 setgray stroke grestore\n"))
    return(FALSE);
  if (! ps_put("} bind def\n"))
    return(FALSE);

/* define a function to draw a filled-in circle for a star */
  if (! ps_put("/star {\n"))
    return(FALSE);
  if (! ps_put("    factor div 2 div\n"))
    return(FALSE);
  if (! ps_put("    newpath 0 360 arc closepath\n"))
    return(FALSE);
  if (! ps_put("    gsave .3 setlinewidth 1 setgray stroke grestore\n"))
    return(FALSE);
  if (! ps_put("    0 setgray fill\n"))
    return(FALSE);
  if (! ps_put("} bind def\n"))
    return(FALSE);

/* define a function to emit a character at an object position */
  if (! ps_put("/put_obj_char {\n"))
    return(FALSE);
  if (! ps_put("    gsave\n"))
    return(FALSE);
  if (! ps_put("    LatinID setfont\n"))
    return(FALSE);
  if (! ps_put("    dup stringwidth pop 2 div\n"))
    return(FALSE);
  if (! ps_put("    4 -1 roll exch sub\n"))
    return(FALSE);
  /* adjusting the y value by 1/6 the font scale is purely empirical */
  if (! ps_put("    3 -1 roll Latinfontscale 6 div sub moveto\n"))
    return(FALSE);
  if (! ps_put("    1 factor div dup scale\n"))
    return(FALSE);
  if (! ps_put("    show\n"))
    return(FALSE);
  if (! ps_put("    grestore\n"))
    return(FALSE);
  if (! ps_put("} bind def\n"))
    return(FALSE);

#if 0
/* define a function to emit object ID text */
  if (! ps_put("/put_id {\n"))
    return(FALSE);
  if (! ps_put("    gsave\n"))
    return(FALSE);
  if (! ps_put("    dup stringwidth pop 3 -1 roll exch div\n"))
    return(FALSE);
  if (! ps_put("    4 -2 roll moveto\n"))
    return(FALSE);
  if (! ps_put("    1 factor div scale\n"))
    return(FALSE);
  if (! ps_put("    show\n"))
    return(FALSE);
  if (! ps_put("    grestore\n"))
    return(FALSE);
  if (! ps_put("} bind def\n"))
    return(FALSE);
#else
/* define a function to emit object ID text */
  if (! ps_put("/put_corr_id {\n"))
    return(FALSE);
  if (! ps_put("    6 1 roll\n"))
    return(FALSE);
  if (! ps_put("    setfont\n"))
    return(FALSE);
  if (! ps_put("    dup factor .8 mul div sub sub\n"))
    return(FALSE);
  if (! ps_put("    4 1 roll dup factor div sub add\n"))
    return(FALSE);
  if (! ps_put("    3 -1 roll moveto\n"))
    return(FALSE);
  if (! ps_put("    show\n"))
    return(FALSE);
  if (! ps_put("} bind def\n"))
    return(FALSE);
#endif

  return(TRUE);
}



/* set up to begin chart header */

static boolean ps_begin_header()

{
/* capture the boundaries of the printable area */
  if (! ps_put("gsave initclip clippath pathbbox grestore\n"))
    return(FALSE);
  if (! ps_put("/ur_y exch def /ur_x exch def\n"))
    return(FALSE);
  if (! ps_put("/ll_y exch def /ll_x exch def\n"))
    return(FALSE);

/* set up the Symbol font for Greek letters */
  if (! ps_put("/Times-Roman findfont 18 scalefont setfont\n"))
    return(FALSE);

/* draw the top line of the header box */
  if (! ps_put("newpath ll_x 1 add ur_y 2 sub moveto\n"))
    return(FALSE);
  if (! ps_put("ur_x ll_x sub 2 sub 0 rlineto\n"))
    return(FALSE);
  if (! ps_put("3 setlinewidth stroke\n"))
    return(FALSE);

/* set up the y position for the first header line */
  if (! ps_put("/curr_y ur_y 22 sub def\n"))
    return(FALSE);

/* initialize the header line count */
  headerlines = 0;

  return(TRUE);
}



/* put out lines of header information */

static boolean ps_put_header(line)

char *line;

{
  char *ptr1, *ptr2;
  char buffer[200];

/* draw the left-hand box segment */
  if (! ps_put("newpath ll_x 2 add curr_y 8 sub moveto\n"))
    return(FALSE);
  if (! ps_put("0 28 rlineto 3 setlinewidth stroke\n"))
    return(FALSE);

/* draw the right-hand box segment */
  if (! ps_put("newpath ur_x 2 sub curr_y 8 sub moveto\n"))
    return(FALSE);
  if (! ps_put("0 28 rlineto 3 setlinewidth stroke\n"))
    return(FALSE);

/* move to the text position */
  if (! ps_put("ll_x 10 add curr_y moveto\n"))
    return(FALSE);

/* set up source and destination pointers */
  ptr1 = line;
  ptr2 = buffer;

/* put the caller's line in the buffer, escaping parens and backslashes */
  *ptr2++ = '(';
  do {
    if ((*ptr1 == '(') || (*ptr1 == ')') || (*ptr1 == '\\'))
      *ptr2++ = '\\';
    /* replace twiddles with a degree symbol */
    if (*ptr1 == '~') {
      /* end the current line here */
      strcpy(ptr2,") show\n");
      if (! ps_put(buffer))
	return(FALSE);
      /* put out a degree symbol */
      if (! ps_put("0 7 rmoveto (o) show 0 -7 rmoveto\n"))
	return(FALSE);
      /* restart the line */
      ptr2 = buffer;
      *ptr2++ = '(';
      /* step past the twiddle */
      ptr1++;
      /* check for escapeable characters following the twiddle */
      if ((*ptr1 == '(') || (*ptr1 == ')') || (*ptr1 == '\\'))
	*ptr2++ = '\\';
    }
  } while ((*ptr2++ = *ptr1++) != '\0');
  
/* end the line properly */
  strcat(buffer,") show\n");

/* and put it out */
  if (! ps_put(buffer))
    return(FALSE);

/* move to the next line position */
  if (! ps_put("/curr_y curr_y 20 sub def\n"))
    return(FALSE);

/* count another line of header information */
  headerlines++;

  return(TRUE);
}



/* end the header information */

static boolean ps_end_header()

{
/* draw the bottom line of the header box */
  if (! ps_put("newpath ll_x 1 add curr_y 12 add moveto\n"))
    return(FALSE);
  if (! ps_put("ur_x ll_x sub 2 sub 0 rlineto\n"))
    return(FALSE);
  if (! ps_put("3 setlinewidth stroke\n"))
    return(FALSE);

  return(TRUE);
}



/* set up to begin drawing commands */

static boolean ps_begin_drawing(width,height)

int width, height;

{
/* set up the Times-Roman font for Latin ID designations */
  if (! ps_put("/LatinIDfontscale 10 def\n"))
    return(FALSE);
  if (! ps_put("/LatinID "))
    return(FALSE);
  if (! ps_put("/Times-Roman findfont LatinIDfontscale scalefont "))
    return(FALSE);
  if (! ps_put("def\n"))
    return(FALSE);

/* set up the Times-Roman font for Latin constellation designations */
  if (! ps_put("/Latinconfontscale 14 def\n"))
    return(FALSE);
  if (! ps_put("/LatinCon "))
    return(FALSE);
  if (! ps_put("/Helvetica findfont Latinconfontscale scalefont "))
    return(FALSE);
  if (! ps_put("def\n"))
    return(FALSE);

/* set up the Symbol font for Greek letters */
  if (! ps_put("/Greekfontscale 12 def\n"))
    return(FALSE);
  if (! ps_put("/Greek /Symbol findfont Greekfontscale scalefont def\n"))
    return(FALSE);

/* set a narrow line width and rounded line join style for grid lines */
  if (! ps_put("0 setlinewidth 1 setlinejoin\n"))
    return(FALSE);

/* set a clip region for the area of the chart */
  if (! ps_put("newpath 0 0 moveto\n"))
    return(FALSE);
  if (! ps_put("ur_x ll_x sub 0 rlineto\n"))
    return(FALSE);
  if (! ps_put("0 ur_y ll_y sub rlineto\n"))
    return(FALSE);
  if (! ps_put("ur_x ll_x sub neg 0 rlineto\n"))
    return(FALSE);
  if (! ps_put("closepath clip newpath\n"))
    return(FALSE);

/* compute the correct scale factor */
  sprintf(ps_buffer,"/factor_x ur_x ll_x sub %d div def\n",width);
  if (! ps_put(ps_buffer))
    return(FALSE);
  sprintf(ps_buffer,"/factor_y ur_y ll_y sub %d sub %d div def\n",
	                                        headerlines * 20 + 15,height);
  if (! ps_put(ps_buffer))
    return(FALSE);
  if (! ps_put("/factor factor_x factor_y lt\n"))
    return(FALSE);
  if (! ps_put("    { factor_x } { factor_y } ifelse def\n"))
    return(FALSE);

/* translate coordinates to the imageable lower left */
  if (! ps_put("ll_x ll_y translate\n"))
    return(FALSE);

/* scale the drawing to the correct size */
  if (! ps_put("factor factor scale\n"))
    return(FALSE);

  return(TRUE);
}



/* end a PostScript drawing */

static boolean ps_end_drawing()

{
  return(ps_put("showpage\n\004"));
}



/* draw an object ID for PostScript */

static boolean ps_draw_id(id,height)

struct id_node *id;
int height;

{
#if 0
  float x, y;
#endif
  float x_corr, y_corr;
  char *font;

#ifdef DEBUG
  if (id->obj_catalog == (struct cat_header *)NULL)
    /* this is a user-supplied label; not associated with an object */
    printf("object is %s\n",id->id_buffer);
  else
    printf("object is %s at %d,%d\n",id->id_buffer,id->obj_x,id->obj_y);
  printf("ID text is at %d,%d\n",id->x,id->y);
  printf("ID upper left is at %d,%d\n",id->ul_x,id->ul_y);
  printf("ID lower right is at %d,%d\n",id->lr_x,id->lr_y);
  printf("therefore text length = %d\n",id->lr_x - id->ul_x);
#endif

/* if the ID is not associated with an object, do not correct its position */
  if (id->obj_catalog == (struct cat_header *)NULL)
    x_corr = y_corr = 0;
  else {
/* otherwise, correct ID position to keep near the object after scale change */

#if 0
/* calculate where to place the text */
    if (id->obj_x < id->x)
      /* apply this correction if the ID is to the right of the object */
      x = id->x - ((id->ul_x - id->obj_x) - (id->ul_x - id->obj_x) / factor);
    else if (id->obj_x > id->x)
      /* apply this correction if the ID is to the left of the object */
      x = id->x + ((id->obj_x - id->lr_x) - (id->obj_x - id->lr_x) / factor);
    else
      /* ID coincides with object horizontally; don't correct position */
      x = id->x;
#endif

/* calculate correction to text location */
    if (id->obj_x < id->x)
      /* apply this correction if the ID is to the right of the object */
      x_corr = -(id->ul_x - id->obj_x);
/*    x_corr = -((id->ul_x - id->obj_x) - (id->ul_x - id->obj_x));*/
    else if (id->obj_x > id->x)
      /* apply this correction if the ID is to the left of the object */
      x_corr = id->obj_x - id->lr_x;
/*    x_corr = ((id->obj_x - id->lr_x) - (id->obj_x - id->lr_x));*/
    else
      /* ID coincides with object horizontally; don't correct position */
      x_corr = 0;

#if 0
/* correct the y coordinate in the same manner */
    if (id->obj_y < id->y)
      /* apply this correction if the ID is below the object */
      y = id->y - ((id->y - id->obj_y) - (id->y - id->obj_y) / factor);
    else if (id->obj_y > id->y)
      /* apply this correction if the ID is above the object */
      y = id->y + ((id->obj_y - id->y) - (id->obj_y - id->y) / factor);
    else
      /* ID coincides with object vertically; don't correct position */
      y = id->y;
#endif

/* calculate the y coordinate correction in the same manner */
    if (id->obj_y < id->y)
      /* apply this correction if the ID is below the object */
      y_corr = -(id->y - id->obj_y);
/*    y_corr = -((id->y - id->obj_y) - (id->y - id->obj_y));*/
    else if (id->obj_y > id->y)
      /* apply this correction if the ID is above the object */
      y_corr = id->obj_y - id->y;
/*    y_corr = ((id->obj_y - id->y) - (id->obj_y - id->y));*/
    else
      /* ID coincides with object vertically; don't correct position */
      y_corr = 0;

#if 0
#ifdef DEBUG
    printf("corrected x value is %f\n",x);
    printf("corrected y value is %f\n\n",y);
#endif
#else
#ifdef DEBUG
    printf("x correction is %f\n",x_corr);
    printf("y correction is %f\n\n",y_corr);
#endif
#endif
  }

/* set up a recognizeable font string */
  if (id->font == LATIN_ID)
    font = LATIN_ID_FONT;
  else if (id->font == LATIN_CON)
    font = LATIN_CON_FONT;
  else if (id->font == GREEK)
    font = GREEK_ID_FONT;
  else
    /* default to Latin ID font */
    font = LATIN_ID_FONT;

/* build the call to the ID display routine and write it out */
#if 0
  sprintf(ps_buffer,"%f %f %d (%s) put_id\n",x,height - y,
	                                   id->lr_x - id->ul_x,id->id_buffer);
#else
  sprintf(ps_buffer,"%f %f %f %f %s (%s) put_corr_id\n",(float)id->x,x_corr,
	                                   (float)height - id->y,y_corr,
	                                                  font,id->id_buffer);
#endif
  if (! ps_put(ps_buffer))
    return(FALSE);

  return(TRUE);
}



/* start a new path for a line */

void ps_newpath()

{
/* set a flag and emit a newpath command */
  ps_new_path = TRUE;
  (void)ps_put("newpath\n");

  return;
}



/* end the path for a line */

void ps_endpath()

{
/* stroke the line just produced */
  (void)ps_put("stroke\n");

  return;
}



/* draw a star for PostScript */

boolean ps_draw_star(display,x,y,size)

struct display *display;
double x, y;
double size;

{
/* generate the line of PostScript to call the filled-circle routine */
  sprintf(ps_buffer,"%.3f %.3f %.3f star\n",x,display->height - y,size);

/* return the result of outputting the buffer */
  return(ps_put(ps_buffer));
}



/* draw a single object on the sky */

boolean ps_draw_object(display,x,y,size,shape)

struct display *display;
double x, y;
double size;
ObjShape shape;

{
/* draw an object figure depending on the given shape */
  switch (shape) {
  case ELLIPSE:
    sprintf(ps_buffer,"%.3f %.3f %.3f ellipse\n",x,
	                                        display->height - y,size + 2);
    break;
  case CIRCLE:
    sprintf(ps_buffer,"%.3f %.3f %.3f circle\n",x,
	                                        display->height - y,size + 1);
    break;
  case SQUARE:
    sprintf(ps_buffer,"%.3f %.3f %.3f box\n",x,display->height - y,size);
    break;
  case XMARK:
    sprintf(ps_buffer,"%.3f %.3f (%s) put_obj_char\n",x,
	                                             display->height - y,"x");
    break;
  default:
    break;
  }

/* return the result of outputting the buffer */
  return(ps_put(ps_buffer));
}



/* extend a line to a new point on the PostScript display */

void ps_draw_point(display,x,y)

struct display *display;
double x, y;

{
/* check for Southern Hemisphere */
  if (display->downunder_flag)
    /* if this is the start of a new path, we need a moveto */
    if (ps_new_path) {
      /* the vertical dimension is already flipped in PostScript */
      sprintf(ps_buffer,"%.3f %.3f moveto\n",display->width - x,y);
      ps_new_path = FALSE;
    }
    else
      /* otherwise, just extend the line to the new point */
      sprintf(ps_buffer,"%.3f %.3f lineto\n",display->width - x,y);
  else
    /* if this is the start of a new path, we need a moveto */
    if (ps_new_path) {
      /* we need to unflip PostScript's flipped vertical dimension */
      sprintf(ps_buffer,"%.3f %.3f moveto\n",x,display->height - y);
      ps_new_path = FALSE;
    }
    else
      /* otherwise, just extend the line to the new point */
      sprintf(ps_buffer,"%.3f %.3f lineto\n",x,display->height - y);

  (void)ps_put(ps_buffer);

  return;
}



/* output a string to the PostScript file */

boolean ps_put(string)

char *string;

{
  if (fputs(string,psfile) == EOF) {
    (void)fclose(psfile);
    return(FALSE);
  }
  else
    return(TRUE);

/* NOTREACHED */
}
