/*
 * xsky - an interactive sky atlas
 *
 * Copyright 1992-6, Terry R. Friedrichsen
 *
 * This program may be copied and redistributed, in whole or in part,
 * as long as you don't try to make any money from the sale or redis-
 * tribution of the program or any part of the program, or pretend
 * that you wrote the program or any of its parts unless specifically
 * credited by the original author.
 *
 * You are free to make use of this software in your own programs, as
 * long as you credit the original author where it is due.
 */

/*
 * WARRANTY:
 * xsky was written as a learning project and as a demonstration of
 * X Window System programming.  xsky doesn't do anything; it is not
 * merchantable, and it is not fit for any purpose whatsoever.  In
 * fact, don't use xsky at all; it's free, and you're getting what
 * you paid for.
 */

#include <stdio.h>

#include <stdlib.h>
#include <ctype.h>
#include <string.h>

#include <math.h>
#include <limits.h>

#if 0
#ifdef sgi            /* Silicon Graphics Indigo, courtesy of Jim Sharpe */
#include <nan.h>
#endif
#endif

#include "skydefs.h"

#include "pos.h"

/* static function prototypes */
#ifndef MEEUS
static double pos_to_sep PROTOTYPE((double,double,double,double));
static double pos_to_dir PROTOTYPE((double,double,double,double));
static void sep_to_centric PROTOTYPE((double,double,double *,double *,double));
static void centric_to_X PROTOTYPE((struct display *,double,double,
				                           double *,double *));
#endif

/* global function prototypes */
struct ra_pos rad_to_ra PROTOTYPE((double));
struct dec_pos rad_to_dec PROTOTYPE((double));
#ifdef ROUND_DISPLAY_POS
void round_ra PROTOTYPE((struct ra_pos *));
void round_dec PROTOTYPE((struct dec_pos *));
#endif
double ra_to_rad PROTOTYPE((int,int,int,int));
double dec_to_rad PROTOTYPE((int,int,int,int,int));
void pos_rad_to_X_double PROTOTYPE((struct display *,double,double,
				                           double *,double *));
void pos_rad_to_X PROTOTYPE((struct display *,double,double,int *,int *));
void X_to_pos_rad PROTOTYPE((struct display *,int,int,double *,double *));
boolean within_limits PROTOTYPE((struct display *,double,double));
double dec_to_rad PROTOTYPE((int,int,int,int,int));
double ra_to_rad PROTOTYPE((int,int,int,int));
boolean get_ra PROTOTYPE((char *,struct ra_pos *));
boolean get_dec PROTOTYPE((char *,struct dec_pos *));



/* crack the position given by the text string and center the display on it */

boolean calculate_position(display,position)

struct display *display;
char *position;

{
  char *ptr;
  struct ra_pos new_ra;
  struct dec_pos new_dec;

/* crack the RA and dec strings */
  ptr = strtok(position,",");
  if (! get_ra(ptr,&new_ra))
    return(FALSE);
  ptr = strtok((char *)NULL,",");
  if (! get_dec(ptr,&new_dec))
    return(FALSE);

/* position looks OK; copy it to the display */
  display->ra_pos = new_ra;
  display->dec_pos = new_dec;

/* calculate the position in radians */
  display->center_ra_rad = ra_to_rad((int)display->ra_pos.hours,
				            (int)display->ra_pos.minutes,
				            (int)display->ra_pos.seconds,
				            (int)display->ra_pos.thousandths);

  display->center_dec_rad = dec_to_rad((int)display->dec_pos.sign,
				            (int)display->dec_pos.degrees,
				            (int)display->dec_pos.minutes,
				            (int)display->dec_pos.seconds,
				            (int)display->dec_pos.hundredths);

  return(TRUE);
}



/* update the display center from the position in radians */

void update_center_position(display,position)

struct display *display;
char *position;

{
/* convert the position in radians to sexigesimal and round to minutes */
  display->ra_pos = rad_to_ra(display->center_ra_rad);
  display->dec_pos = rad_to_dec(display->center_dec_rad);

#ifdef ROUND_DISPLAY_POS
/* round the display position for neatness */
  round_ra(&display->ra_pos);
  round_dec(&display->dec_pos);
#endif

/* convert back to radians */
  display->center_ra_rad = ra_to_rad((int)display->ra_pos.hours,
				            (int)display->ra_pos.minutes,
				            (int)display->ra_pos.seconds,
				            (int)display->ra_pos.thousandths);
  display->center_dec_rad = dec_to_rad((int)display->dec_pos.sign,
				            (int)display->dec_pos.degrees,
				            (int)display->dec_pos.minutes,
				            (int)display->dec_pos.seconds,
				            (int)display->dec_pos.hundredths);

/* update the editable display center position buffer */
  sprintf(position,"%02dh %02dm %02ds.%.03d, %c%02dd %02d' %02d\".%.02d",
	                                display->ra_pos.hours,
	                                display->ra_pos.minutes,
	                                display->ra_pos.seconds,
	                                display->ra_pos.thousandths,
	                                display->dec_pos.sign < 0 ? '-' : '+',
	                                display->dec_pos.degrees,
	                                display->dec_pos.minutes,
	                                display->dec_pos.seconds,
	                                display->dec_pos.hundredths);

  return;
}



/* calculate_limits() and within_limits() were written by Brian Wing as part
 * of his successful effort to greatly speed up xsky for large catalogs such
 * as SAO.  I (TRF) hacked on these slightly, so if they're broken, the fault
 * is mine.  Brian is from Canada, so I must apologize to him for "correcting"
 * his spelling of "center" :-). */


/* calculate approximate boundaries of the display */

void calculate_limits(display)

struct display *display;

{
  int pole_x, pole_y;
  boolean north_pole, south_pole;
  double center_top_dec, center_bottom_dec;
  double side_top_dec, side_bottom_dec;
  double dummy;

/* calculate the X coordinates of the north celestial pole */
  pos_rad_to_X(display,0.0,PI / 2,&pole_x,&pole_y);

  /* set a flag if the NCP is within the display */
  if ((pole_x >= 0) && (pole_x <= display->width) &&
                                 (pole_y >= 0) && (pole_y <= display->height))
    north_pole = TRUE;
  else
    north_pole = FALSE;

/* calculate the X coordinates of the south celestial pole */
  pos_rad_to_X(display,0.0,-PI / 2,&pole_x,&pole_y);
  /* set a flag if the SCP is within the display */
  if ((pole_x >= 0) && (pole_x <= display->width) &&
                                 (pole_y >= 0) && (pole_y <= display->height))
    south_pole = TRUE;
  else
    south_pole = FALSE;

/* set the RA and dec limits of the display */
   if (north_pole && south_pole) {
     /* both poles are within the display; this is easy */
     display->left_ra_rad = 2 * PI;
     display->right_ra_rad = 0.0;
     display->top_dec_rad = PI / 2;
     display->bottom_dec_rad = -PI / 2;
   }
   else if (north_pole) {
     /* north pole is within the display; at least half of it's easy */
     display->left_ra_rad = 2 * PI;
     display->right_ra_rad = 0.0;
     display->top_dec_rad = PI / 2;
     X_to_pos_rad(display,display->width / 2,display->height,&dummy,
		                                          &center_bottom_dec);
     X_to_pos_rad(display,0,display->height,&dummy,&side_bottom_dec);
     display->bottom_dec_rad = min(center_bottom_dec + PI / 2,
				           side_bottom_dec + PI / 2) - PI / 2;
   }
   else if (south_pole) {
     /* south pole is within the display; half of it's still easy */
     display->left_ra_rad = 2 * PI;
     display->right_ra_rad = 0.0;
     X_to_pos_rad(display,display->width / 2,0,&dummy,&center_top_dec);
     X_to_pos_rad(display,0,0,&dummy,&side_top_dec);
     display->top_dec_rad = max(center_top_dec + PI/2,
				              side_top_dec + PI / 2) - PI / 2;
     display->bottom_dec_rad = -PI / 2;
   }
   else {
     /* neither pole is within the display; it's all hard */
     display->left_ra_rad = display->right_ra_rad = display->center_ra_rad;

     if (display->center_dec_rad > 0) {
       /* northern hemisphere; calculate the RA of the upper corners */
       X_to_pos_rad(display,0,0,&display->left_ra_rad,&dummy);
       X_to_pos_rad(display,display->width,0,&display->right_ra_rad,&dummy);
     }
     else {
       /* southern hemisphere (or smack on equator); find lower corners */
       X_to_pos_rad(display,0,display->height,&display->left_ra_rad,&dummy);
       X_to_pos_rad(display,display->width,display->height,
		                             &display->right_ra_rad,&dummy);
     }

     /* calculate the positions of the top and bottom center of the display */
     X_to_pos_rad(display,display->width / 2,0,&dummy,&center_top_dec);
     X_to_pos_rad(display,display->width / 2,display->height,&dummy,
		                                          &center_bottom_dec);

     /* calculate the declination of the upper-left and lower-right corners */
     X_to_pos_rad(display,0,0,&dummy,&side_top_dec);
     X_to_pos_rad(display,0,display->height,&dummy,&side_bottom_dec);

     /* set the limits of the display in declination */
     display->top_dec_rad = max(center_top_dec + PI / 2,
				              side_top_dec + PI / 2) - PI / 2;
     display->bottom_dec_rad = min(center_bottom_dec + PI / 2,
				           side_bottom_dec + PI / 2) - PI / 2;
   }

  return;
}



/* quick test to see if a position may fall within the display */

boolean within_limits(display,ra_rad,dec_rad)

struct display *display;
double ra_rad, dec_rad;

{
  if (dec_rad < display->bottom_dec_rad)
    return (FALSE);

  if (dec_rad > display->top_dec_rad)
    return (FALSE);

  if (display->left_ra_rad > display->right_ra_rad)
    /* 0 hours RA not included within display */
    if ((display->left_ra_rad >= ra_rad) && (ra_rad >= display->right_ra_rad))
      return(TRUE);
    else
      return(FALSE);
  else
    /* 0 hours included within display; must skip discontinuity */
    if (((display->left_ra_rad >= ra_rad) && ra_rad >= 0) ||
	            ((2 * PI >= ra_rad) && (ra_rad >= display->right_ra_rad)))
      return(TRUE);
    else
      return(FALSE);

/* NOTREACHED */
}



/* exhaustive test to see whether a position falls within the display */

boolean within_display(display,ra_rad,dec_rad,x,y)

struct display *display;
double ra_rad, dec_rad;
int *x, *y;

{

/* apply the quick test first */
  if (! within_limits(display,ra_rad,dec_rad)) {
    /* hammer the x and y coordinates into insensibility */
    *x = *y = INT_MAX;

    return(FALSE);
  }

/* if it passes the initial test, make absolutely sure */

  /* calculate the exact position of the point */
  pos_rad_to_X(display,ra_rad,dec_rad,x,y);

  /* see if the point lies within the display */
  if ((*x >= 0) && (*x <= display->width) &&
                                         (*y >= 0) && (*y <= display->height))
    return(TRUE);
  else
    return(FALSE);

/* NOTREACHED */
}



/* convert a position in radians to X coordinates */

void pos_rad_to_X(display,ra_rad,dec_rad,x,y)

struct display *display;
double ra_rad;
double dec_rad;
int *x, *y;

{
  double double_x, double_y;

/* convert to double coordinates */
  pos_rad_to_X_double(display,ra_rad,dec_rad,&double_x,&double_y);

/* convert the floating coordinates into integer X coordinates (remember
 * to round) */
  *x = double_x + 0.5;
  *y = double_y + 0.5;

  return;
}



#ifdef MEEUS

/* convert a position in radians to double X coordinates */

void pos_rad_to_X_double(display,ra_rad,dec_rad,x,y)

struct display *display;
double ra_rad;
double dec_rad;
double *x, *y;

{
  double sin_h, h;
  double a;
  double atan2_numerator, atan2_denominator;
  boolean atan2_0_0_flag;
  double xdist, ydist;

/* consider the display as a view of the sky with the center of the display
 * straight up, with the top of the display being north.  it is then easy
 * to see that the distance of an object from the center of the display
 * corresponds to its altitude (distance from the vertical), and the direction
 * of the object fromt he center of the display corresponds to its azimuth.
 * therefore, we use equations 12.5 and 12.6 on page 89 of Astronomical
 * Algorithms, by Jean Meeus, published by Wilmann-Bell, to convert the
 * equatorial coordinates into horizontal coordinates (altitude-azimuth),
 * and then convert those into X coordinates.  In order to apply said
 * equations, realize that the hour angle is the distance in right ascension
 * from the display center, and the latitude is the declination of the display
 * center. */

/* compute the sine of the altitude of the given position */
  sin_h = sin(display->center_dec_rad) * sin(dec_rad) +
                              cos(display->center_dec_rad) * cos(dec_rad) *
			                 cos(display->center_ra_rad - ra_rad);

#ifdef sparc
  /* some systems generate values which are slightly out of range ... */
  sin_h = sin_h > 1.0 ? 1.0 : (sin_h < -1.0 ? -1.0 : sin_h);
#endif

/* compute the altitude and convert into an angle from the vertical */
  h = PI / 2 - asin(sin_h);

/* compute the azimuth angle using the atan2() function to preserve the
 * quadrant information */
  atan2_numerator = sin(display->center_ra_rad - ra_rad);
  atan2_denominator = cos(display->center_ra_rad - ra_rad) *
				 sin(display->center_dec_rad) -
			         tan(dec_rad) * cos(display->center_dec_rad);

  if ((atan2_numerator == 0.0) && (atan2_denominator == 0.0))
    atan2_0_0_flag = TRUE;
  else {
    atan2_0_0_flag = FALSE;
    a = atan2(atan2_numerator,atan2_denominator);
  }

/* convert altitude and azimuth to centric coordinates in pixel units.
 * the coordinates are not true Cartesian in that the y-axis is reversed
 * to correspond to the X convention.  this fell out conveniently because
 * of the astronomical convention that horizontal coordinates measure azi-
 * muth from the south. */

#if 0
#ifdef sgi
  if (NaN(a)) {
#else
  if (isnan(a)) {
#endif
    printf("we don't have a number here\n");
    printf("atan2_numerator = %lf, atan2_denominator = %lf\n",
	                                   atan2_numerator,atan2_denominator);
  }
#endif

#if 0
#ifdef sgi
  if (atan2_0_0_flag || NaN(a))
#else
  if (atan2_0_0_flag || isnan(a))
#endif
#else
  if (atan2_0_0_flag)
#endif
    /* watch out for position in the exact center */
    xdist = ydist = 0.0;
  else {
    xdist = (h * sin(a)) * DEGREES_PER_RAD / display->scale;
    ydist = (h * cos(a)) * DEGREES_PER_RAD / display->scale;
  }

/* convert the centric coordinates into X coordinates */
  *x = xdist + display->width / 2.0;
  *y = ydist + display->height / 2.0;

#ifdef DEBUG
printf("pos_rad_to_X called to convert %lf, %lf\n",ra_rad,dec_rad);
printf("display center is %lf, %lf\n",display->center_ra_rad,
                                                     display->center_dec_rad);
printf("tan_a = %lf, sin_h = %lf\n",tan_a,sin_h);
printf("altitude = %lf, azimuth = %lf\n",h,a);
printf("X coordinates are %lf, %lf\n\n",*x,*y);
#endif

  return;
}

#else   /* not MEEUS */

/* convert a position in radians to double X coordinates */

void pos_rad_to_X_double(display,ra_rad,dec_rad,x,y)

struct display *display;
double ra_rad;
double dec_rad;
double *x, *y;

{
  double sep, dir;
  double xdist, ydist;

/* the way we proceed is to convert the equatorial coordinates to horizontal
 * coordinates, and then project the horizontal coordinates onto the display */

/* calculate the separation from the center of the display */
  sep = pos_to_sep(display->center_ra_rad,display->center_dec_rad,
		                                              ra_rad,dec_rad);

/* calculate the direction from the center of the display */
  if (fabs(sep) < 0.00001) {
    /* worry about an object in the exact center of the display */
    sep = 0.0;
    dir = 0.0;
  }
  else
    dir = pos_to_dir(display->center_ra_rad,display->center_dec_rad,
		                                              ra_rad,dec_rad);

/* convert the separation and direction into centric coordinates */
  sep_to_centric(sep,dir,&xdist,&ydist,display->scale);

/* convert the centric coordinates into double X coordinates */
  centric_to_X(display,xdist,ydist,x,y);

  return;
}



/* return the separation of two objects in radians, given their positions */

static double pos_to_sep(ra1,dec1,ra2,dec2)

double ra1, dec1;
double ra2, dec2;

{
  double cosd;

/* compute the cosine of the separation */
  cosd = sin(dec1) * sin(dec2) + cos(dec1) * cos(dec2) * cos(ra1 - ra2);

/* return the angular separation in radians */
  return(acos(cosd));
}



/* return the direction from object 1 to object 2, in radians, given
 * their positions */

static double pos_to_dir(ra1,dec1,ra2,dec2)

double ra1, dec1;
double ra2, dec2;

{
  double x, y;
  double tana;

/* compute the tangent of the direction (up = 0, positive counterclockwise) */
  y = sin(ra1 - ra2);
  x = cos(ra1 - ra2) * sin(dec1) - tan(dec2) * cos(dec1);
  tana = y/x;

/* quadrant corrections - if the denominator is negative, flip the quadrant */
  if (x < 0)
    return(atan(tana));
  else
    return(atan(tana) + PI);
}



/* convert separation and direction to centric coordinates */

static void sep_to_centric(sep,dir,x,y,scale)

double sep, dir;
double *x, *y;
float scale;

{
  double centric_x, centric_y;

/* calculate coordinates relative to center (+x is up) */
  centric_x = sep * cos(dir);
  centric_y = sep * sin(dir);
#ifdef DEBUG
  printf("centric coords of star = %lf, %lf radians ",centric_x,centric_y);
#endif

/* convert from radians to given scale units */
  *x = (centric_x * DEGREES_PER_RAD) / scale;
  *y = (centric_y * DEGREES_PER_RAD) / scale;
#ifdef DEBUG
  printf("= %lf, %lf in pixels\n",x,y);
#endif

  return;
}



/* convert centric coordinates into X coordinates on the given display */

static void centric_to_X(display,centric_x,centric_y,X_x,X_y)

struct display *display;
double centric_x, centric_y;
double *X_x, *X_y;

{
/* change coords to upper-left relative, flipped and rotated 90 degrees */
  *X_x = display->width / 2.0 - centric_y;
  *X_y = display->height / 2.0 - centric_x;

  return;
}

#endif   /* MEEUS */



#ifdef MEEUS

/* convert an X position into RA and declination in radians */

void X_to_pos_rad(display,x,y,ra_rad,dec_rad)

struct display *display;
int x, y;
double *ra_rad, *dec_rad;

{
  double r;
  double theta;
  double r_angle;
  double h;
  double sin_dec, dec;

/* convert x,y of X position to Cartesian coordinates */
  x = x - display->width / 2;
  y = - (y - display->height / 2);

/* don't let r be zero in polar coordinate conversion below */
  if ((x == 0) && (y == 0)) {
    *ra_rad = display->center_ra_rad;
    *dec_rad = display->center_dec_rad;
    return;
  }

/* convert to polar coordinates, adjusting the sign of the direction */
  r = sqrt((double)(x * x + y * y));
  theta = acos(y / r);

/* convert distance to an angle in radians */
  r_angle = (r * display->scale) / DEGREES_PER_RAD;

/* at this point, r_angle is the distance of the point from the center of
 * the display, expressed as an angle, and theta is a direction, measured
 * from the top of the display; the direction is positive regardless of
 * clockwise or counterclockwise */

/* convert direction into a full-circle angle, counterclockwise from down */
  if (x < 0)
    theta = -theta;
  theta = PI - theta;

/* in actuality, what these polar coordinates are is a modified altazimuth
 * system, from our point of view.  visualize the center of the display as
 * being straight overhead, with the top of the display pointed north.  it
 * is now easy to see that r_angle is the angular distance of the X position
 * from the vertical, and theta is an azimuth angle, measured toward the west
 * from the south.  thus, we have converted the X position to an altazimuth
 * system; we need only change the r_angle to convert to standard horizontal
 * coordinates with the center of the display at the zenith. */

/* convert the distance from the zenith into a distance from the horizon */
  r_angle = (PI / 2) - r_angle;

/* now that we have proper horizontal coordinates, we can apply the equa-
 * tions from Astronomical Algorithms, by Jean Meeus, published by Wilmann-
 * Bell.  The equations are unnumbered, near the bottom of page 89. */

/* compute the hour angle, realizing that the declination of the center of
 * the display is effectively the latitude.  use atan2() to preserve the
 * quadrant information. */
  h = atan2(sin(theta),cos(theta) * sin(display->center_dec_rad) +
		                 tan(r_angle) * cos(display->center_dec_rad));

/* compute the declination */
  sin_dec = sin(display->center_dec_rad) * sin(r_angle) -
                     cos(display->center_dec_rad) * cos(r_angle) * cos(theta);

#ifdef sparc
  /* some systems generate values which are slightly out of range ... */
  sin_dec = sin_dec > 1.0 ? 1.0 : (sin_dec < -1.0 ? -1.0 : sin_dec);
#endif

  dec = asin(sin_dec);

#ifdef DEBUG
printf("altitude = %lf, azimuth = %lf\n",r_angle * 57.29,theta * 57.29);
printf("latitude = %lf\n",display->center_dec_rad * 57.29);
printf("hour angle = %lf\n",h);
printf("sin_dec = %lf\n",sin_dec);
printf("new dec = %lf\n",dec);
#endif

/* convert this hour angle into a new right ascension and store it */
  *ra_rad = display->center_ra_rad - h;
  /* watch out for wrapping */
  if (*ra_rad < 0.0)
    *ra_rad += 2 * PI;
  else if (*ra_rad >= 2 * PI)
    *ra_rad -= 2 * PI;
  else
    ;

/* store the new declination, too */
  *dec_rad = dec;

  return;
}

#else    /* not MEEUS */

/* convert an X position into RA and declination in radians */

void X_to_pos_rad(display,x,y,ra_rad,dec_rad)

struct display *display;
int x, y;
double *ra_rad, *dec_rad;

{
  double r;
  double theta;
  double r_angle;
  double sin_dec, dec;
  double cos_h, h;

/* convert x,y of button press to Cartesian coordinates */
  x = x - display->width / 2;
  y = - (y - display->height / 2);

/* don't let r be zero in polar coordinate conversion below */
  if ((x == 0) && (y == 0)) {
    *ra_rad = display->center_ra_rad;
    *dec_rad = display->center_dec_rad;
    return;
  }

/* convert to polar coordinates, adjusting the sign of the direction */
  r = sqrt((double)(x * x + y * y));
  theta = acos(y/r);
  if (x > 0)
    theta = -theta;

/* convert distance to an angle in radians */
  r_angle = (r * display->scale) / DEGREES_PER_RAD;

/* we can now directly compute the new declination */
  sin_dec = cos(r_angle) * sin(display->center_dec_rad) + sin(r_angle) *
                                    cos(display->center_dec_rad) * cos(theta);

#ifdef sparc
  /* some systems generate values which are slightly out of range ... */
  sin_dec = sin_dec > 1.0 ? 1.0 : (sin_dec < -1.0 ? -1.0 : sin_dec);
#endif

  dec = asin(sin_dec);

/* armed with this, we can compute the hour angle */
  cos_h = (cos(r_angle) / (cos(dec) * cos(display->center_dec_rad))) -
                                      tan(dec) * tan(display->center_dec_rad);

  /* occasionally, the lsb of cos_h can be set when cos_h should be 1.0; this
   * makes cos_h just barely bigger than 1.0 and causes acos() to return NaN.
   * this bit of code fixes that ... */
  if (cos_h > 1.0)
    cos_h = 1.0;
  if (cos_h < -1.0)
    cos_h = -1.0;

  /* now finish computing the hour angle */
  h = acos(cos_h);

/* if the azimuth is negative, flip the sign of the hour angle */
  if (theta < 0)
    h = -h;

/* convert this hour angle into a new right ascension and store it */
  if ((*ra_rad += h) < 0.0)
    /* watch out for wrapping below 0.0 */
    *ra_rad += 2 * PI;

/* store the new declination, too */
  *dec_rad = dec;

  return;
}

#endif   /* MEEUS */



/* convert a sexigesimal RA to radians */

double ra_to_rad(hours,minutes,seconds,thousandths)

int hours;
int minutes;
int seconds;
int thousandths;

{
  return(((((((thousandths / 1000.0) + seconds) / 60) + minutes) /
		           60 + hours) * DEGREES_PER_HOUR) / DEGREES_PER_RAD);
}



/* calculate the RA in radians, given an RA position structure */

double ra_pos_to_rad(ra)

struct ra_pos ra;

{
/* calculate the RA in radians */
  return(ra_to_rad((int)ra.hours,(int)ra.minutes,(int)ra.seconds,
		                                        (int)ra.thousandths));
}



/* calculate the sexigesimal RA, given an RA in radians */

struct ra_pos rad_to_ra(rad)

double rad;

{
  double degrees;
  struct ra_pos ra;

/* this calculation will be a LOT easier to understand in degrees ... */
  degrees = rad * DEGREES_PER_RAD;

/* only go once around the circle */
  degrees -= (int)(degrees / 360) * 360;

/* and that in the positive direction */
  if (degrees < 0)
    degrees += 360;

/* fill in the ra_pos structure hours */
  ra.hours = degrees / (360 / 24);
  degrees -= ra.hours * (360 / 24);
/* degrees is now some number < 15.0, degrees per hour */

  ra.minutes = degrees / (360 / 24 / 60.0);
  degrees -= ra.minutes * (360 / 24 / 60.0);
/* degrees is now some number < .25, degrees per minute */

  ra.seconds = degrees / (360 / 24 / 60.0 / 60.0);
  degrees -= ra.seconds * (360 / 24 / 60.0 / 60.0);
/* degrees is now some number < .00416667, degrees per second */

/* one minute of time is a quarter of a degree; one second is .004167 degree.
 * what we have left is a fraction of a second, but expressed in degrees.
 * therefore, we divide by degrees per second to convert to fractional second,
 * multiply by 1000 to get thousandths, and add 0.5 to get rounding. */
  ra.thousandths = degrees / (360 / 24 / 60.0 / 60.0) * 1000 + 0.5;

/* now take care of the fact that rounding may bump things up */
  if (ra.thousandths >= 1000) {
    ra.seconds++;
    ra.thousandths -= 1000;
  }
  if (ra.seconds >= 60) {
    ra.minutes++;
    ra.seconds -= 60;
  }
  if (ra.minutes >= 60) {
    ra.hours++;
    ra.minutes -= 60;
  }
  if (ra.hours >= 24)
    ra.hours -= 24;

  return(ra);
}



/* convert a sexigesimal declination to radians */

double dec_to_rad(sign,degrees,minutes,seconds,hundredths)

int sign;
int degrees;
int minutes;
int seconds;
int hundredths;

{
  return(sign * (((((hundredths / 100.0) + seconds) / 60) + minutes) /
		                            60 + degrees) / DEGREES_PER_RAD);
}



/* calculate the declination in radians, given a dec. position structure */

double dec_pos_to_rad(dec)

struct dec_pos dec;

{
/* calculate the declination in radians */
  return(dec_to_rad((int)dec.sign,(int)dec.degrees,(int)dec.minutes,
		                       (int)dec.seconds,(int)dec.hundredths));
}



/* calculate the sexigesimal declination, given a declination in radians */

struct dec_pos rad_to_dec(rad)

double rad;

{
  struct dec_pos dec;
  double degrees;

/* save the sign */
  if (rad < 0)
    dec.sign = (short)-1;
  else
    dec.sign = (short)1;

/* this calculation will be a LOT easier to understand in degrees ... */
  degrees = fabs(rad) * DEGREES_PER_RAD;

/* only go once around the circle */
  degrees -= (int)(degrees / 360) * 360;

/* exploit symmetry:  if more than 180 degrees, subtract 180 and flip sign */
  if (degrees > 180) {
    degrees -= 180;
    dec.sign = -dec.sign;
  }

/* fill in the dec_pos structure degrees */
  dec.degrees = degrees;
  degrees -= dec.degrees;
/* degrees is now a fraction of one degree */

  degrees *= 60;
  dec.minutes = degrees;
  degrees -= dec.minutes;
/* degrees is now a fraction of one minute */

  degrees *= 60;
  dec.seconds = degrees;
  degrees -= dec.seconds;
/* degrees is now a fraction of a second */

  degrees *= 100 + 0.5;
  dec.hundredths = degrees;

/* now take care of the fact that the rounding may bump things up */
  if (dec.hundredths >= 100) {
    dec.seconds++;
    dec.hundredths -= 100;
  }
  if (dec.seconds >= 60) {
    dec.minutes++;
    dec.seconds -= 60;
  }
  if (dec.minutes >= 60) {
    dec.degrees++;
    dec.minutes -= 60;
  }
  if (dec.degrees > 90)
    dec.degrees = 90 - (dec.degrees - 90);

  return(dec);
}



/* convert a right ascension in radians to thousandths of a second of time */

int ra_rad_to_thousandths(ra)

double ra;

{
  double degrees;

/* convert the right ascension to degrees */
  degrees = ra * DEGREES_PER_RAD;

  return((int)(degrees * ((60 / 15) * 60 * 1000) + .5));
}



/* convert a declination in radians to hundredths of a second of arc */

int dec_rad_to_hundredths(dec)

double dec;

{
  double degrees;

/* convert the declination to degrees */
  degrees = dec * DEGREES_PER_RAD;

  return((int)((degrees * 60 * 60 * 100) + .5));
}


/* convert a right ascension in thousandths of a second of time to radians */

double ra_thousandths_to_rad(thousandths)

int thousandths;

{
  double degrees;

/* convert the right ascension to degrees */
  degrees = (double)thousandths / (1000 * 60 * 60) * 15;

  return(degrees / DEGREES_PER_RAD);
}



/* convert a declination in hundredths of a second of arc to radians */

double dec_hundredths_to_rad(hundredths)

int hundredths;

{
  double degrees;

/* convert the declination to degrees */
  degrees = (double)hundredths / (100 * 60 * 60);

  return(degrees / DEGREES_PER_RAD);
}



#ifdef ROUND_DISPLAY_POS

/* round RA to seconds */

void round_ra(ra_pos)

struct ra_pos *ra_pos;

{
  if (ra_pos->thousandths >= 500) {
    ra_pos->seconds++;
    ra_pos->thousandths = 0;
  }
  if (ra_pos->seconds >= 60) {
    ra_pos->minutes++;
    ra_pos->seconds -= 60;
  }
  if (ra_pos->minutes >= 60) {
    ra_pos->hours++;
    ra_pos->minutes -= 60;
  }
  if (ra_pos->hours >= 24)
    ra_pos->hours -= 24;

  return;
}



/* round declination to seconds for the display position */

void round_dec(dec_pos)

struct dec_pos *dec_pos;

{
  if (dec_pos->hundredths >= 50) {
    dec_pos->seconds++;
    dec_pos->hundredths = 0;
  }
  if (dec_pos->seconds >= 60) {
    dec_pos->minutes++;
    dec_pos->seconds -= 60;
  }
  if (dec_pos->minutes >= 60) {
    dec_pos->degrees++;
    dec_pos->minutes -= 100;
  }
  if (dec_pos->degrees > 90)
    dec_pos->degrees = 90 - (dec_pos->degrees - 90);

  return;
}

#endif   /* ROUND_DISPLAY_POS */



/*
 * routine for inputting RA
 *
 * general format:
 *
 *	{ [nn]h | [nnh] [nn]m | [nnh] [nnm] [nn]s } [.nnn]
 *
 * zero or more spaces leading the hours, minutes, and seconds fields
 *	and prior to the '.', if present
 *
 * ra_pos structure filled in with hours, minutes, seconds, and thousandths
 *	of a second of time
 */

boolean get_ra(bufptr,ra)

char *bufptr;
struct ra_pos *ra;

{
	int i, j;
	int val;
	int hours, minutes, seconds;
	int fraction;
	int hour_frac, min_frac, sec_frac;
	int unit_pos;
	char frac_buff[20];
	int retval;

#ifdef DEBUG
	printf("getting RA %s\n",bufptr);
#endif

/* init a bunch of values */
	hours = minutes = seconds = 0;
	hour_frac = min_frac = sec_frac = 0;
	fraction = 0;

/* begin at the beginning */
	i = 0;
	do {
/* skip blanks */
		while (isspace(bufptr[i]) && (bufptr[i] != '\n'))
			i++;

		/* watch out for end of input */
		if ((bufptr[i] == '\n') || (bufptr[i] == '\0') ||
						(bufptr[i] == ','))
			break;
		/* if we find a fraction, break out and get it */
		if (bufptr[i] == '.')
			break;

/* input integer */
		val = atoi(&bufptr[i]);
		/* skip to a non-digit */
		while (isdigit(bufptr[i]))
			i++;
		/* upcase the units identifier */
		bufptr[i] = toupper(bufptr[i]);

/* on h, m, or s, store hours, minutes, or seconds */
		if (bufptr[i] == 'H')
			if ((hours = val) >= 24)
				return(FALSE);
			else
				;
		else if (bufptr[i] == 'M')
			if ((minutes = val) >= 60)
				return(FALSE);
			else
				;
		else if (bufptr[i] == 'S')
			if ((seconds = val) >= 60)
				return(FALSE);
			else
				;
		else
			return(FALSE);

		/* remember where the last unit character is */
		unit_pos = i;

		/* advance past the units indicator */
		i++;

/* if that was hours or minutes, get more */
		} while ((bufptr[i - 1] == 'H') || (bufptr[i - 1] == 'M'));

/* if this character is '.', read fraction of final unit */
	if (bufptr[i] == '.') {
		/* point to first decimal digit */
		i++;

		/* copy out the digits */
		j = 0;
		while (isdigit(frac_buff[j++] = bufptr[i++]))
			if (j >= 4)
				break;
		/* plant a null after the digits */
		frac_buff[j - 1] = '\0';

		/* this better be the end of input */
		if ((bufptr[i - 1] == '\n') || (bufptr[i - 1] == '\0'))
			;
		else
			return(FALSE);

		/* get the integer value after the decimal point */
		fraction = atoi(frac_buff);
		/* adjust to thousandths if less than 3 digits */
		if (strlen(frac_buff) == 1)
			fraction *= 100;
		else if (strlen(frac_buff) == 2)
			fraction *= 10;

		/* figure out which unit the fraction belongs to */
		if (bufptr[unit_pos] == 'H')
			hour_frac = fraction;
		else if (bufptr[unit_pos] == 'M')
			min_frac = fraction;
		else if (bufptr[unit_pos] == 'S')
			sec_frac = fraction;
		else
			return(FALSE);
		}

/* calculate the thousandths of a second */
	retval = ((((hours * 60) + minutes) * 60) + seconds) * 1000;
	retval += (hour_frac * 60 * 60) + (min_frac * 60) + sec_frac;
#ifdef DEBUG
	printf("ra value = %d\n",retval);
#endif

/* fill in the ra structure and return */
	ra->thousandths = retval % 1000;
	ra->seconds = (retval /= 1000) % 60;
	ra->minutes = (retval /= 60) % 60;
	ra->hours = (retval / 60) % 24;
#ifdef DEBUG
	printf("get_ra position = %dh %dm %ds.%03d\n",ra->hours,
							ra->minutes,
							ra->seconds,
							ra->thousandths);
#endif

	return(TRUE);
}



/*
 * routine for inputting declination
 *
 * general format:
 *
 *	{ + | - } { [nn]d | [nnd] [nn]' | [nnd] [nn'm] [nn]" } [.nnn]
 *
 * zero or more spaces leading the degrees, minutes, and seconds fields
 *	and prior to the '.', if present
 *
 * dec_pos structure filled in with degrees, minutes, seconds, and hundredths
 *	of a second of arc
 */

boolean get_dec(bufptr,dec)

char *bufptr;
struct dec_pos *dec;

{
	int i, j;
	int val;
	int degrees, minutes, seconds;
	int fraction;
	int deg_frac, min_frac, sec_frac;
	int sign;
	int unit_pos;
	char frac_buff[20];
	int retval;

/* init a bunch of values */
	degrees = minutes = seconds = 0;
	deg_frac = min_frac = sec_frac = 0;
	fraction = 0;

/* assume positive sign */
	sign = 1;

/* begin at the beginning */
	i = 0;
	do {
/* skip blanks */
		while (isspace(bufptr[i]) && (bufptr[i] != '\n'))
			i++;

		/* watch out for end of input */
		if ((bufptr[i] == '\n') || (bufptr[i] == '\0'))
			break;
		/* if we find a fraction, break out and get it */
		if (bufptr[i] == '.')
			break;

/* check for a sign */
		if (bufptr[i] == '+') {
			sign = 1;
			/* skip past the sign */
			i++;
			continue;
			}
		else if (bufptr[i] == '-') {
			sign = -1;
			/* skip past the sign */
			i++;
			continue;
			}

/* input integer */
		val = atoi(&bufptr[i]);
		/* skip to a non-digit */
		while (isdigit(bufptr[i]))
			i++;
		/* upcase the units identifier */
		bufptr[i] = toupper(bufptr[i]);

/* on d, ', or ", store degrees, minutes, or seconds */
		if (bufptr[i] == 'D')
			if (abs(degrees = val) >= 90)
				return(FALSE);
			else
				;
		else if (bufptr[i] == '\'')
			if ((minutes = val) >= 60)
				return(FALSE);
			else
				;
		else if (bufptr[i] == '\"')
			if ((seconds = val) >= 60)
				return(FALSE);
			else
				;
		else
			return(FALSE);

		/* remember where the last unit character is */
		unit_pos = i;

		/* advance past the units indicator */
		i++;

/* if that was hours or minutes, get more */
		} while ((bufptr[i - 1] == 'D') || (bufptr[i - 1] == '\'')
						|| (bufptr[i - 1] == '+')
						|| (bufptr[i - 1] == '-'));

/* if this character is '.', read fraction of final unit */
	if (bufptr[i] == '.') {
		/* point to first decimal digit */
		i++;

		/* copy out the digits */
		j = 0;
		while (isdigit(frac_buff[j++] = bufptr[i++]))
			if (j >= 4)
				break;
		/* plant a null after the digits */
		frac_buff[j - 1] = '\0';

		/* this better be the end of input */
		if ((bufptr[i - 1] == '\n') || (bufptr[i - 1] == '\0'))
			;
		else
			return(FALSE);

		/* get the integer value after the decimal point */
		fraction = atoi(frac_buff);
		/* adjust to hundredths if less than 2 digits */
		if (strlen(frac_buff) == 1)
			fraction *= 10;

		/* figure out which unit the fraction belongs to */
		if (bufptr[unit_pos] == 'D')
			deg_frac = fraction;
		else if (bufptr[unit_pos] == '\'')
			min_frac = fraction;
		else if (bufptr[unit_pos] == '\"')
			sec_frac = fraction;
		else
			return(FALSE);
		}

/* return the value as hundredths of a second of declination */
	retval = ((((degrees * 60) + minutes) * 60) + seconds) * 100;
	retval += (deg_frac * 60 * 60) + (min_frac * 60) + sec_frac;
#ifdef DEBUG
	printf("dec value = %d\n",retval);
#endif

/* fill in the dec structure and return */
	dec->hundredths = retval % 100;
	dec->seconds = (retval /= 100) % 60;
	dec->minutes = (retval /= 60) % 60;
	dec->degrees = (retval / 60) % 90;
	dec->sign = sign;
#ifdef DEBUG
	if (sign == 1)
	  printf("get_dec position = %dd %d\' %d\".%02d\n",dec->degrees,
							    dec->minutes,
							    dec->seconds,
							    dec->hundredths);
	else
	  printf("get_dec position = -%dd %d\' %d\".%02d\n",dec->degrees,
							    dec->minutes,
							    dec->seconds,
							    dec->hundredths);
#endif

	return(TRUE);
}



/*
 * Precession with IAU 1976 constants.
 *
 * The precession algorithm and precession constants used in
 * these routines are from PRECESS.BAS, written by PC Leyland
 */

static double rot[3][3];



/* build the rotation matrix for the precession */

void init_precess(epoch_0,new_equinox)

double epoch_0, new_equinox;

{
  double t0, new_t;
  double dt;
  double c1, c2, c3, c4, c5, c6, c7, c8, c9, c10, c11, c12, c13, c14, c15;
  double zeta, z, theta;
  double cs1, cs2, cs3;
  double sn1, sn2, sn3;

/* compute epochs in decimal centuries since 1900.0 */
  t0 = (epoch_0 - 1900.0) / 100;
  new_t = (new_equinox - 1900.0) / 100;
#ifdef DEBUG
  printf("\nbuilding precession tables for t0 = %lf, new t = %lf\n",t0,new_t);
#endif

/* set up precession constants in centuries */
  dt = new_t - t0;
  c1 = 2306.2181; c2 = 1.39656; c3 = -0.000139;
  c4 = 0.30188; c5 = -0.000345; c6 = 0.017998;
  c7 = 0.7928; c8 = 0.000411; c9 = 0.000205;
  c10 = 2004.3109; c11 = -0.8533; c12 = -0.000217;
  c13 = -0.42665; c14 = -0.000217; c15 = -0.041833;
#ifdef DEBUG
  printf("\ndt = %lf\n",dt);
  printf("c1 = %lf, c2 = %lf, c3 = %lf\n",c1,c2,c3);
  printf("c4 = %lf, c5 = %lf, c6 = %lf\n",c4,c5,c6);
  printf("c7 = %lf, c8 = %lf, c9 = %lf\n",c7,c8,c9);
  printf("c10 = %lf, c11 = %lf, c12 = %lf\n",c10,c11,c12);
  printf("c13 = %lf, c14 = %lf, c15 = %lf\n",c13,c14,c15);
#endif
  zeta = ((c1 + (c2 + c3 * t0) * t0) + ((c4 + c5 * t0) + c6 * dt) * dt ) * dt;
  z = zeta + ((c7 + c8 * t0) + c9 * dt) * dt * dt;
  theta = ((c10 + (c11 + c12 * t0) * t0) + ((c13 + c14 * t0) +
					                 c15 * dt) * dt) * dt;
  zeta = ARCSEC_TO_RAD(zeta);
  z = ARCSEC_TO_RAD(z);
  theta = ARCSEC_TO_RAD(theta);
#ifdef DEBUG
  printf("\n(radians):  zeta = %lf, z = %lf, theta = %lf\n",zeta,z,theta);
#endif

/* compute subexpressions for rotation matrix */
  cs1 = cos(z); cs2 = cos(theta); cs3 = cos(zeta);
  sn1 = sin(z); sn2 = sin(theta); sn3 = sin(zeta);
#ifdef DEBUG
  printf("\ncs1 = %lf, cs2 = %lf, cs3 = %lf\n",cs1,cs2,cs3);
  printf("sn1 = %lf, sn2 = %lf, sn3 = %lf\n",cs1,cs2,cs3);
#endif

/* build rotation matrix */
  rot[0][0] = -sn1 * sn3 + cs1 * cs2 * cs3;
  rot[0][1] = -sn1 * cs3 - cs1 * cs2 * sn3;
  rot[0][2] = -cs1 * sn2;
  rot[1][0] = cs1 * sn3 + sn1 * cs2 * cs3;
  rot[1][1] = cs1 * cs3 - sn1 * cs2 * sn3;
  rot[1][2] = -sn1 * sn2;
  rot[2][0] = sn2 * cs3;
  rot[2][1] = -sn2 * sn3;
  rot[2][2] = cs2;
#ifdef DEBUG
  printf("rotation matrix:  \n");
  printf("%lf\t%lf\t%lf\n",rot[0][0],rot[0][1],rot[0][2]);
  printf("%lf\t%lf\t%lf\n",rot[1][0],rot[1][1],rot[1][2]);
  printf("%lf\t%lf\t%lf\n",rot[2][0],rot[2][1],rot[2][2]);
#endif

  return;
}



/* precess a position using the pre-built rotation matrix */

void precess(alpha_0,delta_0,new_alpha,new_delta)

double alpha_0, delta_0;
double *new_alpha, *new_delta;

{
  double dir_cos_0[3], new_dir_cos[3];
  int i, j;

#ifdef DEBUG
  printf("\nprecessing position (radians):\n");
  printf("RA = %lf, dec = %lf\n",alpha_0,delta_0);
#endif

/* compute direction cosines of given position */
  dir_cos_0[0] = cos(delta_0) * cos(alpha_0);
  dir_cos_0[1] = cos(delta_0) * sin(alpha_0);
  dir_cos_0[2] = sin(delta_0);
#ifdef DEBUG
  printf("initial direction cosines:  %lf, %lf, %lf\n",
                                      dir_cos_0[0],dir_cos_0[1],dir_cos_0[2]);
#endif

/* precess to new direction cosines using rotation matrix */
  for (i = 0; i < 3; i++) {
    new_dir_cos[i] = 0;
    for (j = 0; j < 3; j++)
      new_dir_cos[i] += rot[i][j] * dir_cos_0[j];
  }
#ifdef DEBUG
  printf("new direction cosines:  %lf, %lf, %lf\n",
	                        new_dir_cos[0],new_dir_cos[1],new_dir_cos[2]);
#endif

/* compute new position in radians */
  *new_delta = asin(new_dir_cos[2]);
  *new_alpha = atan2(new_dir_cos[1],new_dir_cos[0]);

/* convert the values from -pi..0 to pi..2*pi */
  if (*new_alpha < 0)
    *new_alpha += 2 * PI;

#ifdef DEBUG
  printf("\nprecessed to position (radians):\n");
  printf("RA = %lf, dec = %lf\n",*new_alpha,*new_delta);
#endif

  return;
}
