/*
 * xsky - an interactive sky atlas
 *
 * Copyright 1992-6, Terry R. Friedrichsen
 *
 * This program may be copied and redistributed, in whole or in part,
 * as long as you don't try to make any money from the sale or redis-
 * tribution of the program or any part of the program, or pretend
 * that you wrote the program or any of its parts unless specifically
 * credited by the original author.
 *
 * You are free to make use of this software in your own programs, as
 * long as you credit the original author where it is due.
 */

/*
 * WARRANTY:
 * xsky was written as a learning project and as a demonstration of
 * X Window System programming.  xsky doesn't do anything; it is not
 * merchantable, and it is not fit for any purpose whatsoever.  In
 * fact, don't use xsky at all; it's free, and you're getting what
 * you paid for.
 */

#include <stdio.h>
#include <stdlib.h>

#include <string.h>
#include <ctype.h>

#include "skydefs.h"

#include "messier.h"
#include "ngc_ic_names.h"

struct ngc_ic_name *ngc_ic_names = (struct ngc_ic_name *)NULL;

/* external function prototypes */
extern char *build_filespec PROTOTYPE((char *,char *));

/* buffer for NGC/IC common name list */
#define MAX_NAME_LIST   55
static char namebuff[MAX_NAME_LIST + 1];



/* find the Messier object in the buffer and update the display coordinates */

boolean find_messier_obj(buffer,find_ngc_func)

char *buffer;
boolean (*find_ngc_func)();

{
  char *ptr;
  int messier_number;
  char ngc_buffer[10];

/* a Messier designation has to start with an "M" */
  if ((*buffer == 'M') || (*buffer == 'm'))
    buffer++;
  else
    return(FALSE);

/* skip white space */
  while (isspace(*buffer))
    buffer++;

/* is the rest of the buffer numeric?  if so, try it as a Messier number */
  ptr = buffer;
  while (isdigit(*ptr))
    ptr++;
  if (*ptr != '\0')
    return(FALSE);

/* get the Messier number */
  messier_number = atoi(buffer);

/* ensure it's reasonableness */
  if ((messier_number < 1) || (messier_number > NUM_MESSIER_OBJS))
    return(FALSE);
  if (messier_trans[messier_number - 1] == 0)
    return(FALSE);

/* build an NGC buffer */
  sprintf(ngc_buffer,"NGC %d",messier_trans[messier_number - 1]);

/* pretend the user entered the NGC number */
  return((*find_ngc_func)(ngc_buffer));
}



/* find the corresponding Messier number for the given NGC number */

int find_messier_number(ngc_no)

int ngc_no;

{
  int i;

/* scan the Messier table, looking for a match to this NGC number */
  for (i = 0; i < NUM_MESSIER_OBJS; i++)
    if (ngc_no == messier_trans[i])
      break;

/* return the Messier number, if found; else return -1 */
  if (i < NUM_MESSIER_OBJS)
    return(i + 1);
  else
    return(-1);

/* NOTREACHED */
}



/* find an NGC object by common name */

boolean find_ngc_by_name(buffer,find_ngc_func)

char *buffer;
boolean (*find_ngc_func)();

{
  struct ngc_ic_name *ngc_ptr, *match_ptr;
  char ngc_buff[10 + 1];

/* loop through the object name list, looking for a match */
  ngc_ptr = ngc_ic_names;
  match_ptr = (struct ngc_ic_name *)NULL;

  while (ngc_ptr != (struct ngc_ic_name *)NULL) {
/* if the buffer matches the object name, this is probably what the user is
 * looking for */
    if (strncasecmp(buffer,ngc_ptr->ngc_ic_name,strlen(buffer)) == EQUAL)
      if (match_ptr == (struct ngc_ic_name *)NULL)
	match_ptr = ngc_ptr;
      else
	/* the requested object name is ambiguous */
	return(FALSE);

    /* step to the next common name in the list */
    ngc_ptr = ngc_ptr->next;
  }

/* if we fell out of the loop with no match, the name is unrecognized */
  if (match_ptr == (struct ngc_ic_name *)NULL)
    return(FALSE);

/* we have a unique match - build an NGC/IC object number description */
  if (match_ptr->ngc_ic_num >= IC_NUM_OFFSET)
    sprintf(ngc_buff,"IC %d",match_ptr->ngc_ic_num - IC_NUM_OFFSET);
  else
    sprintf(ngc_buff,"NGC %d",match_ptr->ngc_ic_num);

/* find the NGC/IC object by number */
  return((*find_ngc_func)(ngc_buff));
}



/* return the common name of the given NGC/IC object, if any */

char *find_ngc_common_names(ngc_ic_no)

int ngc_ic_no;

{
  struct ngc_ic_name *this_ngc;
  boolean match_found = FALSE;
  boolean first_match = TRUE;

/* step through the names, looking for one with this NGC/IC number */
  this_ngc = ngc_ic_names;
  while (this_ngc != (struct ngc_ic_name *)NULL) {
    if (this_ngc->ngc_ic_num == ngc_ic_no) {
      /* this name matches - save it */
      match_found = TRUE;
      if (first_match) {
	/* copy in this NGC/IC common name */
	strcpy(namebuff,this_ngc->ngc_ic_name);
	first_match = FALSE;
      }
      else {
	/* copy in some separation and the new name */
	strcat(namebuff,"      ");
	strcat(namebuff,this_ngc->ngc_ic_name);
      }
    }

    /* step to the next NGC/IC common name */
    this_ngc = this_ngc->next;
  }

/* return the common name buffer or NULL if no match was found */
  if (match_found)
    return(namebuff);
  else
    return((char *)NULL);

/* NOTREACHED */
}



/* read the table of NGC/IC numbers that have common names */

void read_ngc_name_table()

{
  char *filebuff;
  FILE *name_fd;
  char inbuf[BUFSIZ + 1];
  struct ngc_ic_name *this_ngc;
  char *name;

/* only read the table once */
  if (ngc_ic_names != (struct ngc_ic_name *)NULL)
    return;

/* build the file path */
  filebuff = build_filespec("NGC_PATH",NGC_NAME_FILE);

/* open the descriptor file */
  if ((name_fd = fopen(filebuff,"r")) == (FILE *)NULL) {
    printf("sky:  error opening NGC name file %s\n",filebuff);
    perror("sky");
    exit(1);
  }

/* loop through the file, reading in NGC/IC object numbers and names */
  while (fgets(inbuf,BUFSIZ,name_fd) != NULL) {
    /* remove the trailing newline */
    inbuf[strlen(inbuf) - 1] = '\0';

    /* get a node for this NGC/IC name */
    this_ngc = (struct ngc_ic_name *)malloc(sizeof(struct ngc_ic_name));

    /* the NGC/IC identifier is the first token in the buffer */
    if (strcmp(strtok(inbuf," \t"),"NGC") == EQUAL)
      /* the NGC/IC number is the second token in the buffer */
      this_ngc->ngc_ic_num = atoi(strtok((char *)NULL," \t"));
    else
      this_ngc->ngc_ic_num = atoi(strtok((char *)NULL," \t")) + IC_NUM_OFFSET;

    /* point to the object name */
    name = strtok((char *)NULL,"\n");

    /* allocate space for the name and save it */
    this_ngc->ngc_ic_name = (char *)malloc(strlen(name) + 1);
    strcpy(this_ngc->ngc_ic_name,name);

/* link this NGC/IC object into the list */
    this_ngc->next = ngc_ic_names;
    ngc_ic_names = this_ngc;
  }

/* be neat */
  (void)fclose(name_fd);

/* free the filename buffer */
  free((void *)filebuff);

  return;
}
