/*
 * xsky - an interactive sky atlas
 *
 * Copyright 1995, Terry R. Friedrichsen
 *
 * This program may be copied and redistributed, in whole or in part,
 * as long as you don't try to make any money from the sale or redis-
 * tribution of the program or any part of the program, or pretend
 * that you wrote the program or any of its parts unless specifically
 * credited by the original author.
 *
 * You are free to make use of this software in your own programs, as
 * long as you credit the original author where it is due.
 */

/*
 * WARRANTY:
 * xsky was written as a learning project and as a demonstration of
 * X Window System programming.  xsky doesn't do anything; it is not
 * merchantable, and it is not fit for any purpose whatsoever.  In
 * fact, don't use xsky at all; it's free, and you're getting what
 * you paid for.
 */

#include <stdio.h>
#include <ctype.h>

#include <math.h>
#include <stdlib.h>

#include <unistd.h>

#include <string.h>

#include <X11/Intrinsic.h>

#include "ngc2000.h"
#include "ngc_ic_names.h"

#include "skydefs.h"

#include "pos.h"
#include "catalog.h"
#include "draw.h"
#include "catreader.h"

struct ngc_node {
  double ra_rad;
  double dec_rad;
  long filepos;
  shortint cat_number;
  smallint ngc_ic_flag;
  smallint objtype;
};

/* static function prototypes */
static FILE *read_ngc PROTOTYPE((void));
static void calculate_ngc_field PROTOTYPE((struct display *,
					                struct cat_header *));
static boolean find_ngc_obj PROTOTYPE((char *));
static void get_ngc_pos PROTOTYPE((int,double *,double *));
static void draw_ngc_obj PROTOTYPE((Widget,int,int,int,int,int,int,int));
static boolean ps_draw_ngc_obj PROTOTYPE((struct display *,int,int,int));
static boolean display_ngc PROTOTYPE((FILE *,int));
static void format_ngc_name PROTOTYPE((int,char *,IDfont *));
static void fill_ngc_entry PROTOTYPE((char *,int));
static int ngc_sort_compare PROTOTYPE((struct ngc_node *,struct ngc_node *));
static int ngc_search_compare PROTOTYPE((void *,void *,int));

/* external function prototypes */
extern void read_ngc_desc_table PROTOTYPE((void));
extern void read_ngc_name_table PROTOTYPE((void));
extern char *build_filespec PROTOTYPE((char *,char *));
extern boolean read_binary_database PROTOTYPE((char *,void *,int *));
extern void init_precess PROTOTYPE((double,double));
extern void hs_heapsort PROTOTYPE((char [],size_t,size_t,int (*)()));
extern boolean write_binary_database PROTOTYPE((char *,void *,size_t,size_t));
extern double ra_pos_to_rad PROTOTYPE((struct ra_pos));
extern double dec_pos_to_rad PROTOTYPE((struct dec_pos));
extern void precess PROTOTYPE((double,double,double *,double *));
extern void add_to_display_list PROTOTYPE((struct display *,
					           struct cat_header *,
					                  int,double,double));
extern void draw_object PROTOTYPE((Widget,int,int,int,ObjShape));
extern boolean ps_draw_object PROTOTYPE((struct display *,double,double,
					                    double,ObjShape));
extern void format_ngc_display PROTOTYPE((char *));
extern int binsearch PROTOTYPE((void *,int,void *,int (*)()));
extern void update_display_pos PROTOTYPE((double,double));
extern boolean find_ngc_by_name PROTOTYPE((char *,boolean (*)()));
extern boolean find_messier_obj PROTOTYPE((char *,boolean (*)()));
extern int find_messier_number PROTOTYPE((int));

static struct ngc_node ngc_array[NGC_NUMREC];
static int ngc_array_max;

/* AFAIK, these are the three possible ways the CD-ROM file name might be
 * represented by the operating system */
#define NGC_DATA_FILE_1   "NGC2000.DAT;1"
#define NGC_DATA_FILE_2   "NGC2000.DAT"
#define NGC_DATA_FILE_3   "ngc2000.dat"

#define NGC_BINARY_FILE   "ngc2000.bin"

/* NGC record buffer */
static char ngc_rec[NGC_RECLEN + 2 + 1];

/* definitions for object type codes */
struct {
  char *code;
  char *description;
} ngc_type_codes[] = {
  { "Gx",  "galaxy" },
  { "OC",  "open cluster" },
  { "Gb",  "globular cluster" },
  { "Nb",  "emission or reflection nebula" },
  { "Pl",  "planetary nebula" },
  { "C+N", "cluster associated with nebulosity" },
  { "Ast", "asterism" },
  { "Kt",  "knot in external galaxy" },
  { "***", "triple star" },
  { "D*",  "double star" },
  { "D*?", "double star?" },
  { "*",   "star" },
  { "*?",  "star?" },
  { "?",   "type uncertain (nonexistent?)" },
  { "",    "unidentified or type unknown" },
  { "-",   "\"nonexistent\" in RNGC" },
  { "PD",  "plate defect" },
  { (char *)NULL, (char *)NULL }
};

/* definitions for the type code indices - note that they depend on the
 * order of the preceding table */
#define GALAXY                0
#define OPEN_CLUSTER          1
#define GLOBULAR_CLUSTER      2
#define NEBULA                3
#define PLANETARY_NEBULA      4
#define CLUSTER_NEBULOUS      5
#define ASTERISM              6
#define KNOT_IN_GALAXY        7
#define TRIPLE_STAR           8
#define DOUBLE_STAR           9
#define DOUBLE_STAR_MAYBE    10
#define STAR                 11
#define STAR_MAYBE           12
#define UNCERTAIN_NONEXIST   13
#define UNIDENTIFIED_UNKNOWN 14
#define NONEXISTENT_IN_RNGC  15
#define PLATE_DEFECT         16

/* uncomment the next line to suppress display of nonexistent RNGC objects */
/* #define SUPPRESS_NONEXISTENT */



/* initialize the catalog data structure */

void init_ngc2000(catalog)

struct cat_header *catalog;

{
  catalog->read_data = read_ngc;
  catalog->calculate_field = calculate_ngc_field;
  catalog->mag_limit_check = NULLBOOLFUNCTION;
  catalog->find_object = find_ngc_obj;
  catalog->get_object_pos = get_ngc_pos;
  catalog->draw_object = draw_ngc_obj;
  catalog->ps_draw_object = ps_draw_ngc_obj;
  catalog->display_data = display_ngc;
  catalog->format_name = format_ngc_name;
  catalog->menu_name = NGC_MENU_NAME;
  catalog->catalog_name = NGC_CATALOG_NAME;

  return;
}



/* read in the NGC 2000 catalog, extracting needed info */

static FILE *read_ngc()

{
  char *filebuff;
  FILE *ngc_fd;
  char *bin_filebuff;
  int ngc_entry;

/* read in the table of NGC description abbreviations */
  read_ngc_desc_table();

/* read in the table of common NGC object names */
  read_ngc_name_table();

/* build the path to the NGC data file and open it */
  filebuff = build_filespec("NGC2000_CD_PATH",NGC_DATA_FILE_1);
  if ((ngc_fd = fopen(filebuff,"r")) == (FILE *)NULL) {
    free((void *)filebuff);
    filebuff = build_filespec("NGC2000_CD_PATH",NGC_DATA_FILE_2);
    if ((ngc_fd = fopen(filebuff,"r")) == (FILE *)NULL) {
      free((void *)filebuff);
      filebuff = build_filespec("NGC2000_CD_PATH",NGC_DATA_FILE_3);
      if ((ngc_fd = fopen(filebuff,"r")) == (FILE *)NULL) {
	printf("sky:  can't open NGC 2000 data file %s\n",filebuff);
	perror("sky");
	exit(1);
      }
    }
  }

/* get the path to the NGC binary file */
  bin_filebuff = build_filespec("NGC2000_BIN_PATH",NGC_BINARY_FILE);

/* see if we can read the binary file */
  if (access(bin_filebuff,R_OK) == 0)
    /* we can; try to use the pre-built database */
    if (read_binary_database(bin_filebuff,ngc_array,&ngc_array_max)) {
      /* success; free the filename buffers */
      free((void *)filebuff);
      free((void *)bin_filebuff);

      /* return the fd of the open data file */
      return(ngc_fd);
    }

/* prior to reading the NGC catalog, build the rotation matrix for precessing
 * the object coordinates to the common equinox from the given epoch */
  if (NGC_EPOCH != COMMON_EQUINOX)
    init_precess(NGC_EPOCH,COMMON_EQUINOX);

/* read the NGC catalog, saving the information necessary for the display */
  ngc_entry = 0;
  while (TRUE) {
    ngc_array[ngc_entry].filepos = ftell(ngc_fd);
    if (fgets(ngc_rec,sizeof(ngc_rec),ngc_fd) == (char *)NULL)
      break;
    fill_ngc_entry(ngc_rec,ngc_entry++);
  }

/* leave the file open for later use */
/*  (void)fclose(ngc_fd);*/

/* remember the last ngc entry we filled */
  ngc_array_max = ngc_entry - 1;

/* now sort the array by object number, artifically adding an offset to IC */
  hs_heapsort((void *)ngc_array,(size_t)ngc_entry,
	                    (size_t)sizeof(struct ngc_node),ngc_sort_compare);

/* attempt to save the binary database */
  if (access(bin_filebuff,W_OK) == 0)
    /* it's writeable, so let's try */
    if (! write_binary_database(bin_filebuff,(void *)ngc_array,
				           (size_t)sizeof(struct ngc_node),
				                           (size_t)ngc_entry))
      /* dump of binary database failed; be sure the file is gone */
      (void)unlink(bin_filebuff);

/* free the filename buffers */
  free((void *)filebuff);
  free((void *)bin_filebuff);

  return(ngc_fd);
}



/* given an object's index, return its right ascension and declination */

static void get_ngc_pos(array_pos,ra_rad,dec_rad)

int array_pos;
double *ra_rad, *dec_rad;

{
  *ra_rad = ngc_array[array_pos].ra_rad;
  *dec_rad = ngc_array[array_pos].dec_rad;

  return;
}



/* given an NGC record, make a data node for it */

static void fill_ngc_entry(record,ngc_entry)

char *record;
int ngc_entry;

{
  int reclen;
  char field[10];
  char *ptr;
  int i;
  struct ra_pos ra_pos;
  struct dec_pos dec_pos;

/* get the length of the record for GET_FIELD */
  reclen = strlen(record) - 1;

/* get the NGC/IC number */
  GET_FIELD(field,record,CAT_START,CAT_LENGTH);
  ptr = strtok(field," ");
  if (strcmp(ptr,"NGC") == EQUAL)
    ngc_array[ngc_entry].ngc_ic_flag = NGC_ID_CODE;
  else
    ngc_array[ngc_entry].ngc_ic_flag = IC_ID_CODE;  
  ngc_array[ngc_entry].cat_number = atoi(strtok((char *)NULL," "));

/* get the object type */
  GET_FIELD(field,record,TYPE_START,TYPE_LENGTH);

  /* skip leading blanks */
  ptr = field;
  while (*ptr == ' ')
    ptr++;

  /* loop through the type code table, looking for a match */
  i = 0;
  while (ngc_type_codes[i].code != (char *)NULL)
    if (strcmp(ngc_type_codes[i].code,ptr) == EQUAL)
      break;
    else
      i++;
  ngc_array[ngc_entry].objtype = i;

/* complain about an unknown type code if we fall out of the loop */
  if (ngc_type_codes[i].code == (char *)NULL)
    printf("unknown NGC type code %s\n",ptr);

/* get the RA in pieces */
  GET_FIELD(field,record,RA_HOURS_START,RA_HOURS_LENGTH);
  ra_pos.hours = (smallint)atoi(field);

  GET_FIELD(field,record,RA_MIN_START,RA_MIN_LENGTH);
  ra_pos.minutes = (smallint)atoi(field);

  ra_pos.seconds = (smallint)(atoi(&field[RA_MIN_LENGTH - 1]) * (60 / 10));

  ra_pos.thousandths = (shortint)0;

/* similarly, get the declination in pieces */
  GET_FIELD(field,record,DEC_SIGN_START,DEC_SIGN_LENGTH);
  dec_pos.sign = (short)1;
  if (field[0] == '-')
    dec_pos.sign = (short)-1;

  GET_FIELD(field,record,DEC_DEG_START,DEC_DEG_LENGTH);
  dec_pos.degrees = (smallint)atoi(field);

  GET_FIELD(field,record,DEC_MIN_START,DEC_MIN_LENGTH);
  dec_pos.minutes = (smallint)atoi(field);

  dec_pos.seconds = (smallint)0;
  dec_pos.hundredths = (smallint)0;

/* compute the RA and declination in radians */
  ngc_array[ngc_entry].ra_rad = ra_pos_to_rad(ra_pos);
  ngc_array[ngc_entry].dec_rad = dec_pos_to_rad(dec_pos);

/* precess them to the common equinox */
  if (NGC_EPOCH != COMMON_EQUINOX)
    precess(ngc_array[ngc_entry].ra_rad,ngc_array[ngc_entry].dec_rad,
	          &ngc_array[ngc_entry].ra_rad,&ngc_array[ngc_entry].dec_rad);

  return;
}



/* construct a list of those objects which appear in the current display */

static void calculate_ngc_field(display,catalog)

struct display *display;
struct cat_header *catalog;

{
  int i;

/* loop through all NGC entries, checking for visibility */
  for (i = 0; i <= ngc_array_max; i++) {
#ifdef SUPPRESS_NONEXISTENT
    /* if the object is nonexistent, just ignore it */
    if (ngc_array[i].objtype == NONEXISTENT_IN_RNGC)
      continue;
#endif

/* check this object and add it to the list if visible */
    add_to_display_list(display,catalog,i,ngc_array[i].ra_rad,
                                                        ngc_array[i].dec_rad);
  }

#ifdef DEBUG
  printf("calculate_ngc_field returning\n");
#endif
  return;
}



/* draw an NGC object on the sky */

static void draw_ngc_obj(sky_widget,array_pos,obj_x,obj_y,ul_x,ul_y,
		                                                 width,height)

Widget sky_widget;
int array_pos;
int obj_x, obj_y;
int ul_x, ul_y;
int width, height;

{
  int objtype;
  int safety_margin;

/* get the major object type */
  objtype = ngc_array[array_pos].objtype;

#ifdef DEBUG
  printf("draw_obj called to draw figure type %d at %d,%d\n",objtype,
	                                                         obj_x,obj_y);
#endif

#ifdef SUPPRESS_NONEXISTENT
/* if this is a non-existent object in the RNGC, suppress the display */
  if (objtype == NONEXISTENT_IN_RNGC)
    return;
#endif

/* figure a safety margin - XClearArea() seems to clear a tad too much when
 * erasing IDs */
  if (objtype == GALAXY)
    safety_margin = 4;
  else if ((objtype == OPEN_CLUSTER) || (objtype == GLOBULAR_CLUSTER))
    safety_margin = 3;
  else
    safety_margin = 2;

 /* draw the object even if it is only partially in the box */
  if (((obj_x + (OBJSIZE + 1) / 2 + safety_margin) >= ul_x) &&
            ((obj_x - (OBJSIZE + 1) / 2 - safety_margin) <= (ul_x + width)) &&
            ((obj_y + (OBJSIZE + 1) / 2 + safety_margin) >= ul_y) &&
            ((obj_y - (OBJSIZE + 1) / 2 - safety_margin) <= (ul_y + height)))
    ;
  else
    return;

/* draw a different shape for different types of objects */
  if (objtype == GALAXY)
    /* draw an ellipse for a galaxy */
    draw_object(sky_widget,obj_x,obj_y,OBJSIZE,ELLIPSE);
  else if ((objtype == OPEN_CLUSTER) || (objtype == GLOBULAR_CLUSTER))
    /* draw a circle for open and globular clusters */
    draw_object(sky_widget,obj_x,obj_y,OBJSIZE,CIRCLE);
  else
    /* for the moment, draw a square for anything else */
    draw_object(sky_widget,obj_x,obj_y,OBJSIZE,SQUARE);

  return;
}



/* draw one NGC object on the PostScript display */

static boolean ps_draw_ngc_obj(display,array_pos,x,y)

struct display *display;
int array_pos;
int x, y;

{
  int objtype;
  ObjShape shape;

/* get the major object type */
  objtype = ngc_array[array_pos].objtype;

#ifdef SUPPRESS_NONEXISTENT
/* if this is a non-existent object in the RNGC, suppress the display */
  if (objtype == NONEXISTENT_IN_RNGC)
    return(TRUE);
#endif

/* draw the object */
  if (objtype == GALAXY)
    /* draw an ellipse for a galaxy */
    shape = ELLIPSE;
  else if ((objtype == OPEN_CLUSTER) || (objtype == GLOBULAR_CLUSTER))
    /* draw a circle for open and globular clusters */
    shape = CIRCLE;
  else
    /* for the moment, draw a square for anything else */
    shape = SQUARE;

/* draw the object in the correct position with the correct size */
  return(ps_draw_object(display,(double)x,(double)y,(double)OBJSIZE,shape));
}



/* display the catalog data for the given object */

static boolean display_ngc(ngc_fd,ngc_array_pos)

FILE *ngc_fd;
int ngc_array_pos;

{
/* get info from the NGC catalog */
  if (fseek(ngc_fd,ngc_array[ngc_array_pos].filepos,SEEK_SET) == -1)
      return(FALSE);
  if (fgets(ngc_rec,sizeof(ngc_rec),ngc_fd) == (char *)NULL)
      return(FALSE);

/* format the record into an informational display */
  format_ngc_display(ngc_rec);

  return(TRUE);
}



/* find the NGC/IC object in the buffer and update the display coordinates */

static boolean find_ngc_obj(buffer)

char *buffer;

{
  int ngc_ic_flag;
  int ngc_ic_no;
  int ngc_ic_idx;

/* check to see if this is an NGC/IC designation and step to the number */
  if ((strncmp(buffer,"NGC",3) == EQUAL) || strncmp(buffer,"ngc",3) == EQUAL) {
    buffer = &buffer[3];
    ngc_ic_flag = NGC_ID_CODE;
  }
  else if ((strncmp(buffer,"RNGC",4) == EQUAL) ||
	                                   strncmp(buffer,"rngc",4) == EQUAL) {
    buffer = &buffer[4];
    ngc_ic_flag = NGC_ID_CODE;
  }
  else if ((strncmp(buffer,"IC",2) == EQUAL) ||
	                                     strncmp(buffer,"ic",2) == EQUAL) {
    buffer = &buffer[2];
    ngc_ic_flag = IC_ID_CODE;
  }
  else if (find_ngc_by_name(buffer,find_ngc_obj))
    /* this was a recognized object name */
    return(TRUE);
  else
    /* try a Messier designation */
    return(find_messier_obj(buffer,find_ngc_obj));

  /* skip white space */
  while (isspace(*buffer))
    buffer++;

/* get the NGC/IC number */
  ngc_ic_no = atoi(buffer);
  if (ngc_ic_flag == IC_ID_CODE)
    ngc_ic_no += IC_NUM_OFFSET;

/* find the NGC/IC number */
  ngc_ic_idx = binsearch((void *)ngc_array,ngc_array_max,(void *)&ngc_ic_no,
		                                          ngc_search_compare);

  /* return FALSE if we didn't find the NGC/IC object */
  if (ngc_ic_idx == -1)
    return(FALSE);
  else {
/* we have a match; update the display center */
    update_display_pos(ngc_array[ngc_ic_idx].ra_rad,
		                               ngc_array[ngc_ic_idx].dec_rad);
    return(TRUE);
  }

/* NOTREACHED */
}



/* comparison routine for heapsort to sort ngc_array */

static int ngc_sort_compare(ngc_entry_1,ngc_entry_2)

struct ngc_node *ngc_entry_1, *ngc_entry_2;

{
  int key_1, key_2;

/* the key is the object number, with an offset added to IC for uniqueness */
  key_1 = ngc_entry_1->cat_number;
  if (ngc_entry_1->ngc_ic_flag == IC_ID_CODE)
    key_1 += IC_NUM_OFFSET;
  key_2 = ngc_entry_2->cat_number;
  if (ngc_entry_2->ngc_ic_flag == IC_ID_CODE)
    key_2 += IC_NUM_OFFSET;

/* return the key comparison result */
  return(key_1 - key_2);
}



/* comparison routine for binary search to find NGC/IC number in ngc_array */

static int ngc_search_compare(table,key,idx)

void *table;
void *key;
int idx;

{
  int ngc_ic_number;

/* construct the object number key */
  ngc_ic_number = ((struct ngc_node *)table)[idx].cat_number;
  if (((struct ngc_node *)table)[idx].ngc_ic_flag == IC_ID_CODE)
    ngc_ic_number += IC_NUM_OFFSET;

/* return the key comparison result */
  return(*(int *)key - ngc_ic_number);
}



/* format a nice name for the NGC/IC object */

/* ARGSUSED */

static void format_ngc_name(ngc_pos,buffer,font)

int ngc_pos;
char *buffer;
IDfont *font;

{
  int ngc_ic_no;
  int messier_no;

/* get the NGC/IC number */
  ngc_ic_no = ngc_array[ngc_pos].cat_number;
  if (ngc_array[ngc_pos].ngc_ic_flag == IC_ID_CODE)
    ngc_ic_no += IC_NUM_OFFSET;

/* see if this is a Messier object */
  if ((messier_no = find_messier_number(ngc_ic_no)) == -1)
    /* this is not a Messier object */
    if (ngc_array[ngc_pos].ngc_ic_flag == NGC_ID_CODE)
      sprintf(buffer,"NGC %d",ngc_array[ngc_pos].cat_number);
    else
      sprintf(buffer,"IC %d",ngc_array[ngc_pos].cat_number);
  else
    /* use the Messier designation */
    sprintf(buffer,"M %d",messier_no);

  return;
}
