/*
 * xsky - an interactive sky atlas
 *
 * Copyright 1995, Terry R. Friedrichsen
 *
 * This program may be copied and redistributed, in whole or in part,
 * as long as you don't try to make any money from the sale or redis-
 * tribution of the program or any part of the program, or pretend
 * that you wrote the program or any of its parts unless specifically
 * credited by the original author.
 *
 * You are free to make use of this software in your own programs, as
 * long as you credit the original author where it is due.
 */

/*
 * WARRANTY:
 * xsky was written as a learning project and as a demonstration of
 * X Window System programming.  xsky doesn't do anything; it is not
 * merchantable, and it is not fit for any purpose whatsoever.  In
 * fact, don't use xsky at all; it's free, and you're getting what
 * you paid for.
 */

/* this routine formats and displays an NGC record */

#include <stdio.h>
#include <ctype.h>
#include <string.h>

#include "ngc2000.h"

#include "skydefs.h"

#include "catreader.h"

#include "ngc_ic_names.h"

#include "format.h"

/* static function prototypes */
static void display_type PROTOTYPE((char *));
static void display_pos PROTOTYPE((char *));

/* external function prototypes */
extern int find_messier_number PROTOTYPE((int));
extern int find_const_abbrev PROTOTYPE((char *));
extern char *find_ngc_common_names PROTOTYPE((int));
extern void display_ngc_desc PROTOTYPE((smallint,int,char *));

/* table of NGC type code expansions */
extern struct ngc_type ngc_type_codes[];

/* table of common names of NGC objects */
extern struct ngc_ic_name *ngc_ic_names;

extern char *constellations[];

#define MAX_NGC_ROW   70
#define NUM_NGC_ROWS  16



void format_ngc_display(record)

char *record;

{
  int reclen;
  char field[CAT_LENGTH + 1];
  char *catptr;
  int ngc_ic_no;
  smallint ngc_ic_flag;
  int i;
  char *name;

/* start at the beginning of the display */
  idx = 0;
  row = 0;
  row_start = 0;
  max_row = MAX_NGC_ROW;

/* set the record length */
  reclen = NGC_RECLEN + 2;

/* get the catalog designation */
  GET_FIELD(field,record,CAT_START,CAT_LENGTH);

/* get the catalog ID and catalog number */
  catptr = strtok(record," ");
  ngc_ic_no = atoi(strtok((char *)NULL," "));

/* set the NGC/IC object type flag */
  if (strcmp(catptr,"NGC") == EQUAL)
    ngc_ic_flag = NGC_ID_CODE;
  else
    ngc_ic_flag = IC_ID_CODE;

/* display the catalog designation */
  sprintf(&obj_info[idx],"%s %d",catptr,ngc_ic_no);
  idx += strlen(&obj_info[idx]);
  advance_to_tab();
  if ((ngc_ic_flag == IC_ID_CODE) || (ngc_ic_no < 1000))
    advance_to_tab();

/* find the corresponding Messier number, if any */
  if (ngc_ic_flag == NGC_ID_CODE)
    if ((i = find_messier_number(ngc_ic_no)) != -1) {
      sprintf(&obj_info[idx],"M %d",i);
      idx += strlen(&obj_info[idx]);
      advance_to_tab();
    }

/* now display the object type */
  display_type(&record[TYPE_START]);

/* add the constellation */
  GET_FIELD(field,record,CONSTELL_START,CONSTELL_LENGTH);
  i = find_const_abbrev(field);
  if (i >= 0) {
    sprintf(&obj_info[idx]," in %s",constellations[i]);
    idx += strlen(&obj_info[idx]);
  }
  
/* end the line */
  next_line();

/* show the common names, if any */
  if ((name = find_ngc_common_names(ngc_ic_no)) != (char *)NULL) {
    advance_to_tab();
    add_string(name);
    next_line();
  }

/* space down some */
  next_line();

/* display the position */
  display_pos(record);

/* now display the magnitude */
  if (strncmp(&record[MAG_START],"    ",MAG_LENGTH) != EQUAL) { 
    blank_line();
    sprintf(&obj_info[idx],"Mag:  %.4s",&record[MAG_START]);
    idx += strlen(&obj_info[idx]);

    /* indicate visual or photographic magnitude */
    if (record[MAG_CODE_START] == 'p')
      add_string(" - photographic");
    else
      add_string(" - visual");
  }

/* and the size */
  if (strncmp(&record[SIZE_START],"     ",SIZE_LENGTH) != EQUAL) { 
    blank_line();
    sprintf(&obj_info[idx],"Size:  %.5s arcminutes",&record[SIZE_START]);
    idx += strlen(&obj_info[idx]);

    /* indicate whether this is an upper limit */
    if (record[SIZE_LIM_START] == '<')
      add_string(" (upper limit)");
  }

/* add in the description */
  blank_line();
  display_ngc_desc(ngc_ic_flag,ngc_ic_no,&record[DESC_START]);
  idx += strlen(&obj_info[idx]);

/* fill the display out to the correct number of lines */
  while (row < NUM_NGC_ROWS)
    next_line();
  obj_info[idx] = '\0';

/* and go back */
  return;
}



static void display_pos(record)

char *record;

{
/* identify the epoch */
  sprintf(&obj_info[idx],"Position (2000.0)");
  idx += strlen(&obj_info[idx]);

/* display the right ascension */
  advance_to_tab();
  sprintf(&obj_info[idx]," RA:   %.2sh %.2sm",&record[RA_HOURS_START],
	                                               &record[RA_MIN_START]);
  idx += strlen(&obj_info[idx]);
  if (record[RA_MIN_START + 3] != ' ') {
    sprintf(&obj_info[idx],".%.1s",&record[RA_MIN_START + 3]);
    idx += strlen(&obj_info[idx]);
  }
  next_line();

/* and display the declination */
  advance_to_tab();
  advance_to_tab();
  advance_to_tab();
  sprintf(&obj_info[idx],"Dec:  %.3sd %.2s'",&record[DEC_SIGN_START],
	                                              &record[DEC_MIN_START]);
  idx += strlen(&obj_info[idx]);
  next_line();

  return;
}



static void display_type(typeptr)

char *typeptr;

{
  int reclen;
  char field[TYPE_LENGTH + 1];
  char *ptr;
  int i;

/* set the record length */
  reclen = NGC_RECLEN + 2;

/* isolate the type field */
  GET_FIELD(field,typeptr,0,TYPE_LENGTH);

  /* skip leading blanks */
  ptr = field;
  while (*ptr == ' ')
    ptr++;

/* loop through the type code table, looking for a match */
  i = 0;
  while (ngc_type_codes[i].code != (char *)NULL)
    if (strcmp(ngc_type_codes[i].code,ptr) == EQUAL)
      break;
    else
      i++;

/* complain about an unknown type code if we fall out of the loop */
  if (ngc_type_codes[i].code == (char *)NULL)
    printf("unknown NGC type code %s\n",ptr);

/* display the type code's description */
  add_string(ngc_type_codes[i].description);

  return;
}
