/*
 * xsky - an interactive sky atlas
 *
 * Copyright 1995-6, Terry R. Friedrichsen
 *
 * This program may be copied and redistributed, in whole or in part,
 * as long as you don't try to make any money from the sale or redis-
 * tribution of the program or any part of the program, or pretend
 * that you wrote the program or any of its parts unless specifically
 * credited by the original author.
 *
 * You are free to make use of this software in your own programs, as
 * long as you credit the original author where it is due.
 */

/*
 * WARRANTY:
 * xsky was written as a learning project and as a demonstration of
 * X Window System programming.  xsky doesn't do anything; it is not
 * merchantable, and it is not fit for any purpose whatsoever.  In
 * fact, don't use xsky at all; it's free, and you're getting what
 * you paid for.
 */

/* this routine prints out English for the new-style RNGC object
 * descriptions.  the algoritm is to grab each piece delimited
 * by commas, then match the longest possible piece with the
 * RNGC description code table.  that piece's translation is
 * then output and the piece is discarded. */

#include <stdio.h>
#include <ctype.h>

#include <stdlib.h>

#include <string.h>

#include "ngc2000.h"

#include "skydefs.h"

#include "ngc2000_desc.h"

#include "format.h"

/* static function prototypes */
static int find_longest_match PROTOTYPE((char *,int *,struct key_desc *,
					                          boolean *));
static int read_ambig_table PROTOTYPE((struct ambig [],char *));

/* external function prototypes */
extern char *build_filespec PROTOTYPE((char *,char *));

/* table of description abbreviations and translations */
static struct key_desc desc_table[MAXNUMDESCS + 2 + 1];
static int max_desc;

/* indices of ambiguity-resolving descriptions */
static int l_idx = -1;
static int s_idx = -2;

/* tables of NGC2000 objects which need ambiguity resolution */
static struct ambig l_ambig_table[NUM_L_AMBIG], s_ambig_table[NUM_S_AMBIG];
static int max_l_ambig_idx, max_s_ambig_idx;



#ifdef STANDALONE

void read_ngc_desc_table PROTOTYPE((void));
void display_ngc_desc PROTOTYPE((smallint,int,char *));

char obj_info[2400];



main(argc,argv)

int argc;
char *argv[];

{
  int catlen;
  int find_num;
  FILE *ngc_fd;
  char ngc_rec[NGC_RECLEN + 2 + 1];
  char *cat_id;
  smallint ngc_ic_flag;
  int ngc_ic_num;

/* handle arguments */
  find_num = 0;
  if (argc > 1) {
    catlen = strspn(argv[1],"NGCIC");
    if (strspn(&argv[1][catlen],"0123456789") == strlen(&argv[1][catlen]))
      find_num = atoi(&argv[1][catlen]);
  }

/* read in the description table */
  read_ngc_desc_table();

/* open the NGC data file */
  if ((ngc_fd = fopen("/cd/NONSTELL/GALAXIES/NGC2000/NGC2000.DAT","r")) ==
                                                               (FILE *)NULL) {
    printf("can't open /cd/NONSTELL/GALAXIES/NGC2000/NGC2000.DAT\n");
    perror("");
    exit(1);
  }

/* build the description on a single line */
  max_row = 2400;

/* if there is an argument, and it is not all numeric, presume it is an NGC/IC
 * description to translate */
  if ((argc > 1) && (find_num == 0)) {
    idx = row = row_start = 0;
    display_ngc_desc(NGC_ID_CODE,0,argv[1]);
    printf("%s\n",obj_info);

    exit(0);
  }

/* loop through the entire NGC file */
  while (fgets(ngc_rec,sizeof(ngc_rec),ngc_fd) != (char *)NULL) {
    /* get the NGC/IC number and the object type */
    cat_id = strtok(ngc_rec," ");
    ngc_ic_num = atoi(strtok((char *)NULL," "));

    /* set the catalog flag appropriately */
    if (strcmp(cat_id,"NGC") == EQUAL)
      ngc_ic_flag = NGC_ID_CODE;
    else
      ngc_ic_flag = IC_ID_CODE;

    /* if this isn't the NGC/IC number we want, skip it */
    if ((find_num > 0) && (ngc_ic_num != find_num))
      continue;

    /* build the description string and emit it */
    idx = row = row_start = 0;
    display_ngc_desc(ngc_ic_flag,ngc_ic_num,&ngc_rec[DESC_START]);
    printf("%s %d\t%s\n\n",cat_id,ngc_ic_num,obj_info);
  }

/* be neat and close the file */
  (void)fclose(ngc_fd);

  exit(0);

/* NOTREACHED */
}

#endif



void display_ngc_desc(cat_flag,cat_num,record)

smallint cat_flag;
int cat_num;
char *record;

{
/* record is a pointer to the start of the NGC description text */
  int i, j, k;
  char descbuff[MAXDESCLEN + 1];
  int descidx;
  boolean nomatch_flag;
  boolean stars_flag;
  boolean this_num_flag, last_num_flag;

/* strip trailing <cr><lf> from the record */
  i = strlen(record) - 1;
  while ((record[i] == '\n') || (record[i] == '\r'))
    if (i < 0)
      break;
    else
      record[i--] = '\0';

/* strip trailing whitespace from the record */
  if (i >= 0)
    while (isspace(record[i]))
      if (i-- == 0)
	break;
  record[++i] = '\0';

#ifndef STANDALONE
/* output a header */
  add_string("NGC 2000.0 description:  ");

/* show the actual raw description string */
  add_string(record);

/* position for start of expanded description text */
  blank_line();
  advance_to_tab();
#endif

/* assume we will find matches */
  nomatch_flag = FALSE;

/* we haven't seen a "stars" description or a numeric value yet */
  stars_flag = this_num_flag = last_num_flag = FALSE;

/* loop through the description buffer */
  i = 0;
  while (TRUE) {
    /* skip any white space to get to descriptions */
    while (isspace(record[i]))
      i++;

    /* stop if we reach the null */
    if (record[i] == '\0')
      break;

/* copy a description piece until a comma, semicolon, null, or whitespace */
    j = 0;
    while ((record[i] != ',') && (record[i] != ';') &&
	                        (record[i] != '\0') && (! isspace(record[i])))
      descbuff[j++] = record[i++];
    /* end the description buffer with a null */
    descbuff[j] = '\0';

/* now loop through the description, piecing out the code words */
    j = 0;
    while (j < strlen(descbuff)) {
/* now match the largest piece possible from the description table */
      last_num_flag = this_num_flag;
      descidx = find_longest_match(descbuff,&j,desc_table,&this_num_flag);

/* did this piece of description match something in the table? */
      if ((descidx == max_desc) && nomatch_flag)
	/* no, and neither did the last one; print with no preceding space */
	;
      else if (desc_table[descidx].desc_string[0] == '?')
	/* also, don't space in front of a question mark */
	;
      else
	/* precede the piece of description by a space unless space already */
	if ((idx == 0) || (obj_info[idx - 1] == ' '))
	  ;
        else
	  if (FIT(" "))
	    add_string(" ");
          else
	    next_line();

/* put out the expansion of this piece of the description, taking note of
 * special cases */
      if (descidx == l_idx) {
	for (k = 0; k < max_l_ambig_idx; k++)
	  if ((cat_flag == l_ambig_table[k].ngc_ic_flag) &&
	                              (cat_num == l_ambig_table[k].cat_number))
	    break;

	/* if we fell out of the loop, this one isn't in the "long" table */
        if (k >= max_l_ambig_idx) {
	  if (! FIT("little"))
	    next_line();
	  add_string("little");
	}	  
        else {
	  if (! FIT("long"))
	    next_line();
	  add_string("long");
	}
      }
      else if (descidx == s_idx)
	if (last_num_flag) {
	  if (! FIT("seconds"))
	    next_line();
	  add_string("seconds");
	}
	else {
	  for (k = 0; k < max_s_ambig_idx; k++)
	    if ((cat_flag == s_ambig_table[k].ngc_ic_flag) &&
		                      (cat_num == s_ambig_table[k].cat_number))
	      break;

	  /* if we fell out of the loop, this one isn't in the "south" table */
	  if (k >= max_s_ambig_idx) {
	    if (! FIT("suddenly"))
	      next_line();
	    add_string("suddenly");
	  }
	  else {
	    if (! FIT("south"))
	      next_line();
	    add_string("south");
	  }
	}
      else {
	if (! FIT(desc_table[descidx].desc_string))
	  next_line();
	add_string(desc_table[descidx].desc_string);
      }

/* set nomatch flag according to whether this piece matched or not */
      nomatch_flag = (descidx == max_desc);

/* if the "stars" flag is set and this description is numeric, the number is
 * a magnitude */
      if (stars_flag && this_num_flag) {
	if (! FIT("mag"))
	  next_line();
	add_string(" mag");

	/* forget the fact that this description is numeric */
	this_num_flag = FALSE;
      }

/* if the code word was "*" or "st", set the "stars" flag for next time
 * (however, if "*" is not immediately followed by something with no
 * intervening spaces, don't set the "stars" flag) */
      if (strcmp(desc_table[descidx].desc_name,"st") == EQUAL)
	stars_flag = TRUE;
      else if (strcmp(desc_table[descidx].desc_name,"*") == EQUAL)
	if (strlen(descbuff) == 1)
	  stars_flag = FALSE;
        else
	  stars_flag = TRUE;
      else
	  stars_flag = FALSE;
    }

/* put out the delimiter unless it is a blank or a null */
    if (record[i] == '\0')
      /* we are done when we see the null */
      break;
    else if (isspace(record[i]))
      if (nomatch_flag) {
	/* put out the delimiter anyway */
	sprintf(&obj_info[idx],"%c",record[i]);
	idx += strlen(&obj_info[idx]);
      }
      else
	;
    else {
      sprintf(&obj_info[idx],"%c",record[i]);
      idx += strlen(&obj_info[idx]);

      /* reset the flags for the next piece of the description */
      nomatch_flag = stars_flag = this_num_flag = last_num_flag = FALSE;
    }

/* step past the delimiter to the next piece of the description */
    i++;
  }

  return;
}



/* buffer for reading auxiliary data files */

static char inbuf[BUFSIZ + 1];


void read_ngc_desc_table()

{
  char *filebuff;
  FILE *desc_fd;
  int desc;
  int i;

/* build the file path */
  filebuff = build_filespec("NGC_DESC_PATH",NGC_DESC_FILE);

/* open the descriptor file */
  if ((desc_fd = fopen(filebuff,"r")) == (FILE *)NULL) {
    printf("sky:  error opening NGC descriptor file %s\n",filebuff);
    perror("sky");
    exit(1);
  }

/* loop through the file, reading in descriptor table entries */
  desc = 0;
  while (fgets(inbuf,BUFSIZ,desc_fd) != NULL) {
    /* remove the newline at the end */
    inbuf[strlen(inbuf) - 1] = '\0';
    /* watch out for empty lines */
    if (inbuf[0] == '\0')
      continue;

/* copy the description abbreviation until white space */
    i = 0;
    while (! isspace(inbuf[i])) {
      desc_table[desc].desc_name[i] = inbuf[i];
      i++;
    }

    /* watch out for lines of blanks */
    if (i == 0)
      continue;

    /* be sure the abbreviation ends with a null */
    desc_table[desc].desc_name[i] = '\0';

/* skip to the description string */
    while (isspace(inbuf[i]))
      i++;

/* get some space for the description string */
    desc_table[desc].desc_string = (char *)malloc(strlen(&inbuf[i]) + 1);

/* copy in the description string */
    strcpy(desc_table[desc].desc_string,&inbuf[i]);

/* if this is one of the ambiguity resolvers, save the index */
    if ((l_idx == -1) && (strcmp(desc_table[desc].desc_name,"l") == EQUAL)) {
      l_idx = desc;
      s_idx = -1;
    }
    if ((s_idx == -1) && (strcmp(desc_table[desc].desc_name,"s") == EQUAL))
      s_idx = desc;

/* step to the next table entry */
    desc++;
  }

/* be neat */
  (void)fclose(desc_fd);

/* free the filename buffer */
  free((void *)filebuff);

/* make the last description be some question marks */
  strcpy(desc_table[desc].desc_name,"\\");
  desc_table[desc].desc_string = (char *)malloc(strlen("???") + 1);
  strcpy(desc_table[desc].desc_string,"???");

/* remember the number of descriptions read */
  if ((max_desc = desc) >= MAXNUMDESCS) {
    printf("sky:  too many descriptions in NGC/IC description table\n");
    exit(1);
  }

/* read in the "l" and "s" ambiguity resolution tables */
  max_l_ambig_idx = read_ambig_table(l_ambig_table,NGC_L_AMBIG_FILE);
  max_s_ambig_idx = read_ambig_table(s_ambig_table,NGC_S_AMBIG_FILE);

  return;
}



/* read a table of NGC/IC numbers which need modified expansion */

static int read_ambig_table(table,file)

struct ambig table[];
char *file;

{
  char *filebuff;
  FILE *ambig_fd;
  int i;
  char *ptr;

/* build the file path */
  filebuff = build_filespec("NGC2000_PATH",file);

/* open the descriptor file */
  if ((ambig_fd = fopen(filebuff,"r")) == (FILE *)NULL) {
    printf("sky:  error opening NGC index file %s\n",filebuff);
    perror("sky");
    exit(1);
  }

/* loop through the file, reading in NGC/IC object numbers */
  i = 0;
  while (fgets(inbuf,BUFSIZ,ambig_fd) != NULL) {
    ptr = inbuf;

    /* save the NGC/IC flag */
    if (strncmp(ptr,"NGC",3) == EQUAL)
      table[i].ngc_ic_flag = NGC_ID_CODE;
    else
      table[i].ngc_ic_flag = IC_ID_CODE;

    /* save the catalog number */
    while (! isspace(*ptr++))   ;
    table[i++].cat_number = atoi(ptr);
  }

/* be neat */
  (void)fclose(ambig_fd);

/* free the filename buffer */
  free((void *)filebuff);

/* return the maximum ambiguous-description table entry */
  return --i;
}



static int find_longest_match(buff,ind,table,numflag)

char *buff;
int *ind;
struct key_desc *table;
boolean *numflag;

{
  int max_match, max_match_idx;
  char *ptr;
  int desc;
  int i, j;

/* assume this is not a numeric token */
  *numflag = FALSE;

/* make the match length impossibly small */
  max_match = -1;
  max_match_idx = -1;

/* begin at the beginning */
  desc = 0;
  while (desc <= max_desc) {

/* point to the start of the description */
    ptr = &buff[*ind];

/* match characters for a while */
    i = strlen(table[desc].desc_name);
    if (strncmp(ptr,table[desc].desc_name,(size_t)i) == EQUAL)
      if (i > max_match) {

/* the new match is bigger; save the length and the index */
	max_match = i;
	max_match_idx = desc;
      }

/* step to the next description */
    if (table[++desc].desc_name[0] == '\\')
      break;
  }

/* if no match was found, update to bypass a bogus piece of description
 * and set up the unrecognized string as the output */
  if (max_match_idx == -1) {

    /* if the non-match is numeric, grab the entire number */
    if (isdigit(buff[*ind])) {
      i = strspn(&buff[*ind],"-0123456789.\'\"");
      j = 0;
      while (i-- > 0)
	table[desc].desc_string[j++] = buff[(*ind)++];
      table[desc].desc_string[j] = '\0';

      /* set the numeric flag for the caller */
      *numflag = TRUE;
    }
    else {
      /* otherwise, just grab this one character */
      table[desc].desc_string[0] = buff[(*ind)++];
      table[desc].desc_string[1] = '\0';
    }

    /* make the matching index be this one */
    max_match_idx = desc;
  }
  else
/* a match was found; update the index by the 1-relative length */
    *ind += max_match;

/* return the index to the relevant description table entry */
  return(max_match_idx);
}
