/*
 * xsky - an interactive sky atlas
 *
 * Copyright 1992-6, Terry R. Friedrichsen
 *
 * This program may be copied and redistributed, in whole or in part,
 * as long as you don't try to make any money from the sale or redis-
 * tribution of the program or any part of the program, or pretend
 * that you wrote the program or any of its parts unless specifically
 * credited by the original author.
 *
 * You are free to make use of this software in your own programs, as
 * long as you credit the original author where it is due.
 */

/*
 * WARRANTY:
 * xsky was written as a learning project and as a demonstration of
 * X Window System programming.  xsky doesn't do anything; it is not
 * merchantable, and it is not fit for any purpose whatsoever.  In
 * fact, don't use xsky at all; it's free, and you're getting what
 * you paid for.
 */

#include <stdio.h>

#include <stdlib.h>

#include <string.h>

#include <math.h>

#include <X11/Intrinsic.h>

#include "skydefs.h"
#include "catalog.h"
#include "draw.h"
#include "pos.h"

/* static function prototypes */
static void mark_id_texts PROTOTYPE((int,int,int,int));

/* external function prototypes */
extern void draw_id PROTOTYPE((Widget,struct id_node *));
extern void clear_area PROTOTYPE((Widget,int,int,int,int));
extern void calculate_limits PROTOTYPE((struct display *));
extern void calculate_boundary_field PROTOTYPE((struct display *));
extern void pos_rad_to_X PROTOTYPE((struct display *,double,double,
				                                int *,int *));
extern boolean within_display PROTOTYPE((struct display *,double,double,
					                        int *,int *));
extern void get_text_extents PROTOTYPE((char *,IDfont,
					            short *,short *,short *));
extern void X_to_pos_rad PROTOTYPE((struct display *,int,int,
				                          double *,double *));
extern void bad_init_file PROTOTYPE((char *));
extern struct cat_header *find_catalog_by_menu PROTOTYPE((char *));

/* global function prototypes */
struct obj_node *find_object PROTOTYPE((int,int,struct cat_header **));
void remove_this_id PROTOTYPE((Widget,struct id_node *));
void add_id_to_list PROTOTYPE((struct id_node *));

/* freelist pointer for displayable object list nodes */
static struct obj_node *freelist = (struct obj_node *)NULL;

/* list header for list of visual identifications */
struct id_node *id_list = (struct id_node *)NULL;

/* list of catalog header structures */
extern struct cat_header *cat_list_head;

/* limiting magnitude of the display */
extern float mag_limit;

/* object proximity distance for buttonclick selection */
#define OBJ_PROXIMITY   4



/* draw all identification strings in the list that fall in the given box */

void draw_id_texts(sky_widget,x,y,width,height)

Widget sky_widget;
int x, y;
int width, height;

{
  struct id_node *id;

/* step through the ID list, redrawing all interfering ID strings */
  id = id_list;
  while (id != (struct id_node *)NULL) {
    /* display the ID only if enabled */
    if (id->show_flag)
      /* see of the exposed area includes part of this text */
      if ((id->ul_x <= x) && (id->lr_x >= x) ||
	          (id->ul_x <= (x + width)) && (id->lr_x >= (x + width)) ||
	                         (id->ul_x >= x) && (id->lr_x <= (x + width)))
	/* the x dimension overlaps */
	if ((id->ul_y <= y) && (id->lr_y >= y) ||
	        (id->ul_y <= (y + height)) && (id->lr_y >= (y + height)) ||
	                        (id->ul_y >= y) && (id->lr_y <= (y + height)))
	  /* the y dimension overlaps, too */
	  draw_id(sky_widget,id);

    /* step to the next ID */
    id = id->next;
  }

  return;
}



/* mark all identification strings in the list that fall in the given box */

static void mark_id_texts(x,y,width,height)

int x, y;
int width, height;

{
  struct id_node *id;

/* step through the ID list, marking all interfering ID strings */
  id = id_list;
  while (id != (struct id_node *)NULL) {
    /* assume we won't have to redraw this ID */
    id->redraw_flag = FALSE;

    /* worry about the ID only if enabled */
    if (id->show_flag)
      /* see of the exposed area includes part of this text */
      if ((id->ul_x <= x) && (id->lr_x >= x) ||
	          (id->ul_x <= (x + width)) && (id->lr_x >= (x + width)) ||
	                         (id->ul_x >= x) && (id->lr_x <= (x + width)))
	/* the x dimension overlaps */
	if ((id->ul_y <= y) && (id->lr_y >= y) ||
	        (id->ul_y <= (y + height)) && (id->lr_y >= (y + height)) ||
	                        (id->ul_y >= y) && (id->lr_y <= (y + height)))
	  /* the y dimension overlaps, too */
	  id->redraw_flag = TRUE;

    /* step to the next ID */
    id = id->next;
  }

  return;
}



/* draw all marked identification strings */

void draw_marked_ids(sky_widget)

Widget sky_widget;

{
  struct id_node *id;

/* step through the ID list, redrawing all interfering ID strings */
  id = id_list;
  while (id != (struct id_node *)NULL) {
    /* display the ID only if enabled */
    if ((id->show_flag) && (id->redraw_flag))
      draw_id(sky_widget,id);

    /* clear the redraw flag */
    id->redraw_flag = FALSE;

    /* step to the next ID */
    id = id->next;
  }

  return;
}



/* add the ID text to the ID display list */

void add_id_to_list(id)

struct id_node *id;

{
/* link to the front of the visible ID list */
  id->next = (struct id_node *)id_list;
  id_list = id;

  return;
}



/* remove an ID from the list and clear the text from the screen */

void remove_this_id(sky_widget,id)

Widget sky_widget;
struct id_node *id;

{
  struct id_node *prev_id;

/* find this ID in the list */
  if (id_list == id)
    prev_id = (struct id_node *)NULL;
  else {
    prev_id = id_list;
    while (prev_id != (struct id_node *)NULL)
      if (prev_id->next == id)
	break;
      else
	prev_id = prev_id->next;

    /* if we didn't find the ID, fail silently */
    if (prev_id == (struct id_node *)NULL)
      return;
  }

/* remove this ID from the list */
  if (prev_id == (struct id_node *)NULL)
    /* the given node is at the front of the list */
    id_list = id_list->next;
  else
    /* otherwise, bypass the given node in the list */
    prev_id->next = id->next;

/* clear the area taken up by the string */
  clear_area(sky_widget,id->ul_x,id->ul_y,
	                     id->lr_x - id->ul_x + 1,id->lr_y - id->ul_y + 1);

/* mark all the IDs this ID may have obscured */
  mark_id_texts(id->ul_x,id->ul_y,id->lr_x - id->ul_x,id->lr_y - id->ul_y);

  return;
}



/* remove all IDs from the list */

void remove_all_ids()

{
  struct id_node *id, *next_id;

/* start at the beginning of the ID list */
  id = id_list;

/* zap the ID list head */
  id_list = (struct id_node *)NULL;
  
/* free all IDs in the ID list */
  while (id != (struct id_node *)NULL) {
    /* save a pointer to the next ID */
    next_id = id->next;

    /* free the ID node */
    free((void *)id);

    /* step to the next ID */
    id = next_id;
  }

  return;
}



/* find the ID enclosing the given point */

struct id_node *find_id_by_position(x,y)

int x, y;

{
  struct id_node *id;

/* loop through all displayed IDs, looking for one enclosing (x,y) */
  id = id_list;
  while (id != (struct id_node *)NULL)
    /* does this ID enclose the point? */
    if ((x > id->ul_x) && (x < id->lr_x) && (y > id->ul_y) && (y < id->lr_y))
      /* yes, return it to the user */
      break;
    else
      /* step to the next ID node */
      id = id->next;

  return(id);
}



/* find the object within delta of the given position */

struct obj_node *find_object(x,y,catalog)

int x, y;
struct cat_header **catalog;

{
  struct cat_header *cat;
  struct obj_node *obj;

/* figure out which object is close to the given X coordinates */
  cat = cat_list_head;
  while (cat != (struct cat_header *)NULL) {

    /* search the displayed objects list for this catalog */
    obj = cat->obj_list_head;
    while (obj != (struct obj_node *)NULL) {

      /* ignore objects which don't meet the magnitude limit */
      if (cat->mag_limit_check != NULLBOOLFUNCTION)
	if (! (*cat->mag_limit_check)(obj->array_pos,mag_limit)) {
	  /* just step to the next object */
	  obj = obj->next;
	  continue;
	}

      /* check to see if the position is within sight of an object */
      if (sqrt((double)((x - obj->x) * (x - obj->x) +
	                       (y - obj->y) * (y - obj->y))) <= OBJ_PROXIMITY)
	/* use this object */
	break;
      else
	/* step to the next object */
	obj = obj->next;
    }

    /* if we exhausted the list, step to the next catalog */
    if (obj == (struct obj_node *)NULL)
      cat = cat->next_cat;
    else
      /* we found a close-enough object; use it */
      break;
  }

/* tell the caller which catalog we found the object in */
  *catalog = cat;

/* and return a pointer to the object found */
  return(obj);
}



/* decode the object name and try to find the named object */

boolean find_object_by_name(buffer)

char *buffer;

{
  char *ptr1, *ptr2;
  struct cat_header *catalog;

/* skip leading white space */
  while (isspace(*buffer))
    buffer++;

/* squeeze multiple spaces and tabs, and trailing spaces, out of the buffer */
  ptr1 = ptr2 = buffer;
  ptr2++;
  while (*ptr2 != '\0')
    if (isspace(*ptr2)) {
      if (*ptr1 == ' ')
	;
      else
	*++ptr1 = ' ';
      ptr2++;
    }
    else
      *++ptr1 = *ptr2++;

  /* if the trailing character is a space, overwrite it */
  if (*ptr1 == ' ')
    ;
  else
    ptr1++;
  *ptr1 = '\0';

/* ensure there is something in the buffer */
  if (*buffer == '\0')
    return(FALSE);

/* hand the object designation to each active catalog */
  catalog = cat_list_head;
  while (catalog != (struct cat_header *)NULL) {
    if (catalog->show_flag && catalog->find_object != NULLBOOLFUNCTION)
      if ((*catalog->find_object)(buffer))
	break;

    /* step to tne next catalog */
    catalog = catalog->next_cat;
  }

/* do the user a favor and check inactive catalogs if not found */
  if (catalog == (struct cat_header *)NULL) {
    catalog = cat_list_head;
    while (catalog != (struct cat_header *)NULL) {
      if ((! catalog->show_flag) && catalog->find_object != NULLBOOLFUNCTION)
	if ((*catalog->find_object)(buffer))
	  break;

      /* step to tne next catalog */
      catalog = catalog->next_cat;
    }
  }

/* return TRUE or FALSE depending on whether the object was found */
  return(catalog != (struct cat_header *)NULL);
}



/* redraw the objects in the sky in response to an expose event */

void draw_sky(sky_widget,ul_x,ul_y,width,height)

Widget sky_widget;
int ul_x, ul_y;
int width, height;

{
  struct cat_header *catalog;
  struct obj_node *obj;

#ifdef DEBUG
  printf("drawing objects in area %d,%d %d,%d\n",ul_x,ul_y,width,height);
#endif

/* draw objects from each catalog */
  catalog = cat_list_head;
  while (catalog != (struct cat_header *)NULL) {

/* if the catalog is active, draw all objects in its display list */
    if (catalog->show_flag) {
      obj = catalog->obj_list_head;
      while (obj != (struct obj_node *)NULL) {

/* handle display of the object */
	(*catalog->draw_object)(sky_widget,obj->array_pos,
				              (int)obj->x,(int)obj->y,
			                              ul_x,ul_y,width,height);

/* step to the next object */
	obj = obj->next;
      }
    }

/* step to the next catalog */
    catalog = catalog->next_cat;
  }

  return;
}



/* build the lists of displayable objects */

void calculate_field(display)

struct display *display;

{
  struct id_node *id;
  struct cat_header *catalog;
  int x, y;

/* calculate the limits of the display in RA and declination */
  calculate_limits(display);

/* calculate the visible constellation boundaries */
  calculate_boundary_field(display);

/* calculate new lists of objects to display */
  catalog = cat_list_head;
  while (catalog != (struct cat_header *)NULL) {
    /* if there is an old list, free all the nodes */
    if (catalog->obj_list_head != (struct obj_node *)NULL) {
      (catalog->obj_list_rear)->next = freelist;
      freelist = catalog->obj_list_head;
      catalog->obj_list_head = (struct obj_node *)NULL;
      catalog->obj_list_rear = (struct obj_node *)NULL;
    }

    /* use this catalog only if it is active */
    if (catalog->show_flag)
      (*catalog->calculate_field)(display,catalog);

    /* step to the next catalog */
    catalog = catalog->next_cat;
  }

/* update the object identification list */
  id = id_list;
  while (id != (struct id_node *)NULL) {
    /* calculate the new X position of the object */
    pos_rad_to_X(display,id->ra_rad,id->dec_rad,&x,&y);

    /* flip it for the Southern Hemisphere, if necessary */
    if (display->downunder_flag) {
      x = display->width - x;
      y = display->height - y;
    }

    /* if the object is still in the display, update the ID */
    if ((x >= 0) && (x <= display->width) &&
	                                 (y >= 0) && (y <= display->height)) {

      /* flag this ID as displayable */
      id->show_flag = TRUE;

      /* update the ID's text position */
      id->x += x - id->obj_x;
      id->y += y - id->obj_y;

      /* update the ID's bounding box */
      id->ul_x += x - id->obj_x;
      id->ul_y += y - id->obj_y;
      id->lr_x += x - id->obj_x;
      id->lr_y += y - id->obj_y;
      
      /* and update the object's X position */
      id->obj_x = x;
      id->obj_y = y;
    }
    else
      /* flag this ID as not currently displayable */
      id->show_flag = FALSE;

    /* step to the next ID */
    id = id->next;
  }

  return;
}



/* add the given object to the display list if it's visible in the display */

void add_to_display_list(display,catalog,object_id,ra_rad,dec_rad)

struct display *display;
struct cat_header *catalog;
int object_id;
double ra_rad;
double dec_rad;

{
  int x, y;
  struct obj_node *object;

/* see if the object lies within the display */
  if (within_display(display,ra_rad,dec_rad,&x,&y)) {
#ifdef DEBUG
    printf("object %d from catalog %s falls within the display",object_id,
	                                                  catalog->menu_name);
    printf("at %lf, %lf radians\n",ra_rad,dec_rad);
#endif

/* get a node for this object */
    if (freelist == (struct obj_node *)NULL)
      object = (struct obj_node *)malloc(sizeof(struct obj_node));
    else {
      object = freelist;
      freelist = freelist->next;
    }

/* save the object ID (usually array position) of this object */
    object->array_pos = object_id;

/* in the Southern Hemisphere, mirror-image and invert the display */
    if (display->downunder_flag) {
      x = display->width - x;
      y = display->height - y;
    }

/* save the X coordinates of this object */
    object->x = (Position)x;
    object->y = (Position)y;
#ifdef DEBUG
    printf("X coords of object are %d, %d\n",object->x,object->y);
#endif

/* link it into the display list at the rear */
    object->next = (struct obj_node *)NULL;
    if (catalog->obj_list_rear == (struct obj_node *)NULL)
      /* the list is empty */
      catalog->obj_list_head = catalog->obj_list_rear = object;
    else {
      (catalog->obj_list_rear)->next = object;
      catalog->obj_list_rear = object;
    }

#ifdef DEBUG
    printf("object linked into display list\n\n");
#endif
  }

  return;
}



/* return an ID node for the object at (or near) the given position */

struct id_node *get_object_id(x,y)

int x, y;

{
  struct obj_node *obj;
  struct cat_header *catalog;
  struct id_node *id;
  short width;
  short ascent, descent;

/* find the selected object */
  if ((obj = find_object(x,y,&catalog)) == (struct obj_node *)NULL)
    return((struct id_node *)NULL);

/* get an ID node for this object */
  id = (struct id_node *)malloc(sizeof(struct id_node));

/* build the identifying text (assume Latin ID font) */
  id->font = LATIN_ID;
  (*catalog->format_name)(obj->array_pos,id->id_buffer,&id->font);

/* if the object has no designation, we're done */
  if (strlen(id->id_buffer) == 0) {
    free((void *)id);
    return((struct id_node *)NULL);
  }

/* this ID is obviously displayable now */
  id->show_flag = TRUE;

/* initialize this ID's redraw flag */
  id->redraw_flag = FALSE;

/* save the object's catalog and array index */
  id->obj_catalog = catalog;
  id->obj_index = obj->array_pos;

/* save the object's actual RA and declination */
  (*(id->obj_catalog)->get_object_pos)(id->obj_index,&id->ra_rad,&id->dec_rad);

/* save the object's X position */
  id->obj_x = obj->x;
  id->obj_y = obj->y;

/* save the ID box's initial upper-left position */
  id->ul_x = x;
  id->ul_y = y;

/* get the extents of the ID text */
  get_text_extents(id->id_buffer,id->font,&width,&ascent,&descent);

/* remember the lower-right position as an offset for now */
  id->lr_x = width;
  id->lr_y = ascent + descent;

/* save the ID text position as an offset for now */
  id->x = (Position)0;
  id->y = (Position)ascent;

/* return a pointer to the ID */
  return(id);
}



/* return an ID node for for a user-supplied label */

struct id_node *get_userlabel_id(x,y,buffer,font)

int x, y;
char *buffer;
IDfont font;

{
  struct id_node *id;
  short width;
  short ascent, descent;

/* get an ID node for this object */
  id = (struct id_node *)malloc(sizeof(struct id_node));

/* save the requested font */
  id->font = font;

/* save the identifying text */
  strcpy(id->id_buffer,buffer);

/* this ID is obviously displayable now */
  id->show_flag = TRUE;

/* initialize this ID's redraw flag */
  id->redraw_flag = FALSE;

/* null out the catalog and the array index */
  id->obj_catalog = (struct cat_header *)NULL;
  id->obj_index = -1;

/* save the ID box's initial upper-left position */
  id->ul_x = x;
  id->ul_y = y;

/* get the extents of the ID text */
  get_text_extents(id->id_buffer,id->font,&width,&ascent,&descent);

/* save the ID text position as an offset for now */
  id->x = (Position)0;
  id->y = (Position)ascent;

/* remember the lower-right position as an offset for now */
  id->lr_x = width;
  id->lr_y = ascent + descent;

/* return a pointer to the ID */
  return(id);
}



/* prepare to move a label */

void setup_label_move(sky_widget,id)

Widget sky_widget;
struct id_node *id;

{
/* erase this ID and remove it from the ID list, without freeing the ID node */
  remove_this_id(sky_widget,id);

/* convert the lower-right position of the box back to an offset */
  id->lr_x -= id->ul_x;
  id->lr_y -= id->ul_y;

/* convert the text position back to an offset */
  id->x -= id->ul_x;
  id->y -= id->ul_y;

  return;
}



/* finalize the positional information for a label */

void finalize_label_pos(display,id,x,y)

struct display *display;
struct id_node *id;
int x, y;

{
/* save the final position of the box */
  id->ul_x = x;
  id->ul_y = y;

/* if this is a user label, use the upper-left corner as the object position */
  if (id->obj_catalog == (struct cat_header *)NULL) {
    id->obj_x = (Position)id->ul_x;
    id->obj_y = (Position)id->ul_y;

    /* convert the ID box's final upper-left position into RA and dec */
    X_to_pos_rad(display,(int)id->obj_x,(int)id->obj_y,
		                                    &id->ra_rad,&id->dec_rad);
  }

/* calculate the final lower-right position of the box */
  id->lr_x += id->ul_x;
  id->lr_y += id->ul_y;

/* calculate the location of the ID text */
  id->x += id->ul_x;
  id->y += id->ul_y;

  return;
}



/* read a list of IDs from the initialization file and build the initial
 * ID list */

static char idbuff[BUFSIZ + 1 + 1];

void build_id_list(init_fd)

FILE *init_fd;

{
  int i;
  struct id_node *id;
  char catalog_name[20 + 1];
  int obj_x, obj_y;
  int x, y;
  char font_type[5 + 1];
  char show_flag[5 + 1];
  char redraw_flag[5 + 1];
  char *buffp;

/* read ID lines from the initialization file */
  while (fgets(idbuff,sizeof(idbuff),init_fd) != (char *)NULL) {
/* get an ID node for this label */
    id = (struct id_node *)malloc(sizeof(struct id_node));

/* scan in the values for this ID */
    if ((i = sscanf(idbuff,
		    "%s %d %lf %lf %d %d %d %d %d %d %d %d %s %s %s %s\n",
		    catalog_name,
		    &id->obj_index,
		    &id->ra_rad,&id->dec_rad,
		    &obj_x,&obj_y,
		    &x,&y,
		    &id->ul_x,&id->ul_y,
		    &id->lr_x,&id->lr_y,
		    font_type,
		    show_flag,
		    redraw_flag,
		    id->id_buffer)) != 16) {
      printf("catalog name = %s, obj_index = %d\n",catalog_name,id->obj_index);
      printf("sscanf returns %d\n",i);
      bad_init_file("bad ID record");
    }

/* handle the non-integer values (avoid dependence on sizeof(Position)) */
    id->obj_x = (Position)obj_x;
    id->obj_y = (Position)obj_y;
    id->x = (Position)x;
    id->y = (Position)y;

/* handle the catalog menu name */
    if (strcmp(catalog_name,"User") == EQUAL)
      /* user catalog has no list entry */
      id->obj_catalog = (struct cat_header *)NULL;
    else {
      /* search the catalog list for this menu name */
      id->obj_catalog = find_catalog_by_menu(catalog_name);
      if (id->obj_catalog == (struct cat_header *)NULL)
	bad_init_file("bad catalog name");
    }

/* handle the font name */
    if (strcmp(font_type,LATIN_ID_FONT) == EQUAL)
      id->font = LATIN_ID;
    if (strcmp(font_type,LATIN_CON_FONT) == EQUAL)
      id->font = LATIN_CON;
    else if (strcmp(font_type,GREEK_ID_FONT) == EQUAL)
      id->font = GREEK;
    else
      bad_init_file("bad font field in ID");

/* handle the show flag */
    if (strcmp(show_flag,"True") == EQUAL)
      id->show_flag = TRUE;
    else if (strcmp(show_flag,"False") == EQUAL)
      id->show_flag = FALSE;
    else
      bad_init_file("bad show field in ID");

/* handle the redraw flag */
    if (strcmp(redraw_flag,"True") == EQUAL)
      id->redraw_flag = TRUE;
    else if (strcmp(redraw_flag,"False") == EQUAL)
      id->redraw_flag = FALSE;
    else
      bad_init_file("bad redraw field in ID");

/* convert '$' in the ID string back to spaces */
    buffp = id->id_buffer;
    while (*buffp != '\0') {
      if (*buffp == '$')
	*buffp = ' ';
      buffp++;
    }

/* add this ID to the ID list */
    add_id_to_list(id);
  }

  return;
}
