/*
 * xsky - an interactive sky atlas
 *
 * Copyright 1992-5, Terry R. Friedrichsen
 *
 * This program may be copied and redistributed, in whole or in part,
 * as long as you don't try to make any money from the sale or redis-
 * tribution of the program or any part of the program, or pretend
 * that you wrote the program or any of its parts unless specifically
 * credited by the original author.
 *
 * You are free to make use of this software in your own programs, as
 * long as you credit the original author where it is due.
 */

/*
 * WARRANTY:
 * xsky was written as a learning project and as a demonstration of
 * X Window System programming.  xsky doesn't do anything; it is not
 * merchantable, and it is not fit for any purpose whatsoever.  In
 * fact, don't use xsky at all; it's free, and you're getting what
 * you paid for.
 */

#include <stdio.h>

#include <stdlib.h>

#include <math.h>

#include <X11/Intrinsic.h>

#include "skydefs.h"

#include "pos.h"

/* static function prototypes */
static int build_dot_intervals PROTOTYPE((int [],int));
static void draw_ra_line PROTOTYPE((Widget,struct display *,
				            int,int,int,int,int,int,int,int));
static void draw_dec_line PROTOTYPE((Widget,struct display *,
			                    int,int,int,int,int,int,int,int));
static boolean draw_grid_dot PROTOTYPE((Widget,struct display *,
				                    int,int,int,int,int,int));
static int int_compare PROTOTYPE((int *,int *));

/* external function prototypes */
extern double ra_thousandths_to_rad PROTOTYPE((int));
extern double dec_hundredths_to_rad PROTOTYPE((int));
extern int ra_rad_to_thousandths PROTOTYPE((double));
extern int dec_rad_to_hundredths PROTOTYPE((double));
extern void ps_newpath PROTOTYPE((void));
extern void ps_endpath PROTOTYPE((void));
extern void pos_rad_to_X_double PROTOTYPE((struct display *,double,double,
					                  double *,double *));
extern void ps_draw_point PROTOTYPE((struct display *,double,double));
extern void draw_point PROTOTYPE((Widget,Position,Position));

/* table of allowed grid intervals (even divisors of one hour of time
 * or ten degrees of arc, expressed in thousandths of a second of time
 * for right ascension and hundredths of a second of arc for declination */

#define NUM_RA_GRID_INTERVALS   31
static int ra_grid_intervals[NUM_RA_GRID_INTERVALS] = {
  /* from 1 to 1000 we're dividing one second of time into fractions */
  1,		/* .001 second */
  2,		/* .002 second */
  5,		/* .005 second */
  10,		/* .01 second */
  20,		/* .02 second */
  25,		/* .025 second */
  50,		/* .05 second */
  100,		/* .1 second */
  200,		/* .2 second */
  250,		/* .25 second */
  500,		/* .5 second */

/* from 1000 to 60000 we're dividing one minute of time into seconds of time */
  1000,		/* 1 second */
  2000,		/* 2 seconds */
  2500,		/* 2.5 seconds */
  5000,		/* 5 seconds */
  7500,		/* 7.5 seconds */
  10000,	/* 10 seconds */
  15000,	/* 15 seconds */
  20000,	/* 20 seconds */
  30000,	/* 30 seconds */

/* from 60000 to 3600000 we're dividing one hour into minutes of time */
  60000,	/* 1 minute */
  120000,	/* 2 minutes */
  150000,	/* 2.5 minutes */
  300000,	/* 5 minutes */
  450000,	/* 7.5 minutes */
  600000,	/* 10 minutes */
  900000,	/* 15 minutes */
  1200000,	/* 20 minutes */
  1800000,	/* 30 minutes */
  3600000,	/* 1 hour */

/* two hours is the maximum RA grid line interval */
  7200000	/* 2 hours */
  };

#define NUM_DEC_GRID_INTERVALS   31
static int dec_grid_intervals[NUM_DEC_GRID_INTERVALS] = {
/* from 1 to 100 we're dividing one second of arc into fractions */
  1,		/* .01 second */
  2,		/* .02 second */
  5,		/* .05 second */
  10,		/* .1 second */
  20,		/* .2 second */
  25,		/* .25 second */
  50,		/* .5 second */

/* from 100 to 6000 we're dividing one minute of arc into seconds of arc */
  100,		/* 1 second */
  200,		/* 2 seconds */
  250,		/* 2.5 seconds */
  500,		/* 5 seconds */
  750,		/* 7.5 seconds */
  1000,		/* 10 seconds */
  1500,		/* 15 seconds */
  2000,		/* 20 seconds */
  3000,		/* 30 seconds */

/* from 6000 to 360000 we're dividing one degree into minutes of arc */
  6000,		/* 1 minute */
  12000,	/* 2 minutes */
  15000,	/* 2.5 minutes */
  30000,	/* 5 minutes */
  45000,	/* 7.5 minutes */
  60000,	/* 10 minutes */
  90000,	/* 15 minutes */
  120000,	/* 20 minutes */
  180000,	/* 30 minutes */

/* from 36000 to 36000000 we're dividing ten degrees into intervals */
  360000,	/* 1 degree */
  720000,	/* 2 degrees */
  900000,	/* 2.5 degrees */
  1800000,	/* 5 degrees */
  3600000,	/* 10 degrees */

/* fifteen degrees is the maximum declination grid line interval */
  5400000	/* 15 degrees */
  };

/* initial size of the display */
extern int initial_width, initial_height;

/* right ascension and declination line intervals (RA interval is in
 * thousandths of a second of time, dec interval is in hundredths of
 * a second of arc) */
int ra_line_interval;
int dec_line_interval;

/* flag that we're drawing in PostScript, not on the display */
extern boolean ps_draw_flag;



/* draw RA and declination lines in the exposed area of the display */

void draw_grid_lines(sky_widget,display,x,y,width,height)

Widget sky_widget;
struct display *display;
int x, y;
int width, height;

{
  int lr_x, lr_y;
  int i, j;
  float display_width, display_height;
  int dot_intervals[300];
  int dots_per_interval;
  int num_dot_intervals;
  int ra_dot_interval;
  int dec_dot_interval;
  int line_start_ra;
  int line_end_ra;
  int line_start_dec;
  int line_end_dec;
  int start_ra;
  int end_ra;
  int start_dec;
  int end_dec;
  boolean stop_on_greater_eq;
#if 0
  double pole_distance;
#endif
  int dot_start_ra;

#ifdef GRID_DEBUG
  printf("drawing grid lines from %d,%d for width,height = %d,%d\n",
	                                                    x,y,width,height);
#endif

/* compute the lower-right coordinates of the exposed box */
  lr_x = x + width - 1;
  lr_y = y + height - 1;

/* given the upper-left X coordinates and the X width and length of the
 * exposed box, we need to draw lines of right ascension and declination.
 * the first step is to establish the east and west limits of right ascen-
 * sion and the north and south limits of declination of the box.
 *
 * for right ascension, this will occur at the corners; for declination,
 * this will occur either at the corners or in the middle.  a special
 * case occurs if either pole is in the display, in which case the RA
 * must go full-circle and the minimum or maximum declination will be
 * that pole.
 *
 * to draw the lines, we proceed eastward from the right-hand RA limit
 * to the left-hand RA limit, and northward from the southern declination
 * limit to the northern declination limit.  we place dots on the screen
 * along each line at an interval determined by the display scale; the
 * intervals between lines are also determined by display scale.
 *
 * some special cases apply here:  we want to be able to decrease the
 * number of RA lines as we approach either pole, and we draw a small
 * cross instead of a dot at the pole itself. */

/* convert the display limits to integral units */
  start_ra = ra_rad_to_thousandths(display->right_ra_rad);
  end_ra = ra_rad_to_thousandths(display->left_ra_rad);
  start_dec = dec_rad_to_hundredths(display->bottom_dec_rad);
  end_dec = dec_rad_to_hundredths(display->top_dec_rad);

#ifdef GRID_DEBUG
  printf("start ra  = %d, end ra  = %d\n",start_ra,end_ra);
  printf("start dec = %d, end dec = %d\n",start_dec,end_dec);
#endif

/* determine the line intervals and dot intervals for RA and declination */

/* choose the largest grid line interval that divides the display into at
 * least three parts horizontally and vertically */
  display_width = ((display->width * display->scale) / DEGREES_PER_HOUR) *
                                                               60 * 60 * 1000;
  i = NUM_RA_GRID_INTERVALS - 1;
  while (i >= 0)
    /* we need to progressively allow more RA line intervals as we approach
     * the celestial poles; cos(RA) is perfect for this.  we also want to
     * allow more RA line intervals as the display gets wider */
    if ((display_width / ra_grid_intervals[i--]) / cos(display->center_dec_rad)
	                   >= max(3,3 * (float)display->width / initial_width))
      break;
  ra_line_interval = ra_grid_intervals[++i];

#ifdef GRID_DEBUG
  printf("initial RA grid interval = %d\n",ra_line_interval);
#endif

  /* do the same thing in the vertical direction */
  display_height = (display->height * display->scale) * 60 * 60 * 100;
  j = NUM_DEC_GRID_INTERVALS - 1;
  while (j >= 0)
    if ((display_height / dec_grid_intervals[j--]) >=
	                    max(3,3 * (float)display->height / initial_height))
      break;
  dec_line_interval = dec_grid_intervals[++j];

#if 0     /* good idea, but it broke badly near the poles ... */
/* occasionally, the two intervals get farther apart than is esthetically
 * pleasing; fix this by decreasing the larger interval by one step.  this
 * should be enough so the difference is always less than a factor of two */
  while ((i > 0) && (j > 0) && (max(ra_line_interval,dec_line_interval) /
                                 min(ra_line_interval,dec_line_interval) >= 2))
    if (ra_line_interval > dec_line_interval)
      /* reduce the RA line interval by one step */
      ra_line_interval = ra_grid_intervals[--i];
    else
      /* reduce the declination line interval by one step */
      dec_line_interval = dec_grid_intervals[--j];
#endif

/* build the dot interval table for this declination grid line spacing */
  num_dot_intervals = build_dot_intervals(dot_intervals,dec_line_interval);

/* choose the largest dot interval that provides at least the given number of
 * dots per grid interval.  since the RA dot interval is used to draw lines
 * of right ascension, we must choose a dot interval that divides the dec-
 * lination grid interval */
  dots_per_interval = 40;
  i = num_dot_intervals - 1;
  while (i >= 0)
    if ((dec_line_interval / dot_intervals[i--]) >= dots_per_interval)
      break;
  ra_dot_interval = dot_intervals[++i];

#if defined(DEBUG) || defined(GRID_DEBUG)
  printf("scale = %f pixels per degree\n",1 / display->scale);
  printf("display width = %f; display height = %f\n",display_width,
                                                              display_height);
  printf("RA grid interval = %d; dec. grid interval = %d\n",ra_line_interval,
                                                           dec_line_interval);
  printf("RA dot interval = %d\n",ra_dot_interval);
/*
   RA dot units = thousandths of a second of time
   dec dot units = hundredths of a second of arc

   one hour of time = 15 degrees of arc
   one minute of time = 1/4 degree of arc = 15 minutes of arc
   one second of time = 1/4 minute of arc = 15 seconds of arc
   one thousandth of a second of time = .015 seconds of arc =
                                            1.5 hundredths of a second of arc
 */
#endif

/* discussion:
 *    we draw RA lines from west to east.  line_start_ra and line_end_ra are
 *    the actual RAs of the westernmost and easternmost lines of RA we will
 *    draw, respectively.  ra_dot_interval is the spacing at which we will
 *    draw dots when creating the RA lines.  since start_ra is the westernmost
 *    RA on the display, the first RA line we will draw is the next RA to the
 *    east that falls on the RA line interval.  the last RA line we will draw
 *    is the RA immediately west of the eastern display boundary that falls on
 *    the RA line interval. */

/* round the starting right ascension to the next easterly line */
  line_start_ra = (start_ra + ra_line_interval) / ra_line_interval *
                                                             ra_line_interval;

/* round the ending right ascension to the next westerly line */
  line_end_ra = end_ra / ra_line_interval * ra_line_interval;

/* discussion:
 *    we draw declination lines from south to north.  line_start_dec and
 *    line_end_dec are the actual declinations of the southernmost and
 *    northernmost lines of declination we will draw, respectively.
 *    dec_dot_interval is the spacing at which we will draw dots when
 *    creating the declination lines.  since start_dec is the southern-
 *    most declination on the display, the first declination line we will
 *    draw is the next declination to the north that falls on the decli-
 *    nation line interval.  the last declination line we will draw is the
 *    declinatino immediately south of the northern display boundary that
 *    falls on the declination line interval. */

/* round the starting declination to the next northerly line */
  if (start_dec < 0)
    line_start_dec = start_dec / dec_line_interval * dec_line_interval;
  else
    line_start_dec = (start_dec + dec_line_interval) / dec_line_interval *
                                                            dec_line_interval;

/* round the ending declination to the next southerly line */
  if (end_dec < 0)
    line_end_dec = (end_dec - dec_line_interval) / dec_line_interval *
                                                            dec_line_interval;
  else
    line_end_dec = end_dec / dec_line_interval * dec_line_interval;

/* start_dec and end_dec are the southernmost and northernmost limits of
 * the RA lines we draw.  therefore, we move start_dec to the first RA
 * dot location north of the southern display boundary.  note that we
 * don't need to modify end_dec since it merely serves as a limit. */
  if (start_dec < 0)
    start_dec = start_dec / ra_dot_interval * ra_dot_interval;
  else
    start_dec = (start_dec + ra_dot_interval) / ra_dot_interval *
                                                              ra_dot_interval;

/* set a flag if we're not going through the 24-hour line */
  stop_on_greater_eq = (boolean)(line_start_ra <= line_end_ra);

/* draw lines of right ascension */
  while (TRUE) {
    /* draw this line of right ascension */
    draw_ra_line(sky_widget,display,line_start_ra,start_dec,end_dec,
		                               ra_dot_interval,x,y,lr_x,lr_y);

    /* check to see if we're done */
    if ((line_start_ra >= line_end_ra) && stop_on_greater_eq)
      break;

    /* step to the next RA line, being careful when we pass 24 hours */
    line_start_ra += ra_line_interval;
    if (line_start_ra > (24 * 60 * 60 * 1000)) {
      line_start_ra -= 24 * 60 * 60 * 1000;
      stop_on_greater_eq = TRUE;
    }
  };

/* build the dot interval table for this RA grid line spacing */
  num_dot_intervals = build_dot_intervals(dot_intervals,ra_line_interval);

/* draw lines of declination */
  while (line_start_dec <= line_end_dec) {
#if 0
    /* alter the declination dot interval according to the distance of the
     * declination line from the pole relative to the display height */
    pole_distance = (90 - fabs((double)line_start_dec / 100 / 60 / 60)) /
                                             display->scale / display->height;
#ifdef GRID_DEBUG
    printf("pole distance = %lf display heights\n",pole_distance);
#endif
    if (pole_distance < 1)
      dots_per_interval = 40;
    else if (pole_distance < 3)
      dots_per_interval = 40;
    else
      dots_per_interval = 40;
#else
    /* use a fixed 40 dots per interval all the time, just like RA */
    dots_per_interval = 40;
#endif

    /* since the declination dot interval is used to draw lines of declina-
     * tion, we must choose a dot interval that divides the RA grid interval */
    i = num_dot_intervals - 1;
    while (i >= 0)
      if ((ra_line_interval / dot_intervals[i--]) >= dots_per_interval)
	break;
    dec_dot_interval = dot_intervals[++i];

#if defined(DEBUG) || defined(GRID_DEBUG)
    /* as a reminder, declination dot units = hundredths of a second of arc */
    printf("declination dot interval = %d\n",dec_dot_interval);
#endif

    /* start_ra and end_ra are the westernmost and easternmost limits of
     * the declination lines we draw.  therefore, we move start_ra to the
     * first declination dot location north of the southern display boun-
     * dary (this is done in the declination line loop below, since the
     * dot interval varies by declination).  note that we don't need to
     * modify end_ra since it merely serves as a limit. */
    dot_start_ra = (start_ra + dec_dot_interval) / dec_dot_interval *
                                                             dec_dot_interval;

    /* draw the line with the chosen dot interval */
    draw_dec_line(sky_widget,display,dot_start_ra,end_ra,
		                           line_start_dec,dec_dot_interval,
		                                               x,y,lr_x,lr_y);

    /* step to the next declination line */
    line_start_dec += dec_line_interval;
  }

  return;
}



/* draw one line of right ascension for the RA/declination grid */

static void draw_ra_line(sky_widget,display,ra,start_dec,end_dec,dot_interval,
			                                  ul_x,ul_y,lr_x,lr_y)

Widget sky_widget;
struct display *display;
int ra;
int start_dec;
int end_dec;
int dot_interval;
int ul_x, ul_y;
int lr_x, lr_y;

{
  boolean dot_in_flag;

/* if this is PostScript, begin a new path */
  if (ps_draw_flag)
    ps_newpath();

/* draw dots until we pass the ending declination */
  while (start_dec <= end_dec) {
    /* draw one dot along the RA line */
    dot_in_flag = draw_grid_dot(sky_widget,display,ra,start_dec,
				                         ul_x,ul_y,lr_x,lr_y);

    /* if that dot was not in the display, end this path and begin another */
    if (ps_draw_flag && (! dot_in_flag)) {
      ps_endpath();
      ps_newpath();
    }

    /* step to the next dot */
    start_dec += dot_interval;
  }

/* if this is PostScript, end the path */
  if (ps_draw_flag)
    ps_endpath();

  return;
}



/* draw one line of declination for the RA/declination grid */

static void draw_dec_line(sky_widget,display,start_ra,end_ra,dec,dot_interval,
			                                  ul_x,ul_y,lr_x,lr_y)

Widget sky_widget;
struct display *display;
int start_ra;
int end_ra;
int dec;
int dot_interval;
int ul_x, ul_y;
int lr_x, lr_y;

{
  boolean stop_on_greater_eq;
  boolean dot_in_flag;

/* don't draw anything at either pole */
  if (abs(dec) == (90 * 60 * 60 * 100))
    return;

/* if this is PostScript, begin a new path */
  if (ps_draw_flag)
    ps_newpath();

/* set a flag if we're not going through the 24-hour line */
  stop_on_greater_eq = (boolean)(start_ra <= end_ra);

/* draw grid dots for this declination line */
  while (TRUE) {
    /* draw one dot along the declination line */
    dot_in_flag = draw_grid_dot(sky_widget,display,start_ra,dec,
				                         ul_x,ul_y,lr_x,lr_y);

    /* if that dot was not in the display, end this path and begin another */
    if (ps_draw_flag && (! dot_in_flag)) {
      ps_endpath();
      ps_newpath();
    }

    /* step to the next dot */
    start_ra += dot_interval;

    /* check to see if we're done */
    if ((start_ra >= end_ra) && stop_on_greater_eq)
      break;

    /* be careful when we pass 24 hours */
    if (start_ra > (24 * 60 * 60 * 1000)) {
      start_ra -= 24 * 60 * 60 * 1000;
      stop_on_greater_eq = TRUE;
    }
  };

/* if this is PostScript, end the path */
  if (ps_draw_flag)
    ps_endpath();

  return;
}



/* draw one dot for the right ascension/declination grid */

static boolean draw_grid_dot(sky_widget,display,
			                    ra_thousandths,dec_hundredths,
			                                  ul_x,ul_y,lr_x,lr_y)

Widget sky_widget;
struct display *display;
int ra_thousandths;
int dec_hundredths;
int ul_x, ul_y;
int lr_x, lr_y;

{
  int x, y;
  double ps_x, ps_y;

/* convert the position from radians to floating X coordinates */
  pos_rad_to_X_double(display,ra_thousandths_to_rad(ra_thousandths),
	                        dec_hundredths_to_rad(dec_hundredths),
		                                                 &ps_x,&ps_y);

/* convert floating X coordinates into integer pixel positions */
  x = ps_x + 0.5;
  y = ps_y + 0.5;

/* in the Southern Hemisphere, mirror-image and invert the display */
  if (display->downunder_flag) {
    x = display->width - x;
    y = display->height - y;
  }

/* if the dot is within the area, draw it */
  if ((x >= ul_x) && (x <= lr_x) && (y >= ul_y) && (y <= lr_y)) {
    if (ps_draw_flag)
      ps_draw_point(display,ps_x,ps_y);
    else
      draw_point(sky_widget,(Position)x,(Position)y);

    return(TRUE);
  }
  else
    return(FALSE);

/* NOTREACHED */
}



/* build a dot interval table */

static int build_dot_intervals(interval_table,max_interval)

int interval_table[];
int max_interval;

{
  int divisor;
  int i;
  int j, k;

/* build a table of all divisors of the grid interval */
  for (i = 0, divisor = 1; divisor <= sqrt((double)max_interval); divisor++)
    if ((max_interval % divisor) == 0) {
      interval_table[i++] = divisor;
      interval_table[i++] = max_interval / divisor;
    }

/* sort the table */
  qsort((void *)interval_table,(size_t)i,sizeof(int),(int (*)())int_compare);

/* eliminate the duplicates from the table */
  j = k = 0;
  while (k < i)
    if (interval_table[k] == interval_table[k + 1])
      k++;
    else
      interval_table[j++] = interval_table[k++];

/* return the number of divisors */
  return(j);
}



/* compare two integers for qsort */

static int int_compare(int1,int2)

int *int1, *int2;

{
  return(*int1 - *int2);
}
