/*
 * xsky - an interactive sky atlas
 *
 * Copyright 1992-5, Terry R. Friedrichsen
 *
 * This program may be copied and redistributed, in whole or in part,
 * as long as you don't try to make any money from the sale or redis-
 * tribution of the program or any part of the program, or pretend
 * that you wrote the program or any of its parts unless specifically
 * credited by the original author.
 *
 * You are free to make use of this software in your own programs, as
 * long as you credit the original author where it is due.
 */

/*
 * WARRANTY:
 * xsky was written as a learning project and as a demonstration of
 * X Window System programming.  xsky doesn't do anything; it is not
 * merchantable, and it is not fit for any purpose whatsoever.  In
 * fact, don't use xsky at all; it's free, and you're getting what
 * you paid for.
 */

#include <stdio.h>
#include <ctype.h>

#include <math.h>
#include <stdlib.h>

#include <unistd.h>

#include <string.h>

#include <X11/Intrinsic.h>

#include "gcvs.h"

#include "skydefs.h"

#include "pos.h"
#include "catalog.h"
#include "draw.h"
#include "catreader.h"

struct gcvs_node {
  double ra_rad;
  double dec_rad;
  float mag;
  long filepos;
  smallint constellation_index;
  char spectrum;
  shortint desig_idx;
  char component;
};

/* static function prototypes */
static FILE *read_gcvs PROTOTYPE((void));
static void calculate_gcvs_field PROTOTYPE((struct display *,
					                struct cat_header *));
static boolean check_mag_limit PROTOTYPE((int,double));
static boolean find_gcvs_star PROTOTYPE((char *));
static void get_gcvs_pos PROTOTYPE((int,double *,double *));
static void draw_gcvs_star PROTOTYPE((Widget,int,int,int,int,int,int,int));
static boolean ps_draw_gcvs_star PROTOTYPE((struct display *,int,int,int));
static boolean display_gcvs PROTOTYPE((FILE *,int));
static void format_gcvs_name PROTOTYPE((int,char *,IDfont *));
static int mag_compare PROTOTYPE((struct gcvs_node *,struct gcvs_node *));
static int fill_gcvs_entry PROTOTYPE((char *,int));
static boolean find_star_by_argelander PROTOTYPE((char *,int));
static boolean find_star_by_ordinal PROTOTYPE((char *,int));
static boolean find_star_by_bayer PROTOTYPE((char *,int));
static boolean find_star_by_idx PROTOTYPE((int,int));

/* external function prototypes */
extern char *build_filespec PROTOTYPE((char *,char *));
extern boolean read_binary_database PROTOTYPE((char *,void *,int *));
extern void init_precess PROTOTYPE((double,double));
extern void hs_heapsort PROTOTYPE((char [],size_t,size_t,int (*)()));
extern boolean write_binary_database PROTOTYPE((char *,void *,size_t,size_t));
extern int find_const_abbrev PROTOTYPE((char *));
extern double ra_pos_to_rad PROTOTYPE((struct ra_pos));
extern double dec_pos_to_rad PROTOTYPE((struct dec_pos));
extern int find_gcvs_bayer PROTOTYPE((char *));
extern void precess PROTOTYPE((double,double,double *,double *));
extern void add_to_display_list PROTOTYPE((struct display *,
					           struct cat_header *,
				                          int,double,double));
extern void draw_star PROTOTYPE((Widget,int,int,int,StarColor));
extern boolean ps_draw_star PROTOTYPE((struct display *,double,double,double));
extern void format_gcvs_display PROTOTYPE((char *));
extern int find_constellation PROTOTYPE((char *));
extern int find_greek_letter PROTOTYPE((char *));
extern void update_display_pos PROTOTYPE((double,double));
extern void build_argelander_table PROTOTYPE((void));
extern int argelander_to_ordinal PROTOTYPE((char *));
extern void ordinal_to_argelander PROTOTYPE((int,char *));

extern char *constellations_abbrev[];

extern char greek_letter[];

#define ORDINAL_OFFSET    0
#define GREEK_OFFSET  20000

extern char argelander[NUM_ARGELANDER][3];

static struct gcvs_node gcvs_array[MAX_GCVS];

static int gcvs_array_max;

extern float mag_limit;

/* AFAIK, these are the three possible ways the CD-ROM file name might be
 * represented by the operating system */
#define GCVS_DATA_FILE_1   "GCVS.DAT;1"
#define GCVS_DATA_FILE_2   "GCVS.DAT"
#define GCVS_DATA_FILE_3   "gcvs.dat"

#define GCVS_BINARY_FILE   "gcvs.bin"

/* variable star record buffer */
static char gcvs_rec[GCVS_RECLEN + 2 + 1];



/* initialize the catalog data structure */

void init_gcvs(catalog)

struct cat_header *catalog;

{
  catalog->read_data = read_gcvs;
  catalog->calculate_field = calculate_gcvs_field;
  catalog->mag_limit_check = check_mag_limit;
  catalog->find_object = find_gcvs_star;
  catalog->get_object_pos = get_gcvs_pos;
  catalog->draw_object = draw_gcvs_star;
  catalog->ps_draw_object = ps_draw_gcvs_star;
  catalog->display_data = display_gcvs;
  catalog->format_name = format_gcvs_name;
  catalog->menu_name = GCVS_MENU_NAME;
  catalog->catalog_name = GCVS_CATALOG_NAME;

/* build the table for converting ordinals to Argelander */
  build_argelander_table();

  return;
}



/* read in the variable star catalog file, extracting needed info */

static FILE *read_gcvs()

{
  char *filebuff;
  FILE *gcvs_fd;
  char *bin_filebuff;
  int gcvs_entry;

/* build the path to the variable star data file and open it */
  filebuff = build_filespec("GCVS_CD_PATH",GCVS_DATA_FILE_1);
  if ((gcvs_fd = fopen(filebuff,"r")) == (FILE *)NULL) {
    free((void *)filebuff);
    filebuff = build_filespec("GCVS_CD_PATH",GCVS_DATA_FILE_2);
    if ((gcvs_fd = fopen(filebuff,"r")) == (FILE *)NULL) {
      free((void *)filebuff);
      filebuff = build_filespec("GCVS_CD_PATH",GCVS_DATA_FILE_3);
      if ((gcvs_fd = fopen(filebuff,"r")) == (FILE *)NULL) {
	printf("sky:  can't open variable star data file %s\n",filebuff);
	perror("sky");
	exit(1);
      }
    }
  }

/* get the path to the variable star binary file */
  bin_filebuff = build_filespec("GCVS_BIN_PATH",GCVS_BINARY_FILE);

/* see if we can read the binary file */
  if (access(bin_filebuff,R_OK) == 0)
    /* we can; try to use the pre-built database */
    if (read_binary_database(bin_filebuff,gcvs_array,&gcvs_array_max)) {
      /* success; free the filename buffers */
      free((void *)filebuff);
      free((void *)bin_filebuff);

      /* return the fd of the open data file */
      return(gcvs_fd);
    }

/* prior to reading the variable star catalog, build the rotation matrix
 * for precessing the stellar coordinates to the common equinox from the
 * given epoch */
  if (GCVS_EPOCH != COMMON_EQUINOX)
    init_precess(GCVS_EPOCH,COMMON_EQUINOX);

/* read the variable star catalog, saving the info necessary for the display */
  gcvs_entry = 0;
  while (TRUE) {
    gcvs_array[gcvs_entry].filepos = ftell(gcvs_fd);
    if (fgets(gcvs_rec,sizeof(gcvs_rec),gcvs_fd) == (char *)NULL)
      break;
    gcvs_entry = fill_gcvs_entry(gcvs_rec,gcvs_entry);
  }

/* leave the file open for later use */
/*  (void)fclose(gcvs_fd);*/

/* remember the last gcvs entry we filled */
  gcvs_array_max = gcvs_entry - 1;

/* now sort the gcvs array in decreasing order of magnitude */
  hs_heapsort((void *)gcvs_array,(size_t)gcvs_entry,
	                        (size_t)sizeof(struct gcvs_node),mag_compare);

/* attempt to save the binary database */
  if (access(bin_filebuff,W_OK) == 0)
    /* it's writeable, so let's try */
    if (! write_binary_database(bin_filebuff,(void *)gcvs_array,
				            (size_t)sizeof(struct gcvs_node),
				                          (size_t)gcvs_entry))
      /* dump of binary database failed; be sure the file is gone */
      (void)unlink(bin_filebuff);

/* free the filename buffers */
  free((void *)filebuff);
  free((void *)bin_filebuff);

  return(gcvs_fd);
}



/* compare two array entries for magnitude - compare routine for heapsort */

static int mag_compare(gcvs_entry_1,gcvs_entry_2)

struct gcvs_node *gcvs_entry_1, *gcvs_entry_2;

{
/* major key is the magnitude */
  if (gcvs_entry_1->mag < gcvs_entry_2->mag)
    return(-1);
  else if (gcvs_entry_1->mag > gcvs_entry_2->mag)
    return(1);
  else
    return(0);

/* NOTREACHED */
}



/* return TRUE if the magnitude of the star meets the limiting magnitude */

static boolean check_mag_limit(array_pos,limit)

int array_pos;
float limit;

{
  if (gcvs_array[array_pos].mag <= limit)
    return(TRUE);
  else if ((gcvs_array[array_pos].mag == 88.88) && (limit >= 19.0))
    return(TRUE);
  else
    return(FALSE);

/* NOTREACHED */
}



/* given an object's index, return its right ascension and declination */

static void get_gcvs_pos(array_pos,ra_rad,dec_rad)

int array_pos;
double *ra_rad, *dec_rad;

{
  *ra_rad = gcvs_array[array_pos].ra_rad;
  *dec_rad = gcvs_array[array_pos].dec_rad;

  return;
}



/* given a variable star record, make a data node for it */

static int fill_gcvs_entry(record,gcvs_entry)

char *record;
int gcvs_entry;

{
  int reclen;
  char field[11];
  int i;
  struct ra_pos ra_pos;
  struct dec_pos dec_pos;

/* see if this is a real object */
  if (record[NOTE_FLAG] == 'N')
    /* object is nonexistent */
    return(gcvs_entry);

/* don't bother with duplicate records */
  if (record[SPECTRUM_START] == '=')
    return(gcvs_entry);

/* get the length of the record for GET_FIELD */
  reclen = GCVS_RECLEN;

/* get the constellation name name and store as an index */
  GET_FIELD(field,record,CONSTELLATION_NAME,3);
  gcvs_array[gcvs_entry].constellation_index =
                                        (smallint)find_const_abbrev(field);
  if (gcvs_array[gcvs_entry].constellation_index == (smallint)-1)
    printf("constellation abbreviation %s was not found for GCVS\n",field);

/* get the star designation */

  /*
   * here are the rules:
   * 1)  a single letter is Argelander
   * 2)  two letters are Argelander except for PI
   * 3)  three or more letters are Bayer
   * 4)  if the first character is V and the third is a digit, it's ordinal
   */
  GET_FIELD(field,record,NAME_START,5);

  /* handle ordinals first */
  if ((field[0] == 'V') && isdigit(field[2])) {
    /* fill in any missing leading zero */
    if (field[1] == ' ')
      field[1] = '0';

    /* save the ordinal designation */
    gcvs_array[gcvs_entry].desig_idx = atoi(&field[1]) + ORDINAL_OFFSET;
  }
  else if ((field[2] != ' ') || (strncmp(field,"PI",2) == EQUAL)) {
    /* this is a Bayer designation - nul-terminate it */
    i = 1;
    do {
      i++;
    } while ((field[i] != ' ') && (i < 5));
    field[i] = '\0';

    /* find this Bayer designation in the GCVS Greek-letter table */
    i = find_gcvs_bayer(field);

    /* convert to numeric index */
    gcvs_array[gcvs_entry].desig_idx = i + 1 + GREEK_OFFSET;
  }
  else {
    /* this must be an Argelander designation - nul-terminate it */
    if (field[1] == ' ')
      field[1] = '\0';
    else
      field[2] = '\0';

    /* convert to ordinal */
    gcvs_array[gcvs_entry].desig_idx = argelander_to_ordinal(field) +
                                                               ORDINAL_OFFSET;
  }

/* save the component ID */
  gcvs_array[gcvs_entry].component = record[COMPONENT_ID];

/* get the RA in pieces */
  GET_FIELD(field,record,POS_RA_START,2);
  ra_pos.hours = (smallint)atoi(field);

  GET_FIELD(field,record,POS_RA_START + 2,2);
  ra_pos.minutes = (smallint)atoi(field);

  GET_FIELD(field,record,POS_RA_START + 4,2);
  ra_pos.seconds = (smallint)atoi(field);

  ra_pos.thousandths = 0;

/* similarly, get the declination in pieces */
  GET_FIELD(field,record,POS_DEC_START,1);
  dec_pos.sign = (short)1;
  if (field[0] == '-')
    dec_pos.sign = (short)-1;

  GET_FIELD(field,record,POS_DEC_START + 1,2);
  dec_pos.degrees = (smallint)atoi(field);

  GET_FIELD(field,record,POS_DEC_START + 3,2);
  dec_pos.minutes = (smallint)atoi(field);

  GET_FIELD(field,record,POS_DEC_START + 6,1);
  dec_pos.seconds = (smallint)atoi(field) * 6;

  dec_pos.hundredths = 0;

/* compute the RA and declination in radians */
  gcvs_array[gcvs_entry].ra_rad = ra_pos_to_rad(ra_pos);
  gcvs_array[gcvs_entry].dec_rad = dec_pos_to_rad(dec_pos);

/* precess them to the common equinox */
  if (GCVS_EPOCH != COMMON_EQUINOX)
    precess(gcvs_array[gcvs_entry].ra_rad,gcvs_array[gcvs_entry].dec_rad,
	      &gcvs_array[gcvs_entry].ra_rad,&gcvs_array[gcvs_entry].dec_rad);

/* get the maximum magnitude */
  GET_FIELD(field,record,MAX_MAG,MAX_MAG_LENGTH);
  if (strcmp(field,"       ") == EQUAL) {
    /* no maximum; use minimum magnitude instead */
    GET_FIELD(field,record,MIN_MAG,MIN_MAG_LENGTH);
    if (strcmp(field,"       ") == EQUAL)
      /* no minimum either; use a bogus value */
      strcpy(field,"99.9");
  }
  gcvs_array[gcvs_entry].mag = atof(field);

/* get the spectral class */
  GET_FIELD(field,record,SPECTRUM_START,1);
  /* watch out for the fact that they upper-cased the dwarf designator, and
   * handle the case of parenthesized spectral class pairs */
  if ((field[0] == 'D') || (field[0] == '(')) {
    GET_FIELD(field,record,SPECTRUM_START + 1,1);
    /* one special-case hack for the spectrum (CONT+E)-K5 */
    if ((field[0] == '(') && (record[SPECTRUM_START + 1] == 'C'))
      field[0] = 'K';
  }
  gcvs_array[gcvs_entry].spectrum = field[0];

  return(++gcvs_entry);
}



/* construct a list of those stars which appear in the current display */

static void calculate_gcvs_field(display,catalog)

struct display *display;
struct cat_header *catalog;

{
  int i;

/* loop through all variable star entries, checking for visibility */
  for (i = 0; i <= gcvs_array_max; i++)
    /* check this star and add it to the list if visible */
    add_to_display_list(display,catalog,i,gcvs_array[i].ra_rad,
			                              gcvs_array[i].dec_rad);

#ifdef DEBUG
  printf("calculate_gcvs_field returning\n");
#endif
  return;
}



/* draw a variable star on the sky */

static void draw_gcvs_star(sky_widget,array_pos,obj_x,obj_y,ul_x,ul_y,
		                                                 width,height)

Widget sky_widget;
int array_pos;
int obj_x, obj_y;
int ul_x, ul_y;
int width, height;

{
  int size;
  StarColor color;

/* be sure it is bright enough to display */
  if (! check_mag_limit(array_pos,mag_limit))
    return;

  /* calculate size based on magnitude */
  size = STARSIZE(gcvs_array[array_pos].mag);

  /* draw the object even if it is only partially in the box */
  if (((obj_x + (size + 1) / 2) >= ul_x) &&
                               ((obj_x - (size + 1) / 2) <= (ul_x + width)) &&
                               ((obj_y + (size + 1) / 2) >= ul_y) &&
                               ((obj_y - (size + 1) / 2) <= (ul_y + height)))
    ;
  else
    return;

  /* assign a color based on the spectral class */
  switch (gcvs_array[array_pos].spectrum) {
  case 'W':
  case 'O':
    color = O;
    break;
  case 'B':
    color = B;
    break;
  case 'A':
    color = A;
    break;
  case 'F':
    color = F;
    break;
  case 'G':
    color = G;
    break;
  case 'K':
    color = K;
    break;
  case 'M':
  case 'C':
  case 'R':
  case 'N':
  case 'S':
    color = M;
    break;
  case 'E':        /* emission and H-alpha emission spectra; make them white */
    color = A;
    break;
  default:
    printf("GCVS variable star has unrecognized spectral class %c\n",
	                                      gcvs_array[array_pos].spectrum);
  case 'P':        /* some stars are peculiar */
  case ' ':        /* and some are just not given */
    color = A;     /* make unknowns white */
    break;
  }

/* draw the star with correct location, size, and color */
  draw_star(sky_widget,obj_x,obj_y,size,color);

  return;
}



/* draw one variable star on the PostScript display */

static boolean ps_draw_gcvs_star(display,array_pos,x,y)

struct display *display;
int array_pos;
int x, y;

{
  int size;

/* be sure it is bright enough to display */
  if (! check_mag_limit(array_pos,mag_limit))
    return(TRUE);

/* calculate size based on magnitude */
  size = STARSIZE(gcvs_array[array_pos].mag);

/* draw the star in the correct position with the correct size */
  return(ps_draw_star(display,(double)x,(double)y,(double)size));
}



/* display the catalog data for the given star */

static boolean display_gcvs(gcvs_fd,gcvs_array_pos)

FILE *gcvs_fd;
int gcvs_array_pos;

{
/* get info from the General Catalog of Variable Stars */
  if (fseek(gcvs_fd,gcvs_array[gcvs_array_pos].filepos,SEEK_SET) == -1)
    return(FALSE);
  if (fgets(gcvs_rec,sizeof(gcvs_rec),gcvs_fd) == (char *)NULL)
    return(FALSE);

/* format the record into an informational display */
  format_gcvs_display(gcvs_rec);

  return(TRUE);
}



/* find the variable star in the buffer and update the display coordinates */

static boolean find_gcvs_star(buffer)

char *buffer;

{
  char *ptr;
  char desig_buff[10];
  int i;
  int constellation;

/* copy out the first token */
  ptr = buffer;
  i = 0;
  while (! isspace(*ptr))
    if ((i > 9) || (desig_buff[i++] = *ptr++) == '\0')
      return(FALSE);
  desig_buff[i] = '\0';

/* find the second token and retrieve the constellation index */
  while (isspace(*ptr))
    ptr++;

  /* the second token must be a constellation */
  if ((constellation = find_constellation(ptr)) == -1)
    return(FALSE);

/* if the first letter is upper-case, must be Argelander or ordinal */
  if (isupper(buffer[0]))
    if ((buffer[0] == 'V') && isdigit(buffer[1]))
      /* this is an ordinal designation */
      return(find_star_by_ordinal(desig_buff,constellation));
    else
      /* this is an Argelander designation */
      return(find_star_by_argelander(desig_buff,constellation));
  else if ((buffer[0] == 'v') && isdigit(buffer[1]))
    /* this is an ordinal designation */
    return(find_star_by_ordinal(desig_buff,constellation));
  else
    /* lower-case and not ordinal; try Greek letter */
    return(find_star_by_bayer(desig_buff,constellation));

/* NOTREACHED */
}



/* find the variable star by Argelander designation */

static boolean find_star_by_argelander(buffer,constellation)

char *buffer;
int constellation;

{
/* check for a valid Argelander designation */
  if (strlen(buffer) == 1)
    if ((buffer[0] >= 'R') && (buffer[0] <= 'Z'))
      return(find_star_by_idx(argelander_to_ordinal(buffer) + ORDINAL_OFFSET,
			                                      constellation));
    else
      /* we can't possibly recognize this designation */
      return(FALSE);
  else if (strlen(buffer) == 2)
    if (isupper(buffer[0]) && isupper(buffer[1]))
      if (buffer[1] >= buffer[0])
	if ((buffer[0] == 'J') || (buffer[1] == 'J'))
	  return(FALSE);
	else
	  return(find_star_by_idx(argelander_to_ordinal(buffer) +
				               ORDINAL_OFFSET,constellation));
      else
	/* we can't possibly recognize this designation */
	return(FALSE);
    else
      /* Argelander designation must be upper-case, by definition */
      return(FALSE);
  else
    return(FALSE);

/* NOTREACHED */
}



/* find the variable star by ordinal designation */

static boolean find_star_by_ordinal(buffer,constellation)

char *buffer;
int constellation;

{
  int i;

/* we know the buffer begins with 'V'; check the rest for numeric */
  i = 1;
  while (isdigit(buffer[i]))
    i++;

/* if we stopped on the nul, the format is valid */
  if (buffer[i] == '\0')
    return(find_star_by_idx(atoi(&buffer[1]) + ORDINAL_OFFSET,constellation));
  else
    return(FALSE);

/* NOTREACHED */
}



/* find the variable star by Bayer designation */

static boolean find_star_by_bayer(buffer,constellation)

char *buffer;
int constellation;

{
  int i, j;
  char letter[10];
  int superscript;

/* copy out the letter of the designation */
  i = j = 0;
  while (islower(buffer[i]))
    if (i > 9)
      return(FALSE);
    else
      letter[j++] = buffer[i++];
  letter[i] = '\0';

/* check for a superscript */
  superscript = 0;
  if (buffer[i] == '-')
    if (isdigit(buffer[++i])) {
      superscript = (int)(buffer[i] - '0');
      i++;
    }
    else
      return(FALSE);

/* do we understand the buffer format? */
  if (buffer[i] != '\0')
    return(FALSE);

/* find the index of the given Greek letter and search database for the star */
  i = find_greek_letter(letter);
  if (i == -1)
    return(FALSE);
  else
    return(find_star_by_idx((i + 1) * 10 + superscript + GREEK_OFFSET,
			                                      constellation));

/* NOTREACHED */
}



/* find the variable star given its designation index */

static boolean find_star_by_idx(idx,constellation)

int idx;
int constellation;

{
  int i;

/* loop through the array looking for this variable star */
  for (i = 0; i <= gcvs_array_max; i++)
    if (gcvs_array[i].constellation_index == constellation)
      if (gcvs_array[i].desig_idx == idx)
	break;

/* if we found it, update the display position */
  if (i <= gcvs_array_max) {
    update_display_pos(gcvs_array[i].ra_rad,gcvs_array[i].dec_rad);
    return(TRUE);
  }
  else
    return(FALSE);

/* NOTREACHED */
}



/* format a nice name for a star */

static void format_gcvs_name(gcvs_pos,buffer,font)

int gcvs_pos;
char *buffer;
IDfont *font;

{
  int greek_idx;

/* convert the designation index into a name */
  if (gcvs_array[gcvs_pos].desig_idx > GREEK_OFFSET) {
    /* change to Greek font and format as a Bayer designation */
    *font = GREEK;
    greek_idx = gcvs_array[gcvs_pos].desig_idx - GREEK_OFFSET;

    sprintf(buffer,"%c",greek_letter[greek_idx - 1]);
  }
  else if ((gcvs_array[gcvs_pos].desig_idx - ORDINAL_OFFSET) > NUM_ARGELANDER)
    /* format as an ordinal designation */
    sprintf(buffer,"V%d",gcvs_array[gcvs_pos].desig_idx - ORDINAL_OFFSET);
  else
    /* format as an Argelander designation */
    ordinal_to_argelander((int)gcvs_array[gcvs_pos].desig_idx - ORDINAL_OFFSET,
			                                              buffer);

/* is there a superscript? */
  if (isdigit(gcvs_array[gcvs_pos].component))
    sprintf(&buffer[strlen(buffer)],"-%c",gcvs_array[gcvs_pos].component);

/* skip constellation names if Bayer - we can't switch out of Greek font */
  if (gcvs_array[gcvs_pos].desig_idx <= GREEK_OFFSET) {
    /* add the constellation name */
    sprintf(&buffer[strlen(buffer)]," %s",
	    constellations_abbrev[gcvs_array[gcvs_pos].constellation_index]);

    /* add any alphabetic component ID */
    if (isalpha(gcvs_array[gcvs_pos].component))
      sprintf(&buffer[strlen(buffer)]," %c",gcvs_array[gcvs_pos].component);
  }

  return;
}
