/*
 * xsky - an interactive sky atlas
 *
 * Copyright 1992-3, Terry R. Friedrichsen
 *
 * This program may be copied and redistributed, in whole or in part,
 * as long as you don't try to make any money from the sale or redis-
 * tribution of the program or any part of the program, or pretend
 * that you wrote the program or any of its parts unless specifically
 * credited by the original author.
 *
 * You are free to make use of this software in your own programs, as
 * long as you credit the original author where it is due.
 */

/*
 * WARRANTY:
 * xsky was written as a learning project and as a demonstration of
 * X Window System programming.  xsky doesn't do anything; it is not
 * merchantable, and it is not fit for any purpose whatsoever.  In
 * fact, don't use xsky at all; it's free, and you're getting what
 * you paid for.
 */

/* this routine formats and displays a variable star record */

#include <stdio.h>

#include <ctype.h>
#include <string.h>

#include "gcvs.h"

#include "skydefs.h"

#include "gcvs_types.h"

#include "gcvs_greek.h"

#include "format.h"

/* static function prototypes */
static void display_pos PROTOTYPE((char *));
static int find_gcvs_type PROTOTYPE((char *,struct type_table *));

/* external function prototypes */
extern int find_const_abbrev PROTOTYPE((char *));
extern char *strnstr PROTOTYPE((char *,char *,int));

/* global function prototypes */
int find_gcvs_bayer PROTOTYPE((char *));

#define MAX_GCVS_ROW   85
#define NUM_GCVS_ROWS  23

extern char *greek[];

extern char *constellations_latin[];



void format_gcvs_display(record)

char *record;

{
  int i;
  char desig_buff[NAME_LENGTH + 1];
  int greek_idx;
  char const_buff[4];
  char type_buff[VAR_TYPE_LENGTH + 1];
  char *tokptr;
  int toklen_idx;
  char c;
  int match_idx;
  boolean question_flag;
  char epoch_buff[EPOCH_LENGTH + 1];

/* start at the beginning of the display */
  idx = 0;
  row = 0;
  row_start = 0;
  max_row = MAX_GCVS_ROW;

/* copy out the designation and nul-terminate it */
  strncpy(desig_buff,&record[NAME_START],NAME_LENGTH);
  desig_buff[NAME_LENGTH] = '\0';

/* handle an ordinal designation */
  if ((desig_buff[0] == 'V') && isdigit(desig_buff[2]))
    /* put out the ordinal designation */
    if (desig_buff[1] == ' ')
      sprintf(&obj_info[idx],"V%.3s",&desig_buff[2]);
    else
      sprintf(&obj_info[idx],"V%.4s",&desig_buff[1]);
  else {
    /* nul-terminate the designation */
    i = 0;
    do {
      i++;
    } while ((desig_buff[i] != ' ') && (i < NAME_LENGTH));
    desig_buff[i] = '\0';

    /* see if this is a Bayer designation */
    if ((strlen(desig_buff) > 2) || (strncmp(desig_buff,"PI",2) == EQUAL)) {
      /* find this Bayer designation in the GCVS Greek-letter table */
      greek_idx = find_gcvs_bayer(desig_buff);

      /* display the Bayer designation */
      if (greek_idx == -1)
	add_string("unknown");
      else
	sprintf(&obj_info[idx],"%s",greek[greek_idx]);
    }
    else
      /* display the Argelander designation */
      sprintf(&obj_info[idx],"%s",desig_buff);
  }
  idx += strlen(&obj_info[idx]);

/* if the component is numeric, put it in as a qualifier to the designation */
  if (isdigit(record[COMPONENT_ID])) {
    sprintf(&obj_info[idx], "-%c",record[COMPONENT_ID]);
    idx += strlen(&obj_info[idx]);
  }

/* add in the genitive-case constellation name */
  strncpy(const_buff,&record[CONSTELLATION_NAME],3);
  const_buff[3] = '\0';
  sprintf(&obj_info[idx]," %s",
	                 constellations_latin[find_const_abbrev(const_buff)]);
  idx += strlen(&obj_info[idx]);

/* if the component is alphabetic, put it in after the name */
  if (isalpha(record[COMPONENT_ID])) {
    sprintf(&obj_info[idx], " %c",record[COMPONENT_ID]);
    idx += strlen(&obj_info[idx]);
  }

/* space down some */
  blank_line();

/* display the position */
  display_pos(&record[POS_RA_START]);

  next_line();

/* isolate the variable type field */
  strncpy(type_buff,&record[VAR_TYPE_START],VAR_TYPE_LENGTH);
  /* nul-terminte it */
  i = 0;
  do {
    i++;
  } while ((type_buff[i] != ' ') && (i < VAR_TYPE_LENGTH));
  type_buff[i] = '\0';

/* display the raw type information */
  sprintf(&obj_info[idx],"Variable type(s):  %s",type_buff);
  idx += strlen(&obj_info[idx]);

  next_line();
  add_string("    ");

  /* get the first token */
  tokptr = type_buff;
  do {
    toklen_idx = strcspn(tokptr,"(+/");
    c = tokptr[toklen_idx];
    tokptr[toklen_idx] = '\0';

    /* remember any trailing colon */
    question_flag = tokptr[toklen_idx - 1] == ':';
    if (question_flag)
      tokptr[toklen_idx - 1] = '\0';

    /* find it in the table */
    match_idx = find_gcvs_type(tokptr,gcvs_types);
    if (match_idx == -1)
      sprintf(&obj_info[idx],"unknown variable type %s",tokptr);
    else
      sprintf(&obj_info[idx],"%s",gcvs_types[match_idx].type_desc);
    idx += strlen(&obj_info[idx]);

    /* add a question mark, if necessary */
    if (question_flag)
      add_char('?');

    /* output any parenthesized additions */
    if (c == '(') {
      /* put back the open parenthesis and isolate the parenthesized section */
      tokptr[toklen_idx] = c;
      tokptr = &type_buff[toklen_idx];
      toklen_idx = strcspn(tokptr,")") + 1;
      c = tokptr[toklen_idx];
      tokptr[toklen_idx] = '\0';
      
      /* position to output the text */
      next_line();
      advance_to_tab();

      /* find it in the table */
      match_idx = find_gcvs_type(tokptr,gcvs_types);
      if (match_idx == -1)
	sprintf(&obj_info[idx],"unknown variable type %s",tokptr);
      else
	sprintf(&obj_info[idx],"(%s)",gcvs_types[match_idx].type_desc);
      idx += strlen(&obj_info[idx]);
    }

    /* handle augmented or additional types */
    if (c == '+') {
      /* move into position for the next variable type */
      next_line();
      add_string("    ");
      
      add_char(c);
      add_char(' ');
    }
    else if (c == '/') {
      /* move into position for the next variable type */
      next_line();
      advance_to_tab();
      
      add_char(c);
      add_char(' ');
    }
    else
      ;

    /* advance the token pointer */
    tokptr = &tokptr[toklen_idx + 1];

  } while (c != '\0');

/* skip a line before the magnitudes */
  blank_line();

/* now display the magnitudes */
  sprintf(&obj_info[idx],"Maximum mag:  %.7s",&record[MAX_MAG]);
  idx += strlen(&obj_info[idx]);
  if (record[MAX_MAG_UNCERTAIN] == ':')
    add_char('?');

  if (record[MAX_MAG_LIMIT] == ')')
    add_string(" (lower limit)");

  next_line();

  sprintf(&obj_info[idx],"Minimum mag:  %.6s",&record[MIN_MAG]);
  idx += strlen(&obj_info[idx]);
  if (record[MIN_MAG_UNCERTAIN] == ':')
    add_char('?');

  if (record[MIN_MAG_LIMIT] == '(')
    if (record[MIN_MAG_DIFF] == ')') {
      add_string(" (differential magnitude");

      if (record[MIN_MAG_SYSTEM] != ' ') {
	sprintf(&obj_info[idx],", %c system)",record[MIN_MAG_SYSTEM]);
	idx += strlen(&obj_info[idx]);
      }
      else
	add_char(')');
    }
    else
      add_string(" (upper limit)");

  next_line();

/* display the photometric system used */
  if (record[MAGNITUDE_CODE] != ' ') {
    add_string("Photometric system:  ");

    /* expand the system type codes */
    switch (record[MAGNITUDE_CODE]) {
    case 'P':
      add_string("photographic");
      break;
    case 'V':
      add_string("visual");
      break;
    default:
      add_char(record[MAGNITUDE_CODE]);
      break;
    }

  blank_line();
  }

/* and the epoch */
  if (strncmp(&record[EPOCH],"              ",EPOCH_LENGTH) != EQUAL) {
    /* isolate and nul-terminate the buffer, eliminating trailing dot */
    strncpy(epoch_buff,&record[EPOCH],EPOCH_LENGTH);
    for (i = EPOCH_LENGTH; i > 0; i--)
      if (epoch_buff[i - 1] != ' ')
	break;

    if (epoch_buff[i] == '.')
      i--;
    epoch_buff[i] = '\0';

    sprintf(&obj_info[idx],"Epoch:  JD %s",epoch_buff);
    idx += strlen(&obj_info[idx]);

    /* see if the value is in question */
    switch (record[EPOCH_UNCERTAIN]) {
    case ':':
      add_char('?');
      break;
    case '+':
      add_string(" (may be later)");
      break;
    case '-':
      add_string(" (may be earlier)");
      break;
    case ' ':
      break;
    default:
      printf("unknown epoch uncertainty flag |%c|\n",record[EPOCH_UNCERTAIN]);
      break;
    }

  blank_line();
  }

/* show the period */
  if (strncmp(&record[PERIOD],"                ",PERIOD_LENGTH) != EQUAL) {
    add_string("Period:  ");

    /* remove trailing spaces */
    i = PERIOD + PERIOD_LENGTH;
    while (record[--i] == ' ')   ;
    if (i < (PERIOD + PERIOD_LENGTH - 1))
      record[i + 1] = '\0';

    sprintf(&obj_info[idx],"%.16s days",&record[PERIOD]);
    idx += strlen(&obj_info[idx]);

    /* see if the period is uncertain */
    if (record[PERIOD_UNCERTAIN] == ':')
      add_char('?');

    /* the given period may not be known to be between consecutive cycles */
    if (record[PERIOD_UNKNOWN_FLAG] == 'N')
      add_string("(unknown number of cycles");

    /* check for period limit condition */
    switch(record[PERIOD_LIMIT]) {
    case '<':
      add_string("  (upper limit)");
      break;
    case '>':
      add_string("  (lower limit)");
      break;
    case '(':
      add_string("  (mean cycle time)");
      break;
    case ' ':
      break;
    default:
      printf("unknown period limit flag |%c|\n",record[PERIOD_LIMIT]);
      break;
    }

    next_line();
  }

/* show the rise time or duration */
  if (strncmp(&record[RISE_DURATION],"   ",RISE_DURATION_LENGTH) != EQUAL) {
    sprintf(&obj_info[idx],"Rise/duration:  0.%.3s",&record[RISE_DURATION]);
    idx += strlen(&obj_info[idx]);

    /* see if the value is uncertain */
    if (record[RISE_DURATION_UNCERTAIN] == ':')
      add_char('?');

    add_string(" of the period");

    if (record[DURATION_ZERO_FLAG] == '*')
      add_string(" (zero-duration eclipse)");

    blank_line();
  }

/* add the spectral class */
  if (strncmp(&record[SPECTRUM_START],"                 ",SPECTRUM_LENGTH) !=
                                                                      EQUAL) {
    sprintf(&obj_info[idx],"Spectral classification:  %.17s",
		                                     &record[SPECTRUM_START]);
    idx += strlen(&obj_info[idx]);

    blank_line();
  }

/* tell the user if notes are available */
  if (record[REMARK_FLAG] == '*')
    add_string("Additional information is available in the notes");

/* fill the display out to the correct number of lines */
  while (row < NUM_GCVS_ROWS)
    next_line();
  obj_info[idx] = '\0';

/* and go back */
  return;
}



static void display_pos(posptr)

char *posptr;

{
/* identify the epoch */
  add_string("Position (1950.0)");
  next_line();

/* display the right ascension */
  advance_to_tab();
  sprintf(&obj_info[idx]," RA:   %.2sh %.2sm %.2ss",
		                            &posptr[0],&posptr[2],&posptr[4]);
  idx += strlen(&obj_info[idx]);
  next_line();

/* and display the declination */
  advance_to_tab();
  sprintf(&obj_info[idx],"Dec:  %.3sd %.2s'.%.1s",
	                                   &posptr[6],&posptr[9],&posptr[12]);
  idx += strlen(&obj_info[idx]);
  next_line();

  return;
}



static int find_gcvs_type(buff,table)

char *buff;
struct type_table *table;

{
  int max_match, max_match_idx;
  char *ptr;
  int desc;
  int i;

/* make the match length impossibly small */
  max_match = -1;
  max_match_idx = -1;

/* begin at the beginning */
  desc = 0;
  while (table[desc].type_id != NULL) {
/* point to the start of the description */
    ptr = buff;

/* match characters for a while */
    i = strlen(table[desc].type_id);
    if (strncmp(ptr,table[desc].type_id,(size_t)i) == EQUAL)
      if (i > max_match) {

/* the new match is bigger; save the length and the index */
	max_match = i;
	max_match_idx = desc;
      }

/* step to the next description */
    desc++;
  }

/* return the index to the relevant description table entry */
  return(max_match_idx);
}




/* find this Bayer designation in the GCVS Greek-letter table, returning a
 * zero-relative index or -1 on failure */

int find_gcvs_bayer(buff)

char *buff;

{
  int greek_idx;

/* after spelling rho as "PHO" all day, they spell it right just once ... */
  if (strcmp(buff,"RHO") == EQUAL)
    /* change RHO to PHO and drive on */
    buff[0] = 'P';

/* loop through the GCVS Greek-letter table, looking for a match */
  greek_idx = 0;
  while (strcmp(buff,gcvs_greek[greek_idx]) != EQUAL)
    if (++greek_idx >= NUM_GREEK_LETTERS)
      break;

/* if we didn't find it, complain and return a bogus value */
  if (greek_idx >= NUM_GREEK_LETTERS) {
    printf("no Greek letter found to match %s in GCVS\n",buff);
    greek_idx = -1;
  }

  return(greek_idx);
}
