/*
 * xsky - an interactive sky atlas
 *
 * Copyright 1992-6, Terry R. Friedrichsen
 *
 * This program may be copied and redistributed, in whole or in part,
 * as long as you don't try to make any money from the sale or redis-
 * tribution of the program or any part of the program, or pretend
 * that you wrote the program or any of its parts unless specifically
 * credited by the original author.
 *
 * You are free to make use of this software in your own programs, as
 * long as you credit the original author where it is due.
 */

/*
 * WARRANTY:
 * xsky was written as a learning project and as a demonstration of
 * X Window System programming.  xsky doesn't do anything; it is not
 * merchantable, and it is not fit for any purpose whatsoever.  In
 * fact, don't use xsky at all; it's free, and you're getting what
 * you paid for.
 */

#include <stdio.h>

#include <string.h>

#include <X11/Intrinsic.h>

#include "skydefs.h"
#include "catalog.h"
#include "draw.h"

/* static function prototypes */
static void draw_text PROTOTYPE((Widget,char *,IDfont,int,int));

extern unsigned long blackpixelval;
extern unsigned long whitepixelval;
extern unsigned long redpixelval;

extern GC star_gcs[];
extern GC mono_star_gc;
extern GC obj_gc;
extern GC text_gc;
extern GC grid_gc;

extern XFontStruct *latinidfontstr;
extern XFontStruct *latinconfontstr;
extern XFontStruct *greekidfontstr;

extern boolean mono_flag;
extern boolean night_mode_flag;

extern int x_height;
extern int x_width;



/* draw a single star on the sky */

void draw_star(sky_widget,star_x,star_y,star_diam,color)

Widget sky_widget;
int star_x, star_y;
int star_diam;
StarColor color;

{
  Position ul_x, ul_y;

#ifdef DEBUG
  printf("draw_star called to draw size %d dot at %d,%d\n",
	                                             star_diam,star_x,star_y);
#endif

/* calculate the upper-left coordinates of the enclosing box */
  ul_x = star_x - star_diam / 2;
  ul_y = star_y - star_diam / 2;

/* draw a star */
  if (mono_flag)
    /* monochrome display */
    XFillArc(XtDisplay(sky_widget),XtWindow(sky_widget),mono_star_gc,
	                                                  ul_x,ul_y,
		                                          star_diam,star_diam,
	                                                  0,360 * 64);
  else
    /* color display */
    XFillArc(XtDisplay(sky_widget),XtWindow(sky_widget),star_gcs[color],
	                                                  ul_x,ul_y,
		                                          star_diam,star_diam,
	                                                  0,360 * 64);

  return;
}



/* draw a single object on the sky */

void draw_object(sky_widget,obj_x,obj_y,obj_size,shape)

Widget sky_widget;
int obj_x, obj_y;
int obj_size;
ObjShape shape;

{
  Display *sky_display;
  Window sky_window;
  Position ul_x, ul_y;

/* get the display for the widget */
  sky_display = XtDisplay(sky_widget);

/* get the window for the widget */
  sky_window = XtWindow(sky_widget);

/* calculate the upper-left coordinates of the enclosing box */
  ul_x = obj_x - obj_size / 2;
  ul_y = obj_y - obj_size / 2;

/* draw an object figure depending on the given shape */
  switch (shape) {
  case ELLIPSE:
    /* compensate for elongation */
    ul_x--;
    XDrawArc(sky_display,sky_window,obj_gc,ul_x,ul_y,
	                                    obj_size + 2,obj_size,0,360 * 64);
    break;
  case CIRCLE:
    XDrawArc(sky_display,sky_window,obj_gc,ul_x,ul_y,
	                                obj_size + 1,obj_size + 1,0,360 * 64);
    break;
  case SQUARE:
    XDrawRectangle(sky_display,sky_window,obj_gc,ul_x,ul_y,obj_size,obj_size);
    break;
  case XMARK:
    draw_text(sky_widget,"x",LATIN_ID,(Position)(obj_x - x_width / 2),
		                            (Position)(obj_y + x_height / 2));
    break;
  default:
    break;
  }

  return;
}



/* draw an ID on the sky display */

void draw_id(sky_widget,id)

Widget sky_widget;
struct id_node *id;

{
/* clear the area where the ID will go (else overlapping IDs get scrozzled) */
  XClearArea(XtDisplay(sky_widget),XtWindow(sky_widget),id->ul_x,id->ul_y,
	               id->lr_x - id->ul_x + 1,id->lr_y - id->ul_y + 1,False);

/* and draw the text */
  draw_text(sky_widget,id->id_buffer,id->font,id->x,id->y);

  return;
}



/* draw text on the sky display */

static void draw_text(sky_widget,buff,font,x,y)

Widget sky_widget;
char *buff;
IDfont font;
Position x, y;

{
  Display *sky_display;
  Window sky_window;

/* get the display for the widget */
  sky_display = XtDisplay(sky_widget);

/* get the window for the widget */
  sky_window = XtWindow(sky_widget);

/* set the correct font */
  if (font == LATIN_ID)
    XSetFont(XtDisplay(sky_widget),text_gc,latinidfontstr->fid);
  if (font == LATIN_CON)
    XSetFont(XtDisplay(sky_widget),text_gc,latinconfontstr->fid);
  else if (font == GREEK)
    XSetFont(XtDisplay(sky_widget),text_gc,greekidfontstr->fid);
  else
    /* default to Latin */
    XSetFont(XtDisplay(sky_widget),text_gc,latinidfontstr->fid);

/* set the foreground to draw the text characters */
  if (night_mode_flag)
    XSetForeground(sky_display,text_gc,redpixelval);
  else
    XSetForeground(sky_display,text_gc,whitepixelval);

/* use the copy function to scribble over any overlaid objects */
  XSetFunction(sky_display,text_gc,GXcopy);

/* and draw the ID string */
  XDrawImageString(sky_display,sky_window,text_gc,x,y,buff,strlen(buff));

  return;
}



/* draw a box on the display */

void draw_box(sky_widget,x,y,wid,len)

Widget sky_widget;
int x, y;
int wid, len;

{
  Display *sky_display;

/* get the display for the widget */
  sky_display = XtDisplay(sky_widget);

/* set the foreground color to be the XOR of foreground and background */
  if (night_mode_flag)
    XSetForeground(sky_display,text_gc,redpixelval ^ blackpixelval);
  else
    XSetForeground(sky_display,text_gc,whitepixelval ^ blackpixelval);

/* use the XOR function to reversibly scribble over any overlaid objects */
  XSetFunction(sky_display,text_gc,GXxor);

/* draw the box */
  XDrawRectangle(sky_display,XtWindow(sky_widget),text_gc,x,y,wid,len);

  return;
}



/* draw an enclosure on the display */

void draw_enclosure(sky_widget,pointlist,num_points)

Widget sky_widget;
XPoint *pointlist;
int num_points;

{
  Display *sky_display;

/* get the display for the widget */
  sky_display = XtDisplay(sky_widget);

/* set the foreground color to be the XOR of foreground and background */
  if (night_mode_flag)
    XSetForeground(sky_display,text_gc,redpixelval ^ blackpixelval);
  else
    XSetForeground(sky_display,text_gc,whitepixelval ^ blackpixelval);

/* use the XOR function to reversibly scribble over any overlaid objects */
  XSetFunction(sky_display,text_gc,GXxor);

/* draw the enclosure */
  XDrawLines(sky_display,XtWindow(sky_widget),text_gc,
	                                pointlist,num_points,CoordModeOrigin);

  return;
}



/* draw a dot on the X display */

void draw_point(sky_widget,x,y)

Widget sky_widget;
Position x, y;

{
  XDrawPoint(XtDisplay(sky_widget),XtWindow(sky_widget),grid_gc,x,y);

  return;
}



/* draw a constellation boundary on the X display */

void draw_boundary_line(sky_widget,points,numpoints)

Widget sky_widget;
XPoint *points;
int numpoints;

{
  XDrawLines(XtDisplay(sky_widget),XtWindow(sky_widget),grid_gc,
	                                    points,numpoints,CoordModeOrigin);

  return;
}
